-- $Id: vcs-analysesimplogfile.adb 14621 2009-10-28 13:48:23Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse an .SLG file                                           --
--                                                                            --
--Uses SLG_Parser to deconstruct SLG file, finding which rule files used,     --
--which had syntax errors, and which VCs proved with which rules.             --
--                                                                            --
--This info stored in temporary files and recovered later in total            --
--------------------------------------------------------------------------------
with SLG_Parser,
     FatalErrors,
     PathFormatter,
     SPARK_IO;
use type SLG_Parser.Log_Status_T;
separate (VCS)
procedure AnalyseSimpLogFile
     (ReportFile      : in     SPARK_IO.File_Type;
      FileName        : in     ELStrings.T;
      RuleFilesErrors : in out SPARK_IO.File_Type;
      RuleFilesUsed   : in out SPARK_IO.File_Type;
      SLGErrorInFile  :    out Boolean)
is
   RuleLine         : ELStrings.T;
   RuleFile         : ELStrings.T;
   SLGDirectory     : ELStrings.T;
   RuleFileWithPath : ELStrings.T;

   Rule         : EStrings.T;
   OutputLine   : EStrings.T;
   VCNumber     : EStrings.T;

   SLGParserHandle      : SLG_Parser.Log_Info_T;
   SLGParserStatus      : SLG_Parser.Log_Status_T;
   SLGParserRuleStatus  : SLG_Parser.Log_Status_T;
   SLGParserVCStatus    : SLG_Parser.Log_Status_T;

   EncounteredARuleFile : Boolean;
   EncounteredARule     : Boolean;
   EncounteredAVC       : Boolean;
   Duplicated           : Boolean;
   Success              : Boolean;

   procedure FileContains (
      File     : in out SPARK_IO.File_Type;
      EStr     : in     ELStrings.T;
      Contains :    out Boolean)
   --# global in out FatalErrors.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Contains,
   --#         SPARK_IO.File_Sys from EStr,
   --#                                File,
   --#                                SPARK_IO.File_Sys &
   --#         FatalErrors.State from *,
   --#                                EStr,
   --#                                File,
   --#                                SPARK_IO.File_Sys &
   --#         File              from *,
   --#                                SPARK_IO.File_Sys;
   is
      Status : SPARK_IO.File_Status;
      TempStr : ELStrings.T;
   begin
      -- Reset file to read mode
      SPARK_IO.Reset (File, SPARK_IO.In_File, Status);

      -- Loop over all elements in the file
      -- comparing against the EStr
      Contains := False;
      if Status = SPARK_IO.Ok then
         while not SPARK_IO.End_Of_File (File) loop
            ELStrings.Get_Line (File  => File,
                                E_Str => TempStr);
            if ELStrings.Eq_String (E_Str1 => EStr,
                                    E_Str2 => TempStr) then
               Contains := True;
               exit;
            end if;
         end loop;

         -- Reset file to append mode
         SPARK_IO.Reset (File, SPARK_IO.Append_File, Status);
      end if;

      if Status /= SPARK_IO.Ok then
         FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.Empty_String);
      end if;
   end FileContains;

begin -- AnalyseSimpLogFile

   SLGErrorInFile := False;
   SLG_Parser.Init (FileName, SLGParserHandle, SLGParserStatus);

   if SLGParserStatus = SLG_Parser.Success then

      SLG_Parser.Find_Rulefiles_Read (SLGParserHandle, SLGParserStatus);

      if SLGParserStatus = SLG_Parser.Success then
         -- Find path prefix to use for rule files
         SLGDirectory := OSFiling.DirName (FileName);
         SLGDirectory := PathFormatter.Format (SLGDirectory);

         SLG_Parser.Find_Rule_Syntax_Errors (SLGParserHandle, SLGParserStatus);

         if SLGParserStatus = SLG_Parser.Success then
            -- Store syntax errors
            while SLGParserStatus = SLG_Parser.Success loop
               SLG_Parser.Get_Next_Rulefile_Syntax_Error (
                  SLGParserHandle,
                  RuleFile,
                  SLGParserStatus);

               if SLGParserStatus = SLG_Parser.Success then
                  -- Store the syntax error
                  RuleFileWithPath := SLGDirectory;
                  ELStrings.Append_Examiner_Long_String (E_Str1 => RuleFileWithPath,
                                                         E_Str2 => RuleFile);
                  FileContains (RuleFilesErrors, RuleFileWithPath, Duplicated);
                  if not Duplicated then
                     ELStrings.Put_Line (File  => RuleFilesErrors,
                                         E_Str => RuleFileWithPath);
                  end if;
               end if;

               -- Otherwise loop will terminate
            end loop;
         end if;

         -- All syntax errors stored, now proceed to rule summary
         SLG_Parser.Find_Rule_Summary (SLGParserHandle, SLGParserStatus);

         if SLGParserStatus = SLG_Parser.Success then

            EncounteredARuleFile := False;
            while SLGParserStatus = SLG_Parser.Success loop
               SLG_Parser.Get_Next_Rulefile (
                  SLGParserHandle,
                  RuleFile,
                  SLGParserStatus);

               if SLGParserStatus = SLG_Parser.Success then
                  -- Add the rulefile to the list of used files
                  RuleFileWithPath := SLGDirectory;
                  ELStrings.Append_Examiner_Long_String (E_Str1 => RuleFileWithPath,
                                                         E_Str2 => RuleFile);
                  FileContains (RuleFilesUsed, RuleFileWithPath, Duplicated);
                  if not Duplicated then
                     ELStrings.Put_Line (File  => RuleFilesUsed,
                                         E_Str => RuleFileWithPath);
                  end if;

                  -- Output to the report
                  if not EncounteredARuleFile then
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_Line (ReportFile, "The following user rules were used:", 0);
                     EncounteredARuleFile := True;
                  end if;

                  RuleLine := ELStrings.Copy_String (Str => "from ");
                  ELStrings.Append_Examiner_Long_String (E_Str1 => RuleLine,
                                                         E_Str2 => RuleFileWithPath);
                  ELStrings.Put_Line (File  => ReportFile,
                                      E_Str => RuleLine);

                  EncounteredARule := False;
                  SLGParserRuleStatus := SLG_Parser.Success;
                  while SLGParserRuleStatus = SLG_Parser.Success loop
                     SLG_Parser.Get_Next_Rule (
                        SLGParserHandle,
                        Rule,
                        SLGParserRuleStatus);


                     if SLGParserRuleStatus = SLG_Parser.Success then
                        -- Output rule number to report
                        -- Rule.Length should be < 256 - 20 - 3
                        EncounteredARule := True;
                        OutputLine := EStrings.Copy_String (Str => "   ");
                        EStrings.Append_Examiner_String (E_Str1 => OutputLine,
                                                         E_Str2 => Rule);
                        EStrings.Append_String (E_Str => OutputLine,
                                                Str   => " used in proving VCs:");
                        EStrings.Put_Line (File  => ReportFile,
                                           E_Str => OutputLine);

                        EncounteredAVC := False;
                        OutputLine := EStrings.Copy_String (Str => "      ");
                        SLGParserVCStatus := SLG_Parser.Success;
                        while SLGParserVCStatus = SLG_Parser.Success loop
                           SLG_Parser.Get_Next_VC
                             (SLGParserHandle,
                              VCNumber,
                              SLGParserVCStatus);


                           if SLGParserVCStatus = SLG_Parser.Success then
                              -- Output VC number to report
                              EStrings.Append_Examiner_String (E_Str1 => OutputLine,
                                                               E_Str2 => VCNumber);
                              EStrings.Append_String (E_Str => OutputLine,
                                                      Str   => ", ");
                              EncounteredAVC := True;
                           else
                              -- Remove the comma and replace with full stop
                              OutputLine := EStrings.Section
                                (E_Str     => OutputLine,
                                 Start_Pos => 1,
                                 Length    => EStrings.Get_Length (E_Str => OutputLine) - 2);
                              --# accept F, 10, Success, "Ineffective assignment here OK";
                              EStrings.Append_Char (E_Str   => OutputLine,
                                                    Ch      => '.',
                                                    Success => Success);
                              --# end accept;
                           end if;

                        end loop;

                        -- Must have found a VC
                        if not EncounteredAVC or
                           SLGParserVCStatus = SLG_Parser.Unexpected_Text
                        then
                           SLGErrorInFile := True;
                        end if;
                        EStrings.Put_Line (File  => ReportFile,
                                           E_Str => OutputLine);
                     end if;

                  end loop;

                  -- Must have found a rule
                  if not EncounteredARule or
                     SLGParserRuleStatus = SLG_Parser.Unexpected_Text then

                     SLGErrorInFile := True;
                  end if;
               end if;

            end loop;

            -- Must have found a rulefile
            if not EncounteredARuleFile or
               SLGParserStatus = SLG_Parser.Unexpected_Text then

               SLGErrorInFile := True;
            end if;
         end if;
      end if;

      --# accept Flow, 10, SLGParserHandle, "Modify filehandle to close file";
      SLG_Parser.Finalise (SLGParserHandle);
      --# end accept;

   else
      SLGErrorInFile := True;
   end if;

   --# accept Flow,  33, Success, "Expect Success unused" &
   --#        Flow, 601, RuleFilesUsed, ReportFile, "False coupling through SPARK_IO" &
   --#        Flow, 601, RuleFilesUsed, RuleFilesErrors, "False coupling through SPARK_IO";
end AnalyseSimpLogFile;
