-- $Id: vcs-analysereviewfile.adb 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .PRV file                                            --
--                                                                            --
--Grammar for .PRV files:                                                     --
--PRVfile  ::= {Line}                                                         --
--Line     ::= [VCNumber][Comment]end_of_line_character                       --
--VCNumber ::= digit{digit}                                                   --
--Comment  ::= --{non_end_of_line_character}                                  --
--                                                                            --
--The proof review file (prv) gives the user an opportunity to declare that   --
--a VC has been proved by inspection. The VC numbers proved can be placed one --
--per line with Ada style comments.                                           --
--------------------------------------------------------------------------------
separate (VCS)

procedure AnalyseReviewFile
  (ReportFile : in     SPARK_IO.File_Type;
   FileName   : in     ELStrings.T;
   Errors     :    out ReviewErrors)
is

   -- Each line in the PRV file will be categorised as one of the following:
   -- A VCNumber line will contain a valid VC number and may contain a comment.
   -- A CommentLine is a line with a comment and nothing else.
   -- Any other line is deemed an InvalidLine.
   type PRVFileLineType is (VCNumberLine,
                            CommentLine,
                            InvalidLine);

   -- These are the errors that will be reported to the user via the report file.
   type Error is (VCSyntaxError,
                  VCDoesNotExist,
                  VCProvedByExaminer,
                  VCProvedBySimplifier,
                  VCProvedByChecker,
                  VCDuplicated);


   -- The .prv file.
   ReviewFile       : SPARK_IO.File_Type := SPARK_IO.Null_File;
   DummyCloseStatus : SPARK_IO.File_Status;
   OpenStatus       : SPARK_IO.File_Status;

   FinishedWithFile : Boolean;
   ReadLineSuccess  : Boolean;
   TheLineType      : PRVFileLineType;

   FileLine         : EStrings.T;
   VCName           : EStrings.T;
   TheResult        : EStrings.T;

   TmpErrors        : ReviewErrors;

   -- Reads the given (non blank) line.
   -- TheLineType categorises the line.
   -- TheResult is set to:
   --    o The VC number found on the line if the line is a VCNumberLine.
   --    o Null string is the line is a CommentLine.
   --    o The entire line if the line is an InvalidLine.
   procedure ProcessLine (TheLine     : in     EStrings.T;
                          TheLineType :    out PRVFileLineType;
                          TheResult   :    out EStrings.T)
      --# derives TheLineType,
      --#         TheResult   from TheLine;
   is
      subtype Digit is Character range '0' .. '9';

      subtype TwoCharStringIndex is Positive range 1 .. 2;
      subtype TwoCharString is String (TwoCharStringIndex);
      Comment : constant TwoCharString := "--";

      Index    : EStrings.Positions;
      TheChar  : Character;
      Finished : Boolean;

   begin
      TheLineType := InvalidLine;
      Index := 1;
      Finished := False;

      while not Finished and then
        Index <= EStrings.Get_Length (E_Str => TheLine) loop

         TheChar := EStrings.Get_Element (E_Str => TheLine,
                                          Pos   => Index);

         if TheChar in Digit then
            -- We found a number. Keep looking until a non digit is found.
            TheLineType := VCNumberLine;

         elsif EStrings.Eq1_String
            -- If the next characters (ignoring spaces) constitute a comment then we're done.
           (E_Str => EStrings.Section
              (EStrings.Trim
                 (EStrings.Section
                    (E_Str     => TheLine,
                     Start_Pos => Index,
                     Length    => (EStrings.Get_Length (E_Str => TheLine) - Index) + 1)),
               1,
               Comment'Length),
            Str   => Comment) then

            -- This is a comment line unless we already found a VC number in which case
            -- it's a VCNumber line as set above.
            if TheLineType /= VCNumberLine then
               TheLineType := CommentLine;
            end if;
            Finished := True;

         else
            -- This line is invalid as it contains neither a VC number or a comment.
            TheLineType := InvalidLine;
            Finished := True;
         end if;

         if not Finished then
            Index := Index + 1;
         end if;

      end loop;

      case TheLineType is
         when VCNumberLine =>
            TheResult := EStrings.Section (E_Str     => TheLine,
                                           Start_Pos => 1,
                                           Length    => Index - 1);
         when CommentLine =>
            TheResult := EStrings.Empty_String;
         when InvalidLine =>
            TheResult := TheLine;
      end case;

   end ProcessLine;


--     type ErrorsIndex is Integer range 1..1000;
--     type ErrorsList is array (ErrorsIndex) of EStrings.T;

--     type ReviewErrors is record
--        Errors       : Boolean      -- Have we recorded any errors?
--        ErrorList    : ErrorsList;  -- The list of errors
--        LastError    : ErrorsIndex; -- The last error
--        ExcessCount  : Natural;     -- Are there more than 1000 errors?
--     end record;


   procedure InitReviewErrors
   --# global out TmpErrors;
   --# derives TmpErrors from ;
   is
   begin
      TmpErrors := ReviewErrors'(Errors => False,
                                 ErrorList => ErrorsList'(others => EStrings.Empty_String),
                                 LastError => ErrorsIndex'First,
                                 ExcessCount => 0);
   end InitReviewErrors;

   procedure AddReviewError (Err : in EStrings.T)
   --# global in out TmpErrors;
   --# derives TmpErrors from *,
   --#                        Err;
   is
   begin

      -- Is this the first time round?
      if not TmpErrors.Errors then
         TmpErrors.LastError := ErrorsIndex'First;
         TmpErrors.Errors := True;  -- Close the guard.
      end if;

      -- Check if there is any room left
      if TmpErrors.LastError = ErrorsIndex'Last then
         -- FULL: increment the ExcessCount
         TmpErrors.ExcessCount := TmpErrors.ExcessCount + 1;
      else -- Otherwise, increment the end-of-list point
         TmpErrors.LastError := TmpErrors.LastError + 1;
      end if;

      if TmpErrors.ExcessCount = 0 then  -- There was some room left
         TmpErrors.ErrorList (TmpErrors.LastError) := Err;  -- Store the error.
      end if;

   end AddReviewError;


   procedure RecordError (TheError   : in Error;
                          TheDetails : in EStrings.T)
   --# global in out TmpErrors;
   --# derives TmpErrors from *,
   --#                        TheDetails,
   --#                        TheError;
   is
      TmpMessage : EStrings.T;
      MaxErrorLength : constant Integer := 40;
   begin
         case TheError is
            when VCSyntaxError =>
               TmpMessage := EStrings.Copy_String (Str => "Syntax error: ");
               if EStrings.Get_Length (E_Str => TheDetails) > MaxErrorLength then
                  EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                   E_Str2 => EStrings.Section (TheDetails,
                                                                               EStrings.Positions'First,
                                                                               MaxErrorLength));
               else
                  EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                   E_Str2 => TheDetails);
               end if;

            when VCDoesNotExist =>
               TmpMessage := EStrings.Copy_String (Str => "Warning: VC ");
               EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                E_Str2 => TheDetails);
               EStrings.Append_String (E_Str => TmpMessage,
                                       Str   => " not recognised");

            when VCProvedByExaminer =>
               TmpMessage := EStrings.Copy_String (Str => "Warning: VC ");
               EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                E_Str2 => TheDetails);
               EStrings.Append_String (E_Str => TmpMessage,
                                       Str   => " has been proved by the examiner");

            when VCProvedBySimplifier =>
               TmpMessage := EStrings.Copy_String (Str => "Warning: VC ");
               EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                E_Str2 => TheDetails);
               EStrings.Append_String (E_Str => TmpMessage,
                                       Str   => " has been proved by the simplifier");

            when VCProvedByChecker =>
               TmpMessage := EStrings.Copy_String (Str => "Warning: VC ");
               EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                E_Str2 => TheDetails);
               EStrings.Append_String (E_Str => TmpMessage,
                                       Str   => " has been proved by the checker");

            when VCDuplicated =>
               TmpMessage := EStrings.Copy_String (Str => "Warning: VC ");
               EStrings.Append_Examiner_String (E_Str1 => TmpMessage,
                                                E_Str2 => TheDetails);
               EStrings.Append_String (E_Str => TmpMessage,
                                       Str   => " has been duplicated");
         end case;

         AddReviewError (TmpMessage);

   end RecordError;


   -- Reports TheError to the report file. TheDetails are used to give
   -- further information to help track down the offending line in the file.
   procedure ReportError (TheError   : in Error;
                          TheDetails : in EStrings.T)
      --# global in     ReportFile;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TmpErrors;
      --# derives SPARK_IO.File_Sys from *,
      --#                                ReportFile,
      --#                                TheDetails,
      --#                                TheError &
      --#         TmpErrors         from *,
      --#                                TheDetails,
      --#                                TheError;
   is
      -- The max number of characters output for a line containing a syntax error.
      MaxErrorLength : constant := 40;
   begin
      RecordError (TheError, TheDetails);

      case TheError is

         when VCSyntaxError =>
            SPARK_IO.Put_String (ReportFile,
                                    "*** Error: The following line in the proof review file contains a syntax error: ***", 0);
            SPARK_IO.New_Line (ReportFile, 1);
            SPARK_IO.Put_String (ReportFile, "***        """, 0);
            if EStrings.Get_Length (E_Str => TheDetails) > MaxErrorLength then
               EStrings.Put_String
                 (File  => ReportFile,
                  E_Str => EStrings.Section (E_Str     => TheDetails,
                                             Start_Pos => 1,
                                             Length    => MaxErrorLength));
               SPARK_IO.Put_String (ReportFile, "...", 0);
            else
               EStrings.Put_String (File  => ReportFile,
                                    E_Str => TheDetails);
            end if;
            SPARK_IO.Put_String (ReportFile, """", 0);
            SPARK_IO.New_Line (ReportFile, 1);

         when VCDoesNotExist =>
            SPARK_IO.Put_String (ReportFile, "*** Warning: VC ", 0);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TheDetails);
            SPARK_IO.Put_String (ReportFile,  " in proof review file is not recognised ***", 0);
            SPARK_IO.New_Line (ReportFile, 1);

         when VCProvedByExaminer =>
            SPARK_IO.Put_String (ReportFile, "*** Warning: VC ", 0);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TheDetails);
            SPARK_IO.Put_String (ReportFile,
                                 " in proof review file has been proved by the examiner ***", 0);
            SPARK_IO.New_Line (ReportFile, 1);

         when VCProvedBySimplifier =>
            SPARK_IO.Put_String (ReportFile, "*** Warning: VC ", 0);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TheDetails);
            SPARK_IO.Put_String (ReportFile,
                                 " in proof review file has been proved by the simplifier ***", 0);
            SPARK_IO.New_Line (ReportFile, 1);

         when VCProvedByChecker =>
            SPARK_IO.Put_String (ReportFile, "*** Warning: VC ", 0);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TheDetails);
            SPARK_IO.Put_String (ReportFile,
                                 " in proof review file has been proved by the checker ***", 0);
            SPARK_IO.New_Line (ReportFile, 1);

         when VCDuplicated =>
            SPARK_IO.Put_String (ReportFile, "*** Warning: VC ", 0);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TheDetails);
            SPARK_IO.Put_String (ReportFile,
                                 " in proof review file has been duplicated ***", 0);
            SPARK_IO.New_Line (ReportFile, 1);
      end case;
   end ReportError;


begin -- AnalyseReviewFile

   -- open review file
   ELStrings.Open (File         => ReviewFile,
                   Mode_Of_File => SPARK_IO.In_File,
                   Name_Of_File => FileName,
                   Form_Of_File => "",
                   Status       => OpenStatus);

   -- Initialise the error record stack
   InitReviewErrors;

   if OpenStatus /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.Empty_String);
   end if;

   -- Find first non blank line if we get to the end of the file first,
   -- flag a fatal error
   ReadNextNonBlankLine (ReviewFile, ReadLineSuccess, FileLine);

   if ReadLineSuccess then

      FinishedWithFile := False;

      -- Process file line-by-line.
      -- On entry to the loop there is already a valid line in the FileLine buffer
      while not FinishedWithFile loop

         -- Get the first token on the line and act accordingly
         ProcessLine (TheLine => FileLine,
                      TheLineType => TheLineType,
                      TheResult => TheResult);

         case TheLineType is

            when VCNumberLine =>

               VCName := VCHeap.GetVCNamePrefix;
               EStrings.Append_String (E_Str => VCName,
                                       Str   => "_");
               EStrings.Append_Examiner_String (E_Str1 => VCName,
                                                E_Str2 => TheResult);

               if not VCHeap.Exists (VCName) then

                  ReportError (TheError => VCDoesNotExist,
                               TheDetails => TheResult);

               elsif VCHeap.Get_VC_State (VCName) = VCDetails.VC_Proved_By_Examiner then
                  ReportError (TheError => VCProvedByExaminer,
                               TheDetails => TheResult);

               elsif VCHeap.Get_VC_State (VCName) = VCDetails.VC_Proved_By_Inference or
                 VCHeap.Get_VC_State (VCName) = VCDetails.VC_Proved_By_Contradiction or
                 VCHeap.Get_VC_State (VCName) = VCDetails.VC_Proved_Using_User_Proof_Rules then
                  ReportError (TheError => VCProvedBySimplifier,
                               TheDetails => TheResult);

               elsif VCHeap.Get_VC_State (VCName) = VCDetails.VC_Proved_By_Checker then
                  ReportError (TheError => VCProvedByChecker,
                               TheDetails => TheResult);

               elsif VCHeap.Get_VC_State (VCName) = VCDetails.VC_Proved_By_Review then
                  ReportError (TheError => VCDuplicated,
                               TheDetails => TheResult);

               else
                  -- Mark VC as proved by review
                  VCHeap.Set_VC_State (VCName, VCDetails.VC_Proved_By_Review);

               end if;

            when CommentLine => null;

            when InvalidLine =>

               ReportError (TheError => VCSyntaxError,
                            TheDetails => TheResult);

         end case;

         if not FinishedWithFile then
            -- Read next line
            ReadNextNonBlankLine (ReviewFile, ReadLineSuccess, FileLine);

            -- If unsuccessful then check EOF and set FinishedWithFile accordingly
            if not ReadLineSuccess then
               if SPARK_IO.End_Of_File (ReviewFile) then
                  FinishedWithFile := True;
               else
                  FatalErrors.Process (FatalErrors.ProblemReadingFile, ELStrings.Empty_String);
               end if;
            end if;
         end if;

      end loop;

   else

      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "************* Review file empty ************",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);

   end if;

   --# accept F, 10, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 33, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 10, ReviewFile,       "ReviewFile unused here";
   SPARK_IO.Close (ReviewFile, DummyCloseStatus);

   Errors := TmpErrors;

end AnalyseReviewFile;
