-- $Id: vcs-analyseprooflogfile.adb 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .PLG file                                            --
--                                                                            --
--------------------------------------------------------------------------------

separate (VCS)
procedure AnalyseProofLogFile
  (ReportFile      : in     SPARK_IO.File_Type;
   FileName        : in     ELStrings.T;
   SIVFileDateTime : in     EStrings.T;
   PLGFileDateTime :    out EStrings.T;
   ErrorInFile     :    out Boolean;
   FileError       :    out EStrings.T)
is
   DummyCloseStatus           : SPARK_IO.File_Status;
   FileLine                   : EStrings.T;
   FinishedWithFile           : Boolean;
   OpenStatus                 : SPARK_IO.File_Status;
   ReadLineSuccess            : Boolean;
   ProofLogFile               : SPARK_IO.File_Type := SPARK_IO.Null_File;

   VCProofDateTimeFromPLGFile : EStrings.T;
   ProofLogObsolete           : Boolean;
   TrimmedLine                : EStrings.T;

   CurrentVCName              : EStrings.T;
   ---------------------------------------------------------------------------
   procedure ExtractDatesAndTimesFromProofLogFile
     (ProofLogFile     : in     SPARK_IO.File_Type;
      ProofLogDateTime :    out EStrings.T)
   --# global in out SPARK_IO.File_sys;
   --# derives ProofLogDateTime,
   --#         SPARK_IO.File_sys from ProofLogFile,
   --#                                SPARK_IO.File_sys;
   is
      FileLine      : EStrings.T;
      TrimmedLine   : EStrings.T;
      ProofDateTime : EStrings.T;
      StartFound    : Boolean := False;

   begin

      ProofLogDateTime := EStrings.Empty_String;

      while not StartFound loop

         EStrings.Get_Line (File  => ProofLogFile,
                            E_Str => FileLine);
         TrimmedLine := EStrings.Trim (FileLine);

         -- find date
         if EStrings.Eq1_String (E_Str => EStrings.Section (TrimmedLine, 1, 4),
                                 Str   => "DATE") then
            -- extract the proof session date and time from the string
            ProofDateTime := EStrings.Section
                               (TrimmedLine,
                               PLGFileVCProofDateStartColumn,
                               PLGFileVCProofDateLength);
            EStrings.Append_String (E_Str => ProofDateTime,
                                    Str   => " ");
            EStrings.Append_Examiner_String
              (E_Str1 => ProofDateTime,
               E_Str2 => EStrings.Section (TrimmedLine,
                                           PLGFileVCProofTimeStartColumn,
                                           PLGFileVCProofTimeLength));

            ProofLogDateTime := ProofDateTime;
         end if;

         -- find start of instructions, then go on to analyse the rest of the file
         if EStrings.Eq1_String (E_Str => EStrings.Section (TrimmedLine, 1, 7),
                                 Str   => "COMMAND") then
               StartFound := True;
         end if;

      end loop;

      -- if date has not been found must be in plain output mode
      if EStrings.Is_Empty (E_Str => ProofLogDateTime) then
         ProofLogDateTime := EStrings.Copy_String (Str => "Unknown date");
      end if;

   end ExtractDatesAndTimesFromProofLogFile;


   -------------------------------------------------------------------------
   procedure GetDateAndTime (DateString : in     EStrings.T;
                             DateStamp  :    out SPARK_Calendar.Time;
                             TimeStamp  :    out Integer;
                             Error      :    out Boolean)
   --# derives DateStamp,
   --#         Error,
   --#         TimeStamp from DateString;
   is
      subtype MonthIndex is Integer range 0 .. SPARK_Calendar.Month_Number'Last;

      YearNum        : SPARK_Calendar.Year_Number;
      RawMonthNum    : MonthIndex;
      MonthNum       : SPARK_Calendar.Month_Number;
      DayNum         : SPARK_Calendar.Day_Number;
      HoursNum       : Integer;
      MinutesNum     : Integer;
      SecondsNum     : Integer;
      TimeInSeconds  : Integer;
      Stamp          : SPARK_Calendar.Time;
      Stop           : Natural;
      TimeError      : SPARK_Calendar.Error_Code;

      -------------------------------------------------------------------------
      function MonthNameToMonthNum
        (MonthName : EStrings.T) return MonthIndex
      is
         Num : MonthIndex := 0;
      begin -- MonthNameToMonthNum
         if    EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "JAN") then Num := 1;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "FEB") then Num := 2;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "MAR") then Num := 3;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "APR") then Num := 4;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "MAY") then Num := 5;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "JUN") then Num := 6;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "JUL") then Num := 7;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "AUG") then Num := 8;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "SEP") then Num := 9;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "OCT") then Num := 10;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "NOV") then Num := 11;
         elsif EStrings.Eq1_String (E_Str => MonthName,
                                    Str   => "DEC") then Num := 12;
         end if;

         return Num;
      end MonthNameToMonthNum;

   begin --GetDateAndTime

      -- If the provided date string is empty, do not attempt to parse it, and
      -- return an error.  Otherwise constraint errors are raised in trying to
      -- convert an empty string into an integer value.
      -- Note this check does not exclude all situations that may raise a
      -- constraint error. To do this, would need to check that the string
      -- contains numeric characters in the correct locations.
      if EStrings.Is_Empty (E_Str => DateString) then
         Error := True;
         TimeStamp := 0;
      else
         --# accept F, 10, Stop, "Stop unused here";
         EStrings.Get_Int_From_String
           (Source   => EStrings.Section (E_Str     => DateString,
                                          Start_Pos => 8,
                                          Length    => 4),
            Item     => YearNum,
            Start_Pt => 1,
            Stop     => Stop);
         --# end accept;

         RawMonthNum := MonthNameToMonthNum (EStrings.Section (DateString, 4, 3));

         if RawMonthNum = 0 then
            Error := True;
            TimeStamp := 0;
         else
            MonthNum := RawMonthNum;
            --# accept F, 10, Stop, "Stop unused here";
            EStrings.Get_Int_From_String
              (Source   => EStrings.Section (E_Str     => DateString,
                                             Start_Pos => 1,
                                             Length    => 2),
               Item     => DayNum,
               Start_Pt => 1,
               Stop     => Stop);
            --# end accept;
            SPARK_Calendar.Time_Of (YearNum,
                                    MonthNum,
                                    DayNum,
                                    Stamp,
                                    TimeError);

            if not (TimeError = SPARK_Calendar.Valid) then
               Error := True;
               TimeStamp := 0;
            else
               --# accept F, 10, Stop, "Stop unused here";
               EStrings.Get_Int_From_String
                 (Source   => EStrings.Section (E_Str     => DateString,
                                                Start_Pos => 13,
                                                Length    => 2),
                  Item     => HoursNum,
                  Start_Pt => 1,
                  Stop     => Stop);

               EStrings.Get_Int_From_String
                 (Source   => EStrings.Section (E_Str     => DateString,
                                                Start_Pos => 16,
                                                Length    => 2),
                  Item     => MinutesNum,
                  Start_Pt => 1,
                  Stop     => Stop);

               EStrings.Get_Int_From_String
                 (Source   => EStrings.Section (E_Str     => DateString,
                                                Start_Pos => 19,
                                                Length    => 2),
                  Item     => SecondsNum,
                  Start_Pt => 1,
                  Stop     => Stop);
               --# end accept;
               TimeInSeconds := (SecondsNum + (MinutesNum * 60)) + (HoursNum * 3600);

               DateStamp := Stamp;
               TimeStamp := TimeInSeconds;
               Error := False;
            end if;
         end if;
      end if;
      --# accept F,  33, Stop, "Stop unused here" &
      --#        F, 602, DateStamp, DateStamp, "Always well-defined when no error";
   end GetDateAndTime;

   -------------------------------------------------------------------------
   procedure CheckProofLogObsolescence (ProofLogDateTime : in     EStrings.T;
                                        SIVFileDateTime  : in     EStrings.T;
                                        PLGObsolete      :    out Boolean)
   --# global in     CommandLine.Data;
   --#        in out SPARK_IO.File_sys;
   --# derives PLGObsolete       from CommandLine.Data,
   --#                                ProofLogDateTime,
   --#                                SIVFileDateTime &
   --#         SPARK_IO.File_sys from *,
   --#                                ProofLogDateTime,
   --#                                SIVFileDateTime;
   is
      Result       : Boolean;

      PLGTimeStamp : Integer;
      SIVTimeStamp : Integer;

      PLGDateStamp : SPARK_Calendar.Time;
      SIVDateStamp : SPARK_Calendar.Time;

      PLGError     : Boolean;
      SIVError     : Boolean;

   begin

      if EStrings.Eq1_String (E_Str => ProofLogDateTime,
                              Str   => "Unknown date") then

         -- If the /i option is specified then the absence of date is not a problem and we
         -- can assume that the file is valid.
         -- If the /i option is not specified then dates are assumed to be important and a
         -- file that lacks a date is therefore obsolete.
         Result := CommandLine.Data.IgnoreDates;
      else

         GetDateAndTime (ProofLogDateTime, PLGDateStamp, PLGTimeStamp, PLGError);
         GetDateAndTime (SIVFileDateTime,  SIVDateStamp, SIVTimeStamp, SIVError);
         if PLGError then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Date format error in PLG file", 0);
            Result := True;
         elsif SIVError then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Date format error in SIV file", 0);
            Result := True;
         elsif SPARK_Calendar.LT (SIVDateStamp, PLGDateStamp) then
            Result := False;
         elsif SPARK_Calendar.GT (SIVDateStamp, PLGDateStamp) then
            Result := True;
         elsif PLGTimeStamp < SIVTimeStamp then
            Result := True;
         else
            Result := False;
         end if;

      end if;

      PLGObsolete := Result;

--   exception
--      when others =>
--         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Failure in CheckProofLogObsolescence", 0);
--         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "ProofLogDateTime = ", 0);
--         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, ProofLogDateTime.Content, 0);
--         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "SIVFileDateTime = ", 0);
--         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, SIVFileDateTime.Content, 0);

   end CheckProofLogObsolescence;

   --------------------------------------------------------------------------
begin -- AnalyseProofLogFile

   -- open proof log file
   ELStrings.Open (File         => ProofLogFile,
                   Mode_Of_File => SPARK_IO.In_File,
                   Name_Of_File => FileName,
                   Form_Of_File => "",
                   Status       => OpenStatus);
   if OpenStatus /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.Empty_String);
   end if;

   ExtractDatesAndTimesFromProofLogFile (ProofLogFile,
                                         VCProofDateTimeFromPLGFile);

   --  If the SIV file has a Unknown date/time arising from /plain,
   --  then assume the proof log is obsolete, UNLESS we're running
   --  POGS with /i, so...

   if EStrings.Eq1_String (E_Str => SIVFileDateTime,
                           Str   => UnknownSIVDate) then
      ProofLogObsolete := True;
   else
      CheckProofLogObsolescence
        (VCProofDateTimeFromPLGFile, SIVFileDateTime, ProofLogObsolete);
   end if;

   if ProofLogObsolete and not CommandLine.Data.IgnoreDates then
      if not CommandLine.Data.XML then
         SPARK_IO.New_Line (ReportFile, 1);
         SPARK_IO.Put_Line (ReportFile,
                         "*** Warning: Proof Log file out of date ***",
                            0);
         SPARK_IO.Put_String (ReportFile, "SIV file time stamp: ", 0);
         EStrings.Put_Line (File  => ReportFile,
                            E_Str => SIVFileDateTime);
         SPARK_IO.Put_String (ReportFile, "PLG file time stamp: ", 0);
         EStrings.Put_Line (File  => ReportFile,
                            E_Str => VCProofDateTimeFromPLGFile);
      end if;

      ErrorInFile := True;
      FileError := XMLSummary.XStr ("Warning: Proof log file out of date");
      PLGFileDateTime := EStrings.Empty_String;

   else

      SPARK_IO.New_Line (ReportFile, 1);

      -- Only output dates if we are not ignoring them.
      if CommandLine.Data.IgnoreDates = False
        and CommandLine.Data.XML = False then
         SPARK_IO.Put_String (ReportFile, "VCs proved ", 0);
         EStrings.Put_Line (File  => ReportFile,
                            E_Str => VCProofDateTimeFromPLGFile);
      elsif CommandLine.Data.IgnoreDates and CommandLine.Data.XML then
         VCProofDateTimeFromPLGFile := EStrings.Empty_String;
      end if;

      PLGFileDateTime := VCProofDateTimeFromPLGFile;  -- Return the date and time.

      -- find first non blank line
      -- if we get to the end of the file first, flag a fatal error
      ReadNextNonBlankLine (ProofLogFile, ReadLineSuccess, FileLine);

      if not ReadLineSuccess then
         if not CommandLine.Data.XML then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                               "************* Proof Log file empty ************",
                               0);
            SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         end if;
         ErrorInFile := True;
         FileError := XMLSummary.XStr ("Proof log file empty");
      else
         ErrorInFile := False;
         FileError := EStrings.Empty_String;

         FinishedWithFile := False;

         -- process file line-by-line
         -- on entry to the loop there is already a valid line in the
         -- FileLine buffer
         while not FinishedWithFile loop
            -- examine line and act accordingly

            if IsVCProofSuccessLine (FileLine) then
               TrimmedLine := EStrings.Trim (FileLine);
               CurrentVCName := EStrings.Section
                 (E_Str     => TrimmedLine,
                  Start_Pos => 15,
                  Length    => EStrings.Get_Length (E_Str => TrimmedLine) - 14);

               -- In a proof log file, it's possible that the "Proof
               -- Success Line" for a single VC can appear multiple times
               -- (if the user types "done" repeatedly after proving a VC for
               -- example), so we only mark the VC as proved here ONCE,
               -- to stop it being accounted for multiple times in the Totals.

               if VCHeap.Get_VC_State (CurrentVCName) /= VCDetails.VC_Proved_By_Checker then
                  VCHeap.Set_VC_State (CurrentVCName,
                                       VCDetails.VC_Proved_By_Checker);
               end if;

            end if;

            if not FinishedWithFile then
               -- read next line
               ReadNextNonBlankLine (ProofLogFile, ReadLineSuccess, FileLine);

               -- if unsuccessful then check EOF
               -- and set FinishedWithFile accordingly
               if not ReadLineSuccess then
                  if SPARK_IO.End_Of_File (ProofLogFile) then
                     FinishedWithFile := True;
                  else
                     FatalErrors.Process (FatalErrors.ProblemReadingFile, ELStrings.Empty_String);
                  end if;
               end if;
            end if;
         end loop;
      end if;
   end if;

   --# accept F, 10, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 33, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 10, ProofLogFile,     "ProofLogFile unused here";
   SPARK_IO.Close (ProofLogFile, DummyCloseStatus);

end AnalyseProofLogFile;
