-- $Id: vcs-analysesimplifiedvcfile.adb 14621 2009-10-28 13:48:23Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .SPD file                                            --
--                                                                            --
--------------------------------------------------------------------------------
separate (VCS)
procedure Analyse_Summary_DP_File
  (Report_File          : in     SPARK_IO.File_Type;
   File_Name            : in     ELStrings.T;
   DPC_File_Date_Time   : in     EStrings.T;
   Error_In_SDP_File    :    out Boolean)
is
   Bad_File_Format       : Boolean := False;
   Dummy_Close_Status    : SPARK_IO.File_Status;
   File_Line             : EStrings.T;
   Finished_With_File    : Boolean;
   Open_Status           : SPARK_IO.File_Status;
   Process_Success       : Boolean;
   Read_Line_Success     : Boolean;
   Summary_DP_File       : SPARK_IO.File_Type := SPARK_IO.Null_File;
   File_Status           : FileStatusT;

   Summary_Date_Time     : EStrings.T;

   DP_Generation_Date_Time_From_SDP_File : EStrings.T;

   Trimmed_Line          : EStrings.T;
   Current_DPC_Name      : EStrings.T;

   -------------------------------------------------------------------------
   -- NOTE, this procedure also removes the comma inserted in the string
   -- by the simplifier
   --
   procedure Extract_Dates_And_Times_From_Summary_DPC_File
     (Summary_DP_File          : in     SPARK_IO.File_Type;
      DPC_Generation_Date_Time :    out EStrings.T;
      SDP_Date_Time            :    out EStrings.T;
      File_Status              :    out FileStatusT)
   --# global in out SPARK_IO.File_sys;
   --# derives DPC_Generation_Date_Time,
   --#         File_Status,
   --#         SDP_Date_Time,
   --#         SPARK_IO.File_sys        from SPARK_IO.File_sys,
   --#                                       Summary_DP_File;
   is
      File_Line         : EStrings.T;
      Trimmed_Line      : EStrings.T;
      Gen_Date_Time     : EStrings.T;
      Simp_Date_Time    : EStrings.T;
      Subprogram_Found  : Boolean := False;

   begin -- ExtractDatesAndTimesFromSimplifiedVCFile

      File_Status := NotCorrupt;
      DPC_Generation_Date_Time := EStrings.Empty_String;
      SDP_Date_Time := EStrings.Empty_String;

      --Check for completly empty file.
      EStrings.Get_Line (File  => Summary_DP_File,
                         E_Str => File_Line);
      if EStrings.Eq1_String (E_Str => File_Line,
                              Str   => "") and SPARK_IO.End_Of_File (Summary_DP_File) then
         File_Status := CorruptEmptyFile;
      else
         --Keep on reading from this file, until the desired information is retrieved
         --or the end of the file is reached.
         loop
            Trimmed_Line := EStrings.Trim (File_Line);

            -- find date
            -- (There is an implicit assumption that the date, if present, will appear
            --  before the subprogram name.)
            -- When the Examiner is in plain_output mode, the DATE line doesn't appear.
            if EStrings.Eq1_String (E_Str => EStrings.Section (Trimmed_Line, 1, 7),
                                    Str   => "CREATED") then
               -- extract the VC generation date and time from the string
               Gen_Date_Time := EStrings.Section
                 (Trimmed_Line,
                  SDP_File_VC_Generation_Date_Start_Column,
                  SDP_File_VC_Generation_Date_Length);
               EStrings.Append_String (E_Str => Gen_Date_Time,
                                       Str   => " ");
               EStrings.Append_Examiner_String
                 (E_Str1 => Gen_Date_Time,
                  E_Str2 => EStrings.Section (Trimmed_Line,
                                              SDP_File_VC_Generation_Time_Start_Column,
                                              SDP_File_VC_Generation_Time_Length));
               DPC_Generation_Date_Time := Gen_Date_Time;

               -- extract the simplification date and time from the string
               Simp_Date_Time := EStrings.Section
                 (Trimmed_Line,
                  SDP_File_Simplification_Date_Start_Column,
                  SDP_File_Simplification_Date_Length);
               EStrings.Append_String (E_Str => Simp_Date_Time,
                                       Str   => " ");
               EStrings.Append_Examiner_String
                 (E_Str1 => Simp_Date_Time,
                  E_Str2 => EStrings.Section (Trimmed_Line,
                                              SDP_File_Simplification_Time_Start_Column,
                                              SDP_File_Simplification_Time_Length));

               SDP_Date_Time := Simp_Date_Time;
            end if;

            -- don't do anything with the subprogram name, but this brings us to the
            -- correct point in the file
            -- Match against the whole string plus space or newline - to guard against VC
            -- headers ('procedure_x.') being wrongly detected as the subprogram name.
            if (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                Start_Pos => 1,
                                                                Length    => 8),
                                     Str   => "FUNCTION") and then
                  (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                   Start_Pos => 9,
                                                                   Length    => 1),
                                        Str   => " ") or else
                     EStrings.Get_Length (E_Str => Trimmed_Line) = 8)) or else
              (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                               Start_Pos => 1,
                                                               Length    => 9),
                                    Str   => "PROCEDURE") and then
                 (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                  Start_Pos => 10,
                                                                  Length    => 1),
                                       Str   => " ") or else
                    EStrings.Get_Length (E_Str => Trimmed_Line) = 9)) or else
              (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                               Start_Pos => 1,
                                                               Length    => 4),
                                    Str   => "TASK") and then
                 (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                  Start_Pos => 5,
                                                                  Length    => 1),
                                       Str   => " ") or else
                    EStrings.Get_Length (E_Str => Trimmed_Line) = 4)) then
               Subprogram_Found := True;
            end if;

            exit when (Subprogram_Found or SPARK_IO.End_Of_File (Summary_DP_File));
            EStrings.Get_Line (File  => Summary_DP_File,
                               E_Str => File_Line);
         end loop;
      end if;

      if (File_Status = NotCorrupt) and not (Subprogram_Found) then
         File_Status := CorruptUnknownSubprogram;
      end if;

      -- if date has not been found must be in plain output mode
      -- The above is a false assumption -- the file may just be corrupt. However, the
      -- effect below of setting the string as unknown date is reasonable for both cases.
      if EStrings.Eq_String (DPC_Generation_Date_Time, EStrings.Empty_String) then
         EStrings.Append_String (E_Str => DPC_Generation_Date_Time,
                                 Str   => Unknown_DPC_Date);
         EStrings.Append_String (E_Str => SDP_Date_Time,
                                 Str   => Unknown_SDP_Date);
      end if;
   end Extract_Dates_And_Times_From_Summary_DPC_File;

   -------------------------------------------------------------------------
   -- look at the next non-blank line to see whether it starts
   -- "*** true". If so the VC has been discharged. Otherwise, increment
   -- the counter of undischarged VCs, and set the flag that an undischarged
   -- VC has been found
   procedure Process_New_Summary_DP_Line
     (Summary_DP_File  : in     SPARK_IO.File_Type;
      DPC_Name         : in     EStrings.T;
      Success          :    out Boolean)
   --# global in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out VCHeap.State;
   --# derives FatalErrors.State,
   --#         VCHeap.State      from *,
   --#                                Summary_DP_File,
   --#                                SPARK_IO.File_sys,
   --#                                VCHeap.State,
   --#                                DPC_Name &
   --#         SPARK_IO.File_sys,
   --#         Success           from Summary_DP_File,
   --#                                SPARK_IO.File_sys;
   is
      File_Line         : EStrings.T;
      Read_Line_Success : Boolean;

   begin -- ProcessNewSimplifiedVCLine
      ReadNextNonBlankLine (Summary_DP_File, Read_Line_Success, File_Line);
      if not Read_Line_Success then
         Success := False;
      else
         Success := True;
         if EStrings.Eq1_String
           (E_Str => EStrings.Section (File_Line, 1, 11),
            Str   => "*** No dead") then
            VCHeap.Set_DPC_State (DPC_Name, VCDetails.DPC_Live);
         elsif EStrings.Eq1_String
           (E_Str => EStrings.Section (File_Line, 1, 8),
            Str   => "*** Dead") then
            VCHeap.Set_DPC_State (DPC_Name, VCDetails.DPC_Dead);
         elsif EStrings.Eq1_String
         -- Case when Examiner determined that no dead path search
         -- is required for this DPC.
           (E_Str => EStrings.Section (File_Line, 1, 7),
            Str   => "*** DPC") then
            VCHeap.Set_DPC_State (DPC_Name, VCDetails.DPC_Unchecked);
         else
            Success := False;
         end if;
      end if;

   end Process_New_Summary_DP_Line;

   --------------------------------------------------------------------------

begin -- Analyse_Summary_DPC_File
      -- open SDP file
   ELStrings.Open (File         => Summary_DP_File,
                   Mode_Of_File => SPARK_IO.In_File,
                   Name_Of_File => File_Name,
                   Form_Of_File => "",
                   Status       => Open_Status);
   if Open_Status /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.Empty_String);
   end if;

   --No errors, until discover otherwise.
   Error_In_SDP_File := False;
   --FileError := EStrings.Empty_String;

   Extract_Dates_And_Times_From_Summary_DPC_File (Summary_DP_File,
                                                  DP_Generation_Date_Time_From_SDP_File,
                                                  Summary_Date_Time,
                                                  File_Status);
   case File_Status is
      when NotCorrupt =>
         null;
      when CorruptEmptyFile =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* SDP file corrupt: empty file ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         -- FileError := XMLSummary.XStr ("SDP file corrupt: empty file");
         Error_In_SDP_File := True;
      when CorruptUnknownSubprogram =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* SDP file corrupt: missing subprogram name ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         -- FileError := XMLSummary.XStr ("SDP file corrupt: missing subprogram name");
         Error_In_SDP_File := True;
   end case;

   --Only continue working on this file if an error has not been seen.
   --(Previously POGS would attempt to work with corrupt files. This feature has the
   -- capacity to produce confusing and wrong results.)
   if not (Error_In_SDP_File) then
      if CommandLine.Data.IgnoreDates or else
        EStrings.Eq_String (E_Str1 => DP_Generation_Date_Time_From_SDP_File,
                            E_Str2 => DPC_File_Date_Time) then

         if not CommandLine.Data.IgnoreDates and
            not CommandLine.Data.XML then
               SPARK_IO.New_Line (Report_File, 1);
               SPARK_IO.Put_String (Report_File, "DPC ZombieScoped ", 0);
               EStrings.Put_Line (File  => Report_File,
                                  E_Str => Summary_Date_Time);
         end if;

         -- find first non blank line
         -- if we get to the end of the file first, flag a fatal error
         ReadNextNonBlankLine (Summary_DP_File, Read_Line_Success, File_Line);

         if not Read_Line_Success then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                               "************* SDP file corrupt: no data beyond header ************", 0);
            SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
            Bad_File_Format := True;
            --FileError := XMLSummary.XStr ("SDP file corrupt: no data beyond header");
         else

            Finished_With_File := False;
            --FileError := EStrings.Empty_String;

            -- process file line-by-line
            -- on entry to the loop there is already a valid line in the
            -- FileLine buffer
            while not Finished_With_File loop
               -- examine line and act accordingly
               if IsNewRangeLine (File_Line) then
                  AppendNextLineFromFile (File_Line, Summary_DP_File);
               elsif IsNewVCLine (File_Line) then

                  Trimmed_Line := EStrings.Trim (File_Line);
                  Current_DPC_Name := EStrings.Section
                    (E_Str     => Trimmed_Line,
                     Start_Pos => 1,
                     Length    => EStrings.Get_Length (E_Str => Trimmed_Line) - 1);

                  Process_New_Summary_DP_Line
                    (Summary_DP_File,
                     Current_DPC_Name,
                     Process_Success);

                  if not Process_Success then
                     --# accept F, 41, "Expression is stable but cheap";
                     if not CommandLine.Data.XML then
                        SPARK_IO.Put_String
                          (Report_File,
                           "*** Warning: Bad format in summary DP file ***", 0);
                     end if;
                     --# end accept;
                     Finished_With_File := True;
                     Bad_File_Format := True;
                     --FileError := XMLSummary.XStr ("Warning: Bad format in summary DP file");
                  end if;
               end if;

               if not Finished_With_File then
                  -- read next line
                  ReadNextNonBlankLine (Summary_DP_File, Read_Line_Success, File_Line);

                  -- if unsuccessful then check EOF
                  -- and set FinishedWithFile accordingly
                  if not Read_Line_Success then
                     if SPARK_IO.End_Of_File (Summary_DP_File) then
                        Finished_With_File := True;
                     else
                        FatalErrors.Process (FatalErrors.ProblemReadingFile,
                                             ELStrings.Empty_String);
                     end if;
                  end if;
               end if;
            end loop;

         end if;

      else
         -- SIV file is out of date
         if not CommandLine.Data.XML then
            SPARK_IO.New_Line (Report_File, 1);
            SPARK_IO.Put_Line (Report_File,
                               "*** Warning: Summary DP file out of date ***",
                            0);
            SPARK_IO.Put_String (Report_File, "DPCs Generated: ", 0);
            EStrings.Put_String (File  => Report_File,
                                 E_Str => DPC_File_Date_Time);
            SPARK_IO.New_Line (Report_File, 1);

            SPARK_IO.Put_String (Report_File, "SDP File Date: ", 0);
            EStrings.Put_String (File  => Report_File,
                                 E_Str => DP_Generation_Date_Time_From_SDP_File);
            SPARK_IO.New_Line (Report_File, 1);
         end if;

         Bad_File_Format := True;
         --FileError := XMLSummary.XStr ("Warning: Summary DP file out of date");
      end if;
   end if;

   --# accept F, 10, Dummy_Close_Status, "DummyCloseStatus unused here" &
   --#        F, 33, Dummy_Close_Status, "DummyCloseStatus unused here" &
   --#        F, 10, Summary_DP_File,    "Summary_DP_File unused here";
   SPARK_IO.Close (Summary_DP_File, Dummy_Close_Status);

   --Either an error being raised, or a 'BadFileFormat'
   --being detected is an error.
   Error_In_SDP_File  := Error_In_SDP_File or Bad_File_Format;

end Analyse_Summary_DP_File;
