-- $Id: vcs-analysevcfile.adb 15481 2010-01-04 15:32:15Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .DPC file                                            --
-- At this time, the DPC file has the same format as a VCG file. Most of the  --
-- code has been copied across but the format may change over time and thus   --
-- justification of having a separate subprogram to analyse a DPC file.       --
--------------------------------------------------------------------------------
separate (VCS)
procedure Analyse_DPC_File
  (Report_File    : in     SPARK_IO.File_Type;
   File_Name      : in     ELStrings.T;
   Error_In_File  :    out Boolean;
   File_Date_Time :    out EStrings.T)
is
   Dummy_Close_Status : SPARK_IO.File_Status;
   Date_Time          : EStrings.T;
   Open_Status        : SPARK_IO.File_Status;
   File_Line          : EStrings.T;
   Read_Line_Success  : Boolean;
   DPC_File           : SPARK_IO.File_Type := SPARK_IO.Null_File;
   File_Status        : FileStatusT;
   VC_Info            : VCInfoType;
   Current_DPC_Name   : EStrings.T;
   Parsing_State      : ParsingStateType := Initial;
   Trimmed_Line       : EStrings.T;
   Finished_With_File : Boolean;

   ------------------------------------------------------------------------
   procedure Extract_DPC_File_Date_Time
     (DPC_File        : in     SPARK_IO.File_Type;
      File_Date_Time  :    out EStrings.T;
      File_Status     :    out FileStatusT)

   --# global in out SPARK_IO.File_sys;
   --# derives File_Date_Time,
   --#         File_Status,
   --#         SPARK_IO.File_Sys from DPC_File,
   --#                                SPARK_IO.File_sys;
   is
      File_Line         : EStrings.T;
      Trimmed_Line      : EStrings.T;
      Sub_Program_Found : Boolean := False;

   begin
      File_Status := NotCorrupt;
      File_Date_Time := EStrings.Empty_String;

      -- Check for completely empty file.
      EStrings.Get_Line (File  => DPC_File,
                         E_Str => File_Line);
      if EStrings.Eq1_String (E_Str => File_Line,
                              Str   => "") and SPARK_IO.End_Of_File (DPC_File) then
         File_Status := CorruptEmptyFile;
      else
         --Keep on reading from this file, until the desired information is retrieved
         --or the end of the file is reached.
         loop
            Trimmed_Line := EStrings.Trim (File_Line);

            -- find date, there is no need to find details of each dpc as an entry
            -- on the vcheap for each dpc (vc) has already been created.

            if EStrings.Eq1_String (E_Str => EStrings.Section (Trimmed_Line, 1, 4),
                                    Str   => "DATE") then
               File_Date_Time := EStrings.Section (Trimmed_Line,
                                                   DPC_File_Date_Time_Start_Column,
                                                   DPC_File_Date_Time_Length);
            end if;

            -- find and set SubProgramName
            -- once this has been found go on to analyse the rest of the file
            -- Match against the whole string plus space or newline - to guard against VC
            -- headers ('procedure_x.') being wrongly detected as the subprogram name.
            if (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                Start_Pos => 1,
                                                                Length    => 8),
                                     Str   => "FUNCTION") and then
                  (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                   Start_Pos => 9,
                                                                   Length    => 1),
                                        Str   => " ") or else
                     EStrings.Get_Length (E_Str => Trimmed_Line) = 8)) or else
              (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                               Start_Pos => 1,
                                                               Length    => 9),
                                    Str   => "PROCEDURE") and then
                 (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                  Start_Pos => 10,
                                                                  Length    => 1),
                                       Str   => " ") or else
                    EStrings.Get_Length (E_Str => Trimmed_Line) = 9)) or else
              (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                               Start_Pos => 1,
                                                               Length    => 4),
                                    Str   => "TASK") and then
                 (EStrings.Eq1_String (E_Str => EStrings.Section (E_Str     => Trimmed_Line,
                                                                  Start_Pos => 5,
                                                                  Length    => 1),
                                       Str   => " ") or else
                    EStrings.Get_Length (E_Str => Trimmed_Line) = 4)) then
               Sub_Program_Found := True;
            end if;
            exit when (Sub_Program_Found or SPARK_IO.End_Of_File (DPC_File));
            EStrings.Get_Line (File  => DPC_File,
                               E_Str => File_Line);
         end loop;
      end if;

      if EStrings.Eq_String (File_Date_Time, EStrings.Empty_String) then
         File_Date_Time := EStrings.Copy_String (Str => "Unknown Date (for dpc generation)");
      end if;

   end Extract_DPC_File_Date_Time;
   ---------------------------------------------------------------------------
   function Is_DPC_Error_Message (Line : EStrings.T)
                              return Boolean
   is
   begin -- Is_DPC_Error_Message
      return EStrings.Get_Length (E_Str => Line) > 0 and then
      EStrings.Get_Element (E_Str => Line,
                            Pos   => 1) = '!';
   end Is_DPC_Error_Message;
   ---------------------------------------------------------------------------
   function DPC_Is_New_Range_Line (Line : EStrings.T)
                            return Boolean
   is
   begin -- DPC_Is_New_Range_Line
      return EStrings.Eq1_String (E_Str => EStrings.Section (Line, 1, 17),
                                  Str   => "For path(s) from ") or
        EStrings.Eq1_String (E_Str => EStrings.Section (Line, 1, 14),
                             Str   => "For checks of ");
   end DPC_Is_New_Range_Line;
   ---------------------------------------------------------------------------
   function DPC_Is_New_VC_Line (Line : EStrings.T)
                         return Boolean
   is
      RetVal : Boolean;
   begin -- DPC_Is_New_VC_Line
      -- The shortest possible New VC Line is for a function that has
      -- a single letter identifier, followed by a full-stop e.g.
      --   function_g.
      -- which is 11 characters.
      if EStrings.Get_Length (E_Str => Line) >= 11 then
         RetVal :=
           EStrings.Eq1_String
             (E_Str => EStrings.Section (Line, 1, 10),
              Str   => "procedure_") or else
         EStrings.Eq1_String (E_Str => EStrings.Section (Line, 1, 9),
                              Str   => "function_") or else
         EStrings.Eq1_String (E_Str => EStrings.Section (Line, 1, 10),
                              Str   => "task_type_");
         if RetVal then
            for I in EStrings.Lengths range 9 .. EStrings.Get_Length (E_Str => Line) - 1 loop
               if not (Ada.Characters.Handling.Is_Alphanumeric
                       (EStrings.Get_Element (E_Str => Line,
                                              Pos   => I)) or else
                         EStrings.Get_Element (E_Str => Line,
                                               Pos   => I) = '_') then

                  RetVal := False;
                  exit;
               end if;
               --# assert I in 9 .. EStrings.Get_Length (E_Str => Line) - 1 and
               --#        Line = Line% and
               --#        EStrings.Get_Length (E_Str => Line) >= 11;
            end loop;

            if EStrings.Get_Element (E_Str => Line,
                                     Pos   => EStrings.Get_Length (E_Str => Line)) /= '.' then
               RetVal := False;
            end if;
         end if;
      else
         RetVal := False;
      end if;
      return RetVal;
   end DPC_Is_New_VC_Line;
   ---------------------------------------------------------------------------

   function Get_Line_Number (Line_Number : VCLineType)
                           return EStrings.T
   is
      Number         : Integer;
      Number_String  : String_10;
      Trimmed_Result : EStrings.T;
   begin -- Get_Line_Number

      if Line_Number = Refinement_Or_Inheritance_VC then
         Trimmed_Result := EStrings.Copy_String (Str => "     ");
      elsif Line_Number = VCLineStart then
         Trimmed_Result := EStrings.Copy_String (Str => "start");
      elsif Line_Number = VCLineEnd then
         Trimmed_Result := EStrings.Copy_String (Str => "finish");
      else
         Number := Line_Number;
         SPARK_IO.Put_Int_To_String (Number_String,
                                     Number,
                                     1,
                                     10);
         Trimmed_Result := EStrings.Trim
           (EStrings.Copy_String (Str => Number_String));
      end if;

      return Trimmed_Result;

   end Get_Line_Number;
   ---------------------------------------------------------------------------


begin -- Analyse_DPC_File
   -- open DPC file
   ELStrings.Open (File         => DPC_File,
                   Mode_Of_File => SPARK_IO.In_File,
                   Name_Of_File => File_Name,
                   Form_Of_File => "",
                   Status       => Open_Status);
   if Open_Status /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.Empty_String);
   end if;

   --No errors, until discover otherwise.
   Error_In_File := False;

   Extract_DPC_File_Date_Time (DPC_File, Date_Time, File_Status);

   --Report any error to standard out, store in XML summary structute, and
   --set error flag accordingly.
   --Note that XML summary structures seem to be generated regardless, even if XML is
   --not being used. (This is a little confusing and inefficient, but perfectly safe)
   case File_Status is
      when NotCorrupt =>
         null;
      when CorruptEmptyFile =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* DPC file corrupt: empty file ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         -- FileError := XMLSummary.XStr ("DPC file corrupt: empty file");
         Error_In_File := True;
      when CorruptUnknownSubprogram =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* DPC file corrupt: missing subprogram name ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         -- FileError := XMLSummary.XStr ("DPC file corrupt: missing subprogram name");
         Error_In_File := True;
   end case;

   --Record the date regardless of errors. This may be a string of the form 'no date'.
   File_Date_Time := Date_Time;

   if not (Error_In_File) then

      if not CommandLine.Data.XML then
         SPARK_IO.New_Line (Report_File, 1);
         SPARK_IO.Put_String (Report_File, "File ", 0);
         if CommandLine.Data.PlainOutput then
            ELStrings.Put_Line
              (File  => Report_File,
               E_Str => ELStrings.Lower_Case (E_Str => OSFiling.BaseFileName (File_Name)));
         else
            ELStrings.Put_Line (File  => Report_File,
                                E_Str => File_Name);
         end if;

         if CommandLine.Data.IgnoreDates then
            SPARK_IO.Put_Line (Report_File, "*** Warning: DPC date stamps ignored ***", 0);
         else
            SPARK_IO.Put_String (Report_File, "DPCs generated ", 0);
            EStrings.Put_Line (File  => Report_File,
                               E_Str => Date_Time);
         end if;
      end if;

      -- find first non blank line
      -- if we get to the end of the file first, flag a fatal error

      ReadNextNonBlankLine (DPC_File, Read_Line_Success, File_Line);

      if not Read_Line_Success then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* DPC file corrupt: no data beyond header ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         Error_In_File := True;
      else
         if Is_DPC_Error_Message (File_Line) then
            if not CommandLine.Data.XML then
               SPARK_IO.New_Line (Report_File, 1);
               EStrings.Put_String (File  => Report_File,
                                    E_Str => File_Line);
            end if;
            Error_In_File := True;
         else
            Error_In_File := False;
            --FileError := EStrings.Empty_String;

            -- initialize the 'current information' structure
            VC_Info := VCInfoType'(StartLine            => VCLineStart,
                                  EndLine              => VCLineEnd,
                                  EndLinePointType     => VCDetails.Undetermined_Point,
                                  NumberOfVCs          => 0,
                                  ThisStartLinePrinted => False,
                                  FileType             => StandardVCFileType,
                                  AnyVCsPrinted        => False,
                                  Valid                => False);

            Finished_With_File := False;

            -- process file line-by-line
            -- on entry to the loop there is already a valid line in the
            -- FileLine buffer
            while not Finished_With_File loop
               -- examine line and act accordingly
               if DPC_Is_New_Range_Line (File_Line) then
                  case Parsing_State is
                     when Initial =>
                        Parsing_State := FirstRange;
                     when FirstVCName =>
                        Parsing_State := NewRange;
                     when NewVCName =>
                        Parsing_State := NewRange;
                     when others =>
                        null;
                  end case;

                  AppendNextLineFromFile (File_Line, DPC_File);

                  ProcessNewRangeLine (File_Line, VC_Info);

               elsif DPC_Is_New_VC_Line (File_Line) then
                  case Parsing_State is
                     when FirstRange =>
                        -- Initialise VCHeap and store the first VC on the VCHeap
                        Trimmed_Line := EStrings.Trim (File_Line);
                        Current_DPC_Name := EStrings.Section
                          (E_Str     => Trimmed_Line,
                           Start_Pos => 1,
                           Length    => EStrings.Get_Length (E_Str => Trimmed_Line) - 1);

                        Parsing_State := FirstVCName;
                        if not VCHeap.Exists (Current_DPC_Name) then
                           VCHeap.Reinitialize (Current_DPC_Name,
                                                Get_Line_Number (VC_Info.StartLine),
                                                Get_Line_Number (VC_Info.EndLine),
                                                VC_Info.EndLinePointType);
                        end if;
                        VCHeap.Set_DPC_State (Current_DPC_Name,
                                              VCDetails.DPC_SDP_Not_Present);

                     when FirstVCName =>
                        Trimmed_Line := EStrings.Trim (File_Line);
                        Current_DPC_Name := EStrings.Section
                          (E_Str     => Trimmed_Line,
                           Start_Pos => 1,
                           Length    => EStrings.Get_Length (E_Str => Trimmed_Line) - 1);
                        Parsing_State := NewVCName;
                        if not VCHeap.Exists (Current_DPC_Name) then
                           VCHeap.Add (VCHeap.FirstEntry,
                                       Current_DPC_Name,
                                       Get_Line_Number (VC_Info.StartLine),
                                       Get_Line_Number (VC_Info.EndLine),
                                       VC_Info.EndLinePointType,
                                       VCDetails.VC_Not_Present,
                                       VCDetails.DPC_SDP_Not_Present);
                        else
                           VCHeap.Set_DPC_State (Current_DPC_Name,
                                                 VCDetails.DPC_SDP_Not_Present);
                        end if;
                     when NewRange =>
                        -- Store a new VC on the VC Heap
                        Trimmed_Line := EStrings.Trim (File_Line);
                        Current_DPC_Name := EStrings.Section
                          (E_Str     => Trimmed_Line,
                           Start_Pos => 1,
                           Length    => EStrings.Get_Length (E_Str => Trimmed_Line) - 1);
                        --SPARK_IO.Put_Line(ReportFile,"NewVCNameFound - New range",0);
                        Parsing_State := NewVCName;
                        if not VCHeap.Exists (Current_DPC_Name) then
                           VCHeap.Add (VCHeap.FirstEntry,
                                       Current_DPC_Name,
                                       Get_Line_Number (VC_Info.StartLine),
                                       Get_Line_Number (VC_Info.EndLine),
                                       VC_Info.EndLinePointType,
                                       VCDetails.VC_Not_Present,
                                       VCDetails.DPC_SDP_Not_Present);
                        else
                           VCHeap.Set_DPC_State (Current_DPC_Name,
                                                 VCDetails.DPC_SDP_Not_Present);
                        end if;
                     when NewVCName =>
                        -- The range has not changed, but store a new VC on the VC Heap
                        Trimmed_Line := EStrings.Trim (File_Line);
                        Current_DPC_Name := EStrings.Section
                          (E_Str     => Trimmed_Line,
                           Start_Pos => 1,
                           Length    => EStrings.Get_Length (E_Str => Trimmed_Line) - 1);
                        --SPARK_IO.Put_Line(ReportFile,"NewVCNameFound - Same range2",0);
                        Parsing_State := NewVCName;
                        if not VCHeap.Exists (Current_DPC_Name) then
                           VCHeap.Add (VCHeap.FirstEntry,
                                       Current_DPC_Name,
                                       Get_Line_Number (VC_Info.StartLine),
                                       Get_Line_Number (VC_Info.EndLine),
                                       VC_Info.EndLinePointType,
                                       VCDetails.VC_Not_Present,
                                       VCDetails.DPC_SDP_Not_Present);
                        else
                           VCHeap.Set_DPC_State (Current_DPC_Name,
                                                 VCDetails.DPC_SDP_Not_Present);
                        end if;
                     when others =>
                        null;
                  end case;
               end if;

               -- read next line
               ReadNextNonBlankLine (DPC_File, Read_Line_Success, File_Line);

               -- if unsuccessful then check EOF
               -- and set FinishedWithFile accordingly
               if not Read_Line_Success then
                  if SPARK_IO.End_Of_File (DPC_File) then
                     Finished_With_File := True;
                  else
                     FatalErrors.Process (FatalErrors.ProblemReadingFile, ELStrings.Empty_String);
                  end if;
               end if;
            end loop;


            -- write information for last VC
            -- two VCInfo parameters are necessary as WriteVCInfo compares them
            -- in deciding what to write (see definition of WriteVCInfo)
            if VC_Info.Valid then
               -- Reporting is now done as a table, so this has been commented out
               --WriteVCInfo( ReportFile, VCInfo, Dummy );
               null;
            else
               if not CommandLine.Data.XML then
                  SPARK_IO.Put_Line (Report_File, "No DPCs in file", 0);
               end if;
            end if;
         end if;

      end if;
   end if;

   --# accept F, 10, Dummy_Close_Status, "Dummy_Close_Status unused here" &
   --#        F, 33, Dummy_Close_Status, "Dummy_Close_Status unused here" &
   --#        F, 10, DPC_File,           "DPC_File unused here";
   SPARK_IO.Close (DPC_File, Dummy_Close_Status); -- Expect ineffective assigment

end Analyse_DPC_File;
