-- $Id: vcheap.ads 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package combining Heap and VCDetails to give an ordered list of VC details. --
--                                                                            --
--This is implemented as an abstract state machine. This is possible as there --
--is only a single occurrence of the VCHeap in the program.   It is necessary --
--to prevent unacceptable overheads of creating local copies of the embedded  --
--Heap and VCDetails   types to circumvent the entire variable rule. It would --
--also be possible to implement Heap and VCDetails   as ASMs but not          --
--necessarily desirable as this would affect the high level annotations of the--
--program.                                                                    --
--                                                                            --
--------------------------------------------------------------------------------
with EStrings;
with Heap;
with VCDetails;

use type Heap.Atom;

--# inherit ELStrings,
--#         EStrings,
--#         FatalErrors,
--#         Heap,
--#         HeapIndex,
--#         VCDetails;
package VCHeap
--# own I_State,
--#     State;
is

   -- StartIndex is a point in the linked list at which to start the
   -- search. This is used to start insertion at the first VC.
   -- If the name table is full, a fatal error is produced and Add does not
   -- return
   procedure Add (StartIndex         : in Heap.Atom;
                  NewName            : in EStrings.T;
                  PathStart          : in EStrings.T;
                  PathEnd            : in EStrings.T;
                  EndType            : in VCDetails.TerminalPointType;
                  VC_State           : in VCDetails.VC_State_T;
                  DPC_State          : in VCDetails.DPC_State_T
   );
   --# global in out FatalErrors.State;
   --#        in out I_State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                EndType,
   --#                                NewName,
   --#                                PathEnd,
   --#                                PathStart,
   --#                                VC_State,
   --#                                DPC_State,
   --#                                StartIndex,
   --#                                State &
   --#         I_State           from *,
   --#                                NewName,
   --#                                PathEnd,
   --#                                PathStart;

   --------------------------------------------------------------------------
   -- this procedure returns the VC details for the specified entry in the
   -- linked list.
   procedure Details (ListIndex          : in     Heap.Atom;
                      VCName             :    out EStrings.T;
                      PathStart          :    out EStrings.T;
                      PathEnd            :    out EStrings.T;
                      EndType            :    out VCDetails.TerminalPointType;
                      VC_State           :    out VCDetails.VC_State_T;
                      DPC_State          :    out VCDetails.DPC_State_T);
   --# global in State;
   --# derives EndType,
   --#         PathEnd,
   --#         PathStart,
   --#         VC_State,
   --#         DPC_State,
   --#         VCName                   from ListIndex,
   --#                                       State;

   --------------------------------------------------------------------------
   function FirstEntry return Heap.Atom;
   --# global in State;

   --------------------------------------------------------------------------
   procedure Initialize;
   --# global out I_State;
   --#        out State;
   --# derives I_State,
   --#         State   from ;

   --------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in VCDetails.ErrorType);
   --# global in out State;
   --# derives State from *,
   --#                    ErrorKind;

   --------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in VCDetails.ErrorType) return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   procedure Reinitialize (FirstElement   : in   EStrings.T;
                           FirstPathStart : in   EStrings.T;
                           FirstPathEnd   : in   EStrings.T;
                           FirstEndType   : in   VCDetails.TerminalPointType);
   --# global out I_State;
   --#        out State;
   --# derives I_State from FirstElement,
   --#                      FirstPathEnd,
   --#                      FirstPathStart &
   --#         State   from FirstElement,
   --#                      FirstEndType,
   --#                      FirstPathEnd,
   --#                      FirstPathStart;

   --------------------------------------------------------------------------
   -- this procedure returns the 'NextOne' ordered element in VCH after
   -- 'AfterThis'. It is successful if the NextOne is not a 'null' pointer
   procedure Next (AfterThis : in     Heap.Atom;
                   Success   :    out Boolean;
                   NextOne   :    out Heap.Atom);
   --# global in State;
   --# derives NextOne,
   --#         Success from AfterThis,
   --#                      State;

   --------------------------------------------------------------------------
   procedure Set_VC_State (VC_Name  : in EStrings.T;
                           VC_State : in VCDetails.VC_State_T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State from *,
   --#                                State,
   --#                                VC_Name &
   --#         State             from *,
   --#                                State,
   --#                                VC_Name,
   --#                                VC_State;

   function Get_VC_State (VC_Name  : in     EStrings.T) return VCDetails.VC_State_T;
   --# global in State;

   --------------------------------------------------------------------------
--     function Is_VC_State (VC_Name  : EStrings.T;
--                           VC_State : VCDetails.VC_State_T) return Boolean;
--     --# global in State;

   --------------------------------------------------------------------------
   procedure Set_DPC_State (DPC_Name  : in EStrings.T;
                            DPC_State : in VCDetails.DPC_State_T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State from *,
   --#                                State,
   --#                                DPC_Name &
   --#         State             from *,
   --#                                State,
   --#                                DPC_Name,
   --#                                DPC_State;

   --------------------------------------------------------------------------
   function Exists (VCName : EStrings.T) return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   procedure GetVCNameEndType
     (VCName : in     EStrings.T;
      VCType :    out VCDetails.TerminalPointType);
   --# global in     State;
   --#        in out FatalErrors.State;
   --# derives FatalErrors.State from *,
   --#                                State,
   --#                                VCName &
   --#         VCType            from State,
   --#                                VCName;

   --------------------------------------------------------------------------
   function GetLongestVCNameLength return Integer;
   --# global in I_State;

   --------------------------------------------------------------------------
   function GetLongestVCStartLength return Integer;
   --# global in I_State;

   --------------------------------------------------------------------------
   function GetLongestVCEndLength return Integer;
   --# global in I_State;

   --------------------------------------------------------------------------
   function GetVCNamePrefix return EStrings.T;
   --# global in I_State;

end VCHeap;
