-- $Id: vcheap.adb 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package combining Heap and VCDetails to give an ordered list of VC details. --
--                                                                            --
--------------------------------------------------------------------------------
with ELStrings,
     FatalErrors,
     HeapIndex;

use type HeapIndex.IndexType;

package body VCHeap
--# own State is ThePointers,
--#              TheDetails,
--#              StartOfPointersList &
--#     I_State is VCNamePrefix,
--#                LongestVCNameLength,
--#                LongestVCStartLength,
--#                LongestVCEndLength;
is

   ThePointers         : Heap.HeapRecord;
   TheDetails          : VCDetails.DataType;
   StartOfPointersList : Heap.Atom;

   -- this one is used to record the prefix name for the VCs
   VCNamePrefix         : EStrings.T;

   -- the following are used as VC are added to the heap to record information
   -- used later for displaying the results as a table
   LongestVCNameLength  : Integer;
   LongestVCStartLength : Integer;
   LongestVCEndLength   : Integer;


   function FirstEntry return Heap.Atom
   --# global in StartOfPointersList;
   is
   begin
      return StartOfPointersList;
   end FirstEntry;

   ------------------------------------------------------------------------
   procedure Add (StartIndex         : in Heap.Atom;
                  NewName            : in EStrings.T;
                  PathStart          : in EStrings.T;
                  PathEnd            : in EStrings.T;
                  EndType            : in VCDetails.TerminalPointType;
                  VC_State           : in VCDetails.VC_State_T;
                  DPC_State          : in VCDetails.DPC_State_T
)
   --# global in out FatalErrors.State;
   --#        in out LongestVCEndLength;
   --#        in out LongestVCNameLength;
   --#        in out LongestVCStartLength;
   --#        in out TheDetails;
   --#        in out ThePointers;
   --# derives FatalErrors.State,
   --#         TheDetails,
   --#         ThePointers          from *,
   --#                                   EndType,
   --#                                   NewName,
   --#                                   PathEnd,
   --#                                   PathStart,
   --#                                   VC_State,
   --#                                   DPC_State,
   --#                                   StartIndex,
   --#                                   TheDetails,
   --#                                   ThePointers &
   --#         LongestVCEndLength   from *,
   --#                                   PathEnd &
   --#         LongestVCNameLength  from *,
   --#                                   NewName &
   --#         LongestVCStartLength from *,
   --#                                   PathStart;
   is
      ExistingName       : EStrings.T;
      ExistingPathStart  : EStrings.T;
      ExistingPathEnd    : EStrings.T;
      ExistingEndType    : VCDetails.TerminalPointType;

      Existing_VC_State  : VCDetails.VC_State_T;
      Existing_DPC_State : VCDetails.DPC_State_T;

      ListIndex       : Heap.Atom;
      LoopFinished    : Boolean := False;
      NextEntryInList : Heap.Atom;
      OrderResult     : EStrings.Order_Types;
      RetrieveSuccess : Boolean;

      ------------------------------------------------------------------------
      procedure InsertInList
        (ListIndex          : in Heap.Atom;
         NextEntryInList    : in Heap.Atom;
         Name               : in EStrings.T;
         PathStart          : in EStrings.T;
         PathEnd            : in EStrings.T;
         EndType            : in VCDetails.TerminalPointType;
         VC_State           : in VCDetails.VC_State_T;
         DPC_State          : in VCDetails.DPC_State_T)
      --# global in out FatalErrors.State;
      --#        in out TheDetails;
      --#        in out ThePointers;
      --# derives FatalErrors.State from *,
      --#                                TheDetails,
      --#                                ThePointers &
      --#         TheDetails        from *,
      --#                                EndType,
      --#                                Name,
      --#                                PathEnd,
      --#                                PathStart,
      --#                                VC_State,
      --#                                DPC_State &
      --#         ThePointers       from *,
      --#                                ListIndex,
      --#                                NextEntryInList,
      --#                                TheDetails;
      is
         CreateAtomSuccess : Boolean;
         DetailsAddSuccess : Boolean;
         NewDetailsIndex   : HeapIndex.IndexType;
         NewPointersIndex  : Heap.Atom;
      begin -- InsertInList
         -- allocate heap atom
         Heap.CreateAtom (ThePointers,
                          NewPointersIndex,
                          CreateAtomSuccess);

         -- allocate file details entry
         VCDetails.Add (Details            => TheDetails,
                        Index              => NewDetailsIndex,
                        Success            => DetailsAddSuccess,
                        Name               => Name,
                        Path_Start         => PathStart,
                        Path_End           => PathEnd,
                        End_Type           => EndType,
                        VC_State           => VC_State,
                        DPC_State          => DPC_State);

         if not (CreateAtomSuccess and DetailsAddSuccess) then
            FatalErrors.Process (FatalErrors.VCHeapFull, ELStrings.Empty_String);
         end if;

         -- point heap atom to file details entry
         Heap.UpdateAValue (ThePointers, NewPointersIndex, NewDetailsIndex);

         -- link heap atom into list
         Heap.UpdateAPointer (ThePointers, ListIndex, NewPointersIndex);
         Heap.UpdateAPointer (ThePointers, NewPointersIndex, NextEntryInList);

      end InsertInList;

      -------------------------------------------------------------------------
      function LongestOf
        (First_Length : Integer; Second_Length : Integer) return Integer
      is
         Result : Integer;
      begin -- LongestOf
         if First_Length > Second_Length then
            Result := First_Length;
         else
            Result := Second_Length;
         end if;

         return Result;
      end LongestOf;

      -------------------------------------------------------------------------
   begin -- Add
      -- start at point specified in linked list
      ListIndex := StartIndex;

      while not LoopFinished loop
         -- if current item is last then add after it
         NextEntryInList := Heap.APointer (ThePointers, ListIndex);

         if NextEntryInList = 0 then
            InsertInList (ListIndex,
                          NextEntryInList,
                          NewName,
                          PathStart,
                          PathEnd,
                          EndType,
                          VC_State,
                          DPC_State);
            LoopFinished := True;
         else
            -- otherwise get relative order of next item in list and the new item
            --# accept F, 10, ExistingPathStart,  "ExistingPathStart not used here" &
            --#        F, 10, ExistingPathEnd,    "ExistingPathEnd not used here" &
            --#        F, 10, ExistingEndType,    "ExistingEndType not used here" &
            --#        F, 10, Existing_VC_State,  "Existing_VC_State not used here" &
            --#        F, 10, Existing_DPC_State, "Existing_DPC_State not used here" ;

            VCDetails.Retrieve (TheDetails,
                                Heap.AValue (ThePointers, NextEntryInList),
                                RetrieveSuccess,
                                ExistingName,
                                ExistingPathStart,
                                ExistingPathEnd,
                                ExistingEndType,
                                Existing_VC_State,
                                Existing_DPC_State);
            --# end accept;

            if not RetrieveSuccess then
               FatalErrors.Process (FatalErrors.VCDataStructureInconsistency,
                                    ELStrings.Empty_String);
            end if;

            VCDetails.Order (ExistingName,
                             NewName,
                             OrderResult);

            case OrderResult is
               when EStrings.First_One_First =>
                  -- next item in list is first, keep going down list
                  ListIndex := NextEntryInList;

               when EStrings.Second_One_First =>
                  -- new item is first, insert here
                  InsertInList (ListIndex,
                                NextEntryInList,
                                NewName,
                                PathStart,
                                PathEnd,
                                EndType,
                                VC_State,
                                DPC_State);
                  LoopFinished := True;

               when EStrings.Neither_First =>
                  -- items identical: do nothing
                  LoopFinished := True;

            end case;
         end if;
      end loop;

      LongestVCNameLength  := LongestOf (EStrings.Get_Length (E_Str => NewName), LongestVCNameLength);
      LongestVCStartLength := LongestOf (EStrings.Get_Length (E_Str => PathStart), LongestVCStartLength);

      LongestVCEndLength := LongestOf (EStrings.Get_Length (E_Str => PathEnd) +
                                       VCDetails.EndTypeImageLength,
                                       LongestVCEndLength);
      --# accept F, 33, ExistingPathStart, "ExistingPathStart not used here" &
      --#        F, 33, ExistingPathEnd,   "ExistingPathEnd not used here" &
      --#        F, 33, ExistingEndType,   "ExistingEndType not used here" &
      --#        F, 33, Existing_VC_State, "Existing_VC_State not used here" &
      --#        F, 33, Existing_DPC_State,"Existing_DPC_State not used here";
   end Add;

   ----------------------------------------------------------------------------
   -- this procedure returns the file details for the specified entry in the
   -- linked list.
   procedure Details (ListIndex          : in     Heap.Atom;
                      VCName             :    out EStrings.T;
                      PathStart          :    out EStrings.T;
                      PathEnd            :    out EStrings.T;
                      EndType            :    out VCDetails.TerminalPointType;
                      VC_State           :    out VCDetails.VC_State_T;
                      DPC_State          :    out VCDetails.DPC_State_T)
   --# global in TheDetails;
   --#        in ThePointers;
   --# derives EndType,
   --#         PathEnd,
   --#         PathStart,
   --#         VC_State,
   --#         DPC_State,
   --#         VCName                   from ListIndex,
   --#                                       TheDetails,
   --#                                       ThePointers;
   is
      DetailsIndex : HeapIndex.IndexType;
      Dummy        : Boolean;
   begin -- Details
      -- dereference linked list pointer
      DetailsIndex := Heap.AValue (ThePointers, ListIndex);

      -- if not null pointer then follow it
      if DetailsIndex /= 0 then
         --# accept F, 10, Dummy, "Dummy not used here";
         VCDetails.Retrieve (TheDetails,
                             DetailsIndex,
                             Dummy,
                             VCName,
                             PathStart,
                             PathEnd,
                             EndType,
                             VC_State,
                             DPC_State);
         --# end accept;
      else
         -- if null pointer then return failure
         VCName     := EStrings.Empty_String;
         PathStart  := EStrings.Empty_String;
         PathEnd    := EStrings.Empty_String;
         EndType    := VCDetails.Undetermined_Point;
         VC_State   := VCDetails.VC_Not_Present;
         DPC_State  := VCDetails.DPC_Not_Present;
      end if;
      --# accept F, 33, Dummy, "Dummy not used here";
   end Details;

   --------------------------------------------------------------------------
   procedure Initialize
   --# global out LongestVCEndLength;
   --#        out LongestVCNameLength;
   --#        out LongestVCStartLength;
   --#        out StartOfPointersList;
   --#        out TheDetails;
   --#        out ThePointers;
   --#        out VCNamePrefix;
   --# derives LongestVCEndLength,
   --#         LongestVCNameLength,
   --#         LongestVCStartLength,
   --#         StartOfPointersList,
   --#         TheDetails,
   --#         ThePointers,
   --#         VCNamePrefix         from ;
   is
   begin
      Heap.Initialize (ThePointers);
      VCDetails.Initialize (TheDetails);
      StartOfPointersList := 0;

      VCNamePrefix := EStrings.Empty_String;
      LongestVCNameLength  := 0;
      LongestVCStartLength := 0;
      LongestVCEndLength   := 0;
   end Initialize;

   --------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in VCDetails.ErrorType)
   --# global in out TheDetails;
   --# derives TheDetails from *,
   --#                         ErrorKind;
   is
   begin
      VCDetails.RaiseError (ErrorKind, TheDetails);
   end RaiseError;

   --------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in VCDetails.ErrorType) return Boolean
   --# global in TheDetails;
   is
   begin
      return VCDetails.ErrorRaised (ErrorKind, TheDetails);
   end ErrorRaised;

   --------------------------------------------------------------------------
   procedure Reinitialize
     (FirstElement   : in EStrings.T;
      FirstPathStart : in EStrings.T;
      FirstPathEnd   : in EStrings.T;
      FirstEndType   : in VCDetails.TerminalPointType)
   --# global out LongestVCEndLength;
   --#        out LongestVCNameLength;
   --#        out LongestVCStartLength;
   --#        out StartOfPointersList;
   --#        out TheDetails;
   --#        out ThePointers;
   --#        out VCNamePrefix;
   --# derives LongestVCEndLength   from FirstPathEnd &
   --#         LongestVCNameLength,
   --#         VCNamePrefix         from FirstElement &
   --#         LongestVCStartLength from FirstPathStart &
   --#         StartOfPointersList,
   --#         ThePointers          from  &
   --#         TheDetails           from FirstElement,
   --#                                   FirstEndType,
   --#                                   FirstPathEnd,
   --#                                   FirstPathStart;
   is
      Dummy              : Boolean;
      FirstDetailsIndex  : HeapIndex.IndexType;
      FirstPointersIndex : Heap.Atom;
   begin -- Reinitialize
      Heap.Initialize (ThePointers);
      VCDetails.Initialize (TheDetails);

      -- insert first item
      --# accept F, 10, Dummy, "Dummy unused here";
      VCDetails.Add (TheDetails,
                     FirstDetailsIndex,
                     Dummy,
                     FirstElement,
                     FirstPathStart,
                     FirstPathEnd,
                     FirstEndType,
                     VCDetails.VC_Not_Present,
                     VCDetails.DPC_Not_Present);
      Heap.CreateAtom (ThePointers,
                       FirstPointersIndex,
                       Dummy);
      --# end accept;

      Heap.UpdateAValue (ThePointers, FirstPointersIndex, FirstDetailsIndex);
      Heap.UpdateAPointer (ThePointers, FirstPointersIndex, 0);

      StartOfPointersList := FirstPointersIndex;

      LongestVCNameLength  := EStrings.Get_Length (E_Str => FirstElement);
      LongestVCStartLength := EStrings.Get_Length (E_Str => FirstPathStart);
      LongestVCEndLength   := EStrings.Get_Length (E_Str => FirstPathEnd) + VCDetails.EndTypeImageLength;

      VCNamePrefix := EStrings.Section
        (FirstElement, 1, EStrings.Get_Length (E_Str => FirstElement) - 2);
      --# accept F, 33, Dummy, "Dummy unused here";
   end Reinitialize;

   ---------------------------------------------------------------------------
   -- this procedure returns the 'NextOne' ordered element in FH after
   -- 'AfterThis'. It is successful if the NextOne is not a 'null' pointer
   procedure Next (AfterThis : in     Heap.Atom;
                   Success   :    out Boolean;
                   NextOne   :    out Heap.Atom)
   --# global in ThePointers;
   --# derives NextOne,
   --#         Success from AfterThis,
   --#                      ThePointers;
   is
      NextInList : Heap.Atom;
   begin -- Next
      NextInList := Heap.APointer (ThePointers, AfterThis);
      if NextInList = 0 then
         Success := False;
         NextOne := 0;
      else
         Success := True;
         NextOne := NextInList;
      end if;
   end Next;



   procedure FindVCByName
     (VCName  : in     EStrings.T;
      VCIndex :    out HeapIndex.IndexType)
   --# global in     StartOfPointersList;
   --#        in     TheDetails;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --# derives FatalErrors.State from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName &
   --#         VCIndex           from StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      ListIndex    : Heap.Atom;
      Found        : Boolean;
      LoopFinished : Boolean;
      RetrieveSuccess     : Boolean;
      CurrentVCName       : EStrings.T;
      CurrentVCPathStart  : EStrings.T;
      CurrentVCPathEnd    : EStrings.T;
      CurrentVCEndType    : VCDetails.TerminalPointType;
      Current_VC_State    : VCDetails.VC_State_T;
      Current_DPC_State   : VCDetails.DPC_State_T;
   begin
      ListIndex    := StartOfPointersList;
      Found        := False;
      LoopFinished := False;

      while not Heap.IsNullPointer (ListIndex) and not LoopFinished loop
         --# accept F, 10, CurrentVCPathStart, "CurrentVCPathStart not used here" &
         --#        F, 10, CurrentVCPathEnd,   "CurrentVCPathEnd not used here" &
         --#        F, 10, CurrentVCEndType,   "CurrentVCEndType not used here" &
         --#        F, 10, Current_VC_State,   "Current_VC_State not used here" &
         --#        F, 10, Current_DPC_State,  "Current_DPC_State not used here";
         VCDetails.Retrieve (TheDetails,
                             Heap.AValue (ThePointers, ListIndex),
                             RetrieveSuccess,
                             CurrentVCName,
                             CurrentVCPathStart,
                             CurrentVCPathEnd,
                             CurrentVCEndType,
                             Current_VC_State,
                             Current_DPC_State);
         --# end accept;
         if not RetrieveSuccess then
            FatalErrors.Process (FatalErrors.VCDataStructureInconsistency, ELStrings.Empty_String);
         end if;

         if EStrings.Eq_String (E_Str1 => VCName,
                                E_Str2 => CurrentVCName) then
            Found        := True;
            LoopFinished := True;
         else
            ListIndex := Heap.APointer (ThePointers, ListIndex);
         end if;
      end loop;

      if Found then
         VCIndex := Heap.AValue (ThePointers, ListIndex);
      else
         VCIndex := 0;
      end if;
      --# accept F, 33, CurrentVCPathStart, "CurrentVCPathStart not used here" &
      --#        F, 33, CurrentVCPathEnd,   "CurrentVCPathEnd not used here" &
      --#        F, 33, CurrentVCEndType,   "CurrentVCEndType not used here" &
      --#        F, 33, Current_VC_State,   "Current_VC_State not used here" &
      --#        F, 33, Current_DPC_State,  "Current_DPC_State not used here";

   end FindVCByName;

   --------------------------------------------------------------------------
   procedure Set_VC_State (VC_Name  : in EStrings.T;
                           VC_State : in VCDetails.VC_State_T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VC_Name &
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VC_Name,
   --#                                VC_State;
   is
      Details_Index : HeapIndex.IndexType;
   begin
      FindVCByName (VC_Name, Details_Index);

      if Details_Index /= 0 then
         VCDetails.Set_VC_State (TheDetails, Details_Index, VC_State);
      end if;
   end Set_VC_State;

   --------------------------------------------------------------------------
   function Get_VC_State (VC_Name  : EStrings.T) return VCDetails.VC_State_T
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide Get_VC_State
      Details_Index : HeapIndex.IndexType;
   begin
      FindVCByName (VC_Name, Details_Index);
      return VCDetails.Get_VC_State (TheDetails, Details_Index);
   end Get_VC_State;

   --------------------------------------------------------------------------
   procedure Set_DPC_State (DPC_Name  : in EStrings.T;
                            DPC_State : in VCDetails.DPC_State_T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                DPC_Name &
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                DPC_Name,
   --#                                DPC_State;
   is
      Details_Index : HeapIndex.IndexType;
   begin
      FindVCByName (DPC_Name, Details_Index);

      if Details_Index /= 0 then
         VCDetails.Set_DPC_State (TheDetails, Details_Index, DPC_State);
      end if;
   end Set_DPC_State;

   ---------------------------------------------------------------------------
   function Exists (VCName : EStrings.T) return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide Exists;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);
      return (DetailsIndex /= 0);
   end Exists;

   ---------------------------------------------------------------------------
   procedure GetVCNameEndType
     (VCName : in     EStrings.T;
      VCType :    out VCDetails.TerminalPointType)
   --# global in     StartOfPointersList;
   --#        in     TheDetails;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --# derives FatalErrors.State from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName &
   --#         VCType            from StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      VCIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, VCIndex);
      VCType := VCDetails.EndPointType (TheDetails, VCIndex);
   end GetVCNameEndType;

   ---------------------------------------------------------------------------
   function GetLongestVCNameLength return Integer
   --# global in LongestVCNameLength;
   is
   begin
      return LongestVCNameLength;
   end GetLongestVCNameLength;

   ---------------------------------------------------------------------------
   function GetLongestVCStartLength return Integer
   --# global in LongestVCStartLength;
   is
   begin
      return LongestVCStartLength;
   end GetLongestVCStartLength;

   ---------------------------------------------------------------------------
   function GetLongestVCEndLength return Integer
   --# global in LongestVCEndLength;
   is
   begin
      return LongestVCEndLength;
   end GetLongestVCEndLength;


   --------------------------------------------------------------------------
   function GetVCNamePrefix return EStrings.T
   --# global in VCNamePrefix;
   is
   begin
      return VCNamePrefix;
   end GetVCNamePrefix;

end VCHeap;
