-- $Id: vcdetails.ads 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package providing a structure in which to store VC details.                 --
--                                                                            --
--To be used in tandem with the Heap data structure, hence the use of         --
--Heap.Atom as the array range                                                --
--------------------------------------------------------------------------------
with EStrings;
with HeapIndex;

use type HeapIndex.IndexType;

--# inherit EStrings,
--#         HeapIndex;

package VCDetails
is
   type TerminalPointType is (Assert_Point,
                              Precondition_Check_Point,
                              Check_Statement_Point,
                              Runtime_Check_Point,
                              Refinement_VC_Point,
                              Inheritance_VC_Point,
                              Undetermined_Point);

   type ErrorType is (MissingSLGFile, Corrupt_VC_File, Corrupt_DPC_File);

   -- The possible states of a VC are:
   --   * VC_Not_Present: There is no VC (the user has only generated dpc
   --                     files);
   --   * VC_SIV_Not_Present: The Simplifier has not tried to discharge the VC.
   --   * VC_Undischarged: The VC is undischarged - the Simplifier has failed to
   --                      discharge the VC;
   --   * VC_Proved_By_Examiner: VC was discharged by the Examiner;
   --   * VC_Proved_By_Inference: VC was dischaged by the Simplifier and the
   --                             proof is by inference;
   --   * VC_Proved_By_Contradiction: VC was dischaged by the Simplifier
   --                                 and the proof is by contradiction;
   --   * VC_Proved_By_Checker: VC was discharged using the Checker;
   --   * VC_Proved_By_Review: VC was discharged through a proof review file;
   --   * VC_Proved_By_Using_Proof_Rules: VC was discharged using user proof
   --                                     rules.
   --   * VC_False: VC is false.

   type VC_State_T is  (VC_Not_Present, VC_SIV_Not_Present,
                        VC_Undischarged, VC_Proved_By_Examiner,
                        VC_Proved_By_Inference, VC_Proved_By_Contradiction,
                        VC_Proved_Using_User_Proof_Rules,
                        VC_Proved_By_Checker, VC_Proved_By_Review,
                        VC_False);

   -- The possible states of a DPC are:
   --   * DPC_Not_Present: There is no DPC (the user has only generated
   --                      VCG files).
   --   * DPC_SIV_Not_Present: No SDP is present for DPC.
   --   * DPC_Unchecked: The Examiner generated at DPC that does not required
   --                    ZombieScope to check for dead paths - e.g. RTCs
   --                    and asserts.
   --   * DPC_Live: Path is not a dead path.
   --   * DPC_Dead: Path is a dead path.

   type DPC_State_T is (DPC_Not_Present, DPC_SDP_Not_Present,
                        DPC_Unchecked, DPC_Live, DPC_Dead);

   EndTypeImageLength : constant := 12;
   subtype EndTypeImageIndex is Positive range 1 .. EndTypeImageLength;
   subtype EndTypeImageString is String (EndTypeImageIndex);
   type EndTypeToImageArray is array (TerminalPointType) of EndTypeImageString;

   EndTypeImage : constant EndTypeToImageArray :=
     EndTypeToImageArray'
        (Assert_Point             => "   assert @ ",
         Precondition_Check_Point => "pre check @ ",
         Check_Statement_Point    => "check stm @ ",
         Runtime_Check_Point      => "rtc check @ ",
         Refinement_VC_Point      => "refinement  ",
         Inheritance_VC_Point     => "inheritance ",
         Undetermined_Point       => "            ");


   type DataType is private;

   --------------------------------------------------------------------------
   procedure Add (Details                      : in out DataType;
                  Index                        :    out HeapIndex.IndexType;
                  Success                      :    out Boolean;
                  Name                         : in     EStrings.T;
                  Path_Start                   : in     EStrings.T;
                  Path_End                     : in     EStrings.T;
                  End_Type                     : in     TerminalPointType;
                  VC_State                     : in     VC_State_T;
                  DPC_State                    : in     DPC_State_T);
   --# derives Details from *,
   --#                      End_Type,
   --#                      Name,
   --#                      Path_End,
   --#                      Path_Start,
   --#                      VC_State,
   --#                      DPC_State &
   --#         Index,
   --#         Success from Details;

   --------------------------------------------------------------------------
   procedure Initialize (Details : out DataType);
   --# derives Details from ;

   --------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in ErrorType;
                         Details   : in out DataType);
   --# derives Details from *,
   --#                      ErrorKind;

   --------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in ErrorType;
                         Details   : in DataType) return Boolean;

   --------------------------------------------------------------------------
   procedure Set_VC_State (Details  : in out DataType;
                           Index    : in     HeapIndex.IndexType;
                           VC_State : in     VC_State_T);
   --# derives Details from *,
   --#                      Index,
   --#                      VC_State;

   --------------------------------------------------------------------------
   function Get_VC_State (Details : in DataType;
                          Index   : in HeapIndex.IndexType) return VC_State_T;

   --------------------------------------------------------------------------
   procedure Set_DPC_State (Details   : in out DataType;
                            Index     : in     HeapIndex.IndexType;
                            DPC_State : in     DPC_State_T);
   --# derives Details from *,
   --#                      Index,
   --#                      DPC_State;

   --------------------------------------------------------------------------
   function Get_DPC_State (Details   : in     DataType;
                           Index     : in     HeapIndex.IndexType) return DPC_State_T;

   --------------------------------------------------------------------------
   -- this compares the information given and returns
   -- Result is as defined in section 5.4 of the specification
   -- it works directly on values rather than on indices into the Details
   -- structure so that information can be compared before insertion
   -- NOTE : the procedure is successful iff neither Type is Invalid
   --
   procedure Order (FirstName  : in     EStrings.T;
                    SecondName : in     EStrings.T;
                    Result     :    out EStrings.Order_Types);
   --# derives Result from FirstName,
   --#                     SecondName;

   --------------------------------------------------------------------------
   procedure Retrieve (Details            : in     DataType;
                       Index              : in     HeapIndex.IndexType;
                       Success            :    out Boolean;
                       Name               :    out EStrings.T;
                       PathStart          :    out EStrings.T;
                       PathEnd            :    out EStrings.T;
                       EndType            :    out TerminalPointType;
                       VC_State           :    out VC_State_T;
                       DPC_State          :    out DPC_State_T);
   --# derives EndType,
   --#         Name,
   --#         PathEnd,
   --#         PathStart,
   --#         VC_State,
   --#         DPC_State,
   --#         Success                  from Details,
   --#                                       Index;

   --------------------------------------------------------------------------
   -- PathEndToPathType
   --
   -- Parses a Path End string to convert it into a Terminal Point Type
   -- The basic patterns scanned for are:
   --   ... to check ...              -> check statement point
   --   ... to run-time check ...     -> run-time check point
   --   ... to precondition check ... -> precondition check point
   --   ... to assertion ...          -> assertion point
   --   ... refinement ...            -> refinement VC point
   --   ... inheritance  ...          -> inheritance VC point
   --   ... anything else ...         -> undetermined point
   --------------------------------------------------------------------------
   function PathEndToPathType
     (Line : EStrings.T)
     return TerminalPointType;

   function EndPointType
     (Details : in DataType;
      Index   : in HeapIndex.IndexType)
     return TerminalPointType;


private

   type DetailsElement is record
      Name                : EStrings.T;
      PathStart           : EStrings.T;
      PathEnd             : EStrings.T;
      EndType             : TerminalPointType;
      VC_State            : VC_State_T;
      DPC_State           : DPC_State_T;
   end record;

   NullDetailsElement : constant DetailsElement :=
     DetailsElement'(Name               => EStrings.Empty_String,
                     PathStart          => EStrings.Empty_String,
                     PathEnd            => EStrings.Empty_String,
                     EndType            => Undetermined_Point,
                     VC_State           => VC_Not_Present,
                     DPC_State          => DPC_Not_Present);

   type ElementsArray is array (HeapIndex.IndexType) of DetailsElement;

   -- Use an array to store the different error kinds that may be associated
   -- with a subprogram. This construct may be extended by adding a new enumerated
   -- type, without having to extend the subprogram interfaces.
   type ErrorArray  is array (ErrorType) of Boolean;

   -- HighMark is the number of the highest allocated atom - zero if the
   -- structure is empty
   -- ErrorStatus records different kinds of errors that may be encountered
   -- during analysis.
   type DataType is record
      Details          : ElementsArray;
      HighMark         : HeapIndex.IndexType;
      ErrorStatus      : ErrorArray;
   end record;

end VCDetails;
