-- $Id: vcdetails.adb 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package providing a structure in which to store VC details.                 --
--                                                                            --
--To be used in tandem with the Heap data structure, hence the use of         --
--Heap.Atom as the array range                                                --
--------------------------------------------------------------------------------
package body VCDetails
is

   ---------------------------------------------------------------------------
   procedure Add (Details                      : in out DataType;
                  Index                        :    out HeapIndex.IndexType;
                  Success                      :    out Boolean;
                  Name                         : in     EStrings.T;
                  Path_Start                   : in     EStrings.T;
                  Path_End                     : in     EStrings.T;
                  End_Type                     : in     TerminalPointType;
                  VC_State                     : in     VC_State_T;
                  DPC_State                    : in     DPC_State_T)
   is
   begin
      if Details.HighMark < HeapIndex.IndexType'Last then
         Success := True;
         Details.HighMark := Details.HighMark + 1;
         Index := Details.HighMark;
         Details.Details (Details.HighMark) := DetailsElement'(Name,
                                                               Path_Start,
                                                               Path_End,
                                                               End_Type,
                                                               VC_State,
                                                               DPC_State);
      else
         Success := False;
         Index := 0;
      end if;
   end Add;

   --------------------------------------------------------------------------
   procedure Initialize (Details : out DataType)
   is
   begin
      -- Only set HighMark here.  Initializing the whole array is
      -- unnecessary here, and is VERY SLOW on machines with limited
      -- RAM likes VAXes, where the initialization causes massive
      -- VM thrashing.
      Details.HighMark := 0;
      -- Also set that no error has been seen.
      Details.ErrorStatus := ErrorArray'(others => False);

      --# accept F, 31,           Details.Details, "Partial initialization" &
      --#        F, 32,           Details.Details, "Partial initialization" &
      --#        F, 602, Details, Details.Details, "Partial initialization";
   end Initialize;

   -------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in ErrorType;
                         Details   : in out DataType)
   is
   begin
      Details.ErrorStatus (ErrorKind) := True;
   end RaiseError;

   -------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in ErrorType;
                         Details   : in DataType) return Boolean
   is
   begin
      return Details.ErrorStatus (ErrorKind);
   end ErrorRaised;

   --------------------------------------------------------------------------
   procedure Set_VC_State (Details  : in out DataType;
                           Index    : in     HeapIndex.IndexType;
                           VC_State : in     VC_State_T)
   is
   begin
      Details.Details (Index).VC_State := VC_State;
   end Set_VC_State;

   --------------------------------------------------------------------------
   function Get_VC_State (Details  : in     DataType;
                           Index   : in     HeapIndex.IndexType) return VC_State_T
   is
   begin
      return Details.Details (Index).VC_State;
   end Get_VC_State;

   --------------------------------------------------------------------------
   procedure Set_DPC_State (Details   : in out DataType;
                            Index     : in     HeapIndex.IndexType;
                            DPC_State : in     DPC_State_T)
   is
   begin
      Details.Details (Index).DPC_State := DPC_State;
   end Set_DPC_State;


   --------------------------------------------------------------------------
   function Get_DPC_State (Details   : in     DataType;
                            Index     : in     HeapIndex.IndexType) return DPC_State_T
   is
   begin
      return Details.Details (Index).DPC_State;
   end Get_DPC_State;

   -------------------------------------------------------------------------
   procedure Order (FirstName  : in     EStrings.T;
                    SecondName : in     EStrings.T;
                    Result     :    out EStrings.Order_Types)
   is
   begin -- Order
      -- check which name comes first
      if EStrings.Get_Length (E_Str => FirstName) =
        EStrings.Get_Length (E_Str => SecondName) then
         Result := EStrings.Lex_Order (First_Name  => FirstName,
                                       Second_Name => SecondName);
      elsif EStrings.Get_Length (E_Str => FirstName) <
        EStrings.Get_Length (E_Str => SecondName) then
         Result := EStrings.First_One_First;
      else
         Result := EStrings.Second_One_First;
      end if;
   end Order;

   --------------------------------------------------------------------------
   procedure Retrieve (Details            : in     DataType;
                       Index              : in     HeapIndex.IndexType;
                       Success            :    out Boolean;
                       Name               :    out EStrings.T;
                       PathStart          :    out EStrings.T;
                       PathEnd            :    out EStrings.T;
                       EndType            :    out TerminalPointType;
                       VC_State           :    out VC_State_T;
                       DPC_State          :    out DPC_State_T)
   is
   begin
      if Index <= Details.HighMark and Index /= 0 then
         Success                  := True;
         Name                     := Details.Details (Index).Name;
         PathStart                := Details.Details (Index).PathStart;
         PathEnd                  := Details.Details (Index).PathEnd;
         EndType                  := Details.Details (Index).EndType;
         VC_State                 := Details.Details (Index).VC_State;
         DPC_State                := Details.Details (Index).DPC_State;
      else
         Success                  := False;
         Name                     := EStrings.Empty_String;
         PathStart                := EStrings.Empty_String;
         PathEnd                  := EStrings.Empty_String;
         EndType                  := Undetermined_Point;
         VC_State                 := VC_Not_Present;
         DPC_State                := DPC_Not_Present;
      end if;
   end Retrieve;

   --------------------------------------------------------------------------
   function PathEndToPathType (Line : EStrings.T)
                              return TerminalPointType
   is
      DummyPosition    : EStrings.Positions;
      EndPosition      : EStrings.Positions;
      PointType        : TerminalPointType;
      RefinementFound  : Boolean;
      InheritanceFound : Boolean;
      ToFound          : Boolean;
      CheckFound       : Boolean;
      AssertFound      : Boolean;
      FinishFound      : Boolean;

      RuntimeCheckFound        : Boolean;
      PreconditionCheckFound   : Boolean;
   begin -- ExtractLastLinePointType

      --# accept F, 10, DummyPosition, "DummyPosition unused here";
      EStrings.Find_Sub_String (E_Str         => Line,
                                Search_String => "inheritance",
                                String_Found  => InheritanceFound,
                                String_Start  => DummyPosition);

      EStrings.Find_Sub_String (E_Str         => Line,
                                Search_String => "refinement",
                                String_Found  => RefinementFound,
                                String_Start  => DummyPosition);

      --# accept F, 10, ToFound, "ToFound unused here";
      EStrings.Find_Sub_String (E_Str         => Line,
                                Search_String => " to ",
                                String_Found  => ToFound,
                                String_Start  => EndPosition);
      --# end accept;

      EStrings.Find_Sub_String_After (E_Str         => Line,
                                      Search_Start  => EndPosition,
                                      Search_String => "check",
                                      String_Found  => CheckFound,
                                      String_Start  => DummyPosition);

      if InheritanceFound then

         PointType := Inheritance_VC_Point;

      elsif RefinementFound then

         PointType := Refinement_VC_Point;

      elsif CheckFound then
         EStrings.Find_Sub_String_After (E_Str         => Line,
                                         Search_Start  => EndPosition,
                                         Search_String => "precondition",
                                         String_Found  => PreconditionCheckFound,
                                         String_Start  => DummyPosition);

         EStrings.Find_Sub_String_After (E_Str         => Line,
                                         Search_Start  => EndPosition,
                                         Search_String => "run-time",
                                         String_Found  => RuntimeCheckFound,
                                         String_Start  => DummyPosition);

         if PreconditionCheckFound then
            PointType := Precondition_Check_Point;
         elsif RuntimeCheckFound then
            PointType := Runtime_Check_Point;
         else
            PointType := Check_Statement_Point;
         end if;

      else
         EStrings.Find_Sub_String_After (E_Str         => Line,
                                         Search_Start  => EndPosition,
                                         Search_String => "assert",
                                         String_Found  => AssertFound,
                                         String_Start  => DummyPosition);

         EStrings.Find_Sub_String_After (E_Str         => Line,
                                         Search_Start  => EndPosition,
                                         Search_String => "finish",
                                         String_Found  => FinishFound,
                                         String_Start  => DummyPosition);

         if AssertFound or FinishFound then
            PointType := Assert_Point;
         else
            PointType := Undetermined_Point;
         end if;
      end if;
      --# end accept;

      --# accept F, 33, DummyPosition,  "DummyPosition unused here" &
      --#        F, 33, ToFound,        "ToFound unused here";
      return PointType;
   end PathEndToPathType;


   --------------------------------------------------------------------------
   function EndPointType
     (Details : in DataType;
      Index   : in HeapIndex.IndexType)
     return TerminalPointType
   is
      Result : TerminalPointType;
   begin
      if Index <= Details.HighMark and Index /= 0 then

         Result := Details.Details (Index).EndType;
      else
         Result := Undetermined_Point;
      end if;

      return Result;
   end EndPointType;


end VCDetails;
