-- $Id: total.adb 16001 2010-02-09 09:18:07Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package providing data structure to store running totals, and a procedure   --
--to print them.                                                              --
--                                                                            --
--------------------------------------------------------------------------------
with Banner,
     CommandLine,
     ELStrings,
     EStrings,
     FatalErrors,
     Heap,
     VCHeap,
     VCDetails,
     XMLSummary;

use type VCDetails.VC_State_T;
use type VCDetails.DPC_State_T;

package body Total
--# own State is TheTotals;
is

   type VCCounter is array (VCDetails.TerminalPointType) of Natural;

   NullVCCounter : constant VCCounter :=
     VCCounter'(others => 0);

   -- if adding more entries to this type, you must alter the procedure
   -- Output below to display the results, and add code in the appropriate
   -- analysis routines to increment the counter(s). You must also alter
   -- the aggregate in Initialize below to initialize the new counter
   type TotalType is record

      -- Subprograms without errors (all subprograms have vcs).
      SubprogramsWithVCs : Natural;

      -- Subprograms with errors.
      SubprogramsWhereMissingSLGFile           : Natural;
      Subprograms_Where_VC_Analysis_Abandoned  : Natural;
      Subprograms_Where_DPC_Analysis_Abandoned : Natural;

      -- A subprogram with At Least One (ALO) Undischarged VC.
      SubprogramsWithALOUndischargedVC : Natural;

      -- The following fields record the number of subprograms,
      -- which are not necessarily fully proved, but have at least
      -- one VC proved by each of the following strategies:

      -- A subprogram with At Least One (ALO) VC proved by
      -- the Examiner.
      SubprogramsWithALOExaminerVC      : Natural;

      -- A subprogram with At Least One (ALO) VC proved by the
      -- simplifier without a user defined rule.
      SubprogramsWithALOSimplifierVC    : Natural;

      -- A subprogram with At Least One (ALO) VC proved using
      -- a using proof by contradiction.
      SubprogramsWithALOContradictionVC : Natural;

      -- A subprogram with At Least One (ALO) VC proved using
      -- a user defined rule.
      SubprogramsWithALOUserRuleVC      : Natural;

      -- A subprogram with At Least One (ALO) VC proved using
      -- the Checker.
      SubprogramsWithALOCheckerVC       : Natural;

      -- A subprogram with At Least One (ALO) VC discharged by review
      SubprogramsWithALOReviewVC        : Natural;

      -- A subprogram with At Least One (ALO) false VC.
      SubprogramsWithALOFalseVC         : Natural;

      -- The following fields represent the use of a hierachy of
      -- proof strategies:
      --   Examiner -> Simplifier -> User Rules -> Checker -> Review
      -- When a subprogram is proved, the strategy latest in the hierarchy
      -- is considered to have been used to complete the proof, even
      -- if earlier strategies have also been applied.
      -- The hierachy gives a measure of the extent of the strategies
      -- required in proving a subprogram.
      -- The definitions of the hierarchical strategies is given below:
      -- A subprogram proof is completed by the examiner if:
      --    o at LEAST one VC was proved by the examiner and
      --    o ZERO VCs were proved by the simplifier,
      --      checker, review file or left Undischarged and
      --    o the subprogram has no false VCs.
      SubprogramsProvedByExaminer        : Natural;

      -- A subprogram proof is completed by the simplifier if:
      --    o at LEAST one VC was proved by the simplifier and
      --    o ZERO VCs were proved by the checker, review file
      --      or left Undischarged and
      --    o the subprogram has no false VCs.
      SubprogramsProvedBySimplifier      : Natural;

      -- A subprogram proof is completed by a user defined proof rule if:
      --    o at least one VC was proved by the simplifier and
      --    o at least one user defined rule has been used in the proof of a VC and
      --    o ZERO VCs were proved by the checker, review file
      --      or left Undischarged and
      --    o the subprogram has no false VCs.
      SubprogramsProvedWithUserProofRule : Natural;

      -- A subprogram proof is completed by the checker if:
      --    o at LEAST one VC was proved by the checker and
      --    o ZERO VCs were proved by the review file or
      --      left Undischarged and
      --    o the subprogram has no false VCs.
      SubprogramsProvedByChecker         : Natural;

      -- A subprogram proof is completed by review if:
      --    o at LEAST one VC was proved in the review file and
      --    o ZERO VCs were left Undischarged and
      --    o the subprogram has no false VCs.
      SubprogramsProvedByReview        : Natural;

      -- The following fields record the number of VCs proved by
      -- each strategy grouped by the sort of origin (the terminal point)
      -- of the VC.
      VCsTotal                         : VCCounter;
      VCsProvedByExaminer              : VCCounter;
      VCsProvedBySimplifier            : VCCounter;
      VCsProvedByChecker               : VCCounter;
      VCsProvedWithUserProofRule       : VCCounter;
      VCsProvedByReview                : VCCounter;
      VCsProvedFalse                   : VCCounter;
      VCsUndischarged                  : VCCounter;

      -- Record number of subprograms with DPCs
      Subprograms_With_DPCs            : Natural;

      -- The following fields record the number of dead paths ZombieScope
      -- has found.
      Number_Of_Dead_Paths             : Natural;
      Subprograms_With_Dead_Paths      : Natural;
   end record;

   TheTotals : TotalType;

   function Sum (T : in VCCounter) return Natural
   is
      Result : Natural := 0;
   begin
      for I in VCDetails.TerminalPointType loop
         Result := Result + T (I);
      end loop;
      return Result;
   end Sum;

   -- The calculation of percentages is factored out, to keep the
   -- standard and XML summary output consistent.
   procedure CalcuatePercentages (TheTotals                  : in TotalType;
                                  PercentUndischargedStr     : out EStrings.T;
                                  PercentProvedByExaminerStr : out EStrings.T;
                                  PercentProvedByCheckerStr  : out EStrings.T;
                                  PercentProvedByReviewStr   : out EStrings.T;
                                  PercentSimplifiedStr       : out EStrings.T;
                                  PercentWithUserRuleStr     : out EStrings.T;
                                  PercentProvedFalseStr      : out EStrings.T)
   --#derives PercentUndischargedStr,
   --#        PercentProvedByExaminerStr,
   --#        PercentProvedByCheckerStr,
   --#        PercentProvedByReviewStr,
   --#        PercentSimplifiedStr,
   --#        PercentWithUserRuleStr,
   --#        PercentProvedFalseStr from TheTotals;
   is
      subtype Percentage is Natural range 0 .. 100;

      VCsTotal : Natural;

      PercentUndischarged      : Percentage;
      PercentUndischargedC     : Character;
      PercentProvedByExaminer  : Percentage;
      PercentProvedByExaminerC : Character;
      PercentProvedByChecker   : Percentage;
      PercentProvedByCheckerC  : Character;
      PercentProvedByReview    : Percentage;
      PercentProvedByReviewC   : Character;
      PercentSimplified        : Percentage;
      PercentSimplifiedC       : Character;
      PercentWithUserRule      : Percentage;
      PercentWithUserRuleC     : Character;
      PercentProvedFalse       : Percentage;
      PercentProvedFalseC      : Character;

      procedure TotalAndValueToPercentage (OverallTotal : in  Natural;
                                           Value        : in  Natural;
                                           Percent      : out Percentage;
                                           PercentC     : out Character)
      --# derives Percent, PercentC from OverallTotal, Value;
      --# pre (OverallTotal/=0);
      is
         PercisePercentValue : Float;
         RoundedPercentValue : Percentage;
      begin
         PercisePercentValue := (Float (Value)  * 100.0) / Float (OverallTotal);
         RoundedPercentValue := Percentage (PercisePercentValue);

         case RoundedPercentValue is

            -- If the rounded percentage value is zero, but the actual
            -- value is non-zero, then the rounded value is forced to be 1.
            -- This behaviour ensures that a zero percentage really means
            -- zero.
            when 0 =>
               if (Value /= 0) then
                  Percent := 1;

                  -- If the actual percent is less than 0.5 then this is
                  -- indicated with an appropriate leading character.
                  if (PercisePercentValue < 0.5) then
                     PercentC := '<';
                  else
                     PercentC := ' ';
                  end if;
               else
                  Percent := 0;
                  PercentC := ' ';
               end if;

               -- If the rounded percentage value is 100, but the actual value
               -- is not equal to the total, then the rounded value is forced
               -- to be 99. This behaviour ensures that a hundred percent
               -- really means all.
            when 100 =>
               if (Value /= OverallTotal) then
                  Percent := 99;

                  -- If the actual percent is greater than 99.5 then this is
                  -- indicated with an appropriate leading character.
                  if (PercisePercentValue > 99.5) then
                     PercentC := '>';
                  else
                     PercentC := ' ';
                  end if;
               else
                  Percent := 100;
                  PercentC := ' ';
               end if;

               -- In all other cases, accept the rounding approximation.
            when 1 .. 99 =>
               Percent := RoundedPercentValue;
               PercentC := ' ';
         end case;
      end TotalAndValueToPercentage;

      procedure GeneratePercentString (Percent    : in Percentage;
                                       PercentC   : in Character;
                                       PercentStr : out EStrings.T)
      --# derives PercentStr from Percent, PercentC;
      is
         subtype StringLength is Integer range 1 .. 10;
         subtype TempString is String (StringLength);
         PercentPart : TempString;
         DummySuccess : Boolean;
      begin
         --Initialise to empty string.
         PercentStr := EStrings.Empty_String;

         --# accept F, 10, DummySuccess, "Error status ignored as number always fits.";
         --# accept F, 33, DummySuccess, "Error status ignored as number always fits.";

         --For alingment: if percent is one digit, add two spaces.
         --             : if percent is two digits, add one spaces.
         case Percent is
            when 0 .. 9 =>
               EStrings.Append_String (E_Str => PercentStr,
                                       Str   => "  ");
            when 10 .. 99 =>
               EStrings.Append_String (E_Str => PercentStr,
                                       Str   => " ");
            when 100 =>
               null;
         end case;

         --Append the: '>','<',' ' prefix. (max length "X": 1)
         EStrings.Append_Char (E_Str   => PercentStr,
                               Ch      => PercentC,
                               Success => DummySuccess);

         --Append the: percent number. (max length "XYYY": 4)
         SPARK_IO.Put_Int_To_String (PercentPart, Percent, 1, 10);
         EStrings.Append_Examiner_String
           (E_Str1 => PercentStr,
            E_Str2 => EStrings.Trim
              (EStrings.Copy_String (Str => PercentPart)));

         --Append the: symbol '%'. (max length "XYYY%": 5)
         EStrings.Append_Char (E_Str   => PercentStr,
                               Ch      => '%',
                               Success => DummySuccess);

      end GeneratePercentString;

   begin

      VCsTotal := Sum (TheTotals.VCsTotal);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsUndischarged),
                                 PercentUndischarged,
                                 PercentUndischargedC);
      GeneratePercentString (PercentUndischarged,
                             PercentUndischargedC,
                             PercentUndischargedStr);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsProvedByExaminer),
                                 PercentProvedByExaminer,
                                 PercentProvedByExaminerC);
      GeneratePercentString (PercentProvedByExaminer,
                             PercentProvedByExaminerC,
                             PercentProvedByExaminerStr);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsProvedByChecker),
                                 PercentProvedByChecker,
                                 PercentProvedByCheckerC);
      GeneratePercentString (PercentProvedByChecker,
                             PercentProvedByCheckerC,
                             PercentProvedByCheckerStr);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsProvedByReview),
                                 PercentProvedByReview,
                                 PercentProvedByReviewC);
      GeneratePercentString (PercentProvedByReview,
                             PercentProvedByReviewC,
                             PercentProvedByReviewStr);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsProvedBySimplifier),
                                 PercentSimplified,
                                 PercentSimplifiedC);
      GeneratePercentString (PercentSimplified,
                             PercentSimplifiedC,
                             PercentSimplifiedStr);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsProvedWithUserProofRule),
                                 PercentWithUserRule,
                                 PercentWithUserRuleC);
      GeneratePercentString (PercentWithUserRule,
                             PercentWithUserRuleC,
                             PercentWithUserRuleStr);

      TotalAndValueToPercentage (VCsTotal,
                                 Sum (TheTotals.VCsProvedFalse),
                                 PercentProvedFalse,
                                 PercentProvedFalseC);
      GeneratePercentString (PercentProvedFalse,
                             PercentProvedFalseC,
                             PercentProvedFalseStr);
   end CalcuatePercentages;

   -- Never returns from this subprogram.
   -- Null dependency relation used to avoid propagation
   -- of FatalErrors.State impacting existing clients of Total.
   -- FatalErrors.State is of little interest in this context.
   procedure FatalError (Error : in FatalErrors.ErrorType)
   --# derives null from Error;
   is
      --# hide FatalError;
   begin
      FatalErrors.Process (Error, ELStrings.Empty_String);
   end FatalError;

   function TotalsAreBalanced return Boolean
   --# global in TheTotals;
   is
      TotalSubprogramsProved                        : Natural;
      TotalSubprogramsAtLeastOneFalseVC             : Natural;
      TotalSubprogramsNoFalseAtLeastOneUndischarged : Natural;
      TotalSubprograms                              : Natural;
   begin
      -- Total all of the subprogram that have been fully proved.
      TotalSubprogramsProved := ((((TheTotals.SubprogramsProvedByExaminer +
          TheTotals.SubprogramsProvedBySimplifier) +
            TheTotals.SubprogramsProvedWithUserProofRule) +
              TheTotals.SubprogramsProvedByChecker) +
                TheTotals.SubprogramsProvedByReview);

      -- Total all of the subprogram that have at least one false vc.
      TotalSubprogramsAtLeastOneFalseVC := TheTotals.SubprogramsWithALOFalseVC;

      -- Total all of the subprogram that have no false vcs, and at least one
      -- undischarged vc.
      TotalSubprogramsNoFalseAtLeastOneUndischarged := TheTotals.SubprogramsWithALOUndischargedVC;

      -- Total all of the subprograms.
      TotalSubprograms := ((TotalSubprogramsProved +
          TotalSubprogramsAtLeastOneFalseVC) +
            TotalSubprogramsNoFalseAtLeastOneUndischarged);

      -- Return true if the total matches the recorded total number of Subprograms
      -- with VCs and false otherwise.
      return (TotalSubprograms = TheTotals.SubprogramsWithVCs);
   end TotalsAreBalanced;

   procedure UpdateTotals (VCG : in Boolean; DPC : in Boolean)
   --# global in     VCHeap.State;
   --#        in out TheTotals;
   --# derives TheTotals from *,
   --#                        VCG,
   --#                        DPC,
   --#                        VCHeap.State;
   is
      SubProgramIsUndischarged               : Boolean;
      SubProgramHasVCProvedByExaminer        : Boolean;
      SubProgramHasVCProvedBySimplifier      : Boolean;
      SubProgramHasVCProvedByContradiction   : Boolean;
      SubProgramHasVCProvedWithUserProofRule : Boolean;
      SubProgramHasVCProvedByChecker         : Boolean;
      SubProgramHasVCProvedByReview          : Boolean;
      SubProgramHasVCProvedFalse             : Boolean;
      Subprogram_Contains_Dead_Paths         : Boolean;
      MoreVCs : Boolean;

      HeapIndex : Heap.Atom;
      NextIndex : Heap.Atom;

      UnusedVCName       : EStrings.T;
      UnusedPathStart    : EStrings.T;
      UnusedPathEnd      : EStrings.T;
      EndType            : VCDetails.TerminalPointType;

      VC_State  : VCDetails.VC_State_T;
      DPC_State : VCDetails.DPC_State_T;

   begin

      --Always count missing SLG files.
      if VCHeap.ErrorRaised (VCDetails.MissingSLGFile) then
         TheTotals.SubprogramsWhereMissingSLGFile := TheTotals.SubprogramsWhereMissingSLGFile + 1;
      end if;

      --Only perform detailed analysis of subprogram if it did not have an
      --associated corrupt file.
      if VCHeap.ErrorRaised (VCDetails.Corrupt_VC_File) then
         TheTotals.Subprograms_Where_VC_Analysis_Abandoned :=
           TheTotals.Subprograms_Where_VC_Analysis_Abandoned + 1;
      end if;

      if VCHeap.ErrorRaised (VCDetails.Corrupt_DPC_File) then
         TheTotals.Subprograms_Where_DPC_Analysis_Abandoned :=
           TheTotals.Subprograms_Where_DPC_Analysis_Abandoned + 1;
      end if;

      MoreVCs := True;
      HeapIndex := VCHeap.FirstEntry;

      SubProgramIsUndischarged := False;
      SubProgramHasVCProvedByExaminer := False;
      SubProgramHasVCProvedBySimplifier := False;
      SubProgramHasVCProvedByContradiction := False;
      SubProgramHasVCProvedByChecker := False;
      SubProgramHasVCProvedWithUserProofRule := False;
      SubProgramHasVCProvedByReview := False;
      SubProgramHasVCProvedFalse := False;
      Subprogram_Contains_Dead_Paths := False;

      if VCG and (not VCHeap.ErrorRaised (VCDetails.Corrupt_VC_File)) then
         TheTotals.SubprogramsWithVCs := TheTotals.SubprogramsWithVCs + 1;
      end if;

      if DPC and (not VCHeap.ErrorRaised (VCDetails.Corrupt_DPC_File)) then
         TheTotals.Subprograms_With_DPCs := TheTotals.Subprograms_With_DPCs + 1;
      end if;

      while MoreVCs and not Heap.IsNullPointer (HeapIndex) loop

         -- Get the details for the next VC.
         --# accept F, 10, UnusedPathEnd,            "UnusedPathEnd unused here" &
         --#        F, 10, UnusedPathStart,          "UnusedPathStart unused here" &
         --#        F, 10, UnusedVCName,             "UnusedPVCName unused here" ;
         VCHeap.Details (ListIndex => HeapIndex,
                         VCName => UnusedVCName,
                         PathStart => UnusedPathStart,
                         PathEnd => UnusedPathEnd,
                         EndType => EndType,
                         VC_State => VC_State,
                         DPC_State => DPC_State);

         --# end accept;

         --# accept F, 41, "Expression is stable";
         if VCG and (not VCHeap.ErrorRaised (VCDetails.Corrupt_VC_File)) then
            --# end accept;
            TheTotals.VCsTotal (EndType) := TheTotals.VCsTotal (EndType) + 1;

            case VC_State is
               when VCDetails.VC_False =>
                  TheTotals.VCsProvedFalse (EndType) :=
                    TheTotals.VCsProvedFalse (EndType) + 1;
                  SubProgramHasVCProvedFalse := True;
                  SubProgramIsUndischarged := True;
               when VCDetails.VC_Proved_By_Examiner =>
                  SubProgramHasVCProvedByExaminer := True;
                  TheTotals.VCsProvedByExaminer (EndType) :=
                    TheTotals.VCsProvedByExaminer (EndType) + 1;
               when VCDetails.VC_Proved_By_Inference =>
                  SubProgramHasVCProvedBySimplifier := True;
                  TheTotals.VCsProvedBySimplifier (EndType) :=
                    TheTotals.VCsProvedBySimplifier (EndType) + 1;
               when VCDetails.VC_Proved_By_Contradiction  =>
                  SubProgramHasVCProvedByContradiction := True;
                  SubProgramHasVCProvedBySimplifier := True;
                  TheTotals.VCsProvedBySimplifier (EndType) :=
                    TheTotals.VCsProvedBySimplifier (EndType) + 1;
               when VCDetails.VC_Proved_By_Checker =>
                  SubProgramHasVCProvedByChecker := True;

                  TheTotals.VCsProvedByChecker (EndType) :=
                    TheTotals.VCsProvedByChecker (EndType) + 1;
               when VCDetails.VC_Proved_By_Review =>
                  SubProgramHasVCProvedByReview := True;

                  TheTotals.VCsProvedByReview (EndType) :=
                    TheTotals.VCsProvedByReview (EndType) + 1;
               when VCDetails.VC_Proved_Using_User_Proof_Rules =>
                  SubProgramHasVCProvedWithUserProofRule := True;
                  SubProgramHasVCProvedBySimplifier := True;
                  TheTotals.VCsProvedWithUserProofRule (EndType) :=
                    TheTotals.VCsProvedWithUserProofRule (EndType) + 1;
                  TheTotals.VCsProvedBySimplifier (EndType) :=
                    TheTotals.VCsProvedBySimplifier (EndType) + 1;
               when VCDetails.VC_SIV_Not_Present | VCDetails.VC_Undischarged =>
                  TheTotals.VCsUndischarged (EndType) :=
                    TheTotals.VCsUndischarged (EndType) + 1;
                  SubProgramIsUndischarged := True;
               when others => -- VC_Not_Present
                  -- No VC is present.
                  null;
            end case;
         end if;

         --# accept F, 41, "Expression is stable";
         if DPC and (not VCHeap.ErrorRaised (VCDetails.Corrupt_DPC_File)) then
            --# end accept;
            if DPC_State = VCDetails.DPC_Dead then
               -- Update the total number of subprograms containing
               -- dead paths.
               if not Subprogram_Contains_Dead_Paths then
                  TheTotals.Subprograms_With_Dead_Paths :=
                    TheTotals.Subprograms_With_Dead_Paths + 1;
               end if;
               Subprogram_Contains_Dead_Paths := True;
               -- Update the total number of dead paths found.
               TheTotals.Number_Of_Dead_Paths :=
                 TheTotals.Number_Of_Dead_Paths + 1;
            end if;
         end if;

         VCHeap.Next (AfterThis => HeapIndex,
                      Success => MoreVCs,
                      NextOne => NextIndex);

         HeapIndex := NextIndex;

      end loop;

      --# assert True;

      --  Update the At Least One Counts
      if SubProgramIsUndischarged then
         if SubProgramHasVCProvedFalse then
            TheTotals.SubprogramsWithALOFalseVC :=
              TheTotals.SubprogramsWithALOFalseVC + 1;
         else
            TheTotals.SubprogramsWithALOUndischargedVC :=
              TheTotals.SubprogramsWithALOUndischargedVC + 1;
         end if;
      end if;

      --# assert True;


      if SubProgramHasVCProvedByExaminer then
         TheTotals.SubprogramsWithALOExaminerVC :=
           TheTotals.SubprogramsWithALOExaminerVC + 1;
      end if;

      --# assert True;

      if SubProgramHasVCProvedBySimplifier then
         TheTotals.SubprogramsWithALOSimplifierVC :=
           TheTotals.SubprogramsWithALOSimplifierVC + 1;
      end if;

      --# assert True;

      if SubProgramHasVCProvedByContradiction then
         TheTotals.SubprogramsWithALOContradictionVC :=
           TheTotals.SubprogramsWithALOContradictionVC + 1;
      end if;

      --# assert True;

      if SubProgramHasVCProvedWithUserProofRule then
         TheTotals.SubprogramsWithALOUserRuleVC :=
           TheTotals.SubprogramsWithALOUserRuleVC + 1;
      end if;

      --# assert True;

      if SubProgramHasVCProvedByChecker then
         TheTotals.SubprogramsWithALOCheckerVC :=
           TheTotals.SubprogramsWithALOCheckerVC + 1;
      end if;

      --# assert True;

      if SubProgramHasVCProvedByReview then
         TheTotals.SubprogramsWithALOReviewVC :=
           TheTotals.SubprogramsWithALOReviewVC + 1;
      end if;

      --# assert True;

      --  Update the proof strategy use hierarchy (See declaration of TotalType)
      --  Examiner -> Simplifier -> User Rules -> Checker -> Review
      if not SubProgramIsUndischarged then
         if SubProgramHasVCProvedByReview then
            TheTotals.SubprogramsProvedByReview :=
              TheTotals.SubprogramsProvedByReview + 1;

         elsif SubProgramHasVCProvedByChecker then
            TheTotals.SubprogramsProvedByChecker :=
              TheTotals.SubprogramsProvedByChecker + 1;

         elsif SubProgramHasVCProvedWithUserProofRule then
            TheTotals.SubprogramsProvedWithUserProofRule :=
              TheTotals.SubprogramsProvedWithUserProofRule + 1;

         elsif SubProgramHasVCProvedBySimplifier then
            TheTotals.SubprogramsProvedBySimplifier :=
              TheTotals.SubprogramsProvedBySimplifier + 1;

         elsif SubProgramHasVCProvedByExaminer then
            TheTotals.SubprogramsProvedByExaminer :=
              TheTotals.SubprogramsProvedByExaminer + 1;
         end if;
      end if;

      --# assert True;

      --# accept F, 33, UnusedPathEnd,            "UnusedPathEnd unused here" &
      --#        F, 33, UnusedPathStart,          "UnusedPathStart unused here" &
      --#        F, 33, UnusedVCName,             "UnusedPVCName unused here";
   end UpdateTotals;

   procedure Output (ReportFile        : in     SPARK_IO.File_Type;
                     TempFile          : in out SPARK_IO.File_Type;
                     TempFalseFile     : in out SPARK_IO.File_Type;
                     TempContraFile    : in out SPARK_IO.File_Type;
                     TempUserFile      : in out SPARK_IO.File_Type;
                     TempRluErrorFile  : in out SPARK_IO.File_Type;
                     TempRluUsedFile   : in out SPARK_IO.File_Type;
                     TempPRVerrFile    : in out SPARK_IO.File_Type;
                     TempWarnErrorFile : in out SPARK_IO.File_Type;
                     TempSDPErrorFile  : in out SPARK_IO.File_Type;
                     TempDPCErrorFile  : in out SPARK_IO.File_Type)
   --# global in     CommandLine.Data;
   --#        in     TheTotals;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLine.Data,
   --#                                ReportFile,
   --#                                TempContraFile,
   --#                                TempDPCErrorFile,
   --#                                TempFalseFile,
   --#                                TempFile,
   --#                                TempPRVerrFile,
   --#                                TempRluErrorFile,
   --#                                TempRluUsedFile,
   --#                                TempSDPErrorFile,
   --#                                TempUserFile,
   --#                                TempWarnErrorFile,
   --#                                TheTotals &
   --#         TempContraFile,
   --#         TempDPCErrorFile,
   --#         TempFalseFile,
   --#         TempFile,
   --#         TempPRVerrFile,
   --#         TempRluErrorFile,
   --#         TempRluUsedFile,
   --#         TempSDPErrorFile,
   --#         TempUserFile,
   --#         TempWarnErrorFile from *;
   is
      PercentUndischargedStr     : EStrings.T;
      PercentProvedByExaminerStr : EStrings.T;
      PercentProvedByCheckerStr  : EStrings.T;
      PercentProvedByReviewStr   : EStrings.T;
      PercentSimplifiedStr       : EStrings.T;
      PercentWithUserRuleStr     : EStrings.T;
      PercentProvedFalseStr      : EStrings.T;

      TotalSubprogramsProved  : Natural;

      TempStoreInt    : Integer;
      TempStoreString : EStrings.T;
      TempStoreBool   : Boolean;
      TempStatus      : SPARK_IO.File_Status;


   begin
      SPARK_IO.Put_Line (ReportFile, Banner.MajorSeparatorLine, 0);
      SPARK_IO.Put_Line (ReportFile, "Summary:", 0);
      SPARK_IO.New_Line (ReportFile, 1);

      --# accept F, 10, TempStatus,    "TempStatus not used here" &
      --#        F, 10, TempStoreBool, "TempStoreBool not used here";

      -- print out the names of files containing warnings or errors
      SPARK_IO.Reset (TempWarnErrorFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempWarnErrorFile) then
         SPARK_IO.Put_Line (ReportFile, "***WARNING: The following files, or their absence, raised warnings or errors:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempWarnErrorFile) loop
            SPARK_IO.Get_Integer (TempWarnErrorFile, TempStoreInt, 4, TempStoreBool);
            SPARK_IO.Put_Integer (ReportFile, TempStoreInt, 4, 10);
            SPARK_IO.Put_String (ReportFile, "  ", 0);
            EStrings.Get_Line (File  => TempWarnErrorFile,
                               E_Str => TempStoreString);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TempStoreString);
            SPARK_IO.New_Line (ReportFile, 1);
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out the names of any user rule files containing syntax errors
      SPARK_IO.Reset (TempRluErrorFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempRluErrorFile) then
         SPARK_IO.Put_Line (ReportFile, "***WARNING: The following user defined rule files contain syntax errors:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempRluErrorFile) loop
            -- Note no number output here (not possible to know earlier)
            EStrings.Get_Line (File  => TempRluErrorFile,
                               E_Str => TempStoreString);
            -- Switching back and forth between Append and Input modes
            -- causes empty lines to be created
            if EStrings.Get_Length (E_Str => TempStoreString) /= 0 then
               SPARK_IO.Put_String (ReportFile, "      ", 0);
               EStrings.Put_String (File  => ReportFile,
                                    E_Str => TempStoreString);
               SPARK_IO.New_Line (ReportFile, 1);
            end if;
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out any used user-defined rule files
      SPARK_IO.Reset (TempRluUsedFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempRluUsedFile) then
         SPARK_IO.Put_Line (ReportFile, "The following user-defined rule files have been used:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempRluUsedFile) loop
            -- Note no number output here (difficult to calculate earlier)
            EStrings.Get_Line (File  => TempRluUsedFile,
                               E_Str => TempStoreString);
            -- Switching back and forth between Append and Input modes
            -- causes empty lines to be created
            if EStrings.Get_Length (E_Str => TempStoreString) /= 0 then
               SPARK_IO.Put_String (ReportFile, "      ", 0);
               EStrings.Put_String (File  => ReportFile,
                                    E_Str => TempStoreString);
               SPARK_IO.New_Line (ReportFile, 1);
            end if;
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out the file names and numbers of any false VC
      SPARK_IO.Reset (TempFalseFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempFalseFile) then
         SPARK_IO.Put_Line (ReportFile, "The following subprograms have VCs proved false:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempFalseFile) loop
            SPARK_IO.Get_Integer (TempFalseFile, TempStoreInt, 4, TempStoreBool);
            SPARK_IO.Put_Integer (ReportFile, TempStoreInt, 4, 10);
            SPARK_IO.Put_String (ReportFile, "  ", 0);
            EStrings.Get_Line (File  => TempFalseFile,
                               E_Str => TempStoreString);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TempStoreString);
            SPARK_IO.New_Line (ReportFile, 1);
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out the file names and numbers of any undischarged VC (excluding those proved false)
      SPARK_IO.Reset (TempFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempFile) then
         SPARK_IO.Put_Line (ReportFile, "The following subprograms have undischarged VCs (excluding those proved false):", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempFile) loop
            SPARK_IO.Get_Integer (TempFile, TempStoreInt, 4, TempStoreBool);
            SPARK_IO.Put_Integer (ReportFile, TempStoreInt, 4, 10);
            SPARK_IO.Put_String (ReportFile, "  ", 0);
            EStrings.Get_Line (File  => TempFile,
                               E_Str => TempStoreString);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TempStoreString);
            SPARK_IO.New_Line (ReportFile, 1);
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out the file names and numbers of any VC proved by contradiction
      SPARK_IO.Reset (TempContraFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempContraFile) then
         SPARK_IO.Put_Line (ReportFile, "The following subprograms have VCs proved by contradiction:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempContraFile) loop
            SPARK_IO.Get_Integer (TempContraFile, TempStoreInt, 4, TempStoreBool);
            SPARK_IO.Put_Integer (ReportFile, TempStoreInt, 4, 10);
            SPARK_IO.Put_String (ReportFile, "  ", 0);
            EStrings.Get_Line (File  => TempContraFile,
                               E_Str => TempStoreString);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TempStoreString);
            SPARK_IO.New_Line (ReportFile, 1);
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out the file names and numbers of any VC proved by a user-defined proof rule
      SPARK_IO.Reset (TempUserFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempUserFile) then
         SPARK_IO.Put_Line (ReportFile, "The following subprograms have VCs proved using a user-defined proof rule:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempUserFile) loop
            SPARK_IO.Get_Integer (TempUserFile, TempStoreInt, 4, TempStoreBool);
            SPARK_IO.Put_Integer (ReportFile, TempStoreInt, 4, 10);
            SPARK_IO.Put_String (ReportFile, "  ", 0);
            EStrings.Get_Line (File  => TempUserFile,
                               E_Str => TempStoreString);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TempStoreString);
            SPARK_IO.New_Line (ReportFile, 1);
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- print out the file names and numbers of any files with review errors
      SPARK_IO.Reset (TempPRVerrFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempPRVerrFile) then
         SPARK_IO.Put_Line (ReportFile, "***WARNING: The PRV file(s) associated with the following subprograms may be out of date", 0);
         SPARK_IO.Put_Line (ReportFile, "The following subprograms have review files containing VCs already proved elsewhere:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempPRVerrFile) loop
            SPARK_IO.Get_Integer (TempPRVerrFile, TempStoreInt, 4, TempStoreBool);
            SPARK_IO.Put_Integer (ReportFile, TempStoreInt, 4, 10);
            SPARK_IO.Put_String (ReportFile, "  ", 0);
            EStrings.Get_Line (File  => TempPRVerrFile,
                               E_Str => TempStoreString);
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => TempStoreString);
            SPARK_IO.New_Line (ReportFile, 1);
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      -- print out the names of any missing SDP files
      SPARK_IO.Reset (TempSDPErrorFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempSDPErrorFile) then
         SPARK_IO.Put_Line (ReportFile, "***WARNING: The following DPC files have not been ZombieScoped:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempSDPErrorFile) loop
            EStrings.Get_Line (File  => TempSDPErrorFile,
                               E_Str => TempStoreString);
            -- Switching back and forth between Append and Input modes
            -- causes empty lines to be created
            if EStrings.Get_Length (E_Str => TempStoreString) /= 0 then
               SPARK_IO.Put_String (ReportFile, "      ", 0);
               EStrings.Put_String (File  => ReportFile,
                                    E_Str => TempStoreString);
               SPARK_IO.New_Line (ReportFile, 1);
            end if;
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      -- print out the names of any missing DPC files
      SPARK_IO.Reset (TempDPCErrorFile, SPARK_IO.In_File, TempStatus);
      if not SPARK_IO.End_Of_File (TempDPCErrorFile) then
         SPARK_IO.Put_Line (ReportFile, "***WARNING: The following DPC files are missing:", 0);
         SPARK_IO.New_Line (ReportFile, 1);
         while not SPARK_IO.End_Of_File (TempDPCErrorFile) loop
            EStrings.Get_Line (File  => TempDPCErrorFile,
                               E_Str => TempStoreString);
            -- Switching back and forth between Append and Input modes
            -- causes empty lines to be created
            if EStrings.Get_Length (E_Str => TempStoreString) /= 0 then
               SPARK_IO.Put_String (ReportFile, "      ", 0);
               EStrings.Put_String (File  => ReportFile,
                                    E_Str => TempStoreString);
               SPARK_IO.New_Line (ReportFile, 1);
            end if;
         end loop;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;
      --# end accept;

      --# assert True;

      -- Print a summary of the number of subprograms conatining at
      -- least one instance of the following:
      SPARK_IO.Put_Line
        (ReportFile,
         "Proof strategies used by subprograms", 0);
      SPARK_IO.Put_Line
        (ReportFile,
         "-------------------------------------------------------------------------", 0);
      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one VC proved by examiner:           ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOExaminerVC, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one VC proved by simplifier:         ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOSimplifierVC, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one VC proved by contradiction:      ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOContradictionVC, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one VC proved with user proof rule:  ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOUserRuleVC, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one VC proved using checker:         ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOCheckerVC, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one VC discharged by review:         ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOReviewVC, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      --# assert True;

      -- Print out hierarchy of proof strategy use (see declaration of TotalType)
      --  Examiner -> Simplifier -> User Rules -> Checker -> Review
      SPARK_IO.New_Line (ReportFile, 1);
      SPARK_IO.Put_Line
        (ReportFile,
         "Maximum extent of strategies used for fully proved subprograms:", 0);
      SPARK_IO.Put_Line
        (ReportFile,
         "-------------------------------------------------------------------------", 0);
      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with proof completed by examiner:                  ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsProvedByExaminer, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with proof completed by simplifier:                ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsProvedBySimplifier, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with proof completed with user defined rules:      ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsProvedWithUserProofRule, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with proof completed by checker:                   ", 0);
      SPARK_IO.Put_Integer (ReportFile, TheTotals.SubprogramsProvedByChecker, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with VCs discharged by review:                     ", 0);
      SPARK_IO.Put_Integer (ReportFile, TheTotals.SubprogramsProvedByReview, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      TotalSubprogramsProved := ((((
        TheTotals.SubprogramsProvedByExaminer +
          TheTotals.SubprogramsProvedBySimplifier) +
            TheTotals.SubprogramsProvedWithUserProofRule) +
              TheTotals.SubprogramsProvedByChecker) +
                TheTotals.SubprogramsProvedByReview);

      --# assert True;

      SPARK_IO.New_Line (ReportFile, 1);
      SPARK_IO.Put_Line
        (ReportFile,
         "Overall subprogram summary:", 0);
      SPARK_IO.Put_Line
        (ReportFile,
         "-------------------------------------------------------------------------", 0);
      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms fully proved:                                      ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TotalSubprogramsProved, 4, 10);
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one undischarged VC:                 ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOUndischargedVC, 4, 10);

      if TheTotals.SubprogramsWithALOUndischargedVC > 0 then
         SPARK_IO.Put_String (ReportFile, "  <<<", 0);
      end if;
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms with at least one false VC:                        ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.SubprogramsWithALOFalseVC, 4, 10);

      if TheTotals.SubprogramsWithALOFalseVC > 0 then
         SPARK_IO.Put_String (ReportFile, "  <<<", 0);
      end if;
      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "                                                                    -----", 0);

      SPARK_IO.New_Line (ReportFile, 1);
      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms for which VCs have been generated:                 ", 0);
      SPARK_IO.Put_Integer (ReportFile, TheTotals.SubprogramsWithVCs, 4, 10);
      SPARK_IO.New_Line (ReportFile, 2);

      -- Only report errors if there are some.
      if (TheTotals.Subprograms_Where_VC_Analysis_Abandoned > 0) or
        (TheTotals.Subprograms_Where_DPC_Analysis_Abandoned > 0) or
        (TheTotals.SubprogramsWhereMissingSLGFile > 0) then
         SPARK_IO.Put_Line
           (ReportFile,
            "WARNING: Overall error summary:", 0);
         SPARK_IO.Put_Line
           (ReportFile,
            "-------------------------------------------------------------------------", 0);
         SPARK_IO.Put_String
           (ReportFile,
            "Total simplified subprograms with missing slg file:               ", 0);
         SPARK_IO.Put_Integer (ReportFile, TheTotals.SubprogramsWhereMissingSLGFile, 7, 10);
         SPARK_IO.New_Line (ReportFile, 1);
         SPARK_IO.Put_String
           (ReportFile,
            "Total subprograms where VC analysis was abandoned due to errors:     ", 0);
         SPARK_IO.Put_Integer (ReportFile, TheTotals.Subprograms_Where_VC_Analysis_Abandoned, 4, 10);
         SPARK_IO.New_Line (ReportFile, 1);
         SPARK_IO.Put_String
           (ReportFile,
            "Total subprograms where DPC analysis was abandoned due to errors:     ", 0);
         SPARK_IO.Put_Integer (ReportFile, TheTotals.Subprograms_Where_DPC_Analysis_Abandoned, 3, 10);
         SPARK_IO.New_Line (ReportFile, 2);
      else
         -- One blank line between each table in this group, but double blank line
         -- after the last table.
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --  The sum of the subprograms fully proved,
      --  the subprograms with at least undischarged VC and
      --  the subprograms with at least 1 false VC must equal
      --  the number of subprograms for which VCs have been generated.

      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);

      --# assert TotalSubprogramsProved +
      --#        TheTotals.SubprogramsWithALOUndischargedVC +
      --#        TheTotals.SubprogramsWithALOFalseVC
      --#        = TheTotals.SubprogramsWithVCs;

      -- Check the totals are balanced.
      if not TotalsAreBalanced then
         FatalError (FatalErrors.SubprogramTotalsInconsistent);
      end if;

      --# assert True;

      SPARK_IO.Put_Line
        (ReportFile,
         "ZombieScope Summary:", 0);
      SPARK_IO.Put_Line
        (ReportFile,
         "-------------------------------------------------------------------------", 0);
      SPARK_IO.Put_String
        (ReportFile,
         "Total subprograms for which DPCs have been generated:", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.Subprograms_With_DPCs, 20, 10);
      SPARK_IO.New_Line (ReportFile, 1);
      SPARK_IO.Put_String
        (ReportFile,
         "Total number subprograms with dead paths found:", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.Subprograms_With_Dead_Paths, 26, 10);
      SPARK_IO.New_Line (ReportFile, 1);
      SPARK_IO.Put_String
        (ReportFile,
         "Total number of dead paths found:", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.Number_Of_Dead_Paths, 40, 10);
      SPARK_IO.New_Line (ReportFile, 3);

      -- Issue warning message if some DPC files have not been analysed.

      --# assert True;

      SPARK_IO.Put_Line
        (ReportFile,
         "VC summary:", 0);
      SPARK_IO.Put_Line
        (ReportFile,
         "-------------------------------------------------------------------------", 0);
      SPARK_IO.Put_Line
        (ReportFile,
         "Note: U/R denotes where the Simplifier has proved VCs using one or more " &
         "user-", 0);
      SPARK_IO.Put_Line
        (ReportFile, "defined proof rules.", 0);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile,
         "Total VCs by type:                                       ", 0);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_Line
        (ReportFile,
         "                            -----------Proved By Or Using------------", 0);
      SPARK_IO.Put_String
        (ReportFile,
         "                     Total  Examiner Simp(U/R)  Checker Review False Undiscgd", 0);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile, "Assert or Post:    ", 0);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsTotal (VCDetails.Assert_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Assert_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Assert_Point), 7, 10);

      if TheTotals.VCsProvedWithUserProofRule (VCDetails.Assert_Point) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Assert_Point), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Assert_Point), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Assert_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Assert_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsUndischarged (VCDetails.Assert_Point), 8, 10);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile, "Precondition check:", 0);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsTotal (VCDetails.Precondition_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Precondition_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Precondition_Check_Point), 7, 10);

      if TheTotals.VCsProvedWithUserProofRule (VCDetails.Precondition_Check_Point) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Precondition_Check_Point), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Precondition_Check_Point), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Precondition_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Precondition_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsUndischarged (VCDetails.Precondition_Check_Point), 8, 10);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile, "Check statement:   ", 0);
      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsTotal (VCDetails.Check_Statement_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Check_Statement_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Check_Statement_Point), 7, 10);

      if TheTotals.VCsProvedWithUserProofRule (VCDetails.Check_Statement_Point) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Check_Statement_Point), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Check_Statement_Point), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Check_Statement_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Check_Statement_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsUndischarged (VCDetails.Check_Statement_Point), 8, 10);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile, "Runtime check:     ", 0);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsTotal (VCDetails.Runtime_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Runtime_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Runtime_Check_Point), 7, 10);

      if TheTotals.VCsProvedWithUserProofRule (VCDetails.Runtime_Check_Point) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Runtime_Check_Point), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Runtime_Check_Point), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Runtime_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Runtime_Check_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsUndischarged (VCDetails.Runtime_Check_Point), 8, 10);

      SPARK_IO.New_Line (ReportFile, 1);


      SPARK_IO.Put_String
        (ReportFile, "Refinement VCs:    ", 0);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsTotal (VCDetails.Refinement_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Refinement_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Refinement_VC_Point), 7, 10);

      if TheTotals.VCsProvedWithUserProofRule (VCDetails.Refinement_VC_Point) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Refinement_VC_Point), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Refinement_VC_Point), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Refinement_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Refinement_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsUndischarged (VCDetails.Refinement_VC_Point), 8, 10);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile, "Inheritance VCs:   ", 0);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsTotal (VCDetails.Inheritance_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Inheritance_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Inheritance_VC_Point), 7, 10);

      if TheTotals.VCsProvedWithUserProofRule (VCDetails.Inheritance_VC_Point) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Inheritance_VC_Point), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Undetermined_Point), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Inheritance_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Inheritance_VC_Point), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, TheTotals.VCsUndischarged (VCDetails.Inheritance_VC_Point), 8, 10);

      SPARK_IO.New_Line (ReportFile, 1);

      -- Assuming a well-structured set of VCG, SIV and PLG files,
      -- the number of Undetermined_Point VCs should always be
      -- zero.
      --
      -- A corrupt VCG file might lead to an Undetermined_Point
      -- VC, so we only output the total if it's non-zero...

      if TheTotals.VCsTotal (VCDetails.Undetermined_Point) > 0 then
         SPARK_IO.Put_String
           (ReportFile, "Undetermined:      ", 0);

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsTotal (VCDetails.Undetermined_Point), 7, 10);

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedByExaminer (VCDetails.Undetermined_Point), 7, 10);

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedBySimplifier (VCDetails.Undetermined_Point), 7, 10);

         if TheTotals.VCsProvedWithUserProofRule (VCDetails.Undetermined_Point) > 0 then
            SPARK_IO.Put_String (ReportFile, "(", 0);
            SPARK_IO.Put_Integer
              (ReportFile, TheTotals.VCsProvedWithUserProofRule (VCDetails.Undetermined_Point), 4, 10);
            SPARK_IO.Put_String (ReportFile, ")", 0);
         else
            SPARK_IO.Put_String (ReportFile, "      ", 0);
         end if;

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedByChecker (VCDetails.Undetermined_Point), 9, 10);

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedByReview (VCDetails.Undetermined_Point), 7, 10);

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsProvedFalse (VCDetails.Undetermined_Point), 7, 10);

         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.VCsUndischarged (VCDetails.Undetermined_Point), 8, 10);

         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      SPARK_IO.Put_String
        (ReportFile, Banner.MajorSeparatorLine, 0);

      SPARK_IO.New_Line (ReportFile, 1);

      SPARK_IO.Put_String
        (ReportFile, "Totals:            ", 0);

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsTotal), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsProvedByExaminer), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsProvedBySimplifier), 7, 10);

      if Sum (TheTotals.VCsProvedWithUserProofRule) > 0 then
         SPARK_IO.Put_String (ReportFile, "(", 0);
         SPARK_IO.Put_Integer
           (ReportFile, Sum (TheTotals.VCsProvedWithUserProofRule), 4, 10);
         SPARK_IO.Put_String (ReportFile, ")", 0);
      else
         SPARK_IO.Put_String (ReportFile, "      ", 0);
      end if;

      --# assert True;

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsProvedByChecker), 9, 10);

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsProvedByReview), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsProvedFalse), 7, 10);

      SPARK_IO.Put_Integer
        (ReportFile, Sum (TheTotals.VCsUndischarged), 8, 10);

      if Sum (TheTotals.VCsUndischarged) > 0 then
         SPARK_IO.Put_String (ReportFile, "  <<<", 0);
      end if;

      --# assert True;

      SPARK_IO.New_Line (ReportFile, 1);

      if CommandLine.Data.OutputPercentUndischarged and then
        Sum (TheTotals.VCsTotal) /= 0 then

         CalcuatePercentages (TheTotals,
                              PercentUndischargedStr,
                              PercentProvedByExaminerStr,
                              PercentProvedByCheckerStr,
                              PercentProvedByReviewStr,
                              PercentSimplifiedStr,
                              PercentWithUserRuleStr,
                              PercentProvedFalseStr);

         SPARK_IO.Put_String (ReportFile, "% Totals:                    ", 0);
         EStrings.Put_String (File  => ReportFile,
                              E_Str => PercentProvedByExaminerStr);
         SPARK_IO.Put_String (ReportFile, "  ", 0);
         EStrings.Put_String (File  => ReportFile,
                              E_Str => PercentSimplifiedStr);

         if Sum (TheTotals.VCsProvedWithUserProofRule) > 0 then
            SPARK_IO.Put_String (ReportFile, "(", 0); -- 1 char
            EStrings.Put_String (File  => ReportFile,
                                 E_Str => PercentWithUserRuleStr); -- 5 char
            SPARK_IO.Put_String (ReportFile, ")", 0); -- 1 char
         else
            SPARK_IO.Put_String (ReportFile, "       ", 0); -- 1+5+1 = 7 char
         end if;

         SPARK_IO.Put_String (ReportFile, "   ", 0);
         EStrings.Put_String (File  => ReportFile,
                              E_Str => PercentProvedByCheckerStr);
         SPARK_IO.Put_String (ReportFile, "  ", 0);
         EStrings.Put_String (File  => ReportFile,
                              E_Str => PercentProvedByReviewStr);
         SPARK_IO.Put_String (ReportFile, "  ", 0);
         EStrings.Put_String (File  => ReportFile,
                              E_Str => PercentProvedFalseStr);
         SPARK_IO.Put_String (ReportFile, "   ", 0);
         EStrings.Put_String (File  => ReportFile,
                              E_Str => PercentUndischargedStr);

         -- Only suppress the 'look out' tag if everything has been proved, and
         -- there are no false VCs.
         if not ((EStrings.Eq1_String (E_Str => PercentUndischargedStr,
                                       Str   => "   0%") and then
                    EStrings.Eq1_String (E_Str => PercentProvedFalseStr,
                                         Str   => "   0%"))) then
            SPARK_IO.Put_String (ReportFile, " <<<", 0);
         end if;
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      -- Only report errors if there are some.
      if TheTotals.Subprograms_Where_VC_Analysis_Abandoned > 0 then
         SPARK_IO.Put_String
           (ReportFile, "!!!                         ERRORS IN SIV FILES                   !!!", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.Subprograms_Where_VC_Analysis_Abandoned, 8, 10);
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      if TheTotals.Subprograms_Where_DPC_Analysis_Abandoned > 0 then
         SPARK_IO.Put_String
           (ReportFile, "!!!                         ERRORS IN DPC FILES                   !!!", 0);
         SPARK_IO.Put_Integer
           (ReportFile, TheTotals.Subprograms_Where_DPC_Analysis_Abandoned, 8, 10);
         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# assert True;

      --# accept F, 33, TempStatus,    "TempStatus not used here" &
      --#        F, 33, TempStoreBool, "TempStoreBool not used here";
   end Output;


   procedure XMLOutput (ReportFile : in SPARK_IO.File_Type)
   --# global in     TheTotals;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLSummary.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                ReportFile,
   --#                                TheTotals,
   --#                                XMLSummary.State &
   --#         XMLSummary.State  from *,
   --#                                TheTotals;
   is
      PercentUndischargedStr     : EStrings.T;
      PercentProvedByExaminerStr : EStrings.T;
      PercentProvedByCheckerStr  : EStrings.T;
      PercentProvedByReviewStr   : EStrings.T;
      PercentSimplifiedStr       : EStrings.T;
      PercentWithUserRuleStr     : EStrings.T;
      PercentProvedFalseStr      : EStrings.T;
   begin

      -- Check the totals are balanced.
      if not TotalsAreBalanced then
         FatalError (FatalErrors.SubprogramTotalsInconsistent);
      end if;

      XMLSummary.StartSummary (ReportFile);

      XMLSummary.StartSubprogramSummary (TheTotals.SubprogramsWithVCs,
                                         ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.SubprogramsProvedByExaminer,
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.SubprogramsProvedBySimplifier,
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.SubprogramsProvedByChecker,
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.SubprogramsProvedByReview,
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.SubprogramsWithALOFalseVC,
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.SubprogramsWithALOUndischargedVC,
                                 ReportFile);

      XMLSummary.EndSubprogramSummary (ReportFile);

      --# assert True;

      -- In XML always report number of errors, even if they are zero.
      -- (It seems better for parsing not to hide XML entries.)
      -- This is deliberately not nested inside the subprograms tag above, as
      -- this is implicitly describing only well-formed subprograms.
      XMLSummary.StartSubprogramsInErrorSummary (TheTotals.Subprograms_Where_VC_Analysis_Abandoned,
                                                 ReportFile);
      ---- Nothing inside this tag just now.
      XMLSummary.EndSubprogramsInErrorSummary (ReportFile);

      --# assert True;

      XMLSummary.StartVCSummary (ReportFile);

      XMLSummary.StartSummarySection (XMLSummary.SSAssert_Post,
                                      TheTotals.VCsTotal (VCDetails.Assert_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Assert_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Assert_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Assert_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Assert_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Assert_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Assert_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);

      --# assert True;

      XMLSummary.StartSummarySection (XMLSummary.SSPrecondition,
                                      TheTotals.VCsTotal (VCDetails.Precondition_Check_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Precondition_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Precondition_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Precondition_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Precondition_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Precondition_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Precondition_Check_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);

      --# assert True;

      XMLSummary.StartSummarySection (XMLSummary.SSCheck,
                                      TheTotals.VCsTotal (VCDetails.Check_Statement_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Check_Statement_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Check_Statement_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Check_Statement_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Check_Statement_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Check_Statement_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Check_Statement_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);

      --# assert True;

      XMLSummary.StartSummarySection (XMLSummary.SSRuntime,
                                      TheTotals.VCsTotal (VCDetails.Runtime_Check_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Runtime_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Runtime_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Runtime_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Runtime_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Runtime_Check_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Runtime_Check_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);

      --# assert True;


      XMLSummary.StartSummarySection (XMLSummary.SSRefinement,
                                      TheTotals.VCsTotal (VCDetails.Refinement_VC_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Refinement_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Refinement_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Refinement_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Refinement_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Refinement_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Refinement_VC_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);


      --# assert True;

      XMLSummary.StartSummarySection (XMLSummary.SSInheritance,
                                      TheTotals.VCsTotal (VCDetails.Inheritance_VC_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Inheritance_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Inheritance_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Inheritance_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Inheritance_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Inheritance_VC_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Inheritance_VC_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);


      --# assert True;

      XMLSummary.StartSummarySection (XMLSummary.SSUndetermined,
                                      TheTotals.VCsTotal (VCDetails.Undetermined_Point),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 TheTotals.VCsProvedByExaminer (VCDetails.Undetermined_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 TheTotals.VCsProvedBySimplifier (VCDetails.Undetermined_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 TheTotals.VCsProvedByChecker (VCDetails.Undetermined_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 TheTotals.VCsProvedByReview (VCDetails.Undetermined_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 TheTotals.VCsProvedFalse (VCDetails.Undetermined_Point),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 TheTotals.VCsUndischarged (VCDetails.Undetermined_Point),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);


      --# assert True;

      XMLSummary.StartSummarySection (XMLSummary.SSTotals,
                                      Sum (TheTotals.VCsTotal),
                                      ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITExaminer,
                                 Sum (TheTotals.VCsProvedByExaminer),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITSimplifier,
                                 Sum (TheTotals.VCsProvedBySimplifier),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITChecker,
                                 Sum (TheTotals.VCsProvedByChecker),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITReview,
                                 Sum (TheTotals.VCsProvedByReview),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITFalse,
                                 Sum (TheTotals.VCsProvedFalse),
                                 ReportFile);

      XMLSummary.SummaryItemInt (XMLSummary.ITUndischarged,
                                 Sum (TheTotals.VCsUndischarged),
                                 ReportFile);

      XMLSummary.EndSummarySection (ReportFile);

      --# assert True;

      if Sum (TheTotals.VCsTotal) /= 0 then

         CalcuatePercentages (TheTotals,
                              PercentUndischargedStr,
                              PercentProvedByExaminerStr,
                              PercentProvedByCheckerStr,
                              PercentProvedByReviewStr,
                              PercentSimplifiedStr,
                              PercentWithUserRuleStr,
                              PercentProvedFalseStr);

         XMLSummary.StartSummarySection (XMLSummary.SSPercentages,
                                         Sum (TheTotals.VCsTotal),
                                         ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITExaminer,
                                    EStrings.Trim (PercentProvedByExaminerStr),
                                    ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITSimplifier,
                                    EStrings.Trim (PercentSimplifiedStr),
                                    ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITSimplifierUserRule,
                                    EStrings.Trim (PercentWithUserRuleStr),
                                    ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITChecker,
                                    EStrings.Trim (PercentProvedByCheckerStr),
                                    ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITReview,
                                    EStrings.Trim (PercentProvedByReviewStr),
                                    ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITFalse,
                                    EStrings.Trim (PercentProvedFalseStr),
                                    ReportFile);

         XMLSummary.SummaryItemStr (XMLSummary.ITUndischarged,
                                    EStrings.Trim (PercentUndischargedStr),
                                    ReportFile);

         XMLSummary.EndSummarySection (ReportFile);

      end if;

      XMLSummary.EndVCSummary (ReportFile);

      XMLSummary.EndSummary (ReportFile);

   end XMLOutput;

begin
   TheTotals := TotalType'(SubprogramsWithVCs                        => 0,
                           SubprogramsWhereMissingSLGFile            => 0,
                           Subprograms_Where_VC_Analysis_Abandoned   => 0,
                           Subprograms_Where_DPC_Analysis_Abandoned  => 0,
                           SubprogramsWithALOUndischargedVC          => 0,
                           SubprogramsWithALOExaminerVC              => 0,
                           SubprogramsWithALOSimplifierVC            => 0,
                           SubprogramsWithALOContradictionVC         => 0,
                           SubprogramsWithALOUserRuleVC              => 0,
                           SubprogramsWithALOCheckerVC               => 0,
                           SubprogramsWithALOReviewVC                => 0,
                           SubprogramsWithALOFalseVC                 => 0,
                           SubprogramsProvedByExaminer               => 0,
                           SubprogramsProvedBySimplifier             => 0,
                           SubprogramsProvedByChecker                => 0,
                           SubprogramsProvedWithUserProofRule        => 0,
                           SubprogramsProvedByReview                 => 0,
                           VCsTotal                                  => NullVCCounter,
                           VCsProvedByExaminer                       => NullVCCounter,
                           VCsProvedBySimplifier                     => NullVCCounter,
                           VCsProvedByChecker                        => NullVCCounter,
                           VCsProvedWithUserProofRule                => NullVCCounter,
                           VCsProvedByReview                         => NullVCCounter,
                           VCsProvedFalse                            => NullVCCounter,
                           VCsUndischarged                           => NullVCCounter,
                           Subprograms_With_DPCs                     => 0,
                           Subprograms_With_Dead_Paths               => 0,
                           Number_Of_Dead_Paths                      => 0);
end Total;
