-- $Id: toppackage.adb 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Top level package for the summary tool                                      --
--The main program is simply a call to the procedure MainProcedure wrapped in --
--an execption handler to deal with the exceptions raised by FatalErrors      --
--                                                                            --
--------------------------------------------------------------------------------
with Banner;
with CommandLine;
with ELStrings;
with FatalErrors;
with FileDetails;
with FileHeap;
with FindFiles;
with Heap;
with SPARK_IO;
with Total;
with VCHeap;
with VCS;
with XMLSummary;

use type FileDetails.FileTypes;
use type SPARK_IO.File_Status;

package body TopPackage
is

   procedure MainProcedure
   is
      UnusedDirectoryIsResolved : Boolean;
      FileName               : ELStrings.T;
      FileType               : FileDetails.FileTypes;
      NextFile               : Heap.Atom;
      ReportFile             : SPARK_IO.File_Type := SPARK_IO.Null_File;
      Status                 : SPARK_IO.File_Status;
      Success                : Boolean;
      ThisFile               : Heap.Atom;
      TempFile               : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempFalseFile          : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempContraFile         : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempUserFile           : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempRluErrorFile       : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempRluUsedFile        : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempPRVerrFile         : SPARK_IO.File_Type := SPARK_IO.Null_File;
      TempWarnErrorFile      : SPARK_IO.File_Type := SPARK_IO.Null_File;
      Temp_SDP_Error_File    : SPARK_IO.File_Type := SPARK_IO.Null_File;
      Temp_DPC_Error_File    : SPARK_IO.File_Type := SPARK_IO.Null_File;
      StatusTemp             : SPARK_IO.File_Status;

   begin
      VCHeap.Initialize;
      CommandLine.Read;

      if CommandLine.Data.VersionRequested then
         -- print banner on screen
         Banner.ReportVersion;
      else
         Banner.Screen;

         if not CommandLine.Data.Valid then
            FatalErrors.Process (FatalErrors.InvalidCommandLine, ELStrings.Empty_String);
         end if;

         -- read file structure
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "Reading file structure ...",
                            0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);

         FindFiles.Scan (CommandLine.Data.StartDirectory);

         -- open report file
         ELStrings.Create (File         => ReportFile,
                           Name_Of_File => CommandLine.Data.ReportFile,
                           Form_Of_File => "",
                           Status       => Status);
         if Status /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.CouldNotCreateReportFile,
                                 CommandLine.Data.ReportFile);
         end if;

         -- print the report file banner
         Banner.Report (ReportFile);

         -- process the files
         ThisFile := FileHeap.FirstEntry;

         Success := True;

         if CommandLine.Data.XML then
            XMLSummary.StartResults (ReportFile);
         end if;


         -- create temporary file to store names of files with undischarged vcs
         SPARK_IO.Create (TempFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         --# assert True;
         -- create temporary file to store names of files with false vcs
         SPARK_IO.Create (TempFalseFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of files with vcs proved by contradiction
         SPARK_IO.Create (TempContraFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of files with vcs proved using a
         -- user-defined proof rule
         SPARK_IO.Create (TempUserFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         --# assert True;

         -- create temporary file to store names of user rule files that contain
         -- syntax errors
         SPARK_IO.Create (TempRluErrorFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of user rule files that contain
         -- rules used for proof
         SPARK_IO.Create (TempRluUsedFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of review files containing errors
         SPARK_IO.Create (TempPRVerrFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of files containing warnings or errors
         SPARK_IO.Create (TempWarnErrorFile,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of missing SDP file.
         SPARK_IO.Create (Temp_SDP_Error_File,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;

         -- create temporary file to store names of missing DPC file.
         SPARK_IO.Create (Temp_DPC_Error_File,
                          0,
                          "",
                          "",
                          StatusTemp);
         if StatusTemp /= SPARK_IO.Ok then
            FatalErrors.Process (FatalErrors.ProblemCreatingTempFile, ELStrings.Empty_String);
         end if;
         --# assert True;

         while not Heap.IsNullPointer (ThisFile) and Success loop
            -- read file name
            --# accept F, 10, UnusedDirectoryIsResolved, "UnusedDirectoryIsResolved unused here";
            FileHeap.Details (ThisFile,
                              Success,
                              FileName,
                              FileType,
                              UnusedDirectoryIsResolved);
            --# end accept;
            if not Success then
               FatalErrors.Process (FatalErrors.DataStructureInconsistency,
                                    ELStrings.Empty_String);
            end if;

            if FileType = FileDetails.PlainFile then
               --# accept F, 41, "Expression is stable but efficient";
               if CommandLine.Data.AnalyseVCs then

                  --Rather than assume that old values will be overwritten, actually start from
                  --an initiised state each time.
                  VCHeap.Initialize;
                  if CommandLine.Data.ShortSummary then

                     VCS.Analyse (SPARK_IO.Null_File,
                                  FileName,
                                  CommandLine.Data.AnalyseProofLog,
                                  TempFile,
                                  TempFalseFile,
                                  TempContraFile,
                                  TempUserFile,
                                  TempRluErrorFile,
                                  TempRluUsedFile,
                                  TempPRVerrFile,
                                  TempWarnErrorFile,
                                  Temp_SDP_Error_File,
                                  Temp_DPC_Error_File);
                  else

                     VCS.Analyse (ReportFile,
                                  FileName,
                                  CommandLine.Data.AnalyseProofLog,
                                  TempFile,
                                  TempFalseFile,
                                  TempContraFile,
                                  TempUserFile,
                                  TempRluErrorFile,
                                  TempRluUsedFile,
                                  TempPRVerrFile,
                                  TempWarnErrorFile,
                                  Temp_SDP_Error_File,
                                  Temp_DPC_Error_File);
                  end if;
               end if;
               --# end accept;

               -- put command to analyse path function files here
               -- Not Implemented in POGS yet...
               -- if CommandLine.Data.AnalysePFs then
               --   null;
               -- end if;

            end if;

            -- get next item in linked list
            FileHeap.Next (ThisFile,
                           Success,
                           NextFile);

            -- note that Success will be tested next, at the top of the loop
            ThisFile := NextFile;
         end loop;

         if CommandLine.Data.XML then
            Total.XMLOutput (ReportFile);
            XMLSummary.EndResults (ReportFile);
            XMLSummary.EndPogs (ReportFile);
         else

            -- Output totals
            --# accept F, 10, TempFile,            "TempFile unused here" &
            --#        F, 10, TempFalseFile,       "TempFalseFile unused here" &
            --#        F, 10, TempContraFile,      "TempContraFile unused here" &
            --#        F, 10, TempUserFile,        "TempUserFile unused here" &
            --#        F, 10, TempRluErrorFile,    "TempRluErrorFile unused here" &
            --#        F, 10, TempRluUsedFile,     "TempRluUsedFile unused here" &
            --#        F, 10, TempWarnErrorFile,   "TempPRVerrFile unused here" &
            --#        F, 10, TempPRVerrFile,      "TempPRVerrFile unused here" &
            --#        F, 10, Temp_DPC_Error_File, "Temp_DPC_Error_File unused here" &
            --#        F, 10, Temp_SDP_Error_File, "Temp_SDP_Error_File unused here";

            Total.Output (ReportFile,
                          TempFile,
                          TempFalseFile,
                          TempContraFile,
                          TempUserFile,
                          TempRluErrorFile,
                          TempRluUsedFile,
                          TempPRVerrFile,
                          TempWarnErrorFile,
                          Temp_SDP_Error_File,
                          Temp_DPC_Error_File);
            --# end accept;

            Banner.FinishReport (ReportFile);
         end if;

         -- close report file
         --# accept F, 10, Status,     "Status unused here" &
         --#        F, 10, ReportFile, "ReportFile unused here";
         SPARK_IO.Close (ReportFile, Status);
      end if;

      --# accept F, 33, UnusedDirectoryIsResolved, "UnusedDirectoryIsResolved unused here";
   end MainProcedure;
end TopPackage;
