-- $Id: osfiling.adb 15908 2010-02-04 10:36:19Z dean kuo $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with CommandLine;
with ELStrings.Not_Spark;
with FatalErrors;
with GNAT.Directory_Operations;
with GNAT.IO_Aux;
with GNAT.OS_Lib;

package body OSFiling
is

   ------------------------------------------------------------------------------
   --
   -- Given a path, this function changes directory to that path and returns a
   -- string containing the full path. Useful when you have a path such as "."
   -- or ".." and you want to get a meaningful directory name.
   --
   -- Note: has the side-effect of changing the current directory to be the
   -- specified directory so if that isn't what you want then save the current
   -- directory before calling this function, then cd back to it afterwards.
   --
   -- Note also that the returned directory name will have a trailing slash
   -- that you may have to deal with.
   --
   -- If the specified path doesn't exist then a fatal error is reported.
   --
   ------------------------------------------------------------------------------
   function CD_And_Get_Name (Path : ELStrings.T)
      return GNAT.Directory_Operations.Dir_Name_Str
   is
   begin
      GNAT.Directory_Operations.Change_Dir (ELStrings.Not_Spark.Get_String (E_Str => Path));
      return GNAT.Directory_Operations.Get_Current_Dir;
   exception
      when others =>
         -- Note: this call will NOT return so the return statement following
         -- it will never be executed, but is expected by the compiler.
         FatalErrors.Process (FatalErrors.ExpectedDirectoryMissing, Path);
         return "";
   end CD_And_Get_Name;

   ----------------------------------------------------------------------------

   function BaseDirName (Path : ELStrings.T)
                         return ELStrings.T
   is
      Current_Dir : constant GNAT.Directory_Operations.Path_Name
         := GNAT.Directory_Operations.Get_Current_Dir; -- save for later
      Full_Path_With_Slash : constant String := CD_And_Get_Name (Path);
      Full_Path_No_Slash : constant String := Full_Path_With_Slash (1 .. Full_Path_With_Slash'Length - 1);
      Base_Name : constant String := GNAT.Directory_Operations.Base_Name (Full_Path_No_Slash, "");
   begin
      GNAT.Directory_Operations.Change_Dir (Current_Dir); -- return to saved dir
      return ELStrings.Copy_String (Str => Base_Name);
   exception
      when others =>
         -- Note: this call will NOT return so the return statement following
         -- it will never be executed, but is expected by the compiler.
         FatalErrors.Process (FatalErrors.ExpectedDirectoryMissing, Path);
         return ELStrings.Empty_String;
   end BaseDirName;

   ----------------------------------------------------------------------------

   function BaseFileName (Path : ELStrings.T)
                         return ELStrings.T
   is
      C : constant String := GNAT.Directory_Operations.Base_Name
        (ELStrings.Not_Spark.Get_String (E_Str => Path), "");
   begin
      return ELStrings.Copy_String (Str => C);
   end BaseFileName;

   ----------------------------------------------------------------------------
   function DefaultReportExtn return EStrings.T
   --# global in CommandLine.Data;
   is
   begin
      if CommandLine.Data.XML then
         return EStrings.Copy_String (Str => ".sml");
      else
         return EStrings.Copy_String (Str => ".sum");
      end if;
   end DefaultReportExtn;

   ------------------------------------------------------------------------------
   -- this function combines the inputs to produce the path to the
   -- subdirectory
   function DownDirectory (Path         : ELStrings.T;
                           SubDirectory : ELStrings.T)
                          return ELStrings.T
   is
      Result  : ELStrings.T;
      Success : Boolean;
   begin
      Result := Path;
      if ELStrings.Get_Element (E_Str => Path,
                                Pos   => ELStrings.Get_Length (E_Str => Path)) /=
        GNAT.Directory_Operations.Dir_Separator then
         ELStrings.Append_Char (E_Str   => Result,
                                Ch      => GNAT.Directory_Operations.Dir_Separator,
                                Success => Success);  -- Success unused
      end if;
      ELStrings.Append_Examiner_Long_String (E_Str1 => Result,
                                             E_Str2 => SubDirectory);
      return Result;
   end DownDirectory;

   ------------------------------------------------------------------------------
   -- this function combines the inputs to produce a full file name
   function FullFileName (Path     : ELStrings.T;
                          FileName : ELStrings.T)
                         return ELStrings.T
   is
   begin
      return DownDirectory (Path, FileName);
   end FullFileName;

   ------------------------------------------------------------------------------
   function GetWorkingDirectory return ELStrings.T
   is
      --# hide GetWorkingDirectory;
      Current_Dir : constant GNAT.Directory_Operations.Path_Name
        := GNAT.Directory_Operations.Get_Current_Dir; -- save for later
      Cwd         : ELStrings.T;
   begin
      Cwd := ELStrings.Copy_String (Str => Current_Dir);
      -- Get_Current_Dir can return a trailing '/' on NT..
      if ELStrings.Get_Length (E_Str => Cwd) /= 0 and then
        ELStrings.Get_Element (E_Str => Cwd,
                               Pos   => ELStrings.Get_Length (E_Str => Cwd)) =
        GNAT.Directory_Operations.Dir_Separator then
         Cwd := ELStrings.Section (E_Str     => Cwd,
                                   Start_Pos => 1,
                                   Length    => ELStrings.Get_Length (E_Str => Cwd) - 1);
      end if;
      return Cwd;
   end GetWorkingDirectory;

   function IsDirectory (Name : ELStrings.T) return Boolean
   is
      StringName : constant String := ELStrings.Not_Spark.Get_String (E_Str => Name);
   begin
      return GNAT.OS_Lib.Is_Directory (StringName);
   end IsDirectory;

   -----------------------------------------------------------------------------
   function IsFile (Name : ELStrings.T) return Boolean
   is
      StringName : constant String := ELStrings.Not_Spark.Get_String (E_Str => Name);
   begin
      return GNAT.IO_Aux.File_Exists (StringName);
   end IsFile;

   ------------------------------------------------------------------------------
   function Order (FirstName, SecondName : ELStrings.T)
                  return ELStrings.Order_Types
   is
   begin
      return ELStrings.Lex_Order (First_Name  => FirstName,
                                  Second_Name => SecondName);
   end Order;

   ------------------------------------------------------------------------------
   procedure RemoveFileExtension (FileName : in out ELStrings.T)
   is
      Dot_Pos : ELStrings.Positions;
   begin
      Dot_Pos := ELStrings.Get_Length (E_Str => FileName);

      while Dot_Pos > 1 and then
        ELStrings.Get_Element (E_Str => FileName,
                               Pos   => Dot_Pos) /= '.' loop
         Dot_Pos := Dot_Pos - 1;
      end loop;

      if ELStrings.Get_Element (E_Str => FileName,
                                Pos   => Dot_Pos) = '.' then
         FileName := ELStrings.Section (E_Str     => FileName,
                                        Start_Pos => 1,
                                        Length    => Dot_Pos - 1);
      end if;
   end RemoveFileExtension;

   ----------------------------------------------------------------------------
   function SimplifiedVCFileExtension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".siv");
   end SimplifiedVCFileExtension;

   ------------------------------------------------------------------------------

   -- this function removes the last directory name from the supplied string
   function DirName (Path : ELStrings.T)
                     return ELStrings.T
   is
      C : constant String := GNAT.Directory_Operations.Dir_Name
        (ELStrings.Not_Spark.Get_String (E_Str => Path));
   begin
      return ELStrings.Copy_String (Str => C);
   end DirName;

   ----------------------------------------------------------------------------
   function VCFileExtension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".vcg");
   end VCFileExtension;

   ----------------------------------------------------------------------------
   function ProofLogFileExtension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".plg");
   end ProofLogFileExtension;

   ----------------------------------------------------------------------------
   function ReviewFileExtension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".prv");
   end ReviewFileExtension;

   ----------------------------------------------------------------------------
   function SimplifierLogFileExtension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".slg");
   end SimplifierLogFileExtension;

   function DPC_File_Extension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".dpc");
   end DPC_File_Extension;
   ----------------------------------------------------------------------------
   function Summary_DP_File_Extension return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => ".sdp");
   end Summary_DP_File_Extension;

   ----------------------------------------------------------------------------
   function DirectorySeparator return Character
   is
   begin
      return GNAT.Directory_Operations.Dir_Separator;
   end DirectorySeparator;

end OSFiling;
