-- $Id: fileheap.ads 11367 2008-10-07 15:47:17Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package combining Heap and FileDetails to give an ordered list of files     --
--                                                                            --
--This is implemented as an abstract state machine. This is possible as there --
--is only a single occurrence of the FileHeap in the program. It is necessary --
--to prevent unacceptable overheads of creating local copies of the embedded  --
--Heap and FileDetails types to circumvent the entire variable rule. It would --
--also be possible to implement Heap and FileDetails as ASMs but not          --
--necessarily desirable as this would affect the high level annotations of the--
--program.                                                                    --
--                                                                            --
--------------------------------------------------------------------------------
with ELStrings,
     FileDetails,
     Heap;

use type Heap.Atom;

--# inherit ELStrings,
--#         FatalErrors,
--#         FileDetails,
--#         Heap,
--#         HeapIndex;
package FileHeap
--# own State;
is

   -- StartIndex is a point in the linked list at which to start the
   -- search. This is used to start insertion at the parent directory name
   -- If the file table is full, a fatal error is produced and Add does not
   -- return
   procedure Add (StartIndex  : in Heap.Atom;
                  NewName     : in ELStrings.T;
                  NewFileType : in FileDetails.FileTypes);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                NewFileType,
   --#                                NewName,
   --#                                StartIndex,
   --#                                State;

   --------------------------------------------------------------------------
   -- this procedure returns the file details for the specified entry in the
   -- linked list. Success if ListIndex points to a heap record which points to
   -- a valid set of file details
   procedure Details (ListIndex           : in     Heap.Atom;
                      Success             :    out Boolean;
                      Name                :    out ELStrings.T;
                      FileType            :    out FileDetails.FileTypes;
                      DirectoryIsResolved :    out Boolean);
   --# global in State;
   --# derives DirectoryIsResolved,
   --#         FileType,
   --#         Name,
   --#         Success             from ListIndex,
   --#                                  State;

   --------------------------------------------------------------------------
   function FirstEntry return Heap.Atom;
   --# global in State;

   --------------------------------------------------------------------------
   procedure Initialize (TopDirectory : in ELStrings.T);
   --# global out State;
   --# derives State from TopDirectory;

   --------------------------------------------------------------------------
   procedure MarkDirectoryResolved (ListIndex : in Heap.Atom);
   --# global in out State;
   --# derives State from *,
   --#                    ListIndex;

   --------------------------------------------------------------------------
   -- this procedure returns the 'NextOne' ordered element in FH after
   -- 'AfterThis'. It is successful if the NextOne is not a 'null' pointer
   procedure Next (AfterThis : in     Heap.Atom;
                   Success   :    out Boolean;
                   NextOne   :    out Heap.Atom);
   --# global in State;
   --# derives NextOne,
   --#         Success from AfterThis,
   --#                      State;

end FileHeap;
