-- $Id: filedetails.ads 14621 2009-10-28 13:48:23Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package providing a structure in which to store file details, in the form   --
--of a full pathname (without extension) and a type, either PlainFile or      --
--directory.                                                                  --
--                                                                            --
--To be used in tandem with the Heap data structure, hence the use of         --
--Heap.Atom as the array range                                                --
--------------------------------------------------------------------------------
with ELStrings,
     HeapIndex;

use type ELStrings.Order_Types;
use type HeapIndex.IndexType;

--# inherit ELStrings,
--#         HeapIndex,
--#         OSFiling;
package FileDetails
is
   --ListLength : constant Integer := 20000;
   --type IndexType is range 0 .. ListLength;

   type FileTypes is (PlainFile, Directory, Invalid);

   type DataType is private;

   --------------------------------------------------------------------------
   procedure Add (Details  : in out DataType;
                  Index    :    out HeapIndex.IndexType;
                  Success  :    out Boolean;
                  Name     : in     ELStrings.T;
                  FileType : in     FileTypes);
   --# derives Details from *,
   --#                      FileType,
   --#                      Name &
   --#         Index,
   --#         Success from Details;

   --------------------------------------------------------------------------
   procedure Initialize (Details : out DataType);
   --# derives Details from ;

   --------------------------------------------------------------------------
   procedure MarkDirectoryResolved (Details : in out DataType;
                                    Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   -- this compares the information given and returns
   -- Result is as defined in section 5.4 of the specification
   -- it works directly on values rather than on indices into the Details
   -- structure so that information can be compared before insertion
   -- NOTE : the procedure is successful iff neither Type is Invalid
   --
   procedure Order (FirstName  : in     ELStrings.T;
                    FirstType  : in     FileTypes;
                    SecondName : in     ELStrings.T;
                    SecondType : in     FileTypes;
                    Success    :    out Boolean;
                    Result     :    out ELStrings.Order_Types);
   --# derives Result  from FirstName,
   --#                      FirstType,
   --#                      SecondName,
   --#                      SecondType &
   --#         Success from FirstType,
   --#                      SecondType;

   --------------------------------------------------------------------------
   procedure Retrieve (Details             : in     DataType;
                       Index               : in     HeapIndex.IndexType;
                       Success             :    out Boolean;
                       Name                :    out ELStrings.T;
                       FileType            :    out FileTypes;
                       DirectoryIsResolved :    out Boolean);
   --# derives DirectoryIsResolved,
   --#         FileType,
   --#         Name,
   --#         Success             from Details,
   --#                                  Index;

private
   type DetailsElement is record
      Name                : ELStrings.T;
      FileType            : FileTypes;
      DirectoryIsResolved : Boolean;
   end record;

   NullDetailsElement : constant DetailsElement :=
     DetailsElement'(Name                => ELStrings.Empty_String,
                     FileType            => Invalid,
                     DirectoryIsResolved => False);

   type ElementsArray is array (HeapIndex.IndexType) of DetailsElement;

   -- HighMark is the number of the highest allocated atom - zero if the
   -- structure is empty
   type DataType is record
      Details  : ElementsArray;
      HighMark : HeapIndex.IndexType;
   end record;

end FileDetails;
