-- $Id: fatalerrors.adb 14621 2009-10-28 13:48:23Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Output 'fatal' error messages                                               --
--Note that the Process procedure does not return to point of call. Instead   --
--it raises an exception which is trapped by the main program, which causes   --
--the program to stop politely.                                               --
--                                                                            --
--                                                                            --
--------------------------------------------------------------------------------
with EStrings,
     CommandLine,
     POGS_Exceptions,
     SPARK_IO;

package body FatalErrors
is
   --# hide FatalErrors;
   -- hidden to mask the use of exceptions and the non-initialization of
   -- FatalErrors.State

   procedure Process (Error   : in ErrorType;
                      Message : in ELStrings.T)
   is
      T : EStrings.T;
   begin
      case Error is
         when CouldNotOpenInputFile =>
            T := EStrings.Copy_String
              (Str => "Could not open expected input file " &
                 "- unexpected file system error");

         when CouldNotCreateReportFile =>
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => EStrings.Copy_String
                                   (Str => "Could not create report file "));
            ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => Message);
            T := EStrings.Copy_String (Str => "Check you have write permission for directory");

         when DataStructureInconsistency =>
            T := EStrings.Copy_String
              (Str => "Internal file table inconsistent.");

         when VCDataStructureInconsistency =>
            T := EStrings.Copy_String
              (Str => "Internal VC table inconsistent.");

         when ExpectedDirectoryMissing =>
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => EStrings.Copy_String
                                   (Str => "Directory "));
            ELStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                  E_Str => Message);
            T := EStrings.Copy_String
              (Str => " not found or not readable.");

         when FileHeapFull =>
            T := EStrings.Copy_String
              (Str => "Internal file table too small.");

         when VCHeapFull =>
            T := EStrings.Copy_String
              (Str => "Internal VC table too small.");

         when InvalidCommandLine =>
            T := EStrings.Copy_String
              (Str => "Usage: ");
            EStrings.Append_Examiner_String
              (E_Str1 => T,
               E_Str2 => CommandLine.UsageString);

         when ProblemReadingFile =>
            T := EStrings.Copy_String
              (Str => "Could not read from input file " &
                 "- unexpected file system error");

         when ProblemCreatingTempFile =>
            T := EStrings.Copy_String
              (Str => "Could not create temporary file.");

         when SubprogramTotalsInconsistent =>
            T := EStrings.Copy_String
              (Str => "Overall subprogram summary counts inconsistent.");

      end case;

      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => T);

      raise POGS_Exceptions.Usage_Error;
   end Process;
end FatalErrors;
