-- $Id: xmlreport.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with FileSystem;
with SystemErrors;

package body XMLReport
--# own State is Schema, SchemaState, TagStrings, TagIDs, SectionDepths, FileDepth;
is

   -- Set the version of the report file schema.  This should be the CVS version of the
   -- sparkreport.xsd file.
   Schema_Version : constant String := "1.8";

   type MyTag is (MTBrief_Justifications,
                  MTCommandline,
                  MTCompiler_Data,
                  MTCompiler_Item,
                  MTCyclic_Requirements,
                  MTFile,
                  MTFilename,
                  MTFull_Justification,
                  MTFull_Justifications,
                  MTIndex,
                  MTIndexes,
                  MTJustifications_Section,
                  MTListing,
                  MTMessage,
                  MTMessages,
                  MTMetafile,
                  MTMetafiles,
                  MTOption,
                  MTPrologue,
                  MTReport,
                  MTResults,
                  MTScope,
                  MTSuppressed,
                  MTSuppressedPragma,
                  MTSymbol,
                  MTTarget_Config,
                  MTUnit,
                  MTUnits_Notfound,
                  MTWarnings_Config,
                  MTUnitsInFile);

   type TagStringArray is array (MyTag) of EStrings.T;
   type TagIDArray is array (MyTag) of SPARK_XML.TagID;
   type SectionDepthArray is array (Sections) of SPARK_XML.TagDepth;


   -------------------
   -- Own Variables --
   -------------------

   -- Dictionary for the schema
   TagStrings : TagStringArray;
   TagIDs : TagIDArray;

   -- The schema
   Schema : SPARK_XML.SchemaRecord;
   SchemaState : SPARK_XML.SchemaStateRecord;

   -- Additional schema state info
   FileDepth : SPARK_XML.TagDepth;
   SectionDepths : SectionDepthArray;


   --------------------
   -- Error handling --
   --------------------

   procedure HandleError (Status : in SPARK_XML.SchemaStatus)
   --# global in Schema;
   --#        in SchemaState;
   --# derives null from Schema,
   --#                   SchemaState,
   --#                   Status;
   is
   begin
      if SPARK_XML.IsError (Status) then
         SPARK_XML.PrintSchemaError (Status);
         SPARK_XML.PrintWorkingState (Schema, SchemaState);

         case Status is
            when SPARK_XML.SSToManyAttributes =>
               SystemErrors.FatalError (SystemErrors.XMLGenerationError, "Exceeded Attribute Limit");
            when SPARK_XML.SSStackFull =>
               SystemErrors.FatalError (SystemErrors.XMLGenerationError, "Exceeded XML Stack limit");
            when SPARK_XML.SSTagIncomplete =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Attempted to gerenate incomplete tag");
            when SPARK_XML.SSInvalidDepth =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Invalid depth value for tag closure");
            when SPARK_XML.SSStackEmpty =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Tag stack empty");
            when SPARK_XML.SSWrongContentType =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Incorrect content type for attribute");
            when SPARK_XML.SSInvalidAttribute =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Invalid attribute for tag");
            when SPARK_XML.SSInvalidTag =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Invalid tag");
            when SPARK_XML.SSNoSuchTag =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "No such tag defined in schema");
            when SPARK_XML.SSTagNotFound =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt,
                                        "Could not find instance of the tag to close");
            when SPARK_XML.SSOK =>  -- We should never get to this case
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Everything is fine");
         end case;
      end if;

   end HandleError;


   procedure HandleSchemaError (Success : in Boolean;
                                Msg     : in String)
   --# derives null from Msg,
   --#                   Success;
   is
      --# hide HandleSchemaError;
   begin
      if not Success then
         SPARK_XML.PrintWorkingState (Schema, SchemaState);
         SystemErrors.FatalError (SystemErrors.XMLSchemaError, Msg);
      end if;
   end HandleSchemaError;

   function FilterString (Str : in EStrings.T) return EStrings.T
   is
   begin
      return SPARK_XML.FilterString (Str);
   end FilterString;

   function FilterLongString (Str : in ELStrings.T) return ELStrings.T
   is
   begin
      return SPARK_XML.FilterLongString (Str);
   end FilterLongString;



   --------------------
   -- Initialisation --
   --------------------

   -- Build the dictionary, then create the schema representation for SPARK_XML

   procedure Init
   --# global out FileDepth;
   --#        out Schema;
   --#        out SchemaState;
   --#        out SectionDepths;
   --#        out TagIDs;
   --#        out TagStrings;
   --# derives FileDepth,
   --#         Schema,
   --#         SchemaState,
   --#         SectionDepths,
   --#         TagIDs,
   --#         TagStrings    from ;
   is

      -- Load the tag definitions in to the Schema
      procedure LoadTags
      --# global in out Schema;
      --#           out TagIDs;
      --#           out TagStrings;
      --# derives Schema,
      --#         TagIDs     from Schema &
      --#         TagStrings from ;
      is

         TmpTag : SPARK_XML.TagID;

      begin
         TagStrings := TagStringArray'(MTBrief_Justifications =>   EStrings.Copy_String (Str => "brief_justifications"),
                                       MTCommandline =>            EStrings.Copy_String (Str => "commandline"),
                                       MTCompiler_Data =>          EStrings.Copy_String (Str => "compiler_data"),
                                       MTCompiler_Item =>          EStrings.Copy_String (Str => "compiler_item"),
                                       MTCyclic_Requirements =>    EStrings.Copy_String (Str => "cyclic_requirements"),
                                       MTFile =>                   EStrings.Copy_String (Str => "file"),
                                       MTFilename =>               EStrings.Copy_String (Str => "filename"),
                                       MTFull_Justification =>     EStrings.Copy_String (Str => "full_justification"),
                                       MTFull_Justifications =>    EStrings.Copy_String (Str => "full_justifications"),
                                       MTIndex =>                  EStrings.Copy_String (Str => "index"),
                                       MTIndexes =>                EStrings.Copy_String (Str => "indexes"),
                                       MTJustifications_Section => EStrings.Copy_String (Str => "justifications_section"),
                                       MTListing =>                EStrings.Copy_String (Str => "listing"),
                                       MTMessage =>                EStrings.Copy_String (Str => "message"),
                                       MTMessages =>               EStrings.Copy_String (Str => "messages"),
                                       MTMetafile =>               EStrings.Copy_String (Str => "metafile"),
                                       MTMetafiles =>              EStrings.Copy_String (Str => "metafiles"),
                                       MTOption =>                 EStrings.Copy_String (Str => "option"),
                                       MTPrologue =>               EStrings.Copy_String (Str => "prologue"),
                                       MTReport =>                 EStrings.Copy_String (Str => "report"),
                                       MTResults =>                EStrings.Copy_String (Str => "results"),
                                       MTScope =>                  EStrings.Copy_String (Str => "scope"),
                                       MTSuppressed =>             EStrings.Copy_String (Str => "suppressed"),
                                       MTSuppressedPragma =>       EStrings.Copy_String (Str => "pragma"),
                                       MTSymbol =>                 EStrings.Copy_String (Str => "symbol"),
                                       MTTarget_Config =>          EStrings.Copy_String (Str => "target_config"),
                                       MTUnit =>                   EStrings.Copy_String (Str => "unit"),
                                       MTUnits_Notfound =>         EStrings.Copy_String (Str => "units_notfound"),
                                       MTWarnings_Config =>        EStrings.Copy_String (Str => "warnings_config"),
                                       MTUnitsInFile =>            EStrings.Copy_String (Str => "units"));

         TagIDs := TagIDArray'(others => SPARK_XML.NullTag);

         for I in MyTag loop
            SPARK_XML.AddTag (Schema, TagStrings (I), TmpTag);
            TagIDs (I) := TmpTag;
            if SPARK_XML.IsNullTag (TmpTag) then
               -- Run out of room, so generate an error.
               HandleSchemaError (False, "Failure adding tag");
               exit;
            end if;
         end loop;

      end LoadTags;


      -- Load the attribute definitions
      procedure LoadAttributes
      --# global in     TagIDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     TagIDs;
      is
         TmpAttrib : SPARK_XML.AttributeID;
         TmpSuccess : Boolean;

         type Attribute is record
            ID : SPARK_XML.TagID;
            Name : EStrings.T;
            Typ : SPARK_XML.AttributeType;
            Req : Boolean;
         end record;

         -- Set TotalAttributes to the total number of attributes to be added
         -- to tags in the Schema.
         TotalAttributes : constant Integer := 25;

         type AttributeIndex is range 1 .. TotalAttributes;
         type AttributeList is array (AttributeIndex) of Attribute;
         Attributes : AttributeList;

      begin

         -- This table encodes the legal attributes of each XML element, as specified
         -- in the SPARKReport Schema Description in sparkreport.xsd
         Attributes :=
           AttributeList'(Attribute'(TagIDs (MTReport),      EStrings.Copy_String (Str => "version"),  SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTSymbol),      EStrings.Copy_String (Str => "id"),       SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTOption),      EStrings.Copy_String (Str => "id"),       SPARK_XML.ATString,  False),  -- Remove?

                          Attribute'(TagIDs (MTUnit),        EStrings.Copy_String (Str => "name"),     SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTUnit),        EStrings.Copy_String (Str => "type"),     SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTUnit),        EStrings.Copy_String (Str => "status"),   SPARK_XML.ATString,  False),

                          Attribute'(TagIDs (MTMessage),     EStrings.Copy_String (Str => "class"),    SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTMessage),     EStrings.Copy_String (Str => "code"),     SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTMessage),     EStrings.Copy_String (Str => "line"),     SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTMessage),     EStrings.Copy_String (Str => "offset"),   SPARK_XML.ATInteger, True),

                          Attribute'(TagIDs (MTFull_Justification), EStrings.Copy_String (Str => "class"),      SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTFull_Justification), EStrings.Copy_String (Str => "code"),       SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTFull_Justification), EStrings.Copy_String (Str => "line_from"),  SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTFull_Justification), EStrings.Copy_String (Str => "line_to"),    SPARK_XML.ATString, True),
                          Attribute'(TagIDs (MTFull_Justification), EStrings.Copy_String (Str => "match_no"),   SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTFull_Justification), EStrings.Copy_String (Str => "match_line"), SPARK_XML.ATInteger, True),

                          Attribute'(TagIDs (MTBrief_Justifications), EStrings.Copy_String (Str => "matched"),   SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTBrief_Justifications), EStrings.Copy_String (Str => "unmatched"), SPARK_XML.ATInteger, False),

                          Attribute'(TagIDs (MTMetafile),    EStrings.Copy_String (Str => "name"),     SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTFile),        EStrings.Copy_String (Str => "unit"),     SPARK_XML.ATString,  False),
                          Attribute'(TagIDs (MTFile),        EStrings.Copy_String (Str => "type"),     SPARK_XML.ATString,  False),
                          Attribute'(TagIDs (MTFile),        EStrings.Copy_String (Str => "filename"), SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTCompiler_Item), EStrings.Copy_String (Str => "item"),   SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTScope),       EStrings.Copy_String (Str => "id"),       SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTScope),       EStrings.Copy_String (Str => "type"),     SPARK_XML.ATString,  True)
                         );

         for I in AttributeIndex loop
            --# accept Flow, 10, TmpAttrib, "Expected ineffective assignment to TmpAttrib";
            SPARK_XML.AddAttributeToTag (Schema,  -- Expect ineffective statement
                                         Attributes (I).ID,
                                         Attributes (I).Name,
                                         Attributes (I).Typ,
                                         Attributes (I).Req,
                                         TmpAttrib,
                                         TmpSuccess);
            --# end accept;
            HandleSchemaError (TmpSuccess, "Failure adding attribute to tag");
            exit when not TmpSuccess;
         end loop;

      --# accept Flow, 33, TmpAttrib, "Expected to be neither referenced nor exported";
      end LoadAttributes;  -- Expect "TmpAttrib neither referenced nor exported"



      -- Load the hierarchy definitions
      procedure BuildHierarchy
      --# global in     TagIDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     TagIDs;
      is
         type TagRel is record
            Parent : SPARK_XML.TagID;
            Child : SPARK_XML.TagID;
            Required : Boolean;
         end record;

         MaxRelations : constant Integer := 37;

         subtype RelIndex is Integer range 1 .. MaxRelations;

         type RelArray is array (RelIndex) of TagRel;
         Relations : RelArray;

         TmpSuccess : Boolean;

      begin

         -- This table encodes the legal nesting of XML elements, as specified
         -- in the SPARKReport Schema Description in sparkreport.xsd
         Relations :=
           RelArray'(
                     TagRel'(SPARK_XML.NullTag,              TagIDs (MTReport),              False),

                     TagRel'(TagIDs (MTScope),               TagIDs (MTScope),               False),
                     TagRel'(TagIDs (MTScope),               TagIDs (MTMessage),             False),

                     TagRel'(TagIDs (MTMessage),             TagIDs (MTSymbol),              False),

                     TagRel'(TagIDs (MTReport),              TagIDs (MTPrologue),            False),
                     TagRel'(TagIDs (MTReport),              TagIDs (MTResults),             False),

                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTCommandline),         False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTIndexes),             False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTMetafiles),           False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTWarnings_Config),     False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTCompiler_Data),       False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTTarget_Config),       False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTUnits_Notfound),      False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTCyclic_Requirements), False),

                     TagRel'(TagIDs (MTCommandline),         TagIDs (MTFilename),            False),
                     TagRel'(TagIDs (MTCommandline),         TagIDs (MTOption),              False),

                     TagRel'(TagIDs (MTCompiler_Data),       TagIDs (MTCompiler_Item),       False),

                     TagRel'(TagIDs (MTCyclic_Requirements), TagIDs (MTUnit),                False),

                     TagRel'(TagIDs (MTIndexes),             TagIDs (MTIndex),               False),

                     TagRel'(TagIDs (MTTarget_Config),       TagIDs (MTFilename),            False),
                     TagRel'(TagIDs (MTTarget_Config),       TagIDs (MTMessages),            False),

                     TagRel'(TagIDs (MTMessages),            TagIDs (MTMessage),             False),

                     TagRel'(TagIDs (MTMetafiles),           TagIDs (MTMetafile),            False),

                     TagRel'(TagIDs (MTMetafile),            TagIDs (MTMetafile),            False),
                     TagRel'(TagIDs (MTMetafile),            TagIDs (MTFilename),            False),

                     TagRel'(TagIDs (MTUnits_Notfound),      TagIDs (MTUnit),                False),

                     TagRel'(TagIDs (MTWarnings_Config),     TagIDs (MTSuppressed),          False),
                     TagRel'(TagIDs (MTWarnings_Config),     TagIDs (MTSuppressedPragma),    False),

                     TagRel'(TagIDs (MTFile),                TagIDs (MTUnitsInFile),            False),
                     TagRel'(TagIDs (MTFile),                TagIDs (MTScope),                  False),
                     TagRel'(TagIDs (MTFile),                TagIDs (MTMessage),                False),
                     TagRel'(TagIDs (MTFile),                TagIDs (MTJustifications_Section), False),

                     TagRel'(TagIDs (MTJustifications_Section), TagIDs (MTFull_Justifications),  False),
                     TagRel'(TagIDs (MTJustifications_Section), TagIDs (MTBrief_Justifications), False),

                     TagRel'(TagIDs (MTFull_Justifications), TagIDs (MTFull_Justification),  False),

                     TagRel'(TagIDs (MTUnitsInFile),         TagIDs (MTUnit),                False),

                     TagRel'(TagIDs (MTResults),             TagIDs (MTFile),                False));


         for I in RelIndex loop
            SPARK_XML.AddChildTag (Schema,
                                   Relations (I).Parent,
                                   Relations (I).Child,
                                   Relations (I).Required,
                                   TmpSuccess);
            HandleSchemaError (TmpSuccess, "Failure loading tag hierarchy");
            exit when not TmpSuccess;
         end loop;

      end BuildHierarchy;


   begin
      SectionDepths := SectionDepthArray'(others => SPARK_XML.TagDepth'First);
      FileDepth := SPARK_XML.TagDepth'First;

      SPARK_XML.InitSchema (Schema);
      SPARK_XML.InitSchemaState (SchemaState);

      LoadTags;
      LoadAttributes;
      BuildHierarchy;

   end Init;



   ------------------------------------------------------------------------
   -- Tag producing functions                                            --
   ------------------------------------------------------------------------

   ------------------------------------------------------------------------
   -- Producers for simple container tags.                               --
   -- These have no attributes of their own and only contain other tags. --
   -- The only exception is the Listing tag, which contains large        --
   -- amounts of text, and is also included.                             --
   ------------------------------------------------------------------------

   procedure StartSection (Section : in Sections;
                           Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SectionDepths;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState,
   --#         SectionDepths     from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Section,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Section,
   --#                                TagIDs;
   is
      Tag : SPARK_XML.TagID;
      Status : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

      XML : EStrings.T;

   begin

      case Section is
         when SReport             => Tag := TagIDs (MTReport);
         when SPrologue           => Tag := TagIDs (MTPrologue);
         when SCommandline        => Tag := TagIDs (MTCommandline);
         when SCompilerData       => Tag := TagIDs (MTCompiler_Data);
         when SCyclicRequirements => Tag := TagIDs (MTCyclic_Requirements);
         when SIndexes            => Tag := TagIDs (MTIndexes);
         when STargetConfig       => Tag := TagIDs (MTTarget_Config);
         when SMessages           => Tag := TagIDs (MTMessages);
         when SMetaFiles          => Tag := TagIDs (MTMetafiles);
         when SUnitsNotFound      => Tag := TagIDs (MTUnits_Notfound);
         when SWarningsConfig     => Tag := TagIDs (MTWarnings_Config);
         when SResults            => Tag := TagIDs (MTResults);
         when SListing            => Tag := TagIDs (MTListing);
         when SUnitsInFile        => Tag := TagIDs (MTUnitsInFile);
         when SJustifications     => Tag := TagIDs (MTJustifications_Section);
         when SFullJustifications => Tag := TagIDs (MTFull_Justifications);
      end case;

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => Tag,
                                    Status      => Status);

      HandleError (Status => Status);

      if Section = SReport then
         SPARK_XML.AddAttributeStr (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    Name        => EStrings.Copy_String (Str => "version"),
                                    Value       => EStrings.Copy_String (Str => Schema_Version),
                                    Status      => Status);
         HandleError (Status => Status);
      end if;


      SPARK_XML.OutputOpeningTag (Schema      => Schema,
                                  SchemaState => SchemaState,
                                  XML         => XML,
                                  Depth       => Depth, -- Record the section depth.
                                  Status      => Status);
      SectionDepths (Section) := Depth;

      HandleError (Status => Status);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

   end StartSection;


   procedure EndSection (Section : in Sections;
                         Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     SectionDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Section,
   --#                                SectionDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Section,
   --#                                SectionDepths;
   is
      Status : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin

      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => SectionDepths (Section),
                          XML         => XML,
                          Status      => Status);
      HandleError (Status => Status);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

   end EndSection;

   -- Simple tags, ones whose opening and closing tags are generated
   -- at the same time, with simple text contents
   -- E.g. <filename>file.txt</filename>
   procedure Simple_Tag (Tag      : in     SPARK_XML.TagID;
                         Contents : in out EStrings.T)
   --# global in     Schema;
   --#        in out SchemaState;
   --# derives Contents,
   --#         SchemaState from *,
   --#                          Schema,
   --#                          SchemaState,
   --#                          Tag;
   is
      Acc_XML, Tmp_XML : EStrings.T;  -- XML accumulator and temp variable.
      Depth            : SPARK_XML.TagDepth;
      Status           : SPARK_XML.SchemaStatus;
   begin
      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => Tag,
                                    Status      => Status);
      HandleError (Status => Status);
      SPARK_XML.OutputOpeningTag (Schema      => Schema,
                                  SchemaState => SchemaState,
                                  XML         => Acc_XML,
                                  Depth       => Depth, -- Record the section depth.
                                  Status      => Status);
      HandleError (Status => Status);
      EStrings.Append_Examiner_String (E_Str1 => Acc_XML,
                                       E_Str2 => FilterString (Contents));
      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => Depth,
                          XML         => Tmp_XML,
                          Status      => Status);
      EStrings.Append_Examiner_String (E_Str1 => Acc_XML,
                                       E_Str2 => Tmp_XML);
      HandleError (Status => Status);
      Contents := Acc_XML;
   end Simple_Tag;

   procedure Filename (Plain_Output : in     Boolean;
                       File         : in out EStrings.T)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --# derives File        from *,
   --#                          Plain_Output,
   --#                          Schema,
   --#                          SchemaState,
   --#                          TagIDs &
   --#         SchemaState from *,
   --#                          Schema,
   --#                          TagIDs;
   is
   begin
      if Plain_Output then
         File := EStrings.Lower_Case (E_Str => FileSystem.JustFile (File, True));
      end if;
      Simple_Tag (Tag      => TagIDs (MTFilename),
                  Contents => File);
   end Filename;

   procedure Index (Plain_Output : in     Boolean;
                    Idx          : in out EStrings.T)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --# derives Idx         from *,
   --#                          Plain_Output,
   --#                          Schema,
   --#                          SchemaState,
   --#                          TagIDs &
   --#         SchemaState from *,
   --#                          Schema,
   --#                          TagIDs;
   is
   begin
      if Plain_Output then
         Idx := FileSystem.JustFile (Idx, True);
      end if;
      Simple_Tag (Tag      => TagIDs (MTIndex),
                  Contents => Idx);
   end Index;

   procedure Option (Opt : in out EStrings.T)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --# derives Opt,
   --#         SchemaState from *,
   --#                          Schema,
   --#                          SchemaState,
   --#                          TagIDs;
   is
   begin
      Simple_Tag (Tag      => TagIDs (MTOption),
                  Contents => Opt);
   end Option;

   procedure Suppressed (Item : in out EStrings.T)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --# derives Item,
   --#         SchemaState from *,
   --#                          Schema,
   --#                          SchemaState,
   --#                          TagIDs;
   is
   begin
      Simple_Tag (Tag      => TagIDs (MTSuppressed),
                  Contents => Item);
   end Suppressed;

   procedure SuppressedPragma (Item : in out EStrings.T)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --# derives Item,
   --#         SchemaState from *,
   --#                          Schema,
   --#                          SchemaState,
   --#                          TagIDs;
   is
   begin
      Simple_Tag (Tag      => TagIDs (MTSuppressedPragma),
                  Contents => Item);
   end SuppressedPragma;

   ---------------------------------------------
   -- More complex tags that have attributes. --
   ---------------------------------------------

   procedure OpenCompilerItem (Item   : in     EStrings.T;
                               Depth  :    out SPARK_XML.TagDepth;
                               Report : in     SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives Depth,
   --#         SchemaState       from Item,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;

  is
      AccXML : EStrings.T;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTCompiler_Item),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "item"),
                                 Value       => Item,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.OutputOpeningTag (Schema      => Schema,
                                  SchemaState => SchemaState,
                                  XML         => AccXML,
                                  Depth       => Depth, -- Record the section depth.
                                  Status      => Stat);
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => AccXML);  -- Output the opening tag
   end OpenCompilerItem;


   procedure CloseCompilerItem (Depth  : in     SPARK_XML.TagDepth;
                                Report : in     SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Depth &
   --#         SPARK_IO.File_Sys from *,
   --#                                Depth,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState;
   is
      AccXML : EStrings.T;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => Depth,
                          XML         => AccXML,
                          Status      => Stat);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => AccXML);

   end CloseCompilerItem;

   procedure CompilerItem (Item   : in EStrings.T;
                           Val    : in EStrings.T;
                           Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Item,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Val;
   is
      Depth : SPARK_XML.TagDepth;
   begin

      OpenCompilerItem (Item,
                        Depth,
                        Report);

      EStrings.Put_String (File  => Report,
                           E_Str => FilterString (Val));  -- Output the value


      CloseCompilerItem (Depth,
                         Report);
   end CompilerItem;


   procedure LongCompilerItem (Item   : in EStrings.T;
                               Val    : in ELStrings.T;
                               Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Item,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Val;
   is
      Depth : SPARK_XML.TagDepth;
   begin

      OpenCompilerItem (Item,
                        Depth,
                        Report);

      ELStrings.Put_String (File  => Report,
                            E_Str => FilterLongString (Val));  -- Output the value


      CloseCompilerItem (Depth,
                         Report);

   end LongCompilerItem;



   procedure Unit (Name   : in EStrings.T;
                   Typ    : in EStrings.T;
                   Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Name,
   --#                                Schema,
   --#                                TagIDs,
   --#                                Typ &
   --#         SPARK_IO.File_Sys from *,
   --#                                Name,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Typ;
   is
      AccXML, TmpXML : EStrings.T;  -- XML accumulator and temp variable.
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTUnit),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "name"),
                                 Value       => Name,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "type"),
                                 Value       => Typ,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.OutputOpeningTag (Schema      => Schema,
                                  SchemaState => SchemaState,
                                  XML         => AccXML,
                                  Depth       => Depth, -- Record the section depth.
                                  Status      => Stat);
      HandleError (Status => Stat);


      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => Depth,
                          XML         => TmpXML,
                          Status      => Stat);

      EStrings.Append_Examiner_String (E_Str1 => AccXML,
                                       E_Str2 => TmpXML);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => AccXML);

   end Unit;



   procedure AdaUnit (Name       : in EStrings.T;
                      Typ        : in EStrings.T;
                      UnitStatus : in EStrings.T;
                      Report     : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Name,
   --#                                Schema,
   --#                                TagIDs,
   --#                                Typ,
   --#                                UnitStatus &
   --#         SPARK_IO.File_Sys from *,
   --#                                Name,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Typ,
   --#                                UnitStatus;
   is
      AccXML : EStrings.T;  -- XML accumulator.
      Depth : SPARK_XML.TagDepth;    -- Dummy variable.
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTUnit),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "name"),
                                 Value       => Name,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "type"),
                                 Value       => Typ,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "status"),
                                 Value       => UnitStatus,
                                 Status      => Stat);
      HandleError (Status => Stat);


      SPARK_XML.OutputOpeningTag (Schema      => Schema,  -- Expect inneffective assignment to Depth
                                  SchemaState => SchemaState,
                                  XML         => AccXML,
                                  Depth       => Depth, -- Record the section depth.
                                  Status      => Stat);
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => AccXML);

      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => Depth,
                          XML         => AccXML,
                          Status      => Stat);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => AccXML);

   end AdaUnit;  -- Depth is neither referenced or exported.





   ---------------------------
   -- Message construction. --
   ---------------------------

   procedure StartMessage (Class  : in EStrings.T;
                           Code   : in Integer;
                           Line   : in Integer;
                           Offset : in Integer;
                           Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Class,
   --#                                Code,
   --#                                Line,
   --#                                Offset,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Class,
   --#                                Code,
   --#                                Line,
   --#                                Offset,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin

      SPARK_XML.InitOpeningTagNOCHECK (SchemaState => SchemaState,
                                       TID         => TagIDs (MTMessage),
                                       Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "class"),
                                 Value       => Class,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "code"),
                                 Code,
                                 Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "line"),
                                 Line,
                                 Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "offset"),
                                 Offset,
                                 Stat);
      HandleError (Status => Stat);

      --# accept Flow, 10, Depth, "Expected ineffective assignment to Depth";
      SPARK_XML.OutputOpeningTag (Schema      => Schema, -- Expect inneffective assignment
                                  SchemaState => SchemaState,
                                  XML         => XML,
                                  Depth       => Depth,
                                  Status      => Stat);
      --# end accept;
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

      --# accept Flow, 33, Depth, "Expected Depth to be neither referenced nor exported";
   end StartMessage;  -- Expect "Depth not referenced or exported"



   procedure Symbol (Sym    : in EStrings.T;
                     SymNum : in Integer;
                     Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                SymNum,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Sym,
   --#                                SymNum,
   --#                                TagIDs;
   is
      AccXML, TmpXML : EStrings.T;  -- XML accumulator and temp variable.
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTSymbol),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "id"),
                                 SymNum,
                                 Stat);
      HandleError (Status => Stat);

      SPARK_XML.OutputOpeningTag (Schema      => Schema,
                                  SchemaState => SchemaState,
                                  XML         => AccXML,
                                  Depth       => Depth, -- Record the section depth.
                                  Status      => Stat);
      HandleError (Status => Stat);

      -- Add the symbol contents.
      EStrings.Append_Examiner_String (E_Str1 => AccXML,
                                       E_Str2 => FilterString (Sym));

      -- Close the tag.
      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => Depth,
                          XML         => TmpXML,
                          Status      => Stat);
      HandleError (Status => Stat);
      EStrings.Append_Examiner_String (E_Str1 => AccXML,
                                       E_Str2 => TmpXML);

      EStrings.Put_String (File  => Report,
                           E_Str => AccXML);

   end Symbol;


   -- EndMessage will close the open message (of which there will be one
   -- in the hierarchy since only a <symbol> can be placed in a <message>
   procedure EndMessage (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.CloseTagByID (Schema,
                              SchemaState,
                              TagIDs (MTMessage),
                              XML,
                              Stat);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

   end EndMessage;


   --------------------------
   -- Metafile contruction --
   --------------------------

   procedure StartMetaFile (Name   : in     EStrings.T;
                            Report : in     SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Name,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Name,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
      Id : SPARK_XML.TagDepth;
   begin

      -- Generate the XML

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTMetafile),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "name"),
                                 Value       => Name,
                                 Status      => Stat);
      HandleError (Status => Stat);

      --# accept Flow, 10, Id, "Expected ineffective assignment to Id";
      SPARK_XML.OutputOpeningTag (Schema      => Schema, --Expect inneffective assignment to Id
                                  SchemaState => SchemaState,
                                  XML         => XML,
                                  Depth       => Id,
                                  Status      => Stat);
      --# end accept;
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

      --# accept Flow, 33, Id, "Expected Id to be neither referenced nor exported";
   end StartMetaFile;  -- Expect "id neither referenced nor exported"



   procedure EndMetaFile (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin

      SPARK_XML.CloseTopTagByID (Schema,
                                 SchemaState,
                                 TagIDs (MTMetafile),
                                 XML,
                                 Stat);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

   end EndMetaFile;


   ------------------------------
   -- Results Section contents --
   ------------------------------
   procedure Start_File (Plain_Output : in Boolean;
                         F_Name       : in EStrings.T;
                         Report       : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#           out FileDepth;
   --# derives FileDepth,
   --#         SchemaState       from F_Name,
   --#                                Plain_Output,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                F_Name,
   --#                                Plain_Output,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML  : EStrings.T;
   begin
      SPARK_XML.InitOpeningTagNOCHECK (SchemaState => SchemaState,
                                       TID         => TagIDs (MTFile),
                                       Status      => Stat);
      HandleError (Status => Stat);
      -- Add the attributes
      if Plain_Output then
         SPARK_XML.AddAttributeStr (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    Name        => EStrings.Copy_String (Str => "filename"),
                                    Value       => EStrings.Lower_Case (E_Str => FileSystem.JustFile (F_Name, True)),
                                    Status      => Stat);
      else
         SPARK_XML.AddAttributeStr (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    Name        => EStrings.Copy_String (Str => "filename"),
                                    Value       => F_Name,
                                    Status      => Stat);
      end if;
      HandleError (Status => Stat);
      SPARK_XML.OutputOpeningTag (Schema      => Schema,
                                  SchemaState => SchemaState,
                                  XML         => XML,
                                  Depth       => FileDepth, -- Record the depth.
                                  Status      => Stat);
      HandleError (Status => Stat);
      EStrings.Put_String (File  => Report,
                           E_Str => XML);
   end Start_File;

   procedure EndFile (Report  : in SPARK_IO.File_Type)
   --# global in     FileDepth;
   --#        in     Schema;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                FileDepth &
   --#         SPARK_IO.File_Sys from *,
   --#                                FileDepth,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin
      -- Close the tag.
      SPARK_XML.CloseTag (Schema      => Schema,
                          SchemaState => SchemaState,
                          Depth       => FileDepth,
                          XML         => XML,
                          Status      => Stat);
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

   end EndFile;

   --------------------------
   -- Justification output --
   --------------------------

   procedure BriefJustifications (Matched   : in Natural;
                                  Unmatched : in Natural;
                                  Report    : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Matched,
   --#                                Schema,
   --#                                TagIDs,
   --#                                Unmatched &
   --#         SPARK_IO.File_Sys from *,
   --#                                Matched,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Unmatched;
   is
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTBrief_Justifications),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "matched"),
                                 Matched,
                                 Stat);
      HandleError (Status => Stat);

      if Unmatched > 0 then
         SPARK_XML.AddAttributeInt (Schema,
                                    SchemaState,
                                    EStrings.Copy_String (Str => "unmatched"),
                                    Unmatched,
                                    Stat);

         HandleError (Status => Stat);
      end if;

      --# accept Flow, 10, Depth, "Expected ineffective assignment to Depth";
      SPARK_XML.OutputOpeningTag (Schema      => Schema, -- Expect inneffective assignment
                                  SchemaState => SchemaState,
                                  XML         => XML,
                                  Depth       => Depth,
                                  Status      => Stat);
      --# end accept;
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

      SPARK_XML.CloseTagByID (Schema,
                              SchemaState,
                              TagIDs (MTBrief_Justifications),
                              XML,
                              Stat);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

      --# accept Flow, 33, Depth, "Expected Depth to be neither referenced nor exported";
   end BriefJustifications;

   procedure StartFullJustification (Class     : in EStrings.T;
                                     Code      : in Integer;
                                     LineFrom  : in Integer;
                                     LineTo    : in EStrings.T;
                                     MatchNo   : in Integer;
                                     MatchLine : in Integer;
                                     Report    : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Class,
   --#                                Code,
   --#                                LineFrom,
   --#                                LineTo,
   --#                                MatchLine,
   --#                                MatchNo,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Class,
   --#                                Code,
   --#                                LineFrom,
   --#                                LineTo,
   --#                                MatchLine,
   --#                                MatchNo,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin

      SPARK_XML.InitOpeningTagByID (Schema      => Schema,
                                    SchemaState => SchemaState,
                                    TID         => TagIDs (MTFull_Justification),
                                    Status      => Stat);
      HandleError (Status => Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "class"),
                                 Value       => Class,
                                 Status      => Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "code"),
                                 Code,
                                 Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "line_from"),
                                 LineFrom,
                                 Stat);
      HandleError (Status => Stat);

      SPARK_XML.AddAttributeStr (Schema      => Schema,
                                 SchemaState => SchemaState,
                                 Name        => EStrings.Copy_String (Str => "line_to"),
                                 Value       => LineTo,
                                 Status      => Stat);

      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "match_no"),
                                 MatchNo,
                                 Stat);

      HandleError (Status => Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 EStrings.Copy_String (Str => "match_line"),
                                 MatchLine,
                                 Stat);

      HandleError (Status => Stat);

      --# accept Flow, 10, Depth, "Expected ineffective assignment to Depth";
      SPARK_XML.OutputOpeningTag (Schema      => Schema, -- Expect inneffective assignment
                                  SchemaState => SchemaState,
                                  XML         => XML,
                                  Depth       => Depth,
                                  Status      => Stat);
      --# end accept;
      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);

      --# accept Flow, 33, Depth, "Expected Depth to be neither referenced nor exported";
   end StartFullJustification;

   procedure EndFullJustification (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.CloseTagByID (Schema,
                              SchemaState,
                              TagIDs (MTFull_Justification),
                              XML,
                              Stat);

      HandleError (Status => Stat);

      EStrings.Put_String (File  => Report,
                           E_Str => XML);
   end EndFullJustification;

end XMLReport;
