-- $Id: vcg.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Cells;
with Debug;
with Declarations;
with ErrorHandler;
with EStrings;
with Fatal;
with File_Utils;
with FileSystem;
with Graph;
with LexTokenLists;
with ScreenEcho;
with SPARK_IO;

use type SPARK_IO.File_Status;
use type File_Utils.FileTypes;

package body VCG
is

   ------------------------------------------------------------------------------

   procedure ProduceVCs (VCGheap                   : in out Cells.Heap_Record;
                         StartNode                 : in     STree.SyntaxNode;
                         SubprogSym                : in     Dictionary.Symbol;
                         Scope                     : in     Dictionary.Scopes;
                         VCGOutputFile             : in     SPARK_IO.File_Type;
                         DPCOutputFile             : in     SPARK_IO.File_Type;
                         OutputFileName            : in     EStrings.T;
                         EndPosition               : in     LexTokenManager.Token_Position;
                         FlowHeap                  : in out Heap.HeapRecord;
                         SemanticErrorInSubprogram : in     Boolean;
                         DataFlowErrorInSubprogram : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#           out Graph.Table;
   --#           out StmtStack.S;
   --# derives Declarations.State,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         StmtStack.S,
   --#         VCGheap                   from CommandLineData.Content,
   --#                                        DataFlowErrorInSubprogram,
   --#                                        Declarations.State,
   --#                                        Dictionary.Dict,
   --#                                        FlowHeap,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        VCGheap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        DataFlowErrorInSubprogram,
   --#                                        Declarations.State,
   --#                                        Dictionary.Dict,
   --#                                        DPCOutputFile,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FlowHeap,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        VCGheap,
   --#                                        VCGOutputFile &
   --#         FlowHeap                  from *,
   --#                                        CommandLineData.Content,
   --#                                        DataFlowErrorInSubprogram,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        VCGheap &
   --#         Statistics.TableUsage     from *,
   --#                                        CommandLineData.Content,
   --#                                        DataFlowErrorInSubprogram,
   --#                                        Declarations.State,
   --#                                        Dictionary.Dict,
   --#                                        FlowHeap,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        VCGheap &
   --#         null                      from OutputFileName;
      is  separate;

   ------------------------------------------------------------------------------

   procedure FullSymbolName (Item      : in     Dictionary.Symbol;
                             Ancestors :    out LexTokenLists.Lists;
                             List      :    out LexTokenLists.Lists)
   --# global in Dictionary.Dict;
   --# derives Ancestors,
   --#         List      from Dictionary.Dict,
   --#                        Item;
   is
      ReversePrefixList,
      FullSymbolNameList : LexTokenLists.Lists;
      AncestorList       : LexTokenLists.Lists;
      PackSym            : Dictionary.Symbol;
      Scope              : Dictionary.Scopes;
      Lex_Token_Item     : LexTokenManager.Lex_String;
   begin
      ReversePrefixList := LexTokenLists.Null_List;
      Scope := Dictionary.GetScope (Item);
      loop
         exit when Dictionary.IsGlobalScope (Scope) or else
           Scope = Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard) or else
           Dictionary.IsPredefinedScope (Scope);
         LexTokenLists.Append (ReversePrefixList,
                               Dictionary.GetSimpleName (Dictionary.GetRegion (Scope)));
         Scope := Dictionary.GetEnclosingScope (Scope);
      end loop;

      FullSymbolNameList := LexTokenLists.Null_List;

      while LexTokenLists.Get_Length (List => ReversePrefixList) > 0 loop
         LexTokenLists.Pop (List => ReversePrefixList,
                            Item => Lex_Token_Item);
         LexTokenLists.Append (List => FullSymbolNameList,
                               Item => Lex_Token_Item);
      end loop;

      LexTokenLists.Append (FullSymbolNameList, Dictionary.GetSimpleName (Item));
      List := FullSymbolNameList;
      AncestorList := LexTokenLists.Null_List;
      PackSym := Dictionary.GetLibraryPackage (Dictionary.GetScope (Item));

      if PackSym /= Dictionary.GetPredefinedPackageStandard then
         ReversePrefixList := LexTokenLists.Null_List;
         loop
            PackSym := Dictionary.GetPackageParent (PackSym);
            exit when PackSym = Dictionary.NullSymbol;
            LexTokenLists.Append (ReversePrefixList,
                                  Dictionary.GetSimpleName (PackSym));
         end loop;

         while LexTokenLists.Get_Length (List => ReversePrefixList) > 0 loop
            LexTokenLists.Pop (List => ReversePrefixList,
                               Item => Lex_Token_Item);
            LexTokenLists.Append (List => AncestorList,
                                  Item => Lex_Token_Item);
         end loop;
      end if;
      Ancestors := AncestorList;

   end FullSymbolName;

   --------------------------------------------

   procedure ProduceOutputFiles (SubprogSym       : in     Dictionary.Symbol;
                                 VCGOutputFile    : in out SPARK_IO.File_Type;
                                 DPCOutputFile    : in out SPARK_IO.File_Type;
                                 DeclarationsFile : in out SPARK_IO.File_Type;
                                 RuleFile         : in out SPARK_IO.File_Type;
                                 OutputFileName   :    out EStrings.T;
                                 OK               :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives DeclarationsFile,
   --#         DPCOutputFile,
   --#         RuleFile,
   --#         VCGOutputFile     from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                SPARK_IO.FILE_SYS,
   --#                                SubprogSym &
   --#         Ok,
   --#         SPARK_IO.FILE_SYS from CommandLineData.Content,
   --#                                DeclarationsFile,
   --#                                Dictionary.Dict,
   --#                                DPCOutputFile,
   --#                                LexTokenManager.State,
   --#                                RuleFile,
   --#                                SPARK_IO.FILE_SYS,
   --#                                SubprogSym,
   --#                                VCGOutputFile &
   --#         OutputFileName    from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                SPARK_IO.FILE_SYS,
   --#                                SubprogSym;
   is
      VCG_Extension : constant String := "vcg";
      DPC_Extension : constant String := "dpc";
      FDL_Extension : constant String := "fdl";
      RLS_Extension : constant String := "rls";

      LocalOK          : Boolean;

      UnitName         : LexTokenLists.Lists;
      AncestorName     : LexTokenLists.Lists;
      FileName         : EStrings.T;

      --------------------------------------------

      procedure BuildFileNameNest
      --# global in     AncestorName;
      --#        in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in     UnitName;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out FileName;
      --#           out LocalOk;
      --# derives FileName          from AncestorName,
      --#                                LexTokenManager.State,
      --#                                UnitName &
      --#         LocalOk,
      --#         SPARK_IO.FILE_SYS from AncestorName,
      --#                                CommandLineData.Content,
      --#                                LexTokenManager.State,
      --#                                SPARK_IO.FILE_SYS,
      --#                                UnitName;
      is
         Pos         : LexTokenLists.Lengths;
         Str         : EStrings.T;
         AdjustedDir : EStrings.T;
      begin
         LocalOK := True;
         FileName := EStrings.Empty_String;
         if LexTokenLists.Get_Length (List => UnitName) /= 1 then -- is not a main program so build nested dirs
            EStrings.Append_Examiner_String (E_Str1 => FileName,
                                             E_Str2 => FileSystem.StartOfDirectory);

            Pos := 1;
            loop
               exit when Pos > LexTokenLists.Get_Length (List => AncestorName);
               Str := LexTokenManager.Lex_String_To_String
                 (Lex_Str => LexTokenLists.Get_Element (List => AncestorName,
                                                        Pos  => Pos));
               if EStrings.Get_Length (E_Str => Str) > FileSystem.MaxFileNameLength - 1 then
                  Str := EStrings.Section (E_Str     => Str,
                                           Start_Pos => 1,
                                           Length    => FileSystem.MaxFileNameLength - 1);
               end if;

               -- Note that directories for VCG files are always created
               -- using lower-case names on all platforms.
               EStrings.Append_Examiner_String (E_Str1 => FileName,
                                                E_Str2 => EStrings.Lower_Case (E_Str => Str));
               EStrings.Append_String (E_Str => FileName,
                                       Str   => "_");
               if LocalOK then
                  -- If the user has asked for an alterative output directory, then start
                  -- there, otherwise start at current working directory
                  AdjustedDir := FileName;
                  CommandLineData.Normalize_FileName_To_Output_Directory (AdjustedDir);

                  FileSystem.IdempotentCreateSubdirectory (AdjustedDir, LocalOK);
               end if;
               EStrings.Append_Examiner_String (E_Str1 => FileName,
                                                E_Str2 => FileSystem.BetweenDirectories);
               Pos := Pos + 1;
            end loop;

            Pos := 1;
            loop
               Str := LexTokenManager.Lex_String_To_String
                 (Lex_Str => LexTokenLists.Get_Element (List => UnitName,
                                                        Pos  => Pos));
               if EStrings.Get_Length (E_Str => Str) > FileSystem.MaxFileNameLength then
                  Str := EStrings.Section (E_Str     => Str,
                                           Start_Pos => 1,
                                           Length    => FileSystem.MaxFileNameLength);
               end if;

               -- Note that directories for VCG files are always created
               -- using lower-case names on all platforms.
               EStrings.Append_Examiner_String (E_Str1 => FileName,
                                                E_Str2 => EStrings.Lower_Case (E_Str => Str));

               -- If the user has asked for an alterative output directory, then start
               -- there, otherwise start at current working directory. Note that we
               -- must preserve the case of the user-specified directory.
               AdjustedDir := FileName;
               CommandLineData.Normalize_FileName_To_Output_Directory (AdjustedDir);

               FileSystem.IdempotentCreateSubdirectory (AdjustedDir, LocalOK);

               exit when Pos = LexTokenLists.Get_Length (List => UnitName) - 1;
               EStrings.Append_Examiner_String (E_Str1 => FileName,
                                                E_Str2 => FileSystem.BetweenDirectories);
               Pos := Pos + 1;
            end loop;
            EStrings.Append_Examiner_String (E_Str1 => FileName,
                                             E_Str2 => FileSystem.EndOfPath);
         end if;

         Str := LexTokenManager.Lex_String_To_String
           (Lex_Str => LexTokenLists.Get_Element
              (List => UnitName,
               Pos  => LexTokenLists.Get_Length (List => UnitName)));
         if EStrings.Get_Length (E_Str => Str) > FileSystem.MaxFileNameLength then
            Str := EStrings.Section (E_Str     => Str,
                                     Start_Pos => 1,
                                     Length    => FileSystem.MaxFileNameLength);
         end if;
         EStrings.Append_Examiner_String (E_Str1 => FileName,
                                          E_Str2 => EStrings.Lower_Case (E_Str => Str));
         LocalOK := LocalOK and then
           EStrings.Get_Length (E_Str => FileName) < EStrings.Max_String_Length - 4;
      end BuildFileNameNest;

      --------------------------------------------

      procedure PutSubprogramName (File     : in SPARK_IO.File_Type;
                                   Sym      : in Dictionary.Symbol;
                                   FileType : in File_Utils.FileTypes)
      --# global in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     UnitName;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                FileType,
      --#                                LexTokenManager.State,
      --#                                Sym,
      --#                                UnitName;
      is
         HeadLine : EStrings.T;
         PageWidth : constant Natural := 78;
      begin
         if Dictionary.IsFunction (Sym) then
            HeadLine := EStrings.Copy_String (Str => "function ");
         elsif Dictionary.IsProcedure (Sym) then
            HeadLine := EStrings.Copy_String (Str => "procedure ");
         elsif Dictionary.IsTaskType (Sym) then
            HeadLine := EStrings.Copy_String (Str => "task body ");
         else
            HeadLine := EStrings.Copy_String (Str => "initialization of ");
         end if;
         EStrings.Append_Examiner_String
           (E_Str1 => HeadLine,
            E_Str2 => LexTokenLists.Token_List_To_String (Token_List => UnitName));
         if (EStrings.Get_Length (E_Str => HeadLine) + 1) < PageWidth then
            SPARK_IO.Set_Col (File,
                              (PageWidth - EStrings.Get_Length (E_Str => HeadLine)) / 2);
         end if;
         if FileType = File_Utils.DecFile then
            SPARK_IO.Put_Char (File, '{');
            EStrings.Put_String (File  => File,
                                 E_Str => HeadLine);
            SPARK_IO.Put_Char (File, '}');
            SPARK_IO.New_Line (File, 1);
         elsif FileType = File_Utils.RuleFile then
            SPARK_IO.Put_String (File, "/*", 0);
            EStrings.Put_String (File  => File,
                                 E_Str => HeadLine);
            SPARK_IO.Put_String (File, "*/", 0);
            SPARK_IO.New_Line (File, 1);
         else
            EStrings.Put_Line (File  => File,
                               E_Str => HeadLine);
         end if;
         SPARK_IO.New_Line (File, 2);
      end PutSubprogramName;

      ---------------------------------------------------------

      procedure ProduceVCGOutputFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out VCGOutputFile;
      --#           out LocalOk;
      --#           out OutputFileName;
      --# derives LocalOk,
      --#         VCGOutputFile     from FileName,
      --#                                SPARK_IO.FILE_SYS,
      --#                                VCGOutputFile &
      --#         OutputFileName    from FileName &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                FileName,
      --#                                LexTokenManager.State,
      --#                                SubprogSym,
      --#                                UnitName,
      --#                                VCGOutputFile;
      is
         Success : SPARK_IO.File_Status;
      begin
         OutputFileName := FileName;
         FileSystem.CheckExtension
           (Fn  => OutputFileName,
            Ext => EStrings.Copy_String (Str => VCG_Extension));

         EStrings.Create (File         => VCGOutputFile,
                          Name_Of_File => OutputFileName,
                          Form_Of_File => "",
                          Status       => Success);

         LocalOK := Success = SPARK_IO.Ok;
         if LocalOK then
            File_Utils.PrintAHeader (VCGOutputFile,
                                     "Semantic Analysis of SPARK Text",
                                     File_Utils.OtherFile);
            PutSubprogramName (VCGOutputFile, SubprogSym, File_Utils.OtherFile);
         end if;
      end ProduceVCGOutputFile;

      ---------------------------------------------------------

      procedure ProduceDPCOutputFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out DPCOutputFile;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out LocalOk;
      --# derives DPCOutputFile,
      --#         LocalOk           from DPCOutputFile,
      --#                                FileName,
      --#                                SPARK_IO.FILE_SYS &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                DPCOutputFile,
      --#                                FileName,
      --#                                LexTokenManager.State,
      --#                                SubprogSym,
      --#                                UnitName;
      is
         Success           : SPARK_IO.File_Status;
         DPCOutputFileName : EStrings.T;
      begin
         DPCOutputFileName := FileName;
         FileSystem.CheckExtension
           (Fn  => DPCOutputFileName,
            Ext => EStrings.Copy_String (Str => DPC_Extension));

         EStrings.Create (File         => DPCOutputFile,
                          Name_Of_File => DPCOutputFileName,
                          Form_Of_File => "",
                          Status       => Success);

         LocalOK := Success = SPARK_IO.Ok;
         if LocalOK then
            File_Utils.PrintAHeader (DPCOutputFile,
                                 "Semantic Analysis of SPARK Text",
                                 File_Utils.OtherFile);
            PutSubprogramName (DPCOutputFile, SubprogSym, File_Utils.OtherFile);
         end if;
      end ProduceDPCOutputFile;

      ------------------------------------------------------------

      procedure ProduceDeclarationsFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out DeclarationsFile;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out LocalOk;
      --# derives DeclarationsFile,
      --#         LocalOk           from DeclarationsFile,
      --#                                FileName,
      --#                                SPARK_IO.FILE_SYS &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                DeclarationsFile,
      --#                                Dictionary.Dict,
      --#                                FileName,
      --#                                LexTokenManager.State,
      --#                                SubprogSym,
      --#                                UnitName;
      is
         DeclarationsFileName : EStrings.T;
         Success        : SPARK_IO.File_Status;
      begin
         DeclarationsFileName := FileName;
         FileSystem.CheckExtension
           (Fn  => DeclarationsFileName,
            Ext => EStrings.Copy_String (Str => FDL_Extension));

         EStrings.Create (File         => DeclarationsFile,
                          Name_Of_File => DeclarationsFileName,
                          Form_Of_File => "",
                          Status       => Success);
         LocalOK := Success = SPARK_IO.Ok;
         if LocalOK then
            File_Utils.PrintAHeader (DeclarationsFile,
                                 "FDL Declarations",
                                 File_Utils.DecFile);
         end if;
         PutSubprogramName (DeclarationsFile, SubprogSym, File_Utils.DecFile);
      end ProduceDeclarationsFile;

      ------------------------------------------------------------------

      procedure ProduceRuleFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out RuleFile;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out LocalOk;
      --# derives LocalOk,
      --#         RuleFile          from FileName,
      --#                                RuleFile,
      --#                                SPARK_IO.FILE_SYS &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                FileName,
      --#                                LexTokenManager.State,
      --#                                RuleFile,
      --#                                SubprogSym,
      --#                                UnitName;
      is
         RuleFileName   : EStrings.T;
         Success        : SPARK_IO.File_Status;
      begin
         RuleFileName := FileName;
         FileSystem.CheckExtension
           (Fn  => RuleFileName,
            Ext => EStrings.Copy_String (Str => RLS_Extension));

         EStrings.Create (File         => RuleFile,
                          Name_Of_File => RuleFileName,
                          Form_Of_File => "",
                          Status       => Success);
         LocalOK := Success = SPARK_IO.Ok;

         if LocalOK then
            File_Utils.PrintAHeader (RuleFile,
                                     "Proof Rule Declarations",
                                     File_Utils.RuleFile);
         end if;
         PutSubprogramName (RuleFile, SubprogSym, File_Utils.RuleFile);
      end ProduceRuleFile;

      ------------------------------------------------------------------

   begin -- ProduceOutputFiles
      FullSymbolName (SubprogSym,
                      AncestorName,
                      UnitName);

      BuildFileNameNest;

      -- FileName now contains the basename of the required output file(s)
      -- without an extension.  It is also a relative to the current working
      -- directory - for example for subprogram P.Q, we end up with FileName
      -- being "p/q"
      --
      -- If the user has requested an alternative output directory, then we adjust
      -- FileName now
      CommandLineData.Normalize_FileName_To_Output_Directory (FileName);

      OutputFileName := EStrings.Empty_String;

      if LocalOK then
         ProduceDeclarationsFile;

         if LocalOK then
            ProduceRuleFile;

            if LocalOK and CommandLineData.Content.VCG then
               ProduceVCGOutputFile;
            end if;

            if LocalOK and CommandLineData.Content.DPC then
               ProduceDPCOutputFile;
            end if;
         end if;
      end if;

      OK := LocalOK;
      --# accept Flow, 601, VCGOutputFile, DeclarationsFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, VCGOutputFile, RuleFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, DPCOutputFile, RuleFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, DPCOutputFile, DeclarationsFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, DPCOutputFile, VCGOutputFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, OutputFileName, RuleFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, OutputFileName, DeclarationsFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, RuleFile, DeclarationsFile, "ignore data coupling between files thro' SPARK_IO";
   end ProduceOutputFiles;

   ---------------------------------------------------------------------------

   procedure GenerateVCsLocal (StartNode                 : in     STree.SyntaxNode;
                               Scope                     : in     Dictionary.Scopes;
                               EndPosition               : in     LexTokenManager.Token_Position;
                               FlowHeap                  : in out Heap.HeapRecord;
                               SemanticErrorInSubprogram : in     Boolean;
                               DataFlowErrorInSubprogram : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --# derives Declarations.State,
   --#         Dictionary.Dict,
   --#         FlowHeap,
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         StmtStack.S               from *,
   --#                                        CommandLineData.Content,
   --#                                        DataFlowErrorInSubprogram,
   --#                                        Dictionary.Dict,
   --#                                        FlowHeap,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        StartNode,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        DataFlowErrorInSubprogram,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FlowHeap,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        StartNode,
   --#                                        STree.Table;
   is
      SubprogSym        : Dictionary.Symbol;
      VCGOutputFile     : SPARK_IO.File_Type;
      DPCOutputFile     : SPARK_IO.File_Type;
      OutputFileName    : EStrings.T;
      DeclarationsFile  : SPARK_IO.File_Type;
      RuleFile          : SPARK_IO.File_Type;
      OK                : Boolean;
      Success           : SPARK_IO.File_Status;
      VCGheap           : Cells.Heap_Record;


      -- In case of a fatal error, we generate a single "False" VC in the
      -- VCG or DPC file using this procedure.
      procedure Generate_False_VC (OutputFile : in SPARK_IO.File_Type)
      --# global in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                Dictionary.Dict,
      --#                                LexTokenManager.State,
      --#                                OutputFile,
      --#                                SubprogSym;
      is
         HeadLine     : EStrings.T;
         UnitName     : LexTokenLists.Lists;
         AncestorName : LexTokenLists.Lists;
      begin
         --# accept F, 10, AncestorName, "Ineffective assignment here OK";
         FullSymbolName (SubprogSym,
                         AncestorName,
                         UnitName);
         --# end accept;

         -- The header of the VC has to have the correct name and prefix
         -- for POGS, so we have to deduce this here from SubprogSym
         if Dictionary.IsFunction (SubprogSym) then
            HeadLine := EStrings.Copy_String (Str => "function_");
         elsif Dictionary.IsTaskType (SubprogSym) then
            HeadLine := EStrings.Copy_String (Str => "task_type_");
         else -- must be a procedure
            HeadLine := EStrings.Copy_String (Str => "procedure_");
         end if;

         EStrings.Append_Examiner_String
           (E_Str1 => HeadLine,
            E_Str2 => EStrings.Lower_Case
              (E_Str => LexTokenManager.Lex_String_To_String
                 (Lex_Str => LexTokenLists.Get_Element (List => UnitName,
                                                        Pos  => LexTokenLists.Get_Length (List => UnitName)))));
         EStrings.Append_String (E_Str => HeadLine,
                                 Str   => "_1.");

         SPARK_IO.New_Line (OutputFile, 1);
         SPARK_IO.Put_Line (OutputFile,
                            "/* False VC generated due to VCG heap exhausted */", 0);
         SPARK_IO.New_Line (OutputFile, 2);
         SPARK_IO.Put_Line (OutputFile,
                            "For path(s) from start to finish:", 0);
         SPARK_IO.New_Line (OutputFile, 1);
         EStrings.Put_Line (File  => OutputFile,
                            E_Str => HeadLine);
         SPARK_IO.Put_Line (OutputFile, "H1:    true .", 0);
         SPARK_IO.Put_Line (OutputFile, "        ->", 0);
         SPARK_IO.Put_Line (OutputFile, "C1:    false .", 0);
         SPARK_IO.New_Line (OutputFile, 2);
         --# accept F, 33, AncestorName, "AncestorName not referenced here OK";
      end Generate_False_VC;



   begin
      VCGOutputFile    := SPARK_IO.Null_File;
      DPCOutputFile    := SPARK_IO.Null_File;
      DeclarationsFile := SPARK_IO.Null_File;
      RuleFile         := SPARK_IO.Null_File;

      SubprogSym := Dictionary.GetRegion (Scope);
      ProduceOutputFiles (SubprogSym,
                          VCGOutputFile,
                          DPCOutputFile,
                          DeclarationsFile,
                          RuleFile,
                          OutputFileName,
                          OK);
      if OK then
         Cells.Initialize (VCGheap);
         Declarations.StartProcessing (VCGheap);
         ProduceVCs (VCGheap,
                     StartNode,
                     SubprogSym,
                     Scope,
                     VCGOutputFile,
                     DPCOutputFile,
                     OutputFileName,
                     EndPosition,
                     FlowHeap,
                     SemanticErrorInSubprogram,
                     DataFlowErrorInSubprogram);
         Declarations.OutputDeclarations (VCGheap,
                                          DeclarationsFile,
                                          RuleFile,
                                          Scope,
                                          True,
                                          EndPosition);
         Cells.Report_Usage (VCGheap);
      else
         -- Unable to create output files
         ErrorHandler.SemanticWarning (406,
                                       EndPosition,
                                       LexTokenManager.Null_String);
      end if;
      --# accept Flow, 10, Success, "Expected ineffective assignment to Success" &
      --#        Flow, 10, VCGOutputFile, "Expected ineffective assignment to VCGOutputFile" &
      --#        Flow, 10, DPCOutputFile, "Expected ineffective assignment to VCGOutputFile" &
      --#        Flow, 10, DeclarationsFile, "Expected ineffective assignment to DeclarationsFile" &
      --#        Flow, 10, RuleFile, "Expected ineffective assignment to RuleFile";
      SPARK_IO.Close (DeclarationsFile, Success);
      SPARK_IO.Close (RuleFile, Success);

      if SPARK_IO.Is_Open (VCGOutputFile) then
         SPARK_IO.Close (VCGOutputFile, Success);
      end if;

      if SPARK_IO.Is_Open (DPCOutputFile) then
         SPARK_IO.Close (DPCOutputFile, Success);
      end if;
      --# end accept;

      --# accept Flow, 33, Success, "Expected Success to be neither referenced nor exported";
   exception
      --# hide GenerateVCsLocal;
      when Fatal.StaticLimit =>
         -- Here owing to a VCG Heap/table exhausted.
         -- We need to close open files, making sure they are at least
         -- syntactically legal for the Simplifier.
         -- We insert an explicitly False VC here, so it is sure
         -- to be undischarged and picked up by POGS

         ErrorHandler.SemanticWarning (409,
                                       EndPosition,
                                       LexTokenManager.Null_String);


         if SPARK_IO.Is_Open (VCGOutputFile) then
            Generate_False_VC (VCGOutputFile);
            SPARK_IO.Close (VCGOutputFile, Success);
         end if;

         if SPARK_IO.Is_Open (DPCOutputFile) then
            Generate_False_VC (DPCOutputFile);
            SPARK_IO.Close (DPCOutputFile, Success);
         end if;

         if SPARK_IO.Is_Open (RuleFile) then
            SPARK_IO.Close (RuleFile, Success);
         end if;

         if SPARK_IO.Is_Open (DeclarationsFile) then
            -- Make sure the FDL file is termianted properly before closing it
            Declarations.PrintDeclarationTail (DeclarationsFile);
            SPARK_IO.Close (DeclarationsFile, Success);
         end if;

         -- We DONT'T re-raise here - there may be other subprograms
         -- requiring VC Generation in the enclosing unit, so we
         -- carry on.

      when others =>
         -- Any other exception reaching here.
         -- We need to close open files, then re-raise
         if SPARK_IO.Is_Open (VCGOutputFile) then
            SPARK_IO.Close (VCGOutputFile, Success);
         end if;

         if SPARK_IO.Is_Open (DPCOutputFile) then
            SPARK_IO.Close (DPCOutputFile, Success);
         end if;

         if SPARK_IO.Is_Open (RuleFile) then
            SPARK_IO.Close (RuleFile, Success);
         end if;

         if SPARK_IO.Is_Open (DeclarationsFile) then
            -- Make sure the FDL file is termianted properly before closing it
            Declarations.PrintDeclarationTail (DeclarationsFile);
            SPARK_IO.Close (DeclarationsFile, Success);
         end if;

         raise;
   end GenerateVCsLocal;


   procedure GenerateVCs (StartNode                 : in     STree.SyntaxNode;
                          Scope                     : in     Dictionary.Scopes;
                          Do_VCG                    : in     Boolean;
                          EndPosition               : in     LexTokenManager.Token_Position;
                          FlowHeap                  : in out Heap.HeapRecord;
                          SemanticErrorInSubprogram : in     Boolean;
                          DataFlowErrorInSubprogram : in     Boolean)
   is
      ErrorsInSubprogramOrItsSignature : Boolean;
   begin
      if Do_VCG then

         Invoked := True;

         ErrorsInSubprogramOrItsSignature := SemanticErrorInSubprogram or
           (not Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract,
                                                            Dictionary.GetRegion (Scope)));

         if ErrorsInSubprogramOrItsSignature then
            ErrorHandler.SemanticWarning (408,
                                          EndPosition,
                                          LexTokenManager.Null_String);
         end if;

         GenerateVCsLocal (StartNode,
                           Scope,
                           EndPosition,
                           FlowHeap,
                           ErrorsInSubprogramOrItsSignature,
                           DataFlowErrorInSubprogram);

      end if;
   end GenerateVCs;

end VCG;
