-- $Id: vcg-producevcs.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Debug;
with DAG;
with Pile;

separate (VCG)
procedure ProduceVCs (VCGheap                   : in out Cells.Heap_Record;
                      StartNode                 : in     STree.SyntaxNode;
                      SubprogSym                : in     Dictionary.Symbol;
                      Scope                     : in     Dictionary.Scopes;
                      VCGOutputFile             : in     SPARK_IO.File_Type;
                      DPCOutputFile             : in     SPARK_IO.File_Type;
                      OutputFileName            : in     EStrings.T;
                      EndPosition               : in     LexTokenManager.Token_Position;
                      FlowHeap                  : in out Heap.HeapRecord;
                      SemanticErrorInSubprogram : in     Boolean;
                      DataFlowErrorInSubprogram : in     Boolean)
is
   VCGFailure  : Boolean;
   VerboseEcho : Boolean;


   procedure Put_Line (S : in String)
   --# global in     VerboseEcho;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                S,
   --#                                VerboseEcho;
   is
   begin
      if VerboseEcho then
         ScreenEcho.Put_Line (S);
      end if;
   end Put_Line;

   procedure New_Line
   --# global in     VerboseEcho;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                VerboseEcho;
   is
   begin
      if VerboseEcho then
         ScreenEcho.New_Line (1);
      end if;
   end New_Line;

   procedure DumpUsedSymbols (S : in String)
   --# derives null from S;
   is
      --# hide DumpUsedSymbols;
      Iterator : Declarations.UsedSymbolIterator;
      CN       : Cells.Cell;
      CS       : Dictionary.Symbol;
      CDAG     : Cells.Cell;
   begin
      Debug.PrintMsg (S, True);
      Declarations.Initialize (Iterator);
      while not Declarations.IsNullIterator (Iterator) loop
         CN := Declarations.CurrentNode (Iterator);
         CS := Cells.Get_Symbol_Value (VCGheap, CN);
         Debug.PrintSym ("Symbol: ", CS);

         if Dictionary.IsConstant (CS) then
            Debug.PrintMsg ("is a constant and ", False);
            if Dictionary.IsPrivateType (Dictionary.GetType (CS), Scope) then
               Debug.PrintMsg ("is a private type in this scope", True);
            else
               if Dictionary.TypeIsScalar (Dictionary.GetType (CS)) then
                  Debug.PrintMsg ("is a scalar in this scope", True);
               elsif Dictionary.TypeIsArray (Dictionary.GetType (CS)) then
                  Debug.PrintMsg ("is an array in this scope", True);
                  CDAG := Pile.DAG (VCGheap, CN);
                  if Cells.Is_Null_Cell (CDAG) then
                     Debug.PrintMsg ("and its DAG is Null", True);
                  else
                     Debug.PrintDAG ("and its DAG is ", CDAG, VCGheap, Scope);
                  end if;

               elsif Dictionary.TypeIsRecord (Dictionary.GetType (CS)) then
                  Debug.PrintMsg ("is an array in this scope", True);
                  CDAG := Pile.DAG (VCGheap, CN);
                  if Cells.Is_Null_Cell (CDAG) then
                     Debug.PrintMsg ("and its DAG is Null", True);
                  else
                     Debug.PrintDAG ("and its DAG is ", CDAG, VCGheap, Scope);
                  end if;
               else
                  Debug.PrintMsg ("is OTHER in this scope", True);
               end if;
            end if;
         else
            Debug.PrintMsg ("is not a constant", True);
         end if;

         Iterator := Declarations.NextNode (VCGheap, Iterator);
      end loop;
      Debug.PrintMsg ("---End---", True);

   end DumpUsedSymbols;
   pragma Unreferenced (DumpUsedSymbols);

   procedure ProcessCompositeConstants
   --# global in     CommandLineData.Content;
   --#        in     EndPosition;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGheap;
   --# derives Declarations.State,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         VCGheap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Declarations.State,
   --#                                        Dictionary.Dict,
   --#                                        Graph.Table,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        StmtStack.S,
   --#                                        STree.Table,
   --#                                        VCGheap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Declarations.State,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Graph.Table,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        StmtStack.S,
   --#                                        STree.Table,
   --#                                        VCGheap;
   is
      Iterator   : Declarations.UsedSymbolIterator;
      CN         : Cells.Cell;
      CS         : Dictionary.Symbol;
      CT         : Dictionary.Symbol;
      CDAG       : Cells.Cell;
      ChangeMade : Boolean;
      ExpNode    : STree.SyntaxNode;

      function RuleIsRequired (CS : Dictionary.Symbol) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      --#        in Scope;
      is

      begin
         return CommandLineData.Content.ConstantRules = CommandLineData.AllRules or else
           (CommandLineData.Content.ConstantRules = CommandLineData.Keen and then
              not (Dictionary.GetConstantRulePolicy (CS, Scope) = Dictionary.NoRuleRequested)) or else
           (CommandLineData.Content.ConstantRules = CommandLineData.Lazy and then
              Dictionary.GetConstantRulePolicy (CS, Scope) = Dictionary.RuleRequested);
      end RuleIsRequired;

      procedure RaiseWarnings
   --# global in     CommandLineData.Content;
   --#        in     Declarations.State;
   --#        in     Dictionary.Dict;
   --#        in     EndPosition;
   --#        in     LexTokenManager.State;
   --#        in     Scope;
   --#        in     VCGheap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Declarations.State,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        VCGheap;
      is
         Iterator   : Declarations.UsedSymbolIterator;
         CN         : Cells.Cell;
         CS         : Dictionary.Symbol;
         CT         : Dictionary.Symbol;
         CDAG       : Cells.Cell;
      begin
         Declarations.Initialize (Iterator);

         while not Declarations.IsNullIterator (Iterator) loop

            CN := Declarations.CurrentNode (Iterator);
            CS := Cells.Get_Symbol_Value (VCGheap, CN);
            CT := Dictionary.GetType (CS);

            -- We're interested in visible (not private), composite constants...
            if Dictionary.IsConstant (CS) and then
              not Dictionary.IsPrivateType (CT, Scope) and then
              not Dictionary.IsGenericParameterSymbol (CS) and then
              (Dictionary.TypeIsArray (CT) or Dictionary.TypeIsRecord (CT)) and then
              RuleIsRequired (CS) then

               CDAG := Pile.DAG (VCGheap, CN);
               if Cells.Is_Null_Cell (CDAG) then
                  if Dictionary.ConstantExpIsWellFormed (CS) then

                     if Dictionary.TypeIsArray (CT) and then
                       Dictionary.GetNumberOfDimensions (CT) > 1 then

                        -- At present, FDL cannot represent multi-dimensional array aggregates,
                        -- so we simply issue a warning and continue.  The warning
                        -- appears at the end of the listing for the subprogram that
                        -- we are generating VCs for.
                        ErrorHandler.SemanticWarningSym (312,
                                                         EndPosition,
                                                         CS,
                                                         Dictionary.GetScope (CS));
                     end if;

                  else

                     -- Cannot produce rule due to semantic errors in Expression
                     -- we simply issue a warning and continue.  The warning
                     -- appears at the end of the listing for the subprogram that
                     -- we are generating VCs for.
                     ErrorHandler.SemanticWarningSym (313,
                                                      EndPosition,
                                                      CS,
                                                      Dictionary.GetScope (CS));
                  end if;
               end if;

            end if;


            Iterator := Declarations.NextNode (VCGheap, Iterator);
         end loop;

      end RaiseWarnings;

   begin

      loop
         ChangeMade := False;
         Declarations.Initialize (Iterator);

         while not Declarations.IsNullIterator (Iterator) loop

            CN := Declarations.CurrentNode (Iterator);
            CS := Cells.Get_Symbol_Value (VCGheap, CN);
            CT := Dictionary.GetType (CS);

            -- We're interested in visible (not private), composite constants...
            if Dictionary.IsConstant (CS) and then
              not Dictionary.IsPrivateType (CT, Scope) and then
              not Dictionary.IsGenericParameterSymbol (CS) and then
              (Dictionary.TypeIsArray (CT) or Dictionary.TypeIsRecord (CT)) and then
              RuleIsRequired (CS) then

               CDAG := Pile.DAG (VCGheap, CN);
               if Cells.Is_Null_Cell (CDAG) then
                  if Dictionary.ConstantExpIsWellFormed (CS) then

                     ExpNode := STree.RefToNode (Dictionary.GetConstantExpNode (CS));
                     if (not Dictionary.TypeIsArray (CT)) or else
                       Dictionary.GetNumberOfDimensions (CT) = 1 then

                        -- Build the DAG for this initializing expression, and store the
                        -- resulting root Cell in the Pile Node for this constant.
                        -- The initializing expression must be evaluation in the Scope where
                        -- it is declared.
                        DAG.BuildConstantInitializationDAG (ExpNode, Dictionary.GetScope (CS), VCGheap, CDAG);
                        Pile.SetDAG (VCGheap, CN, CDAG);

                        -- This newly generated DAG might contain references to other
                        -- constants and so on which need FDL declarations and Rules, so...
                        Declarations.FindDagDeclarations (VCGheap, CDAG);

                        -- ...that might have changed the state of the Declarations package,
                        -- which we are currrently iterating over.  This means our Iterator
                        -- is no longer valid, so we have to give up here and start a new
                        -- pass.
                        ChangeMade := True;

                     end if;

                  end if;
               end if;

            end if;

            exit when ChangeMade;

            Iterator := Declarations.NextNode (VCGheap, Iterator);
         end loop;

         -- No changes at all made - that means we must have processed all the constants,
         -- so we can terminate.
         exit when not ChangeMade;
      end loop;

      RaiseWarnings;

   end ProcessCompositeConstants;

begin -- ProduceVCs;
   VerboseEcho := CommandLineData.Content.Echo and not CommandLineData.Content.Brief;

   New_Line;
   Put_Line ("           Building model of subprogram ...");

   Graph.ReInitializeGraph;
   VCGFailure := False;

   DAG.BuildGraph (StartNode,
                   SubprogSym,
                   Scope,
                   VCGOutputFile,
                   EndPosition,
                   VCGFailure,
                   VCGheap,
                   FlowHeap,
                   SemanticErrorInSubprogram,
                   DataFlowErrorInSubprogram);

   if VCGFailure then
      ErrorHandler.SemanticError (962, 0, EndPosition, LexTokenManager.Null_String);
   else
      if CommandLineData.Content.Debug.VCG then
         Debug.PrintMsg ("----------- Dump of VCG State after DAG.BuildGraph ---------------", True);
         Graph.Dump_Graph_Table (VCGheap, Scope, Graph.PFs);
         Graph.Dump_Graph_Dot (VCGheap, OutputFileName, 0, Scope, Graph.PFs);
         Debug.PrintMsg ("------------------------------------------------------------------", True);
      end if;

      New_Line;
      Put_Line ("           Generating VCs ...");
      Graph.GenVCs (VCGheap, VCGOutputFile, OutputFileName, Scope, VCGFailure);
      if VCGFailure then
         New_Line;
         ErrorHandler.SemanticError (962, 0, EndPosition, LexTokenManager.Null_String);
      else
         if CommandLineData.Content.VCG then
            New_Line;
            Put_Line ("           Writing VCs ...");
            Graph.Print_VCs_Or_DPCs (VCGheap, VCGOutputFile, Scope, Graph.VCs);
         end if;

         if CommandLineData.Content.DPC then
            New_Line;
            Put_Line ("           Writing DPCs ...");
            Graph.Print_VCs_Or_DPCs (VCGheap, DPCOutputFile, Scope, Graph.DPCs);
         end if;

         -- We also need to generate DAGs and Declarations for the
         -- initializing expressions of any composite constants that
         -- have been referenced in the VCs printed above.
         if CommandLineData.Content.ConstantRules /= CommandLineData.NoRules then
            ProcessCompositeConstants;
         end if;

      end if;
   end if;
end ProduceVCs;
