-- $Id: systemerrors.adb 13645 2009-06-25 13:57:33Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Fatal,
     ScreenEcho;
package body SystemErrors
is

   type ErrorKinds is (StartUpErr, StaticLimit, OperatingSystemLimit, InternalError);

   type SysErrToErrKindTable is array (SysErrType) of ErrorKinds;

   SETEK : constant SysErrToErrKindTable :=
     SysErrToErrKindTable'(StringTableOverflow   =>  StaticLimit,
                           SyntaxTreeOverflow    =>  StaticLimit,
                           ParseStackOverflow    =>  StaticLimit,
                           SymbolTableOverflowStatic =>  StaticLimit,
                           SymbolTableOverflowDynamic =>  OperatingSystemLimit,
                           InvalidSyntaxTree     =>  InternalError,
                           InvalidSymbolTable     =>  InternalError,
                           EmptyHeap             =>  StaticLimit,
                           RelationStackOverflow =>  StaticLimit,
                           RelationStackUnderflow =>  InternalError,
                           InvalidInit =>  InternalError,
                           ErrorPositionWrong =>  InternalError,
                           ExpressionStackCorrupt =>  InternalError,
                           ExpressionStackUnderflow =>  InternalError,
                           ExpressionStackOverflow =>  StaticLimit,
                           TypeContextStackCorrupt =>  InternalError,
                           TypeContextStackUnderflow =>  InternalError,
                           TypeContextStackOverflow =>  StaticLimit,
                           ListOverflowInExpression =>  StaticLimit,
                           ListOverflowInDependencyClause =>  StaticLimit,
                           ListOverflowInProcedureCall =>  StaticLimit,
                           CaseStackUnderflow =>  InternalError,
                           CaseStackOverflow =>  StaticLimit,
                           VCGGraphSizeExceeded =>  StaticLimit,
                           VCGHeapIsExhausted =>  StaticLimit,
                           VCGHeapIsCorrupted =>  InternalError,
                           RefListKeyCellMissing =>  InternalError,
                           FlowAnalyserExpressionLimit =>  StaticLimit,
                           CaseStatementNestingLimit  =>  StaticLimit,
                           ErrorHandlerTemporaryFiles =>  OperatingSystemLimit,
                           ErrorHandlerSource =>  OperatingSystemLimit,
                           DiskFullError =>  OperatingSystemLimit,
                           MathError =>  InternalError,
                           TooManyNestedArrays =>  StaticLimit,
                           TooManyNestedRecords =>  StaticLimit,
                           ContextUnitStackOverflow =>  StaticLimit,
                           ContextUnitStackUnderflow =>  InternalError,
                           ContextFileHeapOverflow =>  StaticLimit,
                           ContextUnitHeapOverflow =>  StaticLimit,
                           TooManyFileLines =>  StaticLimit,
                           IndexStackFull =>  StaticLimit,
                           IndexComponentListFull =>  StaticLimit,
                           TooManyErrors =>  StaticLimit,
                           WarningNameTooLong =>  StaticLimit,
                           UnitNameInIndexTooLong =>  StaticLimit,
                           FileNameInIndexTooLong =>  StaticLimit,
                           TooManySuppressedWarnings =>  StaticLimit,
                           UnitNestingTooDeep =>  StaticLimit,
                           StatementStackUnderflow =>  InternalError,
                           StatementStackOverflow =>  StaticLimit,
                           WfCompilationUnitStackOverflow =>  StaticLimit,
                           WfCompilationUnitStackUnderflow =>  InternalError,
                           TooManyFlowAnalyserExpressions =>  StaticLimit,
                           TooManyParamsInProcedureCall =>  StaticLimit,
                           StatisticsUsageGreaterThanTableSize =>  StaticLimit,
                           AggregateStackUnderFlow =>  InternalError,
                           AggregateStackOverFlow =>  StaticLimit,
                           MetaFileStackOverflow =>  StaticLimit,
                           LexStackOverflow =>  StaticLimit,
                           LexStackUnderflow =>  InternalError,
                           ComponentManagerOverflow =>  StaticLimit,
                           ComponentErrorOverflow =>  StaticLimit,
                           SyntaxTreeWalkError => InternalError,
                           PreconditionFailure => InternalError,
                           PostConditionFailure => InternalError,
                           AssertionFailure => InternalError,
                           UnimplementedFeature => InternalError,
                           XMLSchemaError => InternalError,
                           XMLGenerationError => InternalError,
                           IllegalXMLGenerationAttempt => InternalError,
                           StringOverFlow => InternalError,
                           QueueOverflow => StaticLimit,
                           XRefTableFull => OperatingSystemLimit,
                           -- Add additional errors here...
                           OtherInternalError     =>  InternalError);

   procedure StopProgram (ErrorKind : in ErrorKinds)
   --# derives null from ErrorKind;
   --# post False; -- does not terminate normally
   is
      --# hide StopProgram;
   begin
      case ErrorKind is
         when StartUpErr =>
            raise Fatal.StartUpError;
         when StaticLimit =>
            raise Fatal.StaticLimit;
         when OperatingSystemLimit =>
            raise Fatal.OperatingSystemLimit;
         when InternalError =>
            raise Fatal.InternalError;
      end case;
   end StopProgram;


   procedure DisplayCause (ErrorKind : in ErrorKinds)
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                ErrorKind;
   is
   begin
      case ErrorKind is
         when StartUpErr =>
            ScreenEcho.Put_Line ("* Unexpected StartUpError in SystemErrors"); --should never happen
         when StaticLimit =>
            ScreenEcho.Put_Line ("* Internal static tool limit reached");
         when OperatingSystemLimit =>
            ScreenEcho.Put_Line ("* Operating system limit reached");
         when InternalError =>
            ScreenEcho.Put_Line ("* Unexpected internal error");
      end case;
   end DisplayCause;

   procedure DisplayBox (SysErr : in SysErrType; Msg : in String)
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Msg,
   --#                                SysErr;
   is
   begin
      -- print 'big box' on screen to draw attention to error
      ScreenEcho.Put_Line ("*****************************************************************************");
      ScreenEcho.Put_Line ("* A fatal error has occurred");
      ScreenEcho.Put_String ("* ");

      case SysErr is
         when StringTableOverflow   =>
            ScreenEcho.Put_Line ("String table overflow");
         when SyntaxTreeOverflow    =>
            ScreenEcho.Put_Line ("Syntax tree overflow");
         when ParseStackOverflow    =>
            ScreenEcho.Put_Line ("Parse stack overflow");
         when SymbolTableOverflowStatic =>
            ScreenEcho.Put_Line ("Symbol table pointers list exhausted");
         when SymbolTableOverflowDynamic =>
            ScreenEcho.Put_Line ("Symbol table allocation overflow");
         when InvalidSyntaxTree     =>
            ScreenEcho.Put_Line ("Invalid syntax tree");
         when InvalidSymbolTable     =>
            ScreenEcho.Put_Line ("Internal Symbol Table Error");
         when EmptyHeap             =>
            ScreenEcho.Put_Line ("Empty heap");
         when RelationStackOverflow =>
            ScreenEcho.Put_Line ("Relation stack overflow");
         when RelationStackUnderflow =>
            ScreenEcho.Put_Line ("Relation stack underflow");
         when InvalidInit =>
            ScreenEcho.Put_Line ("Failure in initialisation");
         when ErrorPositionWrong =>
            ScreenEcho.Put_Line ("An error is incorrectly positioned");
         when ExpressionStackCorrupt =>
            ScreenEcho.Put_Line ("The expression stack is corrupt");
         when ExpressionStackUnderflow =>
            ScreenEcho.Put_Line ("Expression stack underflow");
         when ExpressionStackOverflow =>
            ScreenEcho.Put_Line ("Expression stack overflow");
         when TypeContextStackCorrupt =>
            ScreenEcho.Put_Line ("The type context stack is corrupt");
         when TypeContextStackUnderflow =>
            ScreenEcho.Put_Line ("Type context stack underflow");
         when TypeContextStackOverflow =>
            ScreenEcho.Put_Line ("Type context stack overflow");
         when ListOverflowInExpression =>
            ScreenEcho.Put_Line ("List overflow in expression");
         when ListOverflowInDependencyClause =>
            ScreenEcho.Put_Line ("List overflow in dependency clause");
         when ListOverflowInProcedureCall =>
            ScreenEcho.Put_Line ("List overflow in procedure call");
         when CaseStackUnderflow =>
            ScreenEcho.Put_Line ("Case statement stack underflow");
         when CaseStackOverflow =>
            ScreenEcho.Put_Line ("Case statement stack overflow");
         when VCGGraphSizeExceeded =>
            ScreenEcho.Put_Line ("Maximum graph size in VC Generator exceeded");
         when VCGHeapIsExhausted =>
            ScreenEcho.Put_Line ("VC Generator Heap is Exhausted");
         when VCGHeapIsCorrupted =>
            ScreenEcho.Put_Line ("VC Generator Heap is Corrupted");
         when RefListKeyCellMissing =>
            ScreenEcho.Put_Line ("Referenced Variable List Error");
         when FlowAnalyserExpressionLimit =>
            ScreenEcho.Put_Line ("Flow analyser expression limit reached");
         when CaseStatementNestingLimit  =>
            ScreenEcho.Put_Line ("Case statement nesting limit reached");
         when ErrorHandlerTemporaryFiles =>
            ScreenEcho.Put_Line ("Unable to open temporary file in ErrorHandler");
         when ErrorHandlerSource =>
            ScreenEcho.Put_Line ("Unable to open source file in ErrorHandler");
         when DiskFullError =>
            ScreenEcho.Put_Line ("File write operation failed, disk is full");
         when MathError =>
            ScreenEcho.Put_Line ("Internal error in static expression evaluator");
         when TooManyNestedArrays =>
            ScreenEcho.Put_Line ("Array constant nested too deeply");
         when TooManyNestedRecords =>
            ScreenEcho.Put_Line ("Record constant nested too deeply");
         when ContextUnitStackOverflow =>
            ScreenEcho.Put_Line ("Too many pending units in context manager");
         when ContextUnitStackUnderflow =>
            ScreenEcho.Put_Line ("Internal error in context manager: stack underflow");
         when ContextFileHeapOverflow =>
            ScreenEcho.Put_Line ("Too many files in examination");
         when ContextUnitHeapOverflow =>
            ScreenEcho.Put_Line ("Too many units in examination");
         when TooManyFileLines =>
            ScreenEcho.Put_Line ("Too many lines in source file");
         when IndexStackFull =>
            ScreenEcho.Put_Line ("Index files too deeply nested");
         when IndexComponentListFull =>
            ScreenEcho.Put_Line ("Too many components in index file entry");
         when TooManyErrors =>
            ScreenEcho.Put_Line ("Too many errors in a single file");
         when WarningNameTooLong =>
            ScreenEcho.Put_Line ("Line too long in warning control file");
         when UnitNameInIndexTooLong =>
            ScreenEcho.Put_Line ("Unit name too long in index file");
         when FileNameInIndexTooLong =>
            ScreenEcho.Put_Line ("File name too long in index file");
         when TooManySuppressedWarnings =>
            ScreenEcho.Put_Line ("Too many suppressed warnings for a single file");
         when UnitNestingTooDeep =>
            ScreenEcho.Put_Line ("Units too deeply nested");
         when StatementStackUnderflow =>
            ScreenEcho.Put_Line ("VCG statement stack underflow");
         when StatementStackOverflow =>
            ScreenEcho.Put_Line ("VCG statement stack overflow");
         when WfCompilationUnitStackOverflow =>
            ScreenEcho.Put_Line ("Well-formation checker error: compilation unit stack overflow");
         when WfCompilationUnitStackUnderflow =>
            ScreenEcho.Put_Line ("Internal error in well-formation checker: compilation unit stack underflow");
         when TooManyFlowAnalyserExpressions =>
            ScreenEcho.Put_Line ("Too many expressions in flow analyser");
         when TooManyParamsInProcedureCall =>
            ScreenEcho.Put_Line ("Too many parameters in procedure call");
         when StatisticsUsageGreaterThanTableSize =>
            ScreenEcho.Put_Line ("Reported table usage larger than table size");
         when AggregateStackUnderFlow =>
            ScreenEcho.Put_Line ("Aggregate stack underflow");
         when AggregateStackOverFlow =>
            ScreenEcho.Put_Line ("Aggregate stack overflow");
         when MetaFileStackOverflow =>
            ScreenEcho.Put_Line ("Stack overflow while processing meta file");
         when LexStackOverflow =>
            ScreenEcho.Put_Line ("Stack overflow in LexTokenStacks");
         when LexStackUnderflow =>
            ScreenEcho.Put_Line ("Stack under flow in LexTokenStacks");
         when ComponentManagerOverflow =>
            ScreenEcho.Put_Line ("Record component manager overflow");
         when ComponentErrorOverflow =>
            ScreenEcho.Put_Line ("Record component error-manager overflow");
         when SyntaxTreeWalkError =>
            ScreenEcho.Put_Line ("Syntax tree walk error");
         when PreconditionFailure =>
            ScreenEcho.Put_Line ("Precondition failure");
         when PostConditionFailure =>
            ScreenEcho.Put_Line ("Postcondition failure");
         when AssertionFailure =>
            ScreenEcho.Put_Line ("Run-time assertion failure");
         when UnimplementedFeature =>
            ScreenEcho.Put_Line ("Use of an unimplemented SPARK language construct or Examiner feature");
         when XMLSchemaError =>
            ScreenEcho.Put_Line ("Error initialising schema");
         when XMLGenerationError =>
            ScreenEcho.Put_Line ("Internal failure of the XML report generator");
         when IllegalXMLGenerationAttempt =>
            ScreenEcho.Put_Line ("The Examiner attempted to generate invalid XML");
         when StringOverFlow =>
            ScreenEcho.Put_Line ("String operation overflowed");
         when QueueOverflow =>
            ScreenEcho.Put_Line ("Queue operation overflowed");
         when XRefTableFull =>
            ScreenEcho.Put_Line ("Cross-references table full");
         -- Add additional errors here...
         when OtherInternalError =>
            ScreenEcho.Put_Line ("Other internal error");
      end case;
      if Msg /= "" then
         ScreenEcho.Put_String ("* ");
         ScreenEcho.Put_Line (Msg);
      end if;
   end DisplayBox;


   procedure StartupError (StartErr  : in StartErrType;
                           Version   : in EStrings.T;
                           FlexError : in Integer)
   is
   begin -- StartupError
      -- print 'big box' on screen to draw attention to error
      ScreenEcho.Put_Line ("*****************************************************************************");

      ScreenEcho.Put_String ("* SPARK Toolset Release ");
      ScreenEcho.Put_ExaminerString (Version);

      ScreenEcho.Put_Line (" ");
      ScreenEcho.Put_String ("* ");
      case StartErr is
         when NoLmgrd => ScreenEcho.Put_Line ("The licence manager daemon is not running.");
         when NoLicenceFile => ScreenEcho.Put_Line ("Cannot find licence file. Check setting of LM_LICENSE_FILE.");
         when CantReadLicence => ScreenEcho.Put_Line ("No read access to licence file.");
         when Expired => ScreenEcho.Put_Line ("Licence has expired.");
         when TooManyExaminerUsers => ScreenEcho.Put_Line ("All Examiner licences are in use.");
         when TooManyRavenSPARKUsers => ScreenEcho.Put_Line ("All RavenSPARK licences are in use.");
         when TooManyVDSUsers => ScreenEcho.Put_Line ("All VDS licences are in use.");
         when TooManyAdaMagicUsers => ScreenEcho.Put_Line ("All AdaMagic licences are in use.");
         when CorruptedLicence => ScreenEcho.Put_Line ("Licence file is corrupted.");
         when ObsoleteLicence => ScreenEcho.Put_Line ("Obsolete version of licence file is being used.");
         when UnexpectedFLEX => ScreenEcho.Put_Line ("Unexpected FLEXlm error.");
         when WrongMachine => ScreenEcho.Put_Line ("Examiner not licensed for this machine.");
         when WrongExaminerLicenceFile => ScreenEcho.Put_Line ("Examiner licence not found in licence file." &
                                                               " Check LM_LICENSE_FILE setting.");
         when WrongRavenSPARKLicenceFile => ScreenEcho.Put_Line ("RavenSPARK licence not found in licence file." &
                                                                 " Check LM_LICENSE_FILE setting.");
         when WrongVDSLicenceFile => ScreenEcho.Put_Line ("VDS licence not found in licence file." &
                                                                 " Check LM_LICENSE_FILE setting.");
         when WrongAdaMagicLicenceFile => ScreenEcho.Put_Line ("AdaMagic licence not found in licence file." &
                                                               " Check LM_LICENSE_FILE setting.");
         when NoSpex => ScreenEcho.Put_Line ("SPEX.EXE not found in same directory as Examiner.");
      end case;

      if FlexError /= 0 then
         ScreenEcho.Put_String ("* FLEXlm error number: ");
         ScreenEcho.Put_Integer (FlexError, 4, 10);
         ScreenEcho.Put_Line (" ");
      end if;

      --# assert True;

      -- print 'big box' on screen to draw attention to error
      ScreenEcho.Put_Line ("*****************************************************************************");

      StopProgram (StartUpErr);
   end StartupError;


   procedure FatalError (SysErr : in SysErrType; Msg : in String)
   is
      --# hide FatalError;
      ErrorKind : ErrorKinds;
   begin
      ErrorKind := SETEK (SysErr);

      case SysErr is
         when VCGGraphSizeExceeded |
              VCGHeapIsExhausted =>

            -- Following SEPR 2272, these are both caught and handled using
            -- semantic warning 409 in VCG.GenerateVCsLocal
            -- and no longer terminate the Examiner.
            -- Therefore, no need for a display box here.
            null;

         when others =>
            DisplayBox (SysErr, Msg);
            DisplayCause (ErrorKind);
      end case;

      StopProgram (ErrorKind);
   end FatalError;

   procedure RTAssert (C : in Boolean; SysErr : in SysErrType; Msg : in String)
   is
      --# hide RTAssert;
   begin
      if not C then
         FatalError (SysErr, Msg);
      end if;
   end RTAssert;

end SystemErrors;
