-- $Id: statistics.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with ExaminerConstants,
     SystemErrors;

package body Statistics
is

   subtype Percentage is Integer range 0 .. 100;

   subtype Index_19 is Integer range 1 .. 19;
   subtype String_19 is String (Index_19);

   type TableDataRecord is record
      Label       : String_19;
      MaxSize     : Integer;
      UnitsUsed   : Integer;
      PercentUsed : Percentage;
   end record;

   type TableUsageArray is array (TableType) of TableDataRecord;


   -- Initialize data for each of the tables

   TableUsage : TableUsageArray :=
      TableUsageArray'(
                       RelationTable => TableDataRecord'
                       (Label       => String_19'("Relation Table     "),
                        MaxSize     => ExaminerConstants.HeapListLength,
                        UnitsUsed   => 0,
                        PercentUsed => 0),
                       StringTable => TableDataRecord'
                       (Label       => String_19'("String Table       "),
                        MaxSize     => ExaminerConstants.String_Table_Size,
                        UnitsUsed   => 0,
                        PercentUsed => 0),
                       SymbolTable => TableDataRecord'
                       (Label       => String_19'("Symbol Table       "),
                        MaxSize     => ExaminerConstants.SymbolTableSize,
                        UnitsUsed   => 0,
                        PercentUsed => 0),
                       SyntaxTree => TableDataRecord'
                       (Label       => String_19'("Syntax Tree        "),
                        MaxSize     => ExaminerConstants.SyntaxTreeSize,
                        UnitsUsed   => 0,
                        PercentUsed => 0),
                       VCGHeap => TableDataRecord'
                       (Label       => String_19'("VCG Heap           "),
                        MaxSize     => ExaminerConstants.VCGCellListLength,
                        UnitsUsed   => 0,
                        PercentUsed => 0),
                       RecordFields => TableDataRecord'
                       (Label       => String_19'("Record components  "),
                        MaxSize     => ExaminerConstants.MaxRecordComponents,
                        UnitsUsed   => 0,
                        PercentUsed => 0),
                       RecordErrors => TableDataRecord'
                       (Label       => String_19'("Record errors      "),
                        MaxSize     => ExaminerConstants.MaxRecordErrors,
                        UnitsUsed   => 0,
                        PercentUsed => 0));


   procedure SetTableUsage (Table : in     TableType;
                            Size  : in     Integer)
   is
   begin

      -- Is the value of "Size" silly (more than maximum)?

      if Size > TableUsage (Table).MaxSize then

         SystemErrors.FatalError (SystemErrors.StatisticsUsageGreaterThanTableSize,
                                 "in Statistics.SetTableUsage");

         -- Is more of the table being used than previously recorded?

      elsif Size > TableUsage (Table).UnitsUsed then

         -- Update number of units being used in table, and calculate
         -- percentage of table in use.

         TableUsage (Table).UnitsUsed := Size;
         TableUsage (Table).PercentUsed :=
            (100 * Size) / TableUsage (Table).MaxSize;

      end if;

   end SetTableUsage; -- expected flow error: if size larger than max,
   -- system error affects file sys: but max is  a constant
   -- other contents of TableUsage do _not_ affect filesys


   procedure WriteOutput (File   : in     SPARK_IO.File_Type)
   is

      Column2Posn : constant Integer := 20;
      Column3Posn : constant Integer := 30;
      Column4Posn : constant Integer := 40;
      LastColumn  : constant Integer := 50;

   begin

      SPARK_IO.New_Line (File    => File,
                         Spacing => 2);


      SPARK_IO.Put_Line (File    => File,
                         Item    => "Resource statistics",
                         Stop    => 0);
      SPARK_IO.New_Line (File    => File,
                         Spacing => 1);


      -- Write header line
      SPARK_IO.Put_String (File => File,
                           Item => "Table",
                           Stop => 0);

      SPARK_IO.Set_Col (File => File,
                        Posn => Column2Posn);

      SPARK_IO.Put_String (File => File,
                           Item => "Units used",
                           Stop => 0);

      SPARK_IO.Set_Col (File => File,
                        Posn => Column3Posn);

      SPARK_IO.Put_String (File => File,
                           Item => "  Max Size",
                           Stop => 0);

      SPARK_IO.Set_Col (File => File,
                        Posn => Column4Posn);

      SPARK_IO.Put_String (File => File,
                           Item => "    % used",
                           Stop => 0);

      -- Write a line for each of the tables

      for Table in TableType loop

         SPARK_IO.New_Line (File    => File,
                            Spacing => 1);

         SPARK_IO.Put_String (File => File,
                              Item => TableUsage (Table).Label,
                              Stop => 0);

         SPARK_IO.Set_Col (File => File,
                           Posn => Column2Posn);

         SPARK_IO.Put_Integer (File  => File,
                               Item  => TableUsage (Table).UnitsUsed,
                               Width => Column3Posn - Column2Posn,
                               Base  => 10);

         SPARK_IO.Set_Col (File => File,
                           Posn => Column3Posn);

         SPARK_IO.Put_Integer (File  => File,
                               Item  => TableUsage (Table).MaxSize,
                               Width => Column4Posn - Column3Posn,
                               Base  => 10);

         SPARK_IO.Set_Col (File => File,
                           Posn => Column4Posn);

         SPARK_IO.Put_Integer (File  => File,
                               Item  => TableUsage (Table).PercentUsed,
                               Width => LastColumn - Column4Posn,
                               Base  => 10);

      end loop;

      SPARK_IO.New_Line (File    => File,
                         Spacing => 1);


   end WriteOutput;


end Statistics;
