-- $Id: sparklex-linemanager.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (SparkLex)
package body LineManager
is

   procedure ClearLine
   is
   begin
      CurrLine := ProgramLine'
         (SparkLex.InAda,
          SparkLex.StartAnnotation,
          0, 1, 1, 1, 1,
          SparkLex.Line'
            (SparkLex.Line_Lengths => SparkLex.EndOfLine));
   end ClearLine;

   procedure CopyOutLine (Line : out ProgramLine)
   is
   begin
      Line := ProgramLine'
         (Context       => SparkLex.InAda,
          AnnoContext   => SparkLex.StartAnnotation,
          LineLength    => CurrLine.LineLength,
          LineNo        => CurrLine.LineNo,
          LastTokenPos  => CurrLine.LastTokenPos,
          CurrPos       => CurrLine.LastTokenPos,
          LookaheadPos  => CurrLine.LookaheadPos,
          Conts         => CurrLine.Conts);
   end CopyOutLine;

   procedure CopyInLine (Line : in ProgramLine)
   is
   begin
      CurrLine := Line;
   end CopyInLine;

   procedure RecordCurrPos
   is
   begin
      CurrLine.LastTokenPos := CurrLine.CurrPos;
   end RecordCurrPos;

   procedure ResetCurrPos
   is
   begin
      CurrLine.CurrPos := CurrLine.LastTokenPos;
      CurrLine.LookaheadPos := CurrLine.LastTokenPos;
   end ResetCurrPos;

   procedure InspectChar (Ch : out Character)
   is
   begin
      Ch := CurrLine.Conts (CurrLine.CurrPos);
   end InspectChar;

   procedure AcceptChar
   is
   begin
      case CurrLine.Conts (CurrLine.CurrPos) is
         when SparkLex.EndOfText | SparkLex.EndOfLine =>
            null;
         when others =>
            CurrLine.CurrPos := CurrLine.CurrPos + 1;
            CurrLine.LookaheadPos := CurrLine.CurrPos;
      end case;
   end AcceptChar;

   procedure LookaheadChar (Ch : out Character)
   is
   begin
      case CurrLine.Conts (CurrLine.LookaheadPos) is
         when SparkLex.EndOfText =>
            Ch := SparkLex.EndOfText;
         when SparkLex.EndOfLine =>
            Ch := SparkLex.EndOfLine;
         when others =>
            CurrLine.LookaheadPos := CurrLine.LookaheadPos + 1;
            Ch := CurrLine.Conts (CurrLine.LookaheadPos);
      end case;
   end LookaheadChar;

   procedure AcceptLookahead
   is
   begin
      case CurrLine.Conts (CurrLine.LookaheadPos) is
         when SparkLex.EndOfText | SparkLex.EndOfLine =>
            null;
         when others =>
            CurrLine.LookaheadPos := CurrLine.LookaheadPos + 1;
      end case;
      CurrLine.CurrPos := CurrLine.LookaheadPos;
   end AcceptLookahead;

   procedure RejectLookahead
   is
   begin
      CurrLine.LookaheadPos := CurrLine.CurrPos;
   end RejectLookahead;

   procedure NextSigChar (ProgText : in SPARK_IO.File_Type)
   is
      LocPos : EStrings.Lengths;
      SeparatorChar : Boolean;

      procedure GetNextLine (ProgText : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out CurrLine;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives CurrLine,
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        CurrLine,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        ProgText,
      --#                                        SPARK_IO.File_Sys;
      is
         Pos : EStrings.Lengths;
         EndPos : EStrings.Lengths;
         Line   : EStrings.T;
         LineNoBeforeGet : Positive;

         function Separator (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when ' ' |
                  Ada.Characters.Latin_1.HT | Ada.Characters.Latin_1.VT |
                  Ada.Characters.Latin_1.CR | Ada.Characters.Latin_1.LF |
                  Ada.Characters.Latin_1.FF =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Separator;

         procedure IncLineNumber
         --# global in out CurrLine;
         --# derives CurrLine from *;
         is
         begin
            if CurrLine.LineNo = LexTokenManager.Line_Numbers'Last then
               SystemErrors.FatalError (SystemErrors.TooManyFileLines, "in SparkLex.LineManager");
            end if;
            CurrLine.LineNo := CurrLine.LineNo + 1;
         end IncLineNumber;

      begin
         EndPos := 0;
         CurrLine.Conts (0) := ' ';
         loop
            if SPARK_IO.End_Of_Line (ProgText) then
               -- Skip empty line
               if SPARK_IO.End_Of_File (ProgText) then
                  EndPos := 1;
                  CurrLine.Conts (EndPos) := SparkLex.EndOfText;
               else
                  SPARK_IO.Skip_Line (ProgText, 1);
                  IncLineNumber;
               end if;
            else
               -- Attempt to read the line
               LineNoBeforeGet := SPARK_IO.Line (ProgText);
               EStrings.Get_Line (File => ProgText,
                                  E_Str => Line);
               EndPos := EStrings.Get_Length (E_Str => Line);
               IncLineNumber;
               if EndPos < 1 then -- Examiner bug - OK but not acccepted
                  -- Unable to read line, eight-bit character?
                  ErrorHandler.LexError
                     ("Line contains illegal character(s)",
                      "Ignored",
                      LexTokenManager.Lex_Value'
                        (Position  => LexTokenManager.Token_Position'
                           (Start_Line_No => CurrLine.LineNo,
                            Start_Pos     => 2),
                         Token_Str => LexTokenManager.Null_String));
                  SPARK_IO.Skip_Line (ProgText, 1);
                  if LineNoBeforeGet = SPARK_IO.Line (ProgText) then
                     EndPos := 1;
                     CurrLine.Conts (EndPos) := SparkLex.EndOfText;
                  end if;
               else
                  -- got a line!
                  for I in EStrings.Positions range 1 .. EndPos loop
                     CurrLine.Conts (I) := EStrings.Get_Element (E_Str => Line,
                                                                 Pos   => I);
                  end loop;
                  if EndPos = EStrings.Lengths'Last then
                     -- Line too long
                     if LineNoBeforeGet = SPARK_IO.Line (ProgText) then
                        -- skip unread bit of line.
                        SPARK_IO.Skip_Line (ProgText, 1);
                     end if;
                     EndPos := EndPos - 1;
                     CurrLine.Conts (EndPos) := SparkLex.LineTooLong;
                  end if;
               end if;
            end if;

            CurrLine.Conts (EndPos + 1) := SparkLex.EndOfLine;
            -- Skip over leading seperators.
            Pos := 0;
            loop
               exit when not Separator (CurrLine.Conts (Pos));
               exit when Pos = EndPos;
               Pos := Pos + 1;
            end loop;

            exit when not Separator (CurrLine.Conts (Pos));
            exit when CurrLine.Conts (EndPos) = SparkLex.EndOfText;
            exit when CurrLine.Conts (EndPos) = SparkLex.LineTooLong;
         end loop;

         -- Context sensitive annotation continuation check
         if CurrLine.Context = SparkLex.InAnnotation then
            if EndPos - Pos >= 2 and then
               CurrLine.Conts (Pos) = '-'      and then
               CurrLine.Conts (Pos + 1) = '-'  and then
               CurrLine.Conts (Pos + 2) = CommandLineData.Content.AnnoChar then
               null;
            else
               -- Insert end of annotation
               Pos := 0;
               CurrLine.Conts (Pos) := SparkLex.EndOfAnnotation;
            end if;
         end if;

         CurrLine.LineLength := EndPos;
         CurrLine.CurrPos := Pos;
         CurrLine.LookaheadPos := Pos;
      end GetNextLine;

   begin -- NextSigChar
      SeparatorChar := True;
      LocPos := CurrLine.CurrPos;
      while SeparatorChar loop
         case CurrLine.Conts (LocPos) is
            when SparkLex.EndOfLine =>
               GetNextLine (ProgText);
               LocPos := CurrLine.CurrPos;
            when ' ' | Ada.Characters.Latin_1.HT |
                       Ada.Characters.Latin_1.VT |
                       Ada.Characters.Latin_1.LF |
                       Ada.Characters.Latin_1.FF =>
               LocPos := LocPos + 1;  -- Skip separator
            when others =>
               SeparatorChar := False;
         end case;
      end loop;
      CurrLine.LastTokenPos := LocPos;
      CurrLine.CurrPos := LocPos;
      CurrLine.LookaheadPos := LocPos;
   end NextSigChar;

   procedure SetContext (NewContext : in SparkLex.ProgramContext)
   is
   begin
      CurrLine.Context := NewContext;
   end SetContext;

   procedure SetAnnoContext (NewContext : in SparkLex.AnnotationContext)
   is
   begin
      CurrLine.AnnoContext := NewContext;
   end SetAnnoContext;

end LineManager;
