-- $Id: sparkhtml.adb 16643 2010-03-30 16:23:21Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SPARK_IO;
with Ada.Characters.Handling;
with FileSystem;
with ScreenEcho;
with EStrings;
with ELStrings;
with CommandLineData;
with ContextManager.Ops;

use type SPARK_IO.File_Status;
use type FileSystem.TypFileSpecStatus;

package body SparkHTML is

   Error_Reference_File_Name : constant String := "errors.htm";

   -- These states are used in the parsing of the report file.
   type ReportFileStates is (ReportJustStarted,
                             ReportBannerStarted,
                             ReportBannerFinished,
                             ReportDateFound,
                             ReportOptionsFound,
                             ReportIndexFileFound,
                             ReportWarningFileFound,
                             ReportTargetCompilerDataFound,
                             ReportTargetConfigFileFound,
                             ReportSourceExtensionFound,
                             ReportListingExtensionFound,
                             ReportDictionaryFound,
                             ReportReportFileFound,
                             ReportHTMLFlagFound,
                             ReportStatisticsFlagFound,
                             ReportFDLIdentifiersFound,
                             ReportFlowAnalysisFound,
                             ReportLanguageOptionFound,
                             ReportAnnotationCharacterFound,
                             ReportSelectedFilesStarted,
                             ReportIndexFilesStarted,
                             ReportMetaFilesStarted,
                             ReportWarningListStarted,
                             ReportSourceListStarted,
                             ReportMissingFilesStarted,
                             ReportSourceFileStarted,
                             ReportListingFileNameFound,
                             ReportUnitNameFound,
                             ReportNoUnitsFound,
                             ReportUnitTypeFound,
                             ReportAnalysedMessageFound,
                             ReportStartOfErrors,
                             ReportEndOfErrors,
                             ReportJustificationsSummaryFound,
                             ReportSummarizedWarningsFound,
                             ReportLineHeaderFound,
                             ReportErrorSourceLineFound,
                             ReportErrorSourcePointerFound,
                             ReportErrorMessageFound,
                             ReportBlankAfterErrorFound,
                             ReportTargetConfigListStarted,
                             ReportTargetErrorLine,
                             ReportTargetErrorNextLine,
                             ReportEndOfReportFile);

   type ListingFileStates is (ListingJustStarted,
                              ListingBannerStarted,
                              ListingBannerFinished,
                              ListingDateFound,
                              ListingLineHeadingFound,
                              ListingSourceLineFound,
                              ListingErrorSourcePointerFound,
                              ListingErrorMessageFound,
                              ListingEndOfListingFile);

   GenerateHTML : Boolean := True;
   -- Set to false if fatal HTML error occurs to prevent further HTML generation.

   HTMLWorkDir  : ELStrings.T;
   SparkWorkDir : ELStrings.T;

   ---------------------------------------------------------------------------

   -- This function returns true if and only if C is a character representing
   -- a digit.

   function Digit (C : Character) return Boolean
   is

   begin
      return C in '0' .. '9';
   end Digit;

   ---------------------------------------------------------------------------

   -- This function prepends the given string with the name of the selected
   -- direcotry into which HTML is being generated.  The appropriate
   -- target-dependant directory separator is also added.
   --
   -- No error checking is performed.  If the string overflows it is
   -- truncated by the AppendString routines.

   function HTMLFileName (FileName : ELStrings.T)
                         return ELStrings.T
   --# global in CommandLineData.Content;
   is

      ReturnFileName : ELStrings.T;

   begin

      ReturnFileName := ELStrings.Empty_String;
      ELStrings.Append_Examiner_String (E_Str1 => ReturnFileName,
                                        E_Str2 => FileSystem.StartOfDirectory);
      ELStrings.Append_Examiner_String (E_Str1 => ReturnFileName,
                                        E_Str2 => CommandLineData.Content.HTMLDirectory);
      ELStrings.Append_Examiner_String (E_Str1 => ReturnFileName,
                                        E_Str2 => FileSystem.EndOfPath);
      ELStrings.Append_Examiner_Long_String (E_Str1 => ReturnFileName,
                                             E_Str2 => FileName);

      return ReturnFileName;

   end HTMLFileName;


   ---------------------------------------------------------------------------

   -- This function returns InString with all occurences of FromChar changed
   -- to ToChar.
   --
   -- No error checking is necessary.  The for loop ensures that bounds aren't
   -- exceeded.

   function Translate (InString : ELStrings.T;
                       FromChar : Character;
                       ToChar   : Character) return ELStrings.T
   is
      Out_Content : ELStrings.T := ELStrings.Empty_String;
      Success     : Boolean;
   begin
      for I in ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => InString) loop

         if ELStrings.Get_Element (E_Str => InString,
                                   Pos   => I) = FromChar then
            --# accept F, 10, Success, "Ineffective assignment here OK";
            ELStrings.Append_Char (E_Str   => Out_Content,
                                   Ch      => ToChar,
                                   Success => Success);
            --# end accept;
         else
            --# accept F, 10, Success, "Ineffective assignment here OK";
            ELStrings.Append_Char (E_Str   => Out_Content,
                                   Ch      => ELStrings.Get_Element (E_Str => InString,
                                                                     Pos   => I),
                                   Success => Success);
            --# end accept;
         end if;
      end loop;
      --# accept F, 33, Success, "Expect Success unused";
      return Out_Content;
   end Translate;


   ---------------------------------------------------------------------------

   -- This function returns the index of the first character on a line that isn't
   -- a space.
   --
   -- No error checking is necessary.  The for loop ensures that bounds aren't
   -- exceeded.

   function FirstChar (TheString : ELStrings.T) return ELStrings.Positions
   is

      Pos : ELStrings.Positions;

   begin

      Pos := ELStrings.Positions'First;

      for I in ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => TheString) loop
         if ELStrings.Get_Element (E_Str => TheString,
                                   Pos   => I) /= ' ' then
            Pos := I;
            exit;
         end if;
      end loop;

      return Pos;

   end FirstChar;


   ---------------------------------------------------------------------------

   -- This procedure appends the given character to the given ExaminerLongString.
   -- A similar procedure exists in the ExaminerLongStrings package but it has an
   -- overflow parameter.  This is often not necessary for my uses and so this
   -- simpler version avoids the need for local variables and ineffective
   -- assignments.
   --
   -- If the maximum ExaminerLongString length is exceeded then nothing happens.

   procedure AppendCharWithoutCheck (EStr : in out ELStrings.T;
                                     Char : in     Character)
   --# derives EStr from *,
   --#                   Char;
   is
      Success : Boolean;
   begin
      --# accept F, 10, Success, "Ineffective assignment here OK";
      ELStrings.Append_Char (E_Str   => EStr,
                             Ch      => Char,
                             Success => Success);
      --# end accept;
      --# accept F, 33, Success, "Expect Success unused";
   end AppendCharWithoutCheck;

   ---------------------------------------------------------------------------

   -- This function returns the given character if the character can be used in
   -- an HTML name and '_' otherwise.
   -- Characters the are allowed in HTML names are letters ([A-Za-z]), digits ([0-9]
   -- hyphens ("-"), underscores ("_"), colons (":") and periods (".").
   --
   -- No error checking necessary.

   function HTMLNameChar (C : Character) return Character is

      OutChar : Character;

   begin

      if C in 'A' .. 'Z' or
         C in 'a' .. 'z' or
         C in '0' .. '9' or
         C = '-' or
         C = '_' or
         C = ':' or
         C = '.'
      then
         OutChar := C;
      else
         OutChar := '_';
      end if;

      return OutChar;

   end HTMLNameChar;


   ---------------------------------------------------------------------------

   -- Applies HTMLNameChar to each character in the string.
   --
   -- No error checking is necessary.  The for loop ensures that bounds aren't
   -- exceeded.

   function HTMLName (EStr : ELStrings.T) return ELStrings.T
   is

      OutString : ELStrings.T;

   begin

      OutString := ELStrings.Empty_String;

      for I in ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => EStr) loop
         AppendCharWithoutCheck
           (OutString,
            HTMLNameChar (ELStrings.Get_Element (E_Str => EStr,
                                                 Pos   => I)));
      end loop;

      return OutString;

   end HTMLName;


   ---------------------------------------------------------------------------

   -- This function removes all occurrences of special HTML characters and replaces
   -- them with the HTML codes required to display them.
   -- The conversions done are:
   --
   --    >   becomes   &gt;
   --    <   becomes   &lt;
   --    &   becomes   &amp;
   --    "   becomes   &quot;
   --
   -- The for loop ensures that bounds of the input string aren't exceeded.  If
   -- string overflow occurs then the output string is truncated by the Append
   -- routines.

   function ConvertSpecialHTMLChars (Line : ELStrings.T)
                                    return ELStrings.T

   is

      OutString : ELStrings.T;

   begin

      OutString := ELStrings.Empty_String;

      for I in ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => Line) loop

         case ELStrings.Get_Element (E_Str => Line,
                                     Pos   => I) is

            when '<' =>
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => "&lt;");
            when '>' =>
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => "&gt;");
            when '&' =>
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => "&amp;");
            when '"' =>
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => "&quot;");
            when others =>
               AppendCharWithoutCheck
                 (OutString,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
         end case;

      end loop;

      return OutString;

   end ConvertSpecialHTMLChars;

   ---------------------------------------------------------------------------

   -- This function encloses the line given in HTML tags to make it bold type.
   --
   -- If the line given is too long for the tags to be added then it is
   -- returned unchanged.

   function HTMLEmbolden (Line : ELStrings.T)
                         return ELStrings.T
   is
      OutString : ELStrings.T;
   begin
      if ELStrings.Get_Length (E_Str => Line) < ELStrings.Max_String_Length - 7 then
         OutString := ELStrings.Copy_String (Str => "<B>");
         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => Line);
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => "</B>");
      else
         OutString := Line;
      end if;

      return OutString;
   end HTMLEmbolden;

   ---------------------------------------------------------------------------

   -- This function does a target-specific append of a subdirectory name.
   -- The function works with or without trailing separators.
   --
   -- The path returned will always have a trailing separator.

   function AppendDirectoryString (Path : ELStrings.T;
                                   Dir  : EStrings.T)
                                  return ELStrings.T
   is

      PathOut : ELStrings.T;
      DirIn   : ELStrings.T;

   begin

      DirIn := ELStrings.ToExaminerLongString (Dir);

      PathOut := Path;

      -- Add trailing separator if necessary.
      if ELStrings.Get_Element (E_Str => PathOut,
                                Pos   => ELStrings.Get_Length (E_Str => PathOut)) /=
        EStrings.Get_Element (E_Str => FileSystem.BetweenDirectories,
                              Pos   => EStrings.Get_Length (E_Str => FileSystem.BetweenDirectories)) then
         ELStrings.Append_Examiner_String (E_Str1 => PathOut,
                                           E_Str2 => FileSystem.BetweenDirectories);
      end if;

      ELStrings.Append_Examiner_Long_String (E_Str1 => PathOut,
                                             E_Str2 => DirIn);
      ELStrings.Append_Examiner_String (E_Str1 => PathOut,
                                        E_Str2 => FileSystem.EndOfPath);

      return PathOut;

   end AppendDirectoryString;

   ---------------------------------------------------------------------------
   -- This procedure takes two paths and returns the URI of the first path
   -- relative to the second path using the global value of the SparkWorkDir as
   -- the current directory.
   --
   -- The directories passed must have trailing separators.  The directory returned has
   -- a trailing separator.  A separator is defined as being the first character
   -- of the ExaminerLongString returned by FileSystem.BetweenDirectories.
   --
   -- For use on Windows, it is not possible to give a URI in all instances if
   -- the directories exist on two different drives.  For example, if the
   -- paths given are "C:\foo\" and "D:\bar\".
   --
   -- We could simply return the absolute path but if we then want to view
   -- the URI on a Unix system we will find that the path does not exist.
   --
   -- I therefore have included a "Success" parameter which is set to false if
   -- the paths are on 2 different drives.
   --
   procedure HTMLRelativeDir (DirIn, RelativeToIn : in     ELStrings.T;
                              DirRelative         :    out ELStrings.T;
                              SuccessOut          :    out Boolean)
      --# global in SparkWorkDir;
      --# derives DirRelative,
      --#         SuccessOut  from DirIn,
      --#                          RelativeToIn,
      --#                          SparkWorkDir;
   is

      Dir, RelativeTo       : ELStrings.T;
      OutString             : ELStrings.T;
      RemainingDir          : ELStrings.T;
      RemainingRelativeTo   : ELStrings.T;
      PoppedDirectory       : ELStrings.T;

      WorkingDevice         : ELStrings.T;
      DirDevice             : ELStrings.T;
      RelativeToDevice      : ELStrings.T;

      I                     : ELStrings.Positions;

      DirectoryFound        : Boolean;
      WorkingDeviceFound    : Boolean;
      DirDeviceFound        : Boolean;
      RelativeToDeviceFound : Boolean;

      Success               : Boolean;

      -- This function chops the first character from the ExaminerLongString.
      -- If the EmptyString is given then the EmptyString is returned.

      function GetRest (S : ELStrings.T) return ELStrings.T
      is
         Result : ELStrings.T;
      begin
         if ELStrings.Get_Length (E_Str => S) <= 1 then
            Result := ELStrings.Empty_String;
         else
            Result := ELStrings.Section
              (S,
               ELStrings.Positions'First + 1,
               ELStrings.Get_Length (E_Str => S) - 1);
         end if;

         return Result;
      end GetRest;

      -- This function looks for a device name and returns it.
      --
      -- If no device name is present then the EmptyString is returned.
      --
      -- A device name is ususally the name of a Windows drive (of the form "X:")
      function GetDevicePrefix (Path : in ELStrings.T)
                               return ELStrings.T
      is
         ColonFound        : Boolean;
         Pos               : ELStrings.Positions;
         Device            : ELStrings.T;
      begin

         if ELStrings.Get_Length (E_Str => Path) = 0 then

            Device := ELStrings.Empty_String;

         else

            ColonFound  := False;
            Pos         := ELStrings.Get_Length (E_Str => Path);

            for I in ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => Path) loop

               -- Have we found a colon?
               if ELStrings.Get_Element (E_Str => Path,
                                         Pos   => I) = ':' then
                  ColonFound := True;
                  Pos        := I;
                  exit;
               end if;

               -- If we find a directory separator or StartOfDirectory then
               -- exit the loop with ColonFound = False.
               if ELStrings.Get_Element  (E_Str => Path,
                                          Pos   => I) =
                 EStrings.Get_Element (E_Str => FileSystem.StartOfDirectory,
                                       Pos   => 1) or else
                 ELStrings.Get_Element (E_Str => Path,
                                        Pos   => I) =
                 EStrings.Get_Element (E_Str => FileSystem.BetweenDirectories,
                                       Pos   => 1) then
                  exit;  -- ColonFound is already False;
               end if;

            end loop;

            if ColonFound then
               Device := ELStrings.Section (E_Str     => Path,
                                            Start_Pos => ELStrings.Positions'First,
                                            Length    => Pos);
            else
               -- Leave Path as it is.  Set Device to empty string.
               Device := ELStrings.Empty_String;
            end if;

         end if;

         return Device;

      end GetDevicePrefix;

      -- This procedure does a GetDevicePrefix and removes the device name from
      -- the path given.
      procedure ChopDevicePrefix (Path   : in out ELStrings.T;
                                  Device :    out ELStrings.T;
                                  Found  :    out Boolean)
         --# derives Device,
         --#         Found,
         --#         Path   from Path;
      is

         DeviceName  : ELStrings.T;

      begin

         DeviceName := GetDevicePrefix (Path);

         -- GetDevicePrefix returns EmptyString if no device name was found.
         if ELStrings.Get_Length (E_Str => DeviceName) > 0 then

            Path   := ELStrings.Section
              (Path,
               ELStrings.Get_Length (E_Str => DeviceName) + 1,
               ELStrings.Get_Length (E_Str => Path) - ELStrings.Get_Length (E_Str => DeviceName));
            Device := DeviceName;
            Found  := True;

         else

            Device := ELStrings.Empty_String;
            Found  := False;

         end if;

      end ChopDevicePrefix;


      -- This function takes a pathname (must NOT be preceded by an device name) parameter.
      -- If the pathname is absolute (begins with a BetweenDirectories character) then it is
      -- returned unchanged.  Otherwise it is returned with SparkWorkDir prepended to it.
      -- The returned path will not have a device name.
      function MakeAbsolute (Path : ELStrings.T) return ELStrings.T
         --# global in SparkWorkDir;
      is

         OutString   : ELStrings.T;
         Device      : ELStrings.T;

      begin

         if Ada.Characters.Handling.To_Lower (ELStrings.Get_Element (E_Str => Path,
                                                                     Pos   => 1)) =
           Ada.Characters.Handling.To_Lower (EStrings.Get_Element (E_Str => FileSystem.BetweenDirectories,
                                                                   Pos   => 1)) then
            OutString := Path;
         else
            -- Directory is relative to current - append CurrentDir
            OutString := SparkWorkDir;
            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => Path);
         end if;

         -- Remove device name if it exists.
         Device := GetDevicePrefix (OutString);

         -- Length of device name is 0 if no device was found.
         if ELStrings.Get_Length (E_Str => Device) > 0 then
            OutString := ELStrings.Section
              (OutString,
               ELStrings.Get_Length (E_Str => Device) + 1,
               ELStrings.Get_Length (E_Str => OutString) - ELStrings.Get_Length (E_Str => Device));
         end if;

         return OutString;

      end MakeAbsolute;

      -- This procedure returns and removes the first subdirectory name from
      -- the given path.  The given path must not have an device name or a
      -- leading separator.  The popped subdirectory name will also not have a
      -- leading separator. The remaining directory name has no leading
      -- separator.
      --
      -- e.g. if the input path is "foo/bar/baz/" then the path returned
      -- will be "bar/baz/" and the subdirectory returned will be "foo".
      --
      -- If the URL flag is set then the separator '/' is used rather than
      -- a target specific separator.
      --
      -- DirFound is true if and only if a directory separator was found in the
      -- path.
      --
      -- If the path has length 0 or a directory separator is not found then
      --       - Path is returned unchanged
      --       - Dir is ELStrings.EmptyString
      --       - Dirfound is false
      --
      procedure PopSubDir (Path     : in out ELStrings.T;
                           URL      : in     Boolean;
                           Dir      :    out ELStrings.T;
                           DirFound :    out Boolean)
         --# derives Dir,
         --#         Dirfound,
         --#         Path     from Path,
         --#                       URL;
      is

         BetweenSeparator : ELStrings.T;
         EndSeparator     : ELStrings.T;
         SeparatorPos     : ELStrings.Positions;
         SeparatorFound   : Boolean;

      begin

         if ELStrings.Get_Length (E_Str => Path) = 0 then

            Dir := ELStrings.Empty_String;
            DirFound := False;

         else

            if URL then

               BetweenSeparator := ELStrings.Copy_String (Str => "/");
               EndSeparator := BetweenSeparator;

            else

               BetweenSeparator := ELStrings.ToExaminerLongString (FileSystem.BetweenDirectories);
               EndSeparator     := ELStrings.ToExaminerLongString (FileSystem.EndOfPath);

            end if;

            ELStrings.Find_Examiner_Sub_String (E_Str         => Path,
                                                Search_String => BetweenSeparator,
                                                String_Found  => SeparatorFound,
                                                String_Start  => SeparatorPos);

            if not SeparatorFound then
               -- Maybe last directory?
               ELStrings.Find_Examiner_Sub_String (E_Str         => Path,
                                                   Search_String => EndSeparator,
                                                   String_Found  => SeparatorFound,
                                                   String_Start  => SeparatorPos);
            end if;

            if SeparatorFound then

               Dir := ELStrings.Section
                 (Path,
                  ELStrings.Positions'First,
                  SeparatorPos - 1);
               Path := ELStrings.Section
                 (Path,
                  SeparatorPos + 1,
                  ELStrings.Get_Length (E_Str => Path) - SeparatorPos);
               DirFound := True;

            else

               Dir := ELStrings.Empty_String;
               DirFound := False;

            end if;

         end if;

      end PopSubDir;

      -- This function removes the last directory in the URL given.
      -- The input path should have a trailing separator and the output
      -- path will also have a trailing separator.
      --
      -- If no separator is found then the string is returned unchanged.
      --
      -- e.g. RemoveLastDirectory ("foo/bar/baz/") = "foo/bar/"
      --
      -- This routine is used when removing "../" from pathnames.
      --
      -- NOTE: This only works with URLs - the directory separator must be '/'
      --

      function RemoveLastDirectory (Path : ELStrings.T)
                                   return ELStrings.T
      is

         OutString : ELStrings.T;
         Pos       : ELStrings.Lengths;

      begin

         OutString := Path;
         Pos       := ELStrings.Get_Length (E_Str => Path);

         -- Remember not to include the trailing BetweenDirectories character in the search.
         for I in reverse ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => Path) - 1 loop
            if ELStrings.Get_Element (E_Str => Path,
                                      Pos   => I) = '/' then
               Pos := I;
               exit;
            end if;
         end loop;

         -- Pos will still equal Path.Length if separator was not found.
         if Pos /= ELStrings.Get_Length (E_Str => Path) then  -- separator was found
            OutString := ELStrings.Section (E_Str     => Path,
                                            Start_Pos => ELStrings.Positions'First,
                                            Length    => Pos);
         end if;

         return OutString;

      end RemoveLastDirectory;

      -- This function resolves a URL containing references to the previous
      -- directory "../" and the current directory "./".  The path given must have
      -- a trailing '/' character.
      -- The function works with or without a leading directory separator (this is
      -- copied if it exists.
      --
      -- NOTE: This function only works with URL's.  The directory separator
      --       used is always '/' and current and parent directories are
      --       "." and ".." respectively.

      function RemoveDots (InPath : ELStrings.T)
                          return ELStrings.T
      is

         SubDirFound : Boolean;
         Path        : ELStrings.T;
         OutPath     : ELStrings.T;
         NextDir     : ELStrings.T;

      begin

         Path    := InPath;
         OutPath := ELStrings.Empty_String;

         if ELStrings.Get_Element (E_Str => Path,
                                   Pos   => ELStrings.Positions'First) = '/' then
            -- Copy the leading separator.
            Path := GetRest (Path);
            AppendCharWithoutCheck (OutPath, '/');
         end if;

         loop

            PopSubDir (Path, True, NextDir, SubDirFound);

            exit when not SubDirFound;

            if ELStrings.Eq1_String (E_Str => NextDir,
                                     Str   => ".") then
               null; -- do nothing
            elsif ELStrings.Eq1_String (E_Str => NextDir,
                                        Str   => "..") then
               OutPath := RemoveLastDirectory (OutPath);
            else
               ELStrings.Append_Examiner_Long_String (E_Str1 => OutPath,
                                                      E_Str2 => NextDir);
               AppendCharWithoutCheck (OutPath, '/');
            end if;

         end loop;

         return OutPath;

      end RemoveDots;

      -- This function converts a directory name into a URL.
      function ConvertToURL (Path : ELStrings.T) return ELStrings.T
      is
      begin
         -- On Windows, pathnames might contain '\' which need
         -- to be transformed unto '/' to be a URL.  On other
         -- platforms, this is a no-op.
         return Translate (Path, '\', '/');
      end ConvertToURL;

   begin  -- HTMLRelativeDir

      -- Initialise variables
      Success    := True;
      OutString  := ELStrings.Empty_String;

      -- Copy input parameters to local variables.
      Dir := DirIn;
      RelativeTo := RelativeToIn;

      -- Get device names.
      -- Device names are removed from the input directories.
      -- SparkWorkDir is not to be modified.

      WorkingDevice := GetDevicePrefix  (SparkWorkDir);
      WorkingDeviceFound := ELStrings.Get_Length (E_Str => WorkingDevice) > 0;

      ChopDevicePrefix (Dir, DirDevice, DirDeviceFound);
      ChopDevicePrefix (RelativeTo, RelativeToDevice, RelativeToDeviceFound);

      -- We cannot create links if files are on different NT drives or different
      -- VAX devices.
      if DirDeviceFound then
         --Dir contains a device name.
         if RelativeToDeviceFound then
            -- RelativeTo also contains a device name.  Are they the same?
            -- Fail if different devices.
            if not ELStrings.Eq_String (E_Str1 => DirDevice,
                                        E_Str2 => RelativeToDevice) then
               Success := False;  -- Files are on different drives.
            end if;
         else
            -- Dir contains a device name and RelativeTo is on the current drive.
            -- Is the current device equal to the device that Dir is on?
            if (not WorkingDeviceFound) or else
              (not ELStrings.Eq_String (E_Str1 => DirDevice,
                                        E_Str2 => WorkingDevice)) then
               Success := False;
            end if;
         end if;
      else
         -- Dir does not contain a drive specification - does RelativeTo?
         if RelativeToDeviceFound then
            -- RelativeTo contains a device name and Dir is on the current drive.
            -- Is the current device equal to the device that RelativeTo is on?
            if (not WorkingDeviceFound) or else
              (not ELStrings.Eq_String (E_Str1 => RelativeToDevice,
                                        E_Str2 => WorkingDevice)) then
               Success := False;
            end if;
         end if;
      end if;

      -- Do nothing (return empty string) if directories are identical, or if
      -- the previous checks failed.
      if (not ELStrings.Eq_String (E_Str1 => Dir,
                                   E_Str2 => RelativeTo)) and then
        Success then
         -- Make directories absolute
         Dir        := MakeAbsolute (Dir);
         RelativeTo := MakeAbsolute (RelativeTo);

         -- Convert directories to URL's.
         Dir        := ConvertToURL (Dir);
         RelativeTo := ConvertToURL (RelativeTo);

         -- Remove "./" and "../" and make case-insensitive where required.
         Dir := ELStrings.ToExaminerLongString
           (ELStrings.ToExaminerString (RemoveDots (Dir)));
         RelativeTo := ELStrings.ToExaminerLongString
           (ELStrings.ToExaminerString (RemoveDots (RelativeTo)));

         -- Initialize counter.
         I := ELStrings.Positions'First;

         -- Skip common prefix.  We want I to point to the character that begins
         -- the name of the first different subdirectory.  e.g. if we have
         -- /foo/bar/ and /foo/baz/ we want to point to the 'b'
         loop
            exit when I > ELStrings.Get_Length (E_Str => Dir) or else
              I > ELStrings.Get_Length (E_Str => RelativeTo) or else
              ELStrings.Get_Element (E_Str => Dir,
                                     Pos   => I) /=
              ELStrings.Get_Element (E_Str => RelativeTo,
                                     Pos   => I);
            I := I + 1;
         end loop;

         if I < ELStrings.Get_Length (E_Str => Dir) and then
           I < ELStrings.Get_Length (E_Str => RelativeTo) then
            -- Back up to previous directory separator (in case we're comparing, e.g.
            -- \foo\bar and \foo\baz, in which case the common prefix is c:\foo\ba)
            while ELStrings.Get_Element (E_Str => Dir,
                                         Pos   => I) /= '/'
               and I > ELStrings.Positions'First loop
               I := I - 1;
            end loop;

            -- Now we want to point to just past the separator so
            I := I + 1;
         end if;

         -- Now remove the common ancestor directories.
         if I > ELStrings.Get_Length (E_Str => Dir) then
            RemainingDir := ELStrings.Empty_String;
         else
            RemainingDir := ELStrings.Section
              (Dir,
               I,
               (ELStrings.Get_Length (E_Str => Dir) - I) + 1);
         end if;

         if I > ELStrings.Get_Length (E_Str => RelativeTo) then
            RemainingRelativeTo := ELStrings.Empty_String;
         else
            RemainingRelativeTo := ELStrings.Section
              (RelativeTo,
               I,
               (ELStrings.Get_Length (E_Str => RelativeTo) - I) + 1);
         end if;

         -- For each subdirectory remaining in RelativeTo we add a "../" to OutString
         loop
            --# accept Flow, 10, PoppedDirectory, "Expected ineffective assignment to PoppedDirectory";
            PopSubDir (RemainingRelativeTo,
                       True,
                       PoppedDirectory,
                       DirectoryFound);  -- Flow error expected - PoppedDir not used.
            --# end accept;

            exit when not DirectoryFound;

            ELStrings.Append_String (E_Str => OutString,
                                     Str   => "../");

         end loop;

         -- For each subdirectory remaining in Dir we add "<subdir>/"
         loop

            PopSubDir (RemainingDir, True, PoppedDirectory, DirectoryFound);
            exit when not DirectoryFound;

            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => PoppedDirectory);
            ELStrings.Append_String (E_Str => OutString,
                                     Str   => "/");

         end loop;

      end if;

      DirRelative := OutString;
      SuccessOut  := Success;

   end HTMLRelativeDir;  -- Flow error expected.  SparkWorkDir is NOT updated.

   ---------------------------------------------------------------------------

   -- This procedure uses HTMLRelativeDir to give the URI of a file relative
   -- to a given directory and the global SparkWorkDir as the current directory.
   --
   -- The

   procedure HTMLRelativeFile (File, RelativeTo  : in     ELStrings.T;
                               FileRelative      :    out ELStrings.T;
                               SuccessOut        :    out Boolean)
      --# global in SparkWorkDir;
      --# derives FileRelative,
      --#         SuccessOut   from File,
      --#                           RelativeTo,
      --#                           SparkWorkDir;
   is

      DirectoryName   : ELStrings.T;
      RelativeDirName : ELStrings.T;
      FileName        : ELStrings.T;
      OutString       : ELStrings.T;
      SeparatorPos    : ELStrings.Positions;
      Success         : Boolean;

   begin

      SeparatorPos := ELStrings.Get_Length (E_Str => File);

      -- Get the filename and extension.  Filename might have a path
      -- or a drive name as a prefix.
      for I in reverse ELStrings.Positions range ELStrings.Positions'First .. ELStrings.Get_Length (E_Str => File) loop
         if ELStrings.Get_Element (E_Str => File,
                                   Pos   => I) =
           EStrings.Get_Element (E_Str => FileSystem.EndOfPath,
                                 Pos   => EStrings.Get_Length (E_Str => FileSystem.EndOfPath)) or else
           ELStrings.Get_Element (E_Str => File,
                                  Pos   => I) = ':' then
            SeparatorPos := I;
            exit;
         end if;
      end loop;

      -- Separate the file and directory name.
      if SeparatorPos = ELStrings.Get_Length (E_Str => File) then -- no pathname given, just filename.
         DirectoryName := ELStrings.Empty_String;
         FileName      := File;
      else
         DirectoryName := ELStrings.Section
           (File,
            ELStrings.Positions'First,
            SeparatorPos);
         FileName      := ELStrings.Section
           (File,
            SeparatorPos + 1,
            ELStrings.Get_Length (E_Str => File) - SeparatorPos);
      end if;

      -- Interpret the directory name as a URL relative to RelativeTo
      HTMLRelativeDir (DirectoryName, RelativeTo, RelativeDirName, Success);

      -- If the relative directory operation was successful then we use the
      -- new directory name.  Otherwise we simply return the filename given.
      if Success then
         OutString := RelativeDirName;
         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => FileName);
      else
         OutString := File;
      end if;

      FileRelative := OutString;
      SuccessOut   := Success;

   end HTMLRelativeFile;

   ---------------------------------------------------------------------------

   -- This procedure is global as it is used by both the report and listing file
   -- translators.
   --
   -- This procedure processes an error message line.  Some errors get wrapped over
   -- more than one line and this procedure should only be used in generating
   -- HTML for the first line of any error message.
   --
   -- It assumes that the line is formatted as follows:
   --
   --   <flash> (nnnn)  <error_type> : <xxx> : <error_message>
   --
   -- The actual pattern matched is slightly different but we only expect lines
   -- of this type (or the special cases described below) and lines of this type
   -- will be parsed correctly.
   --
   -- The <error_type> and the error code <xxx> are used to generate an HTML
   -- link of the form <A HREF="error_file#error_link">, where error_file is
   -- the (relative) location of the error messages file and error_link is the
   -- name of an anchor in the file, typically made from the first word of the
   -- error type (converted to lower case) and the error number, separated by a hyphen.
   -- Note that there are some special cases where the name of the anchor is not
   -- constructed in this way.
   --
   -- The HTML link is placed before the start of the line (the closing tag is
   -- placed at the end of the line).  The link is also returned in the ErrorLink
   -- out parameter so that it can be saved for use in any continuations
   -- of this message line.
   --
   -- For example, if the Line parameter is the ExaminerLongString:
   --
   --    *** (419)   Semantic Error      : 1: Message
   --
   -- The line output will be:
   --
   -- <A HREF="error_file#semantic-1">  *** (419)   Semantic Error      : 1: Message</A>
   --
   -- and the ErrorLink parameter will be:
   --
   -- <A HREF="error_file#semantic-1">
   --
   -- If no tag is added then the ErrorLink parameter is set to EmptyString.
   --
   -- The special cases (i.e. those which don't fit the above format, mainly
   -- because they don't have error numbers) are handled by the procedure
   -- HandleSpecialCases as described below.
   procedure ProcessFirstErrorMessageLine (Line       : in out ELStrings.T;
                                           Lookahead  : in     ELStrings.T;
                                           ErrorLink  :    out ELStrings.T)
      --# global in out SPARK_IO.FILE_SYS;
      --# derives ErrorLink,
      --#         Line              from Line,
      --#                                Lookahead &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                Line,
      --#                                Lookahead;
   is

      SavedLine   : ELStrings.T;
      ErrorType   : ELStrings.T;
      ErrorNo     : ELStrings.T;
      OutString   : ELStrings.T;
      Char        : Character;
      Char2       : Character;
      Char3       : Character;
      Error       : Boolean;
      SpecialCase : Boolean;

      -- The function FlashCharacter is true if and only if the input is
      -- a character used by the Examiner in a "flash" at the start of
      -- an error message.

      function FlashCharacter (Char : Character) return Boolean
      is
      begin
         return Char = '?' or else
            Char = '-' or else
            Char = '+' or else
            Char = '*' or else
            Char = '!';
      end FlashCharacter;
      pragma Inline (FlashCharacter);

      -- This procedure handles the special cases of error messages that are not
      -- of the form described in the commentary for ProcessFirstErrorMessageLine.
      -- These special cases are mainly errors that don't have error numbers
      -- associated with them or that have no reference in the error reference
      -- file.
      --
      -- The "special case" errors are as follows:
      --
      --   *** Syntax Error : ";" expected.
      --   *** Syntax Error : No APRAGMA can be start with reserved word "IS".
      --   *** Syntax Error : No complete PROCEDURE_SPECIFICATION can be followed by ANNOTATION_START here.
      --   *** Syntax Error : No complete PROCEDURE_SPECIFICATION can be followed by reserved word "IS" here.
      --   *** Syntax Error : reserved word "INHERIT" expected.
      --   Any other syntax errors (these won't have links)
      --   Warning : No semantic checks carried out, text may not be legal SPARK.
      --   Any lexical errors (these won't have links)
      --
      --   The line output is the line input but with an HTML link to the correct
      --   place in the error reference file (or no link if no such reference
      --   exists).
      --
      --   The error_link parameter is set to the HTML tag used to generate the link
      --   or, if not link is generated, the empty string.
      --
      --   The SpecialCase parameter is set to true if and only if a special case
      --   was found.  If this parameter is set then the caller should not try
      --   to process the error message line in the usual way.
      --
      --   The procedure uses a fairly naive SubString search mechanism that could
      --   potentially incorrectly match a string and flag it as a SpecialCase
      --   but this is unlikely because when this procedure is called we know that
      --   we have a line containing an error message and we know what the error
      --   messages are.
      procedure HandleSpecialCases (Line        : in out ELStrings.T;
                                    Lookahead   : in     ELStrings.T;
                                    ErrorLink   :    out ELStrings.T;
                                    SpecialCase :    out Boolean)
         --# derives ErrorLink,
         --#         Line        from Line,
         --#                          Lookahead &
         --#         SpecialCase from Line;

      is

         AddTag           : Boolean;
         SyntaxErrorFound : Boolean;
         WarningFound     : Boolean;
         LexicalFound     : Boolean;
         ErrorLinkName    : ELStrings.T;
         ErrorLinkString  : ELStrings.T;
         OutString        : ELStrings.T;
         String_Found     : Boolean;
         String_Start     : ELStrings.Positions;

      begin

         AddTag        := False;
         SpecialCase   := False;
         ErrorLink     := ELStrings.Empty_String;
         ErrorLinkName := ELStrings.Empty_String;

         -- Check for the various error types that have special cases.

         --# accept F, 10, String_Start, "Ineffective assignment here OK";
         ELStrings.Find_Sub_String (E_Str         => Line,
                                    Search_String => "Syntax Error",
                                    String_Found  => SyntaxErrorFound,
                                    String_Start  => String_Start);
         ELStrings.Find_Sub_String (E_Str         => Line,
                                    Search_String => "Warning",
                                    String_Found  => WarningFound,
                                    String_Start  => String_Start);
         ELStrings.Find_Sub_String (E_Str         => Line,
                                    Search_String => "Lexical Error",
                                    String_Found  => LexicalFound,
                                    String_Start  => String_Start);
         --# end accept;
         if SyntaxErrorFound then

            -- HTML directives:
            --! <NameFormat> <"syntax-"><Name>
            --! <ErrorFormat> <"*** Syntax Error : "><Error>

            -- HTML output
            --! <Name> semicolon-expected
            --! <Error> &quot;;&quot; expected.
            --! If this is reported at the end of the input file it may well
            --!  be caused by the misspelling of an identifier in a hide directive.
            --!  The parser then skips all the following text looking for the
            --!  misspelled identifier but finds the end of file first where it
            --!  reports a syntax error.

            --! <Name> no-apragma
            --! <Error> No APRAGMA can be start with reserved word &quot;IS&quot;
            --! This can occur when a stub for an embedded subprogram is wrongly
            --! terminated by a semicolon.

            --! <Name> procedure-spec-annotation-start
            --! <Error> No complete PROCEDURE_SPECIFICATION can be followed by ANNOTATION_START here.
            --! This can occur when the reserved word </i>body<i> has been
            --! omitted from the declaration of a package body. This error
            --! will occur at the annotation placed between the
            --! specification and the reserved word </i>is<i> of the first
            --! subprogram.

            --! <Name> procedure-spec-is
            --! <Error> No complete PROCEDURE_SPECIFICATION can be followed by reserved word &quot;IS&quot; here.
            --! This can occur when the reserved word </i>body<i> has been omitted
            --! from the declaration of a package body. This error will occur at the
            --! reserved word </i>is<i> which introduces the body of the first subprogram.

            --! <Name> inherit-expected
            --! <Error> reserved word &quot;INHERIT&quot; expected.
            --! This occurs where the annotation on a subprogram body is placed after
            --! the reserved word </i>is<i> instead of before it.

            --! <Name> simple-expression-rbracket
            --! <Error> No complete SIMPLE_EXPRESSION can be followed by &quot;)&quot; here.
            --! This can occur in an aggregate expression when there is a mixure of
            --! named and positional association being used.

            --! <Name> simple-expression-comma
            --! <Error> No complete SIMPLE_EXPRESSION can be followed by &quot;,&quot; here.
            --! This can occur in an aggregate expression when there is a mixure of
            --! named and positional association being used.

            -- All syntax errors are special cases.
            SpecialCase := True;
            --# accept F, 10, String_Start, "Ineffective assignment here OK";
            ELStrings.Find_Sub_String (E_Str         => Line,
                                       Search_String => ";&quot; expected",
                                       String_Found  => String_Found,
                                       String_Start  => String_Start);
            --# end accept;
            if String_Found then
               ErrorLinkName := ELStrings.Copy_String (Str => "syntax-semicolon-expected");
               AddTag := True;
            end if;
            --# accept F, 10, String_Start, "Ineffective assignment here OK";
            ELStrings.Find_Sub_String (E_Str         => Line,
                                       Search_String => "No APRAGMA",
                                       String_Found  => String_Found,
                                       String_Start  => String_Start);
            --# end accept;
            if String_Found then
               ErrorLinkName := ELStrings.Copy_String (Str => "syntax-no-apragma");
               AddTag := True;
            end if;
            --# accept F, 10, String_Start, "Ineffective assignment here OK";
            ELStrings.Find_Sub_String (E_Str         => Line,
                                       Search_String => "No complete PROCEDURE_SPECIFICATION",
                                       String_Found  => String_Found,
                                       String_Start  => String_Start);
            --# end accept;
            if String_Found then
               --# accept F, 10, String_Start, "Ineffective assignment here OK";
               ELStrings.Find_Sub_String (E_Str         => Lookahead,
                                          Search_String => "_START here.",
                                          String_Found  => String_Found,
                                          String_Start  => String_Start);
               --# end accept;
               if String_Found then

                  ErrorLinkName := ELStrings.Copy_String (Str => "syntax-procedure-spec-annotation-start");
                  AddTag := True;

               end if;
               --# accept F, 10, String_Start, "Ineffective assignment here OK";
               ELStrings.Find_Sub_String (E_Str         => Lookahead,
                                          Search_String => "&quot;IS&quot; here.",
                                          String_Found  => String_Found,
                                          String_Start  => String_Start);
               --# end accept;
               if String_Found then

                  ErrorLinkName := ELStrings.Copy_String (Str => "syntax-procedure-spec-is");
                  AddTag := True;

               end if;

            end if;
            --# accept F, 10, String_Start, "Ineffective assignment here OK";
            ELStrings.Find_Sub_String (E_Str         => Line,
                                       Search_String => "No complete SIMPLE_EXPRESSION",
                                       String_Found  => String_Found,
                                       String_Start  => String_Start);
            --# end accept;
            if String_Found then
               --# accept F, 10, String_Start, "Ineffective assignment here OK";
               ELStrings.Find_Sub_String (E_Str         => Lookahead,
                                          Search_String => "&quot;)&quot; here.",
                                          String_Found  => String_Found,
                                          String_Start  => String_Start);
               --# end accept;
               if String_Found then

                  ErrorLinkName := ELStrings.Copy_String (Str => "syntax-simple-expression-rbracket");
                  AddTag := True;

               end if;
               --# accept F, 10, String_Start, "Ineffective assignment here OK";
               ELStrings.Find_Sub_String (E_Str         => Lookahead,
                                          Search_String => "&quot;,&quot; here.",
                                          String_Found  => String_Found,
                                          String_Start  => String_Start);
               --# end accept;
               if String_Found then

                  ErrorLinkName := ELStrings.Copy_String (Str => "syntax-simple-expression-comma");
                  AddTag := True;

               end if;

            end if;
            --# accept F, 10, String_Start, "Ineffective assignment here OK";
            ELStrings.Find_Sub_String (E_Str         => Line,
                                       Search_String => "reserved word &quot;INHERIT&quot; expected",
                                       String_Found  => String_Found,
                                       String_Start  => String_Start);
            --# end accept;
            if String_Found then
               ErrorLinkName := ELStrings.Copy_String (Str => "syntax-inherit-expected");
               AddTag := True;
            end if;

         elsif WarningFound then

            -- Not all warnings are special cases - only set SpecialCase to True if
            -- a special case is detected.
            --# accept F, 10, String_Start, "Ineffective assignment here OK";
            ELStrings.Find_Sub_String (E_Str         => Line,
                                       Search_String => "No semantic checks carried out,",
                                       String_Found  => String_Found,
                                       String_Start  => String_Start);
            --# end accept;
            if String_Found then
               ErrorLinkName := ELStrings.Copy_String (Str => "warning-no-semantic-checks");
               AddTag := True;
               SpecialCase := True;
            end if;

         elsif LexicalFound then

            -- All lexical errors are special cases.
            SpecialCase := True;

            -- Lexical errors are not included in the file of error explanations
            -- so no processing is done.

         end if;

         if AddTag then
            ErrorLinkString := ELStrings.Copy_String (Str => "<A TARGET=""rbottom"" HREF=""");
            ELStrings.Append_String (E_Str => ErrorLinkString,
                                     Str   => Error_Reference_File_Name);
            ELStrings.Append_String (E_Str => ErrorLinkString,
                                     Str   => "#");
            ELStrings.Append_Examiner_Long_String (E_Str1 => ErrorLinkString,
                                                   E_Str2 => ErrorLinkName);
            ELStrings.Append_String (E_Str => ErrorLinkString,
                                     Str   => """>");
            ErrorLink := ErrorLinkString;

            OutString := ErrorLinkString;
            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => Line);
            ELStrings.Append_String (E_Str => OutString,
                                     Str   => "</A>");
            Line := OutString;
         end if;
         --# accept F, 33, String_Start, "Expect String_Start unused";
      end HandleSpecialCases;

   begin  -- ProcessFirstErrorMessageLine

      HandleSpecialCases (Line, Lookahead, ErrorLink, SpecialCase);

      if not SpecialCase then

         Error := False;

         ErrorType := ELStrings.Empty_String;
         ErrorNo   := ELStrings.Empty_String;

         SavedLine := Line;

         -- Get characters until flash is found (either ???, ---, +++, ***, !!!)
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when FlashCharacter (Char) or else
              ELStrings.Is_Empty (E_Str => Line);
         end loop;

         -- Look for two more flash characters
         ELStrings.Pop_Char (E_Str => Line,
                             Char  => Char2);
         ELStrings.Pop_Char (E_Str => Line,
                             Char  => Char3);

         if Char2 /= Char or Char3 /= Char then
            Error := True;
         end if;

         -- Look for a space and a '('
         ELStrings.Pop_Char (E_Str => Line,
                             Char  => Char2);
         ELStrings.Pop_Char (E_Str => Line,
                             Char  => Char3);

         if Char2 /= ' ' or Char3 /= '(' then
            Error := True;
         end if;

         -- Skip line number (up to next ')')
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char = ')' or else
              ELStrings.Is_Empty (E_Str => Line);
         end loop;

         -- Skip whitespace
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char /= ' ' or else
              ELStrings.Is_Empty (E_Str => Line);
         end loop;

         -- Char is first character of ErrorType
         AppendCharWithoutCheck (ErrorType, Char);

         -- Get rest of ErrorType (up to next ' ')
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char = ' ' or else
              ELStrings.Is_Empty (E_Str => Line);
            AppendCharWithoutCheck (ErrorType, Char);
         end loop;

         -- Skip up to colon
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char = ':' or else
              ELStrings.Is_Empty (E_Str => Line);
         end loop;

         -- Skip whitespace
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char /= ' ' or else
              ELStrings.Is_Empty (E_Str => Line);
         end loop;

         -- Char is first character of ErrorNo
         AppendCharWithoutCheck (ErrorNo, Char);

         -- Get rest of ErrorNo (up to next ':')
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char = ':' or else
              ELStrings.Is_Empty (E_Str => Line);
            AppendCharWithoutCheck (ErrorNo, Char);
         end loop;

         -- If an error occurred report this and show which line it occurred on.
         if Error or else
           ELStrings.Is_Empty (E_Str => Line) then
            ScreenEcho.Put_Line ("An error occurred while parsing the following error message line.");
            ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => SavedLine);
         end if;

         ErrorType := ELStrings.Lower_Case (E_Str => ErrorType);

         -- Generate the output string.
         OutString := ELStrings.Copy_String (Str => "<A HREF=""");
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => Error_Reference_File_Name);
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => "#");
         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => ErrorType);
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => "-");
         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => ErrorNo);
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => """ TARGET=""rbottom"">");

         ErrorLink := OutString;

         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => SavedLine);
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => "</A>");

         Line := OutString;

      end if;

      Line := HTMLEmbolden (Line);

   end ProcessFirstErrorMessageLine;

   -- This procedure processes subsequent error message lines.  These contain
   -- word-wrapped portions of the error message, but no flash, error number
   -- and so on.  We therefore need to use the tag stored in SavedErrorLink
   -- as the link for this line.
   --
   -- The line is assumed to have some spaces at the front of the it.  Placing the
   -- tag before these turns the spaces into links which looks really silly, so
   -- the tag is placed immediately before the first non-space character.
   --
   -- If SavedErrorLink is EmptyString then no tag is added.

   procedure ProcessNextErrorMessageLine (Line : in out ELStrings.T;
                                          Link : in     ELStrings.T)
      --# derives Line from *,
      --#                   Link;
   is

      OutString : ELStrings.T;
      Char      : Character;

   begin

      if not ELStrings.Is_Empty (E_Str => Line) then
         OutString := ELStrings.Empty_String;

         -- Copy up to first non-space character
         loop
            ELStrings.Pop_Char (E_Str => Line,
                                Char  => Char);
            exit when Char /= ' ';
            AppendCharWithoutCheck (OutString, Char);
         end loop;

         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => Link);

         AppendCharWithoutCheck (OutString, Char);
         ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                E_Str2 => Line);
         ELStrings.Append_String (E_Str => OutString,
                                  Str   => "</A>");

         Line := OutString;

      end if;

      Line := HTMLEmbolden (Line);

   end ProcessNextErrorMessageLine;

   --
   --  -------------
   --  InitSparkHTML
   --  -------------
   --
   --  This procedure initialises the HTML output of the examiner.
   --
   --  It is assumed that the HTML output flag has already been tested.  The call to
   --  InitSparkHTML should look something like:
   --
   --    if CommandLineData.Content.HTML then
   --       SparkHTML.InitSparkHTML;
   --    end if;
   --
   --  The procedure does exactly the following:
   --
   --     - creates a subdirectory of the current directory if one
   --       doesn't already exist.  The directory name is given by
   --       CommandLineData.Content.HTMLDirectory
   --     - prints a message to the screen informing the user that HTML output
   --       has been requested;
   --     - creates a top-level frames file in the HTML directory as described in
   --       the design;
   --     - creates an empty HTML file to fill the bottom frame (this is so that
   --       Netscape Navigator behaves correctly);
   --     - sets GenerateHTML to false if anything went wrong, to prevent further
   --       HTML generation.
   --
   --  Error trapping:
   --
   --  If anything unusual happens then the procedure will echo a message to the
   --  screen saying what went wrong and, in the case of errors which should stop
   --  HTML generation, GenerateHTML will be set to "false".
   --
   --  Failed HTML initialisation should not affect the analysis.  If intialisation
   --  fails, the GenerateHTML flag will stop calls to other SparkHTML routines
   --  from doing anything.  It is therefore safe to make those calls.

   procedure InitSparkHTML is

      NewDirOK    : Boolean;
      FrameFileOK : Boolean;
      DirName     : EStrings.T;

      -- This procedure writes HTML content to the base frame file.
      -- The procedure does not open or close the frame file, it assumes
      -- that a file has been opened.
      -- The FrameFile parameter must be the handle of an open file.
      -- The ReportFileName parameter is the location of the file to
      -- go in the top frame, relative to the frame file.
      -- The BlankfileName parameter is the location of the file to go
      -- in the top frame, relative to the frame file.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.  The SPARK_IO
      -- procedures called do not return status information and so error
      -- trapping is not possible here.

      procedure WriteFrameFile (FrameFile      : in SPARK_IO.File_Type;
                                ReportFileName : in ELStrings.T;
                                BlankFileName  : in ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                BlankFileName,
         --#                                CommandLineData.Content,
         --#                                FrameFile,
         --#                                ReportFileName;
      is
         TempExaminerLongString : ELStrings.T;
      begin
         SPARK_IO.Put_Line (FrameFile, "<!DOCTYPE HTML PUBLIC ""-//W3C//DTD HTML 4.0 Frameset//EN""", 57);
         SPARK_IO.Put_Line (FrameFile, "            ""http://www.w3.org/TR/REC-html40/frameset.dtd"">", 59);
         SPARK_IO.Put_Line (FrameFile, "<html>", 6);
         SPARK_IO.Put_Line (FrameFile, "  <head>", 8);
         SPARK_IO.Put_Line (FrameFile, "    <title>SPARK HTML Output</title>", 36);
         SPARK_IO.Put_Line (FrameFile, "  </head>", 9);
         SPARK_IO.Put_Line (FrameFile, "  <frameset rows=""60%,*"">", 25);

         TempExaminerLongString := ELStrings.Copy_String (Str => "    <frame name=""rtop"" scrolling=""auto"" src=""");
         ELStrings.Append_Examiner_Long_String (E_Str1 => TempExaminerLongString,
                                                E_Str2 => ReportFileName);
         ELStrings.Append_String (E_Str => TempExaminerLongString,
                                  Str   => """>");

         if CommandLineData.Content.PlainOutput then
            TempExaminerLongString := ELStrings.Lower_Case (E_Str => TempExaminerLongString);
         end if;
         ELStrings.Put_Line (File  => FrameFile,
                             E_Str => TempExaminerLongString);

         TempExaminerLongString := ELStrings.Copy_String (Str => "    <frame name=""rbottom"" scrolling=""auto"" src=""");
         ELStrings.Append_Examiner_Long_String (E_Str1 => TempExaminerLongString,
                                                E_Str2 => BlankFileName);
         ELStrings.Append_String (E_Str => TempExaminerLongString,
                                  Str   => """>");

         ELStrings.Put_Line (File  => FrameFile,
                             E_Str => TempExaminerLongString);

         SPARK_IO.Put_Line (FrameFile, "  </frameset>", 13);
         SPARK_IO.Put_Line (FrameFile, "</html>", 7);
      end WriteFrameFile;

      -- This procedure does the following:
      --
      --    - creates a file called "spark.htm" in the HTML subdirectory
      --      of the current directory (the HTML directory is assumed to
      --      already exist),
      --    - derives the report file name from that held in
      --      CommandLineData.Content for inclusion in the frame file,
      --    - creates a blank file to fill the bottom frame,
      --    - calls WriteFrameFile to write the content of the base frame
      --      file,
      --    - closes the frame file and blank file,
      --    - returns a status value of True if no errors occurred,
      --    - returns a status value of False and displays a message to
      --      the user if an error occurred.
      --
      -- Error Trapping:
      --
      -- Where calls to procedures in SPARK_IO return status parameters the
      -- parameter is checked.  If the status is not OK then a message is
      -- echoed to the screen to inform the user.  The message does not
      -- display the type of error (but could in future) but explains what
      -- was happening when the error occurred.
      --
      -- This procedure returns a status value itself.  If the status value
      -- is False then it is suggested that no further HTML generation should
      -- take place.  The caller should test the returned status and set the
      -- GenerateHTML flag appropriately.

      procedure CreateFrameFile (Success : out Boolean)
         --# global in     CommandLineData.Content;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS,
         --#         Success           from CommandLineData.Content,
         --#                                SPARK_IO.FILE_SYS;
      is
         FrameFile          : SPARK_IO.File_Type;
         BlankFile          : SPARK_IO.File_Type;
         FrameFileCreatedOK : SPARK_IO.File_Status;
         FrameFileClosedOK  : SPARK_IO.File_Status;
         BlankFileCreatedOK : SPARK_IO.File_Status;
         BlankFileClosedOK  : SPARK_IO.File_Status;

         HTMLReportFileName : ELStrings.T;

         FrameFileName      : ELStrings.T;
         BlankFileName      : ELStrings.T;
         FullFrameFileName  : ELStrings.T;
         FullBlankFileName  : ELStrings.T;

      begin
         -- Initialise variables.
         Success      := True;  -- Set to false when fatal error occurs.
         FrameFile    := SPARK_IO.Null_File;
         BlankFile    := SPARK_IO.Null_File;

         FrameFileName := ELStrings.Copy_String (Str => "spark.htm");
         BlankFileName := ELStrings.Copy_String (Str => "blank.htm");

         -- These files both reside in the HTML directory so their
         -- names need to have the HTML directory name prepended.
         FullFrameFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (FrameFileName));
         FullBlankFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (BlankFileName));

         -- Get the name of the HTML report file for inclusion in the HTML.
         HTMLReportFileName := ELStrings.ToExaminerLongString (
                                  FileSystem.JustFile (CommandLineData.Content.ReportFileName, True));
         HTMLReportFileName := Translate (HTMLReportFileName, '.', '_');
         ELStrings.Append_String (E_Str => HTMLReportFileName,
                                  Str   => ".htm");

         -- Create the frame file.
         ELStrings.Create (File          => FrameFile,
                           Name_Of_File  => FullFrameFileName,
                           Form_Of_File  => "",
                           Status        => FrameFileCreatedOK);

         if FrameFileCreatedOK /= SPARK_IO.Ok then

            ScreenEcho.Put_Line ("An error occurred while attemping to create the HTML frame file.");
            Success := False;

         else

            ELStrings.Create (File          => BlankFile,
                              Name_Of_File  => FullBlankFileName,
                              Form_Of_File  => "",
                              Status        => BlankFileCreatedOK);

            if BlankFileCreatedOK /= SPARK_IO.Ok then

               ScreenEcho.Put_Line ("An error occurred while attempting to create the blank HTML file.");
               -- There is not an else here as not being able to write the blank file is
               -- not fatal.  Success remains true and we continue as we still have to
               -- close the FrameFile.

            end if;

            WriteFrameFile (FrameFile, HTMLReportFileName, BlankFileName);

            --# accept Flow, 10, FrameFile, "Expected ineffective assignment to FrameFile";
            SPARK_IO.Close (File   => FrameFile,
                            Status => FrameFileClosedOK);
            --# end accept;

            if FrameFileClosedOK /= SPARK_IO.Ok then
               ScreenEcho.Put_Line ("An error occurred while attempting to close the HTML frame file.");
               Success := False;
            end if;
            -- There is not an else here as we need to try and close the other file.

            --# accept Flow, 10, BlankFile, "Expected ineffective assignment to BlankFile";
            SPARK_IO.Close (File   => BlankFile,
                            Status => BlankFileClosedOK);
            --# end accept;

            if BlankFileClosedOK /= SPARK_IO.Ok then
               ScreenEcho.Put_Line ("An error occurred while attempting to close the blank HTML file.");
               -- Not closing the blank file is non-fatal, Success remains true.
            end if;

         end if;

      end CreateFrameFile;

      -- Returns the name of the HTML subdirectory required for passing
      -- to FileSystem.IdempotentCreateSubdir.

      function HTMLSubDirName return EStrings.T
      --# global in CommandLineData.Content;
      is

         DirName : EStrings.T;

      begin

         DirName := FileSystem.StartOfDirectory;

         EStrings.Append_Examiner_String (E_Str1 => DirName,
                                          E_Str2 => CommandLineData.Content.HTMLDirectory);

         -- IdempotentCreateSubdir expects:
         --   on Unix/NT:  just the name of the directory

         return DirName;

      end HTMLSubDirName;

      procedure CopyErrorsFile
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content;
      is

         SourceFileName        : EStrings.T;
         SourceFullFileName    : EStrings.T;
         SourceFileStatus      : FileSystem.TypFileSpecStatus;
         SourceFile            : SPARK_IO.File_Type;
         SourceFileOpenOK      : SPARK_IO.File_Status;
         SourceFileClosedOK    : SPARK_IO.File_Status;

         DestFileName          : ELStrings.T;
         DestFile              : SPARK_IO.File_Type;
         DestFileCreatedOK     : SPARK_IO.File_Status;
         DestFileClosedOK      : SPARK_IO.File_Status;

         CopyBuffer            : ELStrings.T;

      begin

         SourceFile := SPARK_IO.Null_File;
         DestFile   := SPARK_IO.Null_File;

         -- Start by trying to open output file.
         DestFileName := FileSystem.CaseOfFilesForCreateL
           (HTMLFileName (ELStrings.Copy_String (Str => "errors.htm")));

         ELStrings.Create (File          => DestFile,
                           Name_Of_File  => DestFileName,
                           Form_Of_File  => "",
                           Status        => DestFileCreatedOK);

         if DestFileCreatedOK /= SPARK_IO.Ok then
            ScreenEcho.Put_Line ("An error occurred while trying to create the HTML errors file.");
         else
            -- Now locate the input file.

            SourceFileName := FileSystem.ExaminerLibDirectory;

            EStrings.Append_Examiner_String (E_Str1 => SourceFileName,
                                             E_Str2 => FileSystem.EndOfPath);

            EStrings.Append_String (E_Str => SourceFileName,
                                    Str   => "errors.htm");
            FileSystem.FindFullFileName (SourceFileName, SourceFileStatus, SourceFullFileName);
            if SourceFileStatus /= FileSystem.FileFound then
               -- Output simple message to destination file.
               SPARK_IO.Put_Line (DestFile, "Sorry, could not locate errors.htm.", 35);
               --# accept Flow, 10, DestFileClosedOK, "Expected ineffective assignment to DestFileClosedOK";
               SPARK_IO.Close (DestFile, DestFileClosedOK);    -- Unused variable DestFileClosedOK
                                                               --Ignore error on close.
               --# end accept;
            else

               -- Open the file and copy it.
               EStrings.Open (File         => SourceFile,
                              Mode_Of_File => SPARK_IO.In_File,
                              Name_Of_File => SourceFullFileName,
                              Form_Of_File => "",
                              Status       => SourceFileOpenOK);

               if SourceFileOpenOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_Line ("An error occurred while opening the HTML errors file");
               else
                  while not SPARK_IO.End_Of_File (SourceFile) loop
                     ELStrings.Get_Line (File  => SourceFile,
                                         E_Str => CopyBuffer);
                     ELStrings.Put_Line (File  => DestFile,
                                         E_Str => CopyBuffer);
                  end loop;
               end if;

               --# accept Flow, 10, SourceFile, "Expected ineffective assignment to SourceFile" &
               --#        Flow, 10, SourceFileClosedOK, "Expected ineffective assignment to SourceFileClosedOK";
               SPARK_IO.Close (SourceFile, SourceFileClosedOK); -- 2 ineffective assignments OK
               --# end accept;

            end if;
         end if;
         --# accept Flow, 10, DestFile, "Expected ineffective assignment to DestFile" &
         --#        Flow, 10, DestFileClosedOK, "Expected ineffective assignment to DestFileClosedOK";
         SPARK_IO.Close (DestFile, DestFileClosedOK);  -- 2 ineffective assignments OK
         --# end accept;

         --# accept Flow, 33, SourceFileClosedOK, "Expected SourceFileClosedOK to be neither referenced nor exported" &
         --#        Flow, 33, DestFileClosedOK, "Expected DestFileClosedOK to be neither referenced nor exported";
      end CopyErrorsFile;  -- 2 Flow errors OK


   begin -- InitSparkHTML

      -- Initialise working directories - ensure trailing slashes for use in
      -- relative filename procedures.
      SparkWorkDir := ELStrings.ToExaminerLongString (FileSystem.WorkingDirectory);

      -- Ensure trailing separator.
      if ELStrings.Get_Element (E_Str => SparkWorkDir,
                                Pos   => ELStrings.Get_Length (E_Str => SparkWorkDir)) /=
        EStrings.Get_Element (E_Str => FileSystem.EndOfPath,
                              Pos   => EStrings.Get_Length (E_Str => FileSystem.EndOfPath)) then
         ELStrings.Append_Examiner_String (E_Str1 => SparkWorkDir,
                                           E_Str2 => FileSystem.EndOfPath);
      end if;

      HTMLWorkDir := ELStrings.ToExaminerLongString (FileSystem.WorkingDirectory);
      HTMLWorkDir := AppendDirectoryString (HTMLWorkDir, CommandLineData.Content.HTMLDirectory);

      -- Get name for creating subdirectory.
      DirName  := HTMLSubDirName;

      -- Create subdirectory.
      FileSystem.IdempotentCreateSubdirectory (Path => DirName,
                                               Ok   => NewDirOK);
      CopyErrorsFile;
      CreateFrameFile (FrameFileOK);
      if not (NewDirOK and FrameFileOK) then
         ScreenEcho.Put_Line ("Error ocurred while initialising HTML generation.");
         ScreenEcho.Put_Line ("No further HTML generation will occur.");
         GenerateHTML := False;
         -- Suppress further HTML generation.
      end if;
   end InitSparkHTML;  -- flow errors expected due to false coupling through SPARK_IO and
   -- FileStatus not being used.

   ---------------------------------------------------------------------------

   --
   --  -------------
   --  GenReportHTML
   --  -------------
   --
   --  This procedure generates the HTMLified report file from the plain text
   --  report file.
   --
   --  It is assumed that the HTML output flag has already been tested.  The call to
   --  GenReportHTML should look something like:
   --
   --    if CommandLineData.Content.HTML then
   --       SparkHTML.GenReportHTML;
   --    end if;
   --
   --  The checking of the GenerateHTML flag is done internally.
   --
   --  If the GenerateHTML flag is false this procedure does nothing, otherwise,
   --  it does exactly the following:
   --
   --     - creates an HTML file (named <rep_file>.htm where <rep_file> is the name
   --       of the report file with all '.' characters changed to '_' characters;
   --     - processes each line of the plain text report file using the
   --       ProcessReportLine procedure (the functionality of this is described
   --       at the declaration of ProcessReportLine);
   --     - writes the processed lines to the HTML report file;
   --     - sets GenerateHTML to false if anything goes wrong, to prevent further
   --       HTML generation.
   --
   --  Error trapping:
   --
   --  The HTML report file is essentially a guide to the HTMLified listings and
   --  so if something causes the HTML report file generation to fail we should
   --  suppress all following HTML generation by settign GenerateHTML to "false".
   --
   --  If anything unusual happens then the procedure will echo a message to the
   --  screen saying what went wrong and, in the case of errors which should stop
   --  HTML generation, GenerateHTML will be set to "false".
   --
   procedure GenReportHTML is

      HTMLReportFile          : SPARK_IO.File_Type;
      HTMLReportFileName      : ELStrings.T;
      HTMLReportFileCreatedOK : SPARK_IO.File_Status;
      HTMLReportFileClosedOK  : SPARK_IO.File_Status;

      PlainReportFile         : SPARK_IO.File_Type;
      PlainReportFileName     : EStrings.T;
      PlainReportFileOpenOK   : SPARK_IO.File_Status;
      PlainReportFileClosedOK : SPARK_IO.File_Status;

      LineBuffer              : ELStrings.T;
      LookaheadBuffer         : ELStrings.T;
      SavedErrorLink          : ELStrings.T;
      SavedListingFile        : ELStrings.T;

      ReportFileState         : ReportFileStates := ReportJustStarted;

      -- Sub Programs
      -- ------------

      -- This sub-procedure writes HTML content to the start of the report file.
      -- It assumes that the file handle HTMLReportFile is the open HTML
      -- report file.  This procedure will not open or close the file.
      --
      -- The HTML written specifies the title of the page and some formatting
      -- tags.  The formatting is <PRE> (pre-processed text) which displays
      -- text exactly as given, and <TT> which sets the typewriter-text font;
      -- I use this because it usually results in a fixed-width font being
      -- used.
      --
      -- When writing the end of the report file these tags need to be closed,
      -- as do the <BODY> and <HTML> tags.  This should be done by calling
      -- the WriteHTMLReportFooter procedure.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLReportHeader
         --# global in     HTMLReportFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                HTMLReportFile;

      is

      begin

         SPARK_IO.Put_Line (HTMLReportFile,
                            "<!DOCTYPE HTML PUBLIC ""-//W3C//DTD HTML 4.0 Transitional//EN""", 61);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "            ""http://www.w3.org/TR/REC-html40/loose.dtd"">", 56);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "<html>", 6);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "  <head>", 8);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "    <title>SPARK Examiner HTML Report File</title>", 50);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "  </head>", 9);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "  <body>", 8);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "    <pre>", 9);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "      <tt>", 10);

      end WriteHTMLReportHeader;


      -- This subprocedure writes HTML content to the end of the report file.
      -- It assumes that the file handle HTMLReportFile is the open HTML
      -- report file.  This procedure will not open or close the file.
      --
      -- The HTML written closes all the formatting tags that were opened
      -- by the call to WriteHTMLReportHeader.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLReportFooter
         --# global in     HTMLReportFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                HTMLReportFile;

      is

      begin

         SPARK_IO.Put_Line (HTMLReportFile, "      </tt>", 11);
         SPARK_IO.Put_Line (HTMLReportFile, "    </pre>", 10);
         SPARK_IO.Put_Line (HTMLReportFile, "  </body>", 9);
         SPARK_IO.Put_Line (HTMLReportFile, "</html>", 7);

      end WriteHTMLReportFooter;

      -- This procedure is used to convert a line of the report file into HTML.
      -- It is effectively a parser for the report file.
      --
      -- The procedure design is based on a state machine.  The global variable
      -- ReportFileState records our current location within the report file (in
      -- terms of what information has already been read).
      --
      -- Given a line, we can determine the line's meaning from our current state
      -- and the contents of the line (all blank lines are ignored).  For example,
      -- if we are reading the warning list and we discover a line starting with
      -- the string "Source Filename(s) used were:" then we know that we are now
      -- processing the source file list.
      --
      -- We can use this method to parse almost all the information in the source
      -- file.  Those bits that we don't parse are usually optional (such as the
      -- flag "rtc" in the options list) and require no translation to HTML anyway.
      --
      -- Once the procedure understands what a line represents it updates the
      -- ReportFileState and processes the line by calling an appropriate
      -- subroutine.
      --
      -- The procedure contains a debugging feature which reports each line that it
      -- finds to the screen along with a message for each line that it recognises.
      -- This only happens if the -debug switch is given on the commandline.

      procedure ProcessReportLine (Line      : in out ELStrings.T;
                                   Lookahead : in     ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     HTMLWorkDir;
         --#        in     SparkWorkDir;
         --#        in out ReportFileState;
         --#        in out SavedErrorLink;
         --#        in out SavedListingFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Line              from *,
         --#                                CommandLineData.Content,
         --#                                HTMLWorkDir,
         --#                                Lookahead,
         --#                                ReportFileState,
         --#                                SavedErrorLink,
         --#                                SavedListingFile,
         --#                                SparkWorkDir &
         --#         ReportFileState   from *,
         --#                                Line &
         --#         SavedErrorLink    from *,
         --#                                Line,
         --#                                Lookahead,
         --#                                ReportFileState &
         --#         SavedListingFile  from *,
         --#                                CommandLineData.Content,
         --#                                Line,
         --#                                ReportFileState &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Line,
         --#                                Lookahead,
         --#                                ReportFileState;

      is

         StartPos      : Integer;
         CompareString : ELStrings.T;
         DebugMessage  : ELStrings.T;

         -- This function takes a filename and some text as input and returns the text
         -- enclosed in HTML tags that form a link to the file relative to the
         -- specified "RelativeTo" directory.
         --
         -- A filename beginning with an @ symbol is interpreted as a metafile and
         -- the @ is removed from the location placed in the tag.
         --
         -- Error checking is assumed to occur within the string handling routines.
         -- If the HTMLRelativeFile procedure can not generate the URI then
         -- the link is not added and the file name returned, followed by a
         -- message saying "[Link unavailable]".

         function CreateFileLinkTagWithText (FileNameIn          : ELStrings.T;
                                             RelativeTo          : ELStrings.T;
                                             Text                : ELStrings.T)
                                            return ELStrings.T
            --# global in SparkWorkDir;
         is

            OutString            : ELStrings.T;
            FileName             : ELStrings.T;
            Location             : ELStrings.T;
            Success              : Boolean;

         begin

            FileName := FileNameIn;

            -- The OutString is the complete string that will be output including
            -- the tags and the original filename.
            OutString := ELStrings.Empty_String;

            -- Chop off the leading @ (if there is one).
            if ELStrings.Get_Element (E_Str => FileName,
                                      Pos   => ELStrings.Positions'First) = '@' then
               FileName := ELStrings.Section
                 (FileName,
                  ELStrings.Positions'First + 1,
                  ELStrings.Get_Length (E_Str => FileName) - 1);
            end if;

            -- Find the relative URI.
            HTMLRelativeFile (FileName, RelativeTo, Location, Success);

            if not Success then
               OutString := FileNameIn;
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => " [Link unavailable] ");
            else
               -- Create the tag.
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => "<A HREF=""");

               ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                      E_Str2 => Location);
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => """  TYPE=""x-text/spark"" TARGET=""rbottom"" >");
               ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                      E_Str2 => Text);
               ELStrings.Append_String (E_Str => OutString,
                                        Str   => "</A>");
            end if;

            return OutString;

         end CreateFileLinkTagWithText;

         -- This function does the same as CreateFileLinkTagWithText but the text
         -- used is the filename given.

         function CreateFileLinkTag (FileNameIn          : ELStrings.T;
                                     RelativeTo          : ELStrings.T)
                                    return ELStrings.T
            --# global in SparkWorkDir;
         is

         begin

            return CreateFileLinkTagWithText (FileNameIn => FileNameIn,
                                              RelativeTo => RelativeTo,
                                              Text       => FileNameIn);
         end CreateFileLinkTag;


         -- This file turns a line from the options section of the report file
         -- that specifies a file location and adds HTML tags to create a link
         -- to the file.
         --
         -- An "Option Line" is defined to be a line from the options section
         -- of the report file.  This line is of the form:
         --
         --    <option_name> = <file_location>
         --
         -- The procedure simply copies the line up to the '=' character
         -- and assumes that whatever follows the '=' is the filename.

         function CreateOptionFileLink (Line : ELStrings.T)
                                       return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            TempLine      : ELStrings.T;
            I             : ELStrings.Positions;
            FileName      : ELStrings.T;
            FileLink      : ELStrings.T;

         begin

            TempLine      := ELStrings.Empty_String;
            FileName      := ELStrings.Empty_String;
            I             := ELStrings.Positions'First;

            -- Copy up to the '='
            loop
               AppendCharWithoutCheck
                 (TempLine,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) = '=';
               I := I + 1;
            end loop;

            -- Point to the first character of the filename.
            I := I + 1;

            -- Get the name of the index file by copying up to the end of line
            while I <= ELStrings.Get_Length (E_Str => Line) loop
               AppendCharWithoutCheck
                 (FileName,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            -- Add HTML tag
            FileLink := CreateFileLinkTag (FileName, HTMLWorkDir);

            ELStrings.Append_Examiner_Long_String (E_Str1 => TempLine,
                                                   E_Str2 => FileLink);

            return TempLine;

         end CreateOptionFileLink;

         -- This function takes a line containing a number of spaces followed by
         -- a file name and puts HTML link tags in to form a link to the file.
         -- If the file starts with an @ symbol it is interpreted as a metafile
         -- and the @ is removed.
         --
         -- This function is used to generate HTML for lines which just contain
         -- whitespace and filenames, such as the selected files list, index files
         -- list, and metafiles list.

         function CreateFileLink (Line : ELStrings.T)
                                 return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            TempLine      : ELStrings.T;
            I             : ELStrings.Positions;
            FileName      : ELStrings.T;
            FileLink      : ELStrings.T;

         begin

            TempLine      := ELStrings.Empty_String;
            FileName      := ELStrings.Empty_String;
            I             := ELStrings.Positions'First;

            -- Copy whitespace up to the first character.
            loop
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) /= ' ';
               AppendCharWithoutCheck
                 (TempLine,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            -- Get the name of the file by copying up to the end of line
            while I <= ELStrings.Get_Length (E_Str => Line) loop
               AppendCharWithoutCheck
                 (FileName,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            -- Add HTML tag
            FileLink := CreateFileLinkTag (FileName, HTMLWorkDir);

            ELStrings.Append_Examiner_Long_String (E_Str1 => TempLine,
                                                   E_Str2 => FileLink);

            return TempLine;

         end CreateFileLink;

         -- This function is used to process a line from the source file list.
         -- The line is assumed to be some space characters followed by a
         -- file name.  The function creates the file link as described in the
         -- commentary for CreateFileLink, and appends to it a link to the
         -- analysis section for that file.
         --
         -- The link created is a link of the form:
         --
         --    <A HREF="#html_name"> ...  </A>
         --
         -- where html_name is the result of applying the HTMLName function to
         -- the filename.
         --
         -- The HTMLName function must also be used to create the anchor that this
         -- link refers to.

         function ProcessSourceListLine (Line : ELStrings.T)
                                        return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            OutString    : ELStrings.T;
            FileName     : ELStrings.T;
            I            : ELStrings.Positions;

         begin

            I := FirstChar (Line);
            FileName := ELStrings.Section
              (Line,
               I,
               (ELStrings.Get_Length (E_Str => Line) - I) + 1);

            OutString := CreateFileLink (Line);

            ELStrings.Append_String (E_Str => OutString,
                                     Str   => " [<A HREF=""#");
            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => HTMLName (FileName));
            ELStrings.Append_String (E_Str => OutString,
                                     Str   => """>View analysis</A>]");

            return OutString;

         end ProcessSourceListLine;

         -- This function processes a line from the main section of the report file
         -- of the form:
         --
         --    some_text: file_name
         --
         --  to produce a line of the form:
         --
         --    some_text: <A NAME="html_name">file_link</A>
         --
         --  where:
         --
         --    html_name is the result of applying the function HTMLName to file_name
         --              to generate the anchor tag referenced from the source file
         --              list (see commentary for ProcessSourceListLine)
         --    file_link is the result of applying the function CreateFileLinkTag to
         --              file_name to produce a link to the file
         --
         --  This function should be used to produce HTML for the lines in the body
         --  of the report file that beign "Source Filename: "

         function ProcessSourceFileNameLine (Line : ELStrings.T)
                                            return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            OutString    : ELStrings.T;
            I            : ELStrings.Positions;
            FileName     : ELStrings.T;
            Anchor       : ELStrings.T;

         begin

            OutString := ELStrings.Empty_String;
            I         := ELStrings.Positions'First;

            -- Copy up to and including ':'
            loop
               AppendCharWithoutCheck
                 (OutString,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) = ':';
               I := I + 1;
            end loop;


            -- Point to next character
            I := I + 1;

            -- Copy spaces up to first char of file name
            loop
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) /= ' ';
               AppendCharWithoutCheck
                 (OutString,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            -- Extract the filename
            FileName := ELStrings.Section
              (Line,
               I,
               (ELStrings.Get_Length (E_Str => Line) - I) + 1);

            -- Create the anchor tag
            Anchor := ELStrings.Copy_String (Str => "<A NAME=""");
            ELStrings.Append_Examiner_Long_String (E_Str1 => Anchor,
                                                   E_Str2 => HTMLName (FileName));
            ELStrings.Append_String (E_Str => Anchor,
                                     Str   => """></A>");

            -- Append the anchor tag to the output line
            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => Anchor);

            -- Create the link to the source file and append that to the output line
            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => CreateFileLinkTag (FileName,
                                                                                HTMLWorkDir));


            return OutString;

         end ProcessSourceFileNameLine;

         -- This procedure processes a line from the main section of the report file
         -- of the form:
         --
         --    some_text: file_name
         --
         --  to produce a line of the form:
         --
         --    some_text: file_link html_link
         --
         --  where:
         --
         --    file_link is the result of applying the function CreateFileLinkTag to
         --              file_name to produce a link to the file
         --
         --    html_link is the text "[View HTML]" enclosed in tags that make it a
         --              link to the HTML version of the above file.  The HTML
         --              version is assumed to be the name (without directory)
         --              of the specified listing file, with all '.'s changed
         --              to '_'s, with a ".htm" extension and residing in the HTML
         --              subdirectory of the current directory; e.g. H:\foo\bar.lst
         --              becomes HTML/bar_lst.htm
         --
         --  It also updates the value of SavedListingFile to be a reference to the
         --  HTML version of the listing for use in processing the source error lines.
         --
         --  This function should be used to produce HTML for the lines in the body
         --  of the report file that beign "Listing Filename: "

         procedure ProcessListingFileNameLine (Line : in out ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     HTMLWorkDir;
         --#        in     SparkWorkDir;
         --#           out SavedListingFile;
         --# derives Line             from *,
         --#                               CommandLineData.Content,
         --#                               HTMLWorkDir,
         --#                               SparkWorkDir &
         --#         SavedListingFile from CommandLineData.Content,
         --#                               Line;
         is

            OutString : ELStrings.T;
            I         : ELStrings.Positions;

            FileName  : ELStrings.T;
            HTMLFileLink : ELStrings.T;

         begin

            OutString := ELStrings.Empty_String;
            I := ELStrings.Positions'First;

            -- Copy up to and including ':'
            loop
               AppendCharWithoutCheck
                 (OutString,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) = ':';
               I := I + 1;
            end loop;

            -- Point to next character
            I := I + 1;

            -- Copy spaces up to first char of file name
            loop
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) /= ' ';
               AppendCharWithoutCheck
                 (OutString,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            -- Extract the filename
            FileName := ELStrings.Section
              (Line,
               I,
               (ELStrings.Get_Length (E_Str => Line) - I) + 1);

            -- Append link to plain text listing.
            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => CreateFileLinkTag (FileNameIn => FileName,
                                                                                RelativeTo => HTMLWorkDir));

            -- And add a space
            AppendCharWithoutCheck (OutString, ' ');

            -- Create URL for HTML listing file in SavedListingFile
            SavedListingFile := ELStrings.ToExaminerLongString (CommandLineData.Content.HTMLDirectory);
            ELStrings.Append_String (E_Str => SavedListingFile,
                                     Str   => "/");
            ELStrings.Append_Examiner_Long_String
              (E_Str1 => SavedListingFile,
               E_Str2 => Translate (ELStrings.ToExaminerLongString
                                      (FileSystem.JustFile (Fn  => ELStrings.ToExaminerString (FileName),
                                                            Ext => True)),
                                    '.',
                                    '_'));

            ELStrings.Append_String (E_Str => SavedListingFile,
                                     Str   => ".htm");

            -- Create HTML tags.
            HTMLFileLink := CreateFileLinkTagWithText
              (FileNameIn => SavedListingFile,
               RelativeTo => HTMLWorkDir,
               Text       => ELStrings.Copy_String (Str => "[View HTML]"));

            ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                   E_Str2 => HTMLFileLink);

            Line := OutString;

         end ProcessListingFileNameLine;

         -- *** Comment

         procedure ProcessErrorSourceLine (Line : in out ELStrings.T)
            --# global in HTMLWorkDir;
            --#        in SavedListingFile;
            --#        in SparkWorkDir;
            --# derives Line from *,
            --#                   HTMLWorkDir,
            --#                   SavedListingFile,
            --#                   SparkWorkDir;
         is

            OutString       : ELStrings.T;
            I               : ELStrings.Positions;
            LineNo          : ELStrings.T;

            RelativeListing : ELStrings.T;
            LinkSuccess     : Boolean;
            Link            : ELStrings.T;

         begin

            if ELStrings.Get_Length (E_Str => SavedListingFile) /= 0 then  -- there is a listing file

               OutString := ELStrings.Empty_String;
               I         := ELStrings.Positions'First;
               LineNo    := ELStrings.Empty_String;

               -- Copy up to the first non-space.
               loop
                  exit when ELStrings.Get_Element (E_Str => Line,
                                                   Pos   => I) /= ' ';
                  AppendCharWithoutCheck
                    (OutString,
                     ELStrings.Get_Element (E_Str => Line,
                                            Pos   => I));
                  I := I + 1;
               end loop;

               -- Copy digits to LineNo
               while Digit (ELStrings.Get_Element (E_Str => Line,
                                                   Pos   => I)) loop
                  AppendCharWithoutCheck
                    (LineNo,
                     ELStrings.Get_Element (E_Str => Line,
                                            Pos   => I));
                  I := I + 1;
               end loop;

               -- Create relative location of SavedListingFile
               HTMLRelativeFile (SavedListingFile,
                                 HTMLWorkDir,
                                 RelativeListing,
                                 LinkSuccess);

               if LinkSuccess then
                  -- Create link to listing based on saved listing file.
                  Link := ELStrings.Copy_String (Str => "<A TARGET=""rbottom"" HREF=""");
                  ELStrings.Append_Examiner_Long_String (E_Str1 => Link,
                                                         E_Str2 => RelativeListing);
                  ELStrings.Append_String (E_Str => Link,
                                           Str   => "#line");
                  ELStrings.Append_Examiner_Long_String (E_Str1 => Link,
                                                         E_Str2 => LineNo);
                  ELStrings.Append_String (E_Str => Link,
                                           Str   => """>");
                  ELStrings.Append_Examiner_Long_String (E_Str1 => Link,
                                                         E_Str2 => LineNo);
                  ELStrings.Append_String (E_Str => Link,
                                           Str   => "</A>");

               else
                  -- The link is just the number.
                  Link := LineNo;
               end if;

               -- Append link to OutString
               ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                      E_Str2 => Link);

               -- Append rest of line
               ELStrings.Append_Examiner_Long_String
                 (E_Str1 => OutString,
                  E_Str2 => ELStrings.Section
                    (Line,
                     I,
                     (ELStrings.Get_Length (E_Str => Line) - I) + 1));

               Line := OutString;

            end if;

         end ProcessErrorSourceLine;

      begin  -- ProcessReportLine
         if ELStrings.Get_Length (E_Str => Line) > 0 then

            StartPos := FirstChar (Line);
            DebugMessage := ELStrings.Empty_String;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                   E_Str => Line);  -- Line used for debugging.
            end if;

            case ReportFileState is
               when ReportJustStarted =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 5),
                     Str   => "*****") then
                     ReportFileState := ReportBannerStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportBannerStarted");
                  end if;
               when ReportBannerStarted =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 5),
                     Str   => "*****") then
                     ReportFileState := ReportBannerFinished;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportBannerFinished");
                  end if;
               when ReportBannerFinished =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 7),
                     Str   => "DATE : ") then
                     ReportFileState := ReportDateFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportDateFound");
                  end if;
               when ReportDateFound =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 8),
                     Str   => "Options:") then
                     ReportFileState := ReportOptionsFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportOptionsFound");
                  end if;
               when ReportOptionsFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 11);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Index_File & "=") then
                     ReportFileState := ReportIndexFileFound;
                     Line := CreateOptionFileLink (Line);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportIndexFileFound");
                  elsif ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => "no" & CommandLineData.Option_Index_File),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 11)) then
                     ReportFileState := ReportIndexFileFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportIndexFileFound");
                  end if;
               when ReportIndexFileFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 13);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Warning_File & "=") then
                     ReportFileState := ReportWarningFileFound;
                     Line := CreateOptionFileLink (Line);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportWarningFileFound");
                  elsif ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => "no" & CommandLineData.Option_Warning_File),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 13)) then
                     ReportFileState := ReportWarningFileFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportWarningFileFound");
                  end if;
               when ReportWarningFileFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 21);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Target_Compiler_Data & "=") then
                     ReportFileState := ReportTargetCompilerDataFound;
                     Line := CreateOptionFileLink (Line);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetCompilerDataFound");
                  elsif ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => "no" & CommandLineData.Option_Target_Compiler_Data),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 21)) then
                     ReportFileState := ReportTargetCompilerDataFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetCompilerDataFound");
                  end if;
               when ReportTargetCompilerDataFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 12);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Config_File & "=") then
                     ReportFileState := ReportTargetConfigFileFound;
                     Line := CreateOptionFileLink (Line);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetConfigFileFound");
                  elsif ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => "no" & CommandLineData.Option_Config_File),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 12)) then
                     ReportFileState := ReportTargetConfigFileFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetConfigFileFound");
                  end if;
               when ReportTargetConfigFileFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 17);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Source_Extension & "=") then
                     ReportFileState := ReportSourceExtensionFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceExtensionFound");
                  end if;
               when ReportSourceExtensionFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 18);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Listing_Extension & "=") then
                     ReportFileState := ReportListingExtensionFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportListingExtensionFound");
                  end if;
               when ReportListingExtensionFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 12);
                  if ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => CommandLineData.Option_Dictionary_File),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 12)) then
                     ReportFileState := ReportDictionaryFound;
                     Line := CreateOptionFileLink (Line);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportDictionaryFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => CommandLineData.Option_No_Dictionary) then
                     ReportFileState := ReportDictionaryFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportDictionaryFound");
                  end if;
               when ReportDictionaryFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 12);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Report_File & "=") then
                     ReportFileState := ReportReportFileFound;
                     Line := CreateOptionFileLink (Line);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportReportFileFound");
                  end if;
                  -- NOTE: checking for "noreport_file" is a stupid thing to do!
               when ReportReportFileFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 4);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Html) then
                     -- NOTE: checking for nohtml is an equally stupid thing to do!
                     ReportFileState := ReportHTMLFlagFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportHTMLFlagFound");
                  end if;
               when ReportHTMLFlagFound =>
                  -- The RTC and VCG options appear here but as they are optional
                  -- we shall ignore them.
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 10);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Statistics) or else
                    ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => "no" & CommandLineData.Option_Statistics),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 10)) then
                     ReportFileState := ReportStatisticsFlagFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportStatisticsFlagFound");
                  end if;
               when ReportStatisticsFlagFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 15);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Fdl_Identifiers) or else
                    ELStrings.Eq_String
                    (E_Str1 => CompareString,
                     E_Str2 => ELStrings.Section
                       (E_Str     => ELStrings.Copy_String (Str => "no" & CommandLineData.Option_Fdl_Identifiers),
                        Start_Pos => ELStrings.Positions'First,
                        Length    => 15)) then
                     ReportFileState := ReportFDLIdentifiersFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportFDLIdentifiersFound");
                  end if;
               when ReportFDLIdentifiersFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 14);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Flow_Analysis & "=") then
                     ReportFileState := ReportFlowAnalysisFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportFlowAnalysisFound");
                  end if;
               when ReportFlowAnalysisFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 5);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "ada83") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "ada95") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "ada20") then
                     ReportFileState := ReportLanguageOptionFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportLanguageOptionFound");
                  end if;
               when ReportLanguageOptionFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 21);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => CommandLineData.Option_Annotation_Character & "=") then
                     ReportFileState := ReportAnnotationCharacterFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportAnnotationCharacterFound");
                  end if;
               when ReportAnnotationCharacterFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 15);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Selected files:") then
                     ReportFileState := ReportSelectedFilesStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSelectedFilesStarted");
                  end if;
               when ReportSelectedFilesStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 24);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Index Filename(s) used w") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "No Index files were used") then
                     ReportFileState := ReportIndexFilesStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportIndexFilesStarted");
                  else
                     Line := CreateFileLink (Line);
                  end if;
               when ReportIndexFilesStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 18);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Meta File(s) used ") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "No Meta Files used") then
                     ReportFileState := ReportMetaFilesStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportMetaFilesStarted");
                  else
                     Line := CreateFileLink (Line);
                  end if;
               when ReportMetaFilesStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 22);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Full warning reporting") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "Summary warning report") then
                     ReportFileState := ReportWarningListStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportWarningListStarted");
                  else
                     Line := CreateFileLink (Line);
                  end if;
               when ReportWarningListStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 26);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Target configuration file:") then
                     ReportFileState := ReportTargetConfigListStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetConfigListStarted");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "Source Filename(s) used we") then
                     ReportFileState := ReportSourceListStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceListStarted");
                  end if;
               when ReportTargetConfigListStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 4);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Line") then
                     ReportFileState := ReportTargetErrorLine;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetErrorLine");
                  end if;
               when ReportTargetErrorLine =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 3);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "!!!") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "***") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "---") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "???") then
                     ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                     ReportFileState := ReportTargetErrorNextLine;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportTargetErrorNextLine");
                  else
                     CompareString := ELStrings.Section (E_Str     => Line,
                                                         Start_Pos => StartPos,
                                                         Length    => 29);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "Source Filename(s) used were:") then
                        ReportFileState := ReportSourceListStarted;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportSourceListStarted");
                     end if;
                  end if;
               when ReportTargetErrorNextLine =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 29);
                  if StartPos > 10 then -- Extra lines of the error message will be indented by 11 characters
                     ProcessNextErrorMessageLine (Line, SavedErrorLink);
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "Source Filename(s) used were:") then
                     ReportFileState := ReportSourceListStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceListStarted");
                  else
                     CompareString := ELStrings.Section (E_Str     => Line,
                                                         Start_Pos => StartPos,
                                                         Length    => 3);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "!!!") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "***") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "---") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "???") then
                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                        ReportFileState := ReportTargetErrorNextLine;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportTargetErrorNextLine");
                     else
                        ReportFileState := ReportTargetErrorLine;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportTargetErrorLine");
                     end if;
                  end if;
               when ReportSourceListStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 16);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Source Filename:") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceFileStarted");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "The following we") then
                     ReportFileState := ReportMissingFilesStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportMissingFilesStarted");
                  else
                     Line := ProcessSourceListLine (Line);
                  end if;
               when ReportMissingFilesStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 16);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Source Filename:") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceFileStarted");
                  end if;
               when ReportSourceFileStarted =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 15);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Listing Filenam") then
                     ProcessListingFileNameLine (Line);
                     ReportFileState := ReportListingFileNameFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportListingFileNameFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "No Listing File") then
                     SavedListingFile := ELStrings.Empty_String;
                     ReportFileState := ReportListingFileNameFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportListingFileNameFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "***     no unit") then
                     ReportFileState := ReportNoUnitsFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportNoUnitsFound");
                  end if;
               when ReportListingFileNameFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 10);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Unit name:") then
                     ReportFileState := ReportUnitNameFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportUnitNameFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "***     no") then
                     ReportFileState := ReportNoUnitsFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportNoUnitsFound");
                  end if;
               when ReportUnitNameFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 10);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Unit type:") then
                     ReportFileState := ReportUnitTypeFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportUnitTypeFound");
                  end if;
               when ReportUnitTypeFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 22);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Unit has been analysed") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "Unit has been parsed o") then
                     ReportFileState := ReportAnalysedMessageFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportAnalysedMessageFound");
                  end if;
               when ReportAnalysedMessageFound | ReportNoUnitsFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 10);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "No errors ") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "No summari") then
                     ReportFileState := ReportEndOfErrors;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfErrors");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "Unit name:") then
                     ReportFileState := ReportUnitNameFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportUnitNameFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "***     Un") then
                     ReportFileState := ReportEndOfErrors;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfErrors");
                  else
                     CompareString := ELStrings.Section
                       (Line,
                        ELStrings.Get_Length (E_Str => Line) - 9,
                        10);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "warning(s)") then
                        ReportFileState := ReportStartOfErrors;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportStartOfErrors");
                     end if;
                  end if;
               when ReportStartOfErrors =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 4);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Line") then
                     ReportFileState := ReportLineHeaderFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportLineHeaderFound");
                  end if;
               when ReportLineHeaderFound =>
                  ProcessErrorSourceLine (Line);
                  ReportFileState := ReportErrorSourceLineFound;
                  DebugMessage := ELStrings.Copy_String (Str => "ReportErrorSourceLineFound");
               when ReportErrorSourceLineFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 1);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "^") then
                     -- Process Error Pointer Line (do nothing?)
                     ReportFileState := ReportErrorSourcePointerFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportErrorSourcePointerFound");
                  else  -- Some errors don't have pointers
                     CompareString := ELStrings.Section (E_Str     => Line,
                                                         Start_Pos => StartPos,
                                                         Length    => 3);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "!!!") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "***") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "---") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "???") then

                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);

                        ReportFileState := ReportErrorMessageFound;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportErrorMessageFound");
                     end if;
                  end if;
               when ReportErrorSourcePointerFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 3);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "!!!") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "***") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "---") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "???") then

                     ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                     ReportFileState := ReportErrorMessageFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportErrorMessageFound");
                  end if;
               when ReportErrorMessageFound | ReportEndOfErrors =>
                  if StartPos > 10 then -- Extra lines of the error message will be indented by 11 characters
                     ProcessNextErrorMessageLine (Line, SavedErrorLink);
                     DebugMessage := ELStrings.Copy_String (Str => "ReportNextErrorMessageLineFound");
                  else
                     CompareString := ELStrings.Section (E_Str     => Line,
                                                         Start_Pos => StartPos,
                                                         Length    => 16);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "Source Filename:") then
                        Line := ProcessSourceFileNameLine (Line);
                        ReportFileState := ReportSourceFileStarted;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportSourceFileStarted");
                     elsif ELStrings.Eq1_String (E_Str => CompareString,
                                                 Str   => "No summarized wa") then
                        ReportFileState := ReportEndOfErrors;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfErrorsFound");
                     elsif ELStrings.Eq1_String (E_Str => CompareString,
                                                 Str   => "--End of file---") then
                        ReportFileState := ReportEndOfReportFile;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfReportFile");

                     elsif ELStrings.Eq1_String (E_Str => CompareString,
                                                 Str   => "Expected message") then
                        ReportFileState := ReportJustificationsSummaryFound;
                        DebugMessage := ELStrings.Copy_String (Str => "ReportJustificationsSummaryFound");

                     else -- next error source line found, next error found or
                        -- summarized warnings
                        CompareString := ELStrings.Section (E_Str     => Line,
                                                            Start_Pos => StartPos,
                                                            Length    => 3);
                        if ELStrings.Eq1_String (E_Str => CompareString,
                                                 Str   => "!!!") or else
                          ELStrings.Eq1_String (E_Str => CompareString,
                                                Str   => "***") or else
                          ELStrings.Eq1_String (E_Str => CompareString,
                                                Str   => "---") or else
                          ELStrings.Eq1_String (E_Str => CompareString,
                                                Str   => "???") then

                           ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                           ReportFileState := ReportErrorMessageFound;
                           DebugMessage := ELStrings.Copy_String (Str => "ReportErrorMessageFound");
                        else  -- error source line or summarized warnings
                           if ELStrings.Get_Length (E_Str => Line) > 9 then
                              CompareString := ELStrings.Section
                                (Line,
                                 ELStrings.Get_Length (E_Str => Line) - 9,
                                 10);
                              if ELStrings.Eq1_String (E_Str => CompareString,
                                                       Str   => "omprising:") then
                                 ReportFileState := ReportSummarizedWarningsFound;
                              else
                                 ReportFileState := ReportErrorSourceLineFound;
                              end if;
                           else
                              ReportFileState := ReportErrorSourceLineFound;
                           end if;
                           if ReportFileState = ReportSummarizedWarningsFound then
                              DebugMessage := ELStrings.Copy_String (Str => "ReportSummarizedWarningsFound");
                           else
                              ProcessErrorSourceLine (Line);
                              DebugMessage := ELStrings.Copy_String (Str => "ReportNextErrorSourceLineFound");
                           end if;
                        end if;
                     end if;
                  end if;
               when ReportJustificationsSummaryFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 16);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "Source Filename:") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceFileStarted");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "No summarized wa") then
                     ReportFileState := ReportEndOfErrors;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfErrorsFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "--End of file---") then
                     ReportFileState := ReportEndOfReportFile;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfReportFile");
                  else
                     -- Here, we could process the justification table line to include
                     -- HTML links the relevant source files, lines of code, and error
                     -- messages.  TBD!
                     -- Both "Brief" and "Full" justifications mode need to be dealt with here.
                     null;
                  end if;

               when ReportSummarizedWarningsFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 7);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "(*Note:") then
                     ReportFileState := ReportEndOfErrors;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportEndOfErrorsFound");
                  elsif ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "Source ") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ReportSourceFileStarted");
                  end if;
               when ReportBlankAfterErrorFound |
                    ReportEndOfReportFile =>
                  ScreenEcho.Put_Line ("An error occurred during HTML report file generation: Invalid ReportFileState");
            end case;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                   E_Str => DebugMessage);  -- Line used for debugging.
            end if;

         end if;
      end ProcessReportLine;

   begin  -- GenReportHTML

      -- Do nothing if previous HTML generation failed in some way.
      if GenerateHTML then

         -- Inform user that HTML generation is taking place.
         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            ScreenEcho.Echo (EStrings.Copy_String (Str => "Generating report file HTML"));
         end if;

         -- Initialise SavedErrorLink and SavedListingFile.
         SavedErrorLink   := ELStrings.Empty_String;
         SavedListingFile := ELStrings.Empty_String;

         -- Initialise file handles.
         HTMLReportFile  := SPARK_IO.Null_File;
         PlainReportFile := SPARK_IO.Null_File;

         -- Generate filename of the form HTML/<report_filename>.htm
         PlainReportFileName := FileSystem.JustFile (CommandLineData.Content.ReportFileName, True);
         HTMLReportFileName := Translate (ELStrings.ToExaminerLongString (PlainReportFileName), '.', '_');
         ELStrings.Append_String (E_Str => HTMLReportFileName,
                                  Str   => ".htm");
         HTMLReportFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (HTMLReportFileName));

         -- Create HTML report file.
         ELStrings.Create (File          => HTMLReportFile,
                           Name_Of_File  => HTMLReportFileName,
                           Form_Of_File  => "",
                           Status        => HTMLReportFileCreatedOK);

         -- Check for errors.  Stop HTML generation if create failed.
         if HTMLReportFileCreatedOK /= SPARK_IO.Ok then

            ScreenEcho.Put_Line ("An error occurred while creating the HTML report file.");
            ScreenEcho.Put_Line ("No further HTML generation will occur.");
            GenerateHTML := False;

         else  -- file created successfully.

            -- Open report file for input
            CommandLineData.Normalize_FileName_To_Output_Directory (PlainReportFileName);

            -- Open report file for input
            EStrings.Open (File         => PlainReportFile,
                           Mode_Of_File => SPARK_IO.In_File,
                           Name_Of_File => PlainReportFileName,
                           Form_Of_File => "",
                           Status       => PlainReportFileOpenOK);

            -- Check for errors.  Stop HTML generation if open failed.
            if PlainReportFileOpenOK /= SPARK_IO.Ok then

               ScreenEcho.Put_Line ("An error occurred while opening the report file for HTML generation.");
               ScreenEcho.Put_Line ("No further HTML generation will occur.");
               GenerateHTML := False;

            else  -- file opened successfully.

               WriteHTMLReportHeader;

               -- Fill LineBuffer and LookaheadBuffer
               if not SPARK_IO.End_Of_File (PlainReportFile) then
                  ELStrings.Get_Line (File  => PlainReportFile,
                                      E_Str => LineBuffer);
                  LineBuffer := ConvertSpecialHTMLChars (LineBuffer);
                  if not SPARK_IO.End_Of_File (PlainReportFile) then
                     ELStrings.Get_Line (File  => PlainReportFile,
                                         E_Str => LookaheadBuffer);
                     LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                     -- Process first line
                     ProcessReportLine (LineBuffer, LookaheadBuffer);

                     -- Write line and process rest of file.
                     loop
                        ELStrings.Put_Line (File  => HTMLReportFile,
                                            E_Str => LineBuffer);

                        if SPARK_IO.End_Of_File (PlainReportFile) then

                           -- Process and output the lookahead buffer.
                           --# accept Flow, 10, SavedErrorLink, "Expected ineffective assignment to SavedErrorLink" &
                           --#        Flow, 10, SavedListingFile, "Expected ineffective assignment to SavedListingFile" &
                           --#        Flow, 10, ReportFileState, "Expected ineffective assignment to ReportFileState";
                           ProcessReportLine (LookaheadBuffer,
                                              ELStrings.Empty_String);  -- Flow errors expected
                           --# end accept;
                           -- this is the last call and so the saved values will not be used.
                           ELStrings.Put_Line (File  => HTMLReportFile,
                                               E_Str => LookaheadBuffer);
                           exit;
                        end if;

                        LineBuffer := LookaheadBuffer;

                        ELStrings.Get_Line (File  => PlainReportFile,
                                            E_Str => LookaheadBuffer);
                        LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                        ProcessReportLine (LineBuffer, LookaheadBuffer);

                     end loop;

                  end if;

               end if;

               WriteHTMLReportFooter;

               -- Close input report file.
               --# accept Flow, 10, PlainReportFile, "Expected ineffective assignment to PlainReportFile";
               SPARK_IO.Close (File   => PlainReportFile,  -- Flow error expected
                               Status => PlainReportFileClosedOK);
               --# end accept;


               -- Check for errors.  Stop HTML generation if close failed.
               if PlainReportFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_Line ("An error occurred while closing the report file after HTML generation.");
                  ScreenEcho.Put_Line ("No further HTML generation will occur.");
                  GenerateHTML := False;
               end if;
               --  We don't use an else here as we need to try and close the HTML file too.

               -- Close HTML output file.
               --# accept Flow, 10, HTMLReportFile, "Expected ineffective assignment to HTMLReportFile";
               SPARK_IO.Close (File   => HTMLReportFile,  -- Flow error expected
                               Status => HTMLReportFileClosedOK);
               --# end accept;

               -- Check for errors.  Stop HTML generation if close failed.
               if HTMLReportFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_Line ("An error occurred while closing the HTML report file after HTML generation.");
                  ScreenEcho.Put_Line ("No further HTML generation will occur.");
                  GenerateHTML := False;
               end if;

            end if;

         end if;

      end if;
      --# accept Flow, 601, GenerateHTML, HTMLWorkDir, "False coupling in SPARK_IO" &
      --#        Flow, 601, GenerateHTML, SparkWorkDir, "False coupling in SPARK_IO";
   end GenReportHTML;  -- Flow errors expected due to false coupling in SPARK_IO.

   -----------------------------------------------------------------------------
   --
   --  -------------
   --  GenListingHTML
   --  -------------
   --
   --  This procedure generates the HTMLified listing file from the plain text
   --  listing file.
   --
   --  It is assumed that the HTML output flag has already been tested.  The call to
   --  GenListingHTML should look something like:
   --
   --    if CommandLineData.Content.HTML then
   --       SparkHTML.GenListingHTML;
   --    end if;
   --
   --  The checking of the GenerateHTML flag is done internally.
   --
   --  If the GenerateHTML flag is false this procedure does nothing, otherwise,
   --  it does exactly the following:
   --
   --     - creates an HTML file (named <lis_file>.htm where <lis_file> is the name
   --       of the listing file associated with the file descriptor passed as a
   --       parameter with all '.' characters changed to '_' characters;
   --     - processes each line of the plain text listing file using the
   --       ProcessListingLine procedure (the functionality of this is described
   --       at the declaration of ProcessListingLine);
   --     - writes the processed lines to the HTML listing file;
   --
   --  Error trapping:
   --
   --  Incorrect generation of a listing files should not affect further HTML generation.
   --  In fact, it is better that we try and generate HTML for as many listing files as
   --  possible.  So if HTML generation fails in this procedure the GenerateHTML flag is
   --  not set to false.
   --
   procedure GenListingHTML (FileDescriptor : in ContextManager.FileDescriptors)
   is

      Message : EStrings.T;

      SavedErrorLink : ELStrings.T;

      HTMLListingFile  : SPARK_IO.File_Type;
      PlainListingFile : SPARK_IO.File_Type;

      HTMLListingFileName   : ELStrings.T;
      PlainListingFileName  : EStrings.T;
      EchoedListingFileName : EStrings.T;

      HTMLListingFileCreatedOK : SPARK_IO.File_Status;
      PlainListingFileOpenOK   : SPARK_IO.File_Status;

      HTMLListingFileClosedOK  : SPARK_IO.File_Status;
      PlainListingFileClosedOK : SPARK_IO.File_Status;

      LineBuffer      : ELStrings.T;
      LookaheadBuffer : ELStrings.T;

      ListingFileState : ListingFileStates := ListingJustStarted;

      --
      -- Subprograms
      --
      -- This sub-procedure writes HTML content to the start of the listing file.
      -- It assumes that the file handle HTMLListingFile is the open HTML
      -- listing file.  This procedure will not open or close the file.
      --
      -- The HTML written specifies the title of the page (using the filename
      -- specified as a parameter) and some formatting tags.  The formatting
      -- is <PRE> (pre-processed text) which displays text exactly as given,
      -- and <TT> which sets the typewriter-text font; I use this because it
      -- usually results in a fixed-width font being used.
      --
      -- When writing the end of the listing file these tags need to be closed,
      -- as do the <BODY> and <HTML> tags.  This should be done by calling
      -- the WriteHTMLListingFooter procedure.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLListingHeader (FileName : in EStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     HTMLListingFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                FileName,
         --#                                HTMLListingFile;

      is
         LocalFileName : EStrings.T;
      begin
         LocalFileName := FileName;

         SPARK_IO.Put_Line (HTMLListingFile,
                            "<!DOCTYPE HTML PUBLIC ""-//W3C//DTD HTML 4.0 Transitional//EN""", 61);
         SPARK_IO.Put_Line (HTMLListingFile,
                            "            ""http://www.w3.org/TR/REC-html40/loose.dtd"">", 56);
         SPARK_IO.Put_Line (HTMLListingFile, "<html>", 6);
         SPARK_IO.Put_Line (HTMLListingFile, "  <head>", 8);
         SPARK_IO.Put_String (HTMLListingFile, "    <title>SPARK Examiner HTML Listing File: ", 45);
         if CommandLineData.Content.PlainOutput then
            LocalFileName := EStrings.Lower_Case (E_Str => FileSystem.JustFile (LocalFileName, True));
         end if;
         EStrings.Put_String (File  => HTMLListingFile,
                              E_Str => LocalFileName);
         SPARK_IO.Put_Line (HTMLListingFile, "</title>", 8);
         SPARK_IO.Put_Line (HTMLListingFile, "  </head>", 9);
         SPARK_IO.Put_Line (HTMLListingFile, "  <body>", 8);
         SPARK_IO.Put_Line (HTMLListingFile, "    <pre>", 9);
         SPARK_IO.Put_Line (HTMLListingFile, "      <tt>", 10);

      end WriteHTMLListingHeader;


      -- This subprocedure writes HTML content to the end of the listing file.
      -- It assumes that the file handle HTMLListingFile is the open HTML
      -- listing file.  This procedure will not open or close the file.
      --
      -- The HTML written closes all the formatting tags that were opened
      -- by the call to WriteHTMLListingHeader.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLListingFooter
         --# global in     HTMLListingFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                HTMLListingFile;

      is

      begin

         SPARK_IO.Put_Line (HTMLListingFile, "      </tt>", 11);
         SPARK_IO.Put_Line (HTMLListingFile, "    </pre>", 10);
         SPARK_IO.Put_Line (HTMLListingFile, "  </body>", 9);
         SPARK_IO.Put_Line (HTMLListingFile, "</html>", 7);

      end WriteHTMLListingFooter;

      -- This procedure is used to convert a line of the listing file into HTML.
      -- It is effectively a parser for the listing file.
      --
      -- The procedure design is based on a state machine.  The global variable
      -- ListingFileState records our current location within the listing file (in
      -- terms of what information has already been read).
      --
      -- Given a line, we can determine the line's meaning from our current state
      -- and the contents of the line (all blank lines are ignored).
      --
      -- We can use this method to parse almost all the information in the source   --***
      -- file.  Those bits that we don't parse are usually optional (such as the    --***
      -- flag "rtc" in the options list) and require no translation to HTML anyway. --***
      --
      -- Once the procedure understands what a line represents it updates the
      -- ListingFileState and processes the line by calling an appropriate
      -- subroutine.
      --
      -- The procedure contains a debugging feature which reports each line that it
      -- finds to the screen along with a message for each line that it recognises.
      -- This only happens if the -debug switch is given on the commandline.

      procedure ProcessListingLine (Line      : in out ELStrings.T;
                                    Lookahead : in     ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in out ListingFileState;
         --#        in out SavedErrorLink;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Line,
         --#         SavedErrorLink    from Line,
         --#                                ListingFileState,
         --#                                Lookahead,
         --#                                SavedErrorLink &
         --#         ListingFileState  from *,
         --#                                Line &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Line,
         --#                                ListingFileState,
         --#                                Lookahead;
      is

         StartPos : ELStrings.Positions;
         CompareString : ELStrings.T;
         DebugMessage : ELStrings.T;

         -- This procedure processes a listing source line.  This line should
         -- consist of some spaces followed by a number followed by some more
         -- spaces and the text of the source line.
         --
         -- The line number is extracted and put into an HTML tag of the form
         -- <A NAME="lineXXX">XXX</A> where XXX is the line number.  The rest of the
         -- source line is unchanged.

         procedure ProcessListingSourceLine (Line : in out ELStrings.T)
            --# derives Line from *;
         is

            OutString : ELStrings.T;
            I         : ELStrings.Positions;
            LineNo    : ELStrings.T;
            Link      : ELStrings.T;

         begin
            OutString := ELStrings.Empty_String;
            I         := ELStrings.Positions'First;
            LineNo     := ELStrings.Empty_String;

            -- Copy up to the first non-space.
            loop
               exit when ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I) /= ' ';
               AppendCharWithoutCheck
                 (OutString,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            -- Copy digits to LineNo
            while Digit (ELStrings.Get_Element (E_Str => Line,
                                                Pos   => I)) loop
               AppendCharWithoutCheck
                 (LineNo,
                  ELStrings.Get_Element (E_Str => Line,
                                         Pos   => I));
               I := I + 1;
            end loop;

            if ELStrings.Get_Length (E_Str => LineNo) > 0 then
               -- Create anchor based on number
               Link := ELStrings.Copy_String (Str => "<A NAME=""line");
               ELStrings.Append_Examiner_Long_String (E_Str1 => Link,
                                                      E_Str2 => LineNo);
               ELStrings.Append_String (E_Str => Link,
                                        Str   => """>");
               ELStrings.Append_Examiner_Long_String (E_Str1 => Link,
                                                      E_Str2 => LineNo);
               ELStrings.Append_String (E_Str => Link,
                                        Str   => "</A>");

               -- Append link to OutString
               ELStrings.Append_Examiner_Long_String (E_Str1 => OutString,
                                                      E_Str2 => Link);

            end if;

            -- Append rest of line
            ELStrings.Append_Examiner_Long_String
              (E_Str1 => OutString,
               E_Str2 => ELStrings.Section
                 (Line,
                  I,
                  (ELStrings.Get_Length (E_Str => Line) - I) + 1));

            Line := OutString;

         end ProcessListingSourceLine;

      begin  -- ProcessListingLine
         if ELStrings.Get_Length (E_Str => Line) > 0 then

            StartPos := FirstChar (Line);
            DebugMessage := ELStrings.Empty_String;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                   E_Str => Line);  -- Line used for debugging.
            end if;

            case ListingFileState is
               when ListingJustStarted =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 5),
                     Str   => "*****") then
                     ListingFileState := ListingBannerStarted;
                     DebugMessage := ELStrings.Copy_String (Str => "ListingBannerStarted");
                  end if;
               when ListingBannerStarted =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 5),
                     Str   => "*****") then
                     ListingFileState := ListingBannerFinished;
                     DebugMessage := ELStrings.Copy_String (Str => "ListingBannerFinished");
                  end if;
               when ListingBannerFinished =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 7),
                     Str   => "DATE : ") then
                     ListingFileState := ListingDateFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ListingDateFound");
                  end if;
               when ListingDateFound =>
                  if ELStrings.Eq1_String
                    (E_Str => ELStrings.Section (E_Str     => Line,
                                                 Start_Pos => StartPos,
                                                 Length    => 4),
                     Str   => "Line") then
                     ListingFileState := ListingLineHeadingFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ListingLineHeadingFound");
                  end if;
               when ListingLineHeadingFound | ListingSourceLineFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 1);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "^") then
                     -- Process Error Pointer Line (do nothing?)
                     ListingFileState := ListingErrorSourcePointerFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ListingErrorSourcePointerFound");
                  else  -- Some errors don't have pointers
                     CompareString := ELStrings.Section (E_Str     => Line,
                                                         Start_Pos => StartPos,
                                                         Length    => 3);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "!!!") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "***") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "---") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "???") then

                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);

                        ListingFileState := ListingErrorMessageFound;
                        DebugMessage := ELStrings.Copy_String (Str => "ListingErrorMessageFound");
                     else
                        ProcessListingSourceLine (Line);
                        ListingFileState := ListingSourceLineFound;
                        DebugMessage := ELStrings.Copy_String (Str => "ListingSourceLineFound");
                     end if;
                  end if;
               when ListingErrorSourcePointerFound =>
                  CompareString := ELStrings.Section (E_Str     => Line,
                                                      Start_Pos => StartPos,
                                                      Length    => 3);
                  if ELStrings.Eq1_String (E_Str => CompareString,
                                           Str   => "!!!") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "***") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "---") or else
                    ELStrings.Eq1_String (E_Str => CompareString,
                                          Str   => "???") then

                     ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                     ListingFileState := ListingErrorMessageFound;
                     DebugMessage := ELStrings.Copy_String (Str => "ListingErrorMessageFound");
                  end if;
               when ListingErrorMessageFound =>
                  if StartPos > 5 then -- Extra lines of the error message will be indented by 11 characters
                     ProcessNextErrorMessageLine (Line, SavedErrorLink);
                     DebugMessage := ELStrings.Copy_String (Str => "ListingNextErrorMessageLineFound");
                  else
                     CompareString := ELStrings.Section (E_Str     => Line,
                                                         Start_Pos => StartPos,
                                                         Length    => 3);
                     if ELStrings.Eq1_String (E_Str => CompareString,
                                              Str   => "!!!") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "***") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "---") or else
                       ELStrings.Eq1_String (E_Str => CompareString,
                                             Str   => "???") then

                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                        ListingFileState := ListingErrorMessageFound;
                        DebugMessage := ELStrings.Copy_String (Str => "ListingErrorMessageFound");
                     else  -- error source line or summarized warnings
                        ProcessListingSourceLine (Line);
                        ListingFileState := ListingSourceLineFound;
                        DebugMessage := ELStrings.Copy_String (Str => "ListingNextSourceLineFound");
                     end if;
                  end if;
               when ListingEndOfListingFile =>
                  ScreenEcho.Put_Line ("An error occurred during HTML listing file generation: Invalid ReportFileState");
            end case;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                   E_Str => DebugMessage);  -- Line used for debugging.
            end if;

         end if;
      end ProcessListingLine;

   begin

      if GenerateHTML then

         -- Get name of listing file.
         ContextManager.Ops.GetListingFileName (FileDescriptor, PlainListingFileName);

         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            -- Echo message to screen.
            Message := EStrings.Copy_String (Str => "Generating listing file HTML for ");

            if CommandLineData.Content.PlainOutput then
               EchoedListingFileName := EStrings.Lower_Case (E_Str => PlainListingFileName);
            else
               EchoedListingFileName := PlainListingFileName;
            end if;
            EStrings.Append_Examiner_String (E_Str1 => Message,
                                             E_Str2 => EchoedListingFileName);
            ScreenEcho.Echo (Message);
         end if;

         -- Initialise SavedErrorLink.
         SavedErrorLink := ELStrings.Empty_String;

         -- Initialise file handles.
         HTMLListingFile  := SPARK_IO.Null_File;
         PlainListingFile := SPARK_IO.Null_File;

         -- Generate filename of the form HTML/<listing_filename>.htm
         HTMLListingFileName := Translate (
            ELStrings.ToExaminerLongString (PlainListingFileName), '.', '_');
         ELStrings.Append_String (E_Str => HTMLListingFileName,
                                  Str   => ".htm");
         HTMLListingFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (HTMLListingFileName));

         -- Create HTML listing file.
         ELStrings.Create (File          => HTMLListingFile,
                           Name_Of_File  => HTMLListingFileName,
                           Form_Of_File  => "",
                           Status        => HTMLListingFileCreatedOK);

         -- Check for errors.
         if HTMLListingFileCreatedOK /= SPARK_IO.Ok then

            ScreenEcho.Put_String ("An error occurred while creating the HTML listing file ");
            ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => HTMLListingFileName);

         else  -- file created successfully.

            CommandLineData.Normalize_FileName_To_Output_Directory (PlainListingFileName);

            -- Open listing file for input
            EStrings.Open (File         => PlainListingFile,
                           Mode_Of_File => SPARK_IO.In_File,
                           Name_Of_File => PlainListingFileName,
                           Form_Of_File => "",
                           Status       => PlainListingFileOpenOK);

            -- Check for errors.
            if PlainListingFileOpenOK /= SPARK_IO.Ok then

               ScreenEcho.Put_String ("An error occurred while opening the listing file ");
               EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                    E_Str => PlainListingFileName);
               ScreenEcho.Put_Line   (" for HTML generation.");

            else  -- file opened successfully.

               WriteHTMLListingHeader (PlainListingFileName);

               -- Fill LineBuffer and LookaheadBuffer
               if not SPARK_IO.End_Of_File (PlainListingFile) then
                  ELStrings.Get_Line (File  => PlainListingFile,
                                      E_Str => LineBuffer);
                  LineBuffer := ConvertSpecialHTMLChars (LineBuffer);
                  if not SPARK_IO.End_Of_File (PlainListingFile) then
                     ELStrings.Get_Line (File  => PlainListingFile,
                                         E_Str => LookaheadBuffer);
                     LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                     -- Process first line
                     ProcessListingLine (LineBuffer, LookaheadBuffer);

                     -- Write line and process rest of file.
                     loop
                        ELStrings.Put_Line (File  => HTMLListingFile,
                                            E_Str => LineBuffer);

                        if SPARK_IO.End_Of_File (PlainListingFile) then

                           -- Process and output the lookahead buffer.
                           --# accept Flow, 10, ListingFileState, "Expected ineffective assignment to ListingFileState" &
                           --#        Flow, 10, SavedErrorLink, "Expected ineffective assignment to SavedErrorLink";
                           ProcessListingLine (LookaheadBuffer,
                                               ELStrings.Empty_String);  -- Flow errors expected
                           --#end accept;
                           -- this is the last call and so the saved values will not be used.
                           ELStrings.Put_Line (File  => HTMLListingFile,
                                               E_Str => LookaheadBuffer);
                           exit;
                        end if;

                        LineBuffer := LookaheadBuffer;

                        ELStrings.Get_Line (File  => PlainListingFile,
                                            E_Str => LookaheadBuffer);
                        LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                        ProcessListingLine (LineBuffer, LookaheadBuffer);

                     end loop;

                  end if;

               end if;

               WriteHTMLListingFooter;

               -- Close input listing file.
               --# accept Flow, 10, PlainListingFile, "Expected ineffective assignment to PlainListingFile";
               SPARK_IO.Close (File   => PlainListingFile,
                               Status => PlainListingFileClosedOK);
               --# end accept;


               -- Check for errors.
               if PlainListingFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_String ("An error occurred while closing the listing file ");
                  EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                       E_Str => PlainListingFileName);
                  ScreenEcho.Put_Line (" after HTML generation.");
               end if;
               --  We don't use an else here as we need to try and close the HTML file too.

               -- Close HTML output file.
               --# accept Flow, 10, HTMLListingFile, "Expected ineffective assignment to HTMLListingFile";
               SPARK_IO.Close (File   => HTMLListingFile,
                               Status => HTMLListingFileClosedOK);
               --# end accept;

               -- Check for errors.
               if HTMLListingFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_String ("An error occurred while closing the HTML listing file ");
                  ELStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                        E_Str => HTMLListingFileName);
                  ScreenEcho.Put_Line (" after HTML generation.");
               end if;

            end if;

         end if;


      end if;

   end GenListingHTML;

end SparkHTML;
