-- $Id: spark_xml.adb 14620 2009-10-28 13:40:30Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SPARK_IO,
     Ada.Characters.Latin_1;

package body SPARK_XML
is


   -------------------------------
   -- General Utility functions --
   -------------------------------

   -- Simple constructor function to build an EStrings.T (presently an ExaminerString)
   -- This is public because is makes definign schemas easier, without having to keep
   -- track of lots of temporary variable for all the strings.
   function XStr (Str : in String) return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => Str);
   end XStr;


   -- Cleans up a string, performing the following replacements
   -- Character   Replacement
   -- =========   ===========
   --    <        &lt;
   --    >        &gt;
   --    &        &amp;
   --    '        &apos;
   --    "        &quot;
   -- These are fundamental characters in XML and cannot occur in
   -- character data (tag attributes, or character data.
   function FilterString (Str : in EStrings.T) return EStrings.T
   is
      OldString : EStrings.T;
      NewString : EStrings.T := EStrings.Empty_String;
      Success : Boolean := True;
      Ch : Character;
   begin

      OldString := Str;

      while (EStrings.Get_Length (E_Str => OldString) > 0) and Success loop
         EStrings.Pop_Char (E_Str => OldString,
                            Char  => Ch);
         case Ch is
            when '<'    => EStrings.Append_String (E_Str => NewString,
                                                   Str   => "&lt;");
            when '>'    => EStrings.Append_String (E_Str => NewString,
                                                   Str   => "&gt;");
            when '&'    => EStrings.Append_String (E_Str => NewString,
                                                   Str   => "&amp;");
            when '''    => EStrings.Append_String (E_Str => NewString,
                                                   Str   => "&apos;");
            when '"'    => EStrings.Append_String (E_Str => NewString,
                                                   Str   => "&quot;");
            when others => EStrings.Append_Char (E_Str   => NewString,
                                                 Ch      => Ch,
                                                 Success => Success);
         end case;
      end loop;

      return NewString;

   end FilterString;

   function FilterLongString (Str : in ELStrings.T) return ELStrings.T
   is
      OldString : ELStrings.T;
      NewString : ELStrings.T := ELStrings.Empty_String;
      Success : Boolean := True;
      Ch : Character;
   begin

      OldString := Str;

      while (ELStrings.Get_Length (E_Str => OldString) > 0) and Success loop
         ELStrings.Pop_Char (E_Str => OldString,
                             Char  => Ch);
         case Ch is
            when '<'    => ELStrings.Append_String (E_Str => NewString,
                                                    Str   => "&lt;");
            when '>'    => ELStrings.Append_String (E_Str => NewString,
                                                    Str   => "&gt;");
            when '&'    => ELStrings.Append_String (E_Str => NewString,
                                                    Str   => "&amp;");
            when '''    => ELStrings.Append_String (E_Str => NewString,
                                                    Str   => "&apos;");
            when '"'    => ELStrings.Append_String (E_Str => NewString,
                                                    Str   => "&quot;");
            when others => ELStrings.Append_Char (E_Str   => NewString,
                                                  Ch      => Ch,
                                                  Success => Success);
         end case;
      end loop;

      return NewString;

   end FilterLongString;


   -------------------
   -- Schema Access --
   -------------------

   function GetTagName (Schema : in SchemaRecord;
                        TID    : in TagID) return EStrings.T
   is
   begin
      return Schema.Tags.TagArray (TID).Name;
   end GetTagName;

   function GetAttributeName (Schema : in SchemaRecord;
                              AID    : in AttributeID) return EStrings.T
   is
   begin
      return (Schema.Attributes.AttributeArray (AID).Name);
   end GetAttributeName;


   function GetTagAttributes (Schema : in SchemaRecord;
                              TID    : in TagID) return TagAttributeArray
   is
   begin
      return Schema.Tags.TagArray (TID).TagAttributes;
   end GetTagAttributes;


   function GetTagAttribute (Schema : in SchemaRecord;
                             TID    : in TagID;
                             TAID   : in TagAttributeArrayIndex) return AttributeID
   is
   begin
      return Schema.Tags.TagArray (TID).TagAttributes (TAID);
   end GetTagAttribute;


   function FindTag (Schema : in SchemaRecord;
                     Name   : in EStrings.T) return TagID
   is
      Found : TagID := NullTag;
   begin

      for I in TagID loop
         if EStrings.Eq_String (E_Str1 => Name,
                                E_Str2 => GetTagName (Schema, I)) then
            Found := I;
            exit;
         end if;
      end loop;

      return Found;

   end FindTag;


   function GetAttributeType (Schema : in SchemaRecord;
                              AID    : in AttributeID) return AttributeType
   is
   begin
      return Schema.Attributes.AttributeArray (AID).ContentType;
   end GetAttributeType;


   function AttributeIsRequired (Schema : in SchemaRecord;
                                 AID    : in AttributeID) return Boolean
   is
   begin
      return Schema.Attributes.AttributeArray (AID).Required;
   end AttributeIsRequired;


   function IsNullAttribute (AID : in AttributeID) return Boolean
   is
   begin
      return (AID = NullAttribute);
   end IsNullAttribute;


   function IsNullTag (TID : in TagID) return Boolean
   is
   begin
      return (TID = NullTag);
   end IsNullTag;


   function GetLastChildTag (Schema : in SchemaRecord;
                             TID    : in TagID) return ChildTagArrayIndex
   is
   begin
      return Schema.Tags.TagArray (TID).LastChild;
   end GetLastChildTag;


   function GetChildTags (Schema : in SchemaRecord;
                          TID    : in TagID) return ChildTagArray
   is
   begin
      return Schema.Tags.TagArray (TID).ChildTags;
   end GetChildTags;

   function GetChildTag (Schema : in SchemaRecord;
                         TID    : in TagID;
                         CTID   : in ChildTagArrayIndex) return ChildTag
   is
   begin
      return Schema.Tags.TagArray (TID).ChildTags (CTID);
   end GetChildTag;

   function IsLegalChild (Schema : in SchemaRecord;
                          Parent : in TagID;
                          Child  : in TagID) return Boolean
   is

      Found    : Boolean := False;
      Children : ChildTagArray;
      Upper    : ChildTagArrayIndex;
   begin

      if not IsNullTag (Child) then  -- The Null tag is never valid.

         Children := GetChildTags (Schema, Parent);

         Upper := GetLastChildTag (Schema, Parent);

         for I in ChildTagArrayIndex range ChildTagArrayIndex'First .. Upper loop

            --# assert Upper = Upper% and Upper in ChildTagArrayIndex;

            if Children (I).Child = Child then
               Found := True;
               exit;
            end if;
         end loop;

      end if;

      return Found;
   end IsLegalChild;





   ---------------------
   -- Schema creation --
   ---------------------

   -- Initialise the schema variables.
   procedure InitSchema (Schema : out SchemaRecord)
   is
   begin
      Schema := EmptySchemaRecord;
   end InitSchema;


   -- Add a tag to the schema
   procedure AddTag (Schema : in out SchemaRecord;
                     Name   : in     EStrings.T;
                     ID     :    out TagID)
   is

   begin

      if (Schema.Tags.LastTag < TagID'Last) then
         Schema.Tags.LastTag := TagID'Succ (Schema.Tags.LastTag);
         Schema.Tags.TagArray (Schema.Tags.LastTag).Name := Name;
         ID := Schema.Tags.LastTag;
      else
         ID := NullTag;
      end if;

   end AddTag;



   procedure AddAttributeToTag (Schema      : in out SchemaRecord;
                                TID         : in     TagID;
                                Name        : in     EStrings.T;
                                ContentType : in     AttributeType;
                                Required    : in     Boolean;
                                ID          :    out AttributeID;
                                Success     :    out Boolean)
   is

      TempAID : AttributeID;

      -- Add an attribute to a tag
      procedure AddAttribute (Schema      : in out SchemaRecord;
                              Name        : in     EStrings.T;
                              ContentType : in     AttributeType;
                              Required    : in     Boolean;
                              ID          :    out AttributeID)
      --# derives ID     from Schema &
      --#         Schema from *,
      --#                     ContentType,
      --#                     Name,
      --#                     Required;
      is

      begin

         if (Schema.Attributes.LastAttribute < AttributeID'Last) then
            Schema.Attributes.LastAttribute := AttributeID'Succ (Schema.Attributes.LastAttribute);
            Schema.Attributes.AttributeArray (Schema.Attributes.LastAttribute) := Attribute'(Name => Name,
                                                                                             ContentType => ContentType,
                                                                                             Required => Required);
            ID := Schema.Attributes.LastAttribute;
         else
            ID := NullAttribute;  -- Return the null attribute to indicate failure
         end if;

      end AddAttribute;


      procedure AttachAttribute (Schema  : in out SchemaRecord;
                                 AID     : in     AttributeID;
                                 TID     : in     TagID;
                                 Success :    out Boolean)
      --# derives Schema  from *,
      --#                      AID,
      --#                      TID &
      --#         Success from Schema,
      --#                      TID;
      is
         TmpTag : Tag;
      begin

         TmpTag := Schema.Tags.TagArray (TID);

         if (TmpTag.LastTagAttribute < MaxAttributesPerTag) then

            TmpTag.TagAttributes (TmpTag.LastTagAttribute) := AID;
            TmpTag.LastTagAttribute := TagAttributeArrayIndex'Succ (TmpTag.LastTagAttribute);

            Schema.Tags.TagArray (TID) := TmpTag;
            Success := True;

         else
            Success := False;
         end if;

      end AttachAttribute;

   begin
      AddAttribute (Schema,
                    Name,
                    ContentType,
                    Required,
                    TempAID);

      if not IsNullAttribute (TempAID) then
         AttachAttribute (Schema,
                          TempAID,
                          TID,
                          Success);
      else  -- Failed to add the attribute to the attribute database
         Success := False;
      end if;

      ID := TempAID;

   end AddAttributeToTag;



   procedure AddChildTag (Schema   : in out SchemaRecord;
                          TID      : in     TagID;
                          Child    : in     TagID;
                          Required : in     Boolean;
                          Success  :    out Boolean)
   is

   begin

      if (Schema.Tags.TagArray (TID).LastChild < ChildTagArrayIndex'Last) then

         Schema.Tags.TagArray (TID).ChildTags (Schema.Tags.TagArray (TID).LastChild) := ChildTag'(Child => Child,
                                                                                                  Required => Required);
         Schema.Tags.TagArray (TID).LastChild := ChildTagArrayIndex'Succ (Schema.Tags.TagArray (TID).LastChild);

         Success := True;
      else
         Success := False;
      end if;

   end AddChildTag;



   procedure AddCDATA (Schema : in out SchemaRecord;
                       TID    : in     TagID)
   is
   begin
      Schema.Tags.TagArray (TID).AllowCDATA := True;
   end AddCDATA;


   function CDATA (Schema : in SchemaRecord;
                   TID    : in TagID) return Boolean
   is
   begin
      return Schema.Tags.TagArray (TID).AllowCDATA;
   end CDATA;



   -----------------------------------
   -- ScheamState access and update --
   -----------------------------------


   procedure InitSchemaState (SchemaState : out SchemaStateRecord)
   is
   begin
      SchemaState := EmptySchemaStateRecord;
   end InitSchemaState;


   function TagStackPeek (SchemaState : in SchemaStateRecord) return TagID
   is
   begin
      return SchemaState.TagStack.Stack (SchemaState.TagStack.Current);
   end TagStackPeek;

   function TagStackPeekN (SchemaState : in SchemaStateRecord;
                           N           : in TagDepth) return TagID
   is
   begin
      return SchemaState.TagStack.Stack (N);
   end TagStackPeekN;

   function TagStackDepth (SchemaState : in SchemaStateRecord) return TagDepth
   is
   begin
      return SchemaState.TagStack.Current;
   end TagStackDepth;

   function TagStackEmpty (SchemaState : in SchemaStateRecord) return Boolean
   is
   begin
      return (TagStackDepth (SchemaState) = TagDepth'First);
   end TagStackEmpty;

   function TagStackFull (SchemaState : in SchemaStateRecord) return Boolean
   is
   begin
      return (TagStackDepth (SchemaState) = TagDepth'Last);
   end TagStackFull;


   procedure TagStackPush (SchemaState : in out SchemaStateRecord;
                          Depth        :    out TagDepth;
                          Status       :    out SchemaStatus)
   --# derives Depth,
   --#         SchemaState,
   --#         Status      from SchemaState;
   is
      ReturnDepth : TagDepth := TagDepth'First;
   begin

      if SchemaState.TagStack.Current = TagDepth'Last then -- Full
         Status := SSStackFull;
      else
         SchemaState.TagStack.Current := TagDepth'Succ (SchemaState.TagStack.Current);
         SchemaState.TagStack.Stack (SchemaState.TagStack.Current) := SchemaState.WorkingTag.TID;
         ReturnDepth := SchemaState.TagStack.Current;
         Status := SSOK;
      end if;

      Depth := ReturnDepth;

   end TagStackPush;


   procedure TagStackPop (SchemaState : in out SchemaStateRecord;
                          TID         :    out TagID;
                          Status      :    out SchemaStatus)
   --# derives SchemaState,
   --#         Status,
   --#         TID         from SchemaState;
   is
   begin

      if SchemaState.TagStack.Current = TagDepth'First then -- Empty
         Status := SSStackEmpty;
         TID := NullTag;
      else
         TID := TagStackPeek (SchemaState);
         SchemaState.TagStack.Current := TagDepth'Pred (SchemaState.TagStack.Current);
         Status := SSOK;
      end if;

   end TagStackPop;


   function GetWorkingAttributeVal (SchemaState : in SchemaStateRecord;
                                    TAID        : in TagAttributeArrayIndex) return EStrings.T
   is
   begin
      return EStrings.Trim (SchemaState.WorkingTag.Attribs (TAID).Val);
   end GetWorkingAttributeVal;


   function GetWorkingAttributeID (SchemaState : in SchemaStateRecord;
                                   TAID        : in TagAttributeArrayIndex) return AttributeID
   is
   begin
      return SchemaState.WorkingTag.Attribs (TAID).AID;
   end GetWorkingAttributeID;


   procedure SetWorkingAttribute (SchemaState : in out SchemaStateRecord;
                                  TAID        : in     TagAttributeArrayIndex;
                                  AID         : in     AttributeID;
                                  Val         : in     EStrings.T)
   --# derives SchemaState from *,
   --#                          AID,
   --#                          TAID,
   --#                          Val;
   is
   begin
      SchemaState.WorkingTag.Attribs (TAID).AID := AID;
      SchemaState.WorkingTag.Attribs (TAID).Val := Val;
   end SetWorkingAttribute;


   function TagStackHuntUp (SchemaState : in SchemaStateRecord;
                            TID         : in TagID) return TagDepth
   is
      Location : TagDepth := TagDepth'First;
      Upper    : TagDepth;
   begin
      Upper := TagStackDepth (SchemaState);
      for I in TagDepth range TagDepth'First .. Upper loop

         --# assert Upper = Upper% and Upper in TagDepth;

         if TID = TagStackPeekN (SchemaState, I) then
            Location := I;
            exit;
         end if;

      end loop;

      return Location;

   end TagStackHuntUp;


   function TagStackHuntDown (SchemaState : in SchemaStateRecord;
                              TID         : in TagID) return TagDepth
   is
      Location : TagDepth := TagDepth'First;
      Upper    : TagDepth;
   begin
      Upper := TagStackDepth (SchemaState);
      for I in reverse TagDepth range TagDepth'First .. Upper loop

         --# assert Upper = Upper% and Upper in TagDepth;

         if TID = TagStackPeekN (SchemaState, I) then
            Location := I;
            exit;
         end if;

      end loop;

      return Location;

   end TagStackHuntDown;



   function GetReqAttributes (Schema      : in SchemaRecord;
                              SchemaState : in SchemaStateRecord) return TagAttributeArray
   is
      TagAttributeList : TagAttributeArray;

      -- Tag Attributes that we have found.
      RAID : TagAttributeArray := TagAttributeArray'(others => NullAttribute);
      NextSlot : TagAttributeArrayIndex := TagAttributeArrayIndex'First;  -- Pointer to end of the list.

   begin

      TagAttributeList := GetTagAttributes (Schema, SchemaState.WorkingTag.TID);

      for CurrentAttribute in TagAttributeArrayIndex loop
         --# assert NextSlot >= TagAttributeArrayIndex'First and
         --#        NextSlot <= CurrentAttribute;
         if AttributeIsRequired (Schema, TagAttributeList (CurrentAttribute)) then
            RAID (NextSlot) := TagAttributeList (CurrentAttribute);

            -- If there is room for another, increment the NextSlot.
            -- As both arrays are the same size, if this condition fails then the loop
            -- will exit.
            if NextSlot < TagAttributeArrayIndex'Last then
               NextSlot := NextSlot + 1;
            end if;
         end if;
      end loop;

      return RAID;

   end GetReqAttributes;


   function AttributeIsSet (SchemaState : in SchemaStateRecord;
                            AID         : in AttributeID)  return Boolean
   is
      Success : Boolean := False;
   begin
      for I in TagAttributeArrayIndex loop

         if SchemaState.WorkingTag.Attribs (I).AID = AID then
            Success := True;
            exit;
         end if;
      end loop;

      return Success;

   end AttributeIsSet;


   function AllRequiredAttributes (Schema      : in SchemaRecord;
                                   SchemaState : in SchemaStateRecord) return Boolean
   is
      Required : TagAttributeArray;
      Success : Boolean := True;
   begin

      Required := GetReqAttributes (Schema, SchemaState);

      for I in TagAttributeArrayIndex loop
         if not AttributeIsSet (SchemaState, Required (I)) then
            Success := False;
            exit;
         end if;
      end loop;

      return Success;

   end AllRequiredAttributes;


   function ValidWorkingTag (Schema      : in SchemaRecord;
                             SchemaState : in SchemaStateRecord) return Boolean
   is
   begin
      return AllRequiredAttributes (Schema, SchemaState)
        and (not (SchemaState.WorkingTag = EmptyWorkingTag));
   end ValidWorkingTag;



   -----------
   -- Debug --
   -----------

   function IsError (Error : in SchemaStatus) return Boolean
   is
   begin
      return not (Error = SSOK);
   end IsError;

   procedure PrintSchemaError (Error : in SchemaStatus)
   is
      --# hide PrintSchemaError;
      type MessageArray is array (SchemaStatus) of String (1 .. 55);

      Messages : constant MessageArray :=
        MessageArray'(SSOK               => "Schema state OK                                        ",
                      SSInvalidAttribute => "Invalid attribute for working tag                      ",
                      SSInvalidTag       => "Invalid tag at this point                              ",
                      SSToManyAttributes => "Reached attribute limit                                ",
                      SSWrongContentType => "Attempt to assign value of incorrect type to attribute ",
                      SSStackFull        => "The Schema Stack is full                               ",
                      SSStackEmpty       => "The Schema Stack is empty                              ",
                      SSTagIncomplete    => "One or more required attribute is missing              ",
                      SSInvalidDepth     => "There are no tags at this depth                        ",
                      SSNoSuchTag        => "The specified tag is not in the schema                 ",
                      SSTagNotFound      => "Could not find an instance of that tag in the hierarchy");

   begin
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                        Messages (Error),
                        55);
   end PrintSchemaError;



   procedure PrintWorkingState (Schema      : in SchemaRecord;
                                SchemaState : in SchemaStateRecord)
   is
      --# hide PrintWorkingState;

      procedure PrintTagSchema (Tag : in TagID)
      is

      begin
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("<<<<<<<<<< Tag Schema >>>>>>>>>>"));
         EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                              E_Str => XStr ("Tag : "));
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => GetTagName (Schema, Tag));
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr (" "));
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("Attributes"));
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("----------"));

         for I in TagAttributeArrayIndex loop

            declare
               TmpAttributeID : constant AttributeID := GetTagAttribute (Schema, Tag, I);
            begin
               if AttributeIsRequired (Schema, TmpAttributeID) then
                  EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                       E_Str => XStr (" * "));
               else
                  EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                       E_Str => XStr ("   "));
               end if;

               EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                    E_Str => GetAttributeName (Schema, TmpAttributeID));

               case GetAttributeType (Schema, TmpAttributeID) is
                  when ATString => EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                                      E_Str => XStr ("  STRING"));
                  when ATInteger => EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                                       E_Str => XStr ("  INTEGER"));
                  when ATFloat => EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                                     E_Str => XStr ("  FLOAT"));
                  when ATNULL => EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                                    E_Str => XStr (""));
               end case;

            end;

         end loop;

         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("Child Tags"));
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("----------"));

         for I in ChildTagArrayIndex loop

            declare
               TmpChild : constant ChildTag := GetChildTag (Schema, Tag, I);
            begin
               if TmpChild.Required then
                  EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                       E_Str => XStr (" * "));
               else
                  EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                       E_Str => XStr ("   "));
               end if;

               EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                  E_Str => GetTagName (Schema, TmpChild.Child));

            end;

         end loop;

      end PrintTagSchema;


      procedure PrintWorkingTagState
      is

         procedure PrintWorkingAttribute (Attrib : in TagAttributeArrayIndex)
         is
         begin

            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => GetAttributeName (Schema,
                                                            SchemaState.WorkingTag.Attribs (Attrib).AID));

            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => XStr (" = "));

            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => SchemaState.WorkingTag.Attribs (Attrib).Val);

         end PrintWorkingAttribute;


      begin

         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("<<<<<<< WorkingTag State >>>>>>>"));

         for I in TagAttributeArrayIndex loop
            PrintWorkingAttribute (I);
         end loop;

      end PrintWorkingTagState;


      procedure PrintStack
      is
      begin
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => XStr ("<<<<<<<<<< Tag Stack  >>>>>>>>>>"));

         for I in TagDepth loop
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => XStr (TagDepth'Image (I)));
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => XStr (" "));
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => GetTagName (Schema, TagStackPeekN (SchemaState, I)));

            exit when I = SchemaState.TagStack.Current;
         end loop;
      end PrintStack;

   begin
      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => XStr ("================"));
      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => XStr ("START DEBUG DUMP "));
      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => XStr ("================"));

      PrintTagSchema (SchemaState.WorkingTag.TID);
      PrintTagSchema (TagStackPeek (SchemaState));
      PrintWorkingTagState;
      PrintStack;

      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => XStr ("================"));
      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => XStr (" END DEBUG DUMP "));
      EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                         E_Str => XStr ("================"));

   end PrintWorkingState;




   ------------------
   -- Tag Creation --
   ------------------


   -- This should only be used carefully.  It opens a tag without first checking that
   -- it is a legal child of the presently open tag.  This is for use when tags have to
   -- be generated out of order.

   procedure InitOpeningTagNOCHECK (SchemaState : in out SchemaStateRecord;
                                    TID         : in     TagID;
                                    Status      :    out SchemaStatus)
   is

   begin

      if TagStackFull (SchemaState) then  -- Check that we can actually generate another tag

         Status := SSStackFull;

      else

         if IsNullTag (TID) then
            Status := SSNoSuchTag;  -- Null tag.
         else
            SchemaState.WorkingTag :=
              WorkingTagType'(TID => TID,
                              Attribs => WorkingAttributeArray'(others =>
                                                                  WorkingAttribute'(AID => NullAttribute,
                                                                                    Val => EStrings.Empty_String)));
            Status := SSOK;
         end if;

      end if;

   end InitOpeningTagNOCHECK;



   -- Opening tags
   -- Initialise the opening tag, then add attributes to it.
   -- Then call OutputOpeningTag to return the string.

   procedure InitOpeningTagByID (Schema      : in     SchemaRecord;
                                 SchemaState : in out SchemaStateRecord;
                                 TID         : in     TagID;
                                 Status      :    out SchemaStatus)
   is

   begin

      if TagStackFull (SchemaState) then  -- Check that we can actually generate another tag

         Status := SSStackFull;

      else

         if IsLegalChild (Schema,
                          TagStackPeek (SchemaState),
                          TID) then

            SchemaState.WorkingTag :=
              WorkingTagType'(TID => TID,
                              Attribs => WorkingAttributeArray'(others =>
                                                                  WorkingAttribute'(AID => NullAttribute,
                                                                                    Val => EStrings.Empty_String)));

            Status := SSOK;

         else
            SchemaState.WorkingTag := EmptyWorkingTag;
            Status := SSInvalidTag;
         end if;
      end if;


   end InitOpeningTagByID;


   procedure InitOpeningTag (Schema      : in     SchemaRecord;
                             SchemaState : in out SchemaStateRecord;
                             Name        : in     EStrings.T;
                             Status      :    out SchemaStatus)
   is

   begin
      InitOpeningTagByID (Schema,
                          SchemaState,
                          FindTag (Schema, Name),
                          Status);
   end InitOpeningTag;





   procedure FindAttribute (Schema   : in     SchemaRecord;
                            TagIdent : in     TagID;
                            Name     : in     EStrings.T;
                            CType    : in     AttributeType;
                            AID      :    out AttributeID;
                            Status   :    out SchemaStatus)
   --# derives AID,
   --#         Status from CType,
   --#                     Name,
   --#                     Schema,
   --#                     TagIdent;
   is
      AArray : TagAttributeArray;
      Found : AttributeID := NullAttribute;
   begin

      AArray := GetTagAttributes (Schema, TagIdent);

      for I in TagAttributeArrayIndex loop
         if EStrings.Eq_String (E_Str1 => Name,
                                E_Str2 => GetAttributeName (Schema,
                                                            AArray (I))) then
            Found := AArray (I);
            exit;
         end if;
      end loop;

      if IsNullAttribute (Found) then
         Status := SSInvalidAttribute;
         AID := NullAttribute;
      elsif not (GetAttributeType (Schema, Found) = CType) then
         Status := SSWrongContentType;
         AID := NullAttribute;
      else
         Status := SSOK;
         AID := Found;
      end if;

   end FindAttribute;


   procedure AddWorkingAttribute (SchemaState : in out SchemaStateRecord;
                                  AID         : in     AttributeID;
                                  Value       : in     EStrings.T;
                                  Status      :    out SchemaStatus)
   --# derives SchemaState from *,
   --#                          AID,
   --#                          Value &
   --#         Status      from SchemaState;
   is

      Found : TagAttributeArrayIndex := TagAttributeArrayIndex'First;

   begin

      -- Find the next free slot
      for I in TagAttributeArrayIndex loop
         if IsNullAttribute (GetWorkingAttributeID (SchemaState, I)) then
            Found := I;
            exit;
         end if;
      end loop;

      -- Add the attribute to the working tag
      if IsNullAttribute (GetWorkingAttributeID (SchemaState, Found)) then
         SetWorkingAttribute (SchemaState, Found, AID, FilterString (Value));
         Status := SSOK;
      else
         -- We didn't find a free spot
         Status := SSToManyAttributes;
      end if;

   end AddWorkingAttribute;


   procedure AddAttributeStr (Schema      : in     SchemaRecord;
                              SchemaState : in out SchemaStateRecord;
                              Name        : in     EStrings.T;
                              Value       : in     EStrings.T;
                              Status      :    out SchemaStatus)
   is
      TmpAttribute : AttributeID;
      TmpStatus : SchemaStatus;
   begin

      -- Find the attribute in the schema
      FindAttribute (Schema,
                     SchemaState.WorkingTag.TID,
                     Name,
                     ATString,
                     TmpAttribute,
                     TmpStatus);

      if not (TmpStatus = SSOK) then
         Status := TmpStatus;
      else
         AddWorkingAttribute (SchemaState,
                              TmpAttribute,
                              Value,
                              Status);
      end if;
   end AddAttributeStr;



   -- Removes all spaces from an ExaminerString
   -- In XML, a non-string attribute cannot contain spaces.
   function StripString (Str : in EStrings.T) return EStrings.T
   is
      Ch : Character;
      OldString : EStrings.T;
      NewString : EStrings.T := EStrings.Empty_String;
      Success : Boolean := True;
   begin

      OldString := Str;

      while (EStrings.Get_Length (E_Str => OldString) > 0) and Success loop
         EStrings.Pop_Char (E_Str => OldString,
                            Char  => Ch);
         if not (Ch = ' ') then
            EStrings.Append_Char (E_Str   => NewString,
                                  Ch      => Ch,
                                  Success => Success);
         end if;
      end loop;

      if not Success then
         NewString := EStrings.Empty_String;
      end if;

      return NewString;

   end StripString;





   procedure AddAttributeInt (Schema      : in     SchemaRecord;
                              SchemaState : in out SchemaStateRecord;
                              Name        : in     EStrings.T;
                              Value       : in     Integer;
                              Status      :    out SchemaStatus)
   is

      TmpAttribute : AttributeID;
      TmpStatus    : SchemaStatus;

      subtype StringLength is Integer range 1 .. 10;
      subtype TempString is String (StringLength);
      TmpString : TempString;

   begin

      -- Find the attribute in the schema
      FindAttribute (Schema,
                     SchemaState.WorkingTag.TID,
                     Name,
                     ATInteger,
                     TmpAttribute,
                     TmpStatus);

      if not (TmpStatus = SSOK) then
         Status := TmpStatus;
      else
         SPARK_IO.Put_Int_To_String (TmpString,
                                     Value,
                                     1,
                                     10);

         AddWorkingAttribute (SchemaState,
                              TmpAttribute,
                              StripString (EStrings.Copy_String (Str => TmpString)),
                              Status);
      end if;

   end AddAttributeInt;




   function OutputAttributes (Schema      : in SchemaRecord;
                              SchemaState : in SchemaStateRecord) return EStrings.T
   is

      TempString : EStrings.T := EStrings.Empty_String;

   begin

      for I in TagAttributeArrayIndex loop

         if not IsNullAttribute (GetWorkingAttributeID (SchemaState, I)) then
            EStrings.Append_String (E_Str => TempString,
                                    Str   => " ");

            EStrings.Append_Examiner_String
              (E_Str1 => TempString,
               E_Str2 => GetAttributeName (Schema,
                                           SchemaState.WorkingTag.Attribs (I).AID));

            EStrings.Append_String (E_Str => TempString,
                                    Str   => "=""");
            EStrings.Append_Examiner_String (E_Str1 => TempString,
                                             E_Str2 => GetWorkingAttributeVal (SchemaState,
                                                                               I));
            EStrings.Append_String (E_Str => TempString,
                                    Str   => """");
         end if;

         exit when IsNullAttribute (GetWorkingAttributeID (SchemaState, I));

      end loop;

      return TempString;

   end OutputAttributes;


   procedure OutputOpeningTag (Schema      : in     SchemaRecord;
                               SchemaState : in out SchemaStateRecord;
                               XML         :    out EStrings.T;
                               Depth       :    out TagDepth;
                               Status      :    out SchemaStatus)
   is
      TempString : EStrings.T := EStrings.Empty_String;
      IndentTempString : EStrings.T := EStrings.Empty_String;
      TmpDepth : TagDepth := TagDepth'First;
      TmpStatus : SchemaStatus;
      Success : Boolean;
   begin

      if ValidWorkingTag (Schema, SchemaState) then

         -- Start with "<tagname "

         EStrings.Append_String (E_Str => TempString,
                                 Str   => "<");
         EStrings.Append_Examiner_String (E_Str1 => TempString,
                                          E_Str2 => GetTagName (Schema, SchemaState.WorkingTag.TID));

         -- Add the attributes

         EStrings.Append_Examiner_String (E_Str1 => TempString,
                                          E_Str2 => OutputAttributes (Schema, SchemaState));

         -- End the opening Tag
         EStrings.Append_String (E_Str => TempString,
                                 Str   => ">");

         -- Push the tag onto the top of the tagstack;
         TagStackPush (SchemaState,
                       TmpDepth,
                       TmpStatus);

         XML := EStrings.Empty_String;
         if (TmpStatus = SSStackFull) then -- The stack is full
            Depth := TmpDepth;
            Status := TmpStatus;
         else
            Status := SSOK;
            Depth := TmpDepth;
            --# accept F, 10, Success, "Ineffective assignment here OK";
            EStrings.Append_Char (E_Str   => IndentTempString,
                                  Ch      => Ada.Characters.Latin_1.LF,
                                  Success => Success);
            --# end accept;
            for I in TagDepth range 2 .. TmpDepth loop
               --# accept F, 10, Success, "Ineffective assignment here OK";
               EStrings.Append_Char (E_Str   => IndentTempString,
                                     Ch      => ' ',
                                     Success => Success); -- Put in some indentation
               --# end accept;
            end loop;
            EStrings.Append_Examiner_String (E_Str1 => IndentTempString,
                                             E_Str2 => TempString);
            XML := IndentTempString;
         end if;

      else
         Status := SSTagIncomplete;
         Depth := TmpDepth;
         XML := EStrings.Empty_String;
      end if;
      --# accept F, 33, Success, "Expect Success unused";
   end OutputOpeningTag;



   function ClosingTagString (Schema : in SchemaRecord;
                              SchemaState : in SchemaStateRecord;
                              TID    : in TagID) return EStrings.T
   is
      TmpString : EStrings.T := EStrings.Empty_String;
      Success   : Boolean;
   begin
      --# accept F, 10, Success, "Ineffective assignment here OK";
      EStrings.Append_Char (E_Str   => TmpString,
                            Ch      => Ada.Characters.Latin_1.LF,
                            Success => Success);
      --# end accept;
      for I in TagDepth range 2 .. TagStackDepth (SchemaState) + 1 loop
         --# accept F, 10, Success, "Ineffective assignment here OK";
         EStrings.Append_Char (E_Str   => TmpString,
                               Ch      => ' ',
                               Success => Success); -- Put in some indentation
         --# end accept;
      end loop;
      EStrings.Append_String (E_Str => TmpString,
                              Str   => "</");
      EStrings.Append_Examiner_String (E_Str1 => TmpString,
                                       E_Str2 => GetTagName (Schema, TID));
      EStrings.Append_String (E_Str => TmpString,
                              Str   => ">");
      --# accept F, 33, Success, "Expect Success unused";
      return TmpString;
   end ClosingTagString;


   -- Closing tags
   procedure CloseTag (Schema      : in     SchemaRecord;
                       SchemaState : in out SchemaStateRecord;
                       Depth       : in     TagDepth;
                       XML         :    out EStrings.T;
                       Status      :    out SchemaStatus)
   is

      TmpString : EStrings.T := EStrings.Empty_String;
      ClosingTag : TagID;
      TmpStatus : SchemaStatus := SSOK;
      ClosingTagStr : EStrings.T;

   begin

      if TagStackEmpty (SchemaState) then  -- Everything is already closed

         Status := SSStackEmpty;
         XML := TmpString;

      elsif (Depth > TagStackDepth (SchemaState)) -- Whoops, trying to close an unopened tag
        or (Depth = TagDepth'First) then  -- or the empty tag.

         Status := SSInvalidDepth;
         XML := TmpString;

      else

         while (Depth <= TagStackDepth (SchemaState)) loop

            TagStackPop (SchemaState, ClosingTag, TmpStatus);

            exit when not (TmpStatus = SSOK);

            ClosingTagStr := ClosingTagString (Schema, SchemaState, ClosingTag);
            EStrings.Append_Examiner_String (E_Str1 => TmpString,
                                             E_Str2 => ClosingTagStr);

         end loop;

         Status := TmpStatus;
         XML := TmpString;

      end if;

   end CloseTag;


   -- Close the lowest tag in the stack that matches TID
   procedure CloseTagByID (Schema      : in     SchemaRecord;
                           SchemaState : in out SchemaStateRecord;
                           TID         : in     TagID;
                           XML         :    out EStrings.T;
                           Status      :    out SchemaStatus)
   is

      Depth : TagDepth;

   begin

      Depth := TagStackHuntUp (SchemaState, TID);

      if Depth = TagDepth'First then  -- Not found

         Status := SSTagNotFound;
         XML := EStrings.Empty_String;

      else

         CloseTag (Schema, SchemaState, Depth, XML, Status);

      end if;

   end CloseTagByID;


      -- Close the lowest tag in the stack that matches TID
   procedure CloseTopTagByID (Schema      : in     SchemaRecord;
                              SchemaState : in out SchemaStateRecord;
                              TID         : in     TagID;
                              XML         :    out EStrings.T;
                              Status      :    out SchemaStatus)
   is

      Depth : TagDepth;

   begin

      Depth := TagStackHuntDown (SchemaState, TID);

      if Depth = TagDepth'First then  -- Not found

         Status := SSTagNotFound;
         XML := EStrings.Empty_String;

      else

         CloseTag (Schema, SchemaState, Depth, XML, Status);

      end if;

   end CloseTopTagByID;



   -- Close the lowest tag in the stack that matches TID
   procedure CloseTagByName (Schema      : in     SchemaRecord;
                             SchemaState : in out SchemaStateRecord;
                             Name        : in     EStrings.T;
                             XML         :    out EStrings.T;
                             Status      :    out SchemaStatus)
   is

      TID : TagID;

   begin

      TID := FindTag (Schema, Name);

      if IsNullTag (TID) then
         Status := SSNoSuchTag;
         XML := EStrings.Empty_String;
      else
         CloseTagByID (Schema, SchemaState, TID, XML, Status);
      end if;

   end CloseTagByName;

end SPARK_XML;
