-- $Id: sem.adb 15946 2010-02-05 13:52:44Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with ErrorHandler,
     ExaminerConstants,
     EStrings,
     ELStrings,
     FlowAnalyser,
     RelationAlgebra,
     SystemErrors,
     SimpleLists,
     CompleteCheck,
     LexTokenManager,
     SPSymbols,
     Lists,
     Heap,
     SeqAlgebra,
     RefList,
     Dictionary,
     Maths,
     ComponentManager,
     Debug,
     SPARK_IO;

use type SPSymbols.SPSymbol;
use type LexTokenManager.Str_Comp_Result;
use type LexTokenManager.Lex_String;
use type Dictionary.Contexts;
use type Dictionary.Symbol;
use type Dictionary.Iterator;
use type Dictionary.Modes;
use type Dictionary.Scopes;
use type Dictionary.Abstractions;
use type Dictionary.PrefixSort;
use type Dictionary.KindsOfOp;
use type ErrorHandler.ErrorLevel;
use type ErrorHandler.JustificationKinds;
use type ErrorHandler.JustificationIdentifier;
use type Maths.ErrorCode;
use type Maths.Value;
use type CompleteCheck.TypRangeState;
use type CompleteCheck.TypCompleteState;
use type CompleteCheck.TypOverlapState;

package body Sem
is
   -- Operator renames


   -- Long subprogram prefix renames
   function Child_Node (CurrentNode : STree.SyntaxNode) return STree.SyntaxNode
      renames STree.Child_Node;

   function Next_Sibling (CurrentNode : STree.SyntaxNode) return STree.SyntaxNode
         renames STree.Next_Sibling;

   function ParentNode (CurrentNode : STree.SyntaxNode) return STree.SyntaxNode
      renames STree.ParentNode;

   function FindFirstNode (NodeKind    : SPSymbols.SPSymbol;
                           FromRoot    : STree.SyntaxNode;
                           InDirection : STree.TraverseDirection)
                          return STree.Iterator
     renames STree.FindFirstNode;

   function FindFirstBranchNode (FromRoot    : STree.SyntaxNode;
                                 InDirection : STree.TraverseDirection)
                          return STree.Iterator
     renames STree.FindFirstBranchNode;

   function GetNode (It : STree.Iterator) return STree.SyntaxNode
     renames STree.GetNode;

   function SyntaxNodeType (Node : STree.SyntaxNode) return SPSymbols.SPSymbol
      renames STree.SyntaxNodeType;

   function NodePosition (Node : STree.SyntaxNode) return
      LexTokenManager.Token_Position
      renames STree.NodePosition;

   function NodeLexString (Node : STree.SyntaxNode) return LexTokenManager.Lex_String
      renames STree.NodeLexString;

   -- Function returns the left most leaf node of the tree.
   function LastChildOf (StartNode : STree.SyntaxNode) return
      STree.SyntaxNode
      renames STree.LastChildOf;

   function LastSiblingOf (StartNode : STree.SyntaxNode) return
      STree.SyntaxNode
      renames STree.LastSiblingOf;

   ----------------------------------------------------------------------------

   type ExpRecordSort is (TypeResult, -- should be this anywhere above primary
                          IsUnknown,
                          IsParameterName, -- used in named association checks
                          IsPackage,
                          IsObject,
                          IsFunction,
                          IsTypeMark);

   type ExpRecord is
      record
         TypeSymbol            : Dictionary.Symbol;
         OtherSymbol           : Dictionary.Symbol;
         StreamSymbol          : Dictionary.Symbol;
         TaggedParameterSymbol : Dictionary.Symbol;
         VariableSymbol        : Dictionary.Symbol;
         ParamCount            : Natural;
         ParamList             : Lists.List;
         Sort                  : ExpRecordSort;
         ArgListFound          : Boolean;
         IsAVariable           : Boolean;
         IsAnEntireVariable    : Boolean;
         ErrorsInExpression    : Boolean;
         HasOperators          : Boolean;

         IsStatic,            -- flags meaningless unless Sort=TypeResult
         IsConstant,
         IsARange              : Boolean;

         Value                 : Maths.Value;
         RangeRHS              : Maths.Value;
      end record;


   NullExpRecord : constant ExpRecord :=
     ExpRecord'(TypeSymbol            => Dictionary.NullSymbol,
                OtherSymbol           => Dictionary.NullSymbol,
                StreamSymbol          => Dictionary.NullSymbol,
                TaggedParameterSymbol => Dictionary.NullSymbol,
                VariableSymbol        => Dictionary.NullSymbol,
                ParamCount            => 0,
                ParamList             => Lists.Null_List,
                Sort                  => IsUnknown,
                ArgListFound          => False,
                IsAVariable           => False,
                IsAnEntireVariable    => False,
                ErrorsInExpression    => False,
                HasOperators          => False,
                IsStatic              => False,
                IsConstant            => False,
                IsARange              => False,
                Value                 => Maths.NoValue,
                RangeRHS              => Maths.NoValue);


   type TypCaseFlags is
      record
         CheckCompleteness : Boolean;
         SignalOutOfRange  : Boolean;
         OutOfRangeSeen    : Boolean;
         CheckOverlap      : Boolean;
         WarnNoOthers      : Boolean;
         OthersMandatory   : Boolean;
      end record;

   NullCaseFlags : constant TypCaseFlags :=
     TypCaseFlags'(False, False, False, False, False, False);

   type TypTypeBound is
      record
         IsDefined : Boolean;
         Value     : Integer;
      end record;

   UnknownTypeBound : constant TypTypeBound := TypTypeBound'(IsDefined => False,
                                                             Value     => 0);


   type TypAggAssociationType is (AggregateIsPositional,
                                  AggregateIsNamed,
                                  AggregateIsLoneOthers);

   type TypAggFlags is
      record
         CheckCompleteness      : Boolean;
         WarnNoOthers           : Boolean;
         CheckOverlap           : Boolean;
         SignalOutOfRange       : Boolean;
         OutOfRangeSeen         : Boolean;
         MoreEntriesThanNatural : Boolean;
         HasOthersPart          : Boolean;
         AssociationType        : TypAggAssociationType;
      end record;

   NullTypAggFlags : constant TypAggFlags :=
     TypAggFlags'(False, False, False, False, False, False, False, TypAggAssociationType'First);

   ----------------------- Subprograms ---------------------------

   -- Put_ExpRecord is handy for debugging expression walking, but
   -- is uncalled in production builds.
   procedure Put_ExpRecord (R : in ExpRecord)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                R;
   is
      --# hide Put_ExpRecord;
      F : SPARK_IO.File_Type;
   begin
      F := SPARK_IO.Standard_Output;
      SPARK_IO.Put_String (F, "Sort                 => ", 0);
      SPARK_IO.Put_Line (F, ExpRecordSort'Image (R.Sort), 0);

      SPARK_IO.Put_String (F, "TypeSymbol           => ", 0);
      EStrings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String
           (Lex_Str => Dictionary.GetSimpleName (R.TypeSymbol)));

      SPARK_IO.Put_String (F, "OtherSymbol          => ", 0);
      EStrings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String
           (Lex_Str => Dictionary.GetSimpleName (R.OtherSymbol)));

      SPARK_IO.Put_String (F, "Value                => ", 0);
      ELStrings.Put_Line (File  => F,
                          E_Str => Maths.ValueToString (R.Value));

      SPARK_IO.Put_String (F, "ArgListFound         => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.ArgListFound), 0);

      SPARK_IO.Put_String (F, "RangeRHS             => ", 0);
      ELStrings.Put_Line (File  => F,
                          E_Str => Maths.ValueToString (R.RangeRHS));

      SPARK_IO.Put_String (F, "ParamCount           =>", 0);
      SPARK_IO.Put_Line   (F, Natural'Image (R.ParamCount), 0);

      SPARK_IO.Put_Line   (F, "ParamList            => (...not printed...)", 0);

      SPARK_IO.Put_String (F, "IsStatic             => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.IsStatic), 0);
      SPARK_IO.Put_String (F, "IsConstant           => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.IsConstant), 0);
      SPARK_IO.Put_String (F, "IsARange             => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.IsARange), 0);

      SPARK_IO.Put_String (F, "VariableSymbol       => ", 0);
      EStrings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String
           (Lex_Str => Dictionary.GetSimpleName (R.VariableSymbol)));
      -- Generates a different string, can be useful in debugging:
      -- SPARK_IO.Put_String (F, "                        ", 0);
      -- Dictionary.GenerateSimpleName (R.VariableSymbol, ".", S);
      -- EStrings.PutLine (F, S);

      SPARK_IO.Put_String (F, "IsAVariable          => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.IsAVariable), 0);
      SPARK_IO.Put_String (F, "IsAnEntireVariable   => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.IsAnEntireVariable), 0);
      SPARK_IO.Put_String (F, "ErrorsInExpression   => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.ErrorsInExpression), 0);
      SPARK_IO.Put_String (F, "HasOperators         => ", 0);
      SPARK_IO.Put_Line   (F, Boolean'Image (R.HasOperators), 0);

      SPARK_IO.Put_String (F, "StreamSymbol         => ", 0);
      EStrings.Put_Line
        (File  => F,
         E_Str => LexTokenManager.Lex_String_To_String
           (Lex_Str => Dictionary.GetSimpleName (R.StreamSymbol)));
   end Put_ExpRecord;

   ---------------------------------------------------------------------

   procedure GetTypeBounds (TypeSymbol   : in     Dictionary.Symbol;
                            LowerBound,
                            UpperBound   :    out TypTypeBound)
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --# derives LowerBound,
   --#         UpperBound from Dictionary.Dict,
   --#                         LexTokenManager.State,
   --#                         TypeSymbol;
   is
      MathsError : Maths.ErrorCode;
      BoundVal   : Integer;
   begin
      if Dictionary.IsBooleanTypeMark (TypeSymbol) then
         LowerBound := TypTypeBound'(IsDefined => True,
                                     Value     => 0);
         UpperBound := TypTypeBound'(IsDefined => True,
                                     Value     => 1);
      else
         Maths.ValueToInteger
            (Maths.ValueRep
             (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                  LexTokenManager.First_Token,
                                                  TypeSymbol)),
             BoundVal,     -- to get
             MathsError);  -- to get

         LowerBound := TypTypeBound'(IsDefined => (MathsError = Maths.NoError),
                                     Value     => BoundVal);

         Maths.ValueToInteger
            (Maths.ValueRep
             (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                  LexTokenManager.Last_Token,
                                                  TypeSymbol)),
             BoundVal,     -- to get
             MathsError);  -- to get

         UpperBound := TypTypeBound'(IsDefined => (MathsError = Maths.NoError),
                                     Value     => BoundVal);
      end if;
   end GetTypeBounds;

   ---------------------------------------------------------------------

   procedure CheckPackagePrefix (Node    : in     STree.SyntaxNode;
                                 PackSym : in     Dictionary.Symbol;
                                 Scope   : in     Dictionary.Scopes;
                                 Ok      :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Ok                        from Dictionary.Dict,
   --#                                        PackSym,
   --#                                        Scope;
   is
   begin
      if Dictionary.PrefixAllowed (PackSym, Scope) then
         Ok := True;
         if Dictionary.IsGenericPackage (PackSym) then
            Ok := False;
            ErrorHandler.SemanticError (655, ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (PackSym));
         end if;
      else
         Ok := False;
         ErrorHandler.SemanticError (337, ErrorHandler.NoReference,
                                    NodePosition (Node),
                                    Dictionary.GetSimpleName (PackSym));
      end if;
   end CheckPackagePrefix;
   ------------------------------------------------------------------

   function InPackageInitialization (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsLocalScope (Scope) and then
         Dictionary.IsPackage (Dictionary.GetEnclosingCompilationUnit (Scope));
   end InPackageInitialization;

   ----------------------------------------------------------------

   function IndexesMatch (Target,
                          Source   : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   -------------------------------------------------------------------

   function IsExternalInterface (PragmaNode : STree.SyntaxNode) return Boolean
   --# global in CommandLineData.Content;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
      is separate;
   -------------------------------------------------------------------

   -- Exported subprogram
   procedure CompUnit (TopNode : in STree.SyntaxNode;
                       Do_VCG  : in Boolean) is separate;

end Sem;
