-- $Id: sem-compunit.adb 16567 2010-03-25 16:09:40Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with ContextManager.Ops;
with VCG;

separate (Sem)
procedure CompUnit (TopNode : in STree.SyntaxNode;
                    Do_VCG  : in Boolean)
is
   type Annotation_Symbol_Table is array (Boolean) of
      SPSymbols.SPSymbol;

   type Visibility_Error_Hint is
      (No_Hint, In_Global_List, In_Derives_Import_List, In_Derives_Export_List, In_Suspends_List);

   type Tilde_Context is (Code, Precondition, Postcondition, FunctionReturn);
   subtype Anno_Tilde_Context is Tilde_Context range Precondition .. FunctionReturn;

   SemanticErrorInSubprogramBody : Boolean;
   DataFlowErrorInSubprogramBody : Boolean;
   UnusedDataFlowErrorFlag       : Boolean;

   type Hidden_Class is (Not_Hidden, Handler_Hidden, All_Hidden);

   -------------------------------------------------------
   --# inherit CommandLineData,
   --#         ExaminerConstants,
   --#         Sem,
   --#         SystemErrors;
   package ExpStack
   is

      type ExpStackType is private;

      procedure Init (Stack : out ExpStackType);
      --# derives Stack from ;

      procedure Push (X     : in     Sem.ExpRecord;
                      Stack : in out ExpStackType);
      --# global in CommandLineData.Content;
      --# derives Stack from *,
      --#                    X &
      --#         null  from CommandLineData.Content;

      procedure Pop (Item  :    out Sem.ExpRecord;
                     Stack : in out ExpStackType);
      --# global in CommandLineData.Content;
      --# derives Item,
      --#         Stack from Stack &
      --#         null  from CommandLineData.Content;

      function Top (Stack : ExpStackType) return Sem.ExpRecord;

      function HasOneEntry (Stack : ExpStackType) return Boolean;

      function IsEmpty (Stack : ExpStackType) return Boolean;

   private

      subtype IndexRange is Integer range
         1 .. ExaminerConstants.WalkExpStackMax;
      type StackArray is array (IndexRange) of Sem.ExpRecord;
      subtype TopRange is Integer range
         0 .. ExaminerConstants.WalkExpStackMax;

      type ExpStackType is record
         S :   StackArray;
         TopPtr : TopRange;
      end record;

   end ExpStack;

   -------------------------------------------------------
   --# inherit Dictionary,
   --#         ExaminerConstants,
   --#         SPARK_IO,
   --#         SystemErrors;
   package TypeContextStack
   is

      type TStackType is private;

      procedure Init (Stack : out TStackType);
      --# derives Stack from ;

      procedure Push (X     : in     Dictionary.Symbol;
                      Stack : in out TStackType);
      --# derives Stack from *,
      --#                    X;

      procedure Pop (Stack : in out TStackType);
      --# derives Stack from *;

      function Top (Stack : in TStackType) return Dictionary.Symbol;

      function HasOneEntry (Stack : in TStackType) return Boolean;

   private

      subtype IndexRange is Integer range
         1 .. ExaminerConstants.WalkExpStackMax;
      type StackArray is array (IndexRange) of Dictionary.Symbol;
      subtype TopRange is Integer range
         0 .. ExaminerConstants.WalkExpStackMax;

      type TStackType is record
         S      : StackArray;
         TopPtr : TopRange;
      end record;

   end TypeContextStack;

   --------------------------------------------

   --# inherit CompleteCheck,
   --#         Dictionary,
   --#         ExaminerConstants,
   --#         Sem,
   --#         SPARK_IO,
   --#         SystemErrors;
   package AggregateStack
      --# own State;
   is
      procedure Init;
      --# global out State;
      --# derives State from ;

      procedure Push (TypeSym     : in Dictionary.Symbol;
                      LowerBound  : in Sem.TypTypeBound;
                      UpperBound  : in Sem.TypTypeBound;
                      AggFlags    : in Sem.TypAggFlags;
                      Counter     : in Natural;
                      CompleteRec : in CompleteCheck.T);
      --# global in out State;
      --# derives State from *,
      --#                    AggFlags,
      --#                    CompleteRec,
      --#                    Counter,
      --#                    LowerBound,
      --#                    TypeSym,
      --#                    UpperBound;

      procedure Pop (TypeSym     : out Dictionary.Symbol;
                     LowerBound  : out Sem.TypTypeBound;
                     UpperBound  : out Sem.TypTypeBound;
                     AggFlags    : out Sem.TypAggFlags;
                     Counter     : out Natural;
                     CompleteRec : out CompleteCheck.T);
      --# global in out State;
      --# derives AggFlags,
      --#         CompleteRec,
      --#         Counter,
      --#         LowerBound,
      --#         State,
      --#         TypeSym,
      --#         UpperBound  from State;

      -- In the analysis of array aggregates, the expression walker
      -- needs to know the expected type of the array index for the
      -- current aggregate.  To make this work for multi-dimensional
      -- aggregates, this has to be stored in the Aggregate Stack, and
      -- is accessed with the following function.
      function TopTypeSym return Dictionary.Symbol;
      --# global in State;
   end AggregateStack;
   -----------------------------------------------------------------

   --# inherit ExaminerConstants,
   --#         SPARK_IO,
   --#         SystemErrors;
   package Stack
      --# own State;
   is

      procedure Init;
      --# global out State;
      --# derives State from ;

      procedure Push (X : in Boolean);
      --# global in out State;
      --# derives State from *,
      --#                    X;

      procedure Pop; -- (Item : out Boolean);
      --# global in out State;
      --# derives State from *;

      -- Note: this is not a proper function as it has a hidden
      --       side effect if a system error is raised
      function Top return Boolean;
      --# global in State;
   end Stack;

   --------------- Procedure variables -----------------------------

   StmtNode,
   LastNode,
   NextNode            : STree.SyntaxNode;
   NodeType            : SPSymbols.SPSymbol;
   ErrStatus           : ErrorHandler.ErrorLevel;
   CurrentScope        : Dictionary.Scopes;
   TheHeap             : Heap.HeapRecord;
   NodeTable           : RefList.HashTable;
   GlobalComponentData : ComponentManager.ComponentData;
   Unused              : Dictionary.Symbol;

   -------------- Package bodies ------------------------------

   package body Stack
      is separate;

   package body ExpStack
      is separate;

   package body TypeContextStack
      is separate;

   package body AggregateStack
      is separate;

   --------------------------------------------
   --Routines for handling list of names
   --------------------------------------------

   procedure CreateNameList (List      :    out Lists.List;
                             HeapParam : in out Lists.List_Heap)
   --# derives HeapParam,
   --#         List      from HeapParam;
   is
      Ok : Boolean;

   begin
      Lists.New_List (Heap     => HeapParam,
                      The_List => List,
                      Ok       => Ok);
      if not Ok then
         SystemErrors.FatalError (SystemErrors.ListOverflowInExpression, "in CreateNameList");
      end if;
   end CreateNameList;

   --------------------------------------------

   procedure DisposeOfNameList (List      : in out Lists.List;
                                HeapParam : in out Lists.List_Heap)
   --# global in LexTokenManager.State;
   --# derives HeapParam from *,
   --#                        LexTokenManager.State,
   --#                        List &
   --#         List      from ;
   is
   begin
      Lists.Delete_List (Heap     => HeapParam,
                         The_List => List);
   end DisposeOfNameList;

   --------------------------------------------

   procedure AddName (Name      : in     LexTokenManager.Lex_String;
                      List      : in     Lists.List;
                      HeapParam : in out Lists.List_Heap;
                      Present   :    out Boolean)
   --# global in LexTokenManager.State;
   --# derives HeapParam,
   --#         Present   from HeapParam,
   --#                        LexTokenManager.State,
   --#                        List,
   --#                        Name;
   is
      Ok : Boolean;
   begin
      Lists.Add_Name (Heap            => HeapParam,
                      The_List        => List,
                      Name            => Name,
                      Already_Present => Present,
                      Ok              => Ok);
      if not Ok then
         SystemErrors.FatalError (SystemErrors.ListOverflowInExpression, "in AddName");
      end if;
   end AddName;

   ---------------- Embedded subprograms ----------------------
   ----------------------------------------------------------------------
   ----------------------------------------------------------------------
   -- The following procedures are used to handle view changes and other
   -- things associated with tagged types and inherited subprogram calls.
   -- They are declared here because they are used in wf_proc_call,
   -- wf_positional_argument_association, wf_named_argument_association,
   -- and wf_primary.

   function ConvertTaggedActual (Actual,
                                 TaggedParameterSym : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   --  Function is only called if we know substitution is required
   is
      ActualLocal : Dictionary.Symbol;
   begin
      ActualLocal := Actual;
      loop
         -- normal exit: successful conversion
         exit when Dictionary.GetType (ActualLocal) = TaggedParameterSym;
         -- abnormal exit: can't follow inherited field pointers because type is hidden
         exit when Dictionary.GetFirstRecordSubcomponent (ActualLocal) = Dictionary.NullSymbol;
         ActualLocal := Dictionary.GetFirstRecordSubcomponent (ActualLocal);
      end loop;
      return ActualLocal;
   end ConvertTaggedActual;

   ---------------------------------------------------------------

   procedure AddRecordSubComponents (RecordVarSym  : in     Dictionary.Symbol;
                                     RecordTypeSym : in     Dictionary.Symbol;
                                     ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    ComponentData,
   --#                                    ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    RecordTypeSym,
   --#                                    RecordVarSym,
   --#                                    TheHeap &
   --#         null                  from CommandLineData.Content;
   is
      TypeComponent       : Dictionary.Symbol;
      VarComponent        : Dictionary.Symbol;

      PrefixComponentNode : ComponentManager.Component;
      CurrentRecordVar    : Dictionary.Symbol;
      CurrentRecordType   : Dictionary.Symbol;

   begin
      CurrentRecordVar := RecordVarSym;

      CurrentRecordType := RecordTypeSym;
      if Dictionary.IsSubtype (CurrentRecordType) then
         CurrentRecordType := Dictionary.GetRootType (CurrentRecordType);
      end if;

      PrefixComponentNode := ComponentManager.GetComponentNode
        (ComponentData,
         CurrentRecordVar);
      -- for non tagged records the outer loop only runs once.  For extended tagged
      -- records we loop through all the inherited fields and add them too

      -- A private extended tagged type where the private part is hidden
      -- might result in PrefixComponentNode being Null here, in which
      -- case nothing can be done.
      loop
         for I in Natural range 1 ..
           Dictionary.GetNumberOfNonExtendedComponents (CurrentRecordType)
         loop
            TypeComponent := Dictionary.GetNonExtendedRecordComponent (CurrentRecordType, I);
            Dictionary.AddRecordSubcomponent (Prefix       => CurrentRecordVar,
                                              Component    => TypeComponent,
                                              Comp_Unit    => ContextManager.Ops.CurrentUnit,
                                              -- to get
                                              Subcomponent => VarComponent);
            PrefixComponentNode := ComponentManager.GetComponentNode
              (ComponentData,
               CurrentRecordVar);
            ComponentManager.AddNextChild (ComponentData,
                                           TheHeap,
                                           PrefixComponentNode,
                                           VarComponent);
         end loop;
         CurrentRecordType := Dictionary.GetRootOfExtendedType (CurrentRecordType);
         -- stop when there are no more inherited fields
         exit when CurrentRecordType = Dictionary.NullSymbol;
         -- if the record has no components (perhaps it is a private type and the private
         -- part is hidden) then we cannot add any subcomponents
         exit when Dictionary.GetNumberOfNonExtendedComponents (CurrentRecordType) = 0;

         -- Simillarly, if the PrefixComponentNode is null, then we
         -- can't find its FirstChild, so we must be done.
         exit when ComponentManager.IsNullComponent (PrefixComponentNode);
         CurrentRecordVar := ComponentManager.GetName
           (ComponentData, ComponentManager.GetFirstChild
              (ComponentData, PrefixComponentNode));
      end loop;

         ComponentManager.DumpAllComponentTrees (ComponentData);
   end AddRecordSubComponents;

   ---------------------------------------------------------------

   procedure TaggedActualMustBeObjectCheck (NodePos         : in     LexTokenManager.Token_Position;
                                            FormalType      : in     Dictionary.Symbol;
                                            ActualType      : in     Dictionary.Symbol;
                                            ControllingType : in     Dictionary.Symbol;
                                            IsAVariable     : in     Boolean;
                                            IsAConstant     : in     Boolean;
                                            ErrorFound      : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorFound                from *,
   --#                                        ActualType,
   --#                                        ControllingType,
   --#                                        FormalType,
   --#                                        IsAConstant,
   --#                                        IsAVariable &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from ActualType,
   --#                                        CommandLineData.Content,
   --#                                        ControllingType,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FormalType,
   --#                                        IsAConstant,
   --#                                        IsAVariable,
   --#                                        LexTokenManager.State,
   --#                                        NodePos,
   --#                                        SPARK_IO.FILE_SYS;
   is
   begin
      if (ControllingType /= Dictionary.NullSymbol) and then
        (FormalType = ControllingType) and then
        (ActualType /= ControllingType) and then -- but must be an extension of it (earlier check)
        not (IsAVariable or IsAConstant) then
         ErrorFound := True;
         ErrorHandler.SemanticError (827,
                                     ErrorHandler.NoReference,
                                     NodePos,
                                     LexTokenManager.Null_String);
      end if;
   end TaggedActualMustBeObjectCheck;

   -----------------------------------------------------------------------------------
   -- CreateImplicitPositiveSubtype extracted from PlantConstrainingType so that it can be
   -- called directly from wf_constant_declaration where it is used to created an implcit
   -- subtype for string constants.
   procedure CreateImplicitPositiveSubtype (StringLength    : in     Maths.Value;
                                            Location        : in     Dictionary.Location;
                                            IndexConstraint :    out Dictionary.Symbol)
   --# global in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    StringLength &
   --#         IndexConstraint       from ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    StringLength &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    StringLength;
   is
      ConstraintStr        : EStrings.T;
      ConstraintLexStr     : LexTokenManager.Lex_String;
      UpperBoundLexStr     : LexTokenManager.Lex_String;
      IndexConstraintLocal : Dictionary.Symbol;
   begin
      -- Create an ExaminerString of the form "Positive__n" where n is the string length
      ConstraintStr := EStrings.Copy_String (Str => "Positive__");
      -- The value of "n" will not exceed a size that can be printed within an ExaminerString
      -- so the conversion will not truncate here.
      EStrings.Append_Examiner_String
        (E_Str1 => ConstraintStr,
         E_Str2 => ELStrings.ToExaminerString (Maths.ValueToString (StringLength)));
      -- Insert this name into the string table; either we add it an get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      LexTokenManager.Insert_Examiner_String (Str     => ConstraintStr,
                                              Lex_Str => ConstraintLexStr);
      -- Look up type in Dictionary in case it has previously been added
      -- Note that we put these implicit subtype in Standard (where Positive itself lives)
      -- and that we declare them in proof rather than Ada context
      IndexConstraintLocal := Dictionary.LookupItem (Name    => ConstraintLexStr,
                                                     Scope   => Dictionary.VisibleScope
                                                       (Dictionary.GetPredefinedPackageStandard),
                                                     Context => Dictionary.ProofContext);
      if IndexConstraintLocal = Dictionary.NullSymbol then
         -- not previously added, so we add a new subtype here
         Maths.StorageRep (StringLength, UpperBoundLexStr);
         Dictionary.AddIntegerSubtype (Name            => ConstraintLexStr,
                                       Static          => True,
                                       Parent          => Dictionary.GetPredefinedIntegerType,
                                       ParentReference => Location,
                                       Lower           => LexTokenManager.One_Value,
                                       Upper           => UpperBoundLexStr,
                                       Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                       Declaration     => Location,
                                       Scope           => Dictionary.VisibleScope
                                         (Dictionary.GetPredefinedPackageStandard),
                                       Context         => Dictionary.ProofContext,
                                       Subtype_Symbol  => IndexConstraintLocal);
      end if;
      IndexConstraint := IndexConstraintLocal;
   end CreateImplicitPositiveSubtype;

   -----------------------------------------------------------------------------------
   procedure CreateImplicitStringSubtype (StringLength     : in     Maths.Value;
                                          Location         : in     Dictionary.Location;
                                          TheStringSubtype :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    CommandLineData.Content,
   --#                                    ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    StringLength &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    StringLength &
   --#         TheStringSubtype      from ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    StringLength;
   is
      StringSubtypeStr      : EStrings.T;
      StringSubtypeLexStr   : LexTokenManager.Lex_String;
      IndexConstraint       : Dictionary.Symbol;
      TheStringSubtypeLocal : Dictionary.Symbol;
   begin
      -- First create an index subtype of the form positive__n
      CreateImplicitPositiveSubtype (StringLength,
                                     Location,
                                       -- to get
                                     IndexConstraint);
      -- Create an ExaminerString of the form "String__n" where n is the string length
      StringSubtypeStr := EStrings.Copy_String (Str => "String__");
      -- The value of "n" will not exceed a size that can be printed within an ExaminerString
      -- so the conversion will not truncate here.
      EStrings.Append_Examiner_String
        (E_Str1 => StringSubtypeStr,
         E_Str2 => ELStrings.ToExaminerString (Maths.ValueToString (StringLength)));
      -- Insert this name into the string table; either we add it an get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      -- Insert this name into the string table; either we add it and get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      LexTokenManager.Insert_Examiner_String (Str     => StringSubtypeStr,
                                              Lex_Str => StringSubtypeLexStr);
      -- Look up type in Dictionary in case it has previously been added
      -- Note that we put these implicit subtype in Standard (where Positive and String themselves live)
      -- and that we declare them in proof rather than Ada context
      TheStringSubtypeLocal := Dictionary.LookupItem (Name    => StringSubtypeLexStr,
                                                      Scope   => Dictionary.VisibleScope
                                                        (Dictionary.GetPredefinedPackageStandard),
                                                      Context => Dictionary.ProofContext);
      if TheStringSubtypeLocal = Dictionary.NullSymbol then
         -- not previously added, so we add a new subtype here
         Dictionary.AddArraySubtype (Name            => StringSubtypeLexStr,
                                     Parent          => Dictionary.GetPredefinedStringType,
                                     ParentReference => Location,
                                     Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                     Declaration     => Location,
                                     Scope           => Dictionary.VisibleScope
                                       (Dictionary.GetPredefinedPackageStandard),
                                     Context         => Dictionary.ProofContext,
                                       -- In SPARK95 and 2005, the implicit string subtype is considered Static
                                     Static          => CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83,
                                       -- to get
                                     ArraySubtype    => TheStringSubtypeLocal);
         Dictionary.AddArrayIndex (TheArrayType       => TheStringSubtypeLocal,
                                   TheIndexType       => IndexConstraint,
                                   Comp_Unit          => ContextManager.Ops.CurrentUnit,
                                   IndexTypeReference => Location);

      end if;
      TheStringSubtype := TheStringSubtypeLocal;
   end CreateImplicitStringSubtype;

   -----------------------------------------------------------------------------------
   -- PlantConstrainingType declared here because it used by wf_proc_call,
   -- wf_position_argument_association and wf_named_argument_association

   procedure PlantConstrainingType (ExpressionType : in Dictionary.Symbol;
                                    StringLength   : in Maths.Value;
                                    ActualNode     : in STree.SyntaxNode)
   --# global in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.FILE_SYS,
   --#         STree.Table           from *,
   --#                                    ActualNode,
   --#                                    ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    ExpressionType,
   --#                                    LexTokenManager.State,
   --#                                    STree.Table,
   --#                                    StringLength &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpressionType,
   --#                                    StringLength;
   is
      -- This procedure is used to plant a type symbol in the syntax tree, at the location of
      -- an actual parameter, for use by the VCG.  IFF the ExpressionType is String
      -- (indicating that the actual parameter is a string literal) then the parameter StringLength
      -- is used to create (or obtain if it's already declared) a subtype of Positive called Positive__n
      -- where n is the string length; this implicitly-declared subtype is then used as the constraint.
      -- For anything other than a string literal actual parameter we
      -- plant the symbol of the constraining array subtype.  The array subtype effectively passes
      -- the symbols of /all/ the constraining indexes however many dimensions the array has.


      TypeToPlant : Dictionary.Symbol := Dictionary.NullSymbol;


   begin -- PlantConstrainingType
      if Dictionary.IsUnconstrainedArrayType (Dictionary.GetRootType (ExpressionType)) and then
        not Dictionary.IsUnconstrainedArrayType (ExpressionType) then

         -- Not a string, so plant the array type
         TypeToPlant := ExpressionType;
      elsif Dictionary.IsPredefinedStringType (ExpressionType) and
        StringLength /= Maths.NoValue then

         -- If the actual is a String Literal like "Hello World", then the ExpressionType
         -- will be Predefined String and StringLength will have a well-defined value.
         -- In this case, we create an implicit constraining subtype.
         CreateImplicitPositiveSubtype (StringLength,
                                        Dictionary.Location'(NodePosition (ActualNode),
                                                             NodePosition (ActualNode)),
                                          -- to get
                                        TypeToPlant);
      end if;
      STree.AddNodeSymbol (ActualNode, TypeToPlant);
   end PlantConstrainingType;

   ----------------------------------------------------------------------
   -- FindPreviousPackage
   --
   -- Given a initial_declarative_item_rep node, we sometimes need to find
   -- if there's a embedded package declaration in the tree rooted at that
   -- node.  This function returns the LexString of that package if it
   -- can be found, or NullString if it can't.
   --
   -- This function is used by wf_renaming_declararation and
   -- wf_use_type_declaration, both of which need to locate such
   -- packages, so this function appears here so it can be called
   -- by both subunits.
   ----------------------------------------------------------------------
   function FindPreviousPackage (Node : STree.SyntaxNode)
                                return LexTokenManager.Lex_String
   --# global in STree.Table;
   is
      LastNode : STree.SyntaxNode;
      PackIdent : LexTokenManager.Lex_String;
   begin
      -- Phase 1 - search down the tree rooted at Node for a
      -- basic_declarative_item node or a package_declaration node
      LastNode := Child_Node (Node);
      loop

         exit when SyntaxNodeType (LastNode) /= SPSymbols.initial_declarative_item_rep;

         if SyntaxNodeType (Next_Sibling (LastNode)) = SPSymbols.basic_declarative_item or
           SyntaxNodeType (Next_Sibling (LastNode)) = SPSymbols.package_declaration
         then
            -- If there's a basic_declarative_item or a package_declaration to the right
            -- of LastNode, then set LastNode to that node and exit.
            LastNode := Next_Sibling (LastNode);
         else
            -- No?  Then go down the tree and try again
            LastNode := Child_Node (LastNode);
         end if;
      end loop;

      -- Phase 2
      -- LastNode should be a basic_declarative_item or a package_declaration
      case SyntaxNodeType (LastNode) is
         when SPSymbols.basic_declarative_item =>
            -- No previous package, so return NullString
            PackIdent := LexTokenManager.Null_String;
         when SPSymbols.package_declaration =>

            -- If this package has an inherit clause, then skip over it
            LastNode := Child_Node (LastNode);
            if SyntaxNodeType (LastNode) = SPSymbols.inherit_clause then
               LastNode := Next_Sibling (LastNode);
            end if;

            -- SyntaxNodeType (LastNode) = package_specification

            -- Find and return the package's identifier node
            LastNode := LastChildOf (LastNode);
            PackIdent := NodeLexString (LastNode);
         when others =>
            PackIdent := LexTokenManager.Null_String; -- to avoid flow error
            SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree, "in WFF.CU.FindPreviousPackage");
      end case;
      return PackIdent;
   end FindPreviousPackage;

   ----------------------------------------------------------------------

   -- Returns whether a subprogram_implementation node has a fully hidden body,
   -- a hidden handler part, or no hiding at all.  This code depends on the grammar
   -- in SPARK.LLA section 6.3
   function Body_Hidden_Class (Node : in STree.SyntaxNode) return Hidden_Class
   --# global in STree.Table;
   is
      Result        : Hidden_Class;
      PragmaRepNode : STree.SyntaxNode;
      EndNode       : STree.SyntaxNode;
   begin
      SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.subprogram_implementation,
                             SystemErrors.PreconditionFailure,
                             "Compunit.Body_Hidden_Class - wrong node type");
      PragmaRepNode := Child_Node (Node);
      EndNode       := LastSiblingOf (PragmaRepNode);
      -- if the PragmaRep is immediately followed by hidden part, then the whole
      -- body must be hidden.  If the PragmaRep is followed by something else, then
      -- we _might_ have a hidden handler part at EndNode.
      if SyntaxNodeType (Next_Sibling (PragmaRepNode)) = SPSymbols.hidden_part then
         Result := All_Hidden;
      else
         if SyntaxNodeType (EndNode) = SPSymbols.hidden_part then
            Result := Handler_Hidden;
         else
            Result := Not_Hidden;
         end if;
      end if;
      return Result;
   end Body_Hidden_Class;

   ----------------------------------------------------------------------

   procedure wf_external_interface (Pragma_Node : in     STree.SyntaxNode;
                                    Entity_Sym  : in     Dictionary.Symbol;
                                    Error_Found :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Error_Found,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        LexTokenManager.State,
   --#                                        Pragma_Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Pragma_Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;

   -----------------------------------------------------------------------------

   procedure wf_elaborate_body (PragmaNode : in STree.SyntaxNode;
                                PackSym    : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict           from *,
   --#                                        PackSym &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        PackSym,
   --#                                        PragmaNode &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        PackSym,
   --#                                        PragmaNode,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;

   -----------------------------------------------------------------------------

   -- Checks Val against Typ'First .. Typ'Last.
   -- NewVal = Val if all OK or if arithmetic overflow raised a warning
   -- NewVal = Maths.NoValue if check performed and failed
   procedure ConstraintCheck (Val          : in     Maths.Value;
                              NewVal       :    out Maths.Value;
                              IsAnnotation : in     Boolean;
                              Typ          : in     Dictionary.Symbol;
                              Position     : in     LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        Typ,
   --#                                        Val &
   --#         NewVal                    from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Typ,
   --#                                        Val;
      is separate;
   --------------------------------------------------------------------------
   function Illegal_Unconstrained (LeftType,
                                      RightType : Dictionary.Symbol)
                                  return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;
   --------------------------------------------------------------------------
   function UnknownTypeRecord return ExpRecord
      --returns a "safe" exp result designed to minimise consequential errs
   --# global in Dictionary.Dict;
   is
   begin
      return ExpRecord'(TypeSymbol            => Dictionary.GetUnknownTypeMark,
                        OtherSymbol           => Dictionary.NullSymbol,
                        StreamSymbol          => Dictionary.NullSymbol,
                        TaggedParameterSymbol => Dictionary.NullSymbol,
                        VariableSymbol        => Dictionary.NullSymbol,
                        ParamCount            => 0,
                        ParamList             => Lists.Null_List,
                        Sort                  => TypeResult,
                        ArgListFound          => False,
                        IsAVariable           => False,
                        IsAnEntireVariable    => False,
                        ErrorsInExpression    => True,
                        HasOperators          => False,
                        IsStatic              => True,
                        IsConstant            => True,
                        IsARange              => False,
                        Value                 => Maths.NoValue,
                        RangeRHS              => Maths.NoValue);
   end UnknownTypeRecord;
   -------------------------------------------------------------------------

   procedure AssignmentCheck (Position   : in     LexTokenManager.Token_Position;
                              Scope      : in     Dictionary.Scopes;
                              TargetType : in     Dictionary.Symbol;
                              ExpResult  : in out ExpRecord)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpResult,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        TargetType &
   --#         ExpResult                 from *,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        TargetType &
   --#         LexTokenManager.State     from *,
   --#                                        Dictionary.Dict,
   --#                                        ExpResult,
   --#                                        TargetType;
      is separate;
   --------------------------------------------------------------------------
   function IsEnclosingPackage (OuterPack : Dictionary.Symbol;
                                Scope     : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
      ScopeChain,
      OuterScope  : Dictionary.Scopes;
      Result      : Boolean;
   begin
      if OuterPack = Dictionary.GetRegion (Scope) then
         Result := True;
      else
         Result := False;
         OuterScope := Dictionary.LocalScope (OuterPack);
         ScopeChain := Scope;
         while not Dictionary.IsGlobalScope (ScopeChain) loop
            ScopeChain := Dictionary.GetEnclosingScope (ScopeChain);
            if ScopeChain = OuterScope then
               Result := True;
               exit;
            end if;
         end loop;
      end if;

      return Result;
   end IsEnclosingPackage;

   -----------------------------------------------------------------------------------------

   procedure AddChild (RootIdNode : in     STree.SyntaxNode;
                       IsPrivate  : in     Boolean;
                       Scope      : in     Dictionary.Scopes;
                       ChildSym   :    out Dictionary.Symbol;
                       ChildStr   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ChildStr                  from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        RootIdNode,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ChildSym,
   --#         Dictionary.Dict           from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsPrivate,
   --#                                        LexTokenManager.State,
   --#                                        RootIdNode,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsPrivate,
   --#                                        LexTokenManager.State,
   --#                                        RootIdNode,
   --#                                        Scope &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsPrivate,
   --#                                        LexTokenManager.State,
   --#                                        RootIdNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsPrivate,
   --#                                        LexTokenManager.State,
   --#                                        RootIdNode,
   --#                                        Scope,
   --#                                        STree.Table;
   is
      CurrNode  : STree.SyntaxNode;
      CurrStr   : LexTokenManager.Lex_String;
      CurrSym   : Dictionary.Symbol;
      ParentSym : Dictionary.Symbol;
      ChildSort : Dictionary.PackageSort;
   begin
      CurrNode := RootIdNode;
      CurrStr := NodeLexString (CurrNode);
      CurrSym := Dictionary.LookupItem (Name    => CurrStr,
                                        Scope   => Scope,
                                        Context => Dictionary.ProofContext);
      loop -- to handle multiple prefixes
         if CurrSym = Dictionary.NullSymbol then
            -- not declared or visible
            ParentSym := Dictionary.NullSymbol;
            ErrorHandler.SemanticError (140,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        CurrStr);
            exit;
         end if;

         if not Dictionary.IsPackage (CurrSym) then
            --can't be a parent
            CurrSym := Dictionary.NullSymbol;
            ParentSym := Dictionary.NullSymbol;
            ErrorHandler.SemanticError (18,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        CurrStr);
            exit;
         end if;

         -- CurrStr (CurrSym) is visible and its a package
         STree.Set_Node_Lex_String (Sym  => CurrSym,
                                    Node => CurrNode);
         ParentSym := CurrSym;
         CurrNode := Next_Sibling (ParentNode (CurrNode));
         CurrStr := NodeLexString (CurrNode);
         CurrSym := Dictionary.LookupSelectedItem (Prefix   => CurrSym,
                                                   Selector => CurrStr,
                                                   Scope    => Scope,
                                                   Context  => Dictionary.ProofContext);
         exit when SyntaxNodeType (Next_Sibling (ParentNode (CurrNode))) /= SPSymbols.identifier;
         -- when no more identifier (s) to right
      end loop;

      if CurrSym /= Dictionary.NullSymbol then
         -- child already declared
         ErrorHandler.SemanticError (10,
                                     ErrorHandler.NoReference,
                                     NodePosition (CurrNode),
                                     CurrStr);
      elsif ParentSym /= Dictionary.NullSymbol then
         -- check that CurrStr has not been declared as a body stub
         CurrSym := Dictionary.LookupImmediateScope (Name    => CurrStr,
                                                     Scope   => Dictionary.LocalScope (ParentSym),
                                                     Context => Dictionary.ProgramContext);
         if CurrSym /= Dictionary.NullSymbol and then Dictionary.HasBodyStub (CurrSym) then
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        CurrStr);
         elsif CurrSym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => CurrSym,
                                       Node => CurrNode);
         end if;

         --check that CurrStr not inherited by parent of private child:
         CurrSym := Dictionary.LookupItem (Name    => CurrStr,
                                           Scope   => Dictionary.VisibleScope (ParentSym),
                                           Context => Dictionary.ProofContext);
         if IsPrivate and then CurrSym /= Dictionary.NullSymbol then
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        CurrStr);
         elsif CurrSym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => CurrSym,
                                       Node => CurrNode);
         end if;
      end if;

      if ParentSym /= Dictionary.NullSymbol then
         if IsPrivate then
            ChildSort := Dictionary.PrivateChild;
         else
            ChildSort := Dictionary.Public;
         end if;
         Dictionary.AddChildPackage (TheParent     => ParentSym,
                                     Sort          => ChildSort,
                                     Name          => CurrStr,
                                     Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                     Specification => Dictionary.Location'(NodePosition (RootIdNode),
                                                                           NodePosition (RootIdNode)),
                                     Scope         => Scope,
                                     --to get
                                     ThePackage    => ChildSym);
      else
         -- Parent is not valid (i.e. undeclared or not a package) so we can't do any more.
         -- Signal failure back to caller with null child symbol
         ChildSym := Dictionary.NullSymbol;
         CurrStr := LexTokenManager.Null_String;
      end if;
      ChildStr := CurrStr;
   end AddChild;

   -----------------------------------------------------------------------------------------

   procedure GetLiteralValue (Node : in     STree.SyntaxNode;
                              Val  :    out Maths.Value)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Val                       from LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   --------------------------------------------
   procedure GetCharacterLiteral (Node : in     STree.SyntaxNode;
                                  Val  :    out Maths.Value)
   --# global in LexTokenManager.State;
   --#        in STree.Table;
   --# derives Val from LexTokenManager.State,
   --#                  Node,
   --#                  STree.Table;
      is separate;
   --------------------------------------------

   procedure GetStringLiteralLength (Node : in     STree.SyntaxNode;
                                     Val  :    out Maths.Value)
   --# global in LexTokenManager.State;
   --#        in STree.Table;
   --# derives Val from LexTokenManager.State,
   --#                  Node,
   --#                  STree.Table;
      is separate;
   --------------------------------------------
   function GetObjectValue (Sym : Dictionary.Symbol) return Maths.Value
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;
   --------------------------------------------
   function GetEnumLitValue (Sym : Dictionary.Symbol) return Maths.Value
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;
   --------------------------------------------
   procedure CalcBinaryOperator (Node          : in     STree.SyntaxNode;
                                 Operator      : in     SPSymbols.SPSymbol;
                                 LeftVal,
                                 RightVal      : in     Maths.Value;
                                 IsAnnotation  : in     Boolean;
                                 Result        : in out ExpRecord)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LeftVal,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Operator,
   --#                                        Result,
   --#                                        RightVal,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Result                    from *,
   --#                                        Dictionary.Dict,
   --#                                        LeftVal,
   --#                                        LexTokenManager.State,
   --#                                        Operator,
   --#                                        RightVal;
      is separate;
   --------------------------------------------
   procedure CalcAttribute (Node         : in     STree.SyntaxNode;
                            AttribName   : in     LexTokenManager.Lex_String;
                            PrefixKind   : in     Dictionary.PrefixSort;
                            Prefix       : in     Dictionary.Symbol;
                            BaseFound    : in     Boolean;
                            IsAnnotation : in     Boolean;
                            Argument     : in out Maths.Value;
                            RHSofRange   :    out Maths.Value)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Argument,
   --#         RHSofRange                from Argument,
   --#                                        AttribName,
   --#                                        BaseFound,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Prefix &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Argument,
   --#                                        AttribName,
   --#                                        BaseFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Prefix,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         null                      from PrefixKind;
      is separate;
   --------------------------------------------------------------------
   function UnknownSymbolRecord return ExpRecord
   --# global in Dictionary.Dict;
      is separate;
   --------------------------------------------------------------------
   function SubstituteProtectedTypeSelfReference (Sym,
                                                  PrefixSymbol : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol;
   begin
      -- if Sym is the implicitly-declared own variable of a protected type
      -- then we must replace it with the "current instance of the protected object"
      -- before checking whether it is visible.
      -- Background: given protected type PT its operations will globally reference and
      -- derive PT meaning, in this case, "myself".
      -- If an object PO of type PT (or a subtype of PT) is declared then calls to its
      -- operations will take the form PO.Op and the calling environment will be annotated
      -- in terms of PO.  Therefore, when checking that the globals necessary for the call
      -- PO.Op are visible (for example), we need to replace all references to PT into
      -- references to PO before making the check.  The Prefix Symbol of the call is the
      -- symbol we need to substitute in.
      Result := Sym;
      if Dictionary.IsOwnVariable (Sym) and then
        Dictionary.IsProtectedType (Dictionary.GetOwner (Sym)) then
         Result := PrefixSymbol;
      end if;
      return Result;
   end SubstituteProtectedTypeSelfReference;
   ---------------------------------------------------------------
   -- this function checks if the symbol passed is a stream variable or
   -- a function which globally accesses a stream variable.  If it is
   -- it returns the symbol otherwise it returns NullSymbol
   function StreamReferencesBy (Sym   : Dictionary.Symbol;
                                Scope : Dictionary.Scopes) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol := Dictionary.NullSymbol;
      It     : Dictionary.Iterator;
   begin -- StreamReferencesBy
      if Dictionary.IsAdaFunction (Sym) then --IsAdaFunction used to block proof functions
         It := Dictionary.FirstGlobalVariable (Dictionary.GetAbstraction (Sym, Scope),
                                               Sym);
         while not Dictionary.IsNullIterator (It) loop
            if Dictionary.IsOwnVariableOrConstituentWithMode (Dictionary.CurrentSymbol (It)) then
               Result := Sym;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      else -- check for stream variable case
         if Dictionary.IsOwnVariableOrConstituentWithMode (Sym) then
            Result := Sym;
         end if;
      end if;
      return Result;
   end StreamReferencesBy;

   --------------------------------------------------------------------
   -- this function checks if the symbol passed is a protected variable or
   -- a function which globally accesses a protected variable.  If it is and
   -- the protected variable is not owned by the region we are calling from
   -- it returns the symbol otherwise it returns NullSymbol
   function ProtectedReferencesBy (Sym   : Dictionary.Symbol;
                                   Scope : Dictionary.Scopes) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result     : Dictionary.Symbol := Dictionary.NullSymbol;
      CurrentVar : Dictionary.Symbol;
      It         : Dictionary.Iterator;

      function IsLocalProtectedState (VarSym : Dictionary.Symbol;
                                      Scope  : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      -- pre IsOwnVariable (VarSym) and GetOwnVariableProtected (VarSym)
      -- returns True if the owner of VarSym is the protected type which also encloses the
      -- scope we are making the function call in
      is
         CompilationUnit : Dictionary.Symbol;

         function GetEnclosingPackageOrProtectedType (Scope : Dictionary.Scopes) return Dictionary.Symbol
         --# global in Dictionary.Dict;
         is
            Current : Dictionary.Scopes;
            Region  : Dictionary.Symbol;
         begin -- GetEnclosingPackageOrProtectedType
            Current := Scope;
            loop
               Region := Dictionary.GetRegion (Current);
               exit when Dictionary.IsPackage (Region);
               exit when Dictionary.IsProtectedType (Region);
               Current := Dictionary.GetEnclosingScope (Current);
            end loop;
            return Region;
         end GetEnclosingPackageOrProtectedType;

      begin -- IsLocalProtectedState
         CompilationUnit := GetEnclosingPackageOrProtectedType (Scope);
         return Dictionary.IsProtectedType (CompilationUnit) and then
           Dictionary.GetOwner (VarSym) = CompilationUnit;
      end IsLocalProtectedState;

   begin -- ProtectedReferencesBy
      if Dictionary.IsAdaFunction (Sym) then --IsAdaFunction used to block proof functions
         It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Sym);
         while not Dictionary.IsNullIterator (It) loop
            CurrentVar := Dictionary.CurrentSymbol (It);
            if Dictionary.IsOwnVariable (CurrentVar) and then
              Dictionary.GetOwnVariableProtected (CurrentVar) and then
              not Dictionary.IsOwnVariableOrConstituentWithMode (CurrentVar) and then
              not IsLocalProtectedState (CurrentVar, Scope) then
               Result := Sym;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      else -- check for pragma atomic protected variable case
         if Dictionary.IsOwnVariable (Sym) and then
           Dictionary.GetOwnVariableProtected (Sym) and then
           not Dictionary.IsOwnVariableOrConstituentWithMode (Sym) then
            Result := Sym;
         end if;
      end if;
      return Result;
   end ProtectedReferencesBy;
   --------------------------------------------------------------------
   procedure StackIdentifier (Sym          : in     Dictionary.Symbol;
                              IdStr        : in     LexTokenManager.Lex_String;
                              Node         : in     STree.SyntaxNode;
                              Prefix       : in     Dictionary.Symbol;
                              Scope        : in     Dictionary.Scopes;
                              EStack       : in out ExpStack.ExpStackType;
                              RefVar       : in     SeqAlgebra.Seq;
                              Dotted       : in     Boolean;
                              Context      : in     Tilde_Context;
                              IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Dotted,
   --#                                        IdStr,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Prefix,
   --#                                        Scope,
   --#                                        Sym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        Dictionary.Dict,
   --#                                        Dotted,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdStr,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Prefix,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Dotted,
   --#                                        IdStr,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Prefix,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        Sym,
   --#                                        TheHeap &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IdStr,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Prefix,
   --#                                        Scope,
   --#                                        Sym;
      is separate;

   procedure wf_tilde (Node           : in     STree.SyntaxNode;
                       Scope          : in     Dictionary.Scopes;
                       EStack         : in out ExpStack.ExpStackType;
                       Context        : in     Anno_Tilde_Context)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Context,
   --#                                        Dictionary.Dict,
   --#                                        Scope;
      is separate;
   --------------------------------------------
   procedure wf_percent (Node           : in     STree.SyntaxNode;
                         Scope          : in     Dictionary.Scopes;
                         EStack         : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        Scope;
      is separate;
   --------------------------------------------
   procedure wf_identifier (Node         : in     STree.SyntaxNode;
                            Scope        : in     Dictionary.Scopes;
                            EStack       : in out ExpStack.ExpStackType;
                            RefVar       : in     SeqAlgebra.Seq;
                            Context      : in     Tilde_Context;
                            IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Context,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   --------------------------------------
   procedure wf_selected_component (Node          : in     STree.SyntaxNode;
                                    Scope         : in     Dictionary.Scopes;
                                    EStack        : in out ExpStack.ExpStackType;
                                    RefVar        : in     SeqAlgebra.Seq;
                                    ComponentData : in out ComponentManager.ComponentData;
                                    Context       : in     Tilde_Context;
                                    IsAnnotation  : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ComponentData             from *,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Scope,
   --#                                        TheHeap &
   --#         Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        Context,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -----------------------------------------------------------------------
   procedure wf_primary (Node          : in     STree.SyntaxNode;
                         Scope         : in     Dictionary.Scopes;
                         EStack        : in out ExpStack.ExpStackType;
                         IsAnnotation  : in     Boolean;
                         RefVar        : in     SeqAlgebra.Seq;
                         ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict           from ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        RefVar,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope;
      is separate;
   -----------------------------------------------------------------------

   procedure wf_attribute (EStack : in out ExpStack.ExpStackType)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --# derives EStack from *,
   --#                     Dictionary.Dict &
   --#         null   from CommandLineData.Content;
      is separate;
   -----------------------------------------------------------------------
   procedure wf_attribute_designator (Node         : in     STree.SyntaxNode;
                                      Scope        : in     Dictionary.Scopes;
                                      EStack       : in out ExpStack.ExpStackType;
                                      IsAnnotation : in     Boolean;
                                      RefVar       : in     SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -----------------------------------------------------------------------
   procedure down_wf_name_argument_list (Node         : in     STree.SyntaxNode;
                                         Scope        : in     Dictionary.Scopes;
                                         EStack       : in out ExpStack.ExpStackType;
                                         HeapParam    : in out Lists.List_Heap;
                                         NextNode     :    out STree.SyntaxNode;
                                         IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         HeapParam                 from *,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table &
   --#         NextNode                  from Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------------
   procedure up_wf_name_argument_list (Node         : in     STree.SyntaxNode;
                                       EStack       : in out ExpStack.ExpStackType;
                                       HeapParam    : in out Lists.List_Heap;
                                       IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         HeapParam                 from *,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_positional_argument_association (Node          : in out STree.SyntaxNode;
                                                 Scope         : in     Dictionary.Scopes;
                                                 EStack        : in out ExpStack.ExpStackType;
                                                 IsAnnotation  : in     Boolean;
                                                 RefVar        : in     SeqAlgebra.Seq;
                                                 ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ComponentData             from *,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        TheHeap &
   --#         Dictionary.Dict           from *,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        TheHeap &
   --#         LexTokenManager.State     from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Scope &
   --#         Node                      from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        RefVar,
   --#                                        TheHeap &
   --#         STree.Table               from *,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope;
      is separate;
   ----------------------------------------------------------------------
   -- This procedure is used in wf_named_argument_association and
   -- expressiontypefromcontext to find a named argument association
   -- parameter.  If NameIsParameterName returns True, then the
   -- identifier simple_name (below Node) is a legal parameter name
   -- for the given subprogram, and ParamSym denotes that parameter name.
   -- If NameIsParameterName return False, then the identifier is not
   -- a legal parameter names, and ParamSym is NullSymbol.
   procedure FindNamedArgumentAssociationParameter
     (Node                : in     STree.SyntaxNode;
      SubprogSym          : in     Dictionary.Symbol;
      IsAnnotation        : in     Boolean;
      NameIsParameterName :    out Boolean;
      ParamSym            :    out Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out STree.Table;
   --# derives NameIsParameterName,
   --#         ParamSym,
   --#         STree.Table         from Dictionary.Dict,
   --#                                  IsAnnotation,
   --#                                  LexTokenManager.State,
   --#                                  Node,
   --#                                  STree.Table,
   --#                                  SubprogSym;
   ---- pre SyntaxNodeType (Node) = SPSymbols.named_argument_association
   is
      IdentNode : STree.SyntaxNode;
      IdentStr  : LexTokenManager.Lex_String;
      It        : Dictionary.Iterator;
      Sym       : Dictionary.Symbol;

      function FindIdentifier (Node : STree.SyntaxNode)
                              return STree.SyntaxNode
      --# global in IsAnnotation;
      --#        in STree.Table;
      is
         SIMPLE_LOOKUP : constant Annotation_Symbol_Table :=
           Annotation_Symbol_Table'(False => SPSymbols.simple_name,
                                    True  => SPSymbols.annotation_simple_name);
         R : STree.SyntaxNode;
      begin
         if SyntaxNodeType (Child_Node (Node)) = SIMPLE_LOOKUP (IsAnnotation) then
            R := Child_Node (Child_Node (Node));
         else
            R := Child_Node (Next_Sibling (Child_Node (Node)));
         end if;
         return R;
      end FindIdentifier;

   begin
      IdentNode := FindIdentifier (Node);
      IdentStr  := NodeLexString (IdentNode);

      NameIsParameterName := False;
      ParamSym := Dictionary.NullSymbol;

      It := Dictionary.FirstSubprogramParameter (SubprogSym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Dictionary.GetSimpleName (Sym),
                                                                 Lex_Str2 => IdentStr) = LexTokenManager.Str_Eq then
            NameIsParameterName := True;
            ParamSym := Sym;
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => IdentNode);
            exit;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;

   end FindNamedArgumentAssociationParameter;

   ----------------------------------------------------------------------
   procedure wf_named_argument_association (Node         : in     STree.SyntaxNode;
                                            Scope        : in     Dictionary.Scopes;
                                            EStack       : in out ExpStack.ExpStackType;
                                            HeapParam    : in out Lists.List_Heap;
                                            IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         LexTokenManager.State     from Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         HeapParam                 from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure HomoImplTypeConv (Operator          : in     SPSymbols.SPSymbol;
                               LeftType,
                               RightType         : in out Dictionary.Symbol;
                               LeftVal           : in     Maths.Value;
                               RightVal          : in     Maths.Value;
                               LeftHasOperators  : in     Boolean;
                               RightHasOperators : in     Boolean;
                               LeftPos           : in     LexTokenManager.Token_Position;
                               RightPos          : in     LexTokenManager.Token_Position;
                               IsAnnotation      : in     Boolean;
                               TStack            : in     TypeContextStack.TStackType;
                               Scope             : in     Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LeftHasOperators,
   --#                                        LeftPos,
   --#                                        LeftType,
   --#                                        LeftVal,
   --#                                        LexTokenManager.State,
   --#                                        Operator,
   --#                                        RightHasOperators,
   --#                                        RightPos,
   --#                                        RightType,
   --#                                        RightVal,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS &
   --#         LeftType                  from *,
   --#                                        Dictionary.Dict,
   --#                                        LeftHasOperators,
   --#                                        Operator,
   --#                                        RightType,
   --#                                        Scope,
   --#                                        TStack &
   --#         RightType                 from *,
   --#                                        Dictionary.Dict,
   --#                                        LeftType,
   --#                                        Operator,
   --#                                        RightHasOperators,
   --#                                        Scope,
   --#                                        TStack;
      is separate;
   -----------------------------------------------------------------
   procedure CheckBinaryOperator (Operator          : in     SPSymbols.SPSymbol;
                                  Left              : in     ExpRecord;
                                  Right             : in     ExpRecord;
                                  Scope             : in     Dictionary.Scopes;
                                  TStack            : in     TypeContextStack.TStackType;
                                  OpPos             : in     LexTokenManager.Token_Position;
                                  LeftPos           : in     LexTokenManager.Token_Position;
                                  RightPos          : in     LexTokenManager.Token_Position;
                                  Convert           : in     Boolean;
                                  IsAnnotation      : in     Boolean;
                                  Result            : in out ExpRecord)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Convert,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        Left,
   --#                                        LeftPos,
   --#                                        LexTokenManager.State,
   --#                                        Operator,
   --#                                        OpPos,
   --#                                        Right,
   --#                                        RightPos,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        TStack &
   --#         Result                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Convert,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        Left,
   --#                                        Operator,
   --#                                        Right,
   --#                                        Scope,
   --#                                        TStack;
      is separate;

   -----------------------------------------------------------------
   function NullTypeRecord return ExpRecord
   --# global in Dictionary.Dict;
      is separate;

   -----------------------------------------------------------------

   function Ops_Are_Same_And_Commutative
     (Op1, Op2 : in SPSymbols.SPSymbol) return Boolean
   is
   begin
      return (Op1 = Op2) and then
        (Op1 = SPSymbols.plus or Op1 = SPSymbols.multiply);
   end Ops_Are_Same_And_Commutative;


   -----------------------------------------------------------------
   procedure wf_simple_expression (Node                  : in     STree.SyntaxNode;
                                   Scope                 : in     Dictionary.Scopes;
                                   EStack                : in out ExpStack.ExpStackType;
                                   TStack                : in     TypeContextStack.TStackType;
                                   ContextRequiresStatic : in     Boolean;
                                   IsAnnotation          : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextRequiresStatic,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        TStack;
      is separate;
   -----------------------------------------------------------------
   procedure wf_simple_expression_opt (Node         : in     STree.SyntaxNode;
                                       Scope        : in     Dictionary.Scopes;
                                       EStack       : in out ExpStack.ExpStackType;
                                       TStack       : in     TypeContextStack.TStackType;
                                       IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        TStack;
      is separate;
   -----------------------------------------------------------------
   procedure wf_expression (Node         : in     STree.SyntaxNode;
                            Scope        : in     Dictionary.Scopes;
                            EStack       : in out ExpStack.ExpStackType;
                            TStack       : in     TypeContextStack.TStackType;
                            IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack;
      is separate;
   -----------------------------------------------------------------
   procedure wf_term (Node                  : in     STree.SyntaxNode;
                      Scope                 : in     Dictionary.Scopes;
                      EStack                : in out ExpStack.ExpStackType;
                      TStack                : in     TypeContextStack.TStackType;
                      ContextRequiresStatic : in     Boolean;
                      IsAnnotation          : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextRequiresStatic,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        TStack;
      is separate;
   -----------------------------------------------------------------
   procedure wf_factor (Node         : in     STree.SyntaxNode;
                        Scope        : in     Dictionary.Scopes;
                        EStack       : in out ExpStack.ExpStackType;
                        TStack       : in     TypeContextStack.TStackType;
                        IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        TStack;
      is separate;
   -----------------------------------------------------------------
   procedure wf_relation (Node         : in     STree.SyntaxNode;
                          Scope        : in     Dictionary.Scopes;
                          EStack       : in out ExpStack.ExpStackType;
                          TStack       : in     TypeContextStack.TStackType;
                          IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope;
      is separate;
   -----------------------------------------------------------------
   procedure wf_arange (Node         : in     STree.SyntaxNode;
                        Scope        : in     Dictionary.Scopes;
                        EStack       : in out ExpStack.ExpStackType;
                        IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure down_wf_aggregate (Node         : in     STree.SyntaxNode;
                                Scope        : in     Dictionary.Scopes;
                                NextNode     :    out STree.SyntaxNode;
                                EStack       : in out ExpStack.ExpStackType;
                                HeapParam    : in out Lists.List_Heap;
                                IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives AggregateStack.State      from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         HeapParam                 from Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         NextNode                  from Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure wf_ancestor_part
     (Node         : in out STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         HeapParam                 from Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         Node                      from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         STree.Table               from *,
   --#                                        EStack,
   --#                                        Node;
      is separate;
   -----------------------------------------------------------------
   function NullParameterRecord return ExpRecord
   --# global in Dictionary.Dict;
      is separate;
   -----------------------------------------------------------------
   procedure down_wf_aggregate_choice_rep (Node         : in     STree.SyntaxNode;
                                           Scope        : in     Dictionary.Scopes;
                                           EStack       : in out ExpStack.ExpStackType;
                                           HeapParam    : in out Lists.List_Heap;
                                           IsAnnotation : in     Boolean;
                                           NextNode     :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         HeapParam,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         NextNode                  from Dictionary.Dict,
   --#                                        EStack,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure wf_record_component_selector_name (Node         : in     STree.SyntaxNode;
                                                Scope        : in     Dictionary.Scopes;
                                                EStack       : in out ExpStack.ExpStackType;
                                                HeapParam    : in out Lists.List_Heap;
                                                NextNode     :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         HeapParam                 from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         NextNode                  from ;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_aggregate_choice (Node   : in     STree.SyntaxNode;
                                     Scope  : in     Dictionary.Scopes;
                                     EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives AggregateStack.State      from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AggregateStack.State,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_named_association_rep (Node         : in     STree.SyntaxNode;
                                          Scope        : in     Dictionary.Scopes;
                                          EStack       : in out ExpStack.ExpStackType;
                                          HeapParam    : in out Lists.List_Heap;
                                          IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         HeapParam                 from Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_named_record_component_association
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         HeapParam                 from Dictionary.Dict,
   --#                                        EStack,
   --#                                        HeapParam,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         LexTokenManager.State     from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        Node;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_positional_association (Node   : in     STree.SyntaxNode;
                                           EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_aggregate_or_expression (Node         : in out STree.SyntaxNode;
                                            Scope        : in     Dictionary.Scopes;
                                            EStack       : in out ExpStack.ExpStackType;
                                            IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         Node,
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_component_association (Node         : in     STree.SyntaxNode;
                                          Scope        : in     Dictionary.Scopes;
                                          EStack       : in out ExpStack.ExpStackType;
                                          IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure wf_positional_record_component_association
     (Node         : in out STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope &
   --#         LexTokenManager.State     from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack &
   --#         Node,
   --#         STree.Table               from Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure up_wf_aggregate (Node         : in     STree.SyntaxNode;
                              Scope        : in     Dictionary.Scopes;
                              EStack       : in out ExpStack.ExpStackType;
                              IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives AggregateStack.State,
   --#         EStack                    from AggregateStack.State,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AggregateStack.State,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure wf_qualified_expression (Node         : in     STree.SyntaxNode;
                                      Scope        : in     Dictionary.Scopes;
                                      EStack       : in out ExpStack.ExpStackType;
                                      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         LexTokenManager.State     from Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;

   -------------------------------------------------------------------

   procedure down_wf_aggregate_or_expression
     (Node         : in     STree.SyntaxNode;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean;
      NextNode     :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         NextNode                  from Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;

   -------------------------------------------------------------------

   ---------------------------------------------------------------------------------
   -- The type context needed for an attribute with arguments (i.e. 'Val, 'Max etc)
   -- depends on the Prefix and the attribute identifier.  Some attributes have two
   -- arguments, so we need a special handler here, since an attribute_designator
   -- node is the closent common parent node of one or both arguments.
   ---------------------------------------------------------------------------------
   function AttributeDesignatorTypeFromContext
     (ExpNode       : in     STree.SyntaxNode;
      IsAnnotation  : in     Boolean;
      EStack        : in     ExpStack.ExpStackType;
      TStack        : in     TypeContextStack.TStackType)
     return Dictionary.Symbol
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
      is separate;

   ---------------------------------------------------------------------------------
   -- Returns the type needed for an expression from its context
   ---------------------------------------------------------------------------------
   procedure ExpressionTypeFromContext (Exp_Node         : in     STree.SyntaxNode;
                                        E_Stack          : in     ExpStack.ExpStackType;
                                        T_Stack          : in     TypeContextStack.TStackType;
                                        New_Context_Type :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out STree.Table;
   --# derives New_Context_Type from CommandLineData.Content,
   --#                               Dictionary.Dict,
   --#                               Exp_Node,
   --#                               E_Stack,
   --#                               LexTokenManager.State,
   --#                               STree.Table,
   --#                               T_Stack &
   --#         STree.Table      from *,
   --#                               CommandLineData.Content,
   --#                               Dictionary.Dict,
   --#                               Exp_Node,
   --#                               E_Stack,
   --#                               LexTokenManager.State;
      is separate;

   ---------------------------------------------------------------------------------
   -- Primary grammer productions nearly always have the same type context
   -- as their parent nodes, but there is a single exception - when a primary
   -- node is the right-hand operand of an exponentiation operator, where the
   -- context is always Integer.
   ---------------------------------------------------------------------------------
   function PrimaryTypeFromContext (Node   : in     STree.SyntaxNode;
                                    TStack : in     TypeContextStack.TStackType)
                                  return Dictionary.Symbol
   --# global in Dictionary.Dict;
   --#        in STree.Table;
      is separate;

   ---------------------------------------------------------------------------------
   -- range constraints appear below aggregate_choice, case_choice, and below
   -- relational operators, and so can have a chage of type context.
   ---------------------------------------------------------------------------------
   function RangeConstraintTypeFromContext (ExpNode       : in     STree.SyntaxNode;
                                            EStack        : in     ExpStack.ExpStackType;
                                            TStack        : in     TypeContextStack.TStackType)
                                     return Dictionary.Symbol
   --# global in STree.Table;
      is separate;

   ---------------------------------------------------------------------------------
   -- Simple expressions appear below aggregate_choice, case_choice, and below
   -- relational operators, and so can have a chage of type context.
   ---------------------------------------------------------------------------------
   function SimpleExpressionTypeFromContext
     (ExpNode : in STree.SyntaxNode;
      TStack  : in TypeContextStack.TStackType)
     return Dictionary.Symbol
   --# global in AggregateStack.State;
   --#        in Dictionary.Dict;
   --#        in STree.Table;
      is separate;

   -------------------------------------------------------------------

   procedure WalkExpression (ExpNode               : in     STree.SyntaxNode;
                             Scope                 : in     Dictionary.Scopes;
                             TypeContext           : in     Dictionary.Symbol;
                             ContextRequiresStatic : in     Boolean;
                             Result                :    out ExpRecord;
                             RefVar                : in     SeqAlgebra.Seq;
                             ComponentData         : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Result                    from CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TypeContext &
   --#         ComponentData,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TypeContext &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        ContextRequiresStatic,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TypeContext;
      is separate;

   --------------------------------------------------------------------------------------
   -- This procedure has essentially the functionality of WalkExpression
   -- but is used to walk expressions which the language rules require to be a
   -- "name".  In addition to the things returned by WalkExpression, a flag is
   -- returned indicating whether the expression was indeed a name.
   -- If the expression is not a name the expression is not traversed at all
   -- and Result is set to the UnknownTypeRecord
   --
   -- After the name node has been found it is traversed by WalkExpression and a subset
   -- of the checks usually done by wf_primary are acarried out on the result.  More
   -- checks are done here than are necessary for this application (e.g. getting bounds
   -- of a type mark) but they have been left in to make WalkName more widely applicable;
   -- it could be moved to Wellformed if ever needed elsewhere.
   --
   --
   procedure WalkName (ExpNode     : in     STree.SyntaxNode;
                       Scope       : in     Dictionary.Scopes;
                       Result      :    out ExpRecord;
                       IsAName     :    out Boolean;
                       RefVarParam : in     SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         IsAName                   from ExpNode,
   --#                                        STree.Table &
   --#         Result                    from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      Node        : STree.SyntaxNode;
      Sym         : Dictionary.Symbol;
      ResultLocal : ExpRecord;
   begin --WalkName
      ResultLocal := UnknownTypeRecord; --preset result for all non-name cases
      -- advance to relation node;
      Node := Child_Node (ExpNode);
      if Next_Sibling (Node) =
         STree.NullNode then
         -- advance to simple_expression;
         Node := Child_Node (Node);
         if Next_Sibling (Node) =
            STree.NullNode then
            -- advance to simple_expression_opt;
            Node := Child_Node (Node);
            if Next_Sibling (Node) =
               STree.NullNode then
               -- advance to term;
               Node := Child_Node (Node);
               if Next_Sibling (Node) =
                  STree.NullNode then
                  -- advance to factor;
                  Node := Child_Node (Node);
                  if Next_Sibling (Node) =
                     STree.NullNode then
                     -- advance to primary;
                     Node := Child_Node (Node);
                     if Next_Sibling (Node) =
                        STree.NullNode then
                        -- advance to name;
                        Node := Child_Node (Node);
                        if SyntaxNodeType (Node) = SPSymbols.name then
                           IsAName := True;
                           WalkExpression (ExpNode               => Node,
                                           Scope                 => Scope,
                                             -- The type of a Name can never be resolved by context - it's
                                             -- either a name which exists in the Dictionary and has a type,
                                             -- or it not, so we pass UnknownTypeMark as the context here.
                                           TypeContext           => Dictionary.GetUnknownTypeMark,
                                           ContextRequiresStatic => False,
                                           Result                => ResultLocal,
                                           RefVar                => RefVarParam,
                                           ComponentData         => GlobalComponentData);
                           -- now perform some checks on the result akin to those
                           -- of wf_primary
                           Sym := ResultLocal.OtherSymbol;

                           case ResultLocal.Sort is
                              when IsUnknown =>
                                 ResultLocal := UnknownTypeRecord;
                              when TypeResult =>
                                 null;
                              when IsPackage =>
                                 ResultLocal := UnknownTypeRecord;
                                 ErrorHandler.SemanticError (5,
                                                             ErrorHandler.NoReference,
                                                             NodePosition (Node),
                                                             Dictionary.GetSimpleName (Sym));
                              when IsFunction =>
                                 ErrorHandler.SemanticError (3,
                                                             ErrorHandler.NoReference,
                                                             NodePosition (Node),
                                                             Dictionary.GetSimpleName (Sym));
                                 ResultLocal.IsStatic   := False;
                                 ResultLocal.IsConstant := False;
                                 ResultLocal.IsARange   := False;
                              when IsObject =>
                                 ResultLocal.IsARange   := False;
                                 if Dictionary.IsVariableOrSubcomponent (Sym) then
                                    SeqAlgebra.AddMember
                                       (TheHeap,
                                        RefVarParam,
                                        Natural (Dictionary.SymbolRef (Sym)));
                                 end if;
                              when IsTypeMark =>
                                 ResultLocal.IsStatic := Dictionary.IsStatic (Sym,
                                                                              Scope);
                                 ResultLocal.IsConstant := True;

                                 if Dictionary.IsScalarTypeMark (Sym,
                                                                 Scope)
                                 then -- we can get bounds of range and so on
                                    ResultLocal.IsARange   := True;
                                    -- get upper and lower bounds of type from dictionary
                                    ResultLocal.Value := Maths.ValueRep
                                       (Dictionary.GetScalarAttributeValue
                                        (False,  --don't want base type
                                         LexTokenManager.First_Token,
                                         Sym));
                                    ResultLocal.RangeRHS := Maths.ValueRep
                                       (Dictionary.GetScalarAttributeValue
                                        (False,  --don't want base type
                                         LexTokenManager.Last_Token,
                                         Sym));

                                 else -- not scalar so we don't do bounds and its not a range
                                    ResultLocal.IsARange   := False;
                                 end if;
                              when IsParameterName =>
                                 null; --should never occur
                           end case;

                           ResultLocal.ParamCount := 0;
                           ResultLocal.ParamList := Lists.Null_List;

                        else
                           IsAName := False;
                        end if;
                     else
                        IsAName := False;
                     end if;
                  else
                     IsAName := False;
                  end if;
               else
                  IsAName := False;
               end if;
            else
               IsAName := False;
            end if;
         else
            IsAName := False;
         end if;
      else
         IsAName := False;
      end if;
      Result := ResultLocal;
   end WalkName;

   ---------------------------------------------------------------------

   procedure CheckPriorityRange (ErrorSym   : in     Dictionary.Symbol;
                                 Scope      : in     Dictionary.Scopes;
                                 PragmaKind : in     Dictionary.RavenscarPragmas;
                                 ErrPos     : in     LexTokenManager.Token_Position;
                                 Value      : in     Maths.Value;
                                 ValueRep   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorSym,
   --#                                        ErrPos,
   --#                                        LexTokenManager.State,
   --#                                        PragmaKind,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        Value &
   --#         LexTokenManager.State,
   --#         ValueRep                  from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        PragmaKind,
   --#                                        Value;
      is separate;
   ---------------------------------------------------------------------
   procedure CheckPriorityPropertyConsistency
     (Sym                     : in     Dictionary.Symbol;
      TypeSym                 : in     Dictionary.Symbol;
      PriorityPropertyValue   : in     LexTokenManager.Lex_String;
      ErrorNode               : in     STree.SyntaxNode;
      Consistent              :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Consistent                from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        PriorityPropertyValue,
   --#                                        Sym,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.State,
   --#                                        PriorityPropertyValue,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym,
   --#                                        TypeSym;
      is separate;
   ---------------------------------------------------------------------
   procedure CheckInterruptPropertyConsistency
     (HasInterruptProperty : in     Boolean;
      Sym                  : in     Dictionary.Symbol;
      TypeSym              : in     Dictionary.Symbol;
      ErrorNode            : in     STree.SyntaxNode;
      Consistent           :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Consistent                from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        HasInterruptProperty,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        HasInterruptProperty,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym,
   --#                                        TypeSym;
      is separate;
   ---------------------------------------------------------------------

   procedure CreateInterruptStreamVariable
              (ForPO             : in     Dictionary.Symbol;
               TheHandler        : in     LexTokenManager.Lex_String;
               TheStreamVariable : in     LexTokenManager.Lex_String;
               ErrorNode         : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        ForPO,
   --#                                        LexTokenManager.State,
   --#                                        TheHandler,
   --#                                        TheStreamVariable &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        ForPO,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHandler,
   --#                                        TheStreamVariable;
      is separate;
   ---------------------------------------------------------------------

   procedure CreateInterruptStreamVariables
     (ForPO     : in Dictionary.Symbol;
      ErrorNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        ForPO,
   --#                                        LexTokenManager.State &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        ForPO,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------------------
   procedure CheckSuspendablePropertyConsistency
     (Sym              : in     Dictionary.Symbol;
      TypeSym          : in     Dictionary.Symbol;
      IsInSuspendsList : in     Boolean;
      ErrorNode        : in     STree.SyntaxNode;
      Consistent       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Consistent                from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsInSuspendsList,
   --#                                        Sym,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        IsInSuspendsList,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym,
   --#                                        TypeSym;
     is separate;
   ---------------------------------------------------------------------
   procedure CheckCeilingPriority (Sym              : in Dictionary.Symbol;
                                   Scope            : in Dictionary.Scopes;
                                   CheckList        : in Dictionary.Iterator;
                                   PriorityLexValue : in LexTokenManager.Lex_String;
                                   ErrorNode        : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CheckList,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.State,
   --#                                        PriorityLexValue,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym;
      is separate;
   ---------------------------------------------------------------------

   function UnexpectedInitialization (Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return -- not an initialized own variable
         ((Dictionary.IsOwnVariable (Sym) and then
          not Dictionary.OwnVariableIsInitialized (Sym)) or else
         -- not an initialized constituent
         (Dictionary.IsConstituent (Sym) and then
          not Dictionary.OwnVariableIsInitialized (Dictionary.GetSubject (Sym)))) and then
         -- not moded (this last limb to suppress error 333 for stream vars
         (Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.DefaultMode);
   end UnexpectedInitialization;

   ------------------------------------------------------------------------

   procedure wf_pragma (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------------------
   procedure down_wf_store (Node     : in     STree.SyntaxNode;
                            Scope    : in     Dictionary.Scopes;
                            EStack   : in out ExpStack.ExpStackType;
                            NextNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        Scope &
   --#         NextNode                  from Dictionary.Dict,
   --#                                        EStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   --------------------------------------------
   procedure wf_store_list (Node   : in out STree.SyntaxNode;
                            Scope  : in     Dictionary.Scopes;
                            EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        Scope &
   --#         Node                      from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------
   procedure down_wf_store_list (Node     : in     STree.SyntaxNode;
                                 Scope    : in     Dictionary.Scopes;
                                 EStack   : in out ExpStack.ExpStackType;
                                 NextNode :    out STree.SyntaxNode)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in STree.Table;
   --# derives EStack   from * &
   --#         NextNode from Dictionary.Dict,
   --#                       EStack,
   --#                       Node,
   --#                       Scope,
   --#                       STree.Table &
   --#         null     from CommandLineData.Content;
      is separate;
   --------------------------------------------
   procedure up_wf_store (Node   : in     STree.SyntaxNode;
                          Scope  : in     Dictionary.Scopes;
                          EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------

   procedure wf_type_mark (Node         : in     STree.SyntaxNode;
                           CurrentScope : in     Dictionary.Scopes;
                           Context      : in     Dictionary.Contexts;
                           TypeSym      :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         STree.Table,
   --#         TypeSym                   from CommandLineData.Content,
   --#                                        Context,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure down_wf_quantifier (Node     : in     STree.SyntaxNode;
                                 Estack   : in out ExpStack.ExpStackType;
                                 Scope    : in out Dictionary.Scopes;
                                 NextNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Scope,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         NextNode                  from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------

   procedure up_wf_quantifier (Node   : in     STree.SyntaxNode;
                               Estack : in out ExpStack.ExpStackType;
                               Scope  : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         Scope                     from *,
   --#                                        Dictionary.Dict;
      is separate;
   ------------------------------------------------------------
   procedure WalkAnnotationExpression (ExpNode       : in     STree.SyntaxNode;
                                       Scope         : in     Dictionary.Scopes;
                                       TypeContext   : in     Dictionary.Symbol;
                                       Context       : in     Anno_Tilde_Context;
                                       Result        :    out ExpRecord;
                                       ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Result                    from CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TypeContext &
   --#         ComponentData,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TypeContext &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        Context,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TypeContext;
      is separate; -- AggregateStack.State will appear RSN
   ---------------------------------------------------------------------
   procedure wf_priority_value (Node       : in     STree.SyntaxNode;
                                PragmaKind : in     Dictionary.RavenscarPragmas;
                                Context    : in     Dictionary.Contexts;
                                ErrorSym   : in     Dictionary.Symbol;
                                Scope      : in     Dictionary.Scopes;
                                ValueRep   :    out LexTokenManager.Lex_String;
                                Compatible :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Compatible                from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorSym,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PragmaKind,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         LexTokenManager.State,
   --#         ValueRep                  from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PragmaKind,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   ---------------------------------------------------------------------
   procedure wf_priority_pragma (Node  : in STree.SyntaxNode;
                                 Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -----------------------------------------------------------------
   procedure wf_renaming_declaration (Node  : in STree.SyntaxNode;
                                      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure Wf_Use_Type_Clause (Node  : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;

   ------------------------------------------------------------------
   procedure wf_proof_renaming_declaration (Node  : in STree.SyntaxNode;
                                            Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   --------------------------------------------------------------------
   procedure wf_predicate (Node        : in     STree.SyntaxNode;
                           Scope       : in     Dictionary.Scopes;
                           Context     : in     Anno_Tilde_Context;
                           ErrorsFound :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         ErrorsFound               from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   --------------------------------------------------------------------
   procedure CheckNamedAssociation
     (TheFormals             : in Dictionary.Symbol;
      Scope                  : in Dictionary.Scopes;
      NamedArgumentAssocNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        NamedArgumentAssocNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheFormals;
      is separate;

   ------------------------------------------------------------------------

   function FindActualNode (ForFormal              : in Dictionary.Symbol;
                            NamedArgumentAssocNode : in STree.SyntaxNode)
     return STree.SyntaxNode
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
   is separate;
   ------------------------------------------------------------------------
   procedure wf_justification_statement (Node     : in STree.SyntaxNode;
                                         Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is separate;
   ------------------------------------------------------------------------
   procedure WalkStatements (SeqNode : in     STree.SyntaxNode;
                             Scope   : in     Dictionary.Scopes;
                             Table   : in out RefList.HashTable)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out GlobalComponentData;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SeqNode,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        Table,
   --#                                        TheHeap &
   --#         GlobalComponentData,
   --#         Table                     from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SeqNode,
   --#                                        STree.Table,
   --#                                        Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SeqNode,
   --#                                        STree.Table,
   --#                                        Table,
   --#                                        TheHeap;
      is separate;
   ---------------------------------------------------------------------
   procedure wf_subtype_declaration (Node  : in STree.SyntaxNode;
                                     Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;
   --------------------------------------------------------------------
   procedure CheckTaskModifierConsistency (TheOwnVarType  : in     Dictionary.Symbol;
                                           TheVarType     : in     Dictionary.Symbol;
                                           ModifierIsTask : in     Boolean;
                                           ErrorNode      : in     STree.SyntaxNode;
                                           Consistent     :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Consistent                from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ModifierIsTask,
   --#                                        TheOwnVarType,
   --#                                        TheVarType &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.State,
   --#                                        ModifierIsTask,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheOwnVarType,
   --#                                        TheVarType;
   is separate;
   --------------------------------------------------------------------
   procedure CheckProtectedModifierConsistency (TheType             : in     Dictionary.Symbol;
                                                ModifierIsProtected : in     Boolean;
                                                ErrorNode           : in     STree.SyntaxNode;
                                                Consistent          :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Consistent                from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ModifierIsProtected,
   --#                                        TheType &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.State,
   --#                                        ModifierIsProtected,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheType;
   is separate;

   -----------------------------------------------------------------------------
   -- wf_variable_declaration
   --
   -- Note: EnclosingUnitScope represents the scope of the enclosing program
   -- unit, whilst DeclarationScope represents the scope of the current
   -- declaration. These are normally the same. The exception is the special
   -- case of a declaration within a protected type, where they may differ.
   -- For example, the declaration could be in the private part of the PT but
   -- the PT itself could be in the visible part of the enclosing unit.
   -----------------------------------------------------------------------------
   procedure wf_variable_declaration (Node               : in STree.SyntaxNode;
                                      EnclosingUnitScope : in Dictionary.Scopes;
                                      DeclarationScope   : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage     from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DeclarationScope,
   --#                                        Dictionary.Dict,
   --#                                        EnclosingUnitScope,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DeclarationScope,
   --#                                        Dictionary.Dict,
   --#                                        EnclosingUnitScope,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         STree.Table,
   --#         TheHeap                   from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DeclarationScope,
   --#                                        Dictionary.Dict,
   --#                                        EnclosingUnitScope,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------
   procedure wf_constant_declaration (Node         : in STree.SyntaxNode;
                                      CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         TheHeap                   from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------------------
   procedure wf_entire_variable (Node         : in     STree.SyntaxNode;
                                 CurrentScope : in     Dictionary.Scopes;
                                 ErrorHint    : in     Visibility_Error_Hint;
                                 VarSym       :    out Dictionary.Symbol;
                                 Dotted       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dotted,
   --#         STree.Table,
   --#         VarSym                    from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorHint,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ----------------------------------------------------------------------------

   procedure CheckDerivesConsistency (SubprogSym : in Dictionary.Symbol;
                                      Position   : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        Dictionary.Dict,
   --#                                        SubprogSym,
   --#                                        TheHeap;
      is separate;
   --------------------------------------------------------------------------
   -- procedure for creating a "full" dependency relation
   procedure CreateFullDependency (Node                         : in STree.SyntaxNode;
                                   SubprogSym                   : in Dictionary.Symbol;
                                   Abstraction                  : in Dictionary.Abstractions;
                                   ImportList,
                                      ExportList                   : in SeqAlgebra.Seq)
   --# global in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in     TheHeap;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.UnitStack,
   --#                                ExportList,
   --#                                ImportList,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym,
   --#                                TheHeap &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.UnitStack,
   --#                                Dictionary.Dict,
   --#                                ExportList,
   --#                                ImportList,
   --#                                LexTokenManager.State,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym,
   --#                                TheHeap;
      is separate;
   -----------------------------------------------------------------------

   procedure AddDerivesStreamEffects (Node        : in STree.SyntaxNode;
                                      SubprogSym  : in Dictionary.Symbol;
                                      Abstraction : in Dictionary.Abstractions)
   --# global in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.UnitStack,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.UnitStack,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym;
      is separate;
   ---------------------------------------------------------------------
   procedure CreateFullSubProgDependency (Node        : in STree.SyntaxNode;
                                          SubprogSym  : in Dictionary.Symbol;
                                          Abstraction : in Dictionary.Abstractions)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        Abstraction,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap;
      is separate;
   ---------------------------------------------------------------------
   procedure wf_global_definition (Node         : in     STree.SyntaxNode;
                                   CurrentScope : in     Dictionary.Scopes;
                                   SubprogSym   : in     Dictionary.Symbol;
                                   FirstSeen    : in     Boolean;
                                   SemErrFound  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         SemErrFound,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        STree.Table,
   --#                                        SubprogSym;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_dependency_relation (Node       : in STree.SyntaxNode;
                                     CurrentScope : in Dictionary.Scopes;
                                     SubprogSym : in Dictionary.Symbol;
                                     FirstSeen  : in Boolean;
                                     GlobDefErr : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        GlobDefErr,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        GlobDefErr,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        GlobDefErr,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        GlobDefErr,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_function_annotation (Node         : in STree.SyntaxNode;
                                     CurrentScope : in Dictionary.Scopes;
                                     SubprogSym   : in Dictionary.Symbol;
                                     FirstSeen    : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        STree.Table,
   --#                                        SubprogSym;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_property_list (Node     : in     STree.SyntaxNode;
                               TypeNode : in     STree.SyntaxNode;
                               Scope    : in     Dictionary.Scopes;
                               TheOwner : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TheOwner,
   --#                                        TypeNode &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap,
   --#                                        TheOwner;
   is separate;
   ----------------------------------------------------------------------
   procedure wf_declare_annotation (Node       : in     STree.SyntaxNode;
                                    Scope      : in     Dictionary.Scopes;
                                    TaskOrProc : in     Dictionary.Symbol;
                                    FirstSeen  : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TaskOrProc,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TaskOrProc,
   --#                                        TheHeap;
      is separate;
   --------------------------------------------------------------------------------
   procedure wf_procedure_annotation (Node         : in STree.SyntaxNode;
                                      CurrentScope : in Dictionary.Scopes;
                                      SubprogSym   : in Dictionary.Symbol;
                                      FirstSeen    : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap;
      is separate;
   ----------------------------------------------------------------------------
   procedure wf_procedure_constraint (Node      : in STree.SyntaxNode;
                                      Scope     : in Dictionary.Scopes;
                                      FirstSeen : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -------------------------------------------------------------------------
   procedure wf_function_constraint (Node      : in STree.SyntaxNode;
                                     Scope     : in Dictionary.Scopes;
                                     FirstSeen : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -------------------------------------------------------------------------
   procedure wf_inherit_clause (Node    : in STree.SyntaxNode;
                                CompSym : in Dictionary.Symbol;
                                Scope   : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        CompSym,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        CompSym,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        CompSym,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        CompSym,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------
   procedure wf_context_clause (Node    : in STree.SyntaxNode;
                                CompSym : in Dictionary.Symbol;
                                Scope   : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CompSym,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        CompSym,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------------------
   procedure wf_formal_part (Node            : in STree.SyntaxNode;
                             CurrentScope    : in Dictionary.Scopes;
                             SubProgSym      : in Dictionary.Symbol;
                             FirstOccurrence : in Boolean;
                             Context         : in Dictionary.Contexts)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstOccurrence,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstOccurrence,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym;
      is separate;
   ------------------------------------------------------------------------
   -- procedure to check whether a properly-defined subprogram or stub creates
   -- overloading by re-using the name of a potentially inheritable root
   -- operation
   procedure CheckNoOverloadingFromTaggedOps
     (SpecNode        : in    STree.SyntaxNode;
      SubprogSym      : in    Dictionary.Symbol;
      Scope           : in    Dictionary.Scopes;
      Abstraction     : in    Dictionary.Abstractions;
      Is_Overriding   : in    Boolean)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SpecNode,
   --#                                        STree.Table,
   --#                                        SubprogSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Is_Overriding,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SpecNode,
   --#                                        STree.Table,
   --#                                        SubprogSym;
      is separate;
   ------------------------------------------------------------------------
   procedure wf_function_specification (Node       : in     STree.SyntaxNode;
                                        Hidden     : in     Boolean;
                                        Scope      : in out Dictionary.Scopes;
                                        SubProgSym :    out Dictionary.Symbol;
                                        FirstSeen  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Hidden,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        Hidden,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         FirstSeen                 from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         Scope,
   --#         SubprogSym                from ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   -------------------------------------------------------------------------
   procedure wf_procedure_specification (Node       : in     STree.SyntaxNode;
                                         Hidden     : in     Boolean;
                                         Scope      : in out Dictionary.Scopes;
                                         SubProgSym :    out Dictionary.Symbol;
                                         FirstSeen  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict           from *,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Hidden,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         FirstSeen,
   --#         STree.Table               from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Scope,
   --#         SubprogSym                from ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Hidden,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ----------------------------------------------------------------------------
   -- function used by wf_subprogram_body and wf_body_stub
   function HasParameterGlobalOrReturnOfLocalPrivateType (SubprogSym : Dictionary.Symbol)
                                                          return Boolean
   --# global in Dictionary.Dict;
   -- returns true if the subprogram has a parameter or global which is of a private type
   -- but which is not private when viewd from the subprogram
   is
      Result : Boolean := False;
      It : Dictionary.Iterator;
      TypeSym : Dictionary.Symbol;

      function IsPrivateButNotPrivateHere (TypeSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in SubprogSym;
      is
      begin -- IsPrivateButNotPrivateHere
         return Dictionary.TypeIsPrivate (TypeSym) and then not
                Dictionary.IsPrivateType (TypeSym, Dictionary.LocalScope (SubprogSym));
      end IsPrivateButNotPrivateHere;

   begin -- HasParameterGlobalOrReturnOfLocalPrivateType

      -- iterate through parameters
      It := Dictionary.FirstSubprogramParameter (SubprogSym);
      while It /= Dictionary.NullIterator loop
         TypeSym := Dictionary.GetType (Dictionary.CurrentSymbol (It));
         Result := IsPrivateButNotPrivateHere (TypeSym);
         exit when Result;

         It := Dictionary.NextSymbol (It);
      end loop;

      if not Result then -- no parameters were private, so check globals
         It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                               SubprogSym);
         while It /= Dictionary.NullIterator loop
            TypeSym := Dictionary.GetType (Dictionary.CurrentSymbol (It));
            Result := IsPrivateButNotPrivateHere (TypeSym);
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
      end if;

      -- no parameters or globals were private, so check for function return type.
      if not Result then
         if Dictionary.IsFunction (SubprogSym) then
            TypeSym := Dictionary.GetType (SubprogSym);
            Result := IsPrivateButNotPrivateHere (TypeSym);
         end if;
      end if;

      return Result;
   end HasParameterGlobalOrReturnOfLocalPrivateType;
   ----------------------------------------------------------------------------
   procedure wf_subprogram_body (Node     : in     STree.SyntaxNode;
                                 Scope    : in out Dictionary.Scopes;
                                 NextNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         NextNode,
   --#         Scope                     from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------
   procedure wf_entry_body (Node     : in     STree.SyntaxNode;
                            Scope    : in out Dictionary.Scopes;
                            NextNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         NextNode,
   --#         Scope                     from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------
   procedure wf_proof_function_declaration (Node       : in     STree.SyntaxNode;
                                            Scope      : in     Dictionary.Scopes;
                                            TheFuncSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         TheFuncSym                from ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        STree.Table;
      is separate;
   -----------------------------------------------------------------------
   procedure wf_body_stub (Node  : in STree.SyntaxNode;
                           Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -----------------------------------------------------------------------
   procedure wf_subunit (Node  : in     STree.SyntaxNode;
                         Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Scope,
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   --------------------------------------------------------------------

   procedure CheckEmbedBodies (CompSym : in Dictionary.Symbol;
                               Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CompSym,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------------------

   procedure CheckSuspendsListAccountedFor (ProcOrTask : in     Dictionary.Symbol;
                                            Node       : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProcOrTask,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;

   ---------------------------------------------------------------------
   procedure up_wf_subprogram_body (Node  : in     STree.SyntaxNode;
                                    Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Scope                     from *,
   --#                                        Dictionary.Dict;
      is separate;

   ---------------------------------------------------------------------
   -- this subprogram used to a subunit of wf_package_declaration.wf_visible; it has
   -- been moved up to this level to make it available to wf_protected_type.wf_protected_operation
   procedure wf_subprogram_declaration (Node            : in     STree.SyntaxNode;
                                        CurrentScope    : in     Dictionary.Scopes;
                                        TheSubprogSym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         TheSubprogSym             from ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
      is separate;
   ---------------------------------------------------------------------
   procedure wf_known_discriminant_part (Node             : in STree.SyntaxNode;
                                         ProtectedTypeSym : in Dictionary.Symbol;
                                         Scope            : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProtectedTypeSym,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProtectedTypeSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ----------
   procedure wf_array_type_definition (Node        : in     STree.SyntaxNode;
                                       Scope       : in     Dictionary.Scopes;
                                       Ident_Node  : in     STree.SyntaxNode;
                                       DecLoc      : in     LexTokenManager.Token_Position;
                                       IsGeneric   : in     Boolean;
                                       ErrorsFound :    out Boolean;
                                       TheArray    :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Ident_Node,
   --#                                        IsGeneric,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         ErrorsFound,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        Ident_Node,
   --#                                        IsGeneric,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Ident_Node,
   --#                                        IsGeneric,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         TheArray                  from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        Ident_Node,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ----------
   procedure wf_full_type_declaration (Node  : in STree.SyntaxNode;
                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   -------------------------------------------------------------------
   procedure wf_basic_declarative_item (Node         : in STree.SyntaxNode;
                                        CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   --------------------------------------------------------------------
   procedure CheckAnnouncedTypesDeclared (PackSym : in Dictionary.Symbol;
                                          Scope   : in Dictionary.Scopes;
                                          Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   --------------------------------------------------------------------
   procedure wf_generic_actual_part (GenericNode      : in     STree.SyntaxNode;
                                     GenericSym       : in     Dictionary.Symbol;
                                     InstantiationSym : in     Dictionary.Symbol;
                                     Scope            : in     Dictionary.Scopes;
                                     ErrorFound       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GenericNode,
   --#                                        GenericSym,
   --#                                        GlobalComponentData,
   --#                                        InstantiationSym,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GenericNode,
   --#                                        GenericSym,
   --#                                        GlobalComponentData,
   --#                                        InstantiationSym,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorFound                from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GenericNode,
   --#                                        GenericSym,
   --#                                        GlobalComponentData,
   --#                                        InstantiationSym,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   --------------------------------------------------------------------
   procedure wf_generic_subprogram_instantiation (Node  : in STree.SyntaxNode;
                                                  Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   ----------------------------------------------------------------------
   -- PNA temporary reduced annotation for phase 1 generics cfr 1340
--     procedure wf_generic_subprogram_body (Node     : in     STree.SyntaxNode;
--                                           Scope    : in out Dictionary.Scopes;
--                                           NextNode :    out STree.SyntaxNode)
--     --# global in     CommandLineData.Content;
--     --#        in out Dictionary.Dict;
--     --#        in out TheHeap;
--     --#        in out Statistics.TableUsage;
--     --#        in out ErrorHandler.ErrorContext;
--     --#        in out SPARK_IO.FILE_SYS;
--     --#        in out LexTokenManager.State;
--     --#        in out STree.Table;
--     --#        in out AggregateStack.State;
--     --#        in out GlobalComponentData;
--     --# derives Dictionary.Dict,
--     --#         TheHeap,
--     --#         Statistics.TableUsage,
--     --#         LexTokenManager.State,
--     --#         STree.Table,
--     --#         AggregateStack.State,
--     --#         GlobalComponentData               from *,
--     --#                                                Dictionary.Dict,
--     --#                                                TheHeap,
--     --#                                                LexTokenManager.State,
--     --#                                                CommandLineData.Content,
--     --#                                                STree.Table,
--     --#                                                Scope,
--     --#                                                Node,
--     --#                                                GlobalComponentData &
--     --#         ErrorHandler.ErrorContext,
--     --#         SPARK_IO.FILE_SYS                 from Dictionary.Dict,
--     --#                                                TheHeap,
--     --#                                                ErrorHandler.ErrorContext,
--     --#                                                SPARK_IO.FILE_SYS,
--     --#                                                LexTokenManager.State,
--     --#                                                CommandLineData.Content,
--     --#                                                STree.Table,
--     --#                                                Scope,
--     --#                                                Node,
--     --#                                                GlobalComponentData &
--     --#         Scope,
--     --#         NextNode                          from Dictionary.Dict,
--     --#                                                STree.Table,
--     --#                                                Scope,
--     --#                                                Node;
--        is separate;
----------------------------------------------------------------------
   -- Factored out of wf_package_declaration to allow reuse in generic package declaration
   procedure wf_package_specification (Node         : in STree.SyntaxNode;
                                       IdentStr     : in LexTokenManager.Lex_String;
                                       PackSym      : in Dictionary.Symbol;
                                       CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        IdentStr,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_package_declaration (Node         : in STree.SyntaxNode;
                                     CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   ----------------------------------------------------------------------
   -- PNA temporary reduced annotation for phase 1 generics cfr 1340
   procedure wf_generic_package_instantiation (Node  : in STree.SyntaxNode;
                                               Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_generic_declaration (Node         : in STree.SyntaxNode;
                                     CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;
   ----------------------------------------------------------------------
   procedure CheckPackageNeedsBody (Node    : in STree.SyntaxNode;
                                    PackSym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ----------------------------------------------------------------------
   procedure wf_package_body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Scope,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;
   ----------------------------------------------------------------------
   procedure up_wf_package_body (Node  : in     STree.SyntaxNode;
                                 Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        STree.Table &
   --#         Scope                     from *,
   --#                                        Dictionary.Dict &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is separate;
   ----------------------------------------------------------------------
   procedure wf_package_initialization (Node  : in STree.SyntaxNode;
                                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------
   procedure wf_protected_body  (Node     : in     STree.SyntaxNode;
                                 Scope    : in out Dictionary.Scopes;
                                 NextNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         NextNode,
   --#         Scope                     from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------
   procedure up_wf_protected_body (Node  : in     STree.SyntaxNode;
                                   Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Scope                     from *,
   --#                                        Dictionary.Dict;
      is separate;
   ------------------------------------------------------------------------
   procedure wf_task_body  (Node     : in     STree.SyntaxNode;
                            Scope    : in out Dictionary.Scopes;
                            NextNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         NextNode,
   --#         Scope                     from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
      is separate;
   ------------------------------------------------------------------------
   procedure up_wf_task_body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Scope                     from *,
   --#                                        Dictionary.Dict;
   is separate;
   ------------------------------------------------------------------------
   procedure wf_machine_code_insertion (Node  : in STree.SyntaxNode;
                                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is separate;

begin -- CompUnit
   NextNode := TopNode;
   Stack.Init;
   Heap.Initialize (TheHeap);
   RefList.Init (NodeTable);
   --# accept Flow, 10, AggregateStack.State, "Ineffective assignment expected";
   AggregateStack.Init;  -- ineffective AS.S OK

   ComponentManager.Initialise (GlobalComponentData);

   CurrentScope := Dictionary.GlobalScope;

   while NextNode /= STree.NullNode loop
      LastNode := NextNode;
      NodeType := SyntaxNodeType (LastNode);
      case NodeType is
         when SPSymbols.package_declaration
           | SPSymbols.private_package_declaration =>
            ErrorHandler.StartUnit;
            wf_package_declaration (  -- ineffective AS.S OK
                                      LastNode,
                                      CurrentScope);
            ErrorHandler.EndUnit;
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.generic_declaration |
           SPSymbols.private_generic_package_declaration =>
            ErrorHandler.StartUnit;
            wf_generic_declaration (LastNode,
                                    CurrentScope);
            ErrorHandler.EndUnit;
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.generic_package_instantiation =>
            wf_generic_package_instantiation (LastNode,
                                              CurrentScope);
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.generic_subprogram_instantiation =>
            wf_generic_subprogram_instantiation (LastNode, -- ineffective AS.S OK
                                                 CurrentScope);
            NextNode := STree.NullNode; --nothing below here used

         --when SPSymbols.generic_subprogram_body =>
         --   wf_generic_subprogram_body (LastNode,
         --                               CurrentScope,
         --                               NextNode);

         when SPSymbols.package_body =>
            ErrorHandler.StartUnit;
            wf_package_body (LastNode, CurrentScope);
            NextNode := Child_Node (LastNode);

         when SPSymbols.subprogram_body =>
            ErrorHandler.ErrorReset;
            ErrorHandler.StartUnit;
            wf_subprogram_body (  -- ineffective AS.S OK
                                  Node     => LastNode,
                                  Scope    => CurrentScope,
                                  NextNode => NextNode);
            ErrorHandler.GetErrorSeverity (ErrStatus);
            if ErrStatus = ErrorHandler.NoError then
               Stack.Push (False);
            else
               Stack.Push (True);
            end if;

         when SPSymbols.entry_body =>
            ErrorHandler.ErrorReset;
            ErrorHandler.StartUnit;
            wf_entry_body (  -- ineffective AS.S OK
                             Node     => LastNode,
                             Scope    => CurrentScope,
                             NextNode => NextNode);
            ErrorHandler.GetErrorSeverity (ErrStatus);
            if ErrStatus = ErrorHandler.NoError then
               Stack.Push (False);
            else
               Stack.Push (True);
            end if;

         when SPSymbols.task_body =>
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.RavenscarSelected then -- expect stable
            --# end accept;
               ErrorHandler.ErrorReset;
               ErrorHandler.StartUnit;
               wf_task_body (LastNode, -- ineffective AS.S OK
                             CurrentScope,
                             NextNode);
               ErrorHandler.GetErrorSeverity (ErrStatus);
               if ErrStatus = ErrorHandler.NoError then
                  Stack.Push (False);
               else
                  Stack.Push (True);
               end if;

            else -- can't use task except in Ravenscar
               ErrorHandler.SemanticError (850,
                                           ErrorHandler.NoReference,
                                           NodePosition (LastNode),
                                           LexTokenManager.Null_String);
               -- prune tree walk, we don't want to enter this production when it's illegal
               NextNode := STree.NullNode;
            end if;

         when SPSymbols.protected_body =>
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.RavenscarSelected then -- expect stable
            --# end accept;
               ErrorHandler.StartUnit;
               wf_protected_body (LastNode,
                                  CurrentScope,
                                  NextNode); -- null if error in prot bod
            else -- can't use protected type except in Ravenscar
               ErrorHandler.SemanticError (850,
                                           ErrorHandler.NoReference,
                                           NodePosition (LastNode),
                                           LexTokenManager.Null_String);
               -- prune tree walk, we don't want to enter this production when it's illegal
               NextNode := STree.NullNode;
            end if;

         when SPSymbols.body_stub =>
            wf_body_stub (  -- ineffective AS.S OK
                            Node   => LastNode,
                            Scope  => CurrentScope);
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.subunit =>
            wf_subunit (LastNode,
                        --to get
                        CurrentScope);
            NextNode := Child_Node (LastNode);

         when SPSymbols.proof_function_declaration =>
            --# accept Flow, 10, Unused, "Expected ineffective assignment to Unused";
            wf_proof_function_declaration (LastNode, CurrentScope, Unused);
            --# end accept;
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.basic_declarative_item =>
            wf_basic_declarative_item (LastNode, CurrentScope);  -- ineffective AS.S OK
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.renaming_declaration =>
            wf_renaming_declaration (LastNode, CurrentScope);
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.proof_renaming_declaration =>
            wf_proof_renaming_declaration (LastNode, CurrentScope);
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.use_type_clause =>
            Wf_Use_Type_Clause (LastNode);
            NextNode := STree.NullNode; --nothing below here used

         when SPSymbols.sequence_of_statements =>
            Heap.ReportUsage (TheHeap);
            Heap.Initialize (TheHeap);
            RefList.Init (NodeTable);
            WalkStatements (  -- ineffective AS.S OK
                              SeqNode       => LastNode,
                              Scope         => CurrentScope,
                              Table         => NodeTable);
            ComponentManager.ReportUsage (GlobalComponentData);
            NextNode := STree.NullNode; --nothing below here used
         --# end accept;
         when SPSymbols.package_initialization =>
            wf_package_initialization (LastNode, CurrentScope);
            ErrorHandler.ErrorReset;
            NextNode := Child_Node (LastNode);

         when SPSymbols.apragma =>
            wf_pragma (LastNode, CurrentScope);
            NextNode := STree.NullNode;

         when SPSymbols.code_insertion =>
            wf_machine_code_insertion (LastNode, CurrentScope);
            NextNode := STree.NullNode;

            --following series of cases are places where we prune tree walk
            --because there is nothing below node of use to us
         when SPSymbols.with_clause          |
            SPSymbols.code_statement         | -- probably can't reach this one any more
            SPSymbols.context_clause_rep     |
            SPSymbols.refinement_definition =>
            NextNode := STree.NullNode;

            --this tests for whether down loop end has been reached at a terminal
         when others =>
            if NodeType in SPSymbols.SPNonTerminal then
               NextNode := Child_Node (LastNode);
            else
               NextNode := STree.NullNode;
            end if;
      end case;

      if NextNode = STree.NullNode then ------up loop----------
         loop
            NextNode := Next_Sibling (LastNode);
            exit when NextNode /= STree.NullNode;

            NextNode := ParentNode (LastNode);
            exit when NextNode = STree.NullNode;

            LastNode := NextNode;
            NodeType := SyntaxNodeType (LastNode);
            case NodeType is
               when SPSymbols.package_body =>
                  up_wf_package_body (LastNode,
                                       --using and to get
                                      CurrentScope);
                  ErrorHandler.EndUnit;


               when SPSymbols.protected_body =>
                  up_wf_protected_body (LastNode,
                                          --using and to get
                                        CurrentScope);
                  ErrorHandler.EndUnit;

               when SPSymbols.task_body =>
                  up_wf_task_body (LastNode,
                                    --using and to get
                                   CurrentScope);
                  Stack.Pop;
                  ErrorHandler.ErrorReset;
                  ErrorHandler.EndUnit;

               when SPSymbols.entry_body =>
                  CurrentScope := Dictionary.GetEnclosingScope (CurrentScope);
                  Stack.Pop;
                  ErrorHandler.ErrorReset;
                  ErrorHandler.EndUnit;

               when SPSymbols.package_initialization =>
                  StmtNode := Child_Node (LastNode);
                  ErrorHandler.GetErrorSeverity (ErrStatus);
                  if (SyntaxNodeType (StmtNode) /=
                      SPSymbols.hidden_part) and
                     ErrStatus = ErrorHandler.NoError
                  then
                     --# accept Flow, 10, UnusedDataFlowErrorFlag, "Expected ineffective assignment to UnusedDataFlowErrorFlag";
                     FlowAnalyser.FlowAnalyse (Dictionary.GetRegion (CurrentScope),
                                               StmtNode,
                                               NodePosition (Next_Sibling (StmtNode)),
                                               GlobalComponentData,
                                               TheHeap,
                                               NodeTable,
                                               UnusedDataFlowErrorFlag); -- not used for pack inits (until VCs gen added)
                     --# end accept;
                     ErrorHandler.GetErrorSeverity (ErrStatus);

                     if ErrStatus = ErrorHandler.NoError then
                        ErrorHandler.ReportSuccess (NodePosition (Next_Sibling (StmtNode)),
                                                    LexTokenManager.Null_String);
                     end if;
                  end if;

               when SPSymbols.subprogram_implementation =>
                  StmtNode := Child_Node (LastNode);
                  while StmtNode /= STree.NullNode and
                     SyntaxNodeType (StmtNode) /=
                     SPSymbols.sequence_of_statements
                  loop
                     StmtNode := Next_Sibling (StmtNode);
                  end loop;
                  if StmtNode /= STree.NullNode then
                     ErrorHandler.GetErrorSeverity (ErrStatus);
                     SemanticErrorInSubprogramBody := not (ErrStatus = ErrorHandler.NoError and Stack.Top = False);
                     DataFlowErrorInSubprogramBody := False; -- default in case flow analyser not called below
                     if not SemanticErrorInSubprogramBody then
                        -- don't do flow analysis in presence of semantic errors
                        FlowAnalyser.FlowAnalyse (Dictionary.GetRegion (CurrentScope),
                                                  StmtNode,
                                                  NodePosition (Next_Sibling (StmtNode)),
                                                  GlobalComponentData,
                                                  TheHeap,
                                                  NodeTable,
                                                   -- to get
                                                  DataFlowErrorInSubprogramBody);
                        ErrorHandler.GetErrorSeverity (ErrStatus);

                        if ErrStatus = ErrorHandler.NoError then
                           ErrorHandler.ReportSuccess (NodePosition (Next_Sibling (StmtNode)),
                                                       Dictionary.GetSimpleName (Dictionary.GetRegion (CurrentScope)));
                        end if;
                        ErrorHandler.FlushEchoMessages;
                     end if;
                     -- Call VCG regardless of error status (used to be inside above IF statement).  Instead, we pass
                     -- the error status to the VCG which generates a false VC (or null PF) for subprograms
                     -- containing static semantic errors
                     VCG.GenerateVCs (StmtNode,
                                      CurrentScope,
                                      Do_VCG,
                                      NodePosition (Next_Sibling (StmtNode)),
                                      TheHeap,
                                      SemanticErrorInSubprogramBody,
                                      DataFlowErrorInSubprogramBody);
                  end if;

               when SPSymbols.subprogram_body =>
                  up_wf_subprogram_body (LastNode,
                                          -- using and to get
                                         CurrentScope);
                  Stack.Pop;
                  ErrorHandler.ErrorReset;
                  ErrorHandler.EndUnit;

               when SPSymbols.main_program_declaration =>
                  if (Dictionary.GetThePartition /= Dictionary.NullSymbol) and then
                    Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract,
                                                                Dictionary.GetThePartition) then
                     FlowAnalyser.FlowAnalysePartition (LastNode,
                                                        TheHeap);
                  end if;

               when others =>
                  null;
            end case;
         end loop;
      end if;
   end loop;

   Heap.ReportUsage (TheHeap);

   --# accept Flow, 33, Unused, "Expected Unused to be neither referenced nor exported" &
   --#        Flow, 33, UnusedDataFlowErrorFlag, "Expected UnusedDataFlowErrorFlag to be neither referenced nor exported";
end CompUnit;
