-- $Id: sem-compunit-wf_variable_declaration.adb 16669 2010-04-01 11:26:15Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit)

procedure wf_variable_declaration (Node               : in STree.SyntaxNode;
                                   EnclosingUnitScope : in Dictionary.Scopes;
                                   DeclarationScope   : in Dictionary.Scopes)
is
   type DeclarationSorts is (InPackage, InSubprogram, InProtectedType);

   DeclarationSort         : DeclarationSorts;
   TypeNode                : STree.SyntaxNode;
   ExpNode                 : STree.SyntaxNode;
   AliasNode               : STree.SyntaxNode;
   VarIsInit               : Boolean;
   IsAliased               : Boolean := False;
   UnwantedSeq             : SeqAlgebra.Seq;
   ExpType                 : ExpRecord;
   TypeSym                 : Dictionary.Symbol;
   UnusedComponentData     : ComponentManager.ComponentData;

   -------------------------------------------------------------------------

   procedure AliasCheck (TypeSym : in Dictionary.Symbol)
   --# global in     AliasNode;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAliased;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AliasNode,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAliased,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TypeSym;
   is
   begin
      if IsAliased and then not Dictionary.TypeIsProtected (TypeSym) then
         ErrorHandler.SemanticError (894,
                                     ErrorHandler.NoReference,
                                     NodePosition (AliasNode),
                                     LexTokenManager.Null_String);

      end if;
   end AliasCheck;

   -------------------------------------------------------------------------

   procedure CheckPODeclaration (Sym       : in     Dictionary.Symbol;
                                 TypeSym   : in     Dictionary.Symbol;
                                 Scope     : in     Dictionary.Scopes;
                                 ErrorNode : in     STree.SyntaxNode;
                                 Valid     :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym,
   --#                                        TypeSym &
   --#         Valid                     from Dictionary.Dict,
   --#                                        Scope,
   --#                                        Sym,
   --#                                        TypeSym;
   is

      procedure CheckForMultipleInstances
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrorNode;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     Sym;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out Valid;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorNode,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        Sym,
      --#                                        TypeSym &
      --#         Valid                     from Dictionary.Dict,
      --#                                        Sym,
      --#                                        TypeSym;
      is
         It          : Dictionary.Iterator;
         RootTypeSym : Dictionary.Symbol;
      begin
         Valid := True;
         RootTypeSym := Dictionary.GetRootType (TypeSym);
         It := Dictionary.FirstVirtualElement (RootTypeSym);
         if It /= Dictionary.NullIterator and then
           Dictionary.GetVirtualElementOwner
           (Dictionary.CurrentSymbol (It)) /= Sym then
            -- Illegal instance. This variable is not the one associated with
            -- the protects list in the own variable clause.
            --
            Valid := False;
            ErrorHandler.SemanticErrorSym (942,
                                           ErrorHandler.NoReference,
                                           NodePosition (ErrorNode),
                                           RootTypeSym,
                                           Scope);
         end if;
      end CheckForMultipleInstances;

      -------------------------------------------------------------------------

      procedure CheckPOOperations
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrorNode;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     Sym;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorNode,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        Sym,
      --#                                        TypeSym;
      is
         SubprogramIt     : Dictionary.Iterator;
         SubprogramSym    : Dictionary.Symbol;
         GlobalVariableIt : Dictionary.Iterator;
      begin
         -- Get the first subprogram for the protected type.
         --
         SubprogramIt := Dictionary.FirstVisibleSubprogram (Dictionary.GetRootType (TypeSym));

         while not Dictionary.IsNullIterator (SubprogramIt) loop

            SubprogramSym := Dictionary.CurrentSymbol (SubprogramIt);

            -- Get the first global variable for this subprogram.
            --
            GlobalVariableIt :=
              Dictionary.FirstGlobalVariable
              (Dictionary.GetAbstraction (SubprogramSym, Scope),
               SubprogramSym);

            -- Check the ceiling priority
            --
            CheckCeilingPriority
              (Sym              => Sym,
               Scope            => Scope,
               CheckList        => GlobalVariableIt,
               PriorityLexValue => Dictionary.GetTypePriority (TypeSym),
               ErrorNode        => ErrorNode);

            while not Dictionary.IsNullIterator (GlobalVariableIt) loop

               if Dictionary.CurrentSymbol (GlobalVariableIt) = Sym then
                  -- A circular dependency has been found
                  --
                  ErrorHandler.SemanticError2
                    (ErrNum    => 916,
                     Reference => ErrorHandler.NoReference,
                     Position  => NodePosition (ErrorNode),
                     IdStr1    => Dictionary.GetSimpleName (Sym),
                     IdStr2    => Dictionary.GetSimpleName (SubprogramSym));
               end if;

               GlobalVariableIt := Dictionary.NextSymbol (GlobalVariableIt);
            end loop;

            -- Get the next subprogram.
            --
            SubprogramIt := Dictionary.NextSymbol (SubprogramIt);
         end loop;
      end CheckPOOperations;

      procedure WarnOfInterruptIDs (Sym   : in Dictionary.Symbol;
                                    Scope : in Dictionary.Scopes)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrorNode;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorNode,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        Sym;
      is
      begin
         if Dictionary.GetHasInterruptProperty (Sym) then
            ErrorHandler.SemanticWarningSym (5,
                                             NodePosition (ErrorNode),
                                             Sym,
                                             Scope);
         end if;
      end WarnOfInterruptIDs;

   begin -- CheckPODeclaration
      if Dictionary.IsLibraryLevel (Scope) then

         if Dictionary.OwnVariableHasType (Sym, Scope) then
            CheckForMultipleInstances;
            CheckPOOperations;
            WarnOfInterruptIDs (Sym, Scope);
         else
            -- The own variable was not type announced.
            Valid := False;
            ErrorHandler.SemanticError (925,
                                        ErrorHandler.NoReference,
                                        NodePosition (ErrorNode),
                                        Dictionary.GetSimpleName (Sym));
         end if;

      else
         Valid := False;

         -- Illegal declaration of protected object
         ErrorHandler.SemanticError (868,
                                     ErrorHandler.NoReference,
                                     NodePosition (ErrorNode),
                                     Dictionary.GetSimpleName (Sym));
      end if;
   end CheckPODeclaration;

   -------------------------------------------------------------------------

   procedure CheckTypeConsistency (Sym       : in     Dictionary.Symbol;
                                   TypeSym   : in     Dictionary.Symbol;
                                   Scope     : in     Dictionary.Scopes;
                                   ErrorNode : in     STree.SyntaxNode;
                                   OkToAdd   :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym,
   --#                                        TypeSym &
   --#         OkToAdd                   from Dictionary.Dict,
   --#                                        Scope,
   --#                                        Sym,
   --#                                        TypeSym;
   is
      AnnouncedType : Dictionary.Symbol;
      RootTypeSym   : Dictionary.Symbol;
      Consistent    : Boolean;
   begin
      AnnouncedType := Dictionary.GetType (Sym);
      RootTypeSym := Dictionary.GetRootType (TypeSym);

      if Dictionary.IsOwnVariable (Sym) then
         if Dictionary.OwnVariableHasType (Sym, Scope) then
            if Dictionary.IsSubtype (TypeSym) then
               if Dictionary.TypeIsProtected (RootTypeSym) then
                  -- Protected types may be declared using a subtype of
                  -- the type used in the own variable declaration.
                  if Dictionary.IsSubtype (Dictionary.GetType (Sym)) then
                     -- The own variable was declared using a subtype.
                     -- Report this error rather than the type mismatch error
                     ErrorHandler.SemanticError (948,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ErrorNode),
                                                 Dictionary.GetSimpleName (Sym));
                     Consistent := True;
                  else
                     Consistent := RootTypeSym = AnnouncedType;
                  end if;
               else
                  -- The types must be the same
                  Consistent := TypeSym = AnnouncedType;
               end if;
            else
               -- The types must be the same
               Consistent := TypeSym = AnnouncedType;
            end if;
         else
            -- No announced type so consistent by default
            Consistent := True;
         end if;
      else -- This is an own task
         Consistent := RootTypeSym = AnnouncedType;
      end if;

      if not Consistent then
         -- Type mismatch between declaration and own variable annotation
         ErrorHandler.SemanticError (22,
                                     ErrorHandler.NoReference,
                                     NodePosition (ErrorNode),
                                     NodeLexString (ErrorNode));
      end if;
      OkToAdd := Consistent;
   end CheckTypeConsistency;

   -------------------------------------------------------------------------

   procedure wf_package_variable (Node,
                                  TypeNode     : in STree.SyntaxNode;
                                  TypeSym      : in Dictionary.Symbol;
                                  CurrentScope : in Dictionary.Scopes)
   --# global in     AliasNode;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     ExpNode;
   --#        in     IsAliased;
   --#        in     LexTokenManager.State;
   --#        in     VarIsInit;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from AliasNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        IsAliased,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeNode,
   --#                                        TypeSym,
   --#                                        VarIsInit &
   --#         Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        IsAliased,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TypeSym,
   --#                                        VarIsInit;
   is
      It          : STree.Iterator;
      NextNode    : STree.SyntaxNode;
      IdentStr    : LexTokenManager.Lex_String;
      Sym         : Dictionary.Symbol;
      RootTypeSym : Dictionary.Symbol;
      Var_Symbol  : Dictionary.Symbol;
      OkToAdd     : Boolean;

      function IsAProtectedOwnVariable (Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.IsOwnVariable (Sym) and then Dictionary.GetOwnVariableProtected (Sym);
      end IsAProtectedOwnVariable;

   begin -- wf_package_variable
      AliasCheck (TypeSym);

      RootTypeSym := Dictionary.GetRootType (TypeSym);

      --# assert True; -- for RTC generation

      It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := GetNode (It);
         OkToAdd  := False;
         IdentStr := NodeLexString (NextNode);
         Sym := Dictionary.LookupItem (Name    => IdentStr,
                                       Scope   => CurrentScope,
                                       Context => Dictionary.ProofContext);
         if Sym = Dictionary.NullSymbol then
            -- package state variable not previously announced
            if Dictionary.TypeIsTask (TypeSym) then
               ErrorHandler.SemanticError (930,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           IdentStr);
            else
               ErrorHandler.SemanticError (151,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           IdentStr);
            end if;
         elsif Dictionary.GetContext (Sym) = Dictionary.ProgramContext then
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         elsif not (Dictionary.IsOwnVariable (Sym) or else
                      Dictionary.IsConstituent (Sym) or else
                      Dictionary.IsOwnTask (Sym)) then
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         elsif Dictionary.IsAbstractOwnVariable (Sym) then
            ErrorHandler.SemanticError (73,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         else
            OkToAdd := True;
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd then
            CheckTaskModifierConsistency
              (TheOwnVarType => Dictionary.GetType (Sym),
               TheVarType => TypeSym,
               ModifierIsTask => Dictionary.IsOwnTask (Sym),
               ErrorNode => NextNode,
               Consistent => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd then
            CheckProtectedModifierConsistency
              (TheType => TypeSym,
               ModifierIsProtected =>
                 Dictionary.IsOwnVariable (Sym) and then
                 Dictionary.GetOwnVariableProtected (Sym),
               ErrorNode => NextNode,
               Consistent => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           ((Dictionary.IsOwnVariable (Sym) or else
               (Dictionary.IsConstituent (Sym)))) then

            CheckSuspendablePropertyConsistency
              (Sym              => Sym,
               TypeSym          => TypeSym,
               IsInSuspendsList => Dictionary.GetIsSuspendable (Sym),
               ErrorNode        => Node,
               Consistent       => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           (Dictionary.IsOwnVariable (Sym) or else
              Dictionary.IsConstituent (Sym)) then

            CheckPriorityPropertyConsistency
              (Sym                     => Sym,
               TypeSym                 => TypeSym,
               PriorityPropertyValue   => Dictionary.GetPriorityProperty (Sym),
               ErrorNode               => NextNode,
               Consistent              => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           (Dictionary.IsOwnVariable (Sym) or else
              Dictionary.IsConstituent (Sym)) then

            CheckInterruptPropertyConsistency
              (HasInterruptProperty => Dictionary.GetHasInterruptProperty (Sym),
               Sym                  => Sym,
               TypeSym              => TypeSym,
               ErrorNode            => NextNode,
               Consistent           => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           ((Dictionary.IsOwnVariable (Sym) and then
               Dictionary.OwnVariableHasType (Sym, CurrentScope)) or else
              (Dictionary.IsOwnTask (Sym))) then

            -- This is a type announced own variable or task own variable.
            -- Check that announced type matches the one used in the declaration.
            --
            CheckTypeConsistency
              (Sym       => Sym,
               TypeSym   => TypeSym,
               Scope     => CurrentScope,
               ErrorNode => NextNode,
               OkToAdd   => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           Dictionary.TypeIsTask (TypeSym) then
            CheckCeilingPriority
              (Sym              => Sym,
               Scope            => CurrentScope,
               CheckList        =>
                 Dictionary.FirstGlobalVariable
                 (Dictionary.IsAbstract,
                  RootTypeSym),
               PriorityLexValue => Dictionary.GetTypePriority (TypeSym),
               ErrorNode        => NextNode);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           VarIsInit and then
           Dictionary.IsOwnVariableOrConstituentWithMode (Sym) and then
           not IsAProtectedOwnVariable (Sym) then
            ErrorHandler.SemanticError (720,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
            OkToAdd := False;
         end if;

         --# assert True; -- for RTC generation

         if Dictionary.FirstVirtualElement (RootTypeSym) /= Dictionary.NullIterator and then
           not Dictionary.TypeIsProtected (RootTypeSym) then
            -- A protects property has been given in an own variable declaration that
            -- announces a type that is not a protected type.
            --
            OkToAdd := False;
            ErrorHandler.SemanticError (937,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        Dictionary.GetSimpleName (Sym));
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           Dictionary.TypeIsProtected (TypeSym) then
            CheckPODeclaration (Sym       => Sym,
                                TypeSym   => TypeSym,
                                Scope     => CurrentScope,
                                ErrorNode => NextNode,
                                Valid     => OkToAdd);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           VarIsInit and then
           UnexpectedInitialization (Sym) then
            ErrorHandler.SemanticError (333,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd and then
           not VarIsInit and then
           Dictionary.PartitionElaborationPolicyIsConcurrent and then
           Dictionary.IsOwnVariable (Sym) and then
           Dictionary.GetOwnVariableMode (Sym) = Dictionary.DefaultMode and then
           (Dictionary.GetOwnVariableProtected (Sym) or else
              Dictionary.IsVirtualElement (Sym)) then
            -- Non moded protected state must be initialized at declaration.
            ErrorHandler.SemanticError (874,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         end if;

         --# assert True; -- for RTC generation

         if OkToAdd then
            --# accept F, 10, Var_Symbol, "Expected ineffective assignment";
            Dictionary.AddVariable (Name          => IdentStr,
                                    TypeMark      => TypeSym,
                                    Initialized   => VarIsInit,
                                    IsAliased     => IsAliased,
                                    ExpNode       => STree.NodeToRef (ExpNode),
                                    TypeReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                          NodePosition (TypeNode)),
                                    Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                    Declaration   => Dictionary.Location'(NodePosition (NextNode),
                                                                          NodePosition (NextNode)),
                                    Scope         => CurrentScope,
                                    Var_Symbol    => Var_Symbol);
            --# end accept;
            if ErrorHandler.Generate_SLI then
               SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                         Parse_Tree     => NextNode,
                                         Symbol         => Sym,
                                         Is_Declaration => True);
            end if;
            -- if we have an in stream, then initially mark it as invalid
            if Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.InMode then
               Dictionary.SetVariableMarkedValid (Sym, False);
            end if;
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => NextNode);
         end if;
         It := STree.NextNode (It);
      end loop;
      --# accept F, 33, Var_Symbol, "Expected unused variable";
   end wf_package_variable;

   -------------------------------------------------------------------------

   procedure wf_protected_element (Node,
                                  TypeNode     : in STree.SyntaxNode;
                                  TypeSym      : in Dictionary.Symbol;
                                  CurrentScope : in Dictionary.Scopes)
   --# global in     AliasNode;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     ExpNode;
   --#        in     IsAliased;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in     VarIsInit;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeNode,
   --#                                        TypeSym,
   --#                                        VarIsInit &
   --#         Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TypeSym,
   --#                                        VarIsInit &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AliasNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        IsAliased,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TypeNode,
   --#                                        TypeSym,
   --#                                        VarIsInit;
   is
      It       : STree.Iterator;
      NextNode : STree.SyntaxNode;
      IdentStr : LexTokenManager.Lex_String;
      Sym      : Dictionary.Symbol;
      OkToAdd  : Boolean;
   begin
      --# assert True; -- for RTC generation

      It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := GetNode (It);
         IdentStr := NodeLexString (NextNode);
         Sym := Dictionary.LookupItem (Name    => IdentStr,
                                       Scope   => CurrentScope,
                                       Context => Dictionary.ProofContext);

         OkToAdd := True;

         if Dictionary.TypeIsProtected (TypeSym) then

            OkToAdd := False;

            -- Illegal declaration of protected object
            ErrorHandler.SemanticError (868,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);

         elsif Dictionary.IsPredefinedSuspensionObjectType (TypeSym) then

            OkToAdd := False;

            -- Illegal declaration of suspension object
            ErrorHandler.SemanticError (901,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);

         elsif Dictionary.TypeIsTask (TypeSym) then

            OkToAdd := False;

            -- illegal declaration of task object.
            ErrorHandler.SemanticError (926,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);

         elsif Sym /= Dictionary.NullSymbol then

            OkToAdd := False;

            -- already exists
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         end if;

         if OkToAdd then
            -- Declaration is legal
            -- First add variable as a refinement constituent of the implicit own variable associated
            -- with the protected type
            Dictionary.AddConstituent (Name                 => IdentStr,
                                       Subject              => Dictionary.GetProtectedTypeOwnVariable
                                         (Dictionary.GetRegion (CurrentScope)),
                                       Mode                 => Dictionary.DefaultMode,
                                       SubjectReference     => Dictionary.Location'(NodePosition (NextNode),
                                                                                    NodePosition (NextNode)),
                                       Comp_Unit            => ContextManager.Ops.CurrentUnit,
                                       ConstituentReference => Dictionary.Location'(NodePosition (NextNode),
                                                                                    NodePosition (NextNode)));
            -- then add the variable itself
            Dictionary.AddVariable (Name          => IdentStr,
                                    TypeMark      => TypeSym,
                                    Initialized   => VarIsInit,
                                    IsAliased     => False, -- not aliased
                                    ExpNode       => STree.NodeToRef (ExpNode),
                                    TypeReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                          NodePosition (TypeNode)),
                                    Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                    Declaration   => Dictionary.Location'(NodePosition (NextNode),
                                                                          NodePosition (NextNode)),
                                    Scope         => CurrentScope,
                                    Var_Symbol    => Sym);
            if ErrorHandler.Generate_SLI then
               SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                         Parse_Tree     => NextNode,
                                         Symbol         => Sym,
                                         Is_Declaration => True);
            end if;
            -- Checking that protected elements are initialized is now done
            -- in wf_protected_type_declaration to allow for justification of
            -- these errors.
         end if;
         It := STree.NextNode (It);
      end loop;
      AliasCheck (TypeSym);
   end wf_protected_element;

   -------------------------------------------------------------------------

   procedure wf_procedure_variable (Node,
                                    TypeNode     : in STree.SyntaxNode;
                                    TypeSym      : in Dictionary.Symbol;
                                    CurrentScope : in Dictionary.Scopes)
   --# global in     AliasNode;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     ExpNode;
   --#        in     IsAliased;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in     VarIsInit;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeNode,
   --#                                        TypeSym,
   --#                                        VarIsInit &
   --#         Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TypeSym,
   --#                                        VarIsInit &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AliasNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        IsAliased,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TypeNode,
   --#                                        TypeSym,
   --#                                        VarIsInit;
   is
      It       : STree.Iterator;
      NextNode : STree.SyntaxNode;
      IdentStr : LexTokenManager.Lex_String;
      Sym      : Dictionary.Symbol;
      OkToAdd  : Boolean;

   begin
      It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := GetNode (It);
         IdentStr := NodeLexString (NextNode);
         Sym := Dictionary.LookupItem (Name    => IdentStr,
                                       Scope   => CurrentScope,
                                       Context => Dictionary.ProofContext);

         OkToAdd := True;

         if Dictionary.TypeIsProtected (TypeSym) then

            OkToAdd := False;

            -- Illegal declaration of protected object
            ErrorHandler.SemanticError (868,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);

         elsif Dictionary.IsPredefinedSuspensionObjectType (TypeSym) then

            OkToAdd := False;

            -- Illegal declaration of suspension object
            ErrorHandler.SemanticError (901,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);

         elsif Dictionary.TypeIsTask (TypeSym) then

            OkToAdd := False;

            -- illegal declaration of task object.
            ErrorHandler.SemanticError (926,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         end if;

         if Sym /= Dictionary.NullSymbol then

            OkToAdd := False;

            -- already exists
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        IdentStr);
         end if;

         if OkToAdd then
            Dictionary.AddVariable (Name          => IdentStr,
                                    TypeMark      => TypeSym,
                                    Initialized   => VarIsInit,
                                    IsAliased     => False, --not aliased
                                    ExpNode       => STree.NodeToRef (ExpNode),
                                    TypeReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                          NodePosition (TypeNode)),
                                    Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                    Declaration   => Dictionary.Location'(NodePosition (NextNode),
                                                                          NodePosition (NextNode)),
                                    Scope         => CurrentScope,
                                    Var_Symbol    => Sym);
            if ErrorHandler.Generate_SLI then
               SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                         Parse_Tree     => NextNode,
                                         Symbol         => Sym,
                                         Is_Declaration => True);
            end if;
         end if;
         It := STree.NextNode (It);
      end loop;
      AliasCheck (TypeSym);
   end wf_procedure_variable;

   -------------------------------------------------------------------------

   function GetDeclarationSort (Scope : Dictionary.Scopes) return DeclarationSorts
   --# global in Dictionary.Dict;
   is
      TheRegion : Dictionary.Symbol;
      Result    : DeclarationSorts;
   begin
      TheRegion := Dictionary.GetRegion (Scope);
      if Dictionary.IsPackage (TheRegion) then
         Result := InPackage;
      elsif Dictionary.IsType (TheRegion) and then Dictionary.TypeIsProtected (TheRegion) then
         Result := InProtectedType;
      else
         -- assume subprogram
         Result := InSubprogram;
      end if;
      return Result;
   end GetDeclarationSort;

   -------------------------------------------------------------------------

   function IsAVariableOfAGenericType (TypeSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.TypeIsGeneric (TypeSym);
   end IsAVariableOfAGenericType;

   -------------------------------------------------------------------------

begin  -- wf_variable_declaration
   -- ASSUME Node = variable_declaration

   Heap.Initialize (TheHeap);

   AliasNode := STree.NullNode;
   DeclarationSort := GetDeclarationSort (DeclarationScope);
   TypeNode :=  Next_Sibling (Child_Node (Node));
   if SyntaxNodeType (TypeNode) = SPSymbols.RWaliased then
      IsAliased := True;
      AliasNode := TypeNode;
      TypeNode := Next_Sibling (TypeNode);
   end if;
   ExpNode := Next_Sibling (TypeNode);

   VarIsInit := False;
   wf_type_mark (TypeNode,
                 EnclosingUnitScope,
                 Dictionary.ProgramContext,
                  --to get
                 TypeSym);

   -- variable initialization
   -- Protected and suspension objects are implicitly initialised on declaration.
   --
   if Dictionary.IsPredefinedSuspensionObjectType (TypeSym) or
        Dictionary.TypeIsProtected (TypeSym) then
      VarIsInit := True;
   end if;

   if ExpNode /= STree.NullNode then
      if IsAVariableOfAGenericType (TypeSym) then
         -- Initialization of such variables not allowed in generic bodies because we rely on VCs
         -- to do range checks that in non-generic situations can be done statically and
         -- no VCs are generated for variable initializations.
         ErrorHandler.SemanticError (651,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      else
         -- Non-generic case, initialization is ok
         VarIsInit := True;
         SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
         ComponentManager.Initialise (UnusedComponentData);
         --# accept Flow, 10, UnusedComponentData, "UnusedComponentData is discarded";
         WalkExpression (ExpNode               => ExpNode,
                         Scope                 => DeclarationScope,
                         TypeContext           => TypeSym,
                         ContextRequiresStatic => False,
                         Result                => ExpType,
                         RefVar                => UnwantedSeq,
                         ComponentData         => UnusedComponentData);
         --# end accept;
         SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
         AssignmentCheck (NodePosition (ExpNode),
                          EnclosingUnitScope,
                          TypeSym,
                          ExpType);
         if not ExpType.IsConstant then
            ErrorHandler.SemanticError (50,
                                        ErrorHandler.NoReference,
                                        NodePosition (ExpNode),
                                        LexTokenManager.Null_String);
         end if;
      end if;
   end if;
   case DeclarationSort is
      when InPackage =>
         wf_package_variable (Child_Node (Node),
                              TypeNode,
                              TypeSym,
                              DeclarationScope);

      when InSubprogram =>
         wf_procedure_variable (Child_Node (Node),
                                TypeNode,
                                TypeSym,
                                DeclarationScope);

      when InProtectedType =>
         wf_protected_element (Child_Node (Node),
                               TypeNode,
                               TypeSym,
                               DeclarationScope);
   end case;

   if Dictionary.IsUnconstrainedArrayType (TypeSym) or else
     Dictionary.IsUnconstrainedTaskType (TypeSym) or else
     Dictionary.IsUnconstrainedProtectedType (TypeSym) then
      ErrorHandler.SemanticError (39,
                                  ErrorHandler.NoReference,
                                  NodePosition (TypeNode),
                                  LexTokenManager.Null_String);
   end if;

   Heap.ReportUsage (TheHeap);
end wf_variable_declaration;
