-- $Id: sem-compunit-wf_type_mark.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview:
-- Starting at node type mark there are 2 possibilities: type mark indicates
-- a type T or a package and type P.T
-- Case 1: type T.   WF if T is a type and T is visible.
-- Case 2: type P.T  WF if P is visible and P is a package and P.T is
--         visible and P.T is a type.
-- Possible errors: Not visible   (semantic error 1 or 754 or 755)
--                  Not a type    (semantic error 63)
--                  Not a package (semantic error 9)
--------------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_type_mark (Node         : in     STree.SyntaxNode;
                        CurrentScope : in     Dictionary.Scopes;
                        Context      : in     Dictionary.Contexts;
                        TypeSym      :    out Dictionary.Symbol)
is

   procedure CheckSymbol (IdNode  : in     STree.SyntaxNode;
                          Sym     : in     Dictionary.Symbol;
                          Prefix  : in     LexTokenManager.Lex_String;
                          TypeSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Prefix,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym &
   --#         TypeSym                   from Dictionary.Dict,
   --#                                        Sym &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        IdNode,
   --#                                        Sym;
   is
   begin
      if Sym = Dictionary.NullSymbol then
         --not declared or visible
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError2 (145,
                                      ErrorHandler.NoReference,
                                      NodePosition (Node),
                                      NodeLexString (IdNode),
                                      Prefix);
      elsif Dictionary.IsTypeMark (Sym) then
         if Dictionary.TypeIsWellformed (Sym) then
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => IdNode);
            TypeSym := Sym;
         else
            TypeSym := Dictionary.GetUnknownTypeMark;
            ErrorHandler.SemanticError2 (145,
                                         ErrorHandler.NoReference,
                                         NodePosition (Node),
                                         NodeLexString (IdNode),
                                         Prefix);
         end if;
      else
         --not a type
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (63,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdNode),
                                     NodeLexString (IdNode));
      end if;
   end CheckSymbol;

   -------------------------------------------------------

   procedure CheckTypeMark (CurrentScope : in     Dictionary.Scopes;
                            TypeSym      :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Context;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         STree.Table,
   --#         TypeSym                   from Context,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
   is
   begin
      CheckSymbol
         (IdNode  => Child_Node (Child_Node (Node)),
          Sym     => Dictionary.LookupItem (Name    => NodeLexString (Child_Node (Child_Node (Node))),
                                            Scope   => CurrentScope,
                                            Context => Context),
          Prefix  => LexTokenManager.Null_String,
          TypeSym => TypeSym);

   end CheckTypeMark;

   -------------------------------------------------------

   procedure CheckDottedTypeMark (CurrentScope : in     Dictionary.Scopes;
                                  TypeSym      :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Context;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Context,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         STree.Table,
   --#         TypeSym                   from CommandLineData.Content,
   --#                                        Context,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
   is
      Sym        : Dictionary.Symbol;
      CurrNode   : STree.SyntaxNode;
      Prefix     : LexTokenManager.Lex_String;
      PrevPrefix : LexTokenManager.Lex_String;
      PrefixOk   : Boolean;
      InPrefix   : Boolean;
      ErrNum     : Natural;

      ----------------------------------------------------------------------------
      -- In the case where a Prefix is not visible, we need to distinguish
      -- between two cases:
      -- 1) Where the Prefix appears in a public child package and might denote
      --    a parent of that package.  In this case, we need to issue a message
      --    saying that the named package needs to be inherited (not NOT withed)
      --    to be visible.
      -- 2) Where the child package may be inherited but the prefix erroneously
      --    includes the grandparent package.
      -- 3) Otherwise.  In these cases, we issue a message saying that the
      --    named package has to be BOTH inherited and withed.
      --
      -- Example 1.  Consider a type mark P.T appearing in a public child A.P.B.C
      -- In this case, we find that the prefix "P" _is_ a potential parent
      -- package, so P needs to be inherited.
      --
      -- Example 2.  Consider a type mark A.P.T appearing in a public child A.P.B.C
      -- In this case, we find that the prefix "P" _is_ a potential parent
      -- package but should not be prefixed by the garndparent A.
      --
      -- Example 3.  Consider a type mark P.T appearing in a public child X.Y.Z
      -- In this case, we find that P cannot be a parent, so P must be
      -- inherited AND withed.
      ----------------------------------------------------------------------------
      function PrefixCanDenoteAnAncestor return Boolean
      --# global in CurrentScope;
      --#        in Dictionary.Dict;
      --#        in LexTokenManager.State;
      --#        in Prefix;
      is
         EP     : Dictionary.Symbol; -- Enclosing package where Prefix appears
         CP     : Dictionary.Symbol; -- Current Parent package
         Result : Boolean;
      begin
         EP := Dictionary.GetEnclosingPackage (CurrentScope);
         CP := Dictionary.GetPackageParent (EP);

         Result := False;
         -- CP = NullSymbol when EP is a library-level package.
         while CP /= Dictionary.NullSymbol loop

            -- If the Prefix matches the current parent, then we're done.
            -- If not, then look for the grand-parent and try again until we
            -- reach library level.
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Prefix,
                                                                    Lex_Str2 => Dictionary.GetSimpleName (CP)) = LexTokenManager.Str_Eq then
               Result := True;
               exit;
            end if;

            CP := Dictionary.GetPackageParent (CP);

         end loop;

         return Result;
      end PrefixCanDenoteAnAncestor;

   begin
      InPrefix := False;
      CurrNode := LastChildOf (Node);
      Prefix := NodeLexString (CurrNode);
      Sym := Dictionary.LookupItem (Name    => Prefix,
                                    Scope   => CurrentScope,
                                    Context => Context);
      loop -- we need a loop to handle multiple prefixes
         if Sym = Dictionary.NullSymbol then
            -- not declared or visible
            TypeSym := Dictionary.GetUnknownTypeMark;

            if PrefixCanDenoteAnAncestor then
               --# accept F, 22, "Invariant expression OK here";
               if InPrefix then
                  ErrNum := 756;
               else
                  ErrNum := 755;
               end if;
               --# end accept;
            else
               ErrNum := 754;
            end if;

            ErrorHandler.SemanticError (ErrNum,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        Prefix);
            exit;
         end if;

         if not Dictionary.IsPackage (Sym) then
            --can't be dotted
            TypeSym := Dictionary.GetUnknownTypeMark;
            ErrorHandler.SemanticError (9,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        Prefix);
            exit;
         end if;

         -- Prefix (Sym) is visible and it's a package
         CheckPackagePrefix (CurrNode,
                             Sym,
                             CurrentScope,
                              --to get
                             PrefixOk);
         if not PrefixOk then
            TypeSym := Dictionary.GetUnknownTypeMark;
            exit;
         end if;
         STree.Set_Node_Lex_String (Sym  => Sym,
                                    Node => CurrNode);
         CurrNode := Next_Sibling (ParentNode (CurrNode));
         Sym := Dictionary.LookupSelectedItem (Prefix   => Sym,
                                               Selector => NodeLexString (CurrNode),
                                               Scope    => CurrentScope,
                                               Context  => Context);
         if Next_Sibling (ParentNode (CurrNode)) = STree.NullNode then
            -- no more identifiers to the right, so we should now have type name
            CheckSymbol (IdNode  => CurrNode,
                         Sym     => Sym,
                         Prefix  => Prefix,
                         TypeSym => TypeSym);
            exit;
         end if;

         -- Check that there are not recursive layers of the
         -- same package name (e.g. A.A.B) as the Dictionary
         -- lookup above will always return the same A
         PrevPrefix := Prefix;
         Prefix := NodeLexString (CurrNode);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => PrevPrefix,
                                                                 Lex_Str2 => Prefix) = LexTokenManager.Str_Eq then
            ErrorHandler.SemanticError (145,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrNode),
                                        NodeLexString (CurrNode));
         end if;

         -- otherwise go round again
         InPrefix := True;
      end loop;
   end CheckDottedTypeMark;

   -------------------------------------------------------

begin -- wf_type_mark
      -- ASSUME Node = type_mark

   if Next_Sibling (Child_Node (Child_Node (Node))) = STree.NullNode then
      CheckTypeMark (CurrentScope, TypeSym);
   else
      CheckDottedTypeMark (CurrentScope, TypeSym);
   end if;
end wf_type_mark;
