-- $Id: sem-compunit-wf_term.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of a
-- term node.  Replaces calls to StaticTerm, BaseTypeTerm and CheckTypeTerm
----------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_term (Node                  : in     STree.SyntaxNode;
                   Scope                 : in     Dictionary.Scopes;
                   EStack                : in out ExpStack.ExpStackType;
                   TStack                : in     TypeContextStack.TStackType;
                   ContextRequiresStatic : in     Boolean;
                   IsAnnotation          : in     Boolean)
is
   Left,
   Right,
   Result    : ExpRecord;
   OpNode    : STree.SyntaxNode;

   Operator  : SPSymbols.SPSymbol;

   POpNode   : STree.SyntaxNode; -- Parent's Operator Node
   POperator : SPSymbols.SPSymbol; -- Parent's Operator Symbol

   LeftPos   : LexTokenManager.Token_Position;
   RightPos  : LexTokenManager.Token_Position;
begin --wf_term
   OpNode := Child_Node (Next_Sibling (Child_Node (Node)));

   if OpNode /= STree.NullNode then
      -- mult_op exists
      Operator := SyntaxNodeType (OpNode);
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left, EStack);
      Result := NullTypeRecord; --safety: we may not set all fields below

      -- do static checks first
      Result.IsConstant   := Left.IsConstant and Right.IsConstant;
      Result.IsStatic     := Left.IsStatic and Right.IsStatic;
      Result.HasOperators := True;
      if Left.IsARange or Right.IsARange then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (90,
                                     ErrorHandler.NoReference,
                                     NodePosition (OpNode),
                                     LexTokenManager.Null_String);
      else -- neither are ranges
         -- now do type compatibility and operator visibility checks
         LeftPos  := NodePosition (Child_Node (Node));
         RightPos := NodePosition
           (Next_Sibling (Next_Sibling (Child_Node (Node))));

         CheckBinaryOperator (Operator          => Operator,
                              Left              => Left,
                              Right             => Right,
                              Scope             => Scope,
                              TStack            => TStack,
                              OpPos             => NodePosition (OpNode),
                              LeftPos           => LeftPos,
                              RightPos          => RightPos,
                              Convert           => True, --including implicit type conversion
                              IsAnnotation      => IsAnnotation,
                              -- using and to get
                              Result            => Result);

         -- Seed OpNode with type to aid selection of operator in VCG
         STree.AddNodeSymbol (OpNode, Result.TypeSymbol);

         CalcBinaryOperator (Node     => Node,
                             Operator => Operator,
                             LeftVal  => Left.Value,
                             RightVal => Right.Value,
                             IsAnnotation => IsAnnotation,
                              --using and to get
                             Result   => Result);
         if (not IsAnnotation) and then
            SyntaxNodeType (ParentNode (Node)) =
            SPSymbols.term
         then
            ----------------------------------------------------------------
            -- If the parent is also a term, then we must have
            -- an unparenthesized expression with two multiplying operators,
            -- such as A * B * C
            --
            -- Here, we issue warning 302 to warn of potential evaluation
            -- order dependency.
            --
            -- We can reduce false-alarm rate here by suppressing the
            -- warning in two specific cases:
            --  a) If the sub-expression under consideration is static
            --     AND the expression as a whole appears in a context
            --     that requires a static expression.  Example: a type
            --     declaration such as
            --       type T is range B * 2 / 3 .. 10;
            --   or
            --  b) A modular-typed expression where the two operators
            --     under consideration are both the same and
            --     commutative.  For example:
            --       A := A * B * C;
            --     where A, B, and C are all of the same modular
            --     (sub-)type.
            --
            -- The same logic is used in wf_simple_expression for
            -- binary adding operators.
            ----------------------------------------------------------------
            POpNode := Child_Node (Next_Sibling (Node));

            POperator := SyntaxNodeType (POpNode);

            if (ContextRequiresStatic and Result.IsStatic)
              or else
              (Dictionary.TypeIsModular (Result.TypeSymbol) and
                 Ops_Are_Same_And_Commutative (Operator, POperator)) then
               null;
            else
               ErrorHandler.SemanticWarning (302,
                                             NodePosition (Node),
                                             LexTokenManager.Null_String);
            end if;
         end if;
      end if;
      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is sued in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.OtherSymbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, EStack);
   end if;
end wf_term;
