-- $Id: sem-compunit-wf_task_body.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure wf_task_body (Node     : in     STree.SyntaxNode;
                        Scope    : in out Dictionary.Scopes;
                        NextNode :    out STree.SyntaxNode)
is
   -- Grammar:
   -- task_body
   --     |
   -- identifier --- procedure_annotation --- subprogram_implementation

   TaskSym   : Dictionary.Symbol;
   IdentNode,
   AnnoNode,
   SubprogramImplementationNode,
   EndNode,
   WithNode   : STree.SyntaxNode;
   OkToAdd,
   InSubunit  : Boolean;
   Hidden     : Hidden_Class;
   TaskScope  : Dictionary.Scopes;
   IdentStr   : LexTokenManager.Lex_String;
   ValidAnnotation : Boolean := False;

   procedure CheckOkToAdd (TypeSym    : in     Dictionary.Symbol;
                           InSubunit  : in     Boolean;
                           OkToAdd    :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     IdentStr;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        IdentStr,
   --#                                        InSubunit,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         OkToAdd                   from Dictionary.Dict,
   --#                                        InSubunit,
   --#                                        TypeSym;
   is
   begin
      OkToAdd := True;
      if InSubunit then
         -- we require a stub but must not have a previous body
         if Dictionary.HasBody (TypeSym) then
            OkToAdd := False;
            ErrorHandler.SemanticError (992,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;

         if not Dictionary.HasBodyStub (TypeSym) then
            OkToAdd := False;
            ErrorHandler.SemanticError (15,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;

      else
         -- we must have neither stub nor previous body
         if Dictionary.HasBody (TypeSym) or else
           Dictionary.HasBodyStub (TypeSym) then
            OkToAdd := False;
            ErrorHandler.SemanticError (992,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;
      end if;
   end CheckOkToAdd;

   ----------------------------------------------------------------------

   function RequiresSecondAnnotation (TaskSym : Dictionary.Symbol)
                                     return Boolean
   --# global in Dictionary.Dict;
   is
      GlobalVar : Dictionary.Symbol;
      Required  : Boolean;
      GlobalItem : Dictionary.Iterator;
      EnclosingRegion : Dictionary.Symbol;

   begin
      Required := False;
      EnclosingRegion := Dictionary.GetRegion (Dictionary.GetScope (TaskSym));
      GlobalItem := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                                    TaskSym);
      while GlobalItem /= Dictionary.NullIterator loop
         GlobalVar := Dictionary.CurrentSymbol (GlobalItem);
         if Dictionary.IsAbstractOwnVariable (GlobalVar) and then
           Dictionary.GetOwner (GlobalVar) = EnclosingRegion
         then
            Required := True;
            exit;
         end if;
         GlobalItem := Dictionary.NextSymbol (GlobalItem);
      end loop;
      return Required;
   end RequiresSecondAnnotation;

   ----------------------------------------------------------------------

   function EmptyConstraint (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
   begin
      return SyntaxNodeType (Node) = SPSymbols.procedure_constraint and then
        Child_Node (Node) = STree.NullNode;
   end EmptyConstraint;

   ----------------------------------------------------------------------

   function EmptyAnnotation return Boolean
   --# global in AnnoNode;
   --#        in STree.Table;
   is
   begin
      return EmptyConstraint (Child_Node (AnnoNode));
   end EmptyAnnotation;

   ----------------------------------------------------------------------

   procedure ProcessAnnotation
   --# global in     AnnoNode;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     Scope;
   --#        in     TaskSym;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#        in out ValidAnnotation;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TaskSym,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TaskSym,
   --#                                        TheHeap &
   --#         ValidAnnotation           from *,
   --#                                        AnnoNode,
   --#                                        STree.Table;
   is
      CurrentNode : STree.SyntaxNode;

      procedure RaiseError (Node : in STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
      begin
         ErrorHandler.SemanticError (990, ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.Null_String);
      end RaiseError;


   begin
      CurrentNode := Child_Node (AnnoNode);
      -- to be legal, CurrentNode must be a moded_global_definition
      if SyntaxNodeType (CurrentNode) = SPSymbols.moded_global_definition then
         CurrentNode := Next_Sibling (CurrentNode);
         if SyntaxNodeType (CurrentNode) = SPSymbols.dependency_relation then
            CurrentNode := Next_Sibling (CurrentNode);
         end if;
         if EmptyConstraint (CurrentNode) then
            ValidAnnotation := True;
            wf_procedure_annotation (Node         => AnnoNode,
                                     CurrentScope => Scope,
                                     SubprogSym   => TaskSym,
                                     FirstSeen    => False);
         else
            RaiseError (CurrentNode);
         end if;
      else
         RaiseError (CurrentNode);
      end if;
   end ProcessAnnotation;

   ----------------------------------------------------------------------

begin -- wf_task_body
   IdentNode := Child_Node (Node);
   AnnoNode := Next_Sibling (IdentNode);
   SubprogramImplementationNode := Next_Sibling (AnnoNode);
   EndNode   := LastSiblingOf (Child_Node (SubprogramImplementationNode));

   Hidden := Body_Hidden_Class (SubprogramImplementationNode);

   IdentStr := NodeLexString (IdentNode);

   -- set up default "pruning" of tree walk in case errors found below
   NextNode := STree.NullNode;

   -- see if we are a subunit or an ordinary in-line declaration
   InSubunit := SyntaxNodeType (ParentNode (ParentNode (Node))) = SPSymbols.subunit;
   -- find any context clause present
   if InSubunit then
      WithNode := Child_Node             -- with_clause
        (Child_Node                      -- context_clause
           (ParentNode                       -- compilation_unit
              (ParentNode                    -- secondary_unit
                 (ParentNode                 -- subunit
                    (ParentNode (Node)))))); -- proper_body
      if SyntaxNodeType (WithNode) /= SPSymbols.with_clause then
         WithNode := STree.NullNode;
      end if;
   else
      WithNode := STree.NullNode;
   end if;

   TaskSym := Dictionary.LookupItem (Name    => IdentStr,
                                     Scope   => Scope,
                                     Context => Dictionary.ProgramContext);
   -- Check that TaskSym is an task type declared in the spec.  Since we are looking up an identifier
   -- not a full, dotted name we can't find any other entry by mistake so a simple check is all that
   -- is needed.
   if Dictionary.IsTaskType (TaskSym) then
      -- The task is valid so far, it may be hidden or it may have a real sequence of statements
      -- see if a body has already been declared etc.
      CheckOkToAdd (TaskSym,
                    InSubunit,
                    OkToAdd);
      if OkToAdd then
         case Hidden is
            when All_Hidden =>
               Dictionary.AddBody (CompilationUnit => TaskSym,
                                   Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                   TheBody         => Dictionary.Location'(NodePosition (Node),
                                                                           NodePosition (Node)),
                                   Hidden          => True);
               ErrorHandler.HiddenText (NodePosition (EndNode),
                                        NodeLexString (IdentNode),
                                        SPSymbols.subprogram_implementation);
            when Not_Hidden =>
               Dictionary.AddBody (CompilationUnit => TaskSym,
                                   Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                   TheBody         => Dictionary.Location'(NodePosition (Node),
                                                                           NodePosition (Node)),
                                   Hidden          => False);
            when Handler_Hidden =>
               Dictionary.AddBody (CompilationUnit => TaskSym,
                                   Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                   TheBody         => Dictionary.Location'(NodePosition (Node),
                                                                           NodePosition (Node)),
                                   Hidden          => False);
               ErrorHandler.HiddenHandler (NodePosition (EndNode),
                                           NodeLexString (IdentNode),
                                           SPSymbols.task_body);
         end case;

         TaskScope := Dictionary.LocalScope (TaskSym);

         -- process context clause if present
         if WithNode /= STree.NullNode then
            wf_context_clause (ParentNode (WithNode),
                               TaskSym,
                               TaskScope);
         end if;

         -- check annotation
         if InSubunit then
            -- no anno expected
            if not EmptyAnnotation then
               ErrorHandler.SemanticError (155, ErrorHandler.NoReference,
                                           NodePosition (AnnoNode),
                                           IdentStr);
            else
               STree.Set_Node_Lex_String (Sym  => TaskSym,
                                          Node => IdentNode);
            end if;
         else -- not in subunit, anno may be needed
            if RequiresSecondAnnotation (TaskSym) then
               if EmptyAnnotation then
                  ErrorHandler.SemanticError (154, ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              IdentStr);
               else -- anno present and required
                  STree.Set_Node_Lex_String (Sym  => TaskSym,
                                             Node => IdentNode);
                  ProcessAnnotation;
               end if;
            else -- second anno not required
               if not EmptyAnnotation then
                  ErrorHandler.SemanticError (155, ErrorHandler.NoReference,
                                              NodePosition (AnnoNode),
                                              IdentStr);
               else
                  STree.Set_Node_Lex_String (Sym  => TaskSym,
                                             Node => IdentNode);
               end if;
            end if;
         end if;

         -- set up scope for rest of tree walk
         Scope := TaskScope;

         --set up next node for rest of tree walk
         NextNode := SubprogramImplementationNode;
      end if;
   else
      -- not a valid Task
      ErrorHandler.SemanticError (991,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  NodeLexString (IdentNode));
   end if;

   -- Create "full" derives if annotation is present, valid and DFA is selected.
   -- We know we are in SPARK 95.
   if ValidAnnotation and then
     not CommandLineData.Content.DoInformationFlow then
      CreateFullSubProgDependency (Node,
                                   TaskSym,
                                   Dictionary.IsRefined);
   end if;

   -- Check closing identifier if present (i.e. not hidden)
   if SyntaxNodeType (EndNode) = SPSymbols.designator then
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (IdentNode),
                                                              Lex_Str2 => NodeLexString (Child_Node (EndNode))) /= LexTokenManager.Str_Eq then
         ErrorHandler.SemanticError (58,
                                     ErrorHandler.NoReference,
                                     NodePosition (EndNode),
                                     NodeLexString (IdentNode));
      end if;
   end if;

end wf_task_body;
