-- $Id: sem-compunit-wf_subtype_declaration.adb 16669 2010-04-01 11:26:15Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit)

procedure wf_subtype_declaration (Node  : in STree.SyntaxNode;
                                  Scope : in Dictionary.Scopes)
is
   type RealType is (IsFloating, IsFixed);

   TypeNode,
   IdentNode,
   ConstraintNode  : STree.SyntaxNode;
   ConstraintType  : SPSymbols.SPSymbol;
   IdStr           : LexTokenManager.Lex_String;
   ConstraintFound,
   OkToAdd         : Boolean;
   AllIndexesOK    : Boolean;
   SubtypeSym,
   TypeSym         : Dictionary.Symbol;
   SubtypeIsStatic : Boolean;
   Lower,
   Upper,
   Accuracy        : LexTokenManager.Lex_String;
   Subtype_Symbol  : Dictionary.Symbol;

   -------------------------------------------------------------------

   function IsNamedAssociation (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
   begin
      return SyntaxNodeType (Child_Node (Child_Node (Node))) = SPSymbols.named_argument_association;
   end IsNamedAssociation;

   -------------------------------------------------------------------

   procedure WalkExpressionAsTypeMark (ExpNode       : in     STree.SyntaxNode;
                                       Scope         : in     Dictionary.Scopes;
                                       ConstraintSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         ConstraintSym,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                   from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Statistics.TableUsage     from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ExpNode,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table;
   is
      UnwantedSeq         : SeqAlgebra.Seq;
      UnusedComponentData : ComponentManager.ComponentData;
      ConstraintRecord    : ExpRecord;
   begin
      Heap.Initialize (TheHeap);
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression (ExpNode,
                      Scope,
                      Dictionary.GetUnknownTypeMark,
                      False,
                        --to get
                      ConstraintRecord,
                      UnwantedSeq,
                      UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);

      if ConstraintRecord.IsARange then
         ConstraintSym := ConstraintRecord.TypeSymbol;
      else
         ErrorHandler.SemanticError (95,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
         ConstraintSym := Dictionary.GetUnknownTypeMark;
      end if;
      Heap.ReportUsage (TheHeap);
   end WalkExpressionAsTypeMark;

   -------------------------------------------------------------------

   procedure CheckIndexConstraints (IsString     : in     Boolean;
                                    AllIndexesOK :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ConstraintNode;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     Scope;
   --#        in     SubTypeSym;
   --#        in     TypeSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         TheHeap                   from CommandLineData.Content,
   --#                                        ConstraintNode,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubTypeSym,
   --#                                        TypeSym &
   --#         AllIndexesOK              from CommandLineData.Content,
   --#                                        ConstraintNode,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsString,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubTypeSym,
   --#                                        TypeSym &
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        ConstraintNode,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubTypeSym,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ConstraintNode,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsString,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        SubTypeSym,
   --#                                        TypeSym;
   is
      IndexIterator         : Dictionary.Iterator;
      CurrentConstraintNode : STree.SyntaxNode;
      IndexSym,
      ConstraintSym         : Dictionary.Symbol;
      IndexesFinished,
      ConstraintsFinished   : Boolean;

      ResultOfFirstCheck    : Maths.Value;
      ResultOfLastCheck     : Maths.Value;

      ConstraintFirst,
      ConstraintLast        : LexTokenManager.Lex_String;

      procedure GetFirstIndex (IndexSym        : out Dictionary.Symbol;
                               IndexesFinished : out Boolean)
      --# global in     Dictionary.Dict;
      --#        in     TypeSym;
      --#           out IndexIterator;
      --# derives IndexesFinished,
      --#         IndexIterator,
      --#         IndexSym        from Dictionary.Dict,
      --#                              TypeSym;
      is
      begin
         IndexIterator := Dictionary.FirstArrayIndex (TypeSym);
         if Dictionary.IsNullIterator (IndexIterator) then
            IndexSym := Dictionary.NullSymbol;
            IndexesFinished := True;
         else
            IndexSym := Dictionary.CurrentSymbol (IndexIterator);
            IndexesFinished := False;
         end if;
      end GetFirstIndex;

      procedure GetNextIndex (IndexSym        : out Dictionary.Symbol;
                              IndexesFinished : out Boolean)
      --# global in     Dictionary.Dict;
      --#        in out IndexIterator;
      --# derives IndexesFinished,
      --#         IndexIterator,
      --#         IndexSym        from Dictionary.Dict,
      --#                              IndexIterator;
      is
      begin
         IndexIterator := Dictionary.NextSymbol (IndexIterator);
         if Dictionary.IsNullIterator (IndexIterator) then
            IndexSym := Dictionary.NullSymbol;
            IndexesFinished := True;
         else
            IndexSym := Dictionary.CurrentSymbol (IndexIterator);
            IndexesFinished := False;
         end if;
      end GetNextIndex;

      procedure GetFirstConstraint (ConstraintSym       : out Dictionary.Symbol;
                                    ConstraintsFinished : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ConstraintNode;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     Scope;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#           out AggregateStack.State;
      --#           out CurrentConstraintNode;
      --#           out TheHeap;
      --# derives AggregateStack.State,
      --#         ConstraintSym,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         STree.Table,
      --#         TheHeap                   from CommandLineData.Content,
      --#                                        ConstraintNode,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        STree.Table &
      --#         ConstraintsFinished       from  &
      --#         CurrentConstraintNode     from ConstraintNode,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SLI.State,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ConstraintNode,
      --#                                        ContextManager.Ops.FileHeap,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SLI.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Statistics.TableUsage     from *,
      --#                                        CommandLineData.Content,
      --#                                        ConstraintNode,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        STree.Table;
      is
      begin
         ConstraintsFinished := False;
         CurrentConstraintNode := Child_Node (Child_Node (ConstraintNode));
         while SyntaxNodeType (CurrentConstraintNode) /= SPSymbols.expression
         loop
            CurrentConstraintNode := Child_Node (CurrentConstraintNode);
         end loop;

         WalkExpressionAsTypeMark (CurrentConstraintNode,
                                   Scope,
                                    --to get
                                   ConstraintSym);
      end GetFirstConstraint;

      procedure GetNextConstraint (ConstraintSym       : out Dictionary.Symbol;
                                   ConstraintsFinished : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     Scope;
      --#        in out AggregateStack.State;
      --#        in out CurrentConstraintNode;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives AggregateStack.State,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentConstraintNode,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SLI.State,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.FileHeap,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentConstraintNode,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SLI.State,
      --#                                        SPARK_IO.File_Sys,
      --#                                        STree.Table &
      --#         ConstraintsFinished,
      --#         CurrentConstraintNode     from CurrentConstraintNode,
      --#                                        STree.Table &
      --#         ConstraintSym             from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentConstraintNode,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        STree.Table;
      is
      begin
         CurrentConstraintNode := Next_Sibling (ParentNode (CurrentConstraintNode));
         if CurrentConstraintNode = STree.NullNode then
            ConstraintsFinished := True;
            ConstraintSym := Dictionary.NullSymbol;
         else
            ConstraintsFinished := False;
            WalkExpressionAsTypeMark (CurrentConstraintNode,
                                      Scope,
                                       --to get
                                      ConstraintSym);
         end if;
      end GetNextConstraint;

   begin -- CheckIndexConstraints
      AllIndexesOK := True;
      GetFirstIndex (IndexSym, IndexesFinished);
      GetFirstConstraint (ConstraintSym, ConstraintsFinished);
      while not (IndexesFinished or ConstraintsFinished) loop
         if not Dictionary.CompatibleTypes (Scope, IndexSym, ConstraintSym) then
            ErrorHandler.SemanticErrorSym2 (107,
                                            ErrorHandler.NoReference,
                                            NodePosition (CurrentConstraintNode),
                                            ConstraintSym,
                                            IndexSym,
                                            Scope);
            AllIndexesOK := False;
         end if;

         ConstraintFirst := Dictionary.GetScalarAttributeValue (False,
                                                                LexTokenManager.First_Token,
                                                                ConstraintSym);
         ConstraintLast  := Dictionary.GetScalarAttributeValue (False,
                                                                LexTokenManager.Last_Token,
                                                                ConstraintSym);

         -- Check that ConstraintSym'First is OK wrt IndexSym'First
         --# accept F, 41,         "Expect stable expression here";
         if IsString then
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ConstraintFirst,
                                                                    Lex_Str2 => LexTokenManager.One_Value) /= LexTokenManager.Str_Eq then
               ErrorHandler.SemanticError (417,
                                           ErrorHandler.NoReference,
                                           NodePosition (ConstraintNode),
                                           LexTokenManager.Null_String);
            end if;
         else
            ConstraintCheck (Maths.ValueRep (ConstraintFirst),
                             ResultOfFirstCheck,
                             False, -- cannot be in annotation here
                             IndexSym,
                             NodePosition (CurrentConstraintNode));
            if ResultOfFirstCheck = Maths.NoValue then
               AllIndexesOK := False;
            end if;

         end if;
         --# end accept;

         -- Check that ConstraintSym'Last is OK wrt IndexSym'Last
         ConstraintCheck (Maths.ValueRep (ConstraintLast),
                          ResultOfLastCheck,
                          False, -- cannot be in annotation here
                          IndexSym,
                          NodePosition (CurrentConstraintNode));
         if ResultOfLastCheck = Maths.NoValue then
            AllIndexesOK := False;
         end if;

         Dictionary.AddArrayIndex (TheArrayType       => SubtypeSym,
                                   TheIndexType       => ConstraintSym,
                                   Comp_Unit          => ContextManager.Ops.CurrentUnit,
                                   IndexTypeReference => Dictionary.Location'(NodePosition (CurrentConstraintNode),
                                                                              NodePosition (CurrentConstraintNode)));
         GetNextIndex (IndexSym, IndexesFinished);
         GetNextConstraint (ConstraintSym, ConstraintsFinished);
      end loop;
      if not (IndexesFinished and ConstraintsFinished) then
         ErrorHandler.SemanticError (93,
                                     ErrorHandler.NoReference,
                                     NodePosition (ConstraintNode),
                                     LexTokenManager.Null_String);
         AllIndexesOK := False;
      end if;
   end CheckIndexConstraints;

   -----------------------------------------------------------------------

   procedure CheckRealAccuracy (Node   : in     STree.SyntaxNode;
                                Scope  : in     Dictionary.Scopes;
                                Sort   : in     RealType;
                                Static :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out Accuracy;
   --#           out TheHeap;
   --# derives Accuracy                  from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        Sort,
   --#                                        STree.Table &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        Sort,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Static,
   --#         TheHeap                   from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table;
   is
      ExpNode             : STree.SyntaxNode;
      AccuracyType        : ExpRecord;
      UnwantedSeq         : SeqAlgebra.Seq;
      UnusedComponentData : ComponentManager.ComponentData;

      -------------------------------

      function TypeCorrect (TypeSym : Dictionary.Symbol;
                            Scope   : Dictionary.Scopes;
                            Sort    : RealType) return Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean;

      begin
         if Sort = IsFloating then
            Result := Dictionary.IsIntegerTypeMark (TypeSym,
                                                    Scope);
         else
            Result := Dictionary.IsRealTypeMark (TypeSym,
                                                 Scope);
         end if;
         return Result or Dictionary.IsUnknownTypeMark (TypeSym);
      end TypeCorrect;

      -------------------------------

   begin --CheckRealAccuracy
      Heap.Initialize (TheHeap);
      ComponentManager.Initialise (UnusedComponentData);

      case CommandLineData.Content.LanguageProfile is
         when CommandLineData.SPARK83 =>

            ExpNode :=  Child_Node (Child_Node (Node));
            SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
            --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
            WalkExpression (ExpNode,
                            Scope,
                            Dictionary.GetUnknownTypeMark,
                            True,
                            --to get
                            AccuracyType,
                            UnwantedSeq,
                            UnusedComponentData);
            --# end accept;
            SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
            Maths.StorageRep (AccuracyType.Value, Accuracy);
            if not TypeCorrect (AccuracyType.TypeSymbol,
                                Scope,
                                Sort)
            then
               Accuracy := LexTokenManager.Null_String;
               ErrorHandler.SemanticError (38,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.Null_String);
            end if;
            Static := AccuracyType.IsStatic;

         when CommandLineData.SPARK95 |
           CommandLineData.SPARK2005 =>

            -- reduced accuracy subtypes of reals are not allowed in 95 or 2005
            Static := True; -- to reduce knock-on errors
            Accuracy := LexTokenManager.Null_String;
            ErrorHandler.SemanticError (608,
                                        9,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);

      end case;

      Heap.ReportUsage (TheHeap);
   end CheckRealAccuracy;

   ------------------------------------------------------------------------

   procedure CheckRange (Node    : in     STree.SyntaxNode;
                           -- this is the arange node
                         TypeSym : in     Dictionary.Symbol;
                         Scope   : in     Dictionary.Scopes;
                         Static  : in out Boolean;
                         Lower,
                         Upper   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Lower,
   --#         STree.Table,
   --#         TheHeap,
   --#         Upper                     from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         Static,
   --#         Statistics.TableUsage     from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TypeSym;
   is
      RangeResult               : ExpRecord;
      UnwantedSeq               : SeqAlgebra.Seq;
      UnusedComponentData       : ComponentManager.ComponentData;

      UpperAfterConstraintCheck,
      LowerAfterConstraintCheck : Maths.Value;
      LowerLocal,
      UpperLocal                : LexTokenManager.Lex_String;
      RHSNode                   : STree.SyntaxNode;
   begin
      Heap.Initialize (TheHeap);
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression (Node,
                      Scope,
                      TypeSym,
                      False,
                        --to get
                      RangeResult,
                      UnwantedSeq,
                      UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Static := Static and RangeResult.IsStatic;
      Maths.StorageRep (RangeResult.Value, LowerLocal);
      Maths.StorageRep (RangeResult.RangeRHS, UpperLocal);

      -- check that range is constant
      if not RangeResult.IsConstant then
         LowerLocal := LexTokenManager.Null_String;
         UpperLocal := LexTokenManager.Null_String;
         ErrorHandler.SemanticError (43,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.Null_String);
      end if;
      if not Dictionary.CompatibleTypes (Scope,
                                         RangeResult.TypeSymbol,
                                         TypeSym)
      then
         LowerLocal := LexTokenManager.Null_String;
         UpperLocal := LexTokenManager.Null_String;
         ErrorHandler.SemanticErrorSym2 (107,
                                         ErrorHandler.NoReference,
                                         NodePosition (Node),
                                         RangeResult.TypeSymbol,
                                         TypeSym,
                                         Scope);
      end if;

      -- checks for bounds outside type being constrained
      -- see whether node is attribute or X..Y form and select suitable place
      -- to report errors on upper range bound
      if SyntaxNodeType (Child_Node (Node)) =
         SPSymbols.attribute
      then
         RHSNode := Node;
      else --must be of form X..Y
         RHSNode := Next_Sibling (Child_Node (Node));
      end if;


      ConstraintCheck (Maths.ValueRep (LowerLocal),
                       LowerAfterConstraintCheck,
                       False, -- can't be in annotation here
                       TypeSym,
                       NodePosition (Node));
      Maths.StorageRep (LowerAfterConstraintCheck, LowerLocal);

      ConstraintCheck (Maths.ValueRep (UpperLocal),
                       UpperAfterConstraintCheck,
                       False, -- can't be in annotation here
                       TypeSym,
                       NodePosition (RHSNode));
      Maths.StorageRep (UpperAfterConstraintCheck, UpperLocal);

      Lower := LowerLocal;
      Upper := UpperLocal;
      Heap.ReportUsage (TheHeap);
   end CheckRange;

   -----------------------------------------------------------------------

   procedure CheckRealRange (Node    : in     STree.SyntaxNode;
                              --this is the constraint node
                             TypeSym : in     Dictionary.Symbol;
                             Scope   : in     Dictionary.Scopes;
                             Static  : in out Boolean;
                             Lower,
                             Upper   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Static,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         Lower,
   --#         Upper                     from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TypeSym;
   is
      RangeNode   : STree.SyntaxNode;
   begin
      RangeNode := Next_Sibling (Child_Node (Node));
      if RangeNode = STree.NullNode then
         --no range supplied so range is unchanged from parent type
         Upper := Dictionary.GetScalarAttributeValue (False, --no 'base
                                                      LexTokenManager.Last_Token,
                                                      TypeSym);
         Lower := Dictionary.GetScalarAttributeValue (False, --no 'base
                                                      LexTokenManager.First_Token,
                                                      TypeSym);
      else --a range is supplied
         CheckRange (Child_Node (RangeNode),
                     TypeSym,
                     Scope,
                     Static,
                     Lower,
                     Upper);
      end if;
   end CheckRealRange;

   -----------------------------------------------------------------------

   procedure AddScalarSubtype
   --# global in     Accuracy;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     ErrorHandler.ErrorContext;
   --#        in     IdentNode;
   --#        in     IdStr;
   --#        in     LexTokenManager.State;
   --#        in     Lower;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in     SubTypeIsStatic;
   --#        in     TypeNode;
   --#        in     TypeSym;
   --#        in     Upper;
   --#        in out Dictionary.Dict;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SLI.State         from *,
   --#                                Accuracy,
   --#                                CommandLineData.Content,
   --#                                ContextManager.Ops.UnitHeap,
   --#                                ContextManager.Ops.UnitStack,
   --#                                Dictionary.Dict,
   --#                                ErrorHandler.ErrorContext,
   --#                                IdentNode,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Lower,
   --#                                Scope,
   --#                                STree.Table,
   --#                                SubTypeIsStatic,
   --#                                TypeSym,
   --#                                Upper &
   --#         Dictionary.Dict   from *,
   --#                                Accuracy,
   --#                                ContextManager.Ops.UnitStack,
   --#                                IdentNode,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Lower,
   --#                                Scope,
   --#                                STree.Table,
   --#                                SubTypeIsStatic,
   --#                                TypeSym,
   --#                                Upper &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Accuracy,
   --#                                CommandLineData.Content,
   --#                                ContextManager.Ops.FileHeap,
   --#                                ContextManager.Ops.UnitHeap,
   --#                                ContextManager.Ops.UnitStack,
   --#                                Dictionary.Dict,
   --#                                ErrorHandler.ErrorContext,
   --#                                IdentNode,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Lower,
   --#                                Scope,
   --#                                SLI.State,
   --#                                STree.Table,
   --#                                SubtypeIsStatic,
   --#                                TypeNode,
   --#                                TypeSym,
   --#                                Upper;
   is
      Subtype_Symbol : Dictionary.Symbol;
   begin
      if Dictionary.TypeIsInteger (TypeSym) then
         Dictionary.AddIntegerSubtype (Name            => IdStr,
                                       Static          => SubtypeIsStatic,
                                       Parent          => TypeSym,
                                       ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                               NodePosition (TypeNode)),
                                       Lower           => Lower,
                                       Upper           => Upper,
                                       Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                       Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                               NodePosition (IdentNode)),
                                       Scope           => Scope,
                                       Context         => Dictionary.ProgramContext,
                                       Subtype_Symbol  => Subtype_Symbol);
         if ErrorHandler.Generate_SLI then
            SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                      Parse_Tree     => IdentNode,
                                      Symbol         => Subtype_Symbol,
                                      Is_Declaration => True);
         end if;
      elsif Dictionary.TypeIsModular (TypeSym) then
         Dictionary.AddModularSubtype (Name            => IdStr,
                                       Parent          => TypeSym,
                                       ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                               NodePosition (TypeNode)),
                                       Lower           => Lower,
                                       Upper           => Upper,
                                       Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                       Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                               NodePosition (IdentNode)),
                                       Scope           => Scope,
                                       Context         => Dictionary.ProgramContext,
                                       Subtype_Symbol  => Subtype_Symbol);
         if ErrorHandler.Generate_SLI then
            SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                      Parse_Tree     => IdentNode,
                                      Symbol         => Subtype_Symbol,
                                      Is_Declaration => True);
         end if;
      elsif Dictionary.TypeIsEnumeration (TypeSym) then
         Dictionary.AddEnumerationSubtype (Name            => IdStr,
                                           Static          => SubtypeIsStatic,
                                           Parent          => TypeSym,
                                           ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                                   NodePosition (TypeNode)),
                                           Lower           => Lower,
                                           Upper           => Upper,
                                           Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                           Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                                   NodePosition (IdentNode)),
                                           Scope           => Scope,
                                           Context         => Dictionary.ProgramContext,
                                           Subtype_Symbol  => Subtype_Symbol);
         if ErrorHandler.Generate_SLI then
            SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                      Parse_Tree     => IdentNode,
                                      Symbol         => Subtype_Symbol,
                                      Is_Declaration => True);
         end if;
      elsif Dictionary.TypeIsFloatingPoint (TypeSym) then
         Dictionary.AddFloatingPointSubtype (Name            => IdStr,
                                             Static          => SubtypeIsStatic,
                                             Parent          => TypeSym,
                                             ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                                     NodePosition (TypeNode)),
                                             Lower           => Lower,
                                             Upper           => Upper,
                                             ErrorBound      => Accuracy,
                                             Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                             Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                                     NodePosition (IdentNode)),
                                             Scope           => Scope,
                                             Context         => Dictionary.ProgramContext,
                                             Subtype_Symbol  => Subtype_Symbol);
         if ErrorHandler.Generate_SLI then
            SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                      Parse_Tree     => IdentNode,
                                      Symbol         => Subtype_Symbol,
                                      Is_Declaration => True);
         end if;
      elsif Dictionary.TypeIsFixedPoint (TypeSym) then
         Dictionary.AddFixedPointSubtype (Name            => IdStr,
                                          Static          => SubtypeIsStatic,
                                          Parent          => TypeSym,
                                          ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                                  NodePosition (TypeNode)),
                                          Lower           => Lower,
                                          Upper           => Upper,
                                          ErrorBound      => Accuracy,
                                          Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                          Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                                  NodePosition (IdentNode)),
                                          Scope           => Scope,
                                          Context         => Dictionary.ProgramContext,
                                          Subtype_Symbol  => Subtype_Symbol);
         if ErrorHandler.Generate_SLI then
            SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                      Parse_Tree     => IdentNode,
                                      Symbol         => Subtype_Symbol,
                                      Is_Declaration => True);
         end if;
      end if;
   end AddScalarSubtype;

   -----------------------------------------------------------------------

   function AlreadyDefined (IdentStr : LexTokenManager.Lex_String;
                            Scope    : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   is
      Sym : Dictionary.Symbol;
   begin
      Sym := Dictionary.LookupItem (Name    => IdentStr,
                                    Scope   => Scope,
                                    Context => Dictionary.ProofContext);

      return not (Sym = Dictionary.NullSymbol or else
                  (Dictionary.IsTypeMark (Sym) and then
                   Dictionary.TypeIsAnnounced (Sym) and then
                   not Dictionary.IsDeclared (Sym)));
   end AlreadyDefined;

   -----------------------------------------------------------------------

   function SelectIndexOrDiscriminantError (ConstraintNode : STree.SyntaxNode) return Natural
   --# global in CommandLineData.Content;
   --#        in STree.Table;
   is
      Result : Natural;
   begin
      -- if an index_or_discriminant_constraint is applied to an inapplicable type we need to report and
      -- error.  This function tries to narrow the scope of the error message returned.
      if CommandLineData.RavenscarSelected then
         -- we could be expecting an array, task or protected type
         if IsNamedAssociation (ConstraintNode) then
            -- must be Task or protected
            Result := 891;
         else
            -- could be any of Task, Protected, Array
            Result := 892;
         end if;
      else
         -- can only be an array
         Result := 41;
      end if;
      return Result;
   end SelectIndexOrDiscriminantError;

   -----------------------------------------------------------------------

   procedure Wf_Ravenscar_Subtype (IdStr          : in LexTokenManager.Lex_String;
                                   TypeSym        : in Dictionary.Symbol;
                                   Scope          : in Dictionary.Scopes;
                                   IdNode         : in STree.SyntaxNode;
                                   ConstraintNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ConstraintNode,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IdNode,
   --#                                        IdStr,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ConstraintNode,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdNode,
   --#                                        IdStr,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeSym;
      is separate;

   -----------------------------------------------------------------------

begin --wf_subtype_declaration
      -- ASSUME Node = subtype_declaration

   Lower    := LexTokenManager.Null_String;
   Upper    := LexTokenManager.Null_String;
   Accuracy := LexTokenManager.Null_String;

   IdentNode := Child_Node (Node);
   IdStr  := NodeLexString (IdentNode);
   TypeNode := Child_Node (Next_Sibling (IdentNode));
   ConstraintNode := Child_Node (Next_Sibling (TypeNode));
   ConstraintFound := ConstraintNode /= STree.NullNode;
   if AlreadyDefined (IdStr, Scope) then
      OkToAdd := False;
      ErrorHandler.SemanticError (10,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdStr);
   else
      OkToAdd := True;
   end if;

   wf_type_mark (TypeNode,
                 Scope,
                 Dictionary.ProgramContext,
                  --to get
                 TypeSym);

   if Dictionary.IsPrivateType (TypeSym, Scope) then
      OkToAdd := False;
      ErrorHandler.SemanticError (48,
                                  ErrorHandler.NoReference,
                                  NodePosition (TypeNode),
                                  LexTokenManager.Null_String);
   end if;

   -- Subtypes of generic types are not allowed because we can't check whether the boudns will be valid
   -- when they are instantiated
   if Dictionary.TypeIsGeneric (TypeSym) then
      OkToAdd := False;
      ErrorHandler.SemanticError (652,
                                  ErrorHandler.NoReference,
                                  NodePosition (TypeNode),
                                  LexTokenManager.Null_String);
   end if;

   -- test to prevent Boolean subtype unless full-range
   if ConstraintFound and
      Dictionary.TypeIsBoolean (TypeSym) then
         OkToAdd := False;
         ErrorHandler.SemanticError (412,
                                     15,
                                     NodePosition (ConstraintNode),
                                     LexTokenManager.Null_String);
   end if;

   if OkToAdd and not Dictionary.IsUnknownTypeMark (TypeSym) then
      if ConstraintFound then
         -- there is a constraint node so proceed as before
         ConstraintType := SyntaxNodeType (ConstraintNode);
         if ConstraintType = SPSymbols.index_or_discriminant_constraint then
            if Dictionary.TypeIsArray (TypeSym) then
               if IsNamedAssociation (ConstraintNode) then
                  ErrorHandler.SemanticError (92,
                                              ErrorHandler.NoReference,
                                              NodePosition (ConstraintNode),
                                              LexTokenManager.Null_String);
               else -- positional association is ok
                  if Dictionary.IsUnconstrainedArrayType (TypeSym) then
                     Dictionary.AddArraySubtype (Name            => IdStr,
                                                 Parent          => TypeSym,
                                                 ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                                         NodePosition (TypeNode)),
                                                 Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                                 Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                                         NodePosition (IdentNode)),
                                                 Scope           => Scope,
                                                 Context         => Dictionary.ProgramContext,
                                                 Static          => False,
                                                 --to get
                                                 ArraySubtype    => SubtypeSym);
                     if ErrorHandler.Generate_SLI then
                        SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                                  Parse_Tree     => IdentNode,
                                                  Symbol         => SubtypeSym,
                                                  Is_Declaration => True);
                     end if;
                     CheckIndexConstraints (Dictionary.IsPredefinedStringType (TypeSym),
                                            AllIndexesOK);
                     Dictionary.SetTypeIsWellformed (SubtypeSym, AllIndexesOK);
                  else -- array already constrained
                     ErrorHandler.SemanticError (99,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (TypeNode),
                                                 LexTokenManager.Null_String);
                  end if;
               end if; -- illegal named association fo index_constraint

            elsif Dictionary.IsProtectedType (TypeSym) or else Dictionary.IsTaskType (TypeSym) then
               Wf_Ravenscar_Subtype (IdStr,
                                     TypeSym,
                                     Scope,
                                     IdentNode,
                                     ConstraintNode);

            else
               -- a type has been supplied for which index_or_discriminant_constraint is
               -- not appropriate
               ErrorHandler.SemanticError (SelectIndexOrDiscriminantError (ConstraintNode),
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.Null_String);
            end if;

         else --some scalar subtype expected
            if not Dictionary.TypeIsScalar (TypeSym) then
               ErrorHandler.SemanticError (59,
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.Null_String);
            else
               SubtypeIsStatic := True; -- default value
               if ConstraintType = SPSymbols.range_constraint then
                  CheckRange (Child_Node (ConstraintNode),
                              TypeSym,
                              Scope,
                              --to get
                              SubtypeIsStatic,
                              Lower,
                              Upper);

                  -- if constraint is a range but type is real then no accuracy
                  -- has been supplied so we need to get it from parent
                  if Dictionary.TypeIsFloatingPoint (TypeSym) then
                     Accuracy := Dictionary.GetScalarAttributeValue
                        (False,
                         LexTokenManager.Digits_Token,
                         TypeSym);
                  elsif Dictionary.TypeIsFixedPoint (TypeSym) then
                     Accuracy := Dictionary.GetScalarAttributeValue
                        (False,
                         LexTokenManager.Delta_Token,
                         TypeSym);
                  end if;
                  SubtypeIsStatic := SubtypeIsStatic and
                     Dictionary.IsStatic (TypeSym, Scope);

               elsif ConstraintType = SPSymbols.floating_point_constraint then
                  if not Dictionary.TypeIsFloatingPoint (TypeSym) then
                     ErrorHandler.SemanticError (100,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (TypeNode),
                                                 LexTokenManager.Null_String);
                  else
                     CheckRealAccuracy (ConstraintNode,
                                        Scope,
                                        IsFloating,
                                        SubtypeIsStatic);
                     CheckRealRange (ConstraintNode,
                                     TypeSym,
                                     Scope,
                                     SubtypeIsStatic,
                                     Lower,
                                     Upper);
                  end if;

               elsif ConstraintType = SPSymbols.fixed_point_constraint then
                  if not Dictionary.TypeIsFixedPoint (TypeSym) then
                     ErrorHandler.SemanticError (101,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (TypeNode),
                                                 LexTokenManager.Null_String);
                  else
                     CheckRealAccuracy (ConstraintNode,
                                        Scope,
                                        IsFixed,
                                        SubtypeIsStatic);
                     CheckRealRange (ConstraintNode,
                                     TypeSym,
                                     Scope,
                                     SubtypeIsStatic,
                                     Lower,
                                     Upper);
                  end if;
               end if;
               AddScalarSubtype;
            end if;
         end if;
      else  -- no constraint node present
         if Dictionary.TypeIsScalar (TypeSym) then
            --scalars are allowed without constraints so process them here
            SubtypeIsStatic := Dictionary.IsStatic (TypeSym, Scope);
            Lower := Dictionary.GetScalarAttributeValue (False,
                                                         LexTokenManager.First_Token,
                                                         TypeSym);
            Upper := Dictionary.GetScalarAttributeValue (False,
                                                         LexTokenManager.Last_Token,
                                                         TypeSym);
            if Dictionary.TypeIsFloatingPoint (TypeSym) then
               Accuracy := Dictionary.GetScalarAttributeValue (False,
                                                               LexTokenManager.Digits_Token,
                                                               TypeSym);
            elsif Dictionary.TypeIsFixedPoint (TypeSym) then
               Accuracy := Dictionary.GetScalarAttributeValue (False,
                                                               LexTokenManager.Delta_Token,
                                                               TypeSym);
            end if;
            AddScalarSubtype;
         -- Full-range subtypes are also allowed for non-tagged records.
         elsif Dictionary.TypeIsRecord (TypeSym) and then
           not Dictionary.TypeIsTagged (TypeSym) then
            Dictionary.AddRecordSubtype (Name            => IdStr,
                                         Parent          => TypeSym,
                                         ParentReference => Dictionary.Location'(NodePosition (TypeNode),
                                                                                 NodePosition (TypeNode)),
                                         Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                         Declaration     => Dictionary.Location'(NodePosition (IdentNode),
                                                                                 NodePosition (IdentNode)),
                                         Scope           => Scope,
                                         Context         => Dictionary.ProgramContext,
                                         Subtype_Symbol  => Subtype_Symbol);
            if ErrorHandler.Generate_SLI then
               SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                         Parse_Tree     => IdentNode,
                                         Symbol         => Subtype_Symbol,
                                         Is_Declaration => True);
            end if;
         else --not scalar or non-tagged record so must be illegal
            ErrorHandler.SemanticError (406,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        LexTokenManager.Null_String);
         end if;
      end if;
   end if;
end wf_subtype_declaration;
