-- $Id: sem-compunit-wf_subprogram_declaration.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview
-- Checks sub-program declarations from node subprogram_declaration.  These
-- nodes occur only in package declarations therefore well formation of
-- function_ and procedure_specifications are handled here as a special case
-- rather than using the more complex and general-purpose
-- wf_procedure_specification and wf_function_specification.
-- NOTE 11/6/02
-- Declarations also occur in protected types but this procedure can
-- deal with those as well
--------------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_subprogram_declaration (Node          : in     STree.SyntaxNode;
                                     CurrentScope  : in     Dictionary.Scopes;
                                     TheSubprogSym :    out Dictionary.Symbol)
is
   SpecNode,
   AnnoNode,
   ConstraintNode,
   FormalPartNode    : STree.SyntaxNode;
   SubprogSym        : Dictionary.Symbol;
   Is_Overriding     : Boolean := False;

   ------------------------------------------------------------------------

   -- If we are declaring a subprogram in a package spec and the spec contains
   -- protected types we search each of these to detect re-use of the subprogram
   -- name.  If we don't trap such re-use at this point then we end up with a
   -- legal package spec for which no legal body could be written (since its
   -- implementation would inevitably involve overload resolution of calls made from
   -- within the protected body.  e.g. type PT in package P declares operation K.  Package
   -- P also declares an operation K.  From inside the body of PT, a call to K could refer
   -- to either of the two Ks since both are directly visible.
   function IsDefinedInVisibleProtectedType (Name  : LexTokenManager.Lex_String;
                                             Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   is
      Result : Boolean := False;
      It : Dictionary.Iterator;
   begin
      if Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         It := Dictionary.FirstVisibleProtectedType (Dictionary.GetRegion (Scope));
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.IsDirectlyDefined (Name,
                                                    Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
                                                    Dictionary.ProofContext);
            exit when Result;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
      return Result;
   end IsDefinedInVisibleProtectedType;

   -- ditto for protected types in package private scope
   function IsDefinedInPrivateProtectedType (Name  : LexTokenManager.Lex_String;
                                             Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   is
      Result : Boolean := False;
      It : Dictionary.Iterator;
   begin
      if Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         It := Dictionary.FirstPrivateProtectedType (Dictionary.GetRegion (Scope));
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.IsDirectlyDefined (Name,
                                                    Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
                                                    Dictionary.ProofContext);
            exit when Result;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
      return Result;
   end IsDefinedInPrivateProtectedType;

   ------------------------------------------------------------------------

   procedure CheckProcedureSpecification (Node         : in     STree.SyntaxNode;
                                          CurrentScope : in     Dictionary.Scopes;
                                          SubprogSym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in     SpecNode;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SpecNode,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         SubprogSym                from ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
   is
      IdentStr        : LexTokenManager.Lex_String;
      LocalSubprogSym : Dictionary.Symbol;

   begin
      IdentStr := NodeLexString (Child_Node (Node));
      if Dictionary.IsDefined (IdentStr,
                               CurrentScope,
                               Dictionary.ProofContext)
      then
         ErrorHandler.SemanticError (10,  ErrorHandler.NoReference, --illegal redeclaration
                                     NodePosition (Node),
                                     IdentStr);
         SubprogSym := Dictionary.NullSymbol;
      elsif IsDefinedInVisibleProtectedType (IdentStr,
                                             CurrentScope) or else
        IsDefinedInPrivateProtectedType (IdentStr,
                                         CurrentScope)
      then
         ErrorHandler.SemanticError (988,  ErrorHandler.NoReference, --illegal redeclaration
                                     NodePosition (Node),
                                     IdentStr);
         SubprogSym := Dictionary.NullSymbol;
      else
         Dictionary.AddSubprogram (Name          => IdentStr,
                                   Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                   Specification => Dictionary.Location'(NodePosition (Node),
                                                                         NodePosition (Node)),
                                   Scope         => CurrentScope,
                                   Context       => Dictionary.ProgramContext,
                                   -- to get
                                   Subprogram    => LocalSubprogSym);
         SubprogSym := LocalSubprogSym;
         if SyntaxNodeType (SpecNode) = SPSymbols.entry_specification then
            Dictionary.SetSubprogramIsEntry (LocalSubprogSym);
         end if;
      end if;
   end CheckProcedureSpecification;

   ------------------------------------------------------------------------

   procedure CheckFunctionSpecification (Node         : in     STree.SyntaxNode;
                                         CurrentScope : in     Dictionary.Scopes;
                                         SubprogSym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SubprogSym                from ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
   is
      ReturnTypeNode  : STree.SyntaxNode;
      IdentStr        : LexTokenManager.Lex_String;
      SubprogSymLocal,
      TypeSym         : Dictionary.Symbol;
   begin
      SubprogSymLocal := Dictionary.NullSymbol;
      IdentStr := NodeLexString (Child_Node (Child_Node (Node)));
      if Dictionary.IsDefined (IdentStr,
                               CurrentScope,
                               Dictionary.ProofContext)
      then
         ErrorHandler.SemanticError (10, --illegal redeclaration
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     IdentStr);
      elsif IsDefinedInVisibleProtectedType (IdentStr,
                                             CurrentScope)
      then
         ErrorHandler.SemanticError (988,  ErrorHandler.NoReference, --illegal redeclaration
                                     NodePosition (Node),
                                     IdentStr);
      else
         Dictionary.AddSubprogram (Name          => IdentStr,
                                   Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                   Specification => Dictionary.Location'(NodePosition (Node),
                                                                         NodePosition (Node)),
                                   Scope         => CurrentScope,
                                   Context       => Dictionary.ProgramContext,
                                   -- to get
                                   Subprogram    => SubprogSymLocal);
      end if;
      ReturnTypeNode := LastSiblingOf (Child_Node (Node));
      wf_type_mark (ReturnTypeNode,
                    CurrentScope,
                    Dictionary.ProgramContext,
                     --to get
                    TypeSym);

      if Dictionary.IsUnconstrainedArrayType (TypeSym) then
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (39, --illegal use of unconstrained array type
                                     ErrorHandler.NoReference,
                                     NodePosition (ReturnTypeNode),
                                     NodeLexString (ReturnTypeNode));
      elsif Dictionary.IsPredefinedSuspensionObjectType (TypeSym) or
        Dictionary.TypeIsProtected (TypeSym) then
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (905,
                                     ErrorHandler.NoReference,
                                     NodePosition (ReturnTypeNode),
                                     LexTokenManager.Null_String);

      elsif Dictionary.TypeIsTagged (TypeSym) and then
        (Dictionary.GetScope (TypeSym) = CurrentScope) then
         -- attempt to declare primitive function with controlling return result
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (840,
                                     ErrorHandler.NoReference,
                                     NodePosition (ReturnTypeNode),
                                     LexTokenManager.Null_String);

      end if;

      if SubprogSymLocal /= Dictionary.NullSymbol then
         Dictionary.AddReturnType (TheFunction   => SubprogSymLocal,
                                   TypeMark      => TypeSym,
                                   Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                   TypeReference => Dictionary.Location'(NodePosition (ReturnTypeNode),
                                                                         NodePosition (ReturnTypeNode)));

         -- mark signature as not wellformed if wf_type_mark has returned the unknown type
         if TypeSym = Dictionary.GetUnknownTypeMark then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                            SubprogSymLocal);
         end if;
      end if;
      SubprogSym := SubprogSymLocal;
   end CheckFunctionSpecification;

   ------------------------------------------------------------------------

   procedure GetAnnoAndConNodes (Node           : in     STree.SyntaxNode;
                                 AnnoNode       :    out STree.SyntaxNode;
                                 ConstraintNode :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives AnnoNode,
   --#         ConstraintNode from Node,
   --#                             STree.Table;
   --  pre   SyntaxNodeType (Node) =
   --           SPSymbols.procedure_annotation or
   --        SyntaxNodeType (Node) =
   --           SPSymbols.function_annotation;
   is
      ConstraintNodeLocal : STree.SyntaxNode;
      NodeType : SPSymbols.SPSymbol;
   begin
      ConstraintNodeLocal := Child_Node (Node);
      NodeType := SyntaxNodeType (ConstraintNodeLocal);
      if NodeType = SPSymbols.function_constraint or else
         NodeType = SPSymbols.procedure_constraint
      then
         AnnoNode := STree.NullNode;      --only a constraint found
         ConstraintNode := ConstraintNodeLocal;
      else
         AnnoNode := Node;
         ConstraintNode := LastSiblingOf (ConstraintNodeLocal);
      end if;
   end GetAnnoAndConNodes;

   ------------------------------------------------------------------------

begin -- wf_subprog
      -- ASSUME Node = [overriding_indicator] subprogram_declaration OR
      --               entry_declaration

   SpecNode := Child_Node (Node);

   --
   -- Determine and record in the variable Overriding_Indicator
   -- if the procedure overrides a parent.
   -- In SPARK 2005 "not overriding Procedure ..." is equivalent
   -- to "Procedure ...".
   -- This differs from Ada 2005 where a procedure may override
   -- a parent procedure when no overriding_indicator is present.
   --

   if SyntaxNodeType (SpecNode) = SPSymbols.overriding_indicator then
      if SyntaxNodeType (Child_Node (SpecNode)) = SPSymbols.RWoverriding then
         Is_Overriding := True;
      end if;
      SpecNode := Next_Sibling (SpecNode);
   end if;

   FormalPartNode := Next_Sibling (Child_Node (SpecNode));
   GetAnnoAndConNodes (Next_Sibling (SpecNode),
                        --to get
                       AnnoNode,
                       ConstraintNode);
   if SyntaxNodeType (SpecNode) = SPSymbols.procedure_specification or else
     SyntaxNodeType (SpecNode) = SPSymbols.entry_specification
   then
      CheckProcedureSpecification (SpecNode,
                                   CurrentScope,
                                    --to get
                                   SubprogSym);
      TheSubprogSym := SubprogSym; -- pass back to caller
      if SubprogSym /= Dictionary.NullSymbol then
         if FormalPartNode /= STree.NullNode then
            wf_formal_part (FormalPartNode,
                            CurrentScope,
                            SubprogSym,
                            True,    --procedure cannot already be declared
                            Dictionary.ProgramContext);
         end if;
         if AnnoNode /= STree.NullNode then
            wf_procedure_annotation (AnnoNode,
                                     CurrentScope,
                                     SubprogSym,
                                     True);

            -- no anno is always an error of 83 or if info flow is turned on
         elsif CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 or
            CommandLineData.Content.DoInformationFlow
         then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                            SubprogSym);
            ErrorHandler.SemanticError (154,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (SubprogSym));
         end if;

         -- If we're in 95 or 2005 mode, and -flow=data, then synthesize
         -- a dependency relation for this subprogram.

         if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
            not CommandLineData.Content.DoInformationFlow
         then
            CreateFullSubProgDependency (Node,
                                         SubprogSym,
                                         Dictionary.IsAbstract);
         end if;

         if ConstraintNode /= STree.NullNode then
            wf_procedure_constraint (ConstraintNode,
                                     Dictionary.LocalScope (SubprogSym),
                                     True);
         end if;
      end if;

      CheckNoOverloadingFromTaggedOps (SpecNode,
                                       SubprogSym,
                                       CurrentScope,
                                       Dictionary.IsAbstract,
                                       Is_Overriding);

   elsif SyntaxNodeType (SpecNode) = SPSymbols.function_specification then
      CheckFunctionSpecification (SpecNode,
                                  CurrentScope,
                                    --to get
                                  SubprogSym);
      TheSubprogSym := SubprogSym; -- pass back to caller
      if SubprogSym /= Dictionary.NullSymbol then
         if FormalPartNode /= STree.NullNode and then
            SyntaxNodeType (FormalPartNode) /= SPSymbols.type_mark then
            wf_formal_part (FormalPartNode,
                            CurrentScope,
                            SubprogSym,
                            True,     --procedure cannot already be declared
                            Dictionary.ProgramContext);
         end if;
         if AnnoNode /= STree.NullNode then
            wf_function_annotation (AnnoNode,
                                    CurrentScope,
                                    SubprogSym,
                                    True);
         end if;
         if ConstraintNode /= STree.NullNode then
            wf_function_constraint (ConstraintNode,
                                    Dictionary.LocalScope (SubprogSym),
                                    True);
         end if;
      end if;
      CheckNoOverloadingFromTaggedOps (SpecNode,
                                       SubprogSym,
                                       CurrentScope,
                                       Dictionary.IsAbstract,
                                       Is_Overriding);
   else
      wf_proof_function_declaration (SpecNode,
                                     CurrentScope,
                                       --to get
                                     TheSubprogSym);
   end if;
end wf_subprogram_declaration;
