-- $Id: sem-compunit-wf_simple_expression_opt.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of a
-- simple_expression_opt node.  Replaces calls to StaticSimpleExpression,
-- BaseTypeSimpleExpression and CheckTypeSimpleExpression
----------------------------------------------------------------------------
separate (Sem.CompUnit)
procedure wf_simple_expression_opt (Node         : in     STree.SyntaxNode;
                                    Scope        : in     Dictionary.Scopes;
                                    EStack       : in out ExpStack.ExpStackType;
                                    TStack       : in     TypeContextStack.TStackType;
                                    IsAnnotation : in     Boolean)
is
   Result   : ExpRecord;
   BaseType : Dictionary.Symbol;
   OpNode   : STree.SyntaxNode;
   Operator : SPSymbols.SPSymbol;

   ----------------------------------------------------

   procedure CalcUnaryPlusMinus (Op        : in     SPSymbols.SPSymbol;
                                 Result    : in out ExpRecord)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Op,
   --#                                        Result,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Result                    from *,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Op;
   is
      type ErrLookUp is array (Boolean) of Integer;
      WhichErr : constant ErrLookUp := ErrLookUp'(False => 402, True => 399);

      TheModulusString : LexTokenManager.Lex_String;
      TempArg          : Maths.Value;
      Err              : Maths.ErrorCode;
   begin
      if Op = SPSymbols.minus then
         Maths.Negate (Result.Value);

         if Dictionary.TypeIsModular (Result.TypeSymbol) then

            TheModulusString := Dictionary.GetScalarAttributeValue
               (Base     => False,
                Name     => LexTokenManager.Modulus_Token,
                TypeMark => Result.TypeSymbol);

            TempArg := Result.Value;
            Maths.Modulus (FirstNum  => TempArg,
                           SecondNum => Maths.ValueRep (TheModulusString),
                           -- to get
                           Result    => Result.Value,
                           Ok        => Err); -- Expect ineffective assignment to Err


            case Err is
               when Maths.NoError =>
                  null;

               when Maths.DivideByZero =>
                  ErrorHandler.SemanticError (400, ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);

               when Maths.ConstraintError =>
                  ErrorHandler.SemanticError (WhichErr (IsAnnotation),
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);

               when Maths.OverFlow =>
                  Result.Value := Maths.NoValue;
                  ErrorHandler.SemanticWarning (200,
                                                NodePosition (Node),
                                                LexTokenManager.Null_String);


               when others => -- indicates internal error in maths package
                  SystemErrors.FatalError (SystemErrors.MathError, "in CalcBinaryPlusMinus");
            end case;

         end if;

      end if;
      --only other possibility is unary plus which has no effect
   end CalcUnaryPlusMinus;

   ----------------------------------------

   procedure Resolve_Universal (T : in out Dictionary.Symbol)
   --# global in Dictionary.Dict;
   --#        in TStack;
   --# derives T from *,
   --#                Dictionary.Dict,
   --#                TStack;
   is

      procedure DebugPrint
      --# derives ;
      is
         --# hide DebugPrint;
      begin
         if CommandLineData.Content.Debug.Expressions then
            SPARK_IO.Put_String
              (SPARK_IO.Standard_Output,
               "Wf_Simple_Expression_Opt encounters a universal expression.  Resolving by context to type ",
               0);
            EStrings.Put_Line
              (File  => SPARK_IO.Standard_Output,
               E_Str => LexTokenManager.Lex_String_To_String
                 (Lex_Str => Dictionary.GetSimpleName (TypeContextStack.Top (TStack))));
         end if;
      end DebugPrint;

   begin
      if TypeContextStack.Top (TStack) /= Dictionary.GetUnknownTypeMark then

         if Dictionary.IsUniversalRealType (T) then
            T := TypeContextStack.Top (TStack);
            DebugPrint;
         elsif Dictionary.IsUniversalIntegerType (T) then
            T := TypeContextStack.Top (TStack);
            DebugPrint;
            -- It's tempting to want to do a ConstraintCheck here against
            -- T'Base.  Unfortunately, this can't be done reliably since
            -- Ada95's "preference rule" _might_ kick in and actualy make
            -- a static expression legal that would be rejected by a simple
            -- minded ConstraintCheck.  For example, consider:
            --
            --   type T is range -128 .. 127;
            --   --# assert T'Base is Short_Short_Integer; -- same range!
            --
            --   C : constant T := -128;
            --
            --   Ada95 - legal, owing to preference rule (which the Examiner doesn't implement!)
            -- SPARK95 - legal, owing to imperfect implementation here
            --   Ada83 - illegal (rejected by DEC Ada, for instance)
            -- SPARK83 - accepted (wrongly) owing to imperfect implementation here
            --
            -- So...the only user-visible mistake is an acceptance of illegal Ada
            -- in SPARK83 mode, which is a long-standing problem and only affects SPARK83
            -- projects anyway.  The risk of messing with this code and incorrectly
            -- rejecting _legal_ SPARK83 is so great, that it's best to leave the
            -- current implementation as is.
         end if;

      end if;
   end Resolve_Universal;

begin --wf_simple_expression_opt
   OpNode := Child_Node (Node);
   if SyntaxNodeType (OpNode) = SPSymbols.unary_adding_operator then

      ExpStack.Pop (Result, EStack);

      Resolve_Universal (Result.TypeSymbol);

      if not IsAnnotation then
         STree.AddNodeSymbol (OpNode, Result.TypeSymbol);
      end if;

      BaseType := Dictionary.GetRootType (Result.TypeSymbol);
      Operator := SyntaxNodeType (Child_Node (OpNode));

      if not Dictionary.UnaryOperatorIsDefined (Operator,
                                                BaseType)
      then
         Result := UnknownTypeRecord;
         if Dictionary.IsModularType (BaseType, Scope) then
            ErrorHandler.SemanticError (803,
                                        ErrorHandler.NoReference,
                                        NodePosition (Next_Sibling (OpNode)),
                                        LexTokenManager.Null_String);
         else
            ErrorHandler.SemanticError (40,
                                        ErrorHandler.NoReference,
                                        NodePosition (Next_Sibling (OpNode)),
                                        LexTokenManager.Null_String);
         end if;
      elsif (not IsAnnotation) and then
         not Dictionary.UnaryOperatorIsVisible (Operator,
                                                BaseType,
                                                Scope)
      then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (309,
                                     ErrorHandler.NoReference,
                                     NodePosition (OpNode),
                                     LexTokenManager.Null_String);

      else
         CalcUnaryPlusMinus (Operator,
                             Result);
         Result.TypeSymbol := BaseType;

         -- (if we decide that unary plus is to be ignored for aliasing purposes
         --  then lines below will have to change to reflect this)
         Result.VariableSymbol := Dictionary.NullSymbol;
         Result.IsAVariable := False;
         Result.IsAnEntireVariable := False;
         Result.HasOperators := True;
      end if;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is sued in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.OtherSymbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, EStack);

   end if;
end wf_simple_expression_opt;
