-- $Id: sem-compunit-wf_selected_component.adb 16669 2010-04-01 11:26:15Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit)

procedure wf_selected_component (Node           : in     STree.SyntaxNode;
                                 Scope          : in     Dictionary.Scopes;
                                 EStack         : in out ExpStack.ExpStackType;
                                 RefVar         : in     SeqAlgebra.Seq;
                                 ComponentData  : in out ComponentManager.ComponentData;
                                 Context        : in     Tilde_Context;
                                 IsAnnotation   : in     Boolean)
is
   Sym                 : Dictionary.Symbol;
   SelectorNode        : STree.SyntaxNode;
   Selector            : LexTokenManager.Lex_String;
   NodePos,
   SelectPos           : LexTokenManager.Token_Position;
   TypeInfo            : ExpRecord;
   PrefixOk            : Boolean;
   PrefixComponentNode : ComponentManager.Component;
   Ident_Context       : Dictionary.Contexts;
begin --wf_selected_component
   if IsAnnotation then
      Ident_Context := Dictionary.ProofContext;
   else
      Ident_Context := Dictionary.ProgramContext;
   end if;
   SelectorNode := LastChildOf (Next_Sibling (Child_Node (Node)));
   Selector := NodeLexString (SelectorNode);
   SelectPos := NodePosition (SelectorNode);
   NodePos := NodePosition (Node);

   ExpStack.Pop (TypeInfo, EStack);

   case TypeInfo.Sort is
      when IsUnknown =>
         TypeInfo.ErrorsInExpression := True;
         ExpStack.Push (TypeInfo, EStack);

      when IsPackage =>
         CheckPackagePrefix (Node,
                             TypeInfo.OtherSymbol,
                             Scope,
                              -- to get
                             PrefixOk);
         if PrefixOk then
            Sym := Dictionary.LookupSelectedItem (Prefix   => TypeInfo.OtherSymbol,
                                                  Selector => Selector,
                                                  Scope    => Scope,
                                                  Context  => Ident_Context);
            if Sym = TypeInfo.OtherSymbol then
               -- found P in P such as P.P.P.X when P.X is intended
               Sym := Dictionary.NullSymbol;
            end if;
            if Sym /= Dictionary.NullSymbol then
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => SelectorNode);
               if IsAnnotation and then
                 ErrorHandler.Generate_SLI then
                  SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                            Parse_Tree     => SelectorNode,
                                            Symbol         => Sym,
                                            Is_Declaration => False);
               end if;
            end if;
            StackIdentifier (Sym            => Sym,
                             IdStr          => Selector,
                             Node           => Node,
                              --Loc            => NodePos,
                             Prefix         => TypeInfo.OtherSymbol,
                             Scope          => Scope,
                             EStack         => EStack,
                             RefVar         => RefVar,
                             Dotted         => True,
                             Context        => Context,
                             IsAnnotation   => IsAnnotation);
         else
            ExpStack.Push (UnknownSymbolRecord, EStack);
         end if;

      when IsObject  =>
         if Dictionary.IsRecordTypeMark (TypeInfo.TypeSymbol,
                                         Scope) then
            -- TypeInfo.TypeSymbol here might denote a record subtype,
            -- so find the root type before looking for the selector.
            TypeInfo.TypeSymbol := Dictionary.GetRootType (TypeInfo.TypeSymbol);

            if IsAnnotation then
               Sym := Dictionary.LookupSelectedItem (Prefix   => TypeInfo.TypeSymbol,
                                                     Selector => Selector,
                                                     Scope    => Scope,
                                                     Context  => Dictionary.ProofContext);
            else -- not IsAnnotation
               if TypeInfo.ArgListFound or else
                 Dictionary.IsConstant (TypeInfo.OtherSymbol) or else
                 Dictionary.IsFunction (TypeInfo.OtherSymbol) then
                  -- do not collect any component entities
                  Sym := Dictionary.LookupSelectedItem (Prefix   => TypeInfo.TypeSymbol,
                                                        Selector => Selector,
                                                        Scope    => Scope,
                                                        Context  => Dictionary.ProgramContext);
               else
                  -- do collect component entities
                  PrefixComponentNode := ComponentManager.GetComponentNode
                    (ComponentData,
                     TypeInfo.OtherSymbol);
                  if not ComponentManager.HasChildren (ComponentData,
                                                       PrefixComponentNode)
                  then
                     -- add allchildren of the prefix to the component mananger
                     -- and declare subcomponents for each in the dictionary
                     AddRecordSubComponents (RecordVarSym  => TypeInfo.OtherSymbol,
                                             RecordTypeSym => TypeInfo.TypeSymbol,
                                             ComponentData => ComponentData);
                  end if; -- not Componentmanager.HasChildren
                  -- subcomponent symbol must be in Dictionary here
                  Sym := Dictionary.LookupSelectedItem (Prefix   => TypeInfo.OtherSymbol,
                                                        Selector => Selector,
                                                        Scope    => Scope,
                                                        Context  => Dictionary.ProgramContext);
                  TypeInfo.OtherSymbol := Sym;
               end if;  -- TypeInfo.ArgListFound...
            end if; -- IsAnnotation

            -- If Sym is found, but it's NOT a record component (e.g. it
            -- denotes the name of a type or something), then something
            -- is very wrong.
            if Sym /= Dictionary.NullSymbol and then
              Dictionary.IsRecordComponent (Sym) then
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => SelectorNode);
               TypeInfo.TypeSymbol := Dictionary.GetType (Sym);
               if not IsAnnotation then
                  TypeInfo.IsAnEntireVariable := False;
               elsif ErrorHandler.Generate_SLI then
                  SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                            Parse_Tree     => SelectorNode,
                                            Symbol         => Sym,
                                            Is_Declaration => False);
               end if;
               TypeInfo.IsConstant := False;
               ExpStack.Push (TypeInfo, EStack);
            else
               ExpStack.Push (UnknownSymbolRecord, EStack);
               ErrorHandler.SemanticError (8, ErrorHandler.NoReference,
                                           SelectPos,
                                           Selector);
            end if;

         elsif Dictionary.IsProtectedType (Dictionary.GetRootType (TypeInfo.TypeSymbol)) then
            -- handle protected function call
            Sym := Dictionary.LookupSelectedItem (Prefix   => TypeInfo.OtherSymbol,
                                                  Selector => Selector,
                                                  Scope    => Scope,
                                                  Context  => Ident_Context);
            if Sym /= Dictionary.NullSymbol then
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => SelectorNode);
               if IsAnnotation and then
                 ErrorHandler.Generate_SLI then
                  SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                            Parse_Tree     => SelectorNode,
                                            Symbol         => Sym,
                                            Is_Declaration => False);
               end if;
            end if;
            StackIdentifier (Sym            => Sym,
                             IdStr          => Selector,
                             Node           => Node,
                             Prefix         => TypeInfo.OtherSymbol,
                             Scope          => Scope,
                             EStack         => EStack,
                             RefVar         => RefVar,
                             Dotted         => False, -- not regarded as "dotted" since prefix is PO name not package
                             Context        => Context,
                             IsAnnotation   => IsAnnotation);

         else -- not Dictionary.IsRecordTypeMark
            ExpStack.Push (UnknownSymbolRecord, EStack);
            ErrorHandler.SemanticError (9,
                                        ErrorHandler.NoReference,
                                        NodePos,
                                        Dictionary.GetSimpleName (TypeInfo.OtherSymbol));
         end if;  -- Dictionary.IsRecordTypeMark

      when IsFunction =>
         ExpStack.Push (UnknownSymbolRecord, EStack);
         ErrorHandler.SemanticError (3, ErrorHandler.NoReference,
                                     NodePos,
                                     Dictionary.GetSimpleName (TypeInfo.OtherSymbol));

      when others =>
         ExpStack.Push (UnknownSymbolRecord, EStack);
         ErrorHandler.SemanticError (5, ErrorHandler.NoReference,
                                     NodePos,
                                     Dictionary.GetSimpleName (TypeInfo.OtherSymbol));
   end case;

   if IsAnnotation and then
     Next_Sibling (SelectorNode) /= STree.NullNode then -- handle ~ or % operator
      case SyntaxNodeType (Next_Sibling (SelectorNode)) is
         when SPSymbols.tilde =>
            wf_tilde (Next_Sibling (SelectorNode),
                      Scope,
                      EStack,
                      Context);
         when SPSymbols.percent =>
            wf_percent (Next_Sibling (SelectorNode),
                        Scope,
                        EStack);
         when others =>
            null; -- unreachable unless grammar of annotation_identifier altered
      end case;
   end if;
end wf_selected_component;
