-- $Id: sem-compunit-wf_relation.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of a relation node.
-- Replaces calls to StaticTerm, BaseTypeTerm and CheckTypeTerm
----------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_relation (Node         : in     STree.SyntaxNode;
                       Scope        : in     Dictionary.Scopes;
                       EStack       : in out ExpStack.ExpStackType;
                       TStack       : in     TypeContextStack.TStackType;
                       IsAnnotation : in     Boolean)
is
   NAME_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.name,
                               True  => SPSymbols.annotation_name);

   OpNode      : STree.SyntaxNode;
   LeftNode    : STree.SyntaxNode;
   RightNode   : STree.SyntaxNode;
   Right,
   Left,
   Result      : ExpRecord;
   IdentStr    : LexTokenManager.Lex_String;
   ErrorsFound : Boolean := False;

   ---------------------------------------------------------------

   procedure DoBooleanBinaryOperator (OpNode    : in     STree.SyntaxNode;
                                      LeftNode  : in     STree.SyntaxNode;
                                      RightNode : in     STree.SyntaxNode;
                                      Left,
                                      Right     : in     ExpRecord;
                                      Scope     : in     Dictionary.Scopes;
                                      Result    : in out ExpRecord)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in     TStack;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        Left,
   --#                                        LeftNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        OpNode,
   --#                                        Result,
   --#                                        Right,
   --#                                        RightNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         Result                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        Left,
   --#                                        LexTokenManager.State,
   --#                                        OpNode,
   --#                                        Right,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack;
   is
      Operator  : SPSymbols.SPSymbol;

   begin --DoBooleanBinaryOperator
      Result.IsConstant := Left.IsConstant and Right.IsConstant;
      Result.IsStatic   := Left.IsStatic  and then
         Right.IsStatic and then
         Dictionary.TypeIsScalar (Left.TypeSymbol) and then
         Dictionary.TypeIsScalar (Right.TypeSymbol);
      Result.HasOperators := True;

      if Left.IsARange or Right.IsARange then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (90, ErrorHandler.NoReference,
                                     NodePosition (OpNode),
                                     LexTokenManager.Null_String);
      else -- neither are ranges
         Operator := SyntaxNodeType (OpNode);
         if Operator = SPSymbols.equals or Operator = SPSymbols.not_equal then
            CheckBinaryOperator (Operator          => Operator,
                                 Left              => Left,
                                 Right             => Right,
                                 Scope             => Scope,
                                 TStack            => TStack,
                                 OpPos             => NodePosition (OpNode),
                                 LeftPos           => NodePosition (LeftNode),
                                 RightPos          => NodePosition (RightNode),
                                 Convert           => True, --with implicit type conversion
                                 IsAnnotation      => IsAnnotation,
                                 -- using and to get
                                 Result            => Result);
            if Result /= UnknownTypeRecord then
               if not Dictionary.IsPredefinedStringType
                  (Dictionary.GetRootType (Left.TypeSymbol)) and then
                  not Dictionary.IsPredefinedStringType
                  (Dictionary.GetRootType (Right.TypeSymbol))
               then
                  -- Unconstrained array types only permitted if:
                  -- 1. we are in annotation context, and
                  -- 2. both sides are unconstrained.
                  -- So you are allowed to say, for example, "post X = T'(others => 0)" for a
                  -- subprogram that initializes an unconstrained array or "post X /= Y" where
                  -- both X and Y are compatible unconstrained array types.
                  -- Note that test 2 is almost certainly redundant because if only one side was
                  -- unconstrained then the incompatibility would be detected elsewhere before
                  -- this code was reached.
                  if (Dictionary.IsUnconstrainedArrayType (Left.TypeSymbol) and
                      not (IsAnnotation and Dictionary.IsUnconstrainedArrayType (Right.TypeSymbol)))
                      or else
                     (Dictionary.IsUnconstrainedArrayType (Right.TypeSymbol) and
                      not (IsAnnotation and Dictionary.IsUnconstrainedArrayType (Left.TypeSymbol)))
                  then
                     Result := UnknownTypeRecord;
                     ErrorHandler.SemanticError (39,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (OpNode),
                                                 LexTokenManager.Null_String);

                  elsif  Illegal_Unconstrained (Left.TypeSymbol, Right.TypeSymbol) then
                     Result := UnknownTypeRecord;
                     ErrorHandler.SemanticError (418,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (OpNode),
                                                 LexTokenManager.Null_String);
                  end if;
               end if;

               if Result /= UnknownTypeRecord then
                  CalcBinaryOperator (Node         => Node,
                                      Operator     => Operator,
                                      LeftVal      => Left.Value,
                                      RightVal     => Right.Value,
                                      IsAnnotation => IsAnnotation,
                                       --using and to get
                                      Result       => Result);
               end if;
            end if;

         else --ordering operator
            if Dictionary.IsUnknownTypeMark (Left.TypeSymbol)          or else
               Dictionary.IsUnknownTypeMark (Right.TypeSymbol)         or else
               (Dictionary.IsScalarTypeMark (Left.TypeSymbol, Scope) and
                Dictionary.IsScalarTypeMark (Right.TypeSymbol, Scope)) or else
               (Dictionary.IsPredefinedStringType (Dictionary.GetRootType (Left.TypeSymbol)) and
                Dictionary.IsPredefinedStringType (Dictionary.GetRootType (Right.TypeSymbol))) or else
               (CommandLineData.RavenscarSelected and then
                Dictionary.IsPredefinedTimeType (Left.TypeSymbol) and then
                Dictionary.IsPredefinedTimeType (Right.TypeSymbol))
            then
               CheckBinaryOperator (Operator          => Operator,
                                    Left              => Left,
                                    Right             => Right,
                                    Scope             => Scope,
                                    TStack            => TStack,
                                    OpPos             => NodePosition (OpNode),
                                    LeftPos           => NodePosition (LeftNode),
                                    RightPos          => NodePosition (RightNode),
                                    Convert           => True, --with implicit type conversion
                                    IsAnnotation      => IsAnnotation,
                                    -- using and to get
                                    Result            => Result);

               CalcBinaryOperator (Node         => Node,
                                   Operator     => Operator,
                                   LeftVal      => Left.Value,
                                   RightVal     => Right.Value,
                                   IsAnnotation => IsAnnotation, --770
                                    --using and to get
                                   Result       => Result);
            elsif Dictionary.IsArrayTypeMark (Left.TypeSymbol, Scope) and
               Dictionary.IsArrayTypeMark (Right.TypeSymbol, Scope)
            then
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (51,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           LexTokenManager.Null_String);
            else
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (52,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           LexTokenManager.Null_String);
            end if;
         end if;
      end if;
      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;
   end DoBooleanBinaryOperator;

   ---------------------------------------------------------------

   function MembershipTest (Operator   : SPSymbols.SPSymbol;
                            Val,
                            LowerBound,
                            UpperBound : Maths.Value) return Maths.Value
   is
      Result  : Maths.Value;
      Ok      : Maths.ErrorCode;
   begin
      --# accept Flow, 10, Ok, "Expected ineffective assignment";
      if Operator = SPSymbols.inside then
         Maths.InsideRange (Val, -- flow error: Ok ineffective
                            LowerBound,
                            UpperBound,
                              --to get
                            Result,
                            Ok);
      else
         Maths.OutsideRange (Val, --flow error: Ok ineffective
                             LowerBound,
                             UpperBound,
                              --to get
                             Result,
                             Ok);
      end if;
      --# accept Flow, 33, Ok, "Expected to be neither referenced nor exported";
      return Result;
   end MembershipTest;  -- OK ignored deliberately

   ---------------------------------------------------------------

   function TypeMembershipTest (Operator : SPSymbols.SPSymbol;
                                Left     : ExpRecord;
                                RHtype   : Dictionary.Symbol) return Maths.Value
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in Scope;
   is
      Result : Maths.Value;

      -------------

      function InvertIfOutside (Operator  : SPSymbols.SPSymbol;
                                RawResult : Maths.Value) return Maths.Value
      is
         Result : Maths.Value;

      begin
         Result := RawResult;
         if Operator = SPSymbols.outside then
            Maths.NotOp (Result);
         end if;
         return Result;
      end InvertIfOutside;

      -------------

      function ScalarTypeMembershipTest (Operator : SPSymbols.SPSymbol;
                                         Val      : Maths.Value;
                                         RHtype   : Dictionary.Symbol) return Maths.Value
      --# global in Dictionary.Dict;
      --#        in LexTokenManager.State;
      is
      begin
         return
            MembershipTest (Operator,
                            Val,
                            Maths.ValueRep
                            (Dictionary.GetScalarAttributeValue (False, --not base type
                                                                 LexTokenManager.First_Token,
                                                                 RHtype)),
                            Maths.ValueRep
                            (Dictionary.GetScalarAttributeValue (False, --not base type
                                                                 LexTokenManager.Last_Token,
                                                                 RHtype)));
      end ScalarTypeMembershipTest;

      -------------

      function NonScalarTypeMembershipTest (Operator : SPSymbols.SPSymbol;
                                            LHtype,
                                            RHtype   : Dictionary.Symbol) return Maths.Value
      --# global in Dictionary.Dict;
      --#        in LexTokenManager.State;
      is
         Result : Maths.Value;
      begin
         if Dictionary.TypeIsRecord (RHtype) then
            Result := Maths.TrueValue; --no record subtypes so must be member

         elsif Dictionary.IsUnconstrainedArrayType (RHtype) then
            Result := Maths.TrueValue; --array must be member of its base type

         else --two constrained arrays
            if IndexesMatch (LHtype, RHtype) then
               Result := Maths.TrueValue;
            else
               Result := Maths.FalseValue;
            end if;
         end if;
         return InvertIfOutside (Operator, Result);
      end NonScalarTypeMembershipTest;

      -------------

   begin --TypeMembershipTest
      if Dictionary.IsPrivateType (RHtype, Scope) or else
        Dictionary.TypeIsBoolean (RHtype) then
         Result := InvertIfOutside (Operator, Maths.TrueValue);

      elsif Dictionary.TypeIsScalar (RHtype) then
         Result := ScalarTypeMembershipTest (Operator,
                                             Left.Value,
                                             RHtype);
      else
         Result := NonScalarTypeMembershipTest (Operator,
                                                Left.TypeSymbol,
                                                RHtype);
      end if;
      return Result;
   end TypeMembershipTest;

   ---------------------------------------------------------------

   -- if we have statically evaluated the result plant it for VCG;
   -- otherwise plant the left hand type so we can distinguish Boolean
   -- models from normal inequality models in the VCG
   procedure PlantResult (OpNode : in STree.SyntaxNode;
                          Result : in Maths.Value;
                          LHtype : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in out STree.Table;
   --# derives STree.Table from *,
   --#                          Dictionary.Dict,
   --#                          LHtype,
   --#                          OpNode,
   --#                          Result;
   is
   begin
      if Result = Maths.TrueValue then
         STree.AddNodeSymbol (OpNode, Dictionary.GetTrue);
      elsif Result = Maths.FalseValue then
         STree.AddNodeSymbol (OpNode, Dictionary.GetFalse);
      else
         --no statically evaluated result available so plant type instead
         STree.AddNodeSymbol (OpNode, LHtype);
      end if;
   end PlantResult;

   ---------------------------------------------------------------

begin --wf_relation
   OpNode := Next_Sibling (Child_Node (Node));

   if OpNode /= STree.NullNode then  -- inside, outside or binary bool
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left, EStack);
      Result := NullTypeRecord;
      if SyntaxNodeType (OpNode) = SPSymbols.inside or -- "in" membership test
         SyntaxNodeType (OpNode) = SPSymbols.outside   -- "not in" membership test
      then
         if SyntaxNodeType (Next_Sibling (OpNode)) =
            NAME_LOOKUP (IsAnnotation) then
            if Right.Sort = IsUnknown then
               Result := UnknownTypeRecord;
            elsif (Right.Sort /= IsTypeMark) then
               Result := UnknownTypeRecord;
               IdentStr := Dictionary.GetSimpleName (Right.OtherSymbol);
               if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                                       Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                  ErrorsFound := True;
                  ErrorHandler.SemanticError (95,
                                              ErrorHandler.NoReference,
                                              NodePosition (Next_Sibling (OpNode)),
                                              IdentStr);
               else
                  ErrorsFound := True;
                  ErrorHandler.SemanticError (63,
                                              ErrorHandler.NoReference,
                                              NodePosition (Next_Sibling (OpNode)),
                                              IdentStr);
               end if;
            else
               if Dictionary.CompatibleTypes (Scope,
                                              Left.TypeSymbol,
                                              Right.TypeSymbol)
               then
                  Result.IsConstant := Left.IsConstant;

                  Result.IsStatic   := CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
                     Left.IsStatic;

                  Result.TypeSymbol := Dictionary.GetPredefinedBooleanType;
                  Result.Value := TypeMembershipTest (SyntaxNodeType (OpNode),
                                                      Left,
                                                      Right.TypeSymbol);
                  Result.HasOperators := True;
                  PlantResult (OpNode, Result.Value, Left.TypeSymbol);
                  -- calculate value here depending on bounds of type mark
                  -- obtained from the dictionary
               else -- type mismatch
                  Result := UnknownTypeRecord;
                  ErrorHandler.SemanticError (42,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);
               end if;
            end if;
         else -- its an explicit range
            if Dictionary.CompatibleTypes (Scope,
                                           Left.TypeSymbol,
                                           Right.TypeSymbol) and
               Right.IsARange
            then
               Result.IsConstant := Left.IsConstant and Right.IsConstant;
               Result.IsStatic   := CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
                  Left.IsStatic and then
                  Right.IsStatic;
               Result.TypeSymbol := Dictionary.GetPredefinedBooleanType;
               -- calculate result value here
               Result.Value := MembershipTest (SyntaxNodeType (OpNode),
                                               Left.Value,
                                               Right.Value,
                                               Right.RangeRHS);
               Result.HasOperators := True;
               PlantResult (OpNode, Result.Value, Left.TypeSymbol);

            else -- type mismatch or RHS is not a range
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (42,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.Null_String);
            end if;
         end if;
         Result.ErrorsInExpression := ErrorsFound or
            Result.ErrorsInExpression or
            Left.ErrorsInExpression or
            Right.ErrorsInExpression;

      else -- must be a Boolean binary operation
         LeftNode  := Child_Node (Node);
         RightNode := Next_Sibling (OpNode);

         DoBooleanBinaryOperator (Child_Node (OpNode),
                                  LeftNode,
                                  RightNode,
                                  Left,
                                  Right,
                                  Scope,
                                    --to get
                                  Result);
      end if;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is sued in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.OtherSymbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, EStack);

   end if;
end wf_relation;
