-- $Id: sem-compunit-wf_priority_pragma.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with EStrings;
separate (Sem.CompUnit)
procedure wf_priority_pragma (Node  : in STree.SyntaxNode;
                              Scope : in Dictionary.Scopes)
is
   -- Node is PragmaNode
   --
   -- Grammar:
   --          PragmaNode
   --              |
   --          identifier --- expression
   --
   -- Rules:
   --         1.  Priority may appear in Task, PT or main_program
   --         2.  Interrupt_Priority may only appear in PT or task
   --         3.  Only one may appear
   --         4.  Only priority or interrupt_priority is valid here
   --
   -- Grammar rules ensure that we only call this check from locations where
   -- some form of priority pragma is expected.

   TheRegion               : Dictionary.Symbol;
   IsProtectedType         : Boolean;
   IsTaskType              : Boolean;
   PragmaKind              : Dictionary.RavenscarPragmasWithValue;
   IdNode                  : STree.SyntaxNode;
   ArgumentExpressionNode  : STree.SyntaxNode;
   ValueRep                : LexTokenManager.Lex_String; -- storage rep of value supplied for pragma
   Compatible              : Boolean;

   function ValidLocation return Boolean
   --# global in IsProtectedType;
   --#        in IsTaskType;
   --#        in PragmaKind;
   is
      Result : Boolean;
   begin
      -- Location must be SYNTACTICALLY correct: we need only worry about things like
      -- Interrupt_Priority in main_program

      -- FOR NOW ALLOW PROTECTED TYPES ONLY - NEEDS EXTENDING FOR TASKS & MAIN PROGRAMS
      case PragmaKind is
         when Dictionary.Priority =>
            Result := IsProtectedType or IsTaskType;
         when Dictionary.InterruptPriority =>
            Result := IsProtectedType or IsTaskType;
      end case;
      return Result;
   end ValidLocation;

   ----

   procedure CheckDiscriminant (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     Scope;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      IdNode : STree.SyntaxNode;
      Sym    : Dictionary.Symbol;

      function IsChain (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      is
         CurrentNode : STree.SyntaxNode;
         Result : Boolean := True;
      begin
         CurrentNode := Node;
         while SyntaxNodeType (CurrentNode) /= SPSymbols.expression loop
            Result := Next_Sibling (CurrentNode) = STree.NullNode;
            exit when not Result; -- fail
            CurrentNode := ParentNode (CurrentNode);
         end loop;
         return Result;
      end IsChain;

      ---

   begin
      -- Check that if a discriminant is used, it is not in an expression.
      -- If it is valid mark it as being used to set priority
      IdNode := LastChildOf (Node);
      if SyntaxNodeType (IdNode) = SPSymbols.identifier then
         -- may be a discriminant
         Sym := Dictionary.LookupItem (Name    => NodeLexString (IdNode),
                                       Scope   => Scope,
                                       Context => Dictionary.ProgramContext);
         if Dictionary.IsKnownDiscriminant (Sym) then
            if IsChain (IdNode) then
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => IdNode);
               Dictionary.SetDiscriminantSetsPriority (Sym);
            else
               ErrorHandler.SemanticError (887,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdNode),
                                           LexTokenManager.Null_String);
            end if;
         end if;
      end if;
   end CheckDiscriminant;

   ----

begin -- wf_priority_pragma
   IdNode := Child_Node (Node);
   if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (IdNode),
                                                           Lex_Str2 => LexTokenManager.Priority_Token) = LexTokenManager.Str_Eq or else
     LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (IdNode),
                                                          Lex_Str2 => LexTokenManager.Interrupt_Priority_Token) = LexTokenManager.Str_Eq then
      -- right sort of pragma
      ArgumentExpressionNode := Next_Sibling (IdNode);
      TheRegion := Dictionary.GetRegion (Scope);
      IsProtectedType := Dictionary.IsType (TheRegion) and then Dictionary.TypeIsProtected (TheRegion);
      IsTaskType      := Dictionary.IsType (TheRegion) and then Dictionary.TypeIsTask (TheRegion);
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (IdNode),
                                                              Lex_Str2 => LexTokenManager.Priority_Token) = LexTokenManager.Str_Eq then
         PragmaKind := Dictionary.Priority;
      else
         PragmaKind := Dictionary.InterruptPriority;
      end if;

      if ValidLocation then
         Dictionary.SetTypeHasPragma (TheRegion, PragmaKind);

         wf_priority_value (Node       => ArgumentExpressionNode,
                            PragmaKind => PragmaKind,
                            Context    => Dictionary.ProgramContext,
                            ErrorSym   => TheRegion,
                            Scope      => Scope,
                            ValueRep   => ValueRep,
                            Compatible => Compatible);
         if Compatible then
            -- return ValueRep will either be a valid static value or NullString so we can add it safely
            Dictionary.SetTypePragmaValue (TheRegion,
                                           PragmaKind,
                                           ValueRep);
            -- see if argument is a discriminant and, if it is, mark it in the dicitonary as being
            -- used to set priority (so that we can do checks on actuals supplied in subtypes)
            CheckDiscriminant (ArgumentExpressionNode);
         end if;

      else -- Invalid location
         ErrorHandler.SemanticError (879,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     NodeLexString (IdNode));

      end if;
   else -- not pragma [Interrupt_]Priority
      ErrorHandler.SemanticError (880,
                                  ErrorHandler.NoReference,
                                  NodePosition (Node),
                                  LexTokenManager.Null_String);

   end if;
end wf_priority_pragma;
