-- $Id: sem-compunit-wf_primary.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure wf_primary (Node          : in     STree.SyntaxNode;
                      Scope         : in     Dictionary.Scopes;
                      EStack        : in out ExpStack.ExpStackType;
                      IsAnnotation  : in     Boolean;
                      RefVar        : in     SeqAlgebra.Seq;
                      ComponentData : in out ComponentManager.ComponentData)
is
   TypeInfo : ExpRecord;
   Sym      : Dictionary.Symbol;

   -----------------------------------------------------------------

   procedure CheckReadability (Sym  : in Dictionary.Symbol;
                               Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        Sym;
   is
      EnclosingSym : Dictionary.Symbol;
   begin
      EnclosingSym := Dictionary.GetMostEnclosingObject (Sym);
      if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 and then
         Dictionary.IsSubprogramParameter (EnclosingSym) and then
         Dictionary.GetSubprogramParameterMode (EnclosingSym) = Dictionary.OutMode
      then
         ErrorHandler.SemanticError (171,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (EnclosingSym));
      end if;
   end CheckReadability;

   -----------------------------------------------------------------

   procedure CheckInvalidStreamUse (PrimaryNode : in STree.SyntaxNode;
                                    Sym         : in Dictionary.Symbol;
                                    Scope       : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     EStack;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.State,
   --#                                        PrimaryNode,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        Sym &
   --#         STree.Table               from *,
   --#                                        PrimaryNode,
   --#                                        Sym;
   is
      CurrentNode                    : STree.SyntaxNode;
      AssignmentOrReturn,
      ParameterToUncheckedConversion,
      ConstantDeclaration,
      HasBranches                    : Boolean;
      ErrorNumber                    : Natural := 0;
   begin --CheckInvalidStreamUse
      if Sym /= Dictionary.NullSymbol then -- a stream has been referenced
         -- put symbol of stream into primary node of syntax tree for use by
         -- vcg-producevcs-buildgraph-modelassignmentstatement
         STree.AddNodeSymbol (PrimaryNode, Sym);

         -- now start search for illegal uses
         -- track up syntax tree until expression or simple_expression
         -- below arange is found
         CurrentNode := PrimaryNode;
         loop
            exit when (SyntaxNodeType (CurrentNode) = SPSymbols.expression) or else
              (SyntaxNodeType (CurrentNode) = SPSymbols.simple_expression and then
                 SyntaxNodeType (ParentNode (CurrentNode)) = SPSymbols.arange);

            CurrentNode := ParentNode (CurrentNode);
         end loop;

         -- to be valid the expression node just found must be below an assignment or return
         AssignmentOrReturn := SyntaxNodeType (ParentNode (CurrentNode)) =
           SPSymbols.return_statement or else
           SyntaxNodeType (ParentNode (CurrentNode)) = SPSymbols.assignment_statement;

         -- or, the single actual parameter to an instance of unchecked conversion.
         -- If the expression is below a name_argument_list then we see if there is a record representing the
         -- not-yet-complete invocation of an unchecked_conversion, thus:
         ParameterToUncheckedConversion := SyntaxNodeType
           (ParentNode (ParentNode (CurrentNode))) = SPSymbols.name_argument_list and then
           not ExpStack.IsEmpty (EStack) and then
           Dictionary.IsAnUncheckedConversion (ExpStack.Top (EStack).OtherSymbol);

         ConstantDeclaration := SyntaxNodeType (ParentNode (CurrentNode)) =
           SPSymbols.constant_declaration;

         -- track back down tree to primary node checking for right branches which show
         -- some kind of expression which is not allowed in this context
         HasBranches := False;
         loop
            if Next_Sibling (CurrentNode) /= STree.NullNode then
               HasBranches := True;
               exit;
            end if;

            exit when CurrentNode = PrimaryNode;

            CurrentNode := Child_Node (CurrentNode);
         end loop;

         if Dictionary.IsPredefinedRealTimeClockOperation (Sym) then
            -- Special case for Ada.Real_Time.Clock.
            -- This function MUST be used
            -- 1. directly
            if HasBranches or else
               -- 2. in a library level constant declaration.
              (Dictionary.IsLibraryLevel (Scope) and then
                 not ConstantDeclaration) or else
               -- 3. in an assignment or return statement..
              (not Dictionary.IsLibraryLevel (Scope) and then
                 not AssignmentOrReturn) then
               ErrorNumber := 960;
            end if;

         elsif HasBranches or not (AssignmentOrReturn or else ParameterToUncheckedConversion) then
            -- illegal use of stream variable or function
            if Dictionary.IsFunction (Sym) then
               ErrorNumber := 715;
            else -- variable
               ErrorNumber := 716;
            end if;
         end if;
      end if;
      if ErrorNumber /= 0 then
         ErrorHandler.SemanticErrorSym (ErrorNumber,
                                        ErrorHandler.NoReference,
                                        NodePosition (PrimaryNode),
                                        Sym,
                                        Scope);
      end if;

   end CheckInvalidStreamUse;

   -----------------------------------------------------------------

   procedure CheckInvalidProtectedFunctionUse (PrimaryNode : in STree.SyntaxNode;
                                               Sym         : in Dictionary.Symbol;
                                               Scope       : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        PrimaryNode,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        Sym;
   is
      CurrentNode        : STree.SyntaxNode;
      AssignmentOrReturn,
      HasBranches        : Boolean;

   begin --CheckInvalidProtectedFunctionUse
      if Sym /= Dictionary.NullSymbol then -- a protected state has been referenced

         -- track up syntax tree until expression found
         CurrentNode := PrimaryNode;
         while SyntaxNodeType (CurrentNode) /= SPSymbols.expression loop
            CurrentNode := ParentNode (CurrentNode);
         end loop;

         -- to be valid the expression node just found must be below an assignment or return
         AssignmentOrReturn := SyntaxNodeType (ParentNode (CurrentNode)) =
           SPSymbols.return_statement or else
           SyntaxNodeType (ParentNode (CurrentNode)) = SPSymbols.assignment_statement;

         -- track back down tree to primary node checking for right branches which show
         -- some kind of expression which is not allowed in this context
         HasBranches := False;
         loop
            if Next_Sibling (CurrentNode) /= STree.NullNode then
               HasBranches := True;
               exit;
            end if;

            exit when CurrentNode = PrimaryNode;

            CurrentNode := Child_Node (CurrentNode);
         end loop;

         if HasBranches or not AssignmentOrReturn then
            ErrorHandler.SemanticErrorSym (725,
                                           ErrorHandler.NoReference,
                                           NodePosition (PrimaryNode),
                                           Sym,
                                           Scope);
         end if;
      end if;
   end CheckInvalidProtectedFunctionUse;

   -----------------------------------------------------------------

   function IsDirectFunctionParameter
     (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
      LocalNode : STree.SyntaxNode;
      Result    : Boolean := False;

      ---------------

      function ValidPositionalArgument (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      is
      begin
         return SyntaxNodeType (ParentNode (Node)) =
           SPSymbols.positional_argument_association;
      end ValidPositionalArgument;

      ---------------

      function ValidNamedArgument (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      is
      begin
         return SyntaxNodeType (ParentNode (Node)) =
           SPSymbols.named_argument_association;
      end ValidNamedArgument;

      ---------------

   begin -- IsDirectFunctionParameter
      LocalNode := Node;
      loop
         -- Success case is when we find an expression which has an argument
         -- association above it.  In any case we don't continue looking once
         -- an expression has been found
         if SyntaxNodeType (LocalNode) = SPSymbols.expression then
            Result := ValidPositionalArgument (LocalNode) or else
              ValidNamedArgument (LocalNode);
            exit;
         end if;

         -- failure condition: right branches exist showing that primary
         -- is part of an expresssion
         exit when Next_Sibling (LocalNode) /= STree.NullNode;

         -- failure condition: we are part of a right branch showing that primary
         -- is part of an expresssion
         exit when Next_Sibling (Child_Node (ParentNode (LocalNode))) /=
           STree.NullNode;

         LocalNode := ParentNode (LocalNode);
      end loop;
      return Result;
   end IsDirectFunctionParameter;

   -----------------------------------------------------------------

   function RootWiseConversionRequired (Node      : STree.SyntaxNode;
                                        ActualSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in EStack;
   --#        in STree.Table;
   is
      Result              : Boolean := False;
      PossibleFunctionSym : Dictionary.Symbol;
      ControllingType     : Dictionary.Symbol;
   begin
      -- conversion is required iff
      -- (1) ActualSym is an object or subcomponent of an object
      -- (2) If it is a direct function parameter (i.e. not an expression)
      -- (3) This primary is being processed in the context of function parameter list
      -- (4) The function has a controlling type
      -- (5) The type of ActualSym is an extension of the controlling type
      if Dictionary.IsVariableOrSubcomponent (ActualSym) then
         if IsDirectFunctionParameter (Node) then
            if not ExpStack.IsEmpty (EStack) then -- there may function info available
               PossibleFunctionSym := ExpStack.Top (EStack).OtherSymbol;
               if Dictionary.IsFunction (PossibleFunctionSym) then
                  ControllingType := Dictionary.GetSubprogramControllingType (PossibleFunctionSym);
                  if ControllingType /= Dictionary.NullSymbol then
                     Result := Dictionary.IsAnExtensionOf (ControllingType,
                                                           Dictionary.GetType (ActualSym));
                  end if;
               end if;
            end if;
         end if;
      end if;
      return Result;
   end RootWiseConversionRequired;

   -----------------------------------------------------------------

begin -- wf_primary
   ExpStack.Pop (TypeInfo, EStack);
   Sym := TypeInfo.OtherSymbol;

   case TypeInfo.Sort is
      when IsUnknown =>
         TypeInfo := UnknownTypeRecord;
      when TypeResult =>
         null;
      when IsPackage =>
         TypeInfo := UnknownTypeRecord;
         ErrorHandler.SemanticError (5,
                                     ErrorHandler.NoReference,
                                     NodePosition (LastChildOf (Node)),
                                     Dictionary.GetSimpleName (Sym));
      when IsFunction =>
         ErrorHandler.SemanticError (3,
                                     ErrorHandler.NoReference,
                                     NodePosition (LastChildOf (Node)),
                                     Dictionary.GetSimpleName (Sym));
         TypeInfo.IsStatic   := False;
         TypeInfo.IsConstant := False;
         TypeInfo.IsARange   := False;
         TypeInfo.ErrorsInExpression := True;

      when IsObject =>
         TypeInfo.IsARange   := False;
         if IsAnnotation then
            TypeInfo.IsStatic := Dictionary.IsStatic (Sym, Scope);
            TypeInfo.IsConstant := Dictionary.IsConstant (Sym);
         else
            CheckReadability (Sym, Node);

            if RootWiseConversionRequired (Node, Sym) then
               -- Actual parameter is a variable so we can select the subset of subcomponents
               -- applicable to the root view required.
               -- We can't convert actual to the appropriate subcomponent unless we add them first
               AddRecordSubComponents (RecordVarSym  => Sym,
                                       RecordTypeSym => Dictionary.GetType (Sym),
                                       ComponentData => ComponentData);
               Sym := ConvertTaggedActual (Sym,
                                           Dictionary.GetSubprogramControllingType
                                           (ExpStack.Top (EStack).OtherSymbol));
            end if;
            if Dictionary.IsVariableOrSubcomponent (Sym) then
               SeqAlgebra.AddMember (TheHeap,
                                     RefVar,
                                     Natural (Dictionary.SymbolRef (Sym)));
            end if;
         end if;
      when IsTypeMark =>
         if Dictionary.IsScalarTypeMark (Sym,
                                         Scope)
         then
            TypeInfo.IsStatic := Dictionary.IsStatic (Sym,
                                                      Scope);
            TypeInfo.IsConstant := True;
            TypeInfo.IsARange   := True;
            TypeInfo.Value := Maths.ValueRep
               (Dictionary.GetScalarAttributeValue (False,  -- don't want base type
                                                    LexTokenManager.First_Token,
                                                    Sym));
            TypeInfo.RangeRHS := Maths.ValueRep
               (Dictionary.GetScalarAttributeValue (
                                                    False,  -- don't want base type
                                                    LexTokenManager.Last_Token,
                                                    Sym));
         else
            TypeInfo := UnknownTypeRecord;
            ErrorHandler.SemanticError (5,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (Sym));
         end if;
      when IsParameterName =>
         null; -- should never occur
   end case;

   -- if a primary references an external stream variable we need to check that it
   -- has done so in a simple assignment statement only (no branches in syntax tree)
   if not IsAnnotation then
      CheckInvalidStreamUse (Node,
                             TypeInfo.StreamSymbol,
                             Scope);
      -- similarly check that protected function is only used directly in assignment statement
      CheckInvalidProtectedFunctionUse (Node,
                                        ProtectedReferencesBy (TypeInfo.OtherSymbol, Scope),
                                        Scope);
   end if;

   TypeInfo.Sort := TypeResult;
   TypeInfo.ParamCount := 0;
   TypeInfo.ParamList := Lists.Null_List;

   -- Normally we set OtherSymbol to null at this point because we have finished with it;
   -- however, if the OtherSymbol represents an in instantiation of unchecked conversion then
   -- we leave it alone.  This allows wf_assign to know that the assigned expression is
   -- an unchecked conversion and for it to seed the syntax tree so that the VCG knows as well.
   -- We need to do this in order to suppress RTCs associated with assignment of unchecked
   -- conversions to an object of the same subtype.  The process is identical to that used
   -- for the assignment of external in variables.
   --
   -- Similarly, we need to keep the component information when processing
   -- the assignment of an in stream variable, in case it is marked 'Always_Valid
   if (IsAnnotation or else
     not Dictionary.IsAnUncheckedConversion (TypeInfo.OtherSymbol)) and then
     not (TypeInfo.IsAVariable and then
         Dictionary.GetOwnVariableOrConstituentMode (TypeInfo.VariableSymbol) =
            Dictionary.InMode) then
      TypeInfo.OtherSymbol := Dictionary.NullSymbol;
   end if;

   if not IsAnnotation then
      TypeInfo.ArgListFound := False;
      if SyntaxNodeType (Child_Node (Node)) /= SPSymbols.name then
         TypeInfo.IsAVariable := False;
         TypeInfo.IsAnEntireVariable := False;
      end if;
   end if;

   ExpStack.Push (TypeInfo, EStack);

   --# accept F, 601, ErrorHandler.ErrorContext, TheHeap, "False coupling through SPARK_IO" &
   --#        F, 601, EStack,                    TheHeap, "False coupling through SPARK_IO" &
   --#        F, 601, EStack,              ComponentData, "False coupling through SPARK_IO";
end wf_primary;
