-- $Id: sem-compunit-wf_positional_argument_association.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- with Debug;
separate (Sem.CompUnit)
procedure wf_positional_argument_association
   (Node         : in out STree.SyntaxNode;
    Scope        : in     Dictionary.Scopes;
    EStack       : in out ExpStack.ExpStackType;
    IsAnnotation : in     Boolean;
    RefVar        : in     SeqAlgebra.Seq;
    ComponentData : in out ComponentManager.ComponentData)

is
   NAME_ARG_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.name_argument_list,
                               True  => SPSymbols.annotation_name_argument_list);

   EXP_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.expression,
                               True  => SPSymbols.annotation_expression);

   ExpResult,
   TypeInfo            : ExpRecord;
   ExpectedType,
   VCGtype,
   Sym,
   ParamSym            : Dictionary.Symbol;
   TypesAreConvertable : Boolean;
   ExpValue,
   UnusedValue         : Maths.Value;
   ErrorFound          : Boolean := False;

   --------------------------------------------------------------

   procedure ChainUpToNameList (Node : in out STree.SyntaxNode)
   --# global in IsAnnotation;
   --#        in STree.Table;
   --# derives Node from *,
   --#                   IsAnnotation,
   --#                   STree.Table;
   is
   begin
      while SyntaxNodeType (Node) /= NAME_ARG_LOOKUP (IsAnnotation)
      loop
         Node := ParentNode (Node);
      end loop;
   end ChainUpToNameList;

   --------------------------------------------------------------

   function ExpressionLocation (Node         : STree.SyntaxNode;
                                IsAnnotation : Boolean)
                               return STree.SyntaxNode
   --# global in STree.Table;
   is
      ExpLoc : STree.SyntaxNode;

   begin
      if SyntaxNodeType (Child_Node (Node)) = EXP_LOOKUP (IsAnnotation) then
         ExpLoc := Node;
      else
         ExpLoc := Next_Sibling (Child_Node (Node));
      end if;
      return ExpLoc;
   end ExpressionLocation;

   --------------------------------------------------------------

   procedure CheckTypesAreConvertable (Node     : in     STree.SyntaxNode;
                                       Target,
                                       Source   : in     Dictionary.Symbol;
                                       Ok       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        Source,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Target &
   --#         Ok                        from Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        Source,
   --#                                        Target;
   --#
   is
      OkLocal   : Boolean;
      Undefined : Boolean := False;

      ---------------------------------------

      function DimensionsMatch (Target, Source : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.GetNumberOfDimensions (Target) =
            Dictionary.GetNumberOfDimensions (Source);
      end DimensionsMatch;

      ---------------------------------------

      function IndexesAreConvertible (Target, Source : Dictionary.Symbol)
                                     return Boolean
      --# global in Dictionary.Dict;
      is
         TgtIt,
         SrcIt : Dictionary.Iterator;
         Ok    : Boolean;

         function Convertible (Src, Tgt : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         is
         begin
            return (Dictionary.TypeIsNumeric (Src) and then
                    Dictionary.TypeIsNumeric (Tgt)) or else
               (Dictionary.GetRootType (Src) = Dictionary.GetRootType (Tgt));

         end Convertible;

      begin --IndexesAreConvertible
         Ok := True;
         TgtIt := Dictionary.FirstArrayIndex (Target);
         SrcIt := Dictionary.FirstArrayIndex (Source);
         while not Dictionary.IsNullIterator (TgtIt) loop

            if not Convertible (Dictionary.CurrentSymbol (SrcIt),
                                Dictionary.CurrentSymbol (TgtIt))
            then
               Ok := False;
               exit;
            end if;
            TgtIt := Dictionary.NextSymbol (TgtIt);
            SrcIt := Dictionary.NextSymbol (SrcIt);
         end loop;
         return Ok;
      end IndexesAreConvertible;

      ---------------------------------------

      function ComponentsSameType (Target, Source : Dictionary.Symbol)
                                  return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.GetRootType (Dictionary.GetArrayComponent (Target)) =
            Dictionary.GetRootType (Dictionary.GetArrayComponent (Source));
      end ComponentsSameType;

      ---------------------------------------

      function ComponentsConstraintsMatch (Target, Source : Dictionary.Symbol)
                                          return Boolean
      --# global in Dictionary.Dict;
      --#        in LexTokenManager.State;
      is
         TgtComponent,
         SrcComponent : Dictionary.Symbol;
         Result : Boolean;

         function ScalarBoundsMatch (SrcSym, TgtSym : Dictionary.Symbol)
                                    return Boolean
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.State;
         is
         begin
            return LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Dictionary.GetScalarAttributeValue
                                                                          (False,
                                                                           LexTokenManager.First_Token,
                                                                           SrcSym),
                                                                        Lex_Str2 => Dictionary.GetScalarAttributeValue
                                                                          (False,
                                                                           LexTokenManager.First_Token,
                                                                           TgtSym)) = LexTokenManager.Str_Eq and then
              LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Dictionary.GetScalarAttributeValue
                                                                     (False,
                                                                      LexTokenManager.Last_Token,
                                                                      SrcSym),
                                                                   Lex_Str2 => Dictionary.GetScalarAttributeValue
                                                                     (False,
                                                                      LexTokenManager.Last_Token,
                                                                      TgtSym)) = LexTokenManager.Str_Eq;
         end ScalarBoundsMatch;

      begin --ComponentsConstraintsMatch
         TgtComponent := Dictionary.GetArrayComponent (Target);
         SrcComponent := Dictionary.GetArrayComponent (Source);
         if Dictionary.TypeIsScalar (TgtComponent) then
            Result := ScalarBoundsMatch (TgtComponent, SrcComponent);
         elsif Dictionary.TypeIsArray (TgtComponent) then
            Result := IndexesMatch (TgtComponent, SrcComponent);
         elsif Dictionary.TypeIsRecord (TgtComponent) then
            Result := True;
         else
            Result := False; --unexpected case, above should trap all components
         end if;
         return Result;
      end ComponentsConstraintsMatch;

      ---------------------------------------

   begin --CheckTypesAreConvertable
         -- UnknownTypes considered convertable to stop error propagation
      if Dictionary.IsUnknownTypeMark (Target) or
         Dictionary.IsUnknownTypeMark (Source)
      then
         OkLocal := True;
         Undefined := True;

      elsif (not IsAnnotation) and then
         (Dictionary.IsPrivateType (Source, Scope) or else
          Dictionary.IsPrivateType (Target, Scope)) and then
         Target /= Source
      then
         OkLocal := False;

      elsif Dictionary.TypeIsNumeric (Target) and then
         Dictionary.TypeIsNumeric (Source)
      then
         OkLocal := True;
      elsif Dictionary.TypeIsArray (Target) and then
         Dictionary.TypeIsArray (Source)
      then
         OkLocal := True;
         if not DimensionsMatch (Target, Source) then
            ErrorHandler.SemanticError (423,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            OkLocal := False;
         elsif not IndexesAreConvertible (Target, Source) then
            ErrorHandler.SemanticError (420,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            OkLocal := False;
         elsif not IndexesMatch (Target, Source) then
            ErrorHandler.SemanticError (418,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            OkLocal := False;
         elsif not ComponentsSameType (Target, Source) then
            ErrorHandler.SemanticError (421,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            OkLocal := False;
         elsif not ComponentsConstraintsMatch (Target, Source) then
            ErrorHandler.SemanticError (422,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            OkLocal := False;
         end if;

         -- allow unnecessary conversions, warning will be produced by later if clause
      elsif Dictionary.GetRootType (Target) = Dictionary.GetRootType (Source) then
         OkLocal := True;
      else
         OkLocal := False;
      end if;

      --if legal (other than undefined case, check if necessary)
      if OkLocal and then
         not Undefined and then
         Dictionary.GetRootType (Target) =
         Dictionary.GetRootType (Source)
      then
         ErrorHandler.SemanticWarning (309,
                                       NodePosition (ParentNode (ParentNode (ParentNode (Node)))),
                                       LexTokenManager.Null_String);
      end if;
      Ok := OkLocal;
   end CheckTypesAreConvertable;

   --------------------------------------------------------------

   procedure DoTaggedTypeConversion (Node   : in     STree.SyntaxNode;
                                     Target : in out ExpRecord;
                                     Source : in     ExpRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     RefVar;
   --#        in     STree.Table;
   --#        in out ComponentData;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict           from ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        Source,
   --#                                        Target,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Source,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Target &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        RefVar,
   --#                                        Source,
   --#                                        Target,
   --#                                        TheHeap &
   --#         Target                    from *,
   --#                                        CommandLineData.Content,
   --#                                        ComponentData,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        Source,
   --#                                        TheHeap;
   is

      procedure RaiseError (ErrNo : in Natural)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out Target;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrNo,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Target                    from Dictionary.Dict;
      is
      begin
         Target := UnknownTypeRecord;
         ErrorHandler.SemanticError (ErrNo,
                                     ErrorHandler.NoReference,
                                     NodePosition
                                     (ParentNode (ParentNode (ParentNode (Node)))),
                                     LexTokenManager.Null_String);
      end RaiseError;

   begin -- DoTaggedTypeConversion
      -- On entry we know Target.TypeSymbol is tagged.  If Source.TypeSymbol is not then we have some grossly
      -- malformed type conversion
      if not Dictionary.TypeIsTagged (Source.TypeSymbol) then
         RaiseError (32);

      elsif not Dictionary.IsAnExtensionOf (Target.TypeSymbol, Source.TypeSymbol) then
         RaiseError (831);

      else
         -- We have two tagged types and the target is an ancestor of the source; basically ok
         if Source.IsAVariable or Source.IsConstant then
            -- we have an object to convert
            if IsAnnotation or else Source.IsConstant then
               -- In an annotation, or for a constant, all we need to is change
               -- the result type to that expected.
               Target.Sort := TypeResult;
               Target.IsConstant := Source.IsConstant;

               Target.IsStatic := Source.IsStatic and
                                   CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83;

               Target.OtherSymbol := Dictionary.NullSymbol;
               Target.VariableSymbol := Source.VariableSymbol;
               Target.IsAVariable := Source.IsAVariable;
               Target.IsAnEntireVariable := False;

            else
               -- In a normal expression with a variable and we need to convert
               -- appropriate record subcomponent symbols.

               -- We can't replace X with X.Inherit unless we add X's subcomponents first
               AddRecordSubComponents (RecordVarSym  => Source.VariableSymbol,
                                       RecordTypeSym => Dictionary.GetType (Source.VariableSymbol),
                                       ComponentData => ComponentData);
               -- Set up ExpRecord
               Target.VariableSymbol := ConvertTaggedActual (Source.VariableSymbol,
                                                             Target.TypeSymbol);
               Target.Sort := TypeResult;
               Target.IsConstant := Source.IsConstant;

               Target.IsStatic := Source.IsStatic and
                                   CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83;

               Target.OtherSymbol := Dictionary.NullSymbol;
               Target.IsAVariable := Source.IsAVariable;
               Target.IsAnEntireVariable := False;

               -- Substitute reference variables to show we only used a subcomponent of Source.
               -- We have to look for the source variable because there may be other items in
               -- the RefVar list if, for example, the type conversion forms part of a larger
               -- expression such as a function call.
               SeqAlgebra.RemoveMember (TheHeap,
                                        RefVar,
                                        Natural (Dictionary.SymbolRef (Source.VariableSymbol)));
               SeqAlgebra.AddMember (TheHeap,
                                     RefVar,
                                     Natural (Dictionary.SymbolRef (Target.VariableSymbol)));

            end if;

         else -- not an object
            RaiseError (832);
         end if;
      end if;
   end DoTaggedTypeConversion;

   --------------------------------------------------------------

   function ConvertValue (Target : Dictionary.Symbol;
                          Exp    : ExpRecord) return Maths.Value
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   is
      Val : Maths.Value;
   begin
      Val := Exp.Value;
      if not Maths.HasNoValue (Val) then
         if Dictionary.IsUnknownTypeMark (Target) then
            Val := Maths.NoValue;
         elsif Dictionary.TypeIsReal (Target) then
            Maths.ConvertToReal (Val);
         elsif Dictionary.TypeIsInteger (Target) and then
            Dictionary.TypeIsReal (Exp.TypeSymbol)
         then
            case CommandLineData.Content.LanguageProfile is
               when CommandLineData.SPARK83 =>

                  Val := Maths.NoValue; -- can't do real to integer safely

               when CommandLineData.SPARK95 |
                 CommandLineData.SPARK2005 =>

                  Val := Maths.Ada95RealToInteger (Val);
            end case;
         end if;
      end if;
      return Val;
   end ConvertValue;

   --------------------------------------------------------------

   procedure RangeCheck (ARange : in Boolean;
                         Node   : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorFound                from *,
   --#                                        ARange &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from ARange,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
   begin
      if ARange then
         ErrorFound := True;
         ErrorHandler.SemanticError (341,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                     LexTokenManager.Null_String);
      end if;
   end RangeCheck;

   --------------------------------------------------------------

   function GetExpectedArrayIndexType (VarSym    : Dictionary.Symbol;
                                       TypeSym   : Dictionary.Symbol;
                                       Dimension : Positive) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol;
   begin
      -- This function determines what type to plant in the syntax tree so that the VCG can check
      -- that array accesses are in bounds.  FOr a constrained object it is edy - we plant the
      -- appropriate index type for the dimension being accessed.  For indexing into unconstrained
      -- objects we plant a symbol of a special kind (ParameterConstraintSymbol) associated with
      -- the array object (rather than its type); this special symbol represents "the index as
      -- constrained by 'something' at this point".  Typically we will no know the actual bounds
      -- of the constraint represented by this symbol.

      if Dictionary.IsUnconstrainedArrayType (TypeSym) then
         -- For unconstrained arrays, obtain the implcitly declared constraint symbol for the array object
         Result := Dictionary.GetSubprogramParameterConstraint (VarSym, Dimension);
      else
         -- For constrained arrays then obtain appropriate index for the array type; this is what the VCG needs
         Result := Dictionary.GetArrayIndex (TypeSym, Dimension);
      end if;
      return Result;
   end GetExpectedArrayIndexType;

   --------------------------------------------------------------

begin --wf_positional_argument_association
   ExpStack.Pop (ExpResult, EStack);
   ExpStack.Pop (TypeInfo, EStack);
   Sym := TypeInfo.OtherSymbol;

   case TypeInfo.Sort is
      when IsTypeMark =>
         -- seed syntax tree with type for run-time check
         STree.AddNodeSymbol (Node, ExpResult.TypeSymbol);

         if Dictionary.IsUnconstrainedArrayType (TypeInfo.TypeSymbol) then
            ErrorHandler.SemanticError (39,
                                        ErrorHandler.NoReference,
                                        NodePosition
                                        (ParentNode (ParentNode (ParentNode (Node)))),
                                        LexTokenManager.Null_String);
            TypeInfo := UnknownTypeRecord;

            -- special handling for type conversion of string literals.
         elsif Dictionary.IsPredefinedStringType (ExpResult.TypeSymbol) and then
            ExpResult.RangeRHS /= Maths.NoValue
         then
            ErrorHandler.SemanticError (425,
                                        22,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            TypeInfo := UnknownTypeRecord;

         elsif Dictionary.TypeIsTagged (TypeInfo.TypeSymbol) then
            DoTaggedTypeConversion (Node   => Node,
                                    Target => TypeInfo,
                                    Source => ExpResult);

         else -- some "normal" conversion case
            if ExpResult.IsARange then
               -- Type conversion of a range is illegal.  This also
               -- catches the illegal case of type-conversion of a
               -- subtype mark, such as Integer (Natural)
               TypeInfo := UnknownTypeRecord;
               ErrorHandler.SemanticError (114, ErrorHandler.NoReference,
                                           NodePosition (ParentNode (Node)),
                                           LexTokenManager.Null_String);

            else
               CheckTypesAreConvertable (Node,
                                         TypeInfo.TypeSymbol,
                                         ExpResult.TypeSymbol,
                                          -- to get
                                         TypesAreConvertable);
               if TypesAreConvertable then
                  ConstraintCheck (ExpResult.Value,
                                   ExpValue,
                                   IsAnnotation,
                                   TypeInfo.TypeSymbol,
                                   NodePosition (Node));
                  ExpResult.Value := ExpValue;
                  TypeInfo.Sort := TypeResult;
                  TypeInfo.IsConstant := ExpResult.IsConstant;

                  TypeInfo.IsStatic := ExpResult.IsStatic
                    and CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83;

                  TypeInfo.OtherSymbol := Dictionary.NullSymbol;
                  TypeInfo.Value := ConvertValue (TypeInfo.TypeSymbol, ExpResult);
                  TypeInfo.VariableSymbol := ExpResult.VariableSymbol;
                  TypeInfo.IsAVariable := False;
                  TypeInfo.IsAnEntireVariable := False;
               else
                  TypeInfo := UnknownTypeRecord;
                  ErrorHandler.SemanticError (32, ErrorHandler.NoReference,
                                              NodePosition
                                                (ParentNode (ParentNode (ParentNode (Node)))),
                                              LexTokenManager.Null_String);
               end if;
            end if;
         end if;

      when IsFunction =>
         if TypeInfo.ParamCount =
            Dictionary.GetNumberOfSubprogramParameters (Sym)
         then
            TypeInfo := UnknownSymbolRecord;
            ErrorHandler.SemanticError (3,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (Sym));
            ChainUpToNameList (Node);
         else
            TypeInfo.ParamCount := TypeInfo.ParamCount + 1;
            ParamSym := Dictionary.GetSubprogramParameter (Sym,
                                                           TypeInfo.ParamCount);
            ExpectedType := Dictionary.GetType (ParamSym);
            -- Seed syntax tree with expected type for run-time check;
            -- but, don't do this for instantiation of unchecked_conversion
            -- because we don't want any RTCs for association of those parameters
            -- (provided the function parameter subtype and actual subtype match)
            if not (Dictionary.IsAnUncheckedConversion (Sym) and then
                      ExpResult.TypeSymbol = ExpectedType) then
               STree.AddNodeSymbol (Node,
                                                ExpectedType);
            end if;

            -- There is a special case involving functions an stream variables.  We allow a stream
            -- variable to be a parameter to an Unchecked_Conversion but need to ensure that
            -- the function inherits the restrictions associated with referencing a stream
            -- (e.g. cannot be used in general expression).  We can do this here by checking
            -- the StreamSymbol of the parameter expression (there will only be one if we are
            -- talking about an unchecked conversion) and if it is non-null then setting the
            -- stream symbol of the function result record (now an object) to the function symbol.
            -- Note that this clause will only be executed for an unchecked conversion because
            -- a parameter which is a stream would hav ebeen rejected at wf_primary in all other
            -- cases
            if ExpResult.StreamSymbol /= Dictionary.NullSymbol then
               TypeInfo.StreamSymbol := Sym;
            end if;

            RangeCheck (ExpResult.IsARange, Node);

            -- function is deemed constant if it is predefined and all its parameters
            -- are constant.
            TypeInfo.IsConstant := TypeInfo.IsConstant and ExpResult.IsConstant;

            if (TypeInfo.TaggedParameterSymbol = ExpResult.TypeSymbol or else
                (TypeInfo.TaggedParameterSymbol = Dictionary.NullSymbol and then
                 Dictionary.CompatibleTypes (Scope,
                                             ExpectedType,  -- always defined here
                                             ExpResult.TypeSymbol)) or else
                (not Dictionary.IsAnExtensionOf (ExpResult.TypeSymbol, TypeInfo.TaggedParameterSymbol) and then
                 Dictionary.CompatibleTypes (Scope,
                                             ExpectedType,  -- always defined here
                                             ExpResult.TypeSymbol)))
            then
               TaggedActualMustBeObjectCheck (NodePos         => NodePosition (Node),
                                              FormalType      => ExpectedType,
                                              ActualType      => ExpResult.TypeSymbol,
                                              ControllingType => Dictionary.GetSubprogramControllingType (Sym),
                                              IsAVariable     => ExpResult.IsAVariable,
                                              IsAConstant     => ExpResult.IsConstant,
                                              ErrorFound      => ErrorFound);
               -- Following call will deal with scalar value constraint checking
               --# accept Flow, 10, UnusedValue, "Expected ineffective assignment";
               ConstraintCheck (ExpResult.Value,
                                UnusedValue,
                                IsAnnotation,
                                ExpectedType,
                                NodePosition (ExpressionLocation (Node, IsAnnotation)));
               --# end accept;
               -- Check array bounds etc.
               if Dictionary.TypeIsArray (Dictionary.GetType (ParamSym)) and then
                 not Dictionary.IsUnconstrainedArrayType (ExpectedType) then
                  -- Formal is a constrained subtype of an unconstrained array

                  if Dictionary.IsUnconstrainedArrayType (ExpResult.TypeSymbol) then
                     -- Actual is unconstrained.  In SPARK95 or 2005, this is OK if
                     -- the actual is a static String expression, but illegal
                     -- otherwise.
                     if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
                       Dictionary.IsPredefinedStringType (ExpResult.TypeSymbol) then
                        -- Formal must be a constrained String subtype, so we need
                        -- to check the upper bound of the actual against the expected
                        -- upper bound of the formal.
                        if ExpResult.RangeRHS = Maths.NoValue then
                           -- Actual is not static, so must be illegal
                           ErrorHandler.SemanticError (39,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                                       LexTokenManager.Null_String);
                        else
                           -- Actual is static, so check upper-bound against that expected
                           if ExpResult.RangeRHS /= Maths.ValueRep
                             (Dictionary.GetScalarAttributeValue (False,
                                                                  LexTokenManager.Last_Token,
                                                                  Dictionary.GetType (ParamSym))) then
                              ErrorHandler.SemanticError (418,
                                                          ErrorHandler.NoReference,
                                                          NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                                          LexTokenManager.Null_String);

                           end if;
                        end if;

                     else
                        -- SPARK83 or not a String type, so illegal
                        ErrorHandler.SemanticError (39,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                                    LexTokenManager.Null_String);
                     end if;

                  elsif Illegal_Unconstrained (ExpResult.TypeSymbol,
                                               ExpectedType) then
                     -- Although both formal and actual are constrained their bounds don't match
                     ErrorHandler.SemanticError (418,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                                 LexTokenManager.Null_String);
                  end if;
               end if;

               -- To help the VCG with generating checks involving unconstrained formal parameters, we
               -- seed the syntax tree with a constraining type mark.  The positional_argument_association
               -- node is already used for RTC purposes, so we seed the expression node instead.
               if not IsAnnotation then
                  PlantConstrainingType (ExpResult.TypeSymbol,
                                         ExpResult.RangeRHS,
                                         STree.ExpressionFromPositionalArgumentAssociation (Node));
               end if;
            else
               ErrorFound := True;
               ErrorHandler.SemanticError (38,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                           LexTokenManager.Null_String);
            end if;
         end if;

      when IsObject =>
         if TypeInfo.ParamCount =
            Dictionary.GetNumberOfDimensions (TypeInfo.TypeSymbol)
         then
            TypeInfo := UnknownSymbolRecord;
            ErrorHandler.SemanticError (93,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (Sym));
            ChainUpToNameList (Node);
         else
            TypeInfo.ParamCount := TypeInfo.ParamCount + 1;
            -- ExpectedType is used to ensure that indexing expression is well-typed
            ExpectedType := Dictionary.GetArrayIndex (TypeInfo.TypeSymbol, TypeInfo.ParamCount);
            -- VCGtype is used to tell VCG what indexing type to expect.  Same as Expected type for a
            -- constrained array but different for uncnstrained.  See comment in function GetExpectedArrayIndexType
            VCGtype := GetExpectedArrayIndexType (TypeInfo.OtherSymbol,
                                                  TypeInfo.TypeSymbol,
                                                  TypeInfo.ParamCount);
            -- seed syntax tree with expected type for run-time check
            -- Debug.PrintSym ("Planting array index type ", VCGtype);
            STree.AddNodeSymbol (Node,
                                             VCGtype);
            RangeCheck (ExpResult.IsARange, Node);
            TypeInfo.IsConstant := TypeInfo.IsConstant and ExpResult.IsConstant;
            if Dictionary.CompatibleTypes (Scope,
                                           ExpectedType,
                                           ExpResult.TypeSymbol)
            then
               --# accept Flow, 10, UnusedValue, "Expected ineffective assignment";
               ConstraintCheck (ExpResult.Value,
                                UnusedValue,
                                IsAnnotation,
                                ExpectedType,
                                NodePosition (ExpressionLocation (Node, IsAnnotation)));
               --# end accept;
            else
               ErrorFound := True;
               ErrorHandler.SemanticError (38,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpressionLocation (Node, IsAnnotation)),
                                           LexTokenManager.Null_String);
            end if;
         end if;

      when others =>
         null;
   end case;

   TypeInfo.ErrorsInExpression := ErrorFound or
      TypeInfo.ErrorsInExpression or
      ExpResult.ErrorsInExpression;

   ExpStack.Push (TypeInfo, EStack);

   --# accept Flow, 33, UnusedValue, "Expected to be neither referenced nor exported";
end wf_positional_argument_association;
