-- $Id: sem-compunit-wf_package_specification.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview:
-- Checks a Package Specification for Sem on down pass through
-- TreeProcessor.  Starts at node package_specification.  May directly raise
-- errors for: mismatch between initial and terminal identifier.
-- Other errors may be raised indirectly by wf_package_annotation which
-- are called from here.
--------------------------------------------------------------------------------

with SLI;

separate (Sem.CompUnit)

procedure wf_package_specification (Node         : in STree.SyntaxNode;
                                    IdentStr     : in LexTokenManager.Lex_String;
                                    PackSym      : in Dictionary.Symbol;
                                    CurrentScope : in Dictionary.Scopes)
is
   type EnclosingScopeTypes is (InLibrary, InPackage, InProcedure);
   EnclosingScopeType        : EnclosingScopeTypes;
   EndPosNode,
   AnnoNode,
   VisPartNode,
   PrivPartNode,
   IdentNode                 : STree.SyntaxNode;
   PackVisScope              : Dictionary.Scopes;

   ------------------------------

   procedure FindKeyNodes
   --# global in     Node;
   --#        in     STree.Table;
   --#           out AnnoNode;
   --#           out IdentNode;
   --#           out PrivPartNode;
   --#           out VisPartNode;
   --# derives AnnoNode,
   --#         IdentNode,
   --#         PrivPartNode,
   --#         VisPartNode  from Node,
   --#                           STree.Table;
   is
   begin
      -- Node is at package_specification
      AnnoNode := Next_Sibling (Child_Node (Node));
      IdentNode := Child_Node (Child_Node (Node));
      if SyntaxNodeType (IdentNode) /= SPSymbols.identifier then
         -- declaring a child package
         IdentNode := LastChildOf (IdentNode);
      end if;

      VisPartNode := Child_Node (Next_Sibling (AnnoNode));
      PrivPartNode := Next_Sibling (Next_Sibling (AnnoNode));
      if SyntaxNodeType (PrivPartNode) = SPSymbols.private_part then
         PrivPartNode := Child_Node (PrivPartNode);
      else
         PrivPartNode := STree.NullNode;
      end if;

   end FindKeyNodes;

   ------------------------------

   procedure FindEnclosingScopeType (Scope : in Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#           out EnclosingScopeType;
   --# derives EnclosingScopeType from Dictionary.Dict,
   --#                                 Scope;
   is
   begin
      if Dictionary.IsGlobalScope (Scope) then
         EnclosingScopeType := InLibrary;
      elsif Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         EnclosingScopeType := InPackage;
      else
         EnclosingScopeType := InProcedure;
      end if;
   end FindEnclosingScopeType;

   ------------------------------

   procedure CheckClosingIdentifier (EndNameNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndNameNode,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      StartIdentNode    : STree.SyntaxNode;
      EndIdentNode      : STree.SyntaxNode;
   begin
      StartIdentNode := IdentNode;
      EndIdentNode := LastChildOf (EndNameNode);
      loop
         -- check identifiers at current positions:
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (StartIdentNode),
                                                                 Lex_Str2 => NodeLexString (EndIdentNode)) /= LexTokenManager.Str_Eq then
            ErrorHandler.SemanticError (58,
                                        ErrorHandler.NoReference,
                                        NodePosition (EndIdentNode),
                                        NodeLexString (StartIdentNode));
            exit;
         end if;

         -- move on to next identifiers:
         StartIdentNode := Next_Sibling (ParentNode (StartIdentNode));
         EndIdentNode   := Next_Sibling (ParentNode (EndIdentNode));

         -- finished when both exhausted:
         exit when SyntaxNodeType (StartIdentNode) /=
            SPSymbols.identifier and
            SyntaxNodeType (EndIdentNode) /=
            SPSymbols.identifier;

         -- check if only one exhausted (length mismatch):
         if SyntaxNodeType (StartIdentNode) /=
            SPSymbols.identifier or
            SyntaxNodeType (EndIdentNode) /=
            SPSymbols.identifier
         then
            ErrorHandler.SemanticError (615,
                                        ErrorHandler.NoReference,
                                        NodePosition (EndNameNode),
                                        LexTokenManager.Null_String);
            exit;
         end if;
      end loop;
   end CheckClosingIdentifier;

   ------------------------------

   procedure CheckBodyRequiredBySpark (Node    : in     STree.SyntaxNode;
                                       PackSym : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It : Dictionary.Iterator;
   begin
      It := Dictionary.FirstInitializedOwnVariable (PackSym);
      while not Dictionary.IsNullIterator (It)
      loop
         if Dictionary.IsDeclared (Dictionary.CurrentSymbol (It)) and then
            not Dictionary.VariableIsInitialized (Dictionary.CurrentSymbol (It))
         then
            case CommandLineData.Content.LanguageProfile is
               when CommandLineData.SPARK83 =>

                  ErrorHandler.SemanticWarning (407,
                                                NodePosition (Node),
                                                LexTokenManager.Null_String);

               when CommandLineData.SPARK95 |
                 CommandLineData.SPARK2005 =>

                  ErrorHandler.SemanticError (607,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);
            end case;
            exit;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckBodyRequiredBySpark;

   ------------------------------

   procedure CheckDeferredItems (EndNode : in STree.SyntaxNode;
                                 PackSym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndNode,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ErrorPosition : LexTokenManager.Token_Position;
      It  : Dictionary.Iterator;
      Sym : Dictionary.Symbol;

   begin
      ErrorPosition := NodePosition (Child_Node (EndNode));
      It := Dictionary.FirstDeferredConstant (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         if not Dictionary.IsDeclared (Sym) then
            ErrorHandler.SemanticError (26, ErrorHandler.NoReference,
                                        ErrorPosition,
                                        Dictionary.GetSimpleName (Sym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;

      It := Dictionary.FirstPrivateType (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         if not Dictionary.IsDeclared (Sym) then
            ErrorHandler.SemanticError (27,
                                        ErrorHandler.NoReference,
                                        ErrorPosition,
                                        Dictionary.GetSimpleName (Sym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckDeferredItems;

   ------------------------------

   procedure CheckModes (Node    : in STree.SyntaxNode;
                         PackSym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is separate;

   ------------------------------

   procedure CheckStateCanBeInitialized (PackSym  : in Dictionary.Symbol;
                                         AnnoNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is separate;

   ------------------------------

   procedure CheckTypesCanBeUsed (PackSym  : in Dictionary.Symbol;
                                  ErrNode  : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrNode,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is separate;

   ------------------------------

   procedure wf_anno (Node         : in STree.SyntaxNode;
                      PackSym      : in Dictionary.Symbol;
                      ScopeType    : in EnclosingScopeTypes;
                      CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        ScopeType,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        ScopeType,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;

   ------------------------------

   procedure wf_visible (Node         : in STree.SyntaxNode;
                         PackSym      : in Dictionary.Symbol;
                         CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;

   ------------------------------

   procedure wf_private (Node  : in STree.SyntaxNode;
                         Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
      is separate;

   ------------------------------

begin -- wf_package_specification
   FindKeyNodes;
   FindEnclosingScopeType (CurrentScope); --tells us where package is being declared

   PackVisScope := Dictionary.VisibleScope (PackSym);

      --# assert True;
      wf_anno (Node         => AnnoNode,
               PackSym      => PackSym,
               ScopeType    => EnclosingScopeType,
               CurrentScope => PackVisScope);

      wf_visible (VisPartNode,
                  PackSym,
                  PackVisScope);

      --# assert True;
      if PrivPartNode /= STree.NullNode then -- private part exists
         if SyntaxNodeType (PrivPartNode) = SPSymbols.hidden_part then
            Dictionary.AddPrivatePart (PackSym,
                                       Dictionary.Location'(NodePosition (PrivPartNode),
                                                            NodePosition (PrivPartNode)),
                                       True);
            ErrorHandler.HiddenText (NodePosition (PrivPartNode),
                                     IdentStr,
                                     SPSymbols.private_part);
            CheckTypesCanBeUsed (PackSym, PrivPartNode);
         else
            Dictionary.AddPrivatePart (PackSym,
                                       Dictionary.Location'(NodePosition (PrivPartNode),
                                                            NodePosition (PrivPartNode)),
                                       False);
            wf_private (PrivPartNode,
                        Dictionary.PrivateScope (PackSym));

            -- check that private types resolved into scalars have not
            -- invalidated parameter declarations of exported procedures
            if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 then
               -- guarded because '83 rules different from 95 and 2005
               CheckModes (VisPartNode, PackSym);
            end if;

            CheckClosingIdentifier (Next_Sibling (PrivPartNode));
            CheckDeferredItems (Next_Sibling (PrivPartNode),
                                PackSym);
            CheckTypesCanBeUsed (PackSym, Next_Sibling (PrivPartNode));
         end if;
      else -- no private part
         CheckClosingIdentifier (LastSiblingOf (AnnoNode));
         CheckDeferredItems (LastSiblingOf (AnnoNode),
                             PackSym);
         CheckTypesCanBeUsed (PackSym, LastSiblingOf (AnnoNode));
      end if;

      -- check for cases where package requires a body
      --# assert True;
      if EnclosingScopeType = InLibrary and then
        not Dictionary.PackageRequiresBody (PackSym)
      then
         EndPosNode := LastSiblingOf (AnnoNode);
         --declaring a library package for which Ada rules do not demand a body so
         --checks must be made to see if Spark rules require a body
         if SyntaxNodeType (Child_Node (EndPosNode)) =
           SPSymbols.basic_declarative_item_rep
         then
            EndPosNode := Next_Sibling (Child_Node (EndPosNode));
         end if;
         CheckBodyRequiredBySpark (EndPosNode, PackSym);
      end if;

      CheckStateCanBeInitialized (PackSym, AnnoNode);
      CheckAnnouncedTypesDeclared (PackSym => PackSym,
                                   Scope => Dictionary.VisibleScope (PackSym),
                                   Node => LastSiblingOf (AnnoNode));

      --  The cross-references for the own variables are generated
      --  after the full semantic analysis of the package
      --  specification because we need to know if an own variable is
      --  actually an abstract own variable or a visble concrete own
      --  variable. If it is an abstract own variable, the own
      --  variable is considered as a declaration, if it is a visible
      --  concrete own variable, the own variable is a usage of the
      --  visible concrete variable that will be declared later in the
      --  same package specification. If the declaration of the
      --  concrete variable only appears in the package body, the own
      --  variable is considered as an abstract declaration.
      if ErrorHandler.Generate_SLI and then
        Child_Node (AnnoNode) /= STree.NullNode then
         SLI.Generate_Xref_Own (Comp_Unit  => ContextManager.Ops.CurrentUnit,
                                Parse_Tree => Child_Node (AnnoNode),
                                Scope      => PackVisScope);
      end if;

end wf_package_specification;
