-- $Id: sem-compunit-wf_package_declaration.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview:
-- Checks a Package Declaration for Sem on down pass through
-- TreeProcessor.  Starts at node package_declaration.  May directly raise
-- errors for: re-declaration of package identifier.  Other errors may be raised
-- indirectly by wf_package_specification, wf_inherit_clause and
-- wf_context_clause which are called from here.
--------------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_package_declaration (Node         : in STree.SyntaxNode;
                                  CurrentScope : in Dictionary.Scopes)
is
   type EnclosingScopeTypes is (InLibrary, InPackage, InProcedure);
   EnclosingScopeType        : EnclosingScopeTypes;
   IdentStr                  : LexTokenManager.Lex_String;
   SpecNode,
   ContextNode,
   InheritNode,
   IdentNode                 : STree.SyntaxNode;
   PackSym                   : Dictionary.Symbol;
   PackVisScope              : Dictionary.Scopes;
   ChildPackageDeclaration   : Boolean;
   PrivatePackageDeclaration : Boolean;
   ValidName                 : Boolean := True;

   ------------------------------

   procedure FindKeyNodes
   --# global in     Node;
   --#        in     STree.Table;
   --#           out ChildPackageDeclaration;
   --#           out ContextNode;
   --#           out IdentNode;
   --#           out IdentStr;
   --#           out InheritNode;
   --#           out PrivatePackageDeclaration;
   --#           out SpecNode;
   --# derives ChildPackageDeclaration,
   --#         ContextNode,
   --#         IdentNode,
   --#         IdentStr,
   --#         InheritNode,
   --#         PrivatePackageDeclaration,
   --#         SpecNode                  from Node,
   --#                                        STree.Table;
   is
   begin
      PrivatePackageDeclaration :=
         (SyntaxNodeType (Node) =
          SPSymbols.private_package_declaration);
      InheritNode := Child_Node (Node);
      if SyntaxNodeType (InheritNode) = SPSymbols.inherit_clause then
         SpecNode := Next_Sibling (InheritNode);
      else
         SpecNode := InheritNode;
         InheritNode := STree.NullNode;
      end if;

      IdentNode := Child_Node (Child_Node (SpecNode));
      if SyntaxNodeType (IdentNode) = SPSymbols.identifier then
         ChildPackageDeclaration := False;
      else  -- declaring a child package
         ChildPackageDeclaration := True;
         IdentNode := LastChildOf (IdentNode);
      end if;

      IdentStr := NodeLexString (IdentNode);

      ContextNode := Child_Node (ParentNode (ParentNode (Node)));
      if SyntaxNodeType (ContextNode) /= SPSymbols.context_clause then
         ContextNode := STree.NullNode;
      end if;

   end FindKeyNodes;

   ------------------------------

   procedure FindEnclosingScopeType (Scope : in Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#           out EnclosingScopeType;
   --# derives EnclosingScopeType from Dictionary.Dict,
   --#                                 Scope;
   is
   begin
      if Dictionary.IsGlobalScope (Scope) then
         EnclosingScopeType := InLibrary;
      elsif Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         EnclosingScopeType := InPackage;
      else
         EnclosingScopeType := InProcedure;
      end if;
   end FindEnclosingScopeType;

   ------------------------------

   function IsNotRefinementAnnouncement (Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in EnclosingScopeType;
   is
   begin
      return ((EnclosingScopeType /= InPackage)
               or
               (Dictionary.GetContext (Sym) /= Dictionary.ProofContext));
   end IsNotRefinementAnnouncement;

   ------------------------------

begin -- wf_package_declaration
   FindKeyNodes;        --sets up key node and also package identifier
   FindEnclosingScopeType (CurrentScope); --tells us where package is being declared
   if  CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
      (ChildPackageDeclaration and EnclosingScopeType = InLibrary)
   then
      AddChild (IdentNode,
                PrivatePackageDeclaration,
                CurrentScope,
                  -- to get
                PackSym,
                IdentStr);
      -- if PackSym is null then something went wrong when we added the child so we need to supress
      -- any further analysis of the package specification
      ValidName := PackSym /= Dictionary.NullSymbol;
   else
      if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 then
         --check that syntax conforms
         if ChildPackageDeclaration or PrivatePackageDeclaration then
            ErrorHandler.SemanticError (610,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        LexTokenManager.Null_String);
            PrivatePackageDeclaration := False;
         end if;
      elsif ChildPackageDeclaration and EnclosingScopeType /= InLibrary then
         ErrorHandler.SemanticError (614,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     LexTokenManager.Null_String);
      end if;
      PackSym := Dictionary.LookupItem (Name    => IdentStr,
                                        Scope   => CurrentScope,
                                        Context => Dictionary.ProofContext);
      --# assert True;
      if PackSym /= Dictionary.NullSymbol and then
        IsNotRefinementAnnouncement (PackSym) then
         ValidName := False;
         ErrorHandler.SemanticError (10,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);
      else
         if PackSym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => PackSym,
                                       Node => IdentNode);
         end if;
         --# assert True;
         if PrivatePackageDeclaration then -- root level private package
            Dictionary.AddPrivatePackage (Name          => IdentStr,
                                          Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                          Specification => Dictionary.Location'(NodePosition (IdentNode),
                                                                                NodePosition (IdentNode)),
                                          Scope         => CurrentScope,
                                          --to get
                                          ThePackage    => PackSym);
         else
            Dictionary.AddPackage (Name          => IdentStr,
                                   Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                   Specification => Dictionary.Location'(NodePosition (IdentNode),
                                                                         NodePosition (IdentNode)),
                                   Scope         => CurrentScope,
                                   --to get
                                   ThePackage    => PackSym);
         end if;
      end if;
   end if;

   -- wff the package specification iff its declaration is valid
   if ValidName then

      --# assert True;

      if SyntaxNodeType (InheritNode) = SPSymbols.inherit_clause then

         Dictionary.AddInheritsAnnotation (PackSym,
                                           Dictionary.Location'(NodePosition (InheritNode),
                                                                NodePosition (InheritNode)));
         wf_inherit_clause (InheritNode,
                            PackSym,
                            CurrentScope);


      end if;

      --# assert True;

      if ContextNode /= STree.NullNode then

         PackVisScope := Dictionary.VisibleScope (PackSym);

         wf_context_clause (ContextNode,
                            PackSym,
                            PackVisScope);
      end if;

      --# assert True;

      wf_package_specification (SpecNode,
                                IdentStr,
                                PackSym,
                                CurrentScope);

   end if;
end wf_package_declaration;
