-- $Id: sem-compunit-wf_package_body.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (Sem.CompUnit)

procedure wf_package_body (Node  : in     STree.SyntaxNode;
                           Scope : in out Dictionary.Scopes)
is
   IdentStr         : LexTokenManager.Lex_String;
   Sym              : Dictionary.Symbol;
   IdentNode,
   WithNode,
   RefNode,
   NextNode         : STree.SyntaxNode;
   GrandParent      : SPSymbols.SPSymbol;
   SpecFound,
   OkToAddBody,
   BodyIsHidden     : Boolean;
   PackScope        : Dictionary.Scopes;

   ----------------------------------------------

   procedure FindPackage (IdentNode  : in out STree.SyntaxNode;
                          IdentStr   : in out LexTokenManager.Lex_String;
                          Scope      : in     Dictionary.Scopes;
                          Found      :    out Boolean;
                          ThePackage :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Found,
   --#         IdentNode,
   --#         IdentStr,
   --#         STree.Table,
   --#         ThePackage                from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        IdentNode,
   --#                                        IdentStr,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        IdentStr,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      Sym,
      ParentSym : Dictionary.Symbol;
      Ok        : Boolean;
   begin
      Sym := Dictionary.LookupImmediateScope (Name    => IdentStr,
                                              Scope   => Scope,
                                              Context => Dictionary.ProgramContext);

      Ok := Sym /= Dictionary.NullSymbol and then Dictionary.IsPackage (Sym);
      if not Ok then
         ErrorHandler.SemanticError (11,  ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);
         --there is no spec, must create one for Dict to add body to later
         Dictionary.AddPackage (Name          => IdentStr,
                                Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                Specification => Dictionary.Location'(NodePosition (IdentNode),
                                                                      NodePosition (IdentNode)),
                                Scope         => Scope,
                                --to get
                                ThePackage    => Sym);
      else
         STree.Set_Node_Lex_String (Sym  => Sym,
                                    Node => IdentNode);
      end if;
      if SyntaxNodeType (Next_Sibling (ParentNode (IdentNode))) = SPSymbols.identifier then
         -- child package form
         if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 then
            ErrorHandler.SemanticError (610,
                                        ErrorHandler.NoReference,
                                        NodePosition
                                        (Next_Sibling (ParentNode (IdentNode))),
                                        LexTokenManager.Null_String);
         elsif Ok then
            loop -- to handle multiple prefixes
               IdentNode := Next_Sibling (ParentNode (IdentNode));
               IdentStr := NodeLexString (IdentNode);
               ParentSym := Sym;
               Sym := Dictionary.LookupSelectedItem (Prefix   => ParentSym,
                                                     Selector => IdentStr,
                                                     Scope    => Scope,
                                                     Context  => Dictionary.ProofContext);
               if Sym = Dictionary.NullSymbol or else not Dictionary.IsPackage (Sym) then
                  ErrorHandler.SemanticError (11,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              IdentStr);
                  --there is no spec, must create one for Dict to add body to later
                  Dictionary.AddChildPackage (TheParent     => ParentSym,
                                              Sort          => Dictionary.Public,
                                              Name          => IdentStr,
                                              Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                              Specification => Dictionary.Location'(NodePosition (IdentNode),
                                                                                    NodePosition (IdentNode)),
                                              Scope         => Scope,
                                              --to get
                                              ThePackage    => Sym);
                  Ok := False;
                  exit;
               end if;
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => IdentNode);
               exit when SyntaxNodeType (Next_Sibling (ParentNode (IdentNode))) /= SPSymbols.identifier;
               -- when no more identifier (s) to right
            end loop;
         end if;
      end if;

      Found := Ok;
      ThePackage := Sym;
   end FindPackage;

   ----------------------------------------------

   --    check that all own variables of private children (and their public
   --    descendents) have appeared as refinement constituents
   procedure CheckOwnedPackages (Owner : in Dictionary.Symbol;
                                 Node  : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Owner,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      OwnedPackages : Dictionary.Iterator;
      OwnVars       : Dictionary.Iterator;
      PackSym       : Dictionary.Symbol;
      VarSym        : Dictionary.Symbol;

      function GetErrorPos (Node : STree.SyntaxNode)
                           return LexTokenManager.Token_Position
      --# global in STree.Table;
      is -- Node is package_body
         ErrNode : STree.SyntaxNode;
      begin
         ErrNode := Next_Sibling (Child_Node (Node));
         if SyntaxNodeType (ErrNode) =
            SPSymbols.refinement_definition
         then  -- report at last constituent
            ErrNode := Child_Node (Child_Node (ErrNode));
            if SyntaxNodeType (ErrNode) /=
               SPSymbols.refinement_clause
            then
               ErrNode := Next_Sibling (Next_Sibling (ErrNode));
            end if;
            -- ErrNode is now a refinement_clause
            ErrNode := Next_Sibling
               (Child_Node
                (Next_Sibling
                 (Child_Node (ErrNode))));
            if SyntaxNodeType (ErrNode) /= SPSymbols.entire_variable then
               ErrNode := Next_Sibling (ErrNode);
            end if;
         else -- no refinement definition - report at package name
            ErrNode := LastChildOf (Node);
         end if;

         return NodePosition (ErrNode);
      end GetErrorPos;

   begin
      OwnedPackages := Dictionary.FirstOwnedPackage (Owner);
      while not Dictionary.IsNullIterator (OwnedPackages) loop
         PackSym := Dictionary.CurrentSymbol (OwnedPackages);
         OwnVars := Dictionary.FirstOwnVariable (PackSym);

         while not Dictionary.IsNullIterator (OwnVars) loop
            VarSym := Dictionary.CurrentSymbol (OwnVars);
            if not Dictionary.IsRefinementConstituent (Owner, VarSym) then
               -- missing own variable
               ErrorHandler.SemanticErrorSym (621,
                                              ErrorHandler.NoReference,
                                              GetErrorPos (Node),
                                              VarSym,
                                              Dictionary.GlobalScope);
            end if;
            OwnVars := Dictionary.NextSymbol (OwnVars);
         end loop;

         OwnedPackages := Dictionary.NextSymbol (OwnedPackages);
      end loop;
   end CheckOwnedPackages;

   ----------------------------------------------

   procedure wf_refine (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
      is  separate;

   ----------------------------------------------

begin --wf_package_body
      -- ASSUME Node = package_body
   IdentNode := LastChildOf (Node);
   IdentStr := NodeLexString (IdentNode);
   GrandParent := SyntaxNodeType (ParentNode (ParentNode (Node)));
   WithNode := Child_Node (Child_Node (ParentNode (ParentNode (ParentNode (Node)))));
   -- if this chain has got us to a parent_unit_name then we need to
   -- look again for the with clause because we are in a subunit
   if SyntaxNodeType (WithNode) = SPSymbols.parent_unit_name then
      WithNode := Child_Node (Child_Node (ParentNode
                                                  (ParentNode (ParentNode (WithNode)))));
   end if;

   --# assert True;
   if SyntaxNodeType (WithNode) /= SPSymbols.with_clause then
      WithNode := STree.NullNode;
   end if;

   RefNode := Next_Sibling (Child_Node (Node));
   if SyntaxNodeType (RefNode) /= SPSymbols.refinement_definition then
      RefNode := STree.NullNode;
   end if;

   OkToAddBody := False;
   FindPackage (IdentNode, IdentStr, Scope,
                  -- to get
                SpecFound, Sym);

   --# assert True;
   if not SpecFound then
      OkToAddBody := True;

   elsif GrandParent = SPSymbols.abody and then
      Dictionary.HasBodyStub (Sym)
   then
      ErrorHandler.SemanticError (17,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdentStr);

   elsif GrandParent = SPSymbols.subunit then
      -- additional if clause to ensure extra package body subunits reported
      if not Dictionary.HasBodyStub (Sym) then
         ErrorHandler.SemanticError (15,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);
      elsif Dictionary.HasBody (Sym) then
         ErrorHandler.SemanticError (16,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);
      else
         OkToAddBody := True;
      end if;
   elsif Dictionary.HasBody (Sym) then
      ErrorHandler.SemanticError (16,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdentStr);

   else --no errors found
      CheckPackageNeedsBody (IdentNode, Sym);
      OkToAddBody := True;
   end if;

   NextNode := Child_Node (LastSiblingOf (Child_Node (Node)));
   BodyIsHidden := SyntaxNodeType (NextNode) =
      SPSymbols.hidden_part;

   --# assert True;
   if OkToAddBody then
      Dictionary.AddBody (CompilationUnit => Sym,
                          Comp_Unit       => ContextManager.Ops.CurrentUnit,
                          TheBody         => Dictionary.Location'(NodePosition (IdentNode),
                                                                  NodePosition (IdentNode)),
                          Hidden          => BodyIsHidden);
   end if;
   PackScope := Dictionary.LocalScope (Sym);

   --# assert True;
   if WithNode /= STree.NullNode then
      wf_context_clause (ParentNode (WithNode),
                         Sym,
                         PackScope);
   end if;

   --# assert True;
   if RefNode /= STree.NullNode then
      Dictionary.AddRefinementDefinition (Sym,
                                          Dictionary.Location'(NodePosition (RefNode),
                                                               NodePosition (RefNode)));
      wf_refine (RefNode, PackScope);
   end if;

   --# assert True;
   if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 then
      CheckOwnedPackages (Sym, Node);
   end if;

   --# assert True;
   if BodyIsHidden then
      ErrorHandler.HiddenText (NodePosition (NextNode),
                               IdentStr,
                               SPSymbols.package_implementation);
   end if;

   Scope := PackScope;

end wf_package_body;
