-- $Id: sem-compunit-wf_known_discriminant_part.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (Sem.CompUnit)

procedure wf_known_discriminant_part (Node             : in STree.SyntaxNode;
                                      ProtectedTypeSym : in Dictionary.Symbol;
                                      Scope            : in Dictionary.Scopes)
is
   --------------------------------------------------------------------------------------------------
   -- Grammar:
   --                            known_discriminant_part
   --                                      |
   --                           known_discriminant_part_rep --- discriminant_specification
   --                                      |                               |
   --                                     ...                    identifier_list type_mark
   --                                      |                               or
   --                           discriminant_specification       identifier_list access_definition
   --                                                                                   |
   --                                                                                type_mark
   --
   -- Rules:
   --          (1) identifier not already visible
   --          (2) access     -> type_mark is protected type (or susp obj type later)
   --          (3) not access -> type is discrete
   --------------------------------------------------------------------------------------------------

   CurrentNode : STree.SyntaxNode;

   procedure CheckDiscriminant (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in     ProtectedTypeSym;
   --#        in     Scope;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProtectedTypeSym,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProtectedTypeSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      TypeSym  : Dictionary.Symbol;
      IsAccess : Boolean;
      TypeNode : STree.SyntaxNode;

      procedure CheckIdentifiers (Node     : in STree.SyntaxNode;
                                  TypeMark : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in     ProtectedTypeSym;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict           from *,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        ProtectedTypeSym,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        TypeMark &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        ProtectedTypeSym,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeMark;
      is
         -- Node = identifier_list
         It       : STree.Iterator;
         NextNode : STree.SyntaxNode;
         IdentStr : LexTokenManager.Lex_String;
         Sym      : Dictionary.Symbol;
      begin -- CheckIdentifiers
         It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                              FromRoot    => Node,
                              InDirection => STree.Down);

         while not STree.IsNull (It) loop
            NextNode := GetNode (It);
            IdentStr := NodeLexString (NextNode);
            Sym := Dictionary.LookupItem (Name    => IdentStr,
                                          Scope   => Scope,
                                          Context => Dictionary.ProofContext);
            if Sym = Dictionary.NullSymbol then
               Dictionary.AddKnownDiscriminant (Name                => IdentStr,
                                                Comp_Unit           => ContextManager.Ops.CurrentUnit,
                                                Declaration         => Dictionary.Location'(NodePosition (NextNode),
                                                                                            NodePosition (NextNode)),
                                                ProtectedOrTaskType => ProtectedTypeSym,
                                                TypeMark            => TypeMark);
            else -- already exists
               ErrorHandler.SemanticError (10,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           IdentStr);
            end if;
            It := STree.NextNode (It);
         end loop;
      end CheckIdentifiers;

      -----------------------

   begin -- CheckDiscriminant

      -- Node = discriminant_specification

      -- check type mark is valid
      IsAccess := False;
      TypeNode := Next_Sibling (Child_Node (Node));
      if SyntaxNodeType (TypeNode) = SPSymbols.access_definition then
         IsAccess := True;
         TypeNode := Child_Node (TypeNode);
      end if;
      wf_type_mark (TypeNode,
                    Scope,
                    Dictionary.ProgramContext,
                     -- to get
                    TypeSym);
      if TypeSym /= Dictionary.GetUnknownTypeMark then
         if IsAccess then
            -- only a protected type is allowed
            if Dictionary.TypeIsProtected (TypeSym) then
               if TypeSym = ProtectedTypeSym then
                  -- "recursive" use in discriminant
                  TypeSym := Dictionary.GetUnknownTypeMark;
                  -- This error cannot be checked because access types are not
                  -- allowed as descriminants
                  ErrorHandler.SemanticError (902,
                                              ErrorHandler.NoReference,
                                              NodePosition (TypeNode),
                                              Dictionary.GetSimpleName (TypeSym));
               else
                  TypeSym := Dictionary.GetAccess (TypeSym);
               end if;
            else
               TypeSym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.SemanticError (875,
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.Null_String);
            end if;
         else
            -- only a discrete type is allowed
            if not Dictionary.TypeIsDiscrete (TypeSym) then
               TypeSym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.SemanticError (46,
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.Null_String);
            end if;
         end if;
      end if; -- UnknownType
      -- check each identifier associated with type
      CheckIdentifiers (Child_Node (Node),
                        TypeSym);
   end CheckDiscriminant;

   ----------------------

begin -- wf_known_discriminant_part
   CurrentNode := LastChildOf (Node);
   while CurrentNode /= Node loop
      if SyntaxNodeType (CurrentNode) = SPSymbols.discriminant_specification then
         CheckDiscriminant (CurrentNode);
      elsif SyntaxNodeType (Next_Sibling (CurrentNode)) = SPSymbols.discriminant_specification then
         CheckDiscriminant (Next_Sibling (CurrentNode));
      end if;
      -- fall through and ignore anything that's not a discriminant_specification
      CurrentNode := ParentNode (CurrentNode);
   end loop;
end wf_known_discriminant_part;
