-- $Id: sem-compunit-wf_global_definition.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit)
procedure wf_global_definition (Node         : in     STree.SyntaxNode;
                                CurrentScope : in     Dictionary.Scopes;
                                SubprogSym   : in     Dictionary.Symbol;
                                FirstSeen    : in     Boolean;
                                SemErrFound  :    out Boolean)
is
   Abstraction   : Dictionary.Abstractions;
   DoingFunction : Boolean;

   ----------------------------------------------------------------------

   procedure GlobalNotRefined (GlobalSym  : in out Dictionary.Symbol;
                               ErrorPos   : in     LexTokenManager.Token_Position;
                               SubProgSym : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorPos,
   --#                                        GlobalSym,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym &
   --#         GlobalSym,
   --#         SemErrFound               from *,
   --#                                        Dictionary.Dict,
   --#                                        GlobalSym,
   --#                                        SubProgSym;
   is
   begin
      if Dictionary.IsAbstractOwnVariable (GlobalSym) and then
         IsEnclosingPackage (Dictionary.GetOwner (GlobalSym),
                             Dictionary.LocalScope (SubProgSym))
      then
         ErrorHandler.SemanticError (314,
                                     ErrorHandler.NoReference,
                                     ErrorPos,
                                     Dictionary.GetSimpleName (GlobalSym));
         GlobalSym := Dictionary.NullSymbol;
         SemErrFound  := True;
      end if;
   end GlobalNotRefined;

   ----------------------------------------------------------------------

   procedure CheckStreamModeConsistency (GlobalSym  : in out Dictionary.Symbol;
                                         Mode       : in     Dictionary.Modes;
                                         ErrorPos   : in     LexTokenManager.Token_Position;
                                         SubProgSym : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorPos,
   --#                                        GlobalSym,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym &
   --#         GlobalSym,
   --#         SemErrFound               from *,
   --#                                        Dictionary.Dict,
   --#                                        GlobalSym,
   --#                                        Mode,
   --#                                        SubProgSym;
   is
      PossibleStreamMode : Dictionary.Modes;
   begin
      PossibleStreamMode := Dictionary.GetOwnVariableOrConstituentMode (GlobalSym);
      -- PossibleStreamMode contains DefaultMode unless the global is a stream
      -- variable or a stream constituent in which case it will be either InMode
      -- or OutMode (it can never be InOutMode because of earlier wffs).

      -- checks are only required if the mode is something other than DefaultMode
      if PossibleStreamMode /= Dictionary.DefaultMode then
         if Dictionary.IsFunction (SubProgSym) then
            -- required check is that global has default mode or mode in
            if PossibleStreamMode = Dictionary.OutMode then
               ErrorHandler.SemanticError (709,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (GlobalSym));
               GlobalSym := Dictionary.NullSymbol;
               SemErrFound  := True;
            end if;

         else -- handle procedure
            -- required check is that a moded own variable must have the same mode
            -- as the global mode if one is given.  Also, global mode in out cannot
            -- be used if the own variable is moded.
            if Mode = Dictionary.InMode and then
               PossibleStreamMode /= Mode then
               ErrorHandler.SemanticError (711,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (GlobalSym));
               GlobalSym := Dictionary.NullSymbol;
               SemErrFound  := True;
            elsif Mode = Dictionary.OutMode and then
               PossibleStreamMode /= Mode then
               ErrorHandler.SemanticError (710,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (GlobalSym));
               GlobalSym := Dictionary.NullSymbol;
               SemErrFound  := True;
            elsif Mode = Dictionary.InOutMode then
               ErrorHandler.SemanticError (712,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (GlobalSym));
               GlobalSym := Dictionary.NullSymbol;
               SemErrFound  := True;
            end if;
         end if;
      end if;
   end CheckStreamModeConsistency;

   ----------------------------------------------------------------------

   procedure UniqueGlobal (GlobalSym  : in out Dictionary.Symbol;
                           Dotted     : in     Boolean;
                           ErrorPos   : in     LexTokenManager.Token_Position;
                           SubprogSym : in     Dictionary.Symbol)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Dotted,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorPos,
   --#                                        GlobalSym,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym &
   --#         GlobalSym,
   --#         SemErrFound               from *,
   --#                                        Abstraction,
   --#                                        Dictionary.Dict,
   --#                                        Dotted,
   --#                                        GlobalSym,
   --#                                        LexTokenManager.State,
   --#                                        SubProgSym;
   is
      It        : Dictionary.Iterator;
      GlobalStr : LexTokenManager.Lex_String;

   begin
      GlobalStr := Dictionary.GetSimpleName (GlobalSym);
      if not Dotted and then
        Dictionary.IsSubprogram (SubprogSym) then -- task types have no parameters so skip search
         It := Dictionary.FirstSubprogramParameter (SubprogSym);
         while It /= Dictionary.NullIterator loop
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GlobalStr,
                                                                    Lex_Str2 => Dictionary.GetSimpleName (Dictionary.CurrentSymbol (It))) = LexTokenManager.Str_Eq then
               ErrorHandler.SemanticError (158,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           GlobalStr);
               GlobalSym := Dictionary.NullSymbol;
               SemErrFound  := True;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;

      if GlobalSym /= Dictionary.NullSymbol then
         It := Dictionary.FirstGlobalVariable (Abstraction,
                                               SubprogSym);
         while It /= Dictionary.NullIterator loop
            if GlobalSym = Dictionary.CurrentSymbol (It) then
               ErrorHandler.SemanticError (157,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           GlobalStr);
               GlobalSym := Dictionary.NullSymbol;
               SemErrFound  := True;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
   end UniqueGlobal;

   ----------------------------------------------------------------------

   procedure CheckImportInit (ImportNode : in STree.SyntaxNode;
                              ImportSym,
                              SubprogSym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ImportNode,
   --#                                        ImportSym,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         SemErrFound               from *,
   --#                                        Dictionary.Dict,
   --#                                        ImportSym,
   --#                                        SubProgSym;
   is

   begin
      if Dictionary.IsMainProgram (SubprogSym) and then
         Dictionary.IsFunction (SubprogSym) and then
         not Dictionary.OwnVariableIsInitialized (ImportSym) and then
         Dictionary.GetOwnVariableOrConstituentMode (ImportSym) = Dictionary.DefaultMode
      then
         SemErrFound := True;
         ErrorHandler.SemanticError (167,
                                     ErrorHandler.NoReference,
                                     NodePosition (ImportNode),
                                     Dictionary.GetSimpleName (ImportSym));
      end if;
   end CheckImportInit;

   -----------------------------------------------------------------------

   procedure CheckOnGlobalRefinement_1 (SubprogSym : in Dictionary.Symbol;
                                        VarSym     : in Dictionary.Symbol;
                                        ErrPos     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrPos,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym,
   --#                                        VarSym &
   --#         SemErrFound               from *,
   --#                                        Dictionary.Dict,
   --#                                        SubProgSym,
   --#                                        VarSym;
   is
      SubjectSym : Dictionary.Symbol;
      Found      : Boolean;
      It         : Dictionary.Iterator;

      function ValidRefinement (Constituent,
                                Subprogram   : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Owner           : Dictionary.Symbol;
         Region          : Dictionary.Symbol;
         EnclosingRegion : Dictionary.Symbol;
         Result          : Boolean;
      begin
         -- We regard Constituent as a ValidRefinement if:
         --   It is a refinement constituent of Subject AND
         --      (Subject is owned by the region in which the Subprogram is declared OR
         --       Subject is owned by the region in which the protected type in which the
         --          Subprogram is declared)
         Owner := Dictionary.GetOwner (Dictionary.GetSubject (Constituent));
         Region := Dictionary.GetRegion (Dictionary.GetScope (Subprogram));

         Result :=  Owner = Region;
         if not Result and then Dictionary.IsProtectedType (Region) then
            EnclosingRegion := Dictionary.GetRegion (Dictionary.GetScope (Region));
            Result := Owner = EnclosingRegion;
         end if;
         return Result;
      end ValidRefinement;


   begin
      if Dictionary.IsConstituent (VarSym) then
         -- look for refinement subject in first global anno
         SubjectSym := Dictionary.GetSubject (VarSym);
         -- if subject not from this package then use own var itself
         --if Dictionary.GetOwner (SubjectSym) /=
         --   Dictionary.GetRegion (Dictionary.GetScope (SubprogSym))
         if not ValidRefinement (VarSym, SubprogSym) then
            SubjectSym := VarSym;
         end if;
      else
         --look for global itself in first global anno
         SubjectSym := VarSym;
      end if;

      Found := False;
      It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                            SubprogSym);
      loop
         exit when Dictionary.IsNullIterator (It);
         Found := SubjectSym = Dictionary.CurrentSymbol (It);
         exit when Found;
         It := Dictionary.NextSymbol (It);
      end loop;
      if not Found then
         ErrorHandler.SemanticErrorSym (85,
                                        ErrorHandler.NoReference,
                                        ErrPos,
                                        VarSym,
                                        Dictionary.GetScope (SubprogSym));
         SemErrFound  := True;
      end if;
   end CheckOnGlobalRefinement_1;

   ----------------------------------------------------------------------

   -- Handle Special case for stream variables here:
   -- Error occurs if the subprogram references a global which is a mode IN refinement
   -- constituent whose abstract subject is unmoded.  This condition is illegal for
   -- a function subprogram (because it would conceal the stream side-efefct in the
   -- abstract view).  For a procedure, we are protected because derives refinement
   -- checks will always reveal the error; however, we can improve error reporting
   -- by putting something out here as well
   procedure CheckStreamVariableRefinement (SubprogSym : in Dictionary.Symbol;
                                            ErrPos     :    LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrPos,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym &
   --#         SemErrFound               from *,
   --#                                        Dictionary.Dict,
   --#                                        SubProgSym;
   is
      RefinedGlobal   : Dictionary.Symbol;
      AbstractGlobal  : Dictionary.Symbol;
      RefinedGlobList : Dictionary.Iterator;

      function IsLocalConstituent (Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in RefinedGlobal;
      --#        in SubProgSym;
      is
      begin
         return Dictionary.IsConstituent (Sym) and then
           (Dictionary.GetEnclosingPackage (Dictionary.GetScope (SubprogSym)) =
              Dictionary.GetOwner (RefinedGlobal));
      end IsLocalConstituent;

   begin --CheckStreamVariableRefinement
      RefinedGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsRefined,
                                                         SubprogSym);
      while not Dictionary.IsNullIterator (RefinedGlobList) loop
         RefinedGlobal := Dictionary.CurrentSymbol (RefinedGlobList);
         if IsLocalConstituent (RefinedGlobal) then
            AbstractGlobal := Dictionary.GetSubject (RefinedGlobal);
            if  Dictionary.GetOwnVariableOrConstituentMode (RefinedGlobal) =
              Dictionary.InMode and then
               -- Abstract has no mode
              Dictionary.GetOwnVariableOrConstituentMode (AbstractGlobal) =
              Dictionary.DefaultMode and then
               -- Abstract is not protected
              not (Dictionary.IsOwnVariable (AbstractGlobal) and then
                     Dictionary.GetOwnVariableProtected (AbstractGlobal)) then
               -- an error condition may exist
               --# accept Flow, 41, "Expected stable expression";
               if Dictionary.IsAdaFunction (SubprogSym) then --expect stable
               --# end accept;
                  -- an error condition does exist
                  ErrorHandler.SemanticError2 (721,
                                               ErrorHandler.NoReference,
                                               ErrPos,
                                               Dictionary.GetSimpleName (RefinedGlobal),
                                               Dictionary.GetSimpleName (AbstractGlobal));
                  SemErrFound := True;
               else -- must be procedure
                  -- extra checks here - for there to be an error the abstract own variable
                  -- must be have global mode IN and the refinement constituent as well
                  if Dictionary.GetGlobalMode (Dictionary.IsAbstract,
                                               SubprogSym,
                                               AbstractGlobal) = Dictionary.InMode and then
                     Dictionary.GetGlobalMode (Dictionary.IsRefined,
                                               SubprogSym,
                                               RefinedGlobal) = Dictionary.InMode then
                     ErrorHandler.SemanticError2 (722,
                                                  ErrorHandler.NoReference,
                                                  ErrPos,
                                                  Dictionary.GetSimpleName (RefinedGlobal),
                                                  Dictionary.GetSimpleName (AbstractGlobal));
                     SemErrFound := True;
                  end if;
               end if;
            end if;
         end if;
         RefinedGlobList := Dictionary.NextSymbol (RefinedGlobList);
      end loop;
   end CheckStreamVariableRefinement;

   ----------------------------------------------------------------------

   function TypeTwoErrorPosition (Node : STree.SyntaxNode)
                                 return LexTokenManager.Token_Position
   --# global in STree.Table;
   is
      TempNode : STree.SyntaxNode;
   begin
      --assume Node = global_definition
      TempNode := Child_Node (Child_Node (Node));
      if SyntaxNodeType (TempNode) = SPSymbols.global_definition_rep then
         TempNode := Next_Sibling (TempNode);
      end if;
      -- TempNode now points at the right-most global_variable_clause
      TempNode := Child_Node (Next_Sibling (Child_Node (TempNode)));
      -- TempNode now either a global_variable or a global_variable_list
      if SyntaxNodeType (TempNode) = SPSymbols.global_variable_list then
         TempNode := Next_Sibling (TempNode);
      end if;
      -- TempNode is right-most global variable
      return NodePosition (TempNode);
   end TypeTwoErrorPosition;

   ----------------------------------------------------------------------

   procedure CheckOnGlobalRefinement_2 (SubprogSym : in Dictionary.Symbol;
                                        ErrPos     : LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrPos,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym &
   --#         SemErrFound               from *,
   --#                                        Dictionary.Dict,
   --#                                        SubProgSym;
   is
      AGloVar        : Dictionary.Symbol;
      --Found          : Boolean;
      FirstGlobList  : Dictionary.Iterator;
      --SecondGlobList : Dictionary.Iterator;

      -- lifted out from loop in enclosing procedure below
      procedure LookForRefinementConstituent
      --# global in     AGloVar;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrPos;
      --#        in     LexTokenManager.State;
      --#        in     SubProgSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SemErrFound;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from AGloVar,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrPos,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        SubProgSym &
      --#         SemErrFound               from *,
      --#                                        AGloVar,
      --#                                        Dictionary.Dict,
      --#                                        SubProgSym;
      is
         Found          : Boolean;
         SecondGlobList : Dictionary.Iterator;

      begin -- LookForRefinementConstituent
         Found := False;
         SecondGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsRefined,
                                                           SubprogSym);
         loop
            exit when Dictionary.IsNullIterator (SecondGlobList);
            Found := Dictionary.IsRefinement (AGloVar,
                                              Dictionary.CurrentSymbol (SecondGlobList));
            exit when Found;
            SecondGlobList := Dictionary.NextSymbol (SecondGlobList);
         end loop;
         if not Found then
            ErrorHandler.SemanticErrorSym (86,
                                           ErrorHandler.NoReference,
                                           ErrPos,
                                           AGloVar,
                                           Dictionary.GetScope (SubprogSym));
            SemErrFound  := True;
         end if;
      end LookForRefinementConstituent;

      -- new procedure similar to above but for unrefined global items
      procedure LookForSelf
      --# global in     AGloVar;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrPos;
      --#        in     LexTokenManager.State;
      --#        in     SubProgSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SemErrFound;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from AGloVar,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrPos,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        SubProgSym &
      --#         SemErrFound               from *,
      --#                                        AGloVar,
      --#                                        Dictionary.Dict,
      --#                                        SubProgSym;
      is
         Found          : Boolean;
         SecondGlobList : Dictionary.Iterator;

      begin -- LookForSelf
         Found := False;
         SecondGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsRefined,
                                                           SubprogSym);
         loop
            exit when Dictionary.IsNullIterator (SecondGlobList);
            Found := AGloVar = Dictionary.CurrentSymbol (SecondGlobList);
            exit when Found;
            SecondGlobList := Dictionary.NextSymbol (SecondGlobList);
         end loop;
         if not Found and then
           AGloVar /= Dictionary.GetNullVariable then -- no refinement of "null" needed
            ErrorHandler.SemanticErrorSym (723,
                                           ErrorHandler.NoReference,
                                           ErrPos,
                                           AGloVar,
                                           Dictionary.GetScope (SubprogSym));
            SemErrFound  := True;
         end if;
      end LookForSelf;

      function ConstituentRequired (TheGlobal,
                                    Subprogram : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean := False;
         Region : Dictionary.Symbol;
      begin
         -- We expect a refinement sonstituent of TheGlobal if:
         --   TheGlobal is an abstract own variable  and
         --      (its owner is the package in which Subprogram is declared or
         --       the Subprogram is declared in a protected type and the owner
         --       of TheGlobal is the package in which the protected type is declared)
         if Dictionary.IsAbstractOwnVariable (TheGlobal) then
            Result := IsEnclosingPackage (Dictionary.GetOwner (TheGlobal),
                                          Dictionary.GetScope (Subprogram));
            if not Result then
               Region := Dictionary.GetRegion (Dictionary.GetScope (Subprogram));
               if Dictionary.IsProtectedType (Region) then
                  Result := IsEnclosingPackage (Dictionary.GetOwner (TheGlobal),
                                                Dictionary.GetScope (Region));
               end if;
            end if;
         end if;
         return Result;
      end ConstituentRequired;

   begin -- CheckOnGlobalRefinement_2
      FirstGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                                       SubprogSym);
      while not Dictionary.IsNullIterator (FirstGlobList) loop
         AGloVar := Dictionary.CurrentSymbol (FirstGlobList);
         if ConstituentRequired (AGloVar, SubprogSym) then
            LookForRefinementConstituent; -- at least on constituent must be present
         else
            LookForSelf;                  -- own variable itself must be present
         end if;
         FirstGlobList := Dictionary.NextSymbol (FirstGlobList);
      end loop;
   end CheckOnGlobalRefinement_2;

   ----------------------------------------------------------------------

   procedure CheckMainProgramGlobal (Node         : in     STree.SyntaxNode;
                                     CurrentScope : in     Dictionary.Scopes;
                                     VarSym       :    out Dictionary.Symbol;
                                     Dotted       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dotted,
   --#         STree.Table,
   --#         VarSym                    from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      IdNode,
      NextIdNode : STree.SyntaxNode;
      PIdStr,
      IdStr      : LexTokenManager.Lex_String;
      PSym,
      Sym        : Dictionary.Symbol;

   begin
      IdNode := LastChildOf (Node);
      IdStr := NodeLexString (IdNode);
      PIdStr := LexTokenManager.Null_String;
      Sym := Dictionary.LookupItem (Name    => IdStr,
                                    Scope   => CurrentScope,
                                    Context => Dictionary.ProofContext);
      loop  -- introduced for multiple prefixes
         if Sym = Dictionary.NullSymbol then
            ErrorHandler.SemanticError2 (144, ErrorHandler.NoReference,
                                         NodePosition (Node),
                                         IdStr,
                                         PIdStr);
            exit;
         end if;

         NextIdNode := Next_Sibling (ParentNode (IdNode));

         if Dictionary.IsOwnVariable (Sym) then
            -- entire variable check
            -- at this point Sym is a variable, final check that there is no dotted
            -- part to the right of it as there would be if a record field was there
            if NextIdNode /= STree.NullNode then
               ErrorHandler.SemanticError (156, --entire var expected
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.Null_String);
               Sym := Dictionary.NullSymbol;
            end if;
            exit;
         end if;

         if not Dictionary.IsPackage (Sym) then
            ErrorHandler.SemanticError (174,                 --own var expected
                                        ErrorHandler.NoReference,
                                        NodePosition (IdNode),
                                        IdStr);
            Sym := Dictionary.NullSymbol;
            exit;
         end if;

         if NextIdNode = STree.NullNode then
            -- package without a selected component
            ErrorHandler.SemanticError (174,                    --own var expected
                                        ErrorHandler.NoReference,
                                        NodePosition (IdNode),
                                        IdStr);
            Sym := Dictionary.NullSymbol;
            exit;
         end if;
         STree.Set_Node_Lex_String (Sym  => Sym,
                                    Node => IdNode);
         PIdStr := IdStr;
         IdNode := NextIdNode;
         IdStr := NodeLexString (IdNode);
         PSym := Sym;
         Sym := Dictionary.LookupSelectedItem (Prefix   => PSym,
                                               Selector => IdStr,
                                               Scope    => Dictionary.VisibleScope (Sym),
                                               Context  => Dictionary.ProofContext);
         if Sym = Dictionary.NullSymbol then
            -- need also to search current scope for inherited child packages
            Sym := Dictionary.LookupSelectedItem (Prefix   => PSym,
                                                  Selector => IdStr,
                                                  Scope    => CurrentScope,
                                                  Context  => Dictionary.ProofContext);
         end if;
      end loop;

      Dotted := LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => PIdStr,
                                                                     Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq and then
        Sym /= Dictionary.NullSymbol;
      VarSym := Sym;
   end CheckMainProgramGlobal;

   --------------------------------------------------

   procedure ProcessOneGlobal (Node        : in     STree.SyntaxNode;
                               Mode        : in     Dictionary.Modes)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     CurrentScope;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in     SubProgSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict           from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SubProgSym &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         SemErrFound               from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym;
   --  pre     SyntaxNodeType (Node) = SPSymbols.global_variable;
   is
      Dotted    : Boolean;
      Sym       : Dictionary.Symbol;
      ParamMode : Dictionary.Modes;
   begin --ProcessOneGlobal
      if Dictionary.IsMainProgram (SubprogSym) then
         CheckMainProgramGlobal (Child_Node (Node),
                                 Dictionary.LocalScope (SubprogSym),
                                 --to get
                                 Sym,
                                 Dotted);
      else
         wf_entire_variable (Child_Node (Node),
                             CurrentScope,
                             In_Global_List,
                              --to get
                             Sym,
                             Dotted);
      end if;

      --# assert True;
      if Sym = Dictionary.NullSymbol then
         SemErrFound := True;
      end if;

      if CommandLineData.RavenscarSelected and then
        (SubprogSym /= Dictionary.GetThePartition) and then
        Dictionary.IsOwnVariable (Sym) and then
        Dictionary.GetOwnVariableIsInterruptStream (Sym) then
         -- An interrupt stream variable is being used outside the partition
         -- wide flow annotation.
         ErrorHandler.SemanticErrorSym (955,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Sym,
                                        CurrentScope);
         Sym := Dictionary.NullSymbol;
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol and then
        (Dictionary.IsMainProgram (SubprogSym) or
           Dictionary.IsTaskType (SubprogSym)) then
         -- check to ensure that global is initialized
         if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
           not CommandLineData.Content.DoInformationFlow and then
           (Mode = Dictionary.InMode or Mode = Dictionary.InOutMode) and then
           not Dictionary.OwnVariableIsInitialized (Sym) and then
           Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.DefaultMode
         then
            ErrorHandler.SemanticError (167,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (Sym));
         end if;
      end if;

      --cfr1203 -- predefined suspension objects not allowed in partition global list
      --cfr1203 if Sym /= Dictionary.NullSymbol and then
      --cfr1203   CommandLineData.RavenscarSelected and then
      --cfr1203   (SubprogSym = Dictionary.GetThePartition) and then
      --cfr1203   Dictionary.IsPredefinedSuspensionObjectType (Dictionary.GetType (Sym)) then
      --cfr1203    SemErrFound := True;
      --cfr1203    ErrorHandler.SemanticError (985,
      --cfr1203                                ErrorHandler.NoReference,
      --cfr1203                                NodePosition (Node),
      --cfr1203                                Dictionary.GetSimpleName (Sym));
      --cfr1203 end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         GlobalNotRefined (Sym,
                           NodePosition (Child_Node (Node)),
                           SubprogSym);
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         UniqueGlobal (Sym,
                       Dotted,
                       NodePosition (Child_Node (Node)),
                       SubprogSym);
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         CheckStreamModeConsistency (Sym,
                                     Mode,
                                     NodePosition (Child_Node (Node)),
                                     SubprogSym);
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol and then
        Dictionary.IsOwnVariable (Sym) and then
        Dictionary.IsVirtualElement (Sym) then
         if Dictionary.IsOrIsInProtectedScope (CurrentScope) and then
           Dictionary.GetEnclosingProtectedRegion (CurrentScope) =
           Dictionary.GetRootType
           (Dictionary.GetType
              (Dictionary.GetVirtualElementOwner (Sym))) then

            -- Mark the virtual element as having been seen by its "owning"
            -- protected type.
            --
            Dictionary.SetVirtualElementSeenByOwner (Sym);
         else
            -- This is an access to a protected virtual element outside
            -- it's protected type.
            --
            ErrorHandler.SemanticErrorSym2
              (946,
               ErrorHandler.NoReference,
               NodePosition (Node),
               Sym,
               Dictionary.GetRootType
                 (Dictionary.GetType
                    (Dictionary.GetVirtualElementOwner (Sym))),
               CurrentScope);
         end if;
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         if not FirstSeen then
            CheckOnGlobalRefinement_1 (SubprogSym,
                                       Sym,
                                       NodePosition (Child_Node (Node)));
         end if;

         CheckImportInit (Node, Sym, SubprogSym);

         -- check that mode on a global of an embedded procedure is compatible with
         -- parameters mode if the global is refering to a parameter of an enclosing
         -- procedure
         if
            Dictionary.IsSubprogramParameter (Sym)
         then
            ParamMode := Dictionary.GetSubprogramParameterMode (Sym);
            if ParamMode = Dictionary.InMode or ParamMode = Dictionary.DefaultMode then
               if Mode /= Dictionary.InMode and Mode /= Dictionary.DefaultMode then
                  ErrorHandler.SemanticError (508,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);

               end if;
            end if;
         end if;

         Dictionary.AddGlobalVariable (Abstraction       => Abstraction,
                                       Subprogram        => SubprogSym,
                                       Variable          => Sym,
                                       Mode              => Mode,
                                       PrefixNeeded      => Dotted,
                                       Comp_Unit         => ContextManager.Ops.CurrentUnit,
                                       VariableReference => Dictionary.Location'(NodePosition (Node),
                                                                                 NodePosition (Node)));

         -- Mark the global variable as being referenced by a thread if the
         -- owner of this global variable is a thread (i.e. task, main program or
         -- interrupt handler) and the variable is not protected. This data is
         -- required by the main program shared variable checks.
         --
         if Dictionary.IsThread (SubprogSym) and then
           Dictionary.IsOwnVariable (Sym) and then
           not Dictionary.GetOwnVariableProtected (Sym) then
            Dictionary.SetUsesUnprotectedVariables (SubprogSym);
         end if;

      end if;
   end ProcessOneGlobal;

   ----------------------------------------------------------------------

   procedure ProcessGlobalList (Node        : in     STree.SyntaxNode;
                                Mode        : in     Dictionary.Modes)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     CurrentScope;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in     SubProgSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         SemErrFound               from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Mode,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym;
   is
      It : STree.Iterator;
   begin
      It := FindFirstNode (NodeKind    => SPSymbols.global_variable,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         ProcessOneGlobal (GetNode (It), Mode);
         It := STree.NextNode (It);
      end loop;
   end ProcessGlobalList;

   ----------------------------------------------------------------------

   procedure wf_moded_global_definition (Node : in     STree.SyntaxNode)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     CurrentScope;
   --#        in     DoingFunction;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in     SubProgSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        DoingFunction,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubProgSym &
   --#         SemErrFound               from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        DoingFunction,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        SubProgSym;
   is
      It       : STree.Iterator;
      NextNode : STree.SyntaxNode;
      Mode     : Dictionary.Modes;

      function GetMode (Node : STree.SyntaxNode)
                       return Dictionary.Modes
      --# global in STree.Table;
      --  pre    SyntaxNodeType (Node) = SPSymbols.global_variable_clause;
      is
         ModeNode : STree.SyntaxNode;
         Result   : Dictionary.Modes;
      begin
         ModeNode := Child_Node (Child_Node (Node));
         if ModeNode = STree.NullNode then
            Result := Dictionary.DefaultMode;
         else
            case SyntaxNodeType (ModeNode) is
               when SPSymbols.in_mode    => Result := Dictionary.InMode;
               when SPSymbols.inout_mode => Result := Dictionary.InOutMode;
               when SPSymbols.outmode    => Result := Dictionary.OutMode;
               when others               => Result := Dictionary.DefaultMode; --cannot occur
            end case;
         end if;
         return Result;
      end GetMode;

      --------------

   begin -- wf_moded_global_definition
      It := FindFirstNode (NodeKind    => SPSymbols.global_variable_clause,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := GetNode (It);
         Mode := GetMode (NextNode);
         --# accept Flow, 41, "Expected stable expression";
         if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 then
         --# end accept;
            if not CommandLineData.Content.DoInformationFlow and then
               not DoingFunction and then
               Mode = Dictionary.DefaultMode
            then
               SemErrFound := True;
               ErrorHandler.SemanticError (500,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           LexTokenManager.Null_String);
            end if;
         end if;

         if DoingFunction and then
            (Mode = Dictionary.OutMode or else
             Mode = Dictionary.InOutMode) then
               SemErrFound := True;
               ErrorHandler.SemanticError (169,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           LexTokenManager.Null_String);
         end if;

         ProcessGlobalList (Next_Sibling (Child_Node (NextNode)),
                            Mode);
         It := STree.NextNode (It);
      end loop;

   end wf_moded_global_definition;

   ----------------------------------------------------------------------

begin -- wf_global_definition
      -- ASSUME Node = moded_global_definition
   SemErrFound  := False;
   DoingFunction := Dictionary.IsFunction (SubprogSym);

   if FirstSeen then
      Abstraction := Dictionary.IsAbstract;
   else
      Abstraction := Dictionary.IsRefined;
   end if;

   Dictionary.AddGlobalAnnotation (Abstraction => Abstraction,
                                   Subprogram  => SubprogSym,
                                   Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                   Annotation  => Dictionary.Location'(NodePosition (Node),
                                                                       NodePosition (Node)));

   wf_moded_global_definition (Node);

   if not FirstSeen then
      CheckOnGlobalRefinement_2 (SubprogSym,
                                 TypeTwoErrorPosition (Node));
      CheckStreamVariableRefinement (SubprogSym,
                                     TypeTwoErrorPosition (Node));
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Global (Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                Parse_Tree  => Node,
                                Scope       => CurrentScope,
                                Subprog_Sym => SubprogSym);
   end if;

end wf_global_definition;
