-- $Id: sem-compunit-wf_generic_declaration.adb 16669 2010-04-01 11:26:15Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Debug;
with SLI;

separate (Sem.CompUnit)

procedure wf_generic_declaration (Node         : in STree.SyntaxNode;
                                  CurrentScope : in Dictionary.Scopes)
is
   CurrentNode : STree.SyntaxNode;

   procedure ProcessGenericFormalPart (Node               : in     STree.SyntaxNode;
                                       GenericUnit        : in     Dictionary.Symbol;
                                       ErrorInDeclaration : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GenericUnit,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Dictionary.Dict,
   --#         ErrorInDeclaration,
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GenericUnit,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table;
   is
      ObjectOrTypeNode : STree.SyntaxNode;
      RepNode          : STree.SyntaxNode;

      procedure ProcessFormalObject (Node               : in     STree.SyntaxNode;
                                     GenericUnit        : in     Dictionary.Symbol;
                                     ErrorInDeclaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         ErrorInDeclaration        from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        GenericUnit,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table &
      --#         STree.Table               from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        GenericUnit,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GenericUnit,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
         IdentListNode : STree.SyntaxNode;
         ModeNode      : STree.SyntaxNode;
         TypeNode      : STree.SyntaxNode;
         FormalType    : Dictionary.Symbol;

         procedure ProcessIdentifiers (Node               : in     STree.SyntaxNode;
                                       TypeSym            : in     Dictionary.Symbol;
                                       GenericUnit        : in     Dictionary.Symbol;
                                       ErrorInDeclaration : in out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.UnitStack;
         --#        in     LexTokenManager.State;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Dictionary.Dict,
         --#         ErrorInDeclaration        from *,
         --#                                        ContextManager.Ops.UnitStack,
         --#                                        Dictionary.Dict,
         --#                                        GenericUnit,
         --#                                        LexTokenManager.State,
         --#                                        Node,
         --#                                        STree.Table,
         --#                                        TypeSym &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
         --#                                        ContextManager.Ops.UnitStack,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        GenericUnit,
         --#                                        LexTokenManager.State,
         --#                                        Node,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table,
         --#                                        TypeSym;
         is
            It       : STree.Iterator;
            NextNode : STree.SyntaxNode;
            IdentStr : LexTokenManager.Lex_String;
            Sym      : Dictionary.Symbol;

         begin -- ProcessIdentifiers
            It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                                 FromRoot    => Node,
                                 InDirection => STree.Down);

            while not STree.IsNull (It) loop
               NextNode := GetNode (It);
               IdentStr := NodeLexString (NextNode);
               Sym := Dictionary.LookupItem (Name    => IdentStr,
                                             Scope   => Dictionary.GlobalScope,
                                             Context => Dictionary.ProofContext);
               if Sym = Dictionary.NullSymbol then
                  Dictionary.AddGenericFormalParameter
                    (Name                => IdentStr,
                     Comp_Unit           => ContextManager.Ops.CurrentUnit,
                     Declaration         => Dictionary.Location'(NodePosition (NextNode),
                                                                 NodePosition (NextNode)),
                     SubprogramOrPackage => GenericUnit,
                     TypeMark            => TypeSym,
                     Kind                => Dictionary.GenericObjectParameter);
               else
                  -- already exists
                  ErrorInDeclaration := True;
                  ErrorHandler.SemanticError (10,
                                              ErrorHandler.NoReference,
                                              NodePosition (NextNode),
                                              IdentStr);
               end if;
               It := STree.NextNode (It);
            end loop;
         end ProcessIdentifiers;

      begin -- ProcessFormalObject

         -- Grammar
         --  formal_object_declaration :
         --         identifier_list mode type_mark  ;
         IdentListNode := Child_Node (Node);
         ModeNode := Child_Node (Next_Sibling (IdentListNode));
         TypeNode := Next_Sibling (Next_Sibling (IdentListNode));

         if SyntaxNodeType (ModeNode) = SPSymbols.inout_mode or else
           SyntaxNodeType (ModeNode) = SPSymbols.outmode then
            ErrorInDeclaration := True;
            ErrorHandler.SemanticError (639,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentListNode),
                                        LexTokenManager.Null_String);
         end if;

         wf_type_mark (TypeNode,
                       Dictionary.LocalScope (GenericUnit),
                       Dictionary.ProgramContext,
                        --to get
                       FormalType);
         if FormalType = Dictionary.GetUnknownTypeMark then
            ErrorInDeclaration := True;

         else -- valid type mark so
            ProcessIdentifiers (IdentListNode,
                                FormalType,
                                GenericUnit,
                                ErrorInDeclaration);
         end if;
      end ProcessFormalObject;

      -------------

      procedure ProcessFormalType (Node               : in     STree.SyntaxNode;
                                   GenericUnit        : in     Dictionary.Symbol;
                                   ErrorInDeclaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SLI.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --# derives SLI.State                 from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GenericUnit,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.File_Sys,
      --#                                        STree.Table &
      --#         Dictionary.Dict,
      --#         ErrorInDeclaration,
      --#         STree.Table               from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        GenericUnit,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.FileHeap,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GenericUnit,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SLI.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
         IdentNode           : STree.SyntaxNode;
         TypeNode            : STree.SyntaxNode;
         IdentStr            : LexTokenManager.Lex_String;
         Declaration         : Dictionary.Location;
         TypeMark            : Dictionary.Symbol;
         ErrorInGenericArray : Boolean;
      begin --ProcessFormalType

         -- Grammar
         --  formal_type_declaration :
         --         identifier formal_type_definition ;

         --  formal_type_definition :
         --         formal_private_type_definition
         --       | formal_discrete_type_definition
         --       | formal_signed_integer_type_definition
         --       | formal_modular_type_definition
         --       | formal_floating_point_definition
         --       | formal_ordinary_fixed_point_definition
         --       | formal_array_type_definition ;

         --  formal_private_type_definition :
         --         RWprivate ;
         --  formal_discrete_type_definition :
         --         left_paren box right_paren ;
         --  formal_signed_integer_type_definition :
         --         RWrange box ;
         --  formal_modular_type_definition :
         --         RWmod box ;
         --  formal_floating_point_definition :
         --         RWdigits box ;
         --  formal_ordinary_fixed_point_definition :
         --         RWdelta box ;
         --  formal_array_type_definition :
         --         array_type_definition ;

         IdentNode := Child_Node (Node);
         IdentStr := NodeLexString (IdentNode);

         -- check whether name already in use
         if Dictionary.IsDefined (IdentStr,
                                  Dictionary.GlobalScope,
                                  Dictionary.ProofContext)
         then
            ErrorHandler.SemanticError (10, --illegal redeclaration
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
            ErrorInDeclaration := True;

         else -- OK to add
            TypeNode := Child_Node (Next_Sibling (IdentNode));
            Declaration := Dictionary.Location'(NodePosition (IdentNode),
                                                NodePosition (IdentNode));
            case SyntaxNodeType (TypeNode) is
               when SPSymbols.formal_private_type_definition =>
                  Dictionary.AddGenericPrivateType (Name        => IdentStr,
                                                    Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                    Declaration => Declaration,
                                                    Scope       => Dictionary.LocalScope (GenericUnit),
                                                    -- to get
                                                    TypeSym     => TypeMark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => TypeMark,
                                               Is_Declaration => True);
                  end if;
               when SPSymbols.formal_discrete_type_definition =>
                  Dictionary.AddGenericOrderedDiscreteType (Name        => IdentStr,
                                                            Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                            Declaration => Declaration,
                                                            Scope       => Dictionary.LocalScope (GenericUnit),
                                                            -- to get
                                                            TypeSym     => TypeMark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => TypeMark,
                                               Is_Declaration => True);
                  end if;
               when SPSymbols.formal_signed_integer_type_definition =>
                  Dictionary.AddGenericIntegerType (Name        => IdentStr,
                                                    Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                    Declaration => Declaration,
                                                    Scope       => Dictionary.LocalScope (GenericUnit),
                                                    -- to get
                                                    TypeSym     => TypeMark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => TypeMark,
                                               Is_Declaration => True);
                  end if;
               when SPSymbols.formal_modular_type_definition =>
                  Dictionary.AddGenericModularType (Name        => IdentStr,
                                                    Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                    Declaration => Declaration,
                                                    Scope       => Dictionary.LocalScope (GenericUnit),
                                                    -- to get
                                                    TypeSym     => TypeMark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => TypeMark,
                                               Is_Declaration => True);
                  end if;
               when SPSymbols.formal_floating_point_definition =>
                  Dictionary.AddGenericFloatingPointType (Name        => IdentStr,
                                                          Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                          Declaration => Declaration,
                                                          Scope       => Dictionary.LocalScope (GenericUnit),
                                                          -- to get
                                                          TypeSym     => TypeMark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => TypeMark,
                                               Is_Declaration => True);
                  end if;
               when SPSymbols.formal_ordinary_fixed_point_definition =>
                  Dictionary.AddGenericFixedPointType (Name        => IdentStr,
                                                       Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                       Declaration => Declaration,
                                                       Scope       => Dictionary.LocalScope (GenericUnit),
                                                       -- to get
                                                       TypeSym     => TypeMark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => TypeMark,
                                               Is_Declaration => True);
                  end if;
               when SPSymbols.formal_array_type_definition =>
                  wf_array_type_definition (Node        => Child_Node (TypeNode),
                                            Scope       => Dictionary.LocalScope (GenericUnit),
                                            Ident_Node  => IdentNode,
                                            DecLoc      => NodePosition (Node),
                                            IsGeneric   => True,
                                             -- to get
                                            ErrorsFound => ErrorInGenericArray,
                                            TheArray    => TypeMark);
                  ErrorInDeclaration := ErrorInDeclaration or ErrorInGenericArray;
               when others =>
                  TypeMark := Dictionary.GetUnknownTypeMark; -- just to avoid data flow error
                  SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                                           "Invalid generic formal type");
            end case;
            Dictionary.AddGenericFormalParameter
              (Name                => LexTokenManager.Null_String, --name not needed for type
               Comp_Unit           => ContextManager.Ops.CurrentUnit,
               Declaration         => Declaration,
               SubprogramOrPackage => GenericUnit,
               TypeMark            => TypeMark,
               Kind                => Dictionary.GenericTypeParameter);
         end if;
      end ProcessFormalType;

      ----------

   begin -- ProcessGenericFormalPart

      -- Node on entry is generic_formal_part

      -- Grammar
      --  generic_formal_part :
      --          generic_formal_part_rep ;

      --  generic_formal_part_rep :
      --         generic_formal_part_rep formal_object_declaration
      --       | generic_formal_part_rep formal_type_declaration
      --       | ;
      RepNode := LastChildOf (Node);
      while Next_Sibling (RepNode) /= STree.NullNode loop
         ObjectOrTypeNode := Next_Sibling (RepNode);
         case SyntaxNodeType (ObjectOrTypeNode) is
            when SPSymbols.formal_object_declaration =>
               ProcessFormalObject (ObjectOrTypeNode,
                                    GenericUnit,
                                    -- using and to get
                                    ErrorInDeclaration);
            when SPSymbols.formal_type_declaration =>
               ProcessFormalType (ObjectOrTypeNode,
                                  GenericUnit,
                                    -- using and to get
                                  ErrorInDeclaration);
            when others =>
               SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                                        "Invalid generic formal parameter");
         end case;
         RepNode := ParentNode (RepNode);
      end loop;
   end ProcessGenericFormalPart;

   -----------------------------------------------------------------------------------

   procedure ProcessGenericSubprogramDeclaration (Node  : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      type SubprogramKinds is (Afunction, Aprocedure, AproofFunction);

      FormalPartNode            : STree.SyntaxNode;
      SubprogramDeclarationNode : STree.SyntaxNode;
      GenericSubprogamSymbol    : Dictionary.Symbol;
      SubprogramKind            : SubprogramKinds;
      ErrorInDeclaration        : Boolean := False;

      procedure CheckAndAddSubprogram (Node           : in     STree.SyntaxNode;
                                       SubprogSym     :    out Dictionary.Symbol;
                                       SubprogramKind :    out SubprogramKinds)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict,
      --#         SubprogSym                from ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         SPARK_IO.FILE_SYS         from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table &
      --#         SubprogramKind            from Node,
      --#                                        STree.Table;
      is
         IdentNode       : STree.SyntaxNode;
         IdentStr        : LexTokenManager.Lex_String;
         SubprogSymLocal : Dictionary.Symbol;
         SubprogramKindLocal : SubprogramKinds;

      begin --CheckAndAddSubprogram

         -- On entry Node is subprogram_declaration

         -- Grammar
         -- subprogram_declaration :
         --    procedure_specification
         --       procedure_annotation
         --  | function_specification
         --       function_annotation
         --  | proof_function_declaration ;

         -- procedure_specification :
         --    identifier
         --  | identifier formal_part ;

         -- function_specification :
         --    designator formal_part type_mark
         --  | designator type_mark ;

         -- proof_function_declaration :
         --    function_specification function_constraint ;

         -- designator :
         --    identifier ;

         -- set default answers
         SubprogramKindLocal := Aprocedure;
         SubprogSymLocal := Dictionary.NullSymbol;

         -- find identifier
         IdentNode := Child_Node (Child_Node (Node));
         if SyntaxNodeType (IdentNode) = SPSymbols.designator then
            IdentNode := Child_Node (IdentNode);
            SubprogramKindLocal := Afunction;
         elsif SyntaxNodeType (IdentNode) = SPSymbols.function_specification then
            IdentNode := Child_Node (Child_Node (IdentNode));
            SubprogramKindLocal := AproofFunction;
         end if;

         -- grammar check
         SystemErrors.RTAssert (SyntaxNodeType (IdentNode) = SPSymbols.identifier,
                                SystemErrors.AssertionFailure,
                                "generic subprogram identifier not in expected place");

         if SubprogramKindLocal = AproofFunction then
            ErrorHandler.SemanticError (999, -- not yet supported
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
         else -- procedure or function
            IdentStr := NodeLexString (IdentNode);
            if Dictionary.IsDefined (IdentStr,
                                     Dictionary.GlobalScope,
                                     Dictionary.ProofContext)
            then
               ErrorHandler.SemanticError (10, --illegal redeclaration
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           IdentStr);
            else
               Dictionary.AddSubprogram (Name          => IdentStr,
                                         Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                         Specification => Dictionary.Location'(NodePosition (Node),
                                                                               NodePosition (Node)),
                                         Scope         => Dictionary.GlobalScope,
                                         Context       => Dictionary.ProgramContext,
                                         -- to get
                                         Subprogram    => SubprogSymLocal);
               Dictionary.SetSubprogramIsGeneric (SubprogSymLocal);
            end if;
         end if;
         SubprogSym := SubprogSymLocal;
         SubprogramKind := SubprogramKindLocal;
      end CheckAndAddSubprogram;

      ---------------

      procedure ProcessFormalPart (Node          : in STree.SyntaxNode;
                                   TheSubprogram : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         STree.Table               from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheSubprogram &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheSubprogram;
      is
         FormalPartNode : STree.SyntaxNode;
      begin -- ProcessFormalPart

         -- On entry Node is subprogram_declaration

         -- Grammar
         -- subprogram_declaration :
         --    procedure_specification
         --       procedure_annotation
         --  | function_specification
         --       function_annotation
         --  | proof_function_declaration ;

         -- procedure_specification :
         --    identifier
         --  | identifier formal_part ;

         -- function_specification :
         --    designator formal_part type_mark
         --  | designator type_mark ;

         FormalPartNode := Next_Sibling (Child_Node (Child_Node (Node)));
         if SyntaxNodeType (FormalPartNode) = SPSymbols.formal_part then
            wf_formal_part (Node => FormalPartNode,
                            CurrentScope => Dictionary.LocalScope (TheSubprogram),
                            SubProgSym => TheSubprogram,
                            FirstOccurrence => True,
                            Context => Dictionary.ProgramContext);
         end if;
      end ProcessFormalPart;

      ---------------

      procedure ProcessReturn (Node               : in     STree.SyntaxNode;
                               TheFunction        : in     Dictionary.Symbol;
                               ErrorInDeclaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --# derives Dictionary.Dict           from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheFunction &
      --#         ErrorInDeclaration,
      --#         STree.Table               from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheFunction &
      --#         ErrorHandler.ErrorContext from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheFunction &
      --#         SPARK_IO.FILE_SYS         from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheFunction;
      is
         ReturnNode : STree.SyntaxNode;
         ReturnType : Dictionary.Symbol;
      begin -- ProcessReturn

         -- Grammar
         -- subprogram_declaration :
         --    function_specification
         --       function_annotation
         --  | proof_function_declaration ;

         -- function_specification :
         --    designator formal_part type_mark
         --  | designator type_mark ;

         ReturnNode := LastSiblingOf (Child_Node (Child_Node (Node)));

         SystemErrors.RTAssert (SyntaxNodeType (ReturnNode) = SPSymbols.type_mark,
                                SystemErrors.AssertionFailure,
                                "generic function return not found");
         wf_type_mark (ReturnNode,
                       Dictionary.LocalScope (TheFunction),
                       Dictionary.ProgramContext,
                        --to get
                       ReturnType);

         -- report errors back to caller
         if ReturnType = Dictionary.GetUnknownTypeMark then
            ErrorInDeclaration := True;
         end if;

         -- GenericTBD check for illegal return types here (e.g. unconstrained arrays)

         -- add return  type to Dictionary
         Dictionary.AddReturnType (TheFunction   => TheFunction,
                                   TypeMark      => ReturnType,
                                   Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                   TypeReference => Dictionary.Location'(NodePosition (ReturnNode),
                                                                         NodePosition (ReturnNode)));
      end ProcessReturn;

      ------------------------------

      procedure GetAnnoAndConNodes (Node           : in     STree.SyntaxNode;
                                    AnnoNode       :    out STree.SyntaxNode;
                                    ConstraintNode :    out STree.SyntaxNode;
                                    GlobalNode     :    out STree.SyntaxNode)
      --# global in STree.Table;
      --# derives AnnoNode,
      --#         ConstraintNode,
      --#         GlobalNode     from Node,
      --#                             STree.Table;
      --  pre   SyntaxNodeType (Node) =
      --           SPSymbols.procedure_annotation or
      --           SPSymbols.function_annotation;
      -- Grammar:
      -- procedure_annotation :
      --   procedure_constraint
      -- | declare_annotation procedure_constraint
      -- | dependency_relation procedure_constraint
      -- | dependency_relation declare_annotation procedure_constraint
      -- | moded_global_definition procedure_constraint
      -- | moded_global_definition declare_annotation procedure_constraint
      -- | moded_global_definition dependency_relation procedure_constraint
      -- | moded_global_definition dependency_relation declare_annotation procedure_constraint ;
      --
      --   function_annotation :
      --      moded_global_definition function_constraint
      -- | function_constraint ;
      is
         NodeToCheck : STree.SyntaxNode;
         GlobalNodeLocal : STree.SyntaxNode := STree.NullNode;
         NodeType : SPSymbols.SPSymbol;
      begin
         NodeToCheck := Child_Node (Node);
         NodeType := SyntaxNodeType (NodeToCheck);
         if NodeType = SPSymbols.function_constraint or else
           NodeType = SPSymbols.procedure_constraint
         then
            AnnoNode := STree.NullNode;      --only a constraint found
            ConstraintNode := NodeToCheck;
         else
            -- soemthing else found other than just a lone constraint
            AnnoNode := Node;
            ConstraintNode := LastSiblingOf (NodeToCheck);
            -- NodeToCheck is now one of declare..., dependency... or moded_global...
            if SyntaxNodeType (NodeToCheck) = SPSymbols.moded_global_definition then
               GlobalNodeLocal := NodeToCheck;
            end if;
         end if;
         GlobalNode := GlobalNodeLocal;
      end GetAnnoAndConNodes;

      ------------------------------------------------------------------------

      procedure ProcessAnno (Node               : in     STree.SyntaxNode;
                             TheSubprogram      : in     Dictionary.Symbol;
                             ErrorInDeclaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in out AggregateStack.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out GlobalComponentData;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives ErrorHandler.ErrorContext,
      --#         SLI.State,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.FileHeap,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GlobalComponentData,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SLI.State,
      --#                                        SPARK_IO.File_Sys,
      --#                                        STree.Table,
      --#                                        TheHeap,
      --#                                        TheSubprogram &
      --#         AggregateStack.State,
      --#         Dictionary.Dict,
      --#         GlobalComponentData,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        GlobalComponentData,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheHeap,
      --#                                        TheSubprogram &
      --#         ErrorInDeclaration        from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheSubprogram;
      is
         AnnoNode       : STree.SyntaxNode;
         GlobalNode     : STree.SyntaxNode;
         ConstraintNode : STree.SyntaxNode;
      begin -- ProcessAnno
         -- Node on entry is procedure_annotation or function_annotation
         SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.procedure_annotation or
                                SyntaxNodeType (Node) = SPSymbols.function_annotation,
                                SystemErrors.AssertionFailure,
                                "wrong node kind passed to ProcessAnno");

         GetAnnoAndConNodes (Node,
                              -- to get
                             AnnoNode,
                             ConstraintNode,
                             GlobalNode);

         -- initial design disallows generic subprograms from having global anno.
         if GlobalNode /= STree.NullNode then
            ErrorHandler.SemanticError (638,
                                        ErrorHandler.NoReference,
                                        NodePosition (Child_Node (Node)),
                                        LexTokenManager.Null_String);
            ErrorInDeclaration := True;

         else -- no global anno
            if Dictionary.IsProcedure (TheSubprogram) then
               if AnnoNode /= STree.NullNode then
                  wf_procedure_annotation (AnnoNode,
                                           Dictionary.LocalScope (TheSubprogram),
                                           TheSubprogram,
                                           True);
               else
                  -- no derives anno
                  if CommandLineData.Content.DoInformationFlow then
                     -- a derives is needed
                     ErrorInDeclaration := True;
                     ErrorHandler.SemanticError (154,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (Node),
                                                 Dictionary.GetSimpleName (TheSubprogram));
                  end if;
               end if;
               if not CommandLineData.Content.DoInformationFlow then
                  CreateFullSubProgDependency (Node,
                                               TheSubprogram,
                                               Dictionary.IsAbstract);
               end if;
            end if;

            if ConstraintNode /= STree.NullNode then
               if Dictionary.IsProcedure (TheSubprogram) then
                  wf_procedure_constraint (ConstraintNode,
                                           Dictionary.LocalScope (TheSubprogram),
                                           True);
               else
                  wf_procedure_constraint (ConstraintNode,
                                           Dictionary.LocalScope (TheSubprogram),
                                           True);
               end if;
            end if;
         end if;
      end ProcessAnno;

   begin -- ProcessGenericSubprogramDeclaration

      -- Grammar
      -- generic_subprogram_declaration :
      --    generic_formal_part subprogram_declaration ;
      --
      -- generic_formal_part :
      --    generic_formal_part_rep ;
      --
      -- generic_formal_part_rep :
      --    generic_formal_part_rep formal_object_declaration
      --  | generic_formal_part_rep formal_type_declaration
      --  | ;
      --

      FormalPartNode := Child_Node (Node);
      SubprogramDeclarationNode := Next_Sibling (FormalPartNode);
      -- First check the generic subprogram name for illegal redeclaration etc.  Add subprogram if OK.
      -- We need to do this before handling the generic formal part because we need somewhere in the
      -- Dictionary where we can attach the list of generic formal parameters.
      CheckAndAddSubprogram (SubprogramDeclarationNode,
                              -- to get
                             GenericSubprogamSymbol,
                             SubprogramKind);

      if GenericSubprogamSymbol /= Dictionary.NullSymbol then
         ProcessGenericFormalPart (FormalPartNode,
                                   GenericSubprogamSymbol,
                                    -- using, and to get
                                   ErrorInDeclaration);

         ProcessFormalPart  (SubprogramDeclarationNode,
                             GenericSubprogamSymbol);
         -- note that the above does not return an erro flag; however, it does set
         -- the subprogram signature as not well formed if there is an error in the
         -- formal part

         if SubprogramKind = Afunction then
            ProcessReturn (SubprogramDeclarationNode,
                           GenericSubprogamSymbol,
                           ErrorInDeclaration);
            ProcessAnno (Next_Sibling (Child_Node (SubprogramDeclarationNode)),
                         GenericSubprogamSymbol,
                         ErrorInDeclaration);
         else
            -- procedure
            ProcessAnno (Next_Sibling (Child_Node (SubprogramDeclarationNode)),
                         GenericSubprogamSymbol,
                         ErrorInDeclaration);
         end if;

         -- PNA to do:
         --    store pre/post nodes

         -- If there are any errors in the generic declaration we mark the subprogram signature as
         -- not well formed.  We can then raise a suitable error when an attempt is made to instantiate
         -- a badly formed generic unit.  Similar to the way we refuse to flow analyse a call to a malformed
         -- subprogram.
         if ErrorInDeclaration then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                            GenericSubprogamSymbol);
         end if;
      end if;
   end ProcessGenericSubprogramDeclaration;

   -----------------------------------------------------------------------------------

   procedure ProcessGenericPackageDeclaration (Node         : in STree.SyntaxNode;
                                               CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      PrivatePackageDeclaration : Boolean;
      ChildPackageDeclaration   : Boolean;
      ErrorInDeclaration        : Boolean; -- := False at start of procedure
      GenericFPNode,
      SpecNode,
      IdentNode,
      ContextNode,
      InheritNode               : STree.SyntaxNode;
      PackSym                   : Dictionary.Symbol;
      PackVisScope              : Dictionary.Scopes;
      IdentStr                  : LexTokenManager.Lex_String;

      procedure FindKeyNodes
      --# global in     Node;
      --#        in     STree.Table;
      --#           out ChildPackageDeclaration;
      --#           out ContextNode;
      --#           out GenericFPNode;
      --#           out IdentNode;
      --#           out IdentStr;
      --#           out InheritNode;
      --#           out PrivatePackageDeclaration;
      --#           out SpecNode;
      --# derives ChildPackageDeclaration,
      --#         ContextNode,
      --#         GenericFPNode,
      --#         IdentNode,
      --#         IdentStr,
      --#         InheritNode,
      --#         PrivatePackageDeclaration,
      --#         SpecNode                  from Node,
      --#                                        STree.Table;
      is
      begin
         -- Grammar

         --       package_specification :
         --       dotted_simple_name package_annotation visible_part private_part
         --       | dotted_simple_name package_annotation visible_part dotted_simple_name ;

         --       private_generic_package_declaration :
         --       inherit_clause generic_formal_part package_specification
         --       | generic_formal_part package_specification ;

         --       generic_package_declaration :
         --       inherit_clause generic_formal_part package_specification
         --       | generic_formal_part package_specification ;

         PrivatePackageDeclaration :=
           (SyntaxNodeType (Node) =
              SPSymbols.private_generic_package_declaration);
         InheritNode := Child_Node (Node);
         if SyntaxNodeType (InheritNode) = SPSymbols.inherit_clause then
            GenericFPNode := Next_Sibling (InheritNode);
         else
            GenericFPNode := InheritNode;
            InheritNode := STree.NullNode;
         end if;

         SpecNode := Next_Sibling (GenericFPNode);

         -- IdentNode is now at package_specification
         IdentNode := Child_Node (Child_Node (SpecNode));
         if SyntaxNodeType (IdentNode) = SPSymbols.identifier then
            ChildPackageDeclaration := False;
         else  -- declaring a child package
            ChildPackageDeclaration := True;
            IdentNode := LastChildOf (IdentNode);
         end if;

         IdentStr := NodeLexString (IdentNode);

--           Debug.PrintNode ("Node = ", Node);
--           Debug.PrintNode ("ParentNode (Node) = ", ParentNode (Node));
--           Debug.PrintNode ("ParentNode (ParentNode (Node)) = ", ParentNode (ParentNode (Node)));
--           Debug.PrintNode ("ParentNode (ParentNode (ParentNode (Node))) = ", ParentNode (ParentNode (ParentNode (Node))));
--           Debug.PrintNode ("Child_Node (ParentNode (ParentNode (ParentNode (Node)))) = ", Child_Node (ParentNode (ParentNode (ParentNode (Node)))));

         ContextNode := Child_Node (ParentNode (ParentNode (ParentNode (Node))));
         if SyntaxNodeType (ContextNode) /= SPSymbols.context_clause then
            ContextNode := STree.NullNode;
         end if;

      end FindKeyNodes;

      procedure CheckAndAddPackage (IdentNode      : in     STree.SyntaxNode;
                                    CurrentScope   : in     Dictionary.Scopes;
                                    PackageSym     :    out Dictionary.Symbol)
      --# global in     ChildPackageDeclaration;
      --#        in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IdentStr;
      --#        in out PrivatePackageDeclaration;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         PackageSym                from ChildPackageDeclaration,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        IdentStr,
      --#                                        LexTokenManager.State,
      --#                                        PrivatePackageDeclaration,
      --#                                        STree.Table &
      --#         STree.Table               from *,
      --#                                        ChildPackageDeclaration,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        PrivatePackageDeclaration &
      --#         IdentStr                  from *,
      --#                                        ChildPackageDeclaration,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext from *,
      --#                                        ChildPackageDeclaration,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        IdentStr,
      --#                                        LexTokenManager.State,
      --#                                        PrivatePackageDeclaration,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         PrivatePackageDeclaration from *,
      --#                                        ChildPackageDeclaration,
      --#                                        CommandLineData.Content &
      --#         SPARK_IO.FILE_SYS         from *,
      --#                                        ChildPackageDeclaration,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IdentNode,
      --#                                        IdentStr,
      --#                                        LexTokenManager.State,
      --#                                        PrivatePackageDeclaration,
      --#                                        STree.Table;
      is
         PackSym   : Dictionary.Symbol;
         ValidName : Boolean := True;
      begin --CheckAndAddPackage

         if  CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then ChildPackageDeclaration then
            AddChild (IdentNode,
                      PrivatePackageDeclaration,
                      CurrentScope,
                        -- to get
                      PackSym,
                      IdentStr);
            -- if PackSym is null then something went wrong when we added the child so we need to supress
            -- any further analysis of the package specification
            ValidName := PackSym /= Dictionary.NullSymbol;
         else
            if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 then
               --check that syntax conforms
               if ChildPackageDeclaration or PrivatePackageDeclaration then
                  ErrorHandler.SemanticError (610,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              LexTokenManager.Null_String);
                  PrivatePackageDeclaration := False;
               end if;
            end if;
            PackSym := Dictionary.LookupItem (Name    => IdentStr,
                                              Scope   => CurrentScope,
                                              Context => Dictionary.ProofContext);
            --# assert True;
            if PackSym /= Dictionary.NullSymbol then
               ValidName := False;
               ErrorHandler.SemanticError (10, --illegal redeclaration
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNode),
                                           IdentStr);
            else
               --# assert True;
               if PrivatePackageDeclaration then -- root level private package
                  Dictionary.AddPrivatePackage (Name          => IdentStr,
                                                Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                                Specification => Dictionary.Location'(NodePosition (IdentNode),
                                                                                      NodePosition (IdentNode)),
                                                Scope         => CurrentScope,
                                                --to get
                                                ThePackage    => PackSym);
               else
                  Dictionary.AddPackage (Name          => IdentStr,
                                         Comp_Unit     => ContextManager.Ops.CurrentUnit,
                                         Specification => Dictionary.Location'(NodePosition (IdentNode),
                                                                               NodePosition (IdentNode)),
                                         Scope         => CurrentScope,
                                         --to get
                                         ThePackage    => PackSym);
               end if;
            end if;
         end if;

         if ValidName then
            PackageSym := PackSym;
         else
            PackageSym := Dictionary.NullSymbol;
         end if;

      end CheckAndAddPackage;

      ---------------

   begin -- ProcessGenericPackageDeclaration
      --# accept Flow, 10, "Required to initialize for flow analysis";
      ErrorInDeclaration := False;
      --# end accept;

      FindKeyNodes;

      if Dictionary.IsGlobalScope (CurrentScope) then
         -- First we need to check the generic package name for illegal redeclaration etc.
         --# accept Flow, 10, PrivatePackageDeclaration, "Code under development";
         CheckAndAddPackage (IdentNode,
                             CurrentScope,
                              -- to get
                             PackSym);
         --# end accept;

         if PackSym /= Dictionary.NullSymbol then

            Debug.PrintSym ("About to call SetPackageIsGeneric ", PackSym);

            Dictionary.SetPackageIsGeneric (PackSym);

            --# accept Flow, 10, ErrorInDeclaration, "Code under development";
            ProcessGenericFormalPart (GenericFPNode,
                                      PackSym,
                                       -- using, and to get
                                      ErrorInDeclaration);
            --# end accept;

            if SyntaxNodeType (InheritNode) = SPSymbols.inherit_clause then

               Dictionary.AddInheritsAnnotation (PackSym,
                                                 Dictionary.Location'(NodePosition (InheritNode),
                                                                        NodePosition (InheritNode)));
               wf_inherit_clause (InheritNode,
                                  PackSym,
                                  CurrentScope);

            end if;

            --# assert True;

            if ContextNode /= STree.NullNode then

               PackVisScope := Dictionary.VisibleScope (PackSym);

               wf_context_clause (ContextNode,
                                  PackSym,
                                  PackVisScope);
            end if;

            --# assert True;

            wf_package_specification (SpecNode,
                                      IdentStr,
                                      PackSym,
                                      CurrentScope);

         end if;

      end if;

   end ProcessGenericPackageDeclaration;

   -----------------------------------------------------------------------------------

begin -- wf_generic_declaration
   -- On entry Node = generic_declaration or private_generic_package_declaration
   --
   -- Grammar
   -- generic_declaration :
   --    generic_subprogram_declaration
   --  | generic_package_declaration ;

   case CommandLineData.Content.LanguageProfile is
      when CommandLineData.SPARK83 =>
         -- not allowed in SPARK 83
         ErrorHandler.SemanticError (637,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.Null_String);

      when CommandLineData.SPARK95 |
        CommandLineData.SPARK2005 =>

         if SyntaxNodeType (Node) = SPSymbols.private_generic_package_declaration then
            CurrentNode := Node;
         else
            CurrentNode := Child_Node (Node);
         end if;

         if SyntaxNodeType (CurrentNode) = SPSymbols.generic_subprogram_declaration then
            ProcessGenericSubprogramDeclaration (CurrentNode);
         elsif SyntaxNodeType (CurrentNode) = SPSymbols.generic_package_declaration then
            ProcessGenericPackageDeclaration (CurrentNode, CurrentScope);
         elsif SyntaxNodeType (CurrentNode) = SPSymbols.private_generic_package_declaration then
            ProcessGenericPackageDeclaration (CurrentNode, CurrentScope);
         else -- can't happen
            SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                                     "Invalid generic unit declaration");
         end if;
   end case;
end wf_generic_declaration;
