-- $Id: sem-compunit-wf_full_type_declaration-wf_integer.adb 16669 2010-04-01 11:26:15Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit.wf_full_type_declaration)

procedure wf_integer (Node       : in STree.SyntaxNode;
                      Scope      : in Dictionary.Scopes;
                      Ident_Node : in STree.SyntaxNode;
                      DecLoc     : in LexTokenManager.Token_Position)
is
   ExpNode             : STree.SyntaxNode;
   LeftExpType,
   RightExpType        : ExpRecord;
   UnwantedSeq         : SeqAlgebra.Seq;
   Lower,
   Upper               : LexTokenManager.Lex_String; -- StoreVals of type's bounds
   UnusedComponentData : ComponentManager.ComponentData;
   Type_Symbol         : Dictionary.Symbol;

   -- Checks that Lower .. Upper are legal wrt System.Min_Int and System.Max_Int
   procedure CheckAgainstRootInteger
   --# global in     CommandLineData.Content;
   --#        in     DecLoc;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Lower;
   --#        in     Upper;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Lower,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Upper;
   is
      SystemSym  : Dictionary.Symbol;
      Min_IntSym : Dictionary.Symbol;
      Min_IntVal : LexTokenManager.Lex_String;
      Max_IntSym : Dictionary.Symbol;
      Max_IntVal : LexTokenManager.Lex_String;
      Result     : Maths.Value;
      Unused     : Maths.ErrorCode;
      RangeOK    : Boolean;
   begin
      -- We only check in 95 or 2005 modes, since System may not be
      -- specified in the target configuration file in SPARK83 mode.
      case CommandLineData.Content.LanguageProfile is
         when CommandLineData.SPARK83 =>
            null;
         when CommandLineData.SPARK95 |
           CommandLineData.SPARK2005 =>

            SystemSym := Dictionary.LookupItem (Name    => LexTokenManager.System_Token,
                                                Scope   => Dictionary.GlobalScope,
                                                Context => Dictionary.ProgramContext);

            -- The user may or may not have bothered to supply
            -- package System, so...
            if SystemSym /= Dictionary.NullSymbol then

               -- Find System.Min_Int and check Lower against it.
               Min_IntSym := Dictionary.LookupSelectedItem (Prefix   => SystemSym,
                                                            Selector => LexTokenManager.Min_Int_Token,
                                                            Scope    => Dictionary.GetScope (SystemSym),
                                                            Context  => Dictionary.ProgramContext);

               -- Even if the user has supplied a package System, they might
               -- not have declared Min_Int, so again we have to guard...
               if Min_IntSym /= Dictionary.NullSymbol then

                  Min_IntVal := Dictionary.GetValue (Min_IntSym);

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Lower,
                                                                          Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Min_IntVal,
                                                                         Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.GreaterOrEqual (Maths.ValueRep (Lower),
                                           Maths.ValueRep (Min_IntVal),
                                           Result,
                                           Unused);
                     --# end accept;

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.ValueToBool (Result, RangeOK, Unused);
                     --# end accept;

                     if not RangeOK then
                        ErrorHandler.SemanticError (781,
                                                    ErrorHandler.NoReference,
                                                    DecLoc,
                                                    LexTokenManager.Null_String);
                     end if;
                  end if;

               end if;

               --# assert True; -- for RTC generation

               -- Find System.Max_Int and check Upper against it.
               Max_IntSym := Dictionary.LookupSelectedItem (Prefix   => SystemSym,
                                                            Selector => LexTokenManager.Max_Int_Token,
                                                            Scope    => Dictionary.GetScope (SystemSym),
                                                            Context  => Dictionary.ProgramContext);

               -- Even if the user has supplied a package System, they might
               -- not have declared Max_Int, so again we have to guard...
               if Max_IntSym /= Dictionary.NullSymbol then

                  Max_IntVal := Dictionary.GetValue (Max_IntSym);

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Lower,
                                                                          Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Max_IntVal,
                                                                         Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.LesserOrEqual (Maths.ValueRep (Upper),
                                          Maths.ValueRep (Max_IntVal),
                                          Result,
                                          Unused);
                     --# end accept;

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.ValueToBool (Result, RangeOK, Unused);
                     --# end accept;

                     if not RangeOK then
                        ErrorHandler.SemanticError (782,
                                                    ErrorHandler.NoReference,
                                                    DecLoc,
                                                    LexTokenManager.Null_String);
                     end if;
                  end if;

               end if;

            end if;
      end case;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
   end CheckAgainstRootInteger;

begin
   --assume Node is integer_type_definition
   ExpNode := Child_Node (Child_Node (Child_Node (Node)));
   SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
   ComponentManager.Initialise (UnusedComponentData);
   --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
   WalkExpression (ExpNode     => ExpNode,
                   Scope       => Scope,
                     -- The context given in the LRM (3.5.4(5)) is "any integer type", but we
                     -- never resolve by context here, so UnknownTypeMark will do.
                   TypeContext           => Dictionary.GetUnknownTypeMark,
                   ContextRequiresStatic => True,
                     -- to get
                   Result  => LeftExpType,
                   RefVar  => UnwantedSeq,
                   ComponentData => UnusedComponentData);
   --# end accept;
   SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
   Maths.StorageRep (LeftExpType.Value, Lower);
   if SyntaxNodeType (ExpNode) = SPSymbols.attribute then
      if LeftExpType.IsARange then
         Maths.StorageRep (LeftExpType.RangeRHS, Upper);
         ErrorHandler.SemanticError (45,
                                     1,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      else
         Lower := LexTokenManager.Null_String; --no value in error case
         Upper := LexTokenManager.Null_String; --no value in error case
         ErrorHandler.SemanticError (98, ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      end if;
   else --not an attribute
      if not (Dictionary.TypeIsInteger (LeftExpType.TypeSymbol) or
              Dictionary.TypeIsModular (LeftExpType.TypeSymbol) or
              Dictionary.IsUnknownTypeMark (LeftExpType.TypeSymbol))
      then
         Lower := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.SemanticError (38,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      elsif LeftExpType.IsARange then
         Lower := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.SemanticError (114,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      end if;

      ExpNode := Next_Sibling (ExpNode);
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression (ExpNode     => ExpNode,
                      Scope       => Scope,
                        -- The context given in the LRM (3.5.4(5)) is "any integer type", but we
                        -- never resolve by context here, so UnknownTypeMark will do.
                      TypeContext           => Dictionary.GetUnknownTypeMark,
                      ContextRequiresStatic => True,
                        -- to get
                      Result  => RightExpType,
                      RefVar  => UnwantedSeq,
                      ComponentData => UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Maths.StorageRep (RightExpType.Value, Upper);

      if not (Dictionary.TypeIsInteger (RightExpType.TypeSymbol) or
              Dictionary.TypeIsModular (RightExpType.TypeSymbol) or
              Dictionary.IsUnknownTypeMark (RightExpType.TypeSymbol))
      then
         Upper := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.SemanticError (38, ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      elsif RightExpType.IsARange then
         Lower := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.SemanticError (114,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      end if;

      if not (LeftExpType.IsStatic and RightExpType.IsStatic) then
         ErrorHandler.SemanticError (45,
                                     1,
                                     NodePosition (ExpNode),
                                     LexTokenManager.Null_String);
      end if;
   end if;

   EmptyTypeCheck (DecLoc, Lower, Upper);
   CheckAgainstRootInteger;
   Dictionary.AddIntegerType (Name        => NodeLexString (Ident_Node),
                              Comp_Unit   => ContextManager.Ops.CurrentUnit,
                              Declaration => Dictionary.Location'(DecLoc, DecLoc),
                              Lower       => Lower,
                              Upper       => Upper,
                              Scope       => Scope,
                              Context     => Dictionary.ProgramContext,
                              Type_Symbol => Type_Symbol);
   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                Parse_Tree     => Ident_Node,
                                Symbol         => Type_Symbol,
                                Is_Declaration => True);
   end if;
   Heap.ReportUsage (TheHeap);
end wf_integer;
