-- $Id: sem-compunit-wf_factor.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of a
-- factor node.  Replaces calls to StaticTerm, BaseTypeTerm and CheckTypeTerm
----------------------------------------------------------------------------
with SPrint;
separate (Sem.CompUnit)
procedure wf_factor (Node         : in     STree.SyntaxNode;
                     Scope        : in     Dictionary.Scopes;
                     EStack       : in out ExpStack.ExpStackType;
                     TStack       : in     TypeContextStack.TStackType;
                     IsAnnotation : in     Boolean)
is
   OpNode    : STree.SyntaxNode;
   LeftNode  : STree.SyntaxNode;
   RightNode : STree.SyntaxNode;

   ---------------------------------------------------------------

   procedure DoStarStar (OpNode    : in     STree.SyntaxNode;
                         LeftNode  : in     STree.SyntaxNode;
                         RightNode : in     STree.SyntaxNode;
                         Scope     : in     Dictionary.Scopes;
                         EStack    : in out ExpStack.ExpStackType;
                         TStack    : in     TypeContextStack.TStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LeftNode,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        OpNode,
   --#                                        RightNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        OpNode,
   --#                                        Scope,
   --#                                        TStack;
   is
      Left,
      Right,
      Result : ExpRecord;
   begin
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left, EStack);
      Result := NullTypeRecord; --safety: we may not set all fields below

      -- do static checks first
      Result.IsConstant   := Left.IsConstant and Right.IsConstant;
      Result.IsStatic     := Left.IsStatic and Right.IsStatic;
      Result.HasOperators := True;
      if Left.IsARange or Right.IsARange then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (90,
                                     ErrorHandler.NoReference,
                                     NodePosition (OpNode),
                                     LexTokenManager.Null_String);
      else -- neither are ranges
         -- Now do type compat and operator visibility checks
         -- First, implicit type conversion

         -- Implicitly convert the Left operand if its type is Universal,
         -- and the context is not unknown.
         if (Dictionary.IsUniversalIntegerType (Left.TypeSymbol) or
             Dictionary.IsUniversalRealType (Left.TypeSymbol)) then

            if TypeContextStack.Top (TStack) /= Dictionary.GetUnknownTypeMark then
               Left.TypeSymbol := TypeContextStack.Top (TStack);
            end if;
         end if;

         -- The right operand of ** is always predefined Integer, so this
         -- does not depend upon the context stack.
         if Dictionary.IsUniversalIntegerType (Right.TypeSymbol) then
            Right.TypeSymbol := Dictionary.GetPredefinedIntegerType;
         end if;

         -- add type of LHS to syntax tree for use by VCG in run-time checks
         STree.AddNodeSymbol (Child_Node (Node),
                                          Left.TypeSymbol);
         --then, operator visibility
         CheckBinaryOperator (Operator => SPSymbols.double_star,
                              Left     => Left,
                              Right    => Right,
                              Scope    => Scope,
                              TStack   => TStack,
                              OpPos    => NodePosition (OpNode),
                              LeftPos  => NodePosition (LeftNode),
                              RightPos => NodePosition (RightNode),
                              -- no implicit type conversion, since
                              -- already done above for this special case.
                              Convert      => False,

                              IsAnnotation => IsAnnotation,
                              -- using and to get
                              Result       => Result);

         STree.AddNodeSymbol (OpNode,
                                          Result.TypeSymbol);
         CalcBinaryOperator (Node     => Node,
                             Operator => SPSymbols.double_star,
                             LeftVal  => Left.Value,
                             RightVal => Right.Value,
                             IsAnnotation => IsAnnotation,
                              --using and to get
                             Result   => Result);
      end if;

      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is sued in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.OtherSymbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, EStack);

   end DoStarStar;

   ---------------------------------------------------------------

   procedure DoAbsOrNot (Node   : in     STree.SyntaxNode;
                         Scope  : in     Dictionary.Scopes;
                         EStack : in out ExpStack.ExpStackType;
                         TStack : in     TypeContextStack.TStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TStack &
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        Node,
   --#                                        TStack;
   is
      Result    : ExpRecord;
      BaseType  : Dictionary.Symbol;
      OpNode    : STree.SyntaxNode;
      Operator  : SPSymbols.SPSymbol;
      Val       : Maths.Value;
      Error     : Boolean;

      procedure Resolve_Universal (T : in out Dictionary.Symbol; Error : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     Operator;
      --#        in     STree.Table;
      --#        in     TStack;
      --#        in     Val;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Error                     from Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Operator,
      --#                                        T,
      --#                                        TStack,
      --#                                        Val &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        Operator,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        T,
      --#                                        TStack,
      --#                                        Val &
      --#         T                         from *,
      --#                                        Dictionary.Dict,
      --#                                        TStack;
      is
         BaseType  : Dictionary.Symbol;
         ValCheck  : Maths.Value;

         procedure DebugPrint
         --# derives ;
         is
            --# hide DebugPrint;
         begin
            if CommandLineData.Content.Debug.Expressions then
               SPARK_IO.Put_String
                 (SPARK_IO.Standard_Output,
                  "Wf_Factor encounters a universal expression.  Resolving by context to type ",
                  0);
               EStrings.Put_Line
                 (File  => SPARK_IO.Standard_Output,
                  E_Str => LexTokenManager.Lex_String_To_String
                    (Lex_Str => Dictionary.GetSimpleName (TypeContextStack.Top (TStack))));
            end if;
         end DebugPrint;

      begin
         Error := False;
         if Dictionary.IsUniversalRealType (T) then -- We want to convert but not check
            T := TypeContextStack.Top (TStack);
            DebugPrint;
         elsif Dictionary.IsUniversalIntegerType (T) then
            T := TypeContextStack.Top (TStack);
            DebugPrint;
            ValCheck := Val;
            if Operator = SPSymbols.RWabs then -- Check against T'Base
               BaseType := Dictionary.GetBaseType (T);
               if BaseType /= Dictionary.NullSymbol then
                  ConstraintCheck (Val,
                                   ValCheck,
                                   False,
                                   BaseType,
                                   NodePosition (LastChildOf (LastSiblingOf (LastChildOf (Node)))));
               end if;
            else -- Operator = SPSymbols.RWnot then -- Check against T
               ConstraintCheck (Val,
                                ValCheck,
                                False,
                                T,
                                NodePosition (LastChildOf (LastSiblingOf (LastChildOf (Node)))));
            end if;
            if Maths.HasNoValue (ValCheck) then
               Error := True;
            end if;
         else
            null;
         end if;
      end Resolve_Universal;

      procedure CalcAbsOrNot (Val : in out Maths.Value)
      --# global in BaseType;
      --#        in Dictionary.Dict;
      --#        in LexTokenManager.State;
      --#        in Operator;
      --# derives Val from *,
      --#                  BaseType,
      --#                  Dictionary.Dict,
      --#                  LexTokenManager.State,
      --#                  Operator;
      is
      begin
         case Operator is
            when SPSymbols.RWabs =>
               Maths.Absolute (Val);

            when SPSymbols.RWnot =>
               if Dictionary.TypeIsModular (BaseType) then
                  Maths.ModularNotOp
                     (Val,
                      Maths.ValueRep
                         (Dictionary.GetScalarAttributeValue (False,
                                                              LexTokenManager.Modulus_Token,
                                                              BaseType)));
               else
                  Maths.NotOp (Val);
               end if;
            when others =>
               null;
         end case;
      end CalcAbsOrNot;

      --------------------------------------------------------

   begin --DoAbsOrNot
      ExpStack.Pop (Result, EStack);    --same result type and flags

      OpNode   := Child_Node (Node);
      Operator := SyntaxNodeType (OpNode);
      Val := Result.Value;

      Resolve_Universal (Result.TypeSymbol, Error);

      BaseType := Dictionary.GetRootType (Result.TypeSymbol);
      STree.AddNodeSymbol (OpNode, Result.TypeSymbol);

      if Error then
         Result := UnknownTypeRecord;
      end if;

      if not Dictionary.UnaryOperatorIsDefined (Operator,
                                                BaseType)
      then
         Result := UnknownTypeRecord;
         if Operator = SPSymbols.RWabs then
            if Dictionary.IsModularType (BaseType, Scope) then
               ErrorHandler.SemanticError (803,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           LexTokenManager.Null_String);
            else
               ErrorHandler.SemanticError (40,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (OpNode)),
                                           LexTokenManager.Null_String);
            end if;
         else
            ErrorHandler.SemanticErrorSym (119,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           BaseType,
                                           Scope);
         end if;
      elsif (not IsAnnotation) and then
         not Dictionary.UnaryOperatorIsVisible (Operator,
                                                BaseType,
                                                Scope)
      then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (309,
                                     ErrorHandler.NoReference,
                                     NodePosition (OpNode),
                                     LexTokenManager.Null_String);

      else
         -- check for misuse of unconstrained boolean array
         if Dictionary.IsUnconstrainedArrayType (Result.TypeSymbol) then
            Result.ErrorsInExpression := True;
            ErrorHandler.SemanticError (39,
                                        ErrorHandler.NoReference,
                                        NodePosition (Next_Sibling (OpNode)),
                                        LexTokenManager.Null_String);
         end if;

         Val := Result.Value;
         CalcAbsOrNot (Val);
         Result.Value := Val;
         if Dictionary.TypeIsScalar (Result.TypeSymbol) then
            Result.TypeSymbol := BaseType;
         end if;
         Result.VariableSymbol := Dictionary.NullSymbol;
         Result.IsAVariable := False;
         Result.IsAnEntireVariable := False;
         Result.HasOperators := True;
      end if;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is used in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.OtherSymbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, EStack);

   end DoAbsOrNot;

   ---------------------------------------------------------------

begin --wf_factor
   OpNode := Next_Sibling (Child_Node (Node));
   if OpNode /= STree.NullNode then   -- ** or abs or not
      if SyntaxNodeType (OpNode) = SPSymbols.double_star then

         LeftNode  := Child_Node (Node);
         RightNode := Next_Sibling (OpNode);

         DoStarStar (OpNode,
                     LeftNode,
                     RightNode,
                     Scope,
                     EStack,
                     TStack);
      else
         DoAbsOrNot (Node,
                     Scope,
                     EStack,
                     TStack);
      end if;
   end if;
end wf_factor;
