-- $Id: sem-compunit-wf_external_interface.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

--Synopsis
--This procedure checks the validity of a pragma interface (Ada83) or pragma
--import (Ada95).  The checks made are:
-- 1.  Internal consistency of associations used, number of parameters etc.
-- 2.  The Entity/Subprogram name is that expected
--------------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure wf_external_interface (Pragma_Node : in     STree.SyntaxNode;
                                 Entity_Sym  : in     Dictionary.Symbol;
                                 Error_Found :    out Boolean)
is

   procedure CheckRepresentSameName (Exp_Node   : in STree.SyntaxNode;
                                     Entity_Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Error_Found;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Error_Found,
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        Exp_Node,
   --#                                        LexTokenManager.State,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Exp_Node,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      IsChain  : Boolean;
      IdNode,
      NextNode : STree.SyntaxNode;
      Name     : LexTokenManager.Lex_String;
   begin
      Name := Dictionary.GetSimpleName (Item => Entity_Sym);
      IdNode := Exp_Node;
      loop
         IsChain := Next_Sibling (IdNode) =
           STree.NullNode;
         NextNode := Child_Node (IdNode);
         exit when not IsChain or NextNode = STree.NullNode;

         IdNode := NextNode;
      end loop;

      if IsChain and then
        SyntaxNodeType (IdNode) = SPSymbols.identifier and then
        LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (IdNode),
                                                             Lex_Str2 => Name) = LexTokenManager.Str_Eq then
         STree.Set_Node_Lex_String (Sym  => Entity_Sym,
                                    Node => IdNode);
      else
         Error_Found := True;
         ErrorHandler.SemanticError (71,
                                     ErrorHandler.NoReference,
                                     NodePosition (Exp_Node),
                                     Name);
      end if;
   end CheckRepresentSameName;

   ------------------------------------------------------------------

   procedure wf_pragma_interface (Pragma_Node : in STree.SyntaxNode;
                                  Entity_Sym  : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Error_Found;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Error_Found,
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        LexTokenManager.State,
   --#                                        Pragma_Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Pragma_Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ArgAssocRepNode : STree.SyntaxNode;
      SubprogNameNode : STree.SyntaxNode;
   begin
      ArgAssocRepNode := Child_Node (Next_Sibling (Child_Node (Pragma_Node)));
      if SyntaxNodeType (Child_Node (ArgAssocRepNode)) =
         SPSymbols.argument_association
      then --pragma has two arguments
         SubprogNameNode := Child_Node (Next_Sibling (ArgAssocRepNode));

         if SyntaxNodeType (SubprogNameNode) /=
            SPSymbols.ADA_expression
         then --form of expression wrong
            Error_Found := True;
            ErrorHandler.SemanticError (71,
                                        ErrorHandler.NoReference,
                                        NodePosition (SubprogNameNode),
                                        Dictionary.GetSimpleName (Item => Entity_Sym));
         else --form of expression ok so check name actually matches
            CheckRepresentSameName (Exp_Node   => SubprogNameNode,
                                    Entity_Sym => Entity_Sym);
         end if;

      else --pragma does nor have exatcly 2 arguments
         Error_Found := True;
         ErrorHandler.SemanticError (69,
                                     ErrorHandler.NoReference,
                                     NodePosition (Pragma_Node),
                                     LexTokenManager.Interface_Token);
      end if;
   end wf_pragma_interface;

   ----------------------

   procedure wf_pragma_import (Pragma_Node : in STree.SyntaxNode;
                               Entity_Sym  : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Error_Found;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Error_Found,
   --#         STree.Table               from *,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        LexTokenManager.State,
   --#                                        Pragma_Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Entity_Sym,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Pragma_Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      type Args is (Illegal, Convention, Entity, ExternalName, LinkName);
      subtype LegalArgs is Args range Convention .. LinkName;
      type Founds is array (LegalArgs) of Boolean;

      Found : Founds := Founds'(LegalArgs => False);
      UsingNamedAssociation : Boolean := False;
      ArgAssNode : STree.SyntaxNode;
      ArgCount : Natural := 0;
      MaxArgs  : constant Natural := 4;

      procedure CheckArgument (Node       : in STree.SyntaxNode;
                               Entity_Sym : in Dictionary.Symbol)
      --# global in     ArgCount;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Error_Found;
      --#        in out Found;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --#        in out UsingNamedAssociation;
      --# derives Error_Found,
      --#         STree.Table               from *,
      --#                                        ArgCount,
      --#                                        Dictionary.Dict,
      --#                                        Entity_Sym,
      --#                                        Found,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        UsingNamedAssociation &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ArgCount,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        Entity_Sym,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        Found,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        UsingNamedAssociation &
      --#         Found                     from *,
      --#                                        ArgCount,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        UsingNamedAssociation &
      --#         UsingNamedAssociation     from *,
      --#                                        Node,
      --#                                        STree.Table;
      is
         ExpNode : STree.SyntaxNode;
         Arg     : Args;

         function GetArg (ArgString : LexTokenManager.Lex_String) return Args
         --# global in LexTokenManager.State;
         is
            Result : Args;
         begin
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ArgString,
                                                                    Lex_Str2 => LexTokenManager.Convention_Token) = LexTokenManager.Str_Eq then
               Result := Convention;
            elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ArgString,
                                                                       Lex_Str2 => LexTokenManager.Entity_Token) = LexTokenManager.Str_Eq then
               Result := Entity;
            elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ArgString,
                                                                       Lex_Str2 => LexTokenManager.External_Name_Token) = LexTokenManager.Str_Eq then
               Result := ExternalName;
            elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ArgString,
                                                                       Lex_Str2 => LexTokenManager.Link_Name_Token) = LexTokenManager.Str_Eq then
               Result := LinkName;
            else
               Result := Illegal;
            end if;
            return Result;
         end GetArg;

         ---------------

      begin --CheckArgument
         ExpNode := Child_Node (Node);
         --must be an identifier or an ADA_expression
         if SyntaxNodeType (ExpNode) = SPSymbols.identifier then
            --named association
            UsingNamedAssociation := True;
            Arg := GetArg (NodeLexString (ExpNode));
            if Arg = Illegal then
               Error_Found := True;
               ErrorHandler.SemanticError (601,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpNode),
                                           LexTokenManager.Null_String);
            elsif Found (Arg) then
               Error_Found := True;
               ErrorHandler.SemanticError (602,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpNode),
                                           NodeLexString (ExpNode));
            else
               Found (Arg) := True;
               if Arg = Entity then
                  ExpNode := Next_Sibling (ExpNode);
                  CheckRepresentSameName (Exp_Node   => ExpNode,
                                          Entity_Sym => Entity_Sym);
               end if;
            end if;

         else --positional association
            if UsingNamedAssociation then --illegal switch form named to positional assoc
               Error_Found := True;
               ErrorHandler.SemanticError (601,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpNode),
                                           LexTokenManager.Null_String);
            else
               Arg := Args'Val (ArgCount);
               Found (Arg) := True;
               if Arg = Entity then
                  CheckRepresentSameName (Exp_Node   => ExpNode,
                                          Entity_Sym => Entity_Sym);
               end if;
            end if;
         end if;
      end CheckArgument;

      ----------------------

   begin --wf_pragma_import
      ArgAssNode := Next_Sibling (Child_Node (Pragma_Node));
      --should be top of argument_asociation_rep chain
      if SyntaxNodeType (ArgAssNode) =
         SPSymbols.argument_association_rep
      then
         while SyntaxNodeType (ArgAssNode) /=
            SPSymbols.argument_association
         loop
            ArgAssNode := Child_Node (ArgAssNode);
         end loop;
         --now pointing at leftmost argument association
         while ArgAssNode /= STree.NullNode
         loop
            if ArgCount = MaxArgs then
               Error_Found := True;
               ErrorHandler.SemanticError (600,
                                           ErrorHandler.NoReference,
                                           NodePosition (ArgAssNode),
                                           LexTokenManager.Null_String);
               exit;
            end if;
            ArgCount := ArgCount + 1;
            CheckArgument (Node       => ArgAssNode,
                           Entity_Sym => Entity_Sym);
            ArgAssNode := Next_Sibling (ParentNode (ArgAssNode));
         end loop;
         if ArgCount < 2 then
            Error_Found := True;
            ErrorHandler.SemanticError (600,
                                        ErrorHandler.NoReference,
                                        NodePosition (Pragma_Node),
                                        LexTokenManager.Convention_Token);

         else
            if not Found (Convention) then
               Error_Found := True;
               ErrorHandler.SemanticError (603,
                                           ErrorHandler.NoReference,
                                           NodePosition (Pragma_Node),
                                           LexTokenManager.Convention_Token);
            end if;
            if not Found (Entity) then
               Error_Found := True;
               ErrorHandler.SemanticError (603,
                                           ErrorHandler.NoReference,
                                           NodePosition (Pragma_Node),
                                           LexTokenManager.Entity_Token);
            end if;
         end if;
      else --there are no arguments
         Error_Found := True;
         ErrorHandler.SemanticError (600,
                                     ErrorHandler.NoReference,
                                     NodePosition (Pragma_Node),
                                     LexTokenManager.Null_String);
      end if;
   end wf_pragma_import;

   ----------------------

begin -- wf_external_interface
   Error_Found := False;
   case CommandLineData.Content.LanguageProfile is
      when CommandLineData.SPARK83 =>

         wf_pragma_interface (Pragma_Node => Pragma_Node,
                              Entity_Sym  => Entity_Sym);

      when CommandLineData.SPARK95 |
        CommandLineData.SPARK2005 =>

         wf_pragma_import (Pragma_Node => Pragma_Node,
                           Entity_Sym  => Entity_Sym);

   end case;
end wf_external_interface;
