-- $Id: sem-compunit-wf_entry_body.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (Sem.CompUnit)
procedure wf_entry_body (Node     : in     STree.SyntaxNode;
                         Scope    : in out Dictionary.Scopes;
                         NextNode :    out STree.SyntaxNode)
is
   --Grammar:
   --       entry_body
   --            |
   --       entry spec --- identifier --- procedure_annotation --- subprogram_implementation
   --            |         (^the guard Boolean)
   --            |
   --       identifier --- formal_part
   --                          |
   --                      formal_part_rep
   --
   -- Actions:
   -- (1) First identifier must be name of (sole) entry declared in spec
   -- (2) Second identifier must be Boolean and must be protected element
   -- (3) If valid, add body, set up a local scope
   -- (4) wff annotation; note FirstSeen is False by definition; however, second anno may not be needed
   -- (5) Allow main tree walk to continue in new scope
   -- (6) Check end designator matches if not hidden

   EntrySym,
   GuardSym   : Dictionary.Symbol;
   EntrySpecNode,
   FormalPartNode,
   IdentNode,
   GuardNode,
   AnnoNode,
   SubprogramImplementationNode,
   PragmaRepNode,
   EndNode         : STree.SyntaxNode;
   Hidden          : Hidden_Class;

   -- check whether a second anno is needed, if it is present, and process it if necessary
   procedure CheckAnnotation (AnnoNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     EntrySym;
   --#        in     Node;
   --#        in     Scope;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EntrySym,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EntrySym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EntrySym,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         STree.Table               from *,
   --#                                        AnnoNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        EntrySym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      ConstraintNode : STree.SyntaxNode;

      -- A second annotation is only needed if the abstract global anno contains
      -- the implicitly-declared "own variable" that shares the name of the type.
      function RequiresSecondAnnotation return Boolean
      --# global in Dictionary.Dict;
      --#        in EntrySym;
      is
         Result : Boolean := False;
         OwnVar : Dictionary.Symbol;
         It     : Dictionary.Iterator;
      begin
         OwnVar := Dictionary.GetProtectedTypeOwnVariable
           (Dictionary.GetRegion (Dictionary.GetScope (EntrySym)));
         It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                               EntrySym);
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.CurrentSymbol (It) = OwnVar;
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
         return Result;
      end RequiresSecondAnnotation;

      function HasSecondAnnotation return Boolean
      --# global in AnnoNode;
      --#        in STree.Table;
      is
      begin
         return SyntaxNodeType (Child_Node (AnnoNode)) = SPSymbols.moded_global_definition or else
           SyntaxNodeType (Child_Node (AnnoNode)) = SPSymbols.dependency_relation;
      end HasSecondAnnotation;

   begin -- CheckAnnotation
      if HasSecondAnnotation then
         if RequiresSecondAnnotation then
            -- wanted and present so process it
            wf_procedure_annotation (Node         => AnnoNode,
                                     CurrentScope => Scope,
                                     SubprogSym   => EntrySym,
                                     FirstSeen    => False);

            -- check for and handle second, concrete constraint
            ConstraintNode := LastSiblingOf (Child_Node (AnnoNode));
            if Child_Node (ConstraintNode) /= STree.NullNode then
               -- a constraint is present, so process it
               wf_procedure_constraint (Node      => ConstraintNode,
                                        Scope     => Dictionary.LocalScope (EntrySym),
                                        FirstSeen => False);
            end if;

            -- If not performing full information flow analysis then
            -- synthesize the "full" dependency clause using moded globals
            if not CommandLineData.Content.DoInformationFlow then
               CreateFullSubProgDependency (Node,
                                            EntrySym,
                                            Dictionary.IsRefined);
            end if;

         else -- anno found but not needed
               ErrorHandler.SemanticError (155,
                                           ErrorHandler.NoReference,
                                           NodePosition (AnnoNode),
                                           Dictionary.GetSimpleName (EntrySym));
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                               EntrySym);
         end if;
      else -- no anno
         if RequiresSecondAnnotation then
            -- anno missing
            ErrorHandler.SemanticError (87,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (EntrySym));
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsRefined,
                                                            EntrySym);
         end if;
      end if;
   end CheckAnnotation;

begin -- wf_entry_body
   EntrySpecNode := Child_Node (Node);
   IdentNode := Child_Node (EntrySpecNode);
   FormalPartNode := Next_Sibling (IdentNode);
   GuardNode := Next_Sibling (EntrySpecNode);
   AnnoNode := Next_Sibling (GuardNode);
   SubprogramImplementationNode := Next_Sibling (AnnoNode);
   PragmaRepNode := Child_Node (SubprogramImplementationNode);
   EndNode   := LastSiblingOf (PragmaRepNode);

   Hidden := Body_Hidden_Class (SubprogramImplementationNode);

   EntrySym := Dictionary.LookupItem (Name    => NodeLexString (IdentNode),
                                      Scope   => Scope,
                                      Context => Dictionary.ProgramContext);
   -- Check that EntrySym is an entry declared in the spec.  Since we are looking up an identifier
   -- not a full, dotted name we can't find any other entry by mistake so a simple check is all that
   -- is needed.
   if Dictionary.IsEntry (EntrySym) then

      if FormalPartNode /= STree.NullNode then
         STree.Set_Node_Lex_String (Sym  => EntrySym,
                                    Node => IdentNode);
         wf_formal_part (Node => FormalPartNode,
                         CurrentScope => Scope,
                         SubProgSym => EntrySym,
                         FirstOccurrence => False,
                         Context => Dictionary.ProgramContext);
      elsif Dictionary.GetNumberOfSubprogramParameters (EntrySym) /= 0 then
         ErrorHandler.SemanticError (152,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (EntrySym));
      else
         STree.Set_Node_Lex_String (Sym  => EntrySym,
                                    Node => IdentNode);
      end if;
      -- ok so far
      -- now check that the Guard is valid
      GuardSym := Dictionary.LookupItem (Name    => NodeLexString (GuardNode),
                                         Scope   => Scope,
                                         Context => Dictionary.ProgramContext);
      if Dictionary.IsVariable (GuardSym) and then
        Dictionary.IsRefinement (Dictionary.GetProtectedTypeOwnVariable
                                   (Dictionary.GetRegion (Scope)),
                                 GuardSym) and then
        Dictionary.IsBooleanTypeMark (Dictionary.GetType (GuardSym)) then
         -- Guard is a protected element of type Boolean, which is OK

         -- store it for use in VCG
         Dictionary.SetSubprogramEntryBarrier (EntrySym, GuardSym);
         STree.Set_Node_Lex_String (Sym  => GuardSym,
                                    Node => GuardNode);
         -- The entry is valid so far, it may be hidden or it may have a real sequence of statements
         if Hidden = All_Hidden then
            Dictionary.AddBody (CompilationUnit => EntrySym,
                                Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                TheBody         => Dictionary.Location'(NodePosition (Node),
                                                                        NodePosition (Node)),
                                Hidden          => True);
            ErrorHandler.HiddenText (NodePosition (EndNode),
                                     NodeLexString (IdentNode),
                                     SPSymbols.subprogram_implementation);
            NextNode := STree.NullNode; -- prune tree walk on hidden part
         else
            Dictionary.AddBody (CompilationUnit => EntrySym,
                                Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                TheBody         => Dictionary.Location'(NodePosition (Node),
                                                                        NodePosition (Node)),
                                Hidden          => False);

            -- check annotation
            CheckAnnotation (AnnoNode);

            if Hidden = Handler_Hidden then
               ErrorHandler.HiddenHandler (NodePosition (EndNode),
                                           NodeLexString (IdentNode),
                                           SPSymbols.entry_body);
            end if;

            -- set up scope for rest of tree walk
            Scope := Dictionary.LocalScope (EntrySym);

            --set up next node for rest of tree walk
            NextNode := SubprogramImplementationNode;
         end if;
      else
         -- Guard is not a protected element or is not Boolean
         ErrorHandler.SemanticError (994,
                                     ErrorHandler.NoReference,
                                     NodePosition (GuardNode),
                                     NodeLexString (GuardNode));
         NextNode := STree.NullNode; -- prune tree walk on error

      end if;
   else
      -- not a valid Entry
      ErrorHandler.SemanticError (995,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  NodeLexString (IdentNode));
      NextNode := STree.NullNode; -- prune tree walk on error
   end if;

   -- check closing identifier
   if SyntaxNodeType (EndNode) = SPSymbols.designator then
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (IdentNode),
                                                              Lex_Str2 => NodeLexString (Child_Node (EndNode))) /= LexTokenManager.Str_Eq then
         ErrorHandler.SemanticError (58,
                                     ErrorHandler.NoReference,
                                     NodePosition (EndNode),
                                     NodeLexString (IdentNode));
      end if;
   end if;
end wf_entry_body;
