-- $Id: sem-compunit-wf_dependency_relation.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit)

procedure wf_dependency_relation (Node         : in STree.SyntaxNode;
                                  CurrentScope : in Dictionary.Scopes;
                                  SubprogSym   : in Dictionary.Symbol;
                                  FirstSeen    : in Boolean;
                                  GlobDefErr   : in Boolean)
is
   It                 : STree.Iterator;
   NextNode           : STree.SyntaxNode;
   Position           : LexTokenManager.Token_Position;
   Abstraction        : Dictionary.Abstractions;
   Error,
   SemanticErrorFound : Boolean;
   ExportList,
   ImportList,
   ExportsInRelation,
   ImportsInRelation  : SeqAlgebra.Seq;
   NullImportNode     : STree.SyntaxNode;  -- location of "derives null..."

   ------------------------------------------------------------------------

   procedure wf_dependency_clause (TheHeap     : in out Heap.HeapRecord;
                                   Node        : in     STree.SyntaxNode;
                                   SubprogSym  : in     Dictionary.Symbol;
                                   Abstraction : in     Dictionary.Abstractions;
                                   DoIFA       : in     Boolean;
                                   NullList    : in     Boolean; -- true -> handling "derives null"
                                   ErrorFound  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     CurrentScope;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out ExportList;
   --#           out ImportList;
   --# derives ErrorFound                from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        DoIFA,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        NullList,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        DoIFA,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        NullList,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        DoIFA,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        NullList,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         ExportList,
   --#         ImportList                from TheHeap;
   is
      StarNode,
      ImportListNode,
      ExportListNode   : STree.SyntaxNode;
      FirstValidExport : Dictionary.Symbol;
      LocalError       : Boolean;
      StarFound        : Boolean;

      -------------------------------------------------------------

      procedure ValidExportListAdd (Sym : in     Dictionary.Symbol)
      --# global in     ExportList;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    ExportList,
      --#                                    Sym,
      --#                                    TheHeap;
      is
      begin
         SeqAlgebra.AddMember (TheHeap,
                               ExportList,
                               Natural (Dictionary.SymbolRef (Sym)));
      end ValidExportListAdd;

      ---------------------------

      procedure GetFirstValidExport (Sym :    out Dictionary.Symbol)
      --# global in ExportList;
      --#        in TheHeap;
      --# derives Sym from ExportList,
      --#                  TheHeap;
      is
         Member : SeqAlgebra.MemberOfSeq;
      begin
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         if SeqAlgebra.IsNullMember (Member) then
            Sym := Dictionary.NullSymbol;
         else
            Sym := Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType
                 (SeqAlgebra.Value_Of_Member
                    (The_Heap => TheHeap,
                     M        => Member)));
         end if;
      end GetFirstValidExport;

      -----------------------------------------------------------------------

      procedure UniqueExport (ExportSym  : in out Dictionary.Symbol;
                              ErrorPos   : in     LexTokenManager.Token_Position;
                              SubprogSym : in     Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorPos,
      --#                                        ExportSym,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        SubprogSym &
      --#         ExportSym,
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ExportSym,
      --#                                        SubprogSym;
      is
         NextExport : Dictionary.Iterator;

      begin
         NextExport := Dictionary.FirstExport (Abstraction,
                                               SubprogSym);
         while NextExport /= Dictionary.NullIterator loop
            if ExportSym = Dictionary.CurrentSymbol (NextExport) then
               LocalError := True;
               ErrorHandler.SemanticError (159,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (ExportSym));
               ExportSym := Dictionary.NullSymbol;
               exit;
            end if;
            NextExport := Dictionary.NextSymbol (NextExport);
         end loop;
      end UniqueExport;

      -----------------------------------------------------------------------

      -- Uses an explicit import list.  The import is passed in,
      -- if it is already in the import list an error is reported and a null symbol
      -- returned; else it is added to the list.  Imports by virtue of * never
      -- appear in the list and do not give rise to duplicate errors.

      procedure UniqueImport (ImportSym  : in out Dictionary.Symbol;
                              ErrorPos   : in     LexTokenManager.Token_Position)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorPos,
      --#                                        ImportList,
      --#                                        ImportSym,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        TheHeap &
      --#         ImportSym,
      --#         LocalError,
      --#         Statistics.TableUsage,
      --#         TheHeap                   from *,
      --#                                        ImportList,
      --#                                        ImportSym,
      --#                                        TheHeap;
      is
         ImportRep : Natural;

      begin
         ImportRep := Natural (Dictionary.SymbolRef (ImportSym));
         if SeqAlgebra.IsMember (TheHeap, ImportList, ImportRep) then
            LocalError := True;
            ErrorHandler.SemanticError (160,
                                        ErrorHandler.NoReference,
                                        ErrorPos,
                                        Dictionary.GetSimpleName (ImportSym));
            ImportSym := Dictionary.NullSymbol;
         else
            SeqAlgebra.AddMember (TheHeap, ImportList, ImportRep);
         end if;
      end UniqueImport;

      -----------------------------------------------------------------------

      -- Similar to UniqueImport but this checks that the import has not appeared as
      -- an import anywhere in the dependency relation whereas UniqueImport only checks
      -- it is not duplicated in the current clause. CheckNotImportedBefore is for use
      -- in handling "derives null from ..." cases where the imports must not have
      -- been imported before.
      procedure CheckNotImportedBefore (ImportSym  : in out Dictionary.Symbol;
                                        ErrorPos   : in     LexTokenManager.Token_Position)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorPos,
      --#                                        ImportSym,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        SubprogSym &
      --#         ImportSym,
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ImportSym,
      --#                                        SubprogSym &
      --#         Statistics.TableUsage,
      --#         TheHeap                   from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ImportList,
      --#                                        ImportSym,
      --#                                        SubprogSym,
      --#                                        TheHeap;
      is
         It : Dictionary.Iterator;
      begin --CheckNotImportedBefore
         It := Dictionary.FirstImport (Abstraction, SubprogSym);
         while not Dictionary.IsNullIterator (It) loop
            if Dictionary.CurrentSymbol (It) = ImportSym then --duplicate
               LocalError := True;
               ErrorHandler.SemanticError (160,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (ImportSym));
               ImportSym := Dictionary.NullSymbol;
            else
               SeqAlgebra.AddMember (TheHeap, ImportList, Natural (Dictionary.SymbolRef (ImportSym)));
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end CheckNotImportedBefore;

      -----------------------------------------------------------------------

      procedure CheckExportMode (ExportSym  : in Dictionary.Symbol;
                                 SubprogSym : in Dictionary.Symbol;
                                 ErrorPos   : in LexTokenManager.Token_Position)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorPos,
      --#                                        ExportSym,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        SubprogSym &
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ExportSym,
      --#                                        SubprogSym;
      --#

      is
         Mode : Dictionary.Modes;

      begin
         if Dictionary.IsFormalParameter (SubprogSym, ExportSym) then
            Mode := Dictionary.GetSubprogramParameterMode (ExportSym);
            if Mode = Dictionary.InMode or
              Mode = Dictionary.DefaultMode
            then
               LocalError := True;
               ErrorHandler.SemanticError (161,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (ExportSym));
            end if;

         else -- check mode, only "IN" is unacceptable
            Mode := Dictionary.GetGlobalMode (Abstraction,
                                              SubprogSym,
                                              ExportSym);
            if Mode = Dictionary.InMode then
               LocalError := True;
               ErrorHandler.SemanticErrorSym (502,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              ExportSym,
                                              CurrentScope);
            end if;
         end if;
      end CheckExportMode;

      -----------------------------------------------------------------------

      -- We check that imports are of the correct mode and (at the
      -- main progam level) that they are initialized.  If the import list
      -- includes a * self reference we do not need to make these checks where the
      -- import being checked is also an export since the checks will already have
      -- been made at the * node.  This function is used by CheckImportMode and
      -- CheckImportInit because the check needed is the same in both cases.

      function ExportImportedByStar (ImportSym : Dictionary.Symbol;
                                     StarFound : Boolean) return Boolean
      --# global in ExportList;
      --#        in TheHeap;
      is
      begin
         return StarFound and then
           SeqAlgebra.IsMember (TheHeap,
                                ExportList,
                                Natural (Dictionary.SymbolRef (ImportSym)));
      end ExportImportedByStar;

      -----------------------------------------------------------------------

      -- ignore checks if the import is an export which has already been
      -- imported by use of the * abbreviation; this is because checks will already
      -- have been carried out at the * node.

      procedure CheckImportMode (ImportSym  : in Dictionary.Symbol;
                                 SubprogSym : in Dictionary.Symbol;
                                 ErrorPos   : in LexTokenManager.Token_Position;
                                 StarFound  : in Boolean)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in     TheHeap;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ErrorPos,
      --#                                        ExportList,
      --#                                        ImportSym,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        StarFound,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ExportList,
      --#                                        ImportSym,
      --#                                        StarFound,
      --#                                        SubprogSym,
      --#                                        TheHeap;
      is

      begin
         if not ExportImportedByStar (ImportSym, StarFound) then
            if Dictionary.IsFormalParameter (SubprogSym, ImportSym) then
               if Dictionary.GetSubprogramParameterMode (ImportSym) =
                 Dictionary.OutMode
               then
                  LocalError := True;
                  ErrorHandler.SemanticError (162,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              Dictionary.GetSimpleName (ImportSym));
               end if;

            elsif -- check mode, only "OUT" is unacceptable
              Dictionary.GetGlobalMode (Abstraction,
                                        SubprogSym,
                                        ImportSym) = Dictionary.OutMode
            then
               LocalError := True;
               ErrorHandler.SemanticErrorSym (503,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              ImportSym,
                                              CurrentScope);
            end if;
            -- check here that we are not trying to import a mode out stream variable
            if Dictionary.GetOwnVariableOrConstituentMode (ImportSym) =
              Dictionary.OutMode then
               ErrorHandler.SemanticError (714,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (ImportSym));
               LocalError := True;
            end if;
         end if;
      end CheckImportMode;

      -----------------------------------------------------------------------
      -- ignore checks if the import is an export which has already been
      -- imported by use of the * abbreviation; this is because checks will already
      -- have been carried out at the * node.

      procedure CheckImportInit (ImportNode : in STree.SyntaxNode;
                                 ImportSym,
                                 SubprogSym : in Dictionary.Symbol;
                                 StarFound  : in Boolean)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TheHeap;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ExportList,
      --#                                        ImportNode,
      --#                                        ImportSym,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        StarFound,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ExportList,
      --#                                        ImportSym,
      --#                                        StarFound,
      --#                                        SubprogSym,
      --#                                        TheHeap;
      is

      begin
         if not ExportImportedByStar (ImportSym, StarFound) then

            if Dictionary.IsTaskType (SubprogSym) and then
              Dictionary.IsOwnVariable (ImportSym) and then
              not Dictionary.GetOwnVariableProtected (ImportSym) and then
              Dictionary.GetOwnVariableMode (ImportSym) /= Dictionary.InMode and then
              Dictionary.PartitionElaborationPolicyIsConcurrent
            then
               -- This is an unprotected import to a task that is not of mode in.
               -- The elaboration policy is concurrent and hence we cannot
               -- guarantee the import is initialised before it is used.
               LocalError := True;
               ErrorHandler.SemanticErrorSym (958,
                                              ErrorHandler.NoReference,
                                              NodePosition (ImportNode),
                                              ImportSym,
                                              CurrentScope);

            elsif (Dictionary.IsMainProgram (SubprogSym) or
                     Dictionary.IsTaskType (SubprogSym)) then
               if Dictionary.IsGlobalVariable (Abstraction,
                                               SubprogSym,
                                               ImportSym) and then
                 Dictionary.IsOwnVariable (ImportSym) and then
                 not Dictionary.OwnVariableIsInitialized (ImportSym) and then
                 Dictionary.GetOwnVariableOrConstituentMode (ImportSym) =
                 Dictionary.DefaultMode
               then
                  LocalError := True;
                  ErrorHandler.SemanticError (167,
                                              ErrorHandler.NoReference,
                                              NodePosition (ImportNode),
                                              Dictionary.GetSimpleName (ImportSym));
               end if;
            end if;
         end if;
      end CheckImportInit;

      -----------------------------------------------------------------------

      procedure DoExportList (Node             : in STree.SyntaxNode;
                              SubprogSym       : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.File_Sys,
      --#                                        STree.Table,
      --#                                        SubprogSym &
      --#         Dictionary.Dict,
      --#         LocalError,
      --#         STree.Table               from *,
      --#                                        Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        SubprogSym &
      --#         Statistics.TableUsage,
      --#         TheHeap                   from *,
      --#                                        Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ExportList,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap;
      is
         Sym        : Dictionary.Symbol;
         It         : STree.Iterator;
         ExportNode : STree.SyntaxNode;
         Unused     : Boolean;
      begin
         It := FindFirstNode (NodeKind    => SPSymbols.entire_variable,
                              FromRoot    => Node,
                              InDirection => STree.Down);

         while not STree.IsNull (It) loop

            ExportNode := GetNode (It);

            --# accept Flow, 10, Unused, "Expected ineffective assignment";
            wf_entire_variable (ExportNode,
                                Dictionary.LocalScope (SubprogSym),
                                In_Derives_Export_List,
                                 --to get
                                Sym,
                                Unused);
            --# end accept;
            LocalError := LocalError or (Sym = Dictionary.NullSymbol);
            if Sym /= Dictionary.NullSymbol then
               UniqueExport (Sym,
                             NodePosition (ExportNode),
                             SubprogSym);
            end if;

            -- check that the export is not a stream variable of mode in
            if Sym /= Dictionary.NullSymbol then
               if Dictionary.GetOwnVariableOrConstituentMode (Sym) =
                 Dictionary.InMode then
                  ErrorHandler.SemanticError (713,
                                              ErrorHandler.NoReference,
                                              NodePosition (ExportNode),
                                              Dictionary.GetSimpleName (Sym));
                  LocalError := True;
                  Sym := Dictionary.NullSymbol;
               end if;
            end if;

            if Dictionary.IsTaskType (SubprogSym) and then
              Dictionary.IsOwnVariable (Sym) and then
              not Dictionary.GetOwnVariableProtected (Sym) and then
              Dictionary.OwnVariableIsInitialized (Sym) and then
              Dictionary.PartitionElaborationPolicyIsConcurrent
            then
               -- This export is defined during elaboration as well by this task.
               -- The order is non-deterministic. Hence we must prohibit the export
               -- being initialised during elaboration.
               ErrorHandler.SemanticErrorSym (959,
                                              ErrorHandler.NoReference,
                                              NodePosition (ExportNode),
                                              Sym,
                                              CurrentScope);
            end if;

            if Sym /= Dictionary.NullSymbol then
               CheckExportMode (Sym,
                                SubprogSym,
                                NodePosition (ExportNode));
               Dictionary.AddExport (Abstraction     => Abstraction,
                                     TheProcedure    => SubprogSym,
                                     TheExport       => Sym,
                                     ExportReference => Dictionary.Location'(NodePosition (ExportNode),
                                                                             NodePosition (ExportNode)),
                                     Annotation      => Dictionary.Location'(NodePosition (Node),
                                                                             NodePosition (Node)));
               ValidExportListAdd (Sym);
            end if;
            It := STree.NextNode (It);
         end loop;

         --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
      end DoExportList;

      -----------------------------------------------------------------------

      procedure DoImportList (Node : in STree.SyntaxNode)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     CurrentScope;
      --#        in     DoIFA;
      --#        in     ExportList;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in     NullList;
      --#        in     StarFound;
      --#        in     SubprogSym;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --#           out FirstValidExport;
      --# derives Dictionary.Dict,
      --#         LocalError,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                   from *,
      --#                                        Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        DoIFA,
      --#                                        ExportList,
      --#                                        ImportList,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        NullList,
      --#                                        StarFound,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        DoIFA,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ExportList,
      --#                                        ImportList,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        NullList,
      --#                                        SPARK_IO.File_Sys,
      --#                                        StarFound,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         FirstValidExport          from ExportList,
      --#                                        TheHeap;
      is
         Sym              : Dictionary.Symbol;
         It               : STree.Iterator;
         ImportNode       : STree.SyntaxNode;
         Unused           : Boolean;

         --------------------------------------------------------------------
         procedure AddDependencies (Abstraction     : in Dictionary.Abstractions;
                                    TheProcedure    : in Dictionary.Symbol;
                                    FirstExport     : in Dictionary.Symbol;
                                    TheImport       : in Dictionary.Symbol;
                                    ImportReference : in Dictionary.Location)
         --# global in     ContextManager.Ops.UnitStack;
         --#        in     ExportList;
         --#        in     LexTokenManager.State;
         --#        in     StarFound;
         --#        in     TheHeap;
         --#        in out Dictionary.Dict;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Dictionary.Dict   from *,
         --#                                Abstraction,
         --#                                ContextManager.Ops.UnitStack,
         --#                                ExportList,
         --#                                FirstExport,
         --#                                ImportReference,
         --#                                StarFound,
         --#                                TheHeap,
         --#                                TheImport,
         --#                                TheProcedure &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                Abstraction,
         --#                                ContextManager.Ops.UnitStack,
         --#                                Dictionary.Dict,
         --#                                ExportList,
         --#                                FirstExport,
         --#                                ImportReference,
         --#                                LexTokenManager.State,
         --#                                StarFound,
         --#                                TheHeap,
         --#                                TheImport,
         --#                                TheProcedure;
         is
            Member        : SeqAlgebra.MemberOfSeq;
            CurrentExport : Dictionary.Symbol;

         begin
            if StarFound then
               Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
               while not SeqAlgebra.IsNullMember (Member)
               loop
                  CurrentExport := Dictionary.ConvertSymbolRef
                    (ExaminerConstants.RefType
                       (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                    M        => Member)));
                  if TheImport /= CurrentExport then    -- don't add if already added by *
                     Dictionary.AddDependency (Abstraction     => Abstraction,
                                               Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                               TheProcedure    => TheProcedure,
                                               TheExport       => CurrentExport,
                                               TheImport       => TheImport,
                                               ImportReference => ImportReference);
                  end if;
                  Member := SeqAlgebra.NextMember (TheHeap, Member);
               end loop;

            else --only need to copy it to first export (CopyImports will do rest)
               Dictionary.AddDependency (Abstraction     => Abstraction,
                                         Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                         TheProcedure    => TheProcedure,
                                         TheExport       => FirstExport,
                                         TheImport       => TheImport,
                                         ImportReference => ImportReference);
            end if;
         end AddDependencies;

         -------------------

      begin --DoImportList
         GetFirstValidExport (FirstValidExport); -- NullSymbol if doing a null derives

         It := FindFirstNode (NodeKind    => SPSymbols.entire_variable,
                              FromRoot    => Node,
                              InDirection => STree.Down);

         while not STree.IsNull (It) loop

            ImportNode := GetNode (It);
            --# accept Flow, 10, Unused, "Expected ineffective assignment";
            wf_entire_variable (ImportNode,
                                Dictionary.LocalScope (SubprogSym),
                                In_Derives_Import_List,
                                 --to get
                                Sym,
                                Unused);
            --# end accept;
            LocalError := LocalError or (Sym = Dictionary.NullSymbol);
            if Sym /= Dictionary.NullSymbol and then
              FirstValidExport /= Dictionary.NullSymbol  -- also covers NullList case
            then
               UniqueImport (Sym,
                             NodePosition (ImportNode));
            end if;

            -- new check, similar to UniqueImport except that for a null import clause
            -- we require that the variable has not appeared as an import anywhere else rather
            -- than just in the current clause
            if NullList and then
              Sym /= Dictionary.NullSymbol then
               CheckNotImportedBefore (Sym,
                                       NodePosition (ImportNode));
            end if;

            if Sym /= Dictionary.NullSymbol then  -- we have valid import so far
               CheckImportMode (Sym,
                                SubprogSym,
                                NodePosition (ImportNode),
                                StarFound);
               CheckImportInit (ImportNode,
                                Sym,
                                SubprogSym,
                                StarFound);

               -- One of two things must happen to the valid import we have at this point.
               -- If we are doing IFA and we have valid export to associated it with we do so;
               -- however, if we are not doing IFA or there is no valid export we just
               -- mark the import in the dictionary as being an import usign the force call
               --# accept Flow, 41, "Expected stable expression";
               if DoIFA and then --  stable index OK
                 FirstValidExport /= Dictionary.NullSymbol then
               --# end accept;
                  AddDependencies (Abstraction,
                                   SubprogSym,
                                   FirstValidExport,
                                   Sym,
                                   Dictionary.Location'(NodePosition (ImportNode),
                                                        NodePosition (ImportNode)));
               else
                  -- No valid export (including null derives case) to associate with  or else we
                  -- are just doing DFA.  In either case mark as import to avoid
                  -- knock-on errors
                  Dictionary.ForceImport
                    (Abstraction,
                     SubprogSym,
                     Sym,
                     Dictionary.Location'(NodePosition (ImportNode),
                                          NodePosition (ImportNode)));
               end if;
            end if;
            It := STree.NextNode (It);
         end loop;
         --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
      end DoImportList;

      -----------------------------------------------------------------------

      procedure DoSelfReferences (StarNode   : in STree.SyntaxNode;
                                  SubprogSym : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     CurrentScope;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TheHeap;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict,
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        Dictionary.Dict,
      --#                                        ExportList,
      --#                                        StarNode,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ExportList,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        StarNode,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap;
      is
         Member    : SeqAlgebra.MemberOfSeq;
         TheExport : Dictionary.Symbol;

         ------------------------------------------------------------------

      begin --DoSelfReferences
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         while not SeqAlgebra.IsNullMember (Member)
         loop
            TheExport := Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType
                 (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                              M        => Member)));
            CheckImportMode (TheExport,
                             SubprogSym,
                             NodePosition (StarNode),
                             False);
            CheckImportInit (StarNode,
                             TheExport,
                             SubprogSym,
                             False);
            Dictionary.AddDependency (Abstraction     => Abstraction,
                                      Comp_Unit       => ContextManager.Ops.CurrentUnit,
                                      TheProcedure    => SubprogSym,
                                      TheExport       => TheExport,
                                      TheImport       => TheExport,
                                      ImportReference => Dictionary.Location'(NodePosition (StarNode),
                                                                              NodePosition (StarNode)));
            Member := SeqAlgebra.NextMember (TheHeap, Member);
         end loop;
      end DoSelfReferences;

      -----------------------------------------------------------------------

      -- This procedure does checks on * node when no IFA selected
      procedure AppendSelfReferences (StarNode   : in STree.SyntaxNode;
                                      SubprogSym : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     ExportList;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LocalError;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ExportList,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        StarNode,
      --#                                        STree.Table,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         LocalError                from *,
      --#                                        Abstraction,
      --#                                        Dictionary.Dict,
      --#                                        ExportList,
      --#                                        SubprogSym,
      --#                                        TheHeap &
      --#         Statistics.TableUsage,
      --#         TheHeap                   from *,
      --#                                        ExportList,
      --#                                        ImportList,
      --#                                        TheHeap;
      is
         Member    : SeqAlgebra.MemberOfSeq;
         TheExport : Dictionary.Symbol;

      begin --AppendSelfReferences
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         while not SeqAlgebra.IsNullMember (Member)
         loop
            TheExport := Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType
                 (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                              M        => Member)));
            CheckImportMode (TheExport,
                             SubprogSym,
                             NodePosition (StarNode),
                             False);
            CheckImportInit (StarNode,
                             TheExport,
                             SubprogSym,
                             False);
            Member := SeqAlgebra.NextMember (TheHeap, Member);
         end loop;
         SeqAlgebra.AugmentSeq (TheHeap, ImportList, ExportList);
      end AppendSelfReferences;

      -----------------------------------------------------------------------

      procedure CopyImports (SubprogSym : in     Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     ExportList;
      --#        in     FirstValidExport;
      --#        in     TheHeap;
      --#        in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              ExportList,
      --#                              FirstValidExport,
      --#                              SubprogSym,
      --#                              TheHeap;
      --#
      --#
      --#
      --#
      --#
      is
         Member : SeqAlgebra.MemberOfSeq;

      begin
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         if not SeqAlgebra.IsNullMember (Member) then
            --there is at leat one valid export, we want to loop through rest
            loop
               Member := SeqAlgebra.NextMember (TheHeap, Member);
               exit when SeqAlgebra.IsNullMember (Member);

               Dictionary.CopyDependencyList
                 (Abstraction,
                  SubprogSym,
                  FirstValidExport,
                  Dictionary.ConvertSymbolRef
                    (ExaminerConstants.RefType
                       (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                    M        => Member))));
            end loop;
         end if;
      end CopyImports;

      -----------------------------------------------------------------------

      procedure CheckStar (Node      : in     STree.SyntaxNode;
                           Found     :    out Boolean;
                           StarNode,
                           ImpNode   :    out STree.SyntaxNode)
      --# global in STree.Table;
      --# derives Found,
      --#         ImpNode,
      --#         StarNode from Node,
      --#                       STree.Table;
      is
         ImpNodeLocal : STree.SyntaxNode;
      begin
         ImpNodeLocal := Next_Sibling (Child_Node (Node));
         ImpNode := ImpNodeLocal;
         if SyntaxNodeType (Child_Node (ImpNodeLocal)) =
           SPSymbols.multiply
         then
            Found := True;
            StarNode := Child_Node (ImpNodeLocal);
         else
            Found := False;
            StarNode := STree.NullNode;
         end if;
      end CheckStar;

      -----------------------------------------------------------------------

   begin  --wf_dependency_clause
      -- ASSUME Node = dependency_clause
      --
      -- For each exported variable E :
      --    E is a visible entire variable;
      --    occurrence of E as an export is unique (Section 6.1.2 Rule 6);
      --    mode of E is in out or out (Section 6.1.2 Rule 2);
      --    for each imported variable I :
      --       I is a visible entire variable;
      --       occurrence of I as an import is unique (Section 6.1.2 Rule 6);
      --       mode of I is in or in out (Section 6.1.2 Rule 1);
      --       if dependency clause belongs to main_program, then I is initialized by
      --          the package of which it is an own variable;
      --

      SeqAlgebra.CreateSeq (TheHeap, ExportList);
      SeqAlgebra.CreateSeq (TheHeap, ImportList);
      LocalError := False;

      if NullList then
         -- we are dealing with " derives null from ..." so there are no stars to worry
         -- about and we don't have to go looking for the node holding the list of imports
         -- because that is what has been passed in
         StarFound := False;
         StarNode := STree.NullNode; -- not used but removes apparent DFA
         ImportListNode := Node;
         -- Add NullVariable as an export to provide a data sink for the imported items
         Dictionary.AddGlobalVariable (Abstraction       => Abstraction,
                                       Subprogram        => SubprogSym,
                                       Variable          => Dictionary.GetNullVariable,
                                       Mode              => Dictionary.OutMode,
                                       PrefixNeeded      => False,
                                       Comp_Unit         => ContextManager.Ops.CurrentUnit,
                                       VariableReference => Dictionary.Location'(NodePosition (Node),
                                                                                 NodePosition (Node)));
         Dictionary.AddExport (Abstraction     => Abstraction,
                               TheProcedure    => SubprogSym,
                               TheExport       => Dictionary.GetNullVariable,
                               ExportReference => Dictionary.Location'(NodePosition (Node),
                                                                       NodePosition (Node)),
                               Annotation      => Dictionary.Location'(NodePosition (Node),
                                                                       NodePosition (Node)));
         ValidExportListAdd (Dictionary.GetNullVariable);
      else
         -- it's an ordinary dependency clause
         ExportListNode := Child_Node (Node);
         CheckStar (Node,
                     --to get
                    StarFound,
                    StarNode,
                    ImportListNode);
         DoExportList (ExportListNode,
                       SubprogSym);
      end if;

      --# accept Flow, 10, FirstValidExport, "Expected ineffective assignment";
      if DoIFA then --DoIFA can only be False here in SPARK 83 DFA mode
         if StarFound then
            DoSelfReferences (StarNode,
                              SubprogSym);
            DoImportList (ImportListNode); -- ineffective FirstValidExport OK
         else
            DoImportList (ImportListNode);
            CopyImports (SubprogSym);
         end if;
      else --not doing IFA
         DoImportList (ImportListNode);  -- ineffective FirstValidExport OK
         if StarFound then
            AppendSelfReferences (StarNode, SubprogSym);
         end if;
      end if;
      ErrorFound := LocalError;
   end wf_dependency_clause;

   ----------------------------------------------------------------------------

   function PositionToReportError (Node : STree.SyntaxNode)
                                  return LexTokenManager.Token_Position
   --# global in STree.Table;
   is
   begin
      return NodePosition (STree.FindLastItemInDependencyRelation (Node));
   end PositionToReportError;

   -------------------------------------------------------------------------

   procedure ParamsAreImportsOrExports (SubprogSym : in Dictionary.Symbol;
                                        ErrorPos   : in LexTokenManager.Token_Position)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemanticErrorFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorPos,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubprogSym &
   --#         SemanticErrorFound        from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        SubprogSym;
   is
      NextParam : Dictionary.Iterator;
      ParamSym  : Dictionary.Symbol;

   begin
      NextParam := Dictionary.FirstSubprogramParameter (SubprogSym);
      while NextParam /= Dictionary.NullIterator loop
         ParamSym := Dictionary.CurrentSymbol (NextParam);
         if not Dictionary.IsImport (Abstraction,
                                     SubprogSym,
                                     ParamSym) and
           not Dictionary.IsExport (Abstraction,
                                    SubprogSym,
                                    ParamSym)
         then
            SemanticErrorFound := True;
            ErrorHandler.SemanticError (200,
                                        ErrorHandler.NoReference,
                                        ErrorPos,
                                        Dictionary.GetSimpleName (ParamSym));
         end if;

         -- test to ensure that parameters of mode IN OUT are imported
         if Dictionary.GetSubprogramParameterMode (ParamSym) =
           Dictionary.InOutMode
         then
            if not Dictionary.IsImport (Abstraction, SubprogSym, ParamSym) then
               --# accept Flow, 41, "Expected stable expression";
               if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 then
               --# end accept;
                  SemanticErrorFound := True;
                  ErrorHandler.SemanticError (504,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              Dictionary.GetSimpleName (ParamSym));

               elsif Dictionary.TypeIsScalar (Dictionary.GetType (ParamSym)) then
                  SemanticErrorFound := True;
                  ErrorHandler.SemanticError (338,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              Dictionary.GetSimpleName (ParamSym));
               end if;
            end if; --end of checks that in out is an import

            -- check that in out is exported
            if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
              not Dictionary.IsExport (Abstraction, SubprogSym,  ParamSym)
            then
               SemanticErrorFound := True;
               ErrorHandler.SemanticError (506,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           Dictionary.GetSimpleName (ParamSym));
            end if;
         end if;
         NextParam := Dictionary.NextSymbol (NextParam);
      end loop;
   end ParamsAreImportsOrExports;

   ---------------------------------------------------------------------------

   procedure GlobalsAreImportsOrExports (SubprogSym : in Dictionary.Symbol;
                                         ErrorPos   : in LexTokenManager.Token_Position)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     CurrentScope;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemanticErrorFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorPos,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubprogSym &
   --#         SemanticErrorFound        from *,
   --#                                        Abstraction,
   --#                                        Dictionary.Dict,
   --#                                        SubprogSym;
   --#
   is
      NextGlobal : Dictionary.Iterator;
      GlobalSym  : Dictionary.Symbol;

   begin
      NextGlobal := Dictionary.FirstGlobalVariable (Abstraction,
                                                    SubprogSym);
      while NextGlobal /= Dictionary.NullIterator loop
         GlobalSym := Dictionary.CurrentSymbol (NextGlobal);
         if not Dictionary.IsImport (Abstraction,
                                     SubprogSym,
                                     GlobalSym) and then
           not Dictionary.IsExport (Abstraction,
                                    SubprogSym,
                                    GlobalSym)
         then
            SemanticErrorFound := True;
            ErrorHandler.SemanticErrorSym (201,
                                           ErrorHandler.NoReference,
                                           ErrorPos,
                                           GlobalSym,
                                           CurrentScope);
         end if;
         -- check that globals with mode in out are both imported and exported.
         if
           Dictionary.GetGlobalMode (Abstraction,
                                     SubprogSym,
                                     GlobalSym) = Dictionary.InOutMode
         then
            if not Dictionary.IsImport (Abstraction,
                                        SubprogSym,
                                        GlobalSym)
            then
               SemanticErrorFound := True;
               ErrorHandler.SemanticErrorSym (505,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              GlobalSym,
                                              CurrentScope);
            end if;
            if not Dictionary.IsExport (Abstraction,
                                        SubprogSym,
                                        GlobalSym)
            then
               SemanticErrorFound := True;
               ErrorHandler.SemanticErrorSym (507,
                                              ErrorHandler.NoReference,
                                              ErrorPos,
                                              GlobalSym,
                                              CurrentScope);
            end if;
         end if;
         NextGlobal := Dictionary.NextSymbol (NextGlobal);
      end loop;
   end GlobalsAreImportsOrExports;

   ---------------------------------------------------------------------------

   procedure CheckInformationFlowPolicy (SubprogSym : in Dictionary.Symbol;
                                         Position   : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     CurrentScope;
   --#        in     Dictionary.Dict;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SemanticErrorFound;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubprogSym &
   --#         SemanticErrorFound        from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        FirstSeen,
   --#                                        LexTokenManager.State,
   --#                                        SubprogSym;
   is
      ExportIt : Dictionary.Iterator;
      ImportIt : Dictionary.Iterator;
      TheExport : Dictionary.Symbol;
      TheImport : Dictionary.Symbol;

      type FSToAbsTable is array (Boolean) of Dictionary.Abstractions;
      ToAbs : constant FSToAbsTable := FSToAbsTable'(False => Dictionary.IsRefined,
                                                     True  => Dictionary.IsAbstract);
   begin
      -- Iterate over all the Exports of this subprogram...
      ExportIt := Dictionary.FirstExport (ToAbs (FirstSeen),
                                          SubprogSym);
      while ExportIt /= Dictionary.NullIterator loop
         TheExport := Dictionary.CurrentSymbol (ExportIt);

         ImportIt := Dictionary.FirstDependency (ToAbs (FirstSeen),
                                                 SubprogSym,
                                                 TheExport);

         -- For each export, iterate over all the imports...
         while ImportIt /= Dictionary.NullIterator loop

            TheImport := Dictionary.CurrentSymbol (ImportIt);

            -- ...and see if each relation violates the current information
            -- flow policy.
            if Dictionary.RelationViolatesInfoFlowPolicy (TheExport, TheImport) then

               ErrorHandler.SemanticErrorSym2 (175,
                                               ErrorHandler.NoReference,
                                               Position,
                                               TheImport,
                                               TheExport,
                                               CurrentScope);
               SemanticErrorFound := True;
            end if;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         ExportIt := Dictionary.NextSymbol (ExportIt);
      end loop;
   end CheckInformationFlowPolicy;

begin --wf_dependency_relation
      -- ASSUME Node = dependency_relation
   Position := PositionToReportError (Node);
   if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
     not CommandLineData.Content.DoInformationFlow
   then -- derives not used in this case, don't process, just issue note
      ErrorHandler.SemanticNote (1,
                                 Position,
                                 LexTokenManager.Null_String);
   else --proceed as usual (even if doing DFA in 83 mode)
      if FirstSeen then
         Abstraction := Dictionary.IsAbstract;
      else
         Abstraction := Dictionary.IsRefined;
      end if;

      SeqAlgebra.CreateSeq (TheHeap, ExportsInRelation);
      SeqAlgebra.CreateSeq (TheHeap, ImportsInRelation);

      SemanticErrorFound := GlobDefErr;
      Dictionary.AddDependencyRelation (Abstraction,
                                        SubprogSym,
                                        Dictionary.Location'(NodePosition (Node),
                                                             NodePosition (Node)));

      It := FindFirstNode (NodeKind    => SPSymbols.dependency_clause,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop

         NextNode := GetNode (It);
         -- globals ImportList, ExportList are created in wf_dependency_clause
         wf_dependency_clause (TheHeap,
                               NextNode,
                               SubprogSym,
                               Abstraction,
                               CommandLineData.Content.DoInformationFlow,
                               False, -- NOT doing a derives null from ... list
                               Error);
         SemanticErrorFound := SemanticErrorFound or Error;

         -- add in the imports and exports from this clause to the total imports/exports
         --# accept Flow, 41, "Expected stable expression";
         if not CommandLineData.Content.DoInformationFlow then  -- stable index OK
         --# end accept;
            SeqAlgebra.AugmentSeq (TheHeap, ExportsInRelation, ExportList);
            SeqAlgebra.AugmentSeq (TheHeap, ImportsInRelation, ImportList);
         end if;
         SeqAlgebra.DisposeOfSeq (TheHeap, ExportList);
         SeqAlgebra.DisposeOfSeq (TheHeap, ImportList);
         It := STree.NextNode (It);
      end loop;

      -- Process any import list of the form "derives null from x, y, z;"
      NullImportNode := Child_Node (Child_Node (Node));
      if SyntaxNodeType (NullImportNode) = SPSymbols.dependency_relation_rep then
         NullImportNode := Next_Sibling (Next_Sibling (NullImportNode));
      end if;
      -- if there is a null import clause then NullImportNode is now pointing at it
      if SyntaxNodeType (NullImportNode) = SPSymbols.null_import_list then
         wf_dependency_clause (TheHeap,
                               Next_Sibling (Child_Node (NullImportNode)),
                               SubprogSym,
                               Abstraction,
                               CommandLineData.Content.DoInformationFlow,
                               True, -- we ARE doing a null list
                               Error);
         SemanticErrorFound := SemanticErrorFound or Error;

         -- add in the imports and exports from this clause to the total imports/exports
         if not CommandLineData.Content.DoInformationFlow then  -- stable index OK
            SeqAlgebra.AugmentSeq (TheHeap, ExportsInRelation, ExportList);
            SeqAlgebra.AugmentSeq (TheHeap, ImportsInRelation, ImportList);
         end if;
         SeqAlgebra.DisposeOfSeq (TheHeap, ExportList);
         SeqAlgebra.DisposeOfSeq (TheHeap, ImportList);
      end if;

      -- make all exports depend on all imports here
      if not CommandLineData.Content.DoInformationFlow then
         ErrorHandler.SemanticNote (2,
                                    Position,
                                    LexTokenManager.Null_String);
         CreateFullDependency (Node        => Node,
                               SubprogSym  => SubprogSym,
                               Abstraction => Abstraction,
                               ImportList  => ImportsInRelation,
                               ExportList  => ExportsInRelation);
      end if;

      SeqAlgebra.DisposeOfSeq (TheHeap, ExportsInRelation);
      SeqAlgebra.DisposeOfSeq (TheHeap, ImportsInRelation);

      ParamsAreImportsOrExports (SubprogSym, Position);
      GlobalsAreImportsOrExports (SubprogSym, Position);

      --do consistency checks if second anno and no semantic errors found and
      --second global annotation is not missing completely
      if not (FirstSeen           or
                SemanticErrorFound  or
                (SyntaxNodeType (Child_Node (ParentNode (Node))) =
                   SPSymbols.dependency_relation)) then
         CheckDerivesConsistency (SubprogSym, Position);
      end if;

      if not SemanticErrorFound then
         CheckInformationFlowPolicy (SubprogSym, Position);
      end if;

      -- mark subprogram as having incorrect signature if necessary
      if SemanticErrorFound then
         Dictionary.SetSubprogramSignatureNotWellformed (Abstraction,
                                                         SubprogSym);
      end if;
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Derives (Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                 Parse_Tree  => Node,
                                 Scope       => Dictionary.LocalScope (SubprogSym),
                                 Subprog_Sym => SubprogSym);
   end if;

end wf_dependency_relation;
