-- $Id: sem-compunit-wf_basic_declarative_item.adb 16669 2010-04-01 11:26:15Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;
separate (Sem.CompUnit)

procedure wf_basic_declarative_item (Node         : in STree.SyntaxNode;
                                     CurrentScope : in Dictionary.Scopes)
is
   NodeToCheck     : STree.SyntaxNode;
   NodeToCheckType : SPSymbols.SPSymbol;

   --------------------------------------------------------------------------

   procedure wf_basic_declaration (Node         : in STree.SyntaxNode;
                                   CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      NodeToCheck     : STree.SyntaxNode;
      NodeToCheckType : SPSymbols.SPSymbol;
   begin
      NodeToCheck := Child_Node (Node);
      NodeToCheckType := SyntaxNodeType (NodeToCheck);

      if NodeToCheckType = SPSymbols.object_declaration then
         NodeToCheck := Child_Node (NodeToCheck);
         if SyntaxNodeType (NodeToCheck) =
            SPSymbols.variable_declaration
         then
            -- In the case of a basic variable declaration the scope of the
            -- enclosing program unit and the scope of the declaration are
            -- the same, hence CurrentScope is used for both parameters.
            wf_variable_declaration (NodeToCheck,
                                     CurrentScope,
                                     CurrentScope);
         else -- must be a constant
            wf_constant_declaration (NodeToCheck,
                                     CurrentScope);
         end if;

      elsif NodeToCheckType = SPSymbols.full_type_declaration then
         wf_full_type_declaration (NodeToCheck,
                                   CurrentScope);

      else -- must be subtype
         wf_subtype_declaration (NodeToCheck,
                                 CurrentScope);

      end if;
   end wf_basic_declaration;

   --------------------------------------------------------------------------

   procedure wf_representation_clause (Node         : in STree.SyntaxNode;
                                       CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      Location    : Dictionary.Location;
      IdentNode   : STree.SyntaxNode;
      SubjectStr  : LexTokenManager.Lex_String;
      SubjectSym  : Dictionary.Symbol;
      RepKind     : SPSymbols.SPSymbol;

      procedure ProcessAddressClause
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in     SubjectStr;
      --#        in     SubjectSym;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict           from *,
      --#                                        SubjectSym &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubjectStr,
      --#                                        SubjectSym;
      is
      begin
         if Dictionary.IsVariable (SubjectSym) then
            -- It's a variable...OK unless it's a mode-less own variable
            Dictionary.AddVariableAddressClause (SubjectSym);
            if Dictionary.GetOwnVariableOrConstituentMode (SubjectSym) =
               Dictionary.DefaultMode then
               ErrorHandler.SemanticWarning (396,
                                             NodePosition (Node),
                                             SubjectStr);
            end if;
         elsif Dictionary.IsConstant (SubjectSym) then
            -- A constant...issue a warning
            ErrorHandler.SemanticWarning (351,
                                          NodePosition (Node),
                                          SubjectStr);
         elsif not Dictionary.IsProgramUnit (SubjectSym) then
            -- if it's not a variable, not a constant, and not a program
            -- unit, then it's illegal.
            ErrorHandler.SemanticError (255,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);

         end if;
      end ProcessAddressClause;

      procedure ProcessSizeClause
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     CurrentScope;
      --#        in     Node;
      --#        in     SubjectStr;
      --#        in     SubjectSym;
      --#        in out AggregateStack.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives AggregateStack.State,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        SubjectSym &
      --#         SLI.State                 from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.FileHeap,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.File_Sys,
      --#                                        STree.Table,
      --#                                        SubjectSym &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.FileHeap,
      --#                                        ContextManager.Ops.UnitHeap,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SLI.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubjectStr,
      --#                                        SubjectSym;
      is
         UnwantedSeq         : SeqAlgebra.Seq;
         UnusedComponentData : ComponentManager.ComponentData;
         SimpleExpression    : ExpRecord;
         IdentNode, SimpNode : STree.SyntaxNode;
         ValueInt            : Integer;
         Error               : Maths.ErrorCode;
         LexValue            : LexTokenManager.Lex_String;
      begin
         if Dictionary.IsType (SubjectSym) then
            Heap.Initialize (TheHeap);
            SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
            ComponentManager.Initialise (UnusedComponentData);
            -- IdentNode is the attribute definition clause
            IdentNode := Child_Node (Node);
            -- SimpNode is simple expression
            SimpNode := Next_Sibling (Child_Node (IdentNode));
            if SyntaxNodeType (SimpNode) = SPSymbols.simple_expression then
               --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
               WalkExpression (SimpNode,
                               CurrentScope,
                               Dictionary.GetUniversalIntegerType,
                               True, -- Static expression expected
                               -- to get
                               SimpleExpression,
                               UnwantedSeq,
                               UnusedComponentData);
               --# end accept;
               SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
               if SimpleExpression.IsStatic then
                  if Dictionary.IsIntegerTypeMark (SimpleExpression.TypeSymbol,
                                                   CurrentScope) then
                     if SimpleExpression.Value /= Maths.NoValue then
                        Maths.ValueToInteger (SimpleExpression.Value, ValueInt, Error);
                        if Error = Maths.NoError then
                           -- Size must not be negative
                           if ValueInt >= 0 then
                              if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => LexTokenManager.Null_String,
                                                                                      Lex_Str2 => Dictionary.TypeSizeAttribute (SubjectSym)) = LexTokenManager.Str_Eq then
                                 Maths.StorageRep (SimpleExpression.Value, LexValue);
                                 Dictionary.AddTypeSizeAttribute (SubjectSym, LexValue);
                                 ErrorHandler.RepresentationClause (NodePosition (Node));
                              else
                                 ErrorHandler.SemanticError (250,
                                                             ErrorHandler.NoReference,
                                                             NodePosition (SimpNode),
                                                             SubjectStr);
                              end if;
                           else
                              ErrorHandler.SemanticError (253,
                                                          ErrorHandler.NoReference,
                                                          NodePosition (SimpNode),
                                                          SubjectStr);
                           end if;
                        else
                           ErrorHandler.SemanticWarning  (200,
                                                          NodePosition (SimpNode),
                                                          LexTokenManager.Null_String);
                        end if;
                     else
                        ErrorHandler.SemanticWarning  (201,
                                                       NodePosition (SimpNode),
                                                       LexTokenManager.Null_String);
                     end if;
                  else
                     ErrorHandler.SemanticError (251,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (SimpNode),
                                                 SubjectStr);
                  end if;
               else
                  ErrorHandler.SemanticError (252,
                                              ErrorHandler.NoReference,
                                              NodePosition (SimpNode),
                                              SubjectStr);
               end if;
            else
               ErrorHandler.SemanticError (252,
                                           ErrorHandler.NoReference,
                                           NodePosition (SimpNode),
                                           SubjectStr);
            end if;
         elsif Dictionary.IsSubtype (SubjectSym) then
            -- Size for a non-first subtype is not permitted
            ErrorHandler.SemanticError (254,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        SubjectStr);
         else
            -- Anything else, just warn - legality is left to the compiler.
            ErrorHandler.RepresentationClause (NodePosition (Node));
         end if;

      end ProcessSizeClause;

      function AttributeIdentifier return LexTokenManager.Lex_String
      --# global in Node;
      --#        in STree.Table;
      -- pre SyntaxNodeType (DerivativeNodwe (Node)) = SPSymbols.attribute_definition_clause
      is
      begin
         return NodeLexString
            (Child_Node
             (Next_Sibling
              (Child_Node
               (Child_Node
                (Child_Node (Node))))));
      end AttributeIdentifier;

   begin --wf_representation_clause
      -- Currently checks only:
      -- (1) that the name of the type or object is declared and visible
      -- (2) if object is a variable and clause is an address then:
      --           (a) set HasAddressClause flag in Dictionary
      --           (b) issues warning if variable is NOT a stream
      -- (3) issues warning that rep clauses cannot be fully understood by Examiner
      -- (4) for enum and record reps calls Dict procedures to add locations to dict file

      Location := Dictionary.Location'(NodePosition (Node), NodePosition (Node));

      IdentNode := LastChildOf (Node);
      SubjectStr := NodeLexString (IdentNode);
      SubjectSym := Dictionary.LookupItem (Name    => SubjectStr,
                                           Scope   => CurrentScope,
                                           Context => Dictionary.ProgramContext);
      if SubjectSym = Dictionary.NullSymbol then
         ErrorHandler.SemanticError (1,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     SubjectStr);
         ErrorHandler.RepresentationClause (NodePosition (Node));
      else
         RepKind := SyntaxNodeType (Child_Node (Node));
         case RepKind is
            when SPSymbols.attribute_definition_clause =>
               if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttributeIdentifier,
                                                                       Lex_Str2 => LexTokenManager.Address_Token) = LexTokenManager.Str_Eq then
                  case CommandLineData.Content.LanguageProfile is
                     when CommandLineData.SPARK83 =>
                        ErrorHandler.SemanticError (54,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (Node),
                                                    LexTokenManager.Address_Token);

                     when CommandLineData.SPARK95 |
                       CommandLineData.SPARK2005 =>
                        STree.Set_Node_Lex_String (Sym  => SubjectSym,
                                                   Node => IdentNode);
                        ProcessAddressClause;
                  end case;

                  ErrorHandler.RepresentationClause (NodePosition (Node));
               elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttributeIdentifier,
                                                                          Lex_Str2 => LexTokenManager.Size_Token) = LexTokenManager.Str_Eq then
                  STree.Set_Node_Lex_String (Sym  => SubjectSym,
                                             Node => IdentNode);
                  ProcessSizeClause;
                  -- error handling done within ProcessSizeClause
               else
                  STree.Set_Node_Lex_String (Sym  => SubjectSym,
                                             Node => IdentNode);
                  ErrorHandler.RepresentationClause (NodePosition (Node));
               end if;
            when SPSymbols.at_clause =>
               case CommandLineData.Content.LanguageProfile is
                  when CommandLineData.SPARK83 =>
                     STree.Set_Node_Lex_String (Sym  => SubjectSym,
                                                Node => IdentNode);
                  when CommandLineData.SPARK95 |
                    CommandLineData.SPARK2005 =>
                     -- "at" clause obsolete in 95 and 2005
                     ErrorHandler.SemanticWarning (310,
                                                   NodePosition (Node),
                                                   LexTokenManager.Null_String);
               end case;
               ProcessAddressClause;
               ErrorHandler.RepresentationClause (NodePosition (Node));
            when SPSymbols.record_representation_clause =>
               if Dictionary.IsRecordTypeMark (SubjectSym, CurrentScope) then
                  STree.Set_Node_Lex_String (Sym  => SubjectSym,
                                             Node => IdentNode);
                  Dictionary.AddRecordRepresentationClause (SubjectSym,
                                                            Location);
               else
                  ErrorHandler.SemanticError (38,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              LexTokenManager.Null_String);
               end if;
               ErrorHandler.RepresentationClause (NodePosition (Node));
            when SPSymbols.enumeration_representation_clause =>
               if Dictionary.IsTypeMark (SubjectSym) and then
                 Dictionary.TypeIsEnumeration (SubjectSym) then
                  STree.Set_Node_Lex_String (Sym  => SubjectSym,
                                             Node => IdentNode);
                  Dictionary.AddEnumerationRepresentationClause (SubjectSym,
                                                                 Location);
               else
                  ErrorHandler.SemanticError (38,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              LexTokenManager.Null_String);
               end if;
               ErrorHandler.RepresentationClause (NodePosition (Node));
            when others => -- gramatically impossible
               null;
         end case;
      end if;
   end wf_representation_clause;

   --------------------------------------------------------------------------

   procedure wf_base_type_assertion (Node         : in STree.SyntaxNode;
                                     CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      NameNode,
      TypeNode,
      AttrNode,
      BaseTypeNode : STree.SyntaxNode;
      TypeStr,
      AttrStr,
      BaseTypeStr  : LexTokenManager.Lex_String;
      TypeSym,
      BaseTypeSym  : Dictionary.Symbol;
      Errors       : Boolean := False;

      procedure CkAttrIsBase
      --# global in     AttrNode;
      --#        in     AttrStr;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from AttrNode,
      --#                                        AttrStr,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Errors                    from *,
      --#                                        AttrStr,
      --#                                        LexTokenManager.State;
      is
      begin
         -- check that the attribute asserted is in fact 'Base
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttrStr,
                                                                 Lex_Str2 => LexTokenManager.Base_Token) /= LexTokenManager.Str_Eq then
            ErrorHandler.SemanticError (789,
                                        ErrorHandler.NoReference,
                                        NodePosition (AttrNode),
                                        LexTokenManager.Null_String);
            Errors := True;
         end if;
      end CkAttrIsBase;

      procedure CkNoExistingBaseType
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TypeNode;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        Errors,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeNode,
      --#                                        TypeSym &
      --#         Errors                    from *,
      --#                                        Dictionary.Dict,
      --#                                        TypeSym;
      is
      begin
         -- check that the assertion is not about a predefined type
         -- or one that already has a base type assertion; guarded by
         -- check on Errors to avoid bogus reports for bad types
         if not Errors and then
           Dictionary.IsType (TypeSym) and then
           Dictionary.GetBaseType (TypeSym) /= Dictionary.NullSymbol then
            ErrorHandler.SemanticError (796,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        LexTokenManager.Null_String);
            Errors := True;
         end if;
      end CkNoExistingBaseType;

      procedure CkIsType
      --# global in     BaseTypeNode;
      --#        in     BaseTypeSym;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TypeNode;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from BaseTypeNode,
      --#                                        BaseTypeSym,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeNode,
      --#                                        TypeSym &
      --#         Errors                    from *,
      --#                                        BaseTypeSym,
      --#                                        Dictionary.Dict,
      --#                                        TypeSym;
      is
      begin
         -- check that the type and base type specified are both actually types
         if not Dictionary.IsType (TypeSym) then
            ErrorHandler.SemanticError (790,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        LexTokenManager.Null_String);
            Errors := True;
         end if;
         if not Dictionary.IsType (BaseTypeSym) then
            ErrorHandler.SemanticError (790,
                                        ErrorHandler.NoReference,
                                        NodePosition (BaseTypeNode),
                                        LexTokenManager.Null_String);
            Errors := True;
         end if;
      end CkIsType;

      procedure CkPredefinedBaseType
      --# global in     BaseTypeNode;
      --#        in     BaseTypeSym;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from BaseTypeNode,
      --#                                        BaseTypeSym,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Errors                    from *,
      --#                                        BaseTypeSym,
      --#                                        Dictionary.Dict;
      is
      begin
         -- check that the base type specified is predefined
         if Dictionary.IsType (BaseTypeSym) then
            if Dictionary.GetBaseType (BaseTypeSym) /= BaseTypeSym then
               ErrorHandler.SemanticError (791,
                                           ErrorHandler.NoReference,
                                           NodePosition (BaseTypeNode),
                                           LexTokenManager.Null_String);
               Errors := True;
            end if;
         end if;
      end CkPredefinedBaseType;

      procedure CkTyping
      --# global in     BaseTypeNode;
      --#        in     BaseTypeSym;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from BaseTypeNode,
      --#                                        BaseTypeSym,
      --#                                        CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeSym &
      --#         Errors                    from *,
      --#                                        BaseTypeSym,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        TypeSym;
      is
      begin
         -- check that the types are either both signed integer, or both
         -- floating point.
         if not ((Dictionary.IsIntegerTypeMark (TypeSym, CurrentScope) and
                    Dictionary.IsIntegerTypeMark (BaseTypeSym, CurrentScope)) or else
                   (Dictionary.IsFloatingPointTypeMark (TypeSym, CurrentScope) and
                      Dictionary.IsFloatingPointTypeMark (BaseTypeSym, CurrentScope))) then
            ErrorHandler.SemanticError (792,
                                        ErrorHandler.NoReference,
                                        NodePosition (BaseTypeNode),
                                        LexTokenManager.Null_String);
            Errors := True;
         end if;
      end CkTyping;

      procedure CkBounding
      --# global in     BaseTypeNode;
      --#        in     BaseTypeSym;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from BaseTypeNode,
      --#                                        BaseTypeSym,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        Errors,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeSym &
      --#         Errors                    from *,
      --#                                        BaseTypeSym,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        TypeSym;
      is
         TypeFirst,
         TypeLast,
         BaseTypeFirst,
         BaseTypeLast  : LexTokenManager.Lex_String;
         TFVal,
         TLVal,
         BTFVal,
         BTLVal,
         CompVal       : Maths.Value;
         BoundsOK      : Boolean;
         MathsError1,
         MathsError2   : Maths.ErrorCode;
      begin
         -- check that there are defined bounds for the base type, and also that
         -- the range of the type fits within the range of the specified base type
         if not Errors and then
           Dictionary.IsTypeMark (TypeSym) and then
           Dictionary.IsTypeMark (BaseTypeSym) then
            TypeFirst     := Dictionary.GetScalarAttributeValue (False,
                                                                 LexTokenManager.First_Token,
                                                                 TypeSym);
            TypeLast      := Dictionary.GetScalarAttributeValue (False,
                                                                 LexTokenManager.Last_Token,
                                                                 TypeSym);
            BaseTypeFirst := Dictionary.GetScalarAttributeValue (False,
                                                                 LexTokenManager.First_Token,
                                                                 BaseTypeSym);
            BaseTypeLast  := Dictionary.GetScalarAttributeValue (False,
                                                                 LexTokenManager.Last_Token,
                                                                 BaseTypeSym);
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => BaseTypeFirst,
                                                                    Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq or else
              LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => BaseTypeLast,
                                                                   Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
               -- we require that the base type have defined bounds,
               ErrorHandler.SemanticError (793,
                                           ErrorHandler.NoReference,
                                           NodePosition (BaseTypeNode),
                                           LexTokenManager.Null_String);
               Errors := True;
            elsif Dictionary.TypeIsReal (TypeSym) and then
              (LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => TypeFirst,
                                                                    Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq or else
                 LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => TypeLast,
                                                                      Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq) then
               -- no check possible with real types with unconstrained ranges
               null;
            else
               -- check that the range of the base type is at least that of the type
               TFVal  := Maths.ValueRep (TypeFirst);
               TLVal  := Maths.ValueRep (TypeLast);
               BTFVal := Maths.ValueRep (BaseTypeFirst);
               BTLVal := Maths.ValueRep (BaseTypeLast);
               Maths.Lesser (TFVal, BTFVal, CompVal, MathsError1);
               BoundsOK := (CompVal = Maths.FalseValue);
               Maths.Greater (TLVal, BTLVal, CompVal, MathsError2);
               BoundsOK := BoundsOK and then (CompVal = Maths.FalseValue);
               if not BoundsOK and
                 (MathsError1 = Maths.NoError) and
                 (MathsError2 = Maths.NoError) then
                  ErrorHandler.SemanticError (794,
                                              ErrorHandler.NoReference,
                                              NodePosition (BaseTypeNode),
                                              LexTokenManager.Null_String);
                  Errors := True;
               end if;
            end if;
         end if;
      end CkBounding;

      procedure CkRealAccuracy
      --# global in     BaseTypeNode;
      --#        in     BaseTypeSym;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from BaseTypeNode,
      --#                                        BaseTypeSym,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        Errors,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeSym &
      --#         Errors                    from *,
      --#                                        BaseTypeSym,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        TypeSym;
      is
         TypeAccy,
         BaseTypeAccy    : LexTokenManager.Lex_String;
         TypeAccyVal,
         BaseTypeAccyVal,
         CompVal         : Maths.Value;
         MathsError      : Maths.ErrorCode;
      begin
         -- This test only relevant if the type is a real
         if not Errors and then
           Dictionary.TypeIsReal (TypeSym) then
            TypeAccy :=
              Dictionary.GetScalarAttributeValue (False,
                                                  LexTokenManager.Digits_Token,
                                                  TypeSym);
            BaseTypeAccy :=
              Dictionary.GetScalarAttributeValue (False,
                                                  LexTokenManager.Digits_Token,
                                                  BaseTypeSym);
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => BaseTypeAccy,
                                                                    Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
               -- we require that the base type have a defined accuracy
               ErrorHandler.SemanticError (797,
                                           ErrorHandler.NoReference,
                                           NodePosition (BaseTypeNode),
                                           LexTokenManager.Null_String);
               Errors := True;
            else
               -- check that the accuracy of the base type is at least that of the type
               TypeAccyVal := Maths.ValueRep (TypeAccy);
               BaseTypeAccyVal := Maths.ValueRep (BaseTypeAccy);
               Maths.Lesser (BaseTypeAccyVal, TypeAccyVal, CompVal, MathsError);
               if (CompVal = Maths.TrueValue) and
                 (MathsError = Maths.NoError) then
                  ErrorHandler.SemanticError (798,
                                              ErrorHandler.NoReference,
                                              NodePosition (BaseTypeNode),
                                              LexTokenManager.Null_String);
                  Errors := True;
               end if;
            end if;
         end if;
      end CkRealAccuracy;

      procedure CkScoping
      --# global in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TypeNode;
      --#        in     TypeSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Errors;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        Errors,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TypeNode,
      --#                                        TypeSym &
      --#         Errors                    from *,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        TypeSym;
      is
         TypeScope : Dictionary.Scopes;
      begin
         -- check that the type assertion occurs in the same scope
         -- as the declaration of the type, or the corresponding
         -- private scope (if the type is private)
         if not Errors then
            TypeScope := Dictionary.GetScope (TypeSym);
            if not (TypeScope = CurrentScope or
                      (Dictionary.PrivateScope (Dictionary.GetRegion (TypeScope)) =
                         CurrentScope and Dictionary.TypeIsPrivate (TypeSym))) then
               ErrorHandler.SemanticError (795,
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.Null_String);
               Errors := True;
            end if;
         end if;
      end CkScoping;

   begin
      NameNode := Child_Node (Node);
      SystemErrors.RTAssert (SyntaxNodeType (NameNode) = SPSymbols.dotted_simple_name,
                             SystemErrors.AssertionFailure,
                             "First node must be a name");
      TypeNode := Child_Node (NameNode);
      if SyntaxNodeType (TypeNode) /= SPSymbols.identifier then
         ErrorHandler.SemanticError (799,
                                     ErrorHandler.NoReference,
                                     NodePosition (TypeNode),
                                     LexTokenManager.Null_String);
      else
         AttrNode     := Next_Sibling (NameNode);
         BaseTypeNode := Next_Sibling (AttrNode);
         TypeStr      := NodeLexString (TypeNode);
         AttrStr      := NodeLexString (AttrNode);
         BaseTypeStr  := NodeLexString (BaseTypeNode);
         TypeSym      := Dictionary.LookupItem (Name    => TypeStr,
                                                Scope   => CurrentScope,
                                                Context => Dictionary.ProgramContext);
         BaseTypeSym  := Dictionary.LookupItem (Name    => BaseTypeStr,
                                                Scope   => CurrentScope,
                                                Context => Dictionary.ProgramContext);

         -- perform static semantic checks
         CkAttrIsBase;
         CkIsType;
         CkTyping;
         CkNoExistingBaseType;
         CkPredefinedBaseType;
         CkBounding;
         CkRealAccuracy;
         CkScoping;

         if not Errors then
            STree.Set_Node_Lex_String (Sym  => TypeSym,
                                       Node => TypeNode);
            STree.Set_Node_Lex_String (Sym  => BaseTypeSym,
                                       Node => BaseTypeNode);
            Dictionary.SetBaseType (TypeSym, BaseTypeSym);
         end if;
      end if;
   end wf_base_type_assertion;

   --------------------------------------------------------------------------

   --------------------------------------------------------------------------
   -- Process a clause of the form:
   --    --# assert A'Always_Valid;
   -- or --# assert A.B.C'Always_Valid;
   -- Where A is a variable, and B anc C are (sub) components.
   -- If successful, mark the Variable/SubComponent symbol as "MarkedValid"
   -- which will suppress the "representation might be invalid" warning,
   -- and cause the appropriate "in range" VCs to be generated.
   --------------------------------------------------------------------------
   procedure Wf_Always_Valid_Variable_Assertion
     (Node         : in STree.SyntaxNode;
      CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      NameNode   : STree.SyntaxNode;
      EntireSym,
      ComponentSym,
      TypeSym       : Dictionary.Symbol;

      procedure ProcessDottedName (NameNode     : in STree.SyntaxNode;
                                   EntireSym    : out Dictionary.Symbol;
                                   ComponentSym : out Dictionary.Symbol)
      -- NameNode     - Node naming a variable, or a (sub) field of a variable
      -- EntireSym    - Symbol of entire variable (ie, first identifer found)
      -- ComponentSym - Symbol of right most element (ie, the variable or
      --                subcomponent to be marked)
      --                == NullSymbol if anything goes wrong

      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     CurrentScope;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out GlobalComponentData;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives ComponentSym              from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        GlobalComponentData,
      --#                                        LexTokenManager.State,
      --#                                        NameNode,
      --#                                        STree.Table,
      --#                                        TheHeap &
      --#         Dictionary.Dict,
      --#         GlobalComponentData,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        GlobalComponentData,
      --#                                        LexTokenManager.State,
      --#                                        NameNode,
      --#                                        STree.Table,
      --#                                        TheHeap &
      --#         EntireSym                 from CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        NameNode,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GlobalComponentData,
      --#                                        LexTokenManager.State,
      --#                                        NameNode,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheHeap;
      is
         It              : STree.Iterator;
         IdNode          : STree.SyntaxNode;
         IdStr, PIdStr   : LexTokenManager.Lex_String;
         LocalSym        : Dictionary.Symbol;
      begin
         -- the first node must be the entire variable
         PIdStr := LexTokenManager.Null_String;

         It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                              FromRoot    => NameNode,
                              InDirection => STree.Down);

         IdNode := GetNode (It);    -- must be the entire variable
         IdStr := NodeLexString (IdNode);
         LocalSym := Dictionary.LookupItem (Name    => IdStr,
                                            Scope   => CurrentScope,
                                            Context => Dictionary.ProofContext);
         EntireSym := LocalSym;

         loop
            -- check that what we have so far (inc the first symbol) is valid
            if LocalSym = Dictionary.NullSymbol then
               ErrorHandler.SemanticError2 (1,
                                            ErrorHandler.NoReference,
                                            NodePosition (IdNode),
                                            IdStr,
                                            PIdStr);
               LocalSym := Dictionary.NullSymbol;
               exit;
            end if;
            STree.Set_Node_Lex_String (Sym  => LocalSym,
                                       Node => IdNode);
            -- get the next symbol, if there is one
            It := STree.NextNode (It);
            exit when STree.IsNull (It);

            IdNode := GetNode (It);
            PIdStr := IdStr;
            IdStr := NodeLexString (IdNode);
            --Debug.PrintLexStr ("Node = ", IdStr);

            -- we have another symbol (field selector), thus the previous
            -- one must be the entire variable, or a subcomponent, and it
            -- must be of a record type

            if not (Dictionary.IsVariableOrSubcomponent (LocalSym) and then
                    Dictionary.TypeIsRecord (Dictionary.GetType (LocalSym))) then
               ErrorHandler.SemanticErrorSym (9,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdNode),
                                              LocalSym,
                                              CurrentScope);
               LocalSym := Dictionary.NullSymbol;
               exit;
            end if;

            -- We have a symbol that might be a subcomponent, before we can
            -- check that it is, we need to add the subcomponent to the enclosing
            -- record symbol, since they are not added unless/until they are
            -- needed
            --Debug.PrintMsg ("Adding subcomponents in CkDotted", True);
            AddRecordSubComponents (RecordVarSym  => LocalSym,
                                    RecordTypeSym => Dictionary.GetType (LocalSym),
                                    ComponentData => GlobalComponentData);

            -- check that the symbol is indeed a subcomponent
            LocalSym := Dictionary.LookupSelectedItem (Prefix   => LocalSym,
                                                       Selector => IdStr,
                                                       Scope    => CurrentScope,
                                                       Context  => Dictionary.ProofContext);

         end loop;
         -- return the innermost component symbol, since this is what
         -- will carry the valid mark
         ComponentSym := LocalSym;
      end ProcessDottedName;

   begin
      -- local grammar (rooted at Node)
      --
      --
      -- Node = dotted_simple_name --- 'Always_Valid --- ";"
      --              |                                 |
      --      dotted_simple_name --- identifier
      --              |
      --               ...
      --           identifier
      NameNode := Child_Node (Node);

      -- check that the attribute asserted is in fact 'Always_Valid
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (Next_Sibling (NameNode)),
                                                              Lex_Str2 => LexTokenManager.Always_Valid_Token) /= LexTokenManager.Str_Eq then
         ErrorHandler.SemanticError (656,
                                     ErrorHandler.NoReference,
                                     NodePosition (Next_Sibling (NameNode)),
                                     LexTokenManager.Null_String);
      else
         -- parse the variable name, which might refer to a record
         -- subcomponent: eg. EntireSym.First.Second.ComponentSym
         -- ComponentSym == NullSymbol if anything goes wrong
         ProcessDottedName (NameNode, EntireSym, ComponentSym);

         -- Check that we are referring ultimately to a variable
         if ComponentSym /= Dictionary.NullSymbol and then
               not Dictionary.IsVariable (EntireSym) then
            ErrorHandler.SemanticError (657,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            ComponentSym := Dictionary.NullSymbol;
         end if;

         -- Check that the variable is declared in the same scope
         if ComponentSym /= Dictionary.NullSymbol and then
               Dictionary.GetScope (EntireSym) /= CurrentScope then
            ErrorHandler.SemanticError (659,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            ComponentSym := Dictionary.NullSymbol;
         end if;

         -- Check that the type of the component is appropriate
         if ComponentSym /= Dictionary.NullSymbol then
            TypeSym := Dictionary.GetType (ComponentSym);
            if not Dictionary.IsScalarTypeMark (TypeSym, CurrentScope) then
               ErrorHandler.SemanticError (658,
                                           ErrorHandler.NoReference,
                                           NodePosition (NameNode),
                                           LexTokenManager.Null_String);
               ComponentSym := Dictionary.NullSymbol;
            end if;
         end if;

         -- Should check that it is a mode in own variable...
         if ComponentSym /= Dictionary.NullSymbol and then
               Dictionary.GetOwnVariableOrConstituentMode (EntireSym) /=
                  Dictionary.InMode then
            ErrorHandler.SemanticError (662,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
            ComponentSym := Dictionary.NullSymbol;
         end if;

         -- Check that this component has not been marked valid already
         -- and if not, then mark it. The Symbol might be a variable,
         -- or a subcomponent, and they must be dealt with separately
         -- because they are different types
         if ComponentSym /= Dictionary.NullSymbol then
            if Dictionary.IsVariable (ComponentSym) then
               if not Dictionary.VariableIsMarkedValid (ComponentSym) then
                  --Debug.PrintSym ("Marking Variable Symbol valid = ", ComponentSym);
                  Dictionary.SetVariableMarkedValid (ComponentSym, True);
               else
                  ErrorHandler.SemanticError (660,
                                              ErrorHandler.NoReference,
                                              NodePosition (NameNode),
                                              LexTokenManager.Null_String);
               end if;
            elsif Dictionary.IsRecordSubcomponent (ComponentSym) then
               --Debug.PrintSym ("Marking Subcomponent Symbol valid = ", ComponentSym);
               if not Dictionary.SubcomponentIsMarkedValid (ComponentSym) then
                  Dictionary.SetSubcomponentMarkedValid (ComponentSym, True);
               else
                  ErrorHandler.SemanticError (660,
                                              ErrorHandler.NoReference,
                                              NodePosition (NameNode),
                                              LexTokenManager.Null_String);
               end if;
            else
               SystemErrors.RTAssert (False,
                         SystemErrors.AssertionFailure,
                         "Component must be a variable or a SubComponent");
            end if;
         end if;
      end if;
   end Wf_Always_Valid_Variable_Assertion;

   --------------------------------------------------------------------------

   procedure wf_object_assertion (Node         : in STree.SyntaxNode;
                                  CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table               from CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        STree.Table;
   is
      IdentNode   : STree.SyntaxNode;
      IdentStr    : LexTokenManager.Lex_String;

      procedure ProcessSimpleNameRep (Node         : in STree.SyntaxNode;
                                      CurrentScope : in Dictionary.Scopes;
                                      RulePolicy   : in Dictionary.RulePolicies)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         STree.Table               from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        RulePolicy,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        ContextManager.Ops.UnitStack,
      --#                                        CurrentScope,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        RulePolicy,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
         It          : STree.Iterator;

         procedure ProcessDottedSimpleName (Node : in STree.SyntaxNode)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.UnitStack;
         --#        in     CurrentScope;
         --#        in     LexTokenManager.State;
         --#        in     RulePolicy;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --#        in out STree.Table;
         --# derives Dictionary.Dict           from *,
         --#                                        CommandLineData.Content,
         --#                                        ContextManager.Ops.UnitStack,
         --#                                        CurrentScope,
         --#                                        LexTokenManager.State,
         --#                                        Node,
         --#                                        RulePolicy,
         --#                                        STree.Table &
         --#         STree.Table               from *,
         --#                                        CommandLineData.Content,
         --#                                        CurrentScope,
         --#                                        Dictionary.Dict,
         --#                                        LexTokenManager.State,
         --#                                        Node &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
         --#                                        CurrentScope,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        Node,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table;
         is
            IdNode,
            NextIdNode : STree.SyntaxNode;
            It         : STree.Iterator;
            PIdStr,
            IdStr      : LexTokenManager.Lex_String;
            Sym,
            SymSoFar   : Dictionary.Symbol;
            SymType    : Dictionary.Symbol;
            PrefixOk   : Boolean;
         begin
            It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                                 FromRoot    => Node,
                                 InDirection => STree.Down);

            IdNode := GetNode (It);
            IdStr := NodeLexString (IdNode);
            PIdStr := LexTokenManager.Null_String;
            Sym := Dictionary.LookupItem (Name    => IdStr,
                                          Scope   => CurrentScope,
                                          Context => Dictionary.ProofContext);

            loop
               if Sym = Dictionary.NullSymbol then
                  ErrorHandler.SemanticError2 (1,
                                               ErrorHandler.NoReference,
                                               NodePosition (Node),
                                               IdStr,
                                               PIdStr);
                  exit;
               end if;

               It := STree.NextNode (It);
               NextIdNode := GetNode (It);

               if Dictionary.IsConstant (Sym) then
                  SymType := Dictionary.GetType (Sym);
                  -- is constant, needs to be a composite constant
                  -- with no dotted part to right.
                  if NextIdNode /= STree.NullNode then

                     ErrorHandler.SemanticError (180, --composite constant expected
                                                 ErrorHandler.NoReference,
                                                 NodePosition (Node),
                                                 LexTokenManager.Null_String);
                     Sym := Dictionary.NullSymbol;
                  end if;

                  if not (Dictionary.TypeIsArray (SymType) or
                            Dictionary.TypeIsRecord (SymType)) then

                     ErrorHandler.SemanticError (180, --composite constant expected
                                                 ErrorHandler.NoReference,
                                                 NodePosition (Node),
                                                 LexTokenManager.Null_String);
                     Sym := Dictionary.NullSymbol;
                  end if;
                  if Sym /= Dictionary.NullSymbol then
                     STree.Set_Node_Lex_String (Sym  => Sym,
                                                Node => IdNode);
                  end if;
                  exit;
               end if;

               if not Dictionary.IsPackage (Sym) then
                  ErrorHandler.SemanticError (180, --composite constant expected
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);
                  Sym := Dictionary.NullSymbol;
                  exit;
               end if;

               if NextIdNode = STree.NullNode then
                  -- package without a selected component
                  ErrorHandler.SemanticError (180, --composite constant expected
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.Null_String);
                  Sym := Dictionary.NullSymbol;
                  exit;
               end if;

               CheckPackagePrefix (IdNode,
                                   Sym,
                                   CurrentScope,
                                   PrefixOk);
               if not PrefixOk then
                  Sym := Dictionary.NullSymbol;
                  exit;
               end if;
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => IdNode);
               PIdStr := IdStr;
               IdNode := NextIdNode;
               IdStr := NodeLexString (IdNode);
               SymSoFar := Sym;
               Sym := Dictionary.LookupSelectedItem (Prefix   => Sym,
                                                     Selector => IdStr,
                                                     Scope    => CurrentScope,
                                                     Context  => Dictionary.ProofContext);
               -- check to see if we are getting the same symbol over and again
               if Sym = SymSoFar then            -- P.P.P.P.X case
                  Sym := Dictionary.NullSymbol;  -- to cause "Not visible" error at top of loop
               end if;

            end loop;

            if Sym /= Dictionary.NullSymbol then
               if Dictionary.IsConstantRulePolicyPresent (Sym, CurrentScope) then
                  ErrorHandler.SemanticError2 (182, -- rule policy already present.
                                               ErrorHandler.NoReference,
                                               NodePosition (Node),
                                               IdStr,
                                               PIdStr);
               else
                  Dictionary.AddConstantRulePolicy (TheConstant => Sym,
                                                    Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                    Declaration => Dictionary.Location'(NodePosition (IdNode),
                                                                                        NodePosition (IdNode)),
                                                    TheScope    => CurrentScope,
                                                    ThePolicy   => RulePolicy);
               end if;
            end if;

         end ProcessDottedSimpleName;


      begin
         It := FindFirstNode (NodeKind    => SPSymbols.dotted_simple_name,
                              FromRoot    => Node,
                              InDirection => STree.Down);

         while not STree.IsNull (It) loop --for each identifier in list of constants
            ProcessDottedSimpleName (GetNode (It));
            It := STree.NextNode (It);

         end loop;

      end ProcessSimpleNameRep;

   begin

      IdentNode := Next_Sibling (Child_Node (Node));
      IdentStr := NodeLexString (IdentNode);
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                              Lex_Str2 => LexTokenManager.Rule_Token) = LexTokenManager.Str_Eq then
         ProcessSimpleNameRep (Child_Node (Node),
                               CurrentScope,
                               Dictionary.RuleRequested);
      elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                                 Lex_Str2 => LexTokenManager.No_Rule_Token) = LexTokenManager.Str_Eq then
         ProcessSimpleNameRep (Child_Node (Node),
                               CurrentScope,
                               Dictionary.NoRuleRequested);
      else
         -- Illegal proof rule switch

         ErrorHandler.SemanticError (181,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     LexTokenManager.Null_String);
      end if;

      if ErrorHandler.Generate_SLI then
         SLI.Generate_Xref_Object_Assertion (Comp_Unit  => ContextManager.Ops.CurrentUnit,
                                             Parse_Tree => Node,
                                             Scope      => CurrentScope);
      end if;

   end wf_object_assertion;
   --------------------------------------------------------------------------

   procedure wf_basic_proof_declaration (Node         : in STree.SyntaxNode;
                                         CurrentScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        SLI.State,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      DeclarationNode,
      IdentNode       : STree.SyntaxNode;
      IdentStr        : LexTokenManager.Lex_String;
      Sym             : Dictionary.Symbol;

      function MoreThanOneOwnVarAnnounced (Sym   : Dictionary.Symbol;
                                           Scope : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      is
         It            : Dictionary.Iterator;
         Count         : Natural := 0;
         CurrentOwnVar : Dictionary.Symbol;
      begin
         It := Dictionary.FirstOwnVariable (Dictionary.GetEnclosingPackage (Scope));
         while not Dictionary.IsNullIterator (It)
         loop
            CurrentOwnVar := Dictionary.CurrentSymbol (It);
            if Dictionary.GetType (CurrentOwnVar) = Sym then
               Count := Count + 1;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
         return Count > 1;
      end MoreThanOneOwnVarAnnounced;

      ---------------------------

   begin -- wf_basic_proof_declaration

      DeclarationNode := Child_Node (Node);
      case SyntaxNodeType (DeclarationNode) is
         when SPSymbols.proof_type_declaration =>
            IdentNode := Child_Node (DeclarationNode);
            IdentStr := NodeLexString (IdentNode);
            Sym := Dictionary.LookupItem (Name    => IdentStr,
                                          Scope   => CurrentScope,
                                          Context => Dictionary.ProofContext);
            if Sym = Dictionary.NullSymbol or else
              (Dictionary.IsTypeMark (Sym) and then
                 Dictionary.TypeIsAnnounced (Sym) and then
                 not Dictionary.IsDeclared (Sym)) then
               --it's ok to add because it either a new type or an announced type.
               --unless more than one own variable has announced it
               if MoreThanOneOwnVarAnnounced (Sym, CurrentScope) then
                  ErrorHandler.SemanticError (149,  ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              IdentStr);

               else -- still ok to add
                  -- Assumption warning: code assumes that proof type is "abstract" because
                  -- that is all the grammar would permit at this point.  If we add
                  -- fdl-based proof types then more checks will be needed here.
                  if Sym /= Dictionary.NullSymbol then
                     STree.Set_Node_Lex_String (Sym  => Sym,
                                                Node => IdentNode);
                  end if;
                  Dictionary.AddAbstractProofType (Name        => IdentStr,
                                                   Comp_Unit   => ContextManager.Ops.CurrentUnit,
                                                   Declaration => Dictionary.Location'(NodePosition (IdentNode),
                                                                                       NodePosition (IdentNode)),
                                                   Scope       => CurrentScope,
                                                   --to get
                                                   TypeSym     => Sym);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol (Comp_Unit      => ContextManager.Ops.CurrentUnit,
                                               Parse_Tree     => IdentNode,
                                               Symbol         => Sym,
                                               Is_Declaration => True);
                  end if;
               end if;
            else --illegal duplicate
               ErrorHandler.SemanticError (10,  ErrorHandler.NoReference,
                                           NodePosition (IdentNode),
                                           IdentStr);
            end if;
         when SPSymbols.type_assertion =>
            DeclarationNode := Child_Node (DeclarationNode);
            if SyntaxNodeType (DeclarationNode) =
              SPSymbols.base_type_assertion then
               wf_base_type_assertion (DeclarationNode, CurrentScope);
            elsif SyntaxNodeType (DeclarationNode) =
              SPSymbols.alwaysvalid_variable_assertion then
               Wf_Always_Valid_Variable_Assertion (DeclarationNode, CurrentScope);
            else
               ErrorHandler.SemanticError (315,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.Null_String);
            end if;
         when SPSymbols.object_assertion =>
            wf_object_assertion (DeclarationNode, CurrentScope);

         when others =>
            -- currently only proof_constant_declaration
            ErrorHandler.SemanticError (315,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
      end case;
   end wf_basic_proof_declaration;

   --------------------------------------------------------------------------

begin
   --assume node = basic_declaritive_item

   NodeToCheck := Child_Node (Node);
   NodeToCheckType := SyntaxNodeType (NodeToCheck);

   if NodeToCheckType = SPSymbols.basic_declaration then
      wf_basic_declaration (NodeToCheck,
                            CurrentScope);
   elsif NodeToCheckType = SPSymbols.representation_clause then
      wf_representation_clause (NodeToCheck,
                                CurrentScope);
   elsif NodeToCheckType = SPSymbols.basic_proof_declaration then
      wf_basic_proof_declaration (NodeToCheck,
                                  CurrentScope);
   elsif NodeToCheckType = SPSymbols.justification_statement then
      wf_justification_statement (NodeToCheck,
                                  CurrentScope); -- PNA ---------------------------------
   end if;
end wf_basic_declarative_item;
