-- $Id: sem-compunit-walkstatements-wf_proc_call.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- This procedure checks the following :
--
-- 1. The parameter associations defined by a procedure call form a bijective
--    function from formal to actual parameters.
-- 2. Every actual parameter corresponding to an exported formal parameter is
--    an entire variable (Section 6.4 Rule 1).
-- 3. Every actual parameter of mode in out or out is the name of a variable.
--    (LRM 6.4.1).
-- 4. Every actual parameter of mode in out or out is not
--       -  a formal parameter of mode in (LRM 6.2), or
--       -  a loop parameter (LRM 5.5), or
-- 5. An actual parameter of mode in out is not a formal parameter of mode out,
--    or a subcomponent thereof (LRM 6.4.1).
-- 6. A variable named in the global definition of a procedure P cannot occur as
--    an actual parameter of P whose corresponding formal parameter is an
--    exported variable (Section 6.4 Rule 2).
-- 7. If a variable V named in the global definition of a procedure P is export
--    then neither V nor any of its subcomponents can occur as an actual
--    statement, and the corresponding formal parameter is exported, then
--    neither V nor any of its subcomponents can occur as another actual
--    parameter in that statement (Section 6.4 Rule 4).
-- 8. The types of Formal and Actual parameters are compatible.
-- 9. The procedure is callable at the point of call.
--10. Globals imported or exported by a procedure are visible at the point of
--    call.
--11. Procedure calls resulting in direct update of non-enclosing package
--    own variable generate a warning.
--12. After substitution of actuals-for-formals, the resulting information
--    flow relation is checked against the selected information flow policy.
--------------------------------------------------------------------------------
separate (Sem.CompUnit.WalkStatements)
procedure wf_proc_call (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
is
   ProcSym               : Dictionary.Symbol;
   ArgListNode,
   NameNode              : STree.SyntaxNode;
   Dotted,
   Valid                 : Boolean;
   ActualParameterList   : SimpleLists.SimpleList;
   FormalPositionList    : SimpleLists.SimpleList;
   Abstraction           : Dictionary.Abstractions
      := Dictionary.IsAbstract;  -- ineffective, but removes spurious flowerrs

   subtype ParamIndex  is Integer range
     1 .. ExaminerConstants.MaxProcCallParams;

   -- A ParamInfo stores the sequence of items referenced in an actual parameter.  For an IN
   -- parameter it will be just the variables for that appear in the general expression
   -- that makes up the actual.  For OUT and IN OUT cases, we normally would expect a singleton
   -- element in the sequence because the actual must be a variable (note in this context that a
   -- record field is still a single variable because each field has its own symbol).  For all the
   -- above cases, the Seq field contains the only useful data and the EntireArraySym is null.
   -- The special case occurs whn the actual parameter is an array element such as A(I).  Here the
   -- Seq field will contain both A and I (in a non-deterministic order); however, we need to know that
   -- the parameter is an array element so we can correct the flow analysis in BuilIORelation.  In this
   -- case, the EntireArraySym will be set to A.
   type ParamInfo is record
      Seq            : Natural;
      EntireArraySym : Dictionary.Symbol;
   end record;
   type ParamTables is array (ParamIndex) of ParamInfo;
   ParamTable            : ParamTables;

   -- The following variable gets set to the type that an actual parameter
   -- must be in a call to an inherited operation
   TaggedParameterSym : Dictionary.Symbol;
   -- and this is the type of the formal of the inherited operation
   RootParameterSym   : Dictionary.Symbol;
   -- we need global access to the PrefixSymbol (which gets set by DoDottedName)
   -- so that we can deal with "self substitutions" in PO calls.
   PrefixSymbol : Dictionary.Symbol;

   ---------------------------------------------------------------

   function TaggedSubstitutionRequired (Formal,
                                        ActualType,
                                        RootParameterSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.GetType (Formal) = RootParameterSym and then
        Dictionary.IsAnExtensionOf (RootParameterSym, ActualType);
   end TaggedSubstitutionRequired;

   ---------------------------------------------------------------

   procedure SubstituteRootTypeForExtendedTypeInSequence
     (TheSeq             : in     SeqAlgebra.Seq;
      TaggedParameterSym : in     Dictionary.Symbol;
      ComponentData      : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in out Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    ComponentData,
   --#                                    ContextManager.Ops.UnitStack,
   --#                                    Dictionary.Dict,
   --#                                    TaggedParameterSym,
   --#                                    TheHeap,
   --#                                    TheSeq &
   --#         null                  from CommandLineData.Content;
   is
      ActualSym   : Dictionary.Symbol;
      MemberOfSeq : SeqAlgebra.MemberOfSeq;
   begin
      MemberOfSeq := SeqAlgebra.FirstMember (TheHeap, TheSeq);
      while not SeqAlgebra.IsNullMember (MemberOfSeq) loop
         -- process each member of sequence in turn
         ActualSym := Dictionary.ConvertSymbolRef
           (ExaminerConstants.RefType
              (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                           M        => MemberOfSeq)));
         -- we only need to do things to tagged types that extend TaggedParameterSym
         if Dictionary.IsAnExtensionOf (TaggedParameterSym,
                                        Dictionary.GetType (ActualSym)) then
            -- We can't replace X with X.Inherit unless we add X's subcomponents first
            AddRecordSubComponents (RecordVarSym  => ActualSym,
                                    RecordTypeSym => Dictionary.GetType (ActualSym),
                                    ComponentData => ComponentData);
            -- now add as may .Inherits as needed to get type match
            ActualSym := ConvertTaggedActual (ActualSym, TaggedParameterSym);
            -- replace existing sequence member with converted one
            SeqAlgebra.RemoveMember (TheHeap,
                                     TheSeq,
                                     SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                 M        => MemberOfSeq));
            SeqAlgebra.AddMember (TheHeap,
                                  TheSeq,
                                  Natural (Dictionary.SymbolRef (ActualSym)));
         end if;
         MemberOfSeq := SeqAlgebra.NextMember (TheHeap, MemberOfSeq);
      end loop;
   end SubstituteRootTypeForExtendedTypeInSequence;

   ---------------------------------------------------------------

   procedure AddSeq (ParamNo        : in Natural;
                     Refs           : in SeqAlgebra.Seq;
                     EntireArraySym : in Dictionary.Symbol)
   -- see comment on type ParamInfo for explanation of parameter EntireArraySym
   --# global in out ParamTable;
   --# derives ParamTable from *,
   --#                         EntireArraySym,
   --#                         ParamNo,
   --#                         Refs;
   is
   begin
      if ParamNo > ParamIndex'Last then
         SystemErrors.FatalError (SystemErrors.TooManyParamsInProcedureCall, "");
      end if;
      ParamTable (ParamNo).Seq := SeqAlgebra.SeqToNatural (Refs);
      ParamTable (ParamNo).EntireArraySym := EntireArraySym;
   end AddSeq;

   ---------------------------------------------------------------

   procedure BuildIORelation
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     ParamTable;
   --#        in     PrefixSymbol;
   --#        in     ProcSym;
   --#        in     RootParameterSym;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                   from *,
   --#                                        Abstraction,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        Node,
   --#                                        ParamTable,
   --#                                        PrefixSymbol,
   --#                                        ProcSym,
   --#                                        RootParameterSym,
   --#                                        Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ParamTable,
   --#                                        PrefixSymbol,
   --#                                        ProcSym,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Table,
   --#                                        TheHeap;
   is
      ExportIt,
      ImportIt  : Dictionary.Iterator;
      ImportSeq : SeqAlgebra.Seq;
      ExportSym,
      ImportSym : Dictionary.Symbol;
      ParamNo   : Positive;
      ActualSym : Dictionary.Symbol;

      procedure CheckRelationInfoFlowPolicy (TheExport  : in Dictionary.Symbol;
                                             TheImports : in SeqAlgebra.Seq)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     TheHeap;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheExport,
      --#                                        TheHeap,
      --#                                        TheImports;
      is
         CurrentMember : SeqAlgebra.MemberOfSeq;
         TheImportSym  : Dictionary.Symbol;
      begin
         CurrentMember := SeqAlgebra.FirstMember (TheHeap, TheImports);

         while not SeqAlgebra.IsNullMember (CurrentMember) loop

            TheImportSym := Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType
                 (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                              M        => CurrentMember)));

            if Dictionary.RelationViolatesInfoFlowPolicy (TheExport, TheImportSym) then
               ErrorHandler.SemanticErrorSym2 (175,
                                               ErrorHandler.NoReference,
                                               NodePosition (Node),
                                               TheImportSym,
                                               TheExport,
                                               Scope);
            end if;

            CurrentMember := SeqAlgebra.NextMember (TheHeap, CurrentMember);
         end loop;
      end CheckRelationInfoFlowPolicy;

   begin -- BuildIORelation
      ExportIt := Dictionary.FirstExport (Abstraction, ProcSym);

      if Dictionary.IsNullIterator (ExportIt) then
         RefList.AddNullRelation (Table,
                                  TheHeap,
                                  Node);
      else
         while not Dictionary.IsNullIterator (ExportIt) loop
            ExportSym := Dictionary.CurrentSymbol (ExportIt);
            SeqAlgebra.CreateSeq (TheHeap, ImportSeq);
            ImportIt := Dictionary.FirstDependency (Abstraction,
                                                    ProcSym,
                                                    ExportSym);
            while not Dictionary.IsNullIterator (ImportIt) loop
               ImportSym := Dictionary.CurrentSymbol (ImportIt);
               if Dictionary.IsGlobalVariable (Abstraction,
                                               ProcSym,
                                               ImportSym)
               then
                  SeqAlgebra.AddMember (TheHeap,
                                        ImportSeq,
                                        Natural (Dictionary.SymbolRef
                                                   (SubstituteProtectedTypeSelfReference
                                                      (ImportSym, PrefixSymbol))));
               else
                  ParamNo := Dictionary.GetSubprogramParameterNumber (ImportSym);
                  if ParamTable (ParamNo).Seq /= 0 then
                     SeqAlgebra.AugmentSeq (TheHeap,
                                            ImportSeq,
                                            SeqAlgebra.NaturalToSeq (ParamTable (ParamNo).Seq));
                  end if;
               end if;
               ImportIt := Dictionary.NextSymbol (ImportIt);
            end loop;


            --now substitue export if necessary
            if Dictionary.IsGlobalVariable (Abstraction,
                                            ProcSym,
                                            ExportSym)
            then
               ExportSym := SubstituteProtectedTypeSelfReference (ExportSym, PrefixSymbol);

            else -- parameter
               ParamNo := Dictionary.GetSubprogramParameterNumber (ExportSym);
               if ParamTable (ParamNo).Seq /= 0 and then
                  not SeqAlgebra.IsEmptySeq (TheHeap,
                                             SeqAlgebra.NaturalToSeq (ParamTable (ParamNo).Seq))
               then
                  -- we have an export to process.
                  -- There are two cases:
                  -- (1) It's NOT an array element.  In which case we know that the export is a single
                  --     variable symbol and must be the sole member of the set of vars associated with the
                  --     current parameter; or
                  -- (2) It is an array element in which case the set of vars includes both the array and the
                  --     indexing expressions.

                  -- We can distinguish these case because the field EntireArraySym
                  --     associated with the current parameter will be null in case 1 and non null in case 2 .

                  if ParamTable (ParamNo).EntireArraySym = Dictionary.NullSymbol then
                     -- case 1

                     -- extract singleton from set
                     ActualSym := Dictionary.ConvertSymbolRef
                       (ExaminerConstants.RefType
                          (SeqAlgebra.Value_Of_Member
                             (The_Heap => TheHeap,
                              M        => SeqAlgebra.FirstMember
                                (TheHeap,
                                 SeqAlgebra.NaturalToSeq (ParamTable (ParamNo).Seq)))));
                  else
                     -- case 2
                     -- use array symbol stored especially for this case
                     ActualSym := ParamTable (ParamNo).EntireArraySym;
                     -- Add a self reference here because the actual parameter is an array element.
                     -- Necessary because, for example, SetToZero (A (1));, must derives A from A.
                     -- Secondly, as well as adding self, add all things the selected array
                     -- element depends on (e.g. variables controlling the selected element).
                     -- The array and all the things it depends on are in the ParamTable, thus:
                     SeqAlgebra.AugmentSeq (TheHeap,
                                            ImportSeq,
                                            SeqAlgebra.NaturalToSeq (ParamTable (ParamNo).Seq));
                  end if;

                  if TaggedSubstitutionRequired (ExportSym,
                                                 Dictionary.GetType (ActualSym),
                                                 RootParameterSym) then
                     -- before we can convert the actual parameter to the appropriate root type
                     -- we must add all its subcomponents to the Dictionary and ComponentManager
                     AddRecordSubComponents (RecordVarSym  => ActualSym,
                                             RecordTypeSym => Dictionary.GetType (ActualSym),
                                             ComponentData => GlobalComponentData);
                     ExportSym := ConvertTaggedActual (ActualSym, RootParameterSym);
                  else
                     ExportSym := ActualSym;
                  end if;
               end if;
            end if;

            RefList.AddRelation (Table,
                                 TheHeap,
                                 Node,
                                 ExportSym,
                                 ImportSeq);

            CheckRelationInfoFlowPolicy (ExportSym, ImportSeq);

            ExportIt := Dictionary.NextSymbol (ExportIt);
         end loop;
      end if;
   end BuildIORelation;

   -----------------------------------------------------

   function ActualParameterAt (Position : Positive) return Dictionary.Symbol
   --# global in ActualParameterList;
   is
      Ok  : Boolean;
      Val : Natural;
      Sym : Dictionary.Symbol;

   begin
      SimpleLists.GetItem (ActualParameterList,
                           Position,
                           --to get
                           Val,
                           Ok);
      if Ok then
         Sym := Dictionary.ConvertSymbolRef (ExaminerConstants.RefType (Val));
      else
         Sym := Dictionary.NullSymbol;
      end if;
      return Sym;
   end ActualParameterAt;

   ---------------------------------------------------------------

   procedure CheckProcedureName (Node     : in     STree.SyntaxNode;
                                 Scope    : in     Dictionary.Scopes;
                                 ProcSym  :    out Dictionary.Symbol;
                                 NameNode :    out STree.SyntaxNode;
                                 Valid    :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out STree.Table;
   --#           out Dotted;
   --#           out PrefixSymbol;
   --#           out RootParameterSym;
   --#           out TaggedParameterSym;
   --# derives Dotted,
   --#         NameNode,
   --#         PrefixSymbol,
   --#         ProcSym,
   --#         RootParameterSym,
   --#         STree.Table,
   --#         TaggedParameterSym,
   --#         Valid                     from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      IdentNode,
      PrefixNode   : STree.SyntaxNode;
      ProcSymLocal : Dictionary.Symbol;

      ----------------------------------------------------------------

      procedure CheckSymbol (Node      : in     STree.SyntaxNode;
                             Sym       : in     Dictionary.Symbol;
                             Prefix    : in     LexTokenManager.Lex_String;
                             Name      : in     LexTokenManager.Lex_String;
                             PrefixSym : in     Dictionary.Symbol;
                             Scope     : in     Dictionary.Scopes;
                             ProcSym   :    out Dictionary.Symbol;
                             Valid     :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --#           out RootParameterSym;
      --#           out TaggedParameterSym;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Name,
      --#                                        Node,
      --#                                        Prefix,
      --#                                        PrefixSym,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        Sym &
      --#         STree.Table               from *,
      --#                                        Dictionary.Dict,
      --#                                        Node,
      --#                                        Sym &
      --#         ProcSym,
      --#         RootParameterSym,
      --#         TaggedParameterSym,
      --#         Valid                     from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        Name,
      --#                                        PrefixSym,
      --#                                        Scope,
      --#                                        Sym;
      is
         KindOfOp     : Dictionary.KindsOfOp := Dictionary.NotASubprogram;
         ProcSymLocal : Dictionary.Symbol;
      begin
         ProcSym            := Dictionary.NullSymbol; -- should never be left with this value, but remove DFE
         TaggedParameterSym := Dictionary.NullSymbol;
         RootParameterSym   := Dictionary.NullSymbol;
         ProcSymLocal       := Sym;
         if ProcSymLocal = Dictionary.NullSymbol then --not declared
            -- If we have failed to find an operation by here then we
            -- need to look for an inherited one of the same name in
            -- some root package from which tagged types may have been
            -- extended.
            --
            -- This search is only made if the prefix is a package.  If it is a protected object
            -- then there is no question of there being any inherited operations to find.
            -- We only need to search if there is no prefix or the prefix is a package
            if PrefixSym = Dictionary.NullSymbol or else
              Dictionary.IsPackage (PrefixSym) then
               Dictionary.SearchForInheritedOperations (Name             => Name,
                                                        Scope            => Scope,
                                                        Prefix           => PrefixSym,
                                                        Context          => Dictionary.ProgramContext,
                                                         -- to get
                                                        OpSym            => ProcSymLocal,
                                                        KindOfOp         => KindOfOp,
                                                        ActualTaggedType => TaggedParameterSym);
            end if;

            if ProcSymLocal = Dictionary.NullSymbol then
               -- we still didn't find anything
               Valid := False;
               ProcSym := Dictionary.NullSymbol;
               ErrorHandler.SemanticError2 (143,
                                            ErrorHandler.NoReference,
                                            NodePosition (Node),
                                            NodeLexString (Node),
                                            Prefix);
            else
               -- the inherit search did find something
               if KindOfOp = Dictionary.AProcedure then
                  Valid := True;
                  ProcSym := ProcSymLocal;
                  RootParameterSym := Dictionary.GetSubprogramControllingType (ProcSymLocal);
               else -- found but not a procedure
                  Valid := False;
                  ProcSym := Dictionary.NullSymbol;
               end if;
            end if;
         elsif Dictionary.IsProcedure (Sym) then --ok
            if Dictionary.IsGenericSubprogram (Sym) then
               Valid := False;
               ErrorHandler.SemanticError (654,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           NodeLexString (Node));
            else
               Valid := True;
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => Node);
               ProcSym := ProcSymLocal;
               RootParameterSym := Dictionary.GetSubprogramControllingType (ProcSymLocal);
            end if;
         else --there but not a procedure
            Valid := False;
            ProcSym := Dictionary.NullSymbol;
            ErrorHandler.SemanticError (7,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        NodeLexString (Node));
         end if;
      end CheckSymbol;

      ----------------------------------------------------------------

      function FindNameNode (Node : STree.SyntaxNode)
                            return STree.SyntaxNode
      --# global in STree.Table;
      is
         TempNode : STree.SyntaxNode;

      begin
         TempNode := Node;
         while SyntaxNodeType (TempNode) /= SPSymbols.name loop
            TempNode := ParentNode (TempNode);
         end loop;
         return TempNode;
      end FindNameNode;

      ----------------------------------------------------------------

      procedure DoSimpleName (IdentNode : in     STree.SyntaxNode;
                              Scope     : in     Dictionary.Scopes;
                              ProcSym   :    out Dictionary.Symbol;
                              NameNode  :    out STree.SyntaxNode;
                              Valid     :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --#           out RootParameterSym;
      --#           out TaggedParameterSym;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         NameNode,
      --#         STree.Table               from Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        STree.Table &
      --#         ProcSym,
      --#         RootParameterSym,
      --#         TaggedParameterSym,
      --#         Valid                     from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        STree.Table;
      is
      begin
         CheckSymbol (Node      => IdentNode,
                      Sym       => Dictionary.LookupItem (Name    => NodeLexString (IdentNode),
                                                          Scope   => Scope,
                                                          Context => Dictionary.ProgramContext),
                      Prefix    => LexTokenManager.Null_String,
                      Name      => NodeLexString (IdentNode),
                      PrefixSym => Dictionary.NullSymbol,
                      Scope     => Scope,
                        --to get
                      ProcSym   => ProcSym,
                      Valid     => Valid);
         NameNode := FindNameNode (IdentNode);
      end DoSimpleName;

      ----------------------------------------------------------------

      procedure DoDottedName (IdentNode : in     STree.SyntaxNode;
                              Prefix    : in     STree.SyntaxNode;
                              Scope     : in     Dictionary.Scopes;
                              ProcSym   :    out Dictionary.Symbol;
                              NameNode  :    out STree.SyntaxNode;
                              Valid     :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out STree.Table;
      --#           out PrefixSymbol;
      --#           out RootParameterSym;
      --#           out TaggedParameterSym;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        Prefix,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         NameNode,
      --#         PrefixSymbol,
      --#         ProcSym,
      --#         RootParameterSym,
      --#         STree.Table,
      --#         TaggedParameterSym,
      --#         Valid                     from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        IdentNode,
      --#                                        LexTokenManager.State,
      --#                                        Prefix,
      --#                                        Scope,
      --#                                        STree.Table;
      is
         Sym            : Dictionary.Symbol;
         IdentNodeLocal : STree.SyntaxNode;
         IdentNodePrev  : STree.SyntaxNode;
         PrefixLocal    : STree.SyntaxNode;
         PrefixOk       : Boolean;
      begin
         -- on entry, IdentNode points at the first identifier in the prefix so if the proc
         -- called is P.C.K then IdentNode points at P

         PrefixSymbol := Dictionary.NullSymbol;
         TaggedParameterSym := Dictionary.NullSymbol; -- in case path taken misses CheckSymbol
         RootParameterSym   := Dictionary.NullSymbol; -- in case path taken misses CheckSymbol
         IdentNodeLocal := IdentNode;
         PrefixLocal := Prefix;

         -- look up start of prefix
         Sym := Dictionary.LookupItem (Name    => NodeLexString (IdentNodeLocal),
                                       Scope   => Scope,
                                       Context => Dictionary.ProgramContext);
         loop -- loop over multiple prefixes
            if Sym = Dictionary.NullSymbol then -- not found
               ProcSym := Dictionary.NullSymbol;
               Valid := False;
               ErrorHandler.SemanticError (142,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNodeLocal),
                                           NodeLexString (IdentNodeLocal));
               exit;
            end if;

            if not (Dictionary.IsPackage (Sym) or
                      (Dictionary.IsObject (Sym) and then
                         Dictionary.TypeIsProtected (Dictionary.GetType (Sym)))) then -- not a valid prefix
               ProcSym := Dictionary.NullSymbol;
               Valid := False;
               ErrorHandler.SemanticError (9,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNodeLocal),
                                           NodeLexString (IdentNodeLocal));
               exit;
            end if;

            if Dictionary.IsGenericPackage (Sym) then -- can't access components of generic packages directly
               ProcSym := Dictionary.NullSymbol;
               Valid := False;
               ErrorHandler.SemanticError (655,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNodeLocal),
                                           NodeLexString (IdentNodeLocal));
               exit;
            end if;

            if Dictionary.PrefixAllowed (Sym, Scope) then
               PrefixOk := True;
            else
               PrefixOk := False;
               ErrorHandler.SemanticError (337,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNodeLocal),
                                           Dictionary.GetSimpleName (Sym));
            end if;

            if not PrefixOk then
               ProcSym := Dictionary.NullSymbol;
               Valid := False;
               exit;
            end if;
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => IdentNodeLocal);
            IdentNodePrev := IdentNodeLocal;
            -- move on to next identifier in prefix chain
            IdentNodeLocal := Child_Node
              (Child_Node (Next_Sibling (PrefixLocal)));
            PrefixLocal := ParentNode (ParentNode (ParentNode (PrefixLocal)));

            if SyntaxNodeType (PrefixLocal) /= SPSymbols.prefix then
               -- store package prefix in case we have to search for inherited ops in it
               PrefixSymbol := Sym;
               -- then check procedure name
               CheckSymbol (Node      => IdentNodeLocal,
                            Sym       => Dictionary.LookupSelectedItem (Prefix   => Sym,
                                                                        Selector => NodeLexString (IdentNodeLocal),
                                                                        Scope    => Scope,
                                                                        Context  => Dictionary.ProgramContext),
                            Prefix    => NodeLexString (IdentNodePrev),
                            Name      => NodeLexString (IdentNodeLocal),
                            PrefixSym => PrefixSymbol,
                            Scope     => Scope,
                              --to get
                            ProcSym   => ProcSym,
                            Valid     => Valid);
               exit;
            end if;

            -- otherwise we have another prefix to deal with
            Sym := Dictionary.LookupSelectedItem (Prefix   => Sym,
                                                  Selector => NodeLexString (IdentNodeLocal),
                                                  Scope    => Scope,
                                                  Context  => Dictionary.ProgramContext);
         end loop;

         NameNode := FindNameNode (IdentNodeLocal);
      end DoDottedName;
      --expect spurious dependencies on CommandLineData.Content (coupling thro. SPARKIO)

      ----------------------------------------------------------------

      function ReallyHasPackagePrefix (ProcSym : Dictionary.Symbol;
                                       Scope   : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      is
         DeclaredScope  : Dictionary.Scopes;
         DeclaredRegion : Dictionary.Symbol;
         CallingRegion  : Dictionary.Symbol;
         Result         : Boolean := False;
      begin
         -- Before protected operations, we knew syntactically whether a subprogram had a
         -- package prefix on it.  Now we don't because even a local protected operation call
         -- is prefixed with the protected object name.  We need to distinguish between
         -- PO.Op which is "not dotted" and P.PO.Op which is.

         if ProcSym /= Dictionary.NullSymbol then
            -- Get Region in which caller is declared
            CallingRegion := Dictionary.GetEnclosingPackage (Scope);

            -- get scope where called subprogram is declared
            DeclaredScope := Dictionary.GetScope (ProcSym);
            DeclaredRegion := Dictionary.GetRegion (DeclaredScope);
            if Dictionary.IsProtectedType (DeclaredRegion) then
               -- we want the scope where the protected type is declared
               DeclaredScope := Dictionary.GetScope (DeclaredRegion);
            end if;

            -- function is true if declared region and calling region are different
            Result := Dictionary.GetRegion (DeclaredScope) /= CallingRegion;
         end if;
         return Result;
      end ReallyHasPackagePrefix;

      ----------------------------------------------------------------

   begin --CheckProcedureName
         --assume node is procedure_call_statement
      IdentNode := LastChildOf (Node);
      PrefixNode := ParentNode (ParentNode (ParentNode (IdentNode)));
      PrefixSymbol := Dictionary.NullSymbol;
      if SyntaxNodeType (PrefixNode) = SPSymbols.prefix then
         DoDottedName (IdentNode,    -- points at first identifier in prefix
                       PrefixNode,
                       Scope,
                        --to get
                       ProcSymLocal,
                       NameNode,
                       Valid);
         Dotted := ReallyHasPackagePrefix (ProcSymLocal, Scope);
         ProcSym := ProcSymLocal;
      else
         Dotted := False;
         DoSimpleName (IdentNode,    -- points at procedure name
                       Scope,
                        --to get
                       ProcSym,
                       NameNode,
                       Valid);
      end if;
   end CheckProcedureName;

   --=====================================================================--

   procedure CheckGlobalsAreVisible (Node    : in STree.SyntaxNode;
                                     ProcSym : in Dictionary.Symbol;
                                     Scope   : in Dictionary.Scopes)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     PrefixSymbol;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PrefixSymbol,
   --#                                        ProcSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It                 : Dictionary.Iterator;
      CallingSym,
      GlobSym            : Dictionary.Symbol;
      CallingAbstraction : Dictionary.Abstractions;

      function IsLocalVariable (CallingSym,
                                GlobSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.GetRegion (Dictionary.GetScope (GlobSym)) = CallingSym;
      end IsLocalVariable;

      function IsOwnVarOfEmbeddedPackage (CallingSym,
                                          GlobSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return (Dictionary.IsOwnVariable (GlobSym)) and then
            (Dictionary.GetScope (Dictionary.GetOwner (GlobSym)) =
             Dictionary.LocalScope (CallingSym));
      end IsOwnVarOfEmbeddedPackage;


      --this procedure can be extended to deal with error case of vars which
      --are exported by a called procedure but not exported by calling procedure

      procedure CheckIllegalUpdate (CallingAbstraction,
                                    Abstraction : in Dictionary.Abstractions;
                                    CallingSym,
                                    ProcSym,
                                    GlobSym     : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        CallingAbstraction,
      --#                                        CallingSym,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GlobSym,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        ProcSym,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
      begin
         if Dictionary.IsFunction (CallingSym) and then
            Dictionary.IsGlobalVariable (CallingAbstraction,
                                         CallingSym,
                                         GlobSym) and then
            Dictionary.IsExport (Abstraction,
                                 ProcSym,
                                 GlobSym) and then not
            -- if the global is a mode in own variable then the "exporting" is only
            -- a modelling artefact that we can ignore
           (Dictionary.GetOwnVariableOrConstituentMode (GlobSym) = Dictionary.InMode)
         then
            ErrorHandler.SemanticError (328,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (GlobSym));
         end if;
      end CheckIllegalUpdate;

   begin
      CallingSym := Dictionary.GetEnclosingCompilationUnit (Scope);
      CallingAbstraction := Dictionary.GetAbstraction (CallingSym, Scope);
      It := Dictionary.FirstGlobalVariable (Abstraction,
                                            ProcSym);
      while not Dictionary.IsNullIterator (It) loop
         GlobSym := SubstituteProtectedTypeSelfReference (Dictionary.CurrentSymbol (It), PrefixSymbol);

         if (not Dictionary.IsGlobalVariable (CallingAbstraction,
                                              CallingSym,
                                              GlobSym)) and then
           (not Dictionary.IsFormalParameter (CallingSym,
                                              GlobSym)) and then
           (not IsLocalVariable (CallingSym, GlobSym)) and then
           (not IsOwnVarOfEmbeddedPackage (CallingSym, GlobSym)) and then
           (GlobSym /= Dictionary.GetNullVariable) -- null variable always deemed visible
         then
            if Dictionary.IsImport (Abstraction,
                                    ProcSym,
                                    GlobSym) and then
               Dictionary.GetOwnVariableOrConstituentMode (GlobSym) /= Dictionary.OutMode
            then
               ErrorHandler.SemanticErrorSym (25,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              GlobSym,
                                              Scope);
            end if;
            if Dictionary.IsExport (Abstraction,
                                    ProcSym,
                                    GlobSym) and then
               Dictionary.GetOwnVariableOrConstituentMode (GlobSym) /= Dictionary.InMode
            then
               ErrorHandler.SemanticErrorSym (24,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              GlobSym,
                                              Scope);
            end if;
         end if;

         CheckIllegalUpdate (CallingAbstraction,
                             Abstraction,
                             CallingSym,
                             ProcSym,
                             GlobSym);

         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckGlobalsAreVisible;

   --=====================================================================--
-- Procedure relocated to compunit to allow reuse
--     --procedure; this which has essentially the functionality of WalkExpression
--     --but is used to walk expressions which the language rules require to be a
--     --"name".  In addition to the things returned by WalkExpression, a flag is
--     --returned indicating whether the expression was indeed a name.
--     --If the expression is not a name the expression is not traversed at all
--     --and Result is set to the UnknownTypeRecord
--     --
--     --After the name node has been found it is traversed by WalkExpression and a subset
--     --of the checks usually done by wf_primary are acarried out on the result.  More
--     --checks are done here than are necessary for this application (e.g. getting bounds
--     --of a type mark) but they have been left in to make WalkName more widely applicable;
--     --it could be moved to Wellformed if ever needed elsewhere.
--     --
--     --
--     procedure WalkName (ExpNode     : in     STree.SyntaxNode;
--                         Scope       : in     Dictionary.Scopes;
--                         Result      :    out ExpRecord;
--                         IsAName     :    out Boolean;
--                         RefVarParam : in     SeqAlgebra.Seq)
--     --# global in     CommandLineData.Content;
--     --#        in out Dictionary.Dict;
--     --#        in out TheHeap;
--     --#        in out Statistics.TableUsage;
--     --#        in out GlobalComponentData;
--     --#        in out STree.Table;
--     --#        in out ErrorHandler.ErrorContext;
--     --#        in out SPARK_IO.File_Sys;
--     --#        in out LexTokenManager.State;
--     --#        in out AggregateStack.State;
--     --# derives Dictionary.Dict,
--     --#         TheHeap,
--     --#         Statistics.TableUsage,
--     --#         GlobalComponentData,
--     --#         STree.Table,
--     --#         LexTokenManager.State,
--     --#         AggregateStack.State              from *,
--     --#                                                Dictionary.Dict,
--     --#                                                TheHeap,
--     --#                                                GlobalComponentData,
--     --#                                                CommandLineData.Content,
--     --#                                                Scope,
--     --#                                                STree.Table,
--     --#                                                LexTokenManager.State,
--     --#                                                ExpNode,
--     --#                                                RefVarParam &
--     --#         ErrorHandler.ErrorContext,
--     --#         SPARK_IO.File_Sys                 from Dictionary.Dict,
--     --#                                                TheHeap,
--     --#                                                GlobalComponentData,
--     --#                                                CommandLineData.Content,
--     --#                                                Scope,
--     --#                                                STree.Table,
--     --#                                                ErrorHandler.ErrorContext,
--     --#                                                SPARK_IO.File_Sys,
--     --#                                                LexTokenManager.State,
--     --#                                                ExpNode,
--     --#                                                RefVarParam &
--     --#         Result                            from Dictionary.Dict,
--     --#                                                TheHeap,
--     --#                                                GlobalComponentData,
--     --#                                                CommandLineData.Content,
--     --#                                                Scope,
--     --#                                                STree.Table,
--     --#                                                LexTokenManager.State,
--     --#                                                ExpNode,
--     --#                                                RefVarParam &
--     --#         IsAName                           from STree.Table,
--     --#                                                ExpNode;
--     is
--        Node        : STree.SyntaxNode;
--        Sym         : Dictionary.Symbol;
--        ResultLocal : ExpRecord;

--     begin --WalkName
--        ResultLocal := UnknownTypeRecord; --preset result for all non-name cases
--        -- advance to relation node;
--        Node := Child_Node (ExpNode);
--        if Next_Sibling (Node) =
--           STree.NullNode then
--           -- advance to simple_expression;
--           Node := Child_Node (Node);
--           if Next_Sibling (Node) =
--              STree.NullNode then
--              -- advance to simple_expression_opt;
--              Node := Child_Node (Node);
--              if Next_Sibling (Node) =
--                 STree.NullNode then
--                 -- advance to term;
--                 Node := Child_Node (Node);
--                 if Next_Sibling (Node) =
--                    STree.NullNode then
--                    -- advance to factor;
--                    Node := Child_Node (Node);
--                    if Next_Sibling (Node) =
--                       STree.NullNode then
--                       -- advance to primary;
--                       Node := Child_Node (Node);
--                       if Next_Sibling (Node) =
--                          STree.NullNode then
--                          -- advance to name;
--                          Node := Child_Node (Node);
--                          if SyntaxNodeType (Node) = SPSymbols.name then
--                             IsAName := True;
--                             WalkExpression (ExpNode       => Node,
--                                             Scope         => Scope,
--                                               -- The type of a Name can never be resovled by context - it's
--                                               -- either a name which exists in the Dictionary and has a type,
--                                               -- or it not, so we pass UnknownTypeMark as the context here.
--                                             TypeContext   => Dictionary.GetUnknownTypeMark,
--                                             Result        => ResultLocal,
--                                             RefVar        => RefVarParam,
--                                             ComponentData => GlobalComponentData);
--                             -- now perform some checks on the result akin to those
--                             -- of wf_primary
--                             Sym := ResultLocal.OtherSymbol;

--                             case ResultLocal.Sort is
--                                when IsUnknown =>
--                                   ResultLocal := UnknownTypeRecord;
--                                when TypeResult =>
--                                   null;
--                                when IsPackage =>
--                                   ResultLocal := UnknownTypeRecord;
--                                   ErrorHandler.SemanticError (5,
--                                                               ErrorHandler.NoReference,
--                                                               NodePosition (Node),
--                                                               Dictionary.GetSimpleName (Sym));
--                                when IsFunction =>
--                                   ErrorHandler.SemanticError (3,
--                                                               ErrorHandler.NoReference,
--                                                               NodePosition (Node),
--                                                               Dictionary.GetSimpleName (Sym));
--                                   ResultLocal.IsStatic   := False;
--                                   ResultLocal.IsConstant := False;
--                                   ResultLocal.IsARange   := False;
--                                when IsObject =>
--                                   ResultLocal.IsARange   := False;
--                                   if Dictionary.IsVariableOrSubcomponent (Sym) then
--                                      SeqAlgebra.AddMember
--                                         (TheHeap,
--                                          RefVarParam,
--                                          Natural (Dictionary.SymbolRef (Sym)));
--                                   end if;
--                                when IsTypeMark =>
--                                   if Dictionary.IsScalarTypeMark (Sym,
--                                                                   Scope)
--                                   then
--                                      ResultLocal.IsStatic := Dictionary.IsStatic (Sym,
--                                                                                   Scope);
--                                      ResultLocal.IsConstant := True;
--                                      ResultLocal.IsARange   := True;
--                                      -- get upper and lower bounds of type from dictionary
--                                      ResultLocal.Value := Maths.ValueRep
--                                         (Dictionary.GetScalarAttributeValue
--                                          (False,  --don't want base type
--                                           LexTokenManager.FirstToken,
--                                           Sym));
--                                      ResultLocal.RangeRHS := Maths.ValueRep
--                                         (Dictionary.GetScalarAttributeValue
--                                          (False,  --don't want base type
--                                           LexTokenManager.LastToken,
--                                           Sym));

--                                   else
--                                      ResultLocal := UnknownTypeRecord;
--                                      ErrorHandler.SemanticError
--                                         (5,
--                                          ErrorHandler.NoReference,
--                                          NodePosition (Node),
--                                          Dictionary.GetSimpleName (Sym));
--                                   end if;
--                                when IsParameterName =>
--                                   null; --should never occur
--                             end case;

--                             ResultLocal.ParamCount := 0;
--                             ResultLocal.ParamList := Lists.NullList;

--                          else
--                             IsAName := False;
--                          end if;
--                       else
--                          IsAName := False;
--                       end if;
--                    else
--                       IsAName := False;
--                    end if;
--                 else
--                    IsAName := False;
--                 end if;
--              else
--                 IsAName := False;
--              end if;
--           else
--              IsAName := False;
--           end if;
--        else
--           IsAName := False;
--        end if;
--        Result := ResultLocal;
--     end WalkName;

   -------------------------------------------------------------------------

   procedure CheckActual (ProcSym,
                          FormalSym       : in     Dictionary.Symbol;
                          FormalPos       : in     Positive;
                          ActualNode      : in     STree.SyntaxNode;
                          RefVarParam     : in     SeqAlgebra.Seq;
                          EntireArraySym  :    out Dictionary.Symbol)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     RootParameterSym;
   --#        in     Scope;
   --#        in     TaggedParameterSym;
   --#        in out ActualParameterList;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out FormalPositionList;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ActualParameterList,
   --#         AggregateStack.State      from *,
   --#                                        Abstraction,
   --#                                        ActualNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FormalSym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         SLI.State                 from *,
   --#                                        Abstraction,
   --#                                        ActualNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FormalSym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        Abstraction,
   --#                                        ActualNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FormalSym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RefVarParam,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         EntireArraySym,
   --#         LexTokenManager.State,
   --#         STree.Table               from Abstraction,
   --#                                        ActualNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FormalSym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        ActualNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FormalSym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RefVarParam,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TaggedParameterSym,
   --#                                        TheHeap &
   --#         FormalPositionList        from *,
   --#                                        Abstraction,
   --#                                        ActualNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        FormalPos,
   --#                                        FormalSym,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RefVarParam,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      EntireVarSym,
      VarType      : Dictionary.Symbol;
      ModeOfFormal : Dictionary.Modes;
      ExpResult    : ExpRecord;
      IsAName      : Boolean;
      --ActualSym    : Dictionary.Symbol;
      ActualType   : Dictionary.Symbol;
      ActualParameterIsAVariable : Boolean := False;
      ActualParameterIsAConstant : Boolean := False;

      --------------------------------------------------------------------

      procedure CheckValidMode (ActualSym    : in      Dictionary.Symbol;
                                ModeOfFormal : in      Dictionary.Modes)
      --# global in     ActualNode;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ActualNode,
      --#                                        ActualSym,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        ModeOfFormal,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
         -- ActualSym represents an actual parameter, whose corresponding formal
         -- parameter is of mode in out or out.  It is the symbol of the associated
         -- entire variable (if the parameter is an entire variable or a
         -- subcomponent of a variable).
         --
         -- The procedure checks that every actual parameter of mode in out or out
         -- is not
         --       -  a formal parameter of mode in, or
         --       -  a loop parameter,
         -- and that an actual parameter of mode in out is not itself a formal
         -- parameter of mode out, or a subcomponent thereof.
         -- Also warns of direct update of non-enclosing package own variable
         --
      is
         ActualMode : Dictionary.Modes;

      begin  --CheckValidMode
         if Dictionary.IsLoopParameter (ActualSym) then
            ErrorHandler.SemanticError (168,
                                        ErrorHandler.NoReference,
                                        NodePosition (ActualNode),
                                        Dictionary.GetSimpleName (ActualSym));
         elsif Dictionary.IsOwnVariable (ActualSym) and then
            not IsEnclosingPackage (Dictionary.GetOwner (ActualSym),
                                    Scope)
         then
            ErrorHandler.SemanticWarningSym (169,
                                             NodePosition (ActualNode),
                                             ActualSym,
                                             Scope);
         end if;

         if Dictionary.IsSubprogramParameter (ActualSym) then
            ActualMode := Dictionary.GetSubprogramParameterMode (ActualSym);
            if  ActualMode = Dictionary.InMode or
               ActualMode = Dictionary.DefaultMode
            then
               ErrorHandler.SemanticError (170,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           Dictionary.GetSimpleName (ActualSym));
            elsif CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 and
               ActualMode = Dictionary.OutMode and
               ModeOfFormal = Dictionary.InOutMode
            then
               ErrorHandler.SemanticError (171,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           Dictionary.GetSimpleName (ActualSym));
            end if;
         end if;
      end CheckValidMode;

      -------------------------------------------------------------------

      procedure CheckType (ActualType     : in     Dictionary.Symbol;
                           ActualRangeRHS : in     Maths.Value;
                           FormalSym      : in     Dictionary.Symbol;
                           ActualNode     : in     STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     TaggedParameterSym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ActualNode,
      --#                                        ActualRangeRHS,
      --#                                        ActualType,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        FormalSym,
      --#                                        LexTokenManager.State,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TaggedParameterSym;
      is
         ExpectedType : Dictionary.Symbol;
      begin
         ExpectedType := Dictionary.GetType (FormalSym);

         if not (TaggedParameterSym = ActualType or else              -- inherited call, right type
                 (TaggedParameterSym = Dictionary.NullSymbol and then -- no inheritance, normal rules
                  Dictionary.CompatibleTypes (Scope,
                                              ActualType,
                                              ExpectedType)) or else
                 (not Dictionary.IsAnExtensionOf (ActualType, TaggedParameterSym) and then -- trap use of root type
                 Dictionary.CompatibleTypes (Scope,
                                             ActualType,
                                             ExpectedType)))
         then
            -- we have a type mismatch but there are two cases to consider: (1) the failure is in an inherited call
            -- indicated by TaggedParameterSym /= NullSymbol or (2) in a normal proc call.  To provide the right
            -- information in the error message we need a condition thus:
            if TaggedParameterSym = Dictionary.NullSymbol then -- normal case
               ErrorHandler.SemanticErrorSym2 (107, --Expression not of expected type
                                               ErrorHandler.NoReference,
                                               NodePosition (ActualNode),
                                               ActualType,
                                               ExpectedType,
                                               Scope);
            else -- inherited call
               ErrorHandler.SemanticErrorSym2 (107, --Expression not of expected type
                                               ErrorHandler.NoReference,
                                               NodePosition (ActualNode),
                                               ActualType,
                                               TaggedParameterSym, -- what we really wanted
                                               Scope);
            end if;

            -- if types match ok, do checks for array bounds, strings etc
         elsif Dictionary.TypeIsArray (ExpectedType) and then
            not Dictionary.IsUnconstrainedArrayType (ExpectedType)
         then
            -- Formal parameter type is Constrained.
            if Dictionary.IsUnconstrainedArrayType (ActualType) then
               -- Actual is unconstrained.  In SPARK95 or 2005, this is OK if
               -- the actual is a static String expression, but illegal
               -- otherwise.
               if CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83 and then
                 Dictionary.IsPredefinedStringType (ActualType) then
                  -- Formal must be a constrained String subtype, so we need
                  -- to check the upper bound of the actual (if it has one) against
                  -- the expected upper bound of the formal.
                  if ActualRangeRHS = Maths.NoValue then
                     -- Actual is not static, so must be illegal
                     ErrorHandler.SemanticError (39,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ActualNode),
                                                 LexTokenManager.Null_String);
                  else
                     -- Actual is static, so check upper-bound against that expected
                     if ActualRangeRHS /= Maths.ValueRep
                       (Dictionary.GetScalarAttributeValue (False,
                                                            LexTokenManager.Last_Token,
                                                            ExpectedType)) then
                        ErrorHandler.SemanticError (418,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (ActualNode),
                                                    LexTokenManager.Null_String);

                     end if;
                  end if;
               else
                  ErrorHandler.SemanticError (39,
                                              ErrorHandler.NoReference,
                                              NodePosition (ActualNode),
                                              LexTokenManager.Null_String);
               end if;


            elsif Illegal_Unconstrained (ExpectedType,
                                         ActualType)
            then
               ErrorHandler.SemanticError (418,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           LexTokenManager.Null_String);
            end if;
         end if;
      end CheckType;

      -------------------------------------------------------------------

      function IsGlobalOf (VarSym,
                           ProcSym : Dictionary.Symbol) return Boolean
      --# global in Abstraction;
      --#        in Dictionary.Dict;
      is
         It    : Dictionary.Iterator;
         Found : Boolean;
      begin
         Found := False;
         It := Dictionary.FirstGlobalVariable (Abstraction,
                                               ProcSym);
         while not Dictionary.IsNullIterator (It) loop
            if Dictionary.CurrentSymbol (It) = VarSym then
               Found := True;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;

         return Found;

      end IsGlobalOf;

      -------------------------------------------------------------------

      function IsExportedGlobalOf (VarSym,
                                   ProcSym : Dictionary.Symbol) return Boolean
      --# global in Abstraction;
      --#        in Dictionary.Dict;
      is
         It    : Dictionary.Iterator;
         Found : Boolean;
      begin
         Found := False;
         It := Dictionary.FirstGlobalVariable (Abstraction,
                                               ProcSym);
         while not Dictionary.IsNullIterator (It) loop
            if Dictionary.CurrentSymbol (It) = VarSym then
               Found := Dictionary.IsExport (Abstraction,
                                             ProcSym,
                                             Dictionary.CurrentSymbol (It));
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;

         return Found;

      end IsExportedGlobalOf;

      -------------------------------------------------------------------

      procedure CheckOverlapOfExportedGlobal (VarSym,
                                              ProcSym : Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     ActualNode;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstraction,
      --#                                        ActualNode,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        ProcSym,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        VarSym;
      is
      begin
         if IsExportedGlobalOf (VarSym, ProcSym) then
            ErrorHandler.SemanticError (166,
                                        ErrorHandler.NoReference,
                                        NodePosition (ActualNode),
                                        LexTokenManager.Null_String);
         end if;
      end CheckOverlapOfExportedGlobal;

      ----------------------------------------------------------

      procedure CheckForUseOfProtectedOrStreamVars (VarSym,
                                                    ProcSym : Dictionary.Symbol)
      --# global in     ActualNode;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ActualNode,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        ProcSym,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        VarSym;
      is
      begin
         -- Stream variables cannot be passed as parameters
         if Dictionary.IsOwnVariableOrConstituentWithMode (VarSym) then

            ErrorHandler.SemanticErrorSym (716,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           VarSym,
                                           Scope);

         -- Protected variable cannot be passed as parameters
         elsif Dictionary.IsOwnVariable (VarSym) and then
           Dictionary.GetOwnVariableProtected (VarSym) then

            -- unless it is a suspension object to a predefined operation.
            if not (Dictionary.IsPredefinedSuspensionObjectType
                      (Dictionary.GetType (VarSym)) and
                      Dictionary.IsPredefined (ProcSym)) then

               ErrorHandler.SemanticErrorSym (725,
                                              ErrorHandler.NoReference,
                                              NodePosition (ActualNode),
                                              VarSym,
                                              Scope);
            end if;
         end if;
      end CheckForUseOfProtectedOrStreamVars;

      ----------------------------------------------------------

      --procedure maintains two parallel list one of actual symbols
      --     and the other of the formal parameter position number applicable
      --     to each.
      procedure AddSym (Sym       : in     Dictionary.Symbol;
                        FormalPos : in     Positive)
      --# global in out ActualParameterList;
      --#        in out FormalPositionList;
      --# derives ActualParameterList from *,
      --#                                  Sym &
      --#         FormalPositionList  from *,
      --#                                  FormalPos;
      is
         Ok : Boolean;
      begin
         SimpleLists.AddItem (Natural (Dictionary.SymbolRef (Sym)),
                              ActualParameterList,
                              Ok);
         if not Ok then
            SystemErrors.FatalError (SystemErrors.ListOverflowInProcedureCall, "");
         end if;

         SimpleLists.AddItem (FormalPos,
                              FormalPositionList,
                              Ok);
         if not Ok then
            SystemErrors.FatalError (SystemErrors.ListOverflowInProcedureCall, "");
         end if;
      end AddSym;

      --------------------------------------------

      function GetRootVariable (VarSym : Dictionary.Symbol) return Dictionary.Symbol
      --# global in Dictionary.Dict;
      --#        in GlobalComponentData;
      is
         Result : Dictionary.Symbol;
      begin
         -- When adding variables to the ActualPArameterList for alias-checking purposes
         -- we need to avoid adding view conversions since these will have different numeric
         -- values from the object from which they are converted.  For example, the first root-wise
         -- conversion of O is O.Inherit; O.Inherit has a different symbol from O even though they
         -- represent the same object (and same memeory location).  If we add O.Inherit then alias
         -- checking fails; this is most likely in tagged type cases.  This function converts things
         -- like O.Inherit to O.
         if Dictionary.IsRecordSubcomponent (VarSym) then
            Result := ComponentManager.GetName
              (GlobalComponentData,
               ComponentManager.GetRoot
                 (GlobalComponentData,
                  ComponentManager.GetComponentNode
                  (GlobalComponentData,
                   VarSym)));
         else
            Result := VarSym;
         end if;
         return Result;
      end GetRootVariable;

      --------------------------------------------

      function IsGlobalOfAnEnclosingFunction (CallingScope : Dictionary.Scopes;
                                              ParamSym     : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         CallingSym : Dictionary.Symbol;
         CallingAbstraction : Dictionary.Abstractions;
         Result : Boolean := False;
      begin
         CallingSym := Dictionary.GetEnclosingCompilationUnit (CallingScope);
         if Dictionary.IsFunction (CallingSym) then
            -- Procedure that exports ParamSym is being called from inside a function body
            -- so there is a possibility that an indirect function side efefct is occurring
            CallingAbstraction := Dictionary.GetAbstraction (CallingSym, CallingScope);
            Result := Dictionary.IsGlobalVariable (CallingAbstraction,
                                                   CallingSym,
                                                   ParamSym);
         end if;
         return Result;
      end IsGlobalOfAnEnclosingFunction;

      --------------------------------------------

   begin -- CheckActual
      ActualType := Dictionary.GetUnknownTypeMark;
      EntireArraySym := Dictionary.NullSymbol; -- default value for all except actual param which is array element
      ModeOfFormal := Dictionary.GetSubprogramParameterMode (FormalSym);
      if Dictionary.IsExport (Abstraction,
                              ProcSym,
                              FormalSym)
      then  --no array elements or indexing permitted
         WalkName (ActualNode,
                   Scope,
                     --to get
                   ExpResult,
                   IsAName,
                   RefVarParam);

         if IsAName then
            if ExpResult.IsAVariable then --valid variable found, do further checks
               ActualParameterIsAVariable := True;
               EntireVarSym := ExpResult.VariableSymbol;
               if ExpResult.ArgListFound then
                  -- actual parameter is an array reference so we need to return the entire array symbol
                  -- so we can correctly handle the flow analysis of calls such as SetToZero (A (I));
                  --
                  -- If it's an entire variable, then the symbol we need is in ExpResult.VariableSymbol
                  --
                  -- If it's NOT an entire var, then it must be record field such as R.F (I).
                  -- If that case, the flow analyser needs the symbol of the field itself,
                  -- which is in ExpResult.OtherSymbol
                  if ExpResult.IsAnEntireVariable then
                     EntireArraySym := ExpResult.VariableSymbol;
                  else
                     EntireArraySym := ExpResult.OtherSymbol;
                  end if;
               end if;
               VarType := ExpResult.TypeSymbol;
               ActualType := ExpResult.TypeSymbol;
               PlantConstrainingType (ActualType,
                                      Maths.NoValue, -- because a string literal can't be an export
                                      ActualNode);
               AddSym (GetRootVariable (EntireVarSym),
                       FormalPos);
               CheckType (VarType,
                          Maths.NoValue,
                          FormalSym,
                          ActualNode);
               CheckValidMode (EntireVarSym,
                               ModeOfFormal);
               CheckForUseOfProtectedOrStreamVars (EntireVarSym,
                                                   ProcSym);

               if IsGlobalOfAnEnclosingFunction (Scope, EntireVarSym) then
                  -- We are attempting to change a function's global via it's use as an actual parameter
                  -- to an procedure within the function
                  ErrorHandler.SemanticErrorSym (328,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ActualNode),
                                                 EntireVarSym,
                                                 Scope);
               end if;

               if IsGlobalOf (EntireVarSym,
                              ProcSym)
               then                    --export overlapping global
                  ErrorHandler.SemanticError (173,
                                              ErrorHandler.NoReference,
                                              NodePosition (ActualNode),
                                              LexTokenManager.Null_String);
               end if;

            else --expression is not an entire variable
               AddSym (Dictionary.NullSymbol,
                       FormalPos);             --put place holder in parameter list
               ErrorHandler.SemanticError (172,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           LexTokenManager.Null_String);
            end if;
         else --its not even a name
            AddSym (Dictionary.NullSymbol,
                    FormalPos);             --put place holder in parameter list
            ErrorHandler.SemanticError (172,
                                        ErrorHandler.NoReference,
                                        NodePosition (ActualNode),
                                        LexTokenManager.Null_String);
         end if;

      elsif (ModeOfFormal = Dictionary.InOutMode) or
         (ModeOfFormal = Dictionary.OutMode)
      then --not an export but must be variable because of mode---------------------
         WalkName (ActualNode,
                   Scope,
                     --to get
                   ExpResult,
                   IsAName,
                   RefVarParam);
         if IsAName then
            if ExpResult.IsAVariable then --ok so far, do further checks
               ActualParameterIsAVariable := True;
               EntireVarSym := ExpResult.VariableSymbol;
               VarType := ExpResult.TypeSymbol;
               ActualType := ExpResult.TypeSymbol;
               PlantConstrainingType (ActualType,
                                      Maths.NoValue, -- we know it's a variable so it can't be a literal
                                      ActualNode);
               AddSym (GetRootVariable (EntireVarSym),
                       FormalPos);
               CheckType (VarType,
                          Maths.NoValue,
                          FormalSym,
                          ActualNode);
               CheckValidMode (EntireVarSym,
                               ModeOfFormal);
               CheckOverlapOfExportedGlobal (EntireVarSym,
                                             ProcSym);
               CheckForUseOfProtectedOrStreamVars (EntireVarSym,
                                                   ProcSym);

            else --its not a variable when it should have been
               AddSym (Dictionary.NullSymbol,
                       FormalPos); --put place holder in parameter list
               ErrorHandler.SemanticError (164,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           LexTokenManager.Null_String);
            end if;
         else --its not a name so it certainly isn't a variable
            AddSym (Dictionary.NullSymbol,
                    FormalPos); --put place holder in parameter list
            ErrorHandler.SemanticError (164,
                                        ErrorHandler.NoReference,
                                        NodePosition (ActualNode),
                                        LexTokenManager.Null_String);
         end if;

      else
         -- Mode is "in", so any expression will do but vars need special alias checks
         WalkExpression (ExpNode     => ActualNode,
                         Scope       => Scope,
                           -- Context type is always the type of the formal parameter.
                         TypeContext           => Dictionary.GetType (FormalSym),
                         ContextRequiresStatic => False,
                           -- to get
                         Result        => ExpResult,
                         RefVar        => RefVarParam,
                         ComponentData => GlobalComponentData);

         ActualType := ExpResult.TypeSymbol;
         PlantConstrainingType (ActualType,
                                ExpResult.RangeRHS, -- the string's length if parameter is a string literal
                                ActualNode);
         if ExpResult.VariableSymbol /= Dictionary.NullSymbol then
            --we have a variable or 'disguised' variable
            ActualParameterIsAVariable := True;
            EntireVarSym := ExpResult.VariableSymbol;
            VarType := ExpResult.TypeSymbol;
            AddSym (GetRootVariable (EntireVarSym),
                    FormalPos);     --put in list of actual parameters
            CheckType (VarType,
                       Maths.NoValue,
                       FormalSym,
                       ActualNode);

            if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK83 and then
               Dictionary.IsSubprogramParameter (EntireVarSym) and then
               Dictionary.GetSubprogramParameterMode (EntireVarSym) =
               Dictionary.OutMode
            then
               ErrorHandler.SemanticError (171,
                                           ErrorHandler.NoReference,
                                           NodePosition (ActualNode),
                                           Dictionary.GetSimpleName (EntireVarSym));
            end if;

            CheckOverlapOfExportedGlobal (EntireVarSym,
                                          ProcSym);


         elsif ExpResult.IsARange then -- check that a type mark isn't being used as an actual parameter
            ErrorHandler.SemanticError (5,
                                        ErrorHandler.NoReference,
                                        NodePosition (ActualNode),
                                        Dictionary.GetSimpleName (ExpResult.TypeSymbol));

         else  --True expression, does not represent a variable and isn't a typemark
            ActualParameterIsAConstant := ExpResult.IsConstant;
            AddSym (Dictionary.NullSymbol,
                    FormalPos);              --put place holder in parameter list
            CheckType (ExpResult.TypeSymbol, --only check necessary and possible
                       ExpResult.RangeRHS,
                       FormalSym,
                       ActualNode);
         end if;
      end if;

      -- We might need to substitute
      -- a subtype of the actual paramter if an inherited root op is being called

      if TaggedSubstitutionRequired (FormalSym,
                                     ActualType,
                                     RootParameterSym) then
         -- Seed actual parameter node with type of actual parameter for use in VCG
         -- (no longer needed, VC uses alternative way of getting actual param type)
         -- STree.AddNodeSymbol (ActualNode, ActualType);

         if ActualParameterIsAVariable then
            SubstituteRootTypeForExtendedTypeInSequence (RefVarParam,
                                                         RootParameterSym,
                                                         GlobalComponentData);
         elsif not ActualParameterIsAConstant then
            ErrorHandler.SemanticError (827,
                                        ErrorHandler.NoReference,
                                        NodePosition (ActualNode),
                                        LexTokenManager.Null_String);
         end if;
      end if;
   end CheckActual;

   --=====================================================================--

   procedure HandleNamedAssociation
   --# global in     Abstraction;
   --#        in     ArgListNode;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     ProcSym;
   --#        in     RootParameterSym;
   --#        in     Scope;
   --#        in     TaggedParameterSym;
   --#        in out ActualParameterList;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out FormalPositionList;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out ParamTable;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ActualParameterList,
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         FormalPositionList,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         ParamTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        Abstraction,
   --#                                        ArgListNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        ArgListNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TaggedParameterSym,
   --#                                        TheHeap;
   is
      CheckNamedAssociationRefVar : SeqAlgebra.Seq;
      FormalSym                   : Dictionary.Symbol;
      It                          : Dictionary.Iterator;
      NamedArgumentAssocNode      : STree.SyntaxNode;
      ExpressionNode              : STree.SyntaxNode;
      EntireArraySym              : Dictionary.Symbol;
   begin

      NamedArgumentAssocNode := Child_Node (ArgListNode);

      CheckNamedAssociation (TheFormals             => ProcSym,
                             Scope                  => Scope,
                             NamedArgumentAssocNode => NamedArgumentAssocNode);

      -- Loop through all the formals
      It := Dictionary.FirstSubprogramParameter (ProcSym);
      while not Dictionary.IsNullIterator (It) loop

         FormalSym := Dictionary.CurrentSymbol (It);

         ExpressionNode :=
           FindActualNode (ForFormal              => FormalSym,
                           NamedArgumentAssocNode => NamedArgumentAssocNode);

         if ExpressionNode /= STree.NullNode then

            SeqAlgebra.CreateSeq (TheHeap, CheckNamedAssociationRefVar);

            CheckActual (ProcSym        => ProcSym,
                         FormalSym      => FormalSym,
                         FormalPos      => Dictionary.GetSubprogramParameterNumber (FormalSym),
                         ActualNode     => ExpressionNode,
                         RefVarParam    => CheckNamedAssociationRefVar,
                         EntireArraySym => EntireArraySym);

            AddSeq (Dictionary.GetSubprogramParameterNumber (FormalSym),
                    CheckNamedAssociationRefVar,
                    EntireArraySym);

         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end HandleNamedAssociation;

   --=====================================================================--

   procedure HandlePositionalAssociation
   --# global in     Abstraction;
   --#        in     ArgListNode;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.FileHeap;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     ProcSym;
   --#        in     RootParameterSym;
   --#        in     Scope;
   --#        in     TaggedParameterSym;
   --#        in out ActualParameterList;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out FormalPositionList;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out ParamTable;
   --#        in out SLI.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ActualParameterList,
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         FormalPositionList,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         ParamTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                   from *,
   --#                                        Abstraction,
   --#                                        ArgListNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SLI.State,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        ArgListNode,
   --#                                        CommandLineData.Content,
   --#                                        ContextManager.Ops.FileHeap,
   --#                                        ContextManager.Ops.UnitHeap,
   --#                                        ContextManager.Ops.UnitStack,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        RootParameterSym,
   --#                                        Scope,
   --#                                        SLI.State,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TaggedParameterSym,
   --#                                        TheHeap;
   is
      ExpressionNode  : STree.SyntaxNode;
      FormalIt        : Dictionary.Iterator;
      ActualIt        : STree.Iterator;
      FormalParameter : Dictionary.Symbol;
      ParameterNumber : Natural;
      CheckPositionalAssociationRefVar : SeqAlgebra.Seq;
      EntireArraySym  : Dictionary.Symbol;
   begin
      FormalIt := Dictionary.FirstSubprogramParameter (ProcSym);

      ActualIt := FindFirstNode
        (NodeKind => SPSymbols.expression,
         FromRoot => ArgListNode,
         InDirection => STree.Down);

      ExpressionNode := ArgListNode;

      while not Dictionary.IsNullIterator (FormalIt) and
        not STree.IsNull (ActualIt) loop

         FormalParameter := Dictionary.CurrentSymbol (FormalIt);
         ExpressionNode  := GetNode (ActualIt);
         ParameterNumber := Dictionary.GetSubprogramParameterNumber (FormalParameter);

         SeqAlgebra.CreateSeq (TheHeap, CheckPositionalAssociationRefVar);
         CheckActual (ProcSym,
                      FormalParameter,
                      ParameterNumber,
                      ExpressionNode,
                      CheckPositionalAssociationRefVar,
                      EntireArraySym);
         AddSeq (ParameterNumber, CheckPositionalAssociationRefVar, EntireArraySym);

         FormalIt := Dictionary.NextSymbol (FormalIt);
         ActualIt := STree.NextNode (ActualIt);
      end loop;

      if not Dictionary.IsNullIterator (FormalIt) or
        not STree.IsNull (ActualIt) then
         ErrorHandler.SemanticError (3,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpressionNode),
                                     Dictionary.GetSimpleName (ProcSym));
      end if;
   end HandlePositionalAssociation;

   --=====================================================================--

   procedure CheckParameterOverlap
   --# global in     Abstraction;
   --#        in     ActualParameterList;
   --#        in     ArgListNode;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     FormalPositionList;
   --#        in     LexTokenManager.State;
   --#        in     ProcSym;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        ActualParameterList,
   --#                                        ArgListNode,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FormalPositionList,
   --#                                        LexTokenManager.State,
   --#                                        ProcSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ActualNode : STree.SyntaxNode;
      NodeType   : SPSymbols.SPSymbol;
      ErrPos     : LexTokenManager.Token_Position;
      ActualSym  : Dictionary.Symbol;
      Overlaps    : Boolean;

      -----------------------------------------------------

      procedure NextErrPos
      --# global in     STree.Table;
      --#        in out ActualNode;
      --#           out ErrPos;
      --# derives ActualNode,
      --#         ErrPos     from ActualNode,
      --#                         STree.Table;
      is
      begin
         ActualNode := LastSiblingOf (ParentNode (ActualNode));
         ErrPos := NodePosition (ActualNode);
      end NextErrPos;

      -----------------------------------------------------

      function FormalNumberOfActualParameterAt (Position : Positive) return Natural
      --# global in FormalPositionList;
      is
         Ok    : Boolean;
         PosNo : Natural;

      begin
         SimpleLists.GetItem (FormalPositionList,
                              Position,
                              --to get
                              PosNo,
                              Ok);
         if not Ok then
            PosNo := 0;
         end if;
         return PosNo;
      end FormalNumberOfActualParameterAt;

      -----------------------------------------------------

      procedure CheckOverlapOfExportedParameter (Position : in     Positive;
                                                 Sym      : in     Dictionary.Symbol;
                                                 OverLaps :    out Boolean)
      --# global in Abstraction;
      --#        in ActualParameterList;
      --#        in Dictionary.Dict;
      --#        in FormalPositionList;
      --#        in ProcSym;
      --# derives Overlaps from Abstraction,
      --#                       ActualParameterList,
      --#                       Dictionary.Dict,
      --#                       FormalPositionList,
      --#                       Position,
      --#                       ProcSym,
      --#                       Sym;
      is
         CurrentActual : Dictionary.Symbol;
         CurrentFormal : Natural;

      begin
         OverLaps := False;
         for CurrentPos in Positive range 1 .. SimpleLists.NumberOfItems (ActualParameterList)
         loop
            CurrentActual := ActualParameterAt (CurrentPos);
            CurrentFormal := FormalNumberOfActualParameterAt (CurrentPos);
            if CurrentActual /= Dictionary.NullSymbol and then
               Position /= CurrentPos and then
               Sym = CurrentActual and then
               Dictionary.IsExport (Abstraction,
                                    ProcSym,
                                    Dictionary.GetSubprogramParameter (ProcSym, CurrentFormal))
            then
               OverLaps := True;
               exit;
            end if;
         end loop;
      end CheckOverlapOfExportedParameter;

      -----------------------------------------------------

   begin --CheckParameterOverlap
      ActualNode := ArgListNode;
      loop
         NodeType := SyntaxNodeType (ActualNode);
         exit when NodeType = SPSymbols.simple_name or
            NodeType = SPSymbols.expression;
         ActualNode := Child_Node (ActualNode);
      end loop;
      if NodeType = SPSymbols.simple_name then --named association
         ActualNode := Next_Sibling (ActualNode);
      end if;
      ErrPos := NodePosition (ActualNode);

      for I in Positive range 1 .. SimpleLists.NumberOfItems (ActualParameterList)
      loop
         ActualSym := ActualParameterAt (I);
         if ActualSym /= Dictionary.NullSymbol then
            CheckOverlapOfExportedParameter (I,
                                             ActualSym,
                                             Overlaps);
            if Overlaps then
               ErrorHandler.SemanticError (165,
                                           ErrorHandler.NoReference,
                                           ErrPos,
                                           LexTokenManager.Null_String);
            end if;
         end if;
         NextErrPos;
      end loop;
   end CheckParameterOverlap;

   --=====================================================================--

   procedure CheckCallable (Node     : in     STree.SyntaxNode;
                            ProcSym  : in     Dictionary.Symbol;
                            Scope    : in     Dictionary.Scopes;
                            Ok       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Dotted;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Dotted,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProcSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Ok                        from Dictionary.Dict,
   --#                                        Dotted,
   --#                                        ProcSym,
   --#                                        Scope;
   is
   begin
      if Dictionary.IsCallable (ProcSym, Dotted, Scope) then
         Ok := True;
      else
         Ok := False;
         ErrorHandler.SemanticError (163,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (ProcSym));
      end if;
   end CheckCallable;

   --=====================================================================--

   procedure CheckNotLocalProcedureCallFromProtectedFunction
     (Node     : in     STree.SyntaxNode;
      ProcSym  : in     Dictionary.Symbol;
      Scope    : in     Dictionary.Scopes;
      Ok       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        ProcSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Ok                        from Dictionary.Dict,
   --#                                        ProcSym,
   --#                                        Scope;
   is
      CallingSym,
      CallingRegion : Dictionary.Symbol;
      CallingScope  : Dictionary.Scopes;

      function IsProtectedType (Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.IsType (Sym) and then Dictionary.TypeIsProtected (Sym);
      end IsProtectedType;

      function DeclaredInCallingRegion (ProcSym : Dictionary.Symbol) return Boolean
      --# global in CallingRegion;
      --#        in Dictionary.Dict;
      is
      begin
         return Dictionary.GetRegion (Dictionary.GetScope (ProcSym)) = CallingRegion;
      end DeclaredInCallingRegion;

   begin
      Ok := True;
      -- get symbol of caller
      CallingSym := Dictionary.GetEnclosingCompilationUnit (Scope);

      -- get package or protected type where caller is declared
      CallingScope := Dictionary.GetScope (CallingSym);
      loop
         CallingRegion := Dictionary.GetRegion (CallingScope);
         exit when IsProtectedType (CallingRegion) or else Dictionary.IsPackage (CallingRegion);
         CallingScope := Dictionary.GetEnclosingScope (CallingScope);
      end loop;

      if Dictionary.IsFunction (CallingSym) and then
        IsProtectedType (CallingRegion) and then
        DeclaredInCallingRegion (ProcSym) then
         Ok := False;
         ErrorHandler.SemanticError (986,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (ProcSym));
      end if;
   end CheckNotLocalProcedureCallFromProtectedFunction;

   --=====================================================================--

   procedure CheckPropertyConsistency (Node       : in     STree.SyntaxNode;
                                       ProcSym    : in     Dictionary.Symbol;
                                       Scope      : in     Dictionary.Scopes)
   --# global in     ActualParameterList;
   --#        in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     PrefixSymbol;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        ActualParameterList,
   --#                                        CommandLineData.Content,
   --#                                        PrefixSymbol,
   --#                                        ProcSym,
   --#                                        Scope &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from ActualParameterList,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PrefixSymbol,
   --#                                        ProcSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ThePOorSOSym : Dictionary.Symbol;
      EncUnitSym   : Dictionary.Symbol;
   begin
      EncUnitSym := Dictionary.GetEnclosingCompilationUnit (Scope);

      -- Check that if the procedure has a delay property then there is a
      -- delay property for the enclosing unit (if it is a procedure).
      -- Also mark the enclosing procedure as having come across a
      -- potentially delaying operation.
      --
      if Dictionary.HasDelayProperty (ProcSym) then
         if Dictionary.IsSubprogram (EncUnitSym) then
            if Dictionary.HasDelayProperty (EncUnitSym) then
               Dictionary.MarkAccountsForDelay (EncUnitSym);
            else
               ErrorHandler.SemanticError (912,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           Dictionary.GetSimpleName (ProcSym));
            end if;
         end if;
      end if;

      -- Check that if this is a call to Suspend_Until_True then the actual parameter
      -- appears in the suspends list for the enclosing procedure. Also mark the
      -- enclosing unit as having seen the suspend until true call.
      --
      if CommandLineData.RavenscarSelected and then
        Dictionary.IsPredefinedSuspendUntilTrueOperation (ProcSym) then
         ThePOorSOSym := ActualParameterAt (1);
         if Dictionary.SuspendsOn (TheTaskOrProc => EncUnitSym,
                                   ThePOorSO => ThePOorSOSym) then
            Dictionary.MarkAccountsForSuspendsListItem (TheTaskOrProc => EncUnitSym,
                                                        ThePOorSO => ThePOorSOSym);
         else
            -- The suspension object does not appear in the suspends list for the
            -- enclosing unit.
            --
            ErrorHandler.SemanticErrorSym (910,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           ThePOorSOSym,
                                           Scope);
         end if;
      end if;

      -- Check that if this is an entry then the protected object appears in the
      -- suspends list for the enclosing procedure. Also mark the enclosing unit
      -- as having seen the entry.
      --
      if Dictionary.IsEntry (ProcSym) then
         if Dictionary.SuspendsOn (TheTaskOrProc => EncUnitSym,
                                   ThePOorSO => PrefixSymbol) then
            Dictionary.MarkAccountsForSuspendsListItem (TheTaskOrProc => EncUnitSym,
                                                        ThePOorSO => PrefixSymbol);
         else
            -- The protected object does not appear in the entry annotation for the enclosing
            -- unit.
            --
            ErrorHandler.SemanticErrorSym (910,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           PrefixSymbol,
                                           Scope);
         end if;
      end if;

      -- Check that all the items in the suspends list for the procedure appear
      -- in the suspends list for the enclosing unit.
      --
      if not Dictionary.SuspendsListIsPropagated (FromProcedure => ProcSym,
                                                  ToTaskOrProc => EncUnitSym) then

         -- The operation being called contains entries that are not present in the
         -- entry list of the calling procedure.
         --
         ErrorHandler.SemanticError (913,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (ProcSym));

      end if;

      -- Mark the entries declared in the called procedure as having been seen by the
      -- enclosing procedure. Any entries not propagated are ignored.
      --
      Dictionary.MarkAccountsForSuspendsListItems (TheTaskOrProc => EncUnitSym,
                                                   TheItemsInProcedure => ProcSym);

   end CheckPropertyConsistency;

   --=====================================================================--

begin --wf_procedure_call_statement
      --assume node is procedure_call_statement

   if InPackageInitialization (Scope) then
      ErrorHandler.SemanticError (34,
                                  ErrorHandler.NoReference,
                                  NodePosition (Node),
                                  LexTokenManager.Null_String);

   else --not in package initialization
      ParamTable := ParamTables'(ParamIndex => ParamInfo'(0, Dictionary.NullSymbol));
      CheckProcedureName (Node,
                          Scope,
                           --to get
                          ProcSym,
                          NameNode,
                          Valid);

      -- above call will also have set TaggedParameterSym which if not null is
      -- the type of the formal parameter to which an actual parameter of some
      -- extended type will have to be converted

      if Valid then
         -- determine which set of annotatios to use
         Abstraction := Dictionary.GetAbstraction (ProcSym, Scope);

         if not Dictionary.SubprogramSignatureIsWellformed (Abstraction, ProcSym) then
            ErrorHandler.SemanticWarning (399,
                                          NodePosition (Node),
                                          LexTokenManager.Null_String);
         end if;

         if Child_Node (Node) /= NameNode and then
            Child_Node (Child_Node (Node)) /= NameNode
         then
            Valid := False;
            ErrorHandler.SemanticError (34,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.Null_String);
         end if;
      end if;

      --# assert True;

      if Valid then
         CheckCallable (Node,
                        ProcSym,
                        Scope,
                        --to get
                        Valid);
      end if;

      --# assert True;

      if Valid then
         CheckNotLocalProcedureCallFromProtectedFunction (Node,
                                                          ProcSym,
                                                          Scope,
                                                            --to get
                                                          Valid);
      end if;

      --# assert True;

      if Valid and then CommandLineData.RavenscarSelected and then
        Dictionary.IsOrIsInProtectedScope (Scope) and then
        Dictionary.SubprogramMayBlock (ProcSym) then
         Valid := False;
         ErrorHandler.SemanticError (917,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (ProcSym));
      end if;

      --# assert True;

      if Valid and then CommandLineData.RavenscarSelected and then
        Dictionary.IsFunction (Dictionary.GetRegion (Scope)) and then
        Dictionary.SubprogramMayBlock (ProcSym) then
         Valid := False;
         ErrorHandler.SemanticError (923,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (ProcSym));
      end if;

      --# assert True;

      if Valid and then
        Dictionary.IsInterruptHandler (ProcSym) then
         Valid := False;
         ErrorHandler.SemanticErrorSym (952,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        ProcSym,
                                        Scope);
      end if;

      --# assert True;

      if Valid then
         SimpleLists.Init (ActualParameterList);
         SimpleLists.Init (FormalPositionList);

         CheckGlobalsAreVisible ( -- Abstraction defined since Valid
                                  Node,
                                  ProcSym,
                                  Scope);
         ArgListNode := Next_Sibling (NameNode);
         if ArgListNode = STree.NullNode then
            --parameterless procedure call
            if Dictionary.GetNumberOfSubprogramParameters (ProcSym) /= 0 then
               ErrorHandler.SemanticError (3,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           Dictionary.GetSimpleName (ProcSym));
            end if;
         else --parameter node found
            if Dictionary.GetNumberOfSubprogramParameters (ProcSym) = 0 then
               ErrorHandler.SemanticError (3,
                                           ErrorHandler.NoReference,
                                           NodePosition (ArgListNode),
                                           Dictionary.GetSimpleName (ProcSym));
            else
               if SyntaxNodeType (Child_Node (ArgListNode)) =
                  SPSymbols.named_argument_association
               then
                  HandleNamedAssociation;  -- Abstraction defined since Valid
               else
                  HandlePositionalAssociation; -- Abstraction defined since Valid
               end if;
               CheckParameterOverlap;  -- Abstraction defined since Valid
            end if;
         end if;

         BuildIORelation; -- Abstraction defined since valid

         CheckPropertyConsistency (Node => Node,
                                   ProcSym => ProcSym,
                                   Scope => Scope);

      end if;

      --# assert True;

      -- If we have ended calling an inherited root operation then we need to tell
      -- the VCG which subprogram actually got called.  We know that a root op has
      -- been called if TaggedParameterSym is not null.  In that case we seed the
      -- procedure_call_statement syntax node with the symbol of the root procedure
      -- called.
      if TaggedParameterSym /= Dictionary.NullSymbol then
         STree.AddNodeSymbol (Node, ProcSym);

         -- If we are not dealing with an inherited root operation then we MAY be dealing with
         -- a protected operation call.  Such calls are annotated in terms of their TYPE but we
         -- will want to substitute the particular PO instance of the type.  We can pass this
         -- prefix (PO in this case) to the VCG via the syntax tree.  Note that previous and
         -- current if clauses are mutually exclusive - there are no protected tagged ops.
      elsif PrefixSymbol /= Dictionary.NullSymbol and then
        Dictionary.IsOwnVariable (PrefixSymbol) and then
        Dictionary.GetOwnVariableProtected (PrefixSymbol) then
         STree.AddNodeSymbol (Node, PrefixSymbol);
      end if;

   end if; -- in package initialization
end wf_proc_call; -- Abstraction always defined if used
